//------------------------------------------------------------------------------
// ldpcDecIpPre.v
// 
// Description
//   A preprocessor to the main input block. This maps from an input of
//   `IP_WIDTH samples to an input of DEC_RAM_IP_SAMPLES as used by the VM RAM.
//
// Inputs:
// (From control)
//   enable         : from enable register.
//   blkNum         : The block number within the packet.
//   nShrtFloor     : Floor of number of shortening bits.
//   shrtMod        : 1 when blkNum < nShrtMod.
//   nPuncFloor     : Floor of number of puncturing bits.
//   puncMod        : 1 when blkNum < nPuncMod.
//   n              : Code size (data + parity).
//   k              : Data-part size for the code.
//   ipWidth        : Width of the input samples.
//   setRdyToRcv    : Pulse at end of a decode encode || on first enable.
// (From sender)
//   inStrobe       : High when inData is valid.
//   inDataWord     : `IP_WIDTH samples of 1 bit.
// (From Op block)
// Outputs:
// (To Ip Block)
//   k2             : Number of input bits after shortening.
//   endParity      : Bit number of last parity bit.
//   strobe         : strobe indicating output data available.
//   dataWordPre    : processed data word.
//   isEndData      : Indicates that this is the last data word before parity.
//  
// 8 Oct 2010 M. Rumsey - Adapted from ldpcEncIp.
//
// (c) Copyright 2010-11, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecIpPre
  #(parameter  N_HI = numBits(`LDEC_N_MAX)-1,
               K_HI = numBits(`LDEC_K_MAX)-1,
               REP_HI = numBits(`LDEC_NREPS_MAX)-1,
               RAM_HI = numBits(`LDEC_DEC_RAM_IP_WIDTH)-1,
               BLK_HI = numBits(`LDEC_MAX_BLKNUM)-1)
  (
   // Clocking/reset
   input                                                 nReset,
   input                                                 clk,
   input                                                 clkEn,
   // From control
   input                                                 enable,
   input [BLK_HI:0]                                      blkNum,
   input                                                 setRdyToRcv,
   input                                                 setRdyToRcvD1,
   input                                                 rdyToRcvD1,
   input [K_HI:0]                                        nShrtFloor,
   input [BLK_HI:0]                                      shrtMod,
   input [numBits(`LDEC_M_MAX)-1:0]                      nPuncFloor,
   input [BLK_HI:0]                                      puncMod,
   input [REP_HI:0]                                      nRepFloor,
   input [BLK_HI:0]                                      repMod,
   input [N_HI:0]                                        n,
   input [K_HI:0]                                        k,
   input [numBits(`LDEC_IP_WIDTH)-1:0]                   ipWidth,
   // From IP block
   input                                                 isLastWord,
   // From sender
   input                                                 inStrobe,
   input [`LDEC_IP_BITS*`LDEC_IP_WIDTH-1:0]              inDataWord,
   // To IP block
   output [`LDEC_LB_HI:0]                                loadBaseOut,
   output reg                                            numIpBitsValid, 
   output reg [numBits(`LDEC_MAX_IPBITS)-1:0]            numIpBits, 
   output [K_HI:0]                                       k2Out,
   output [N_HI:0]                                       endParity2Out,
   output [REP_HI:0]                                     numRepsOut, 
   output                                                strobeOut_1,
   output                                                strobeOut,
   output                                                ipStallOut,
   output reg [`LDEC_IP_BITS*`LDEC_DEC_RAM_IP_WIDTH-1:0] dataWordOut);

`include "ldpcDecFuncs.vh"

  // CollectionReg Hi bit. CollectionReg could have `LDEC_DEC_RAM_IP_WIDTH-1 bits,
  // begin get another `LDEC_IP_WIDTH bits, but only have 1 removed (say at data
  // parity interface). It then needs a further `LDEC_IP_WIDTH to store new ip data
  // unless the input is stalled at that point.
  localparam CR_HI = `LDEC_DEC_RAM_IP_WIDTH - 2 + `LDEC_IP_WIDTH*(1+`LDEC_IP_STALL_OK);
  // WriteOffset Hi Bit
  localparam WO_HI = maximum(ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH)-1, 1);
  // See also LB_HI in ldpcDec.svh. This the index into the collection reg.
 
  // Data is manipulated with 2D arrays (num samples x bits in sample)
  reg `LDEC_ip2DType                         inData2D       [`LDEC_IP_WIDTH-1:0];
  reg `LDEC_ip2DType                         collectionReg  [CR_HI:0];

  reg [`LDEC_LB_HI:0]                        loadBase;
  reg [RAM_HI:0]                             bitsToTake;
                                             
  // Track how much data has been written    
  reg [N_HI:0]                               numOpSamples;
  reg [K_HI:0]                               k2;
  reg [N_HI:0]                               endParity;
  reg [N_HI:0]                               endParity2;
  reg [REP_HI:0]                             numReps; 
  reg                                        isShortened;
  wire [N_HI:0]                              nPuncFloorExt;
  // The 3 mods below are only 1 bit, but we extend to N_HI+1
  // for fewer linting errors.
  wire [N_HI:0]                              shrtBlkMod;
  wire [N_HI:0]                              puncBlkMod;
  wire [N_HI:0]                              repBlkMod;
  reg                                        dataNotParity;
  reg [numBits(`LDEC_NREPS_MAX-1)-1:0]       repsRemaining;
  reg                                        repeatMode;
                                             
  reg                                        wrEnable;
  reg                                        wrEnableD1;
  reg                                        wrEnableNxt;
  reg                                        wrEnableNxtD1;
  reg [`LDEC_IP_BITS*`LDEC_DEC_RAM_IP_WIDTH-1:0]  dataWord;
                                             
  reg [`LDEC_LB_HI:0]                        loadBaseNxt;
  reg [N_HI:0]                               numOpSamplesNxt;
  reg [RAM_HI:0]                             bitsToTakeNxt;
  reg                                        dataNotParityNxt;
  reg [numBits(`LDEC_NREPS_MAX-1)-1:0]       repsRemainingNxt;
  reg                                        repeatModeNxt;
  reg                                        ipComplete;
  reg                                        ipStallNxt;
  wire                                       ipStall;
  reg [N_HI:0]                               endParityLessWidth;
  reg [K_HI:0]                               k2LessWidth;
`ifdef BRC_SIMU_ON
//synthesis translate_off                  
//synopsys translate_off                     
  integer                                    maxColRegUsed = 0;
//synthesis translate_on
//synopsys translate_on
`endif //BRC_SIMU_ON

  localparam [31:0] RAM_IP_WIDTH = `LDEC_DEC_RAM_IP_WIDTH;
  localparam IPB_HI = numBits(`LDEC_MAX_IPBITS)-1;

  always @(inDataWord)
  begin : pIp
    //reg [numBits(`LDEC_IP_BITS*`LDEC_IP_WIDTH-1)-1:0] hi;
    integer hi;
    integer sample;
    
    for (sample=0; sample<=`LDEC_IP_WIDTH-1; sample=sample+1) begin
      // Get the 1D index positions for this sample.
      hi = (sample+1)*`LDEC_IP_BITS-1;
      inData2D[sample] = inDataWord[hi-:`LDEC_IP_BITS];
    end
  end //pIp
  
  //---------------------------------------------------------------------------
  // Work out shortening/puncturing/repetition parameters
  //---------------------------------------------------------------------------
  
  assign shrtBlkMod = `LDEC_PAD((blkNum < shrtMod ) ? 1'd1 : 1'd0, N_HI);
  assign puncBlkMod = `LDEC_PAD((blkNum < puncMod ) ? 1'd1 : 1'd0, N_HI);
  assign repBlkMod  = `LDEC_PAD((blkNum < repMod )   ? 1'd1 : 1'd0, N_HI);
  // Extend punc to n for easier comparison with other n based regs.
  assign nPuncFloorExt = `LDEC_PAD(nPuncFloor,
                              numBits(`LDEC_N_MAX)-numBits(`LDEC_M_MAX));
  
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pShrtBits
    reg [K_HI:0] k2V;
    
    if (nReset == 1'b0) begin
      k2 <= `LDEC_PAD(1'b0, K_HI);
      k2LessWidth <= `LDEC_PAD(1'b0, K_HI);
      isShortened <= 1'b0;
    end else begin
      if (setRdyToRcv == 1'b1) begin
        isShortened <= 1'b0;
        k2V = k - (nShrtFloor + shrtBlkMod[K_HI:0]);
        k2 <= k2V;
        if (k2V < RAM_IP_WIDTH[K_HI:0]) begin
          k2LessWidth <= `LDEC_PAD(1'b0, K_HI);
        end else begin
          k2LessWidth <= k2V - RAM_IP_WIDTH[K_HI:0];
        end
        
        if (nShrtFloor != `LDEC_PAD(1'b0, K_HI) || shrtBlkMod[0] != 1'b0) begin
          isShortened <= 1'b1;
        end
      end
    end

  end //pShrtBits

  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pPuncBits
    reg [N_HI:0] endPV;
    
    if (nReset == 1'b0) begin
      endParity <= `LDEC_PAD(1'b0, N_HI);
      endParity2 <= `LDEC_PAD(1'b0, N_HI);
      endParityLessWidth <= `LDEC_PAD(1'b0, N_HI);
    end else begin
      if (setRdyToRcv == 1'b1) begin
        // endParity as used in this block is based on the number of received
        // bits whereas endParity2 used in ldpcDecIp is based on bit positions
        // such that the bit number incorporates the number of shortened bits.
        endPV = n - (nShrtFloor + shrtBlkMod) -
                (nPuncFloorExt + puncBlkMod);
        endParity <= endPV;      
        endParityLessWidth <= endPV - RAM_IP_WIDTH[N_HI:0];
        endParity2 <= n - (nPuncFloorExt + puncBlkMod);
      end
    end

  end //pPuncBits

  // Repeat bits plus total
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pRepBits
    reg calcTotalBitsV;
    
    if (nReset == 1'b0) begin
      numReps <= `LDEC_PAD(1'b0, REP_HI);
      calcTotalBitsV = 1'b0;
      numIpBits <= `LDEC_PAD(1'b0, IPB_HI);
      numIpBitsValid <= 1'b0;   
    end else begin
      if (!enable) begin
        numIpBits <= `LDEC_PAD(1'b0, IPB_HI);
        numIpBitsValid <= 1'b0;
        numReps <= `LDEC_PAD(1'b0, REP_HI);
      end else begin
        // Calculate total number of bits expected on the input port. This info
        // is sent back to the first input sample stage so that we can force
        // rdyToRcv to drop low on the last word.
        if (calcTotalBitsV == 1'b1) begin
          calcTotalBitsV = 1'b0;
          numIpBitsValid <= 1'b1;
          numIpBits <= `LDEC_PAD(endParity, IPB_HI-N_HI) +
                       numReps;
//                       `LDEC_PAD(numReps, IPB_HI-REP_HI);
        end
        if (setRdyToRcv == 1'b1) begin
          calcTotalBitsV = 1'b1;
          numReps <= nRepFloor + repBlkMod;
          numIpBitsValid <= 1'b0;
        end
      end
    end
    
  end //pRepBits

  //---------------------------------------------------------------------------
  // Control
  //---------------------------------------------------------------------------

  // Work out the indexes etc. There is a combinatorial part because we need
  // wrEnable_1 in ldpcDecIp.

  always @(loadBase, numOpSamples, dataNotParity, inStrobe,
           setRdyToRcv, k2, k2LessWidth, endParity, endParityLessWidth,
           repsRemaining, repeatMode, ipComplete, isLastWord,
           ipWidth, ipStall, isShortened)
  begin : pIdx
    // loadBaseV has working headroom compared to loadBase.
    reg [`LDEC_LB_HI+1:0]                     loadBaseV;
    reg [N_HI:0]                              numOpSamplesV;
    reg [RAM_HI:0]                            bitsToTakeV;
    reg                                       wrEnableV;
    reg                                       dataNotParityV;
    reg [numBits(`LDEC_NREPS_MAX-1)-1:0]      repsRemainingV;
    reg                                       repeatModeV;
    reg                                       resetNumOpSamplesV;
    // Lint: some bits may be unused.
    reg [31:0]                                tmp32;
  
    // Loadbase is the index in collectionReg where the Ip word gets
    // put. Its current value also tells us how many samples are already in the
    // collection register.
    
    // After the following, loadBaseV tells us how many samples will
    // reside in collectionReg if none are shifted out.
    loadBaseV = {1'b0, loadBase};
    if (inStrobe == 1'b1 && ipStall == 1'b0) begin
      loadBaseV = loadBaseV + `LDEC_PAD(ipWidth, (`LDEC_LB_HI+1)-(numBits(`LDEC_IP_WIDTH)-1));
    end
    // Detect when there are enough samples to transfer && correct
    // loadBaseV if data is removed.
    // Initialise vars. 
    dataNotParityV = dataNotParity;
    repsRemainingV = repsRemaining;
    repeatModeV = repeatMode;
    wrEnableV = 1'b0;
    resetNumOpSamplesV = 1'b0;
    ipStallNxt = 1'b0;

    // Mostly we remove the full RAM_IP_WIDTH samples ...
    bitsToTakeV = RAM_IP_WIDTH[RAM_HI:0];
    numOpSamplesV = numOpSamples + RAM_IP_WIDTH[N_HI:0];
    
    // ... except at the transition from data to parity || 
    //  parity to data (case of repetition).
    if (dataNotParity == 1'b1) begin
      // Check for the data/parity interface.
      // The following is what we want to do:
      //if (numOpSamplesV >= k2) begin
      // but the timing is improved if instead we note that at this point
      // numOpSamplesV is numOpSamples+DEC_RAM_IP_WIDTH so
      // we can compare numOpSamples with k2-DEC_RAM_IP_WIDTH.
      if (numOpSamples >= k2LessWidth) begin
        tmp32 = `LDEC_PAD(k2 - numOpSamples, 31-N_HI);
        bitsToTakeV = tmp32[RAM_HI:0];
        numOpSamplesV = k2;
        // We change dataNotParity in sync with the last data sample so it
        // changes a cycle early.
        dataNotParityV = 1'b0;
        // Generate an input stall at the data/parity boundary to
        // reduce the size requirement of collectionReg.
        ipStallNxt = isShortened & inStrobe;
      end
    end else begin
      // Check for the parity/data interface.
      // The following if is what we want to do:
      //if (numOpSamplesV >= endParity) begin
      // but timing is improved if instead we note that at this point
      // numOpSamplesV is numOpSamples+DEC_RAM_IP_WIDTH so
      // we can compare numOpSamples with endParity-DEC_RAM_IP_WIDTH.
      if (numOpSamples >= endParityLessWidth) begin
        bitsToTakeV = endParity - numOpSamples;
        // reset sample count for a possible repeat.
        dataNotParityV = 1'b1;     
        repeatModeV = 1'b1;
        resetNumOpSamplesV = 1'b1;
      end
    end
    // ... || when we have received the last repeat.
    if (repeatMode == 1'b1) begin
      if (`LDEC_PAD(bitsToTakeV, REP_HI-RAM_HI) >= repsRemaining) begin
        bitsToTakeV = repsRemaining[RAM_HI:0];
        repsRemainingV = `LDEC_PAD(1'b0, REP_HI);
      end
      else begin
        repsRemainingV = repsRemaining - `LDEC_PAD(bitsToTakeV, REP_HI-RAM_HI);
      end
      if (resetNumOpSamplesV == 1'b0) begin
        numOpSamplesV = numOpSamples + `LDEC_PAD(bitsToTakeV, N_HI-RAM_HI);
      end
    end
    // Will we have enough data buffered to do a write?
//    if (loadBaseV >= bitsToTakeV) begin 
    if (loadBaseV >= `LDEC_PAD(bitsToTakeV, (`LDEC_LB_HI+1)-RAM_HI)) begin 
//    if (`LDEC_PAD(loadBaseV, RAM_HI-`LDEC_LB_HI) >= bitsToTakeV) begin 
      wrEnableV = 1'b1;
      loadBaseV = loadBaseV - `LDEC_PAD(bitsToTakeV, (`LDEC_LB_HI+1)-RAM_HI);  
//      loadBaseV = loadBaseV - bitsToTakeV[RAM_HI:0];  
//      loadBaseV = loadBaseV - bitsToTakeV[`LDEC_LB_HI+1:0];  
    end
    // Tidy up last write by overriding any spurious calculations.
    if ((setRdyToRcv || ipComplete || isLastWord)  == 1'b1) begin
      wrEnableV = 1'b0;
      ipStallNxt = 1'b0;
    end
    // Override any other calculations at parity/data interface.
    if (resetNumOpSamplesV == 1'b1) begin
      numOpSamplesV = `LDEC_PAD(1'b0, N_HI);
    end
    loadBaseNxt = loadBaseV[`LDEC_LB_HI:0];
    numOpSamplesNxt = numOpSamplesV;
    bitsToTakeNxt = bitsToTakeV;
    wrEnableNxt = wrEnableV;
    dataNotParityNxt = dataNotParityV;
    bitsToTakeNxt = bitsToTakeV;
    repsRemainingNxt = repsRemainingV;
    repeatModeNxt = repeatModeV;

  end //pIdx

  generate
    reg ipStallTmp;
    if (`LDEC_IP_STALL_OK == 1) begin: gIpStall1
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pIpStall
        if (nReset == 1'b0) begin
          ipStallTmp <= 1'b0;
        end else begin
          if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
            // The input stall is only ever 1 cycle.
            ipStallTmp <= ipStallNxt & ! ipStall;
          end
        end
      end //pIpStall
    end
    assign ipStall = ipStallTmp;
  endgenerate //  gIpStall1

  generate
    if (`LDEC_IP_STALL_OK == 0) begin: gIpStall0
      assign ipStall  = 1'b0;
    end
  endgenerate //  gIpStall0
  
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pIdxReg
    if (nReset == 1'b0) begin
      loadBase <= `LDEC_PAD(1'b0, `LDEC_LB_HI);
      dataNotParity <= 1'b1;
      bitsToTake <= `LDEC_PAD(1'b0, RAM_HI);
      dataNotParity <= 1'b1;
      numOpSamples <= `LDEC_PAD(1'b0, N_HI);
      wrEnable <= 1'b0;
      repeatMode <= 1'b0;
      repsRemaining <= `LDEC_PAD(1'b0, REP_HI);
      ipComplete <= 1'b0;
    end else begin
      if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
  
        if (setRdyToRcvD1 == 1'b1) begin
          // nb numReps is updated on setRdyToRcv.
          repsRemaining <= numReps;
        end
        // Catch the isLastWord pulse to create a 'complete' signal that ensures
        // no further writes.
        if (isLastWord == 1'b1) begin
          ipComplete <= 1'b1;
        end
        // `LDEC_CONCAT_IP mode requires that any bits of the input word that are !
        // used by the current block belong to the next, so some registers are
        // ! cleared between blocks, only at start of packet.
        if (setRdyToRcv == 1'b1 && ! `LDEC_CONCAT_IP) begin
          loadBase <= `LDEC_PAD(1'b0, `LDEC_LB_HI);
        end
        // Initialise
        if (setRdyToRcv == 1'b1) begin
          dataNotParity <= 1'b1;
          bitsToTake <= `LDEC_PAD(1'b0, RAM_HI);
          numOpSamples <=`LDEC_PAD(1'b0, N_HI) ;
          wrEnable <= 1'b0;
          repeatMode <= 1'b0;
          repsRemaining <= `LDEC_PAD(1'b0, REP_HI);
          ipComplete <= 1'b0;
          // The enable below is primarily rdyToRcv. wrEnable is just for the
          // end of the block to make sure we sent wrEnable low.
        end
        else if (rdyToRcvD1 == 1'b1 || wrEnable == 1'b1) begin
          // Save the key parameters if a write is confirmed.
          if (wrEnableNxt == 1'b1) begin
            bitsToTake <= bitsToTakeNxt;
            dataNotParity <= dataNotParityNxt;
            numOpSamples <= numOpSamplesNxt;
            repsRemaining <= repsRemainingNxt;
            repeatMode <= repeatModeNxt;
          end
          wrEnable <= wrEnableNxt;
          loadBase <= loadBaseNxt;
          
        end  
      end
      // Synchronous clear is mostly at start of each block, except for this one.
      if (enable == 1'b0) begin
        loadBase <= `LDEC_PAD(1'b0, `LDEC_LB_HI);
      end
    end
  end //pIdxReg
  
  //---------------------------------------------------------------------------
  // Collect input data [realign:RAM word size word]
  //---------------------------------------------------------------------------
  
  // We get `LDEC_IP_WIDTH samples, which are moved into a wider collection register.
  // Data is taken out of the register from the bottom when there are >=
  // `LDEC_DEC_RAM_IP_WIDTH bits present and any remaining data in the collection
  // register is shifted down `LDEC_DEC_RAM_IP_WIDTH bits. When data is loaded into
  // the collection register it must be loaded at loadBase, which is the start
  // of free space above the remaining data. At the transition from data to
  // parity we only shift out the data part and loadBase is updated accordingly.
  // Similarly at the end of parity we shift out just the parity bits, making
  // sure that the rest of the output register is masked (to get zeros as
  // required for puncturing).

  always @(posedge(clk) `LDEC_RESET_STR_B)
  begin : pCollect
    reg     [`LDEC_LB_HI:0]   inIdxV;
    reg     [31:0]            b;
    integer                   i;
    reg     `LDEC_ip2DCr_P    collectionRegP;
    integer                   idx1;
    
    if (!nReset) begin
      for (i=0; i<=CR_HI; i=i+1) begin
        collectionReg[i] <= `LDEC_PAD(1'b0, `LDEC_IP_BITS-1);
      end
    end else begin
      if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
        
        // Shift down collection register if data has been taken out.
        if (wrEnable == 1'b1) begin
          `LDEC_PACKF(collectionReg, collectionRegP, `LDEC_IP_BITS, `LDEC_CR_HI+1);
          collectionRegP = ldpcDecShiftDown(collectionRegP, RAM_HI,
                                            0, bitsToTake);
          `LDEC_UNPACKQ(collectionRegP, collectionReg, `LDEC_IP_BITS, `LDEC_CR_HI+1);
        end
        if (ipStall == 1'b0) begin

`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
          if (loadBase+ipWidth > CR_HI+1)
            $error("Overflow of collection register.");
          if (ipComplete == 1'b0 && maxColRegUsed < loadBase+ipWidth) begin
            maxColRegUsed <= loadBase+ipWidth;
          end
//synthesis translate_on
//synopsys translate_on
`endif //BRC_SIMU_ON
          // Load to the collection reg at loadBase
          for (b=0; b<=CR_HI; b=b+1) begin
            if (b[`LDEC_LB_HI+1:0] >= {1'b0, loadBase} && ipComplete == 1'b0)
            begin
              // Load inData into collection register
              inIdxV = b[`LDEC_LB_HI:0] - loadBase;
              if (inIdxV < ipWidth) begin
                collectionReg[b[`LDEC_LB_HI+1:0]] <= inData2D[inIdxV];
              end
            end
          end
        end
      end
    end
  end //pCollect

  //---------------------------------------------------------------------------
  // Data output
  //---------------------------------------------------------------------------

  // Normally the number of output bits 'bitsToTake' is `LDEC_DEC_RAM_IP_WIDTH, but if
  // it isn't begin we zero the unused bits.

 // always @(collectionReg, bitsToTake)
  always @(*)
  begin : pOp
    //reg [numBits(`LDEC_IP_BITS*`LDEC_DEC_RAM_IP_WIDTH-1)-1:0]  hi;
    integer hi;
    integer sample;
    
    for (sample=0; sample<=`LDEC_DEC_RAM_IP_WIDTH-1; sample=sample+1) begin
      // Get the 1D index positions for this sample.
      hi = (sample+1)*`LDEC_IP_BITS-1;
      if (sample[RAM_HI:0] < bitsToTake) begin
        dataWord[hi-:`LDEC_IP_BITS] = collectionReg[sample];
      end
      else begin
        dataWord[hi-:`LDEC_IP_BITS] = `LDEC_PAD(1'b0, `LDEC_IP_BITS-1);
      end
    end
  end //pOp

  // When timing is difficult add a register here.
  // See also related registering in ldpcDecIp (gRepReg).
  // Note that this pipeline stage is independent of PIPE A/B/C, we merely
  // Currently disabled by using 10.
  generate
    if (`LDEC_PIPE_ABCX >= 10) begin: gOpReg1
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pOpReg
        if (nReset == 1'b0) begin
          wrEnableD1 <= 1'b0;
          wrEnableNxtD1 <= 1'b0;
        end else begin
          if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
            // Because of additional latency the isLastWord from ldpcDecIp did !
            // get back to pIdx in time to turn off wrEnable so we do it here.
            wrEnableD1 <= wrEnable & ! isLastWord;
            wrEnableNxtD1 <= wrEnableNxt;
          end
        end
      end //pOpReg
      
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin
        if (nReset == 1'b0) begin
          dataWordOut <= `LDEC_PAD(1'b0, `LDEC_IP_BITS*`LDEC_DEC_RAM_IP_WIDTH-1);       
        end else begin
          if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
            dataWordOut <= dataWord;
          end
        end
      end
    end
  endgenerate //  gOpReg1

  generate
    if (`LDEC_PIPE_ABCX < 10) begin: gOpReg0
      always @(*) begin
        wrEnableD1 = wrEnable;
        wrEnableNxtD1 = wrEnableNxt;
        dataWordOut = dataWord;
      end
    end
  endgenerate //  gOpReg0    
  
  assign strobeOut  = wrEnableD1;
  assign strobeOut_1  = wrEnableNxtD1;
  assign k2Out  = k2;
  assign endParity2Out  = endParity2;
  assign numRepsOut  = numReps;
  assign loadBaseOut  = loadBase;
  assign ipStallOut  = ipStall;
  
endmodule

