//-----------------------------------------------------------------------------
// ldcpDecRegs.v
// 
// Description
//   "Memory mapped" registers for the LDPC decoder. A basic single cycle
//   interface is provided. This provides raw access to the registers and
//   may be easily interfaced to a more substantial bus protocol with
//   external (or internal) adaption logic.
// 
// Inputs:
//   nReset           : Asynchronous reset.
//   clk              : Master clock.
// (From the processor)
//   writeEn          : Active high write enable.
//   readEn           : Active high read enable.
//   addr             : Address bus from processor.
//   wrData           : Write data bus from processor.
// (Read-only registers)
//   curBlkNum        : Block number within the packet that is being decoded.
//   curIteration     : The current iteration number of the ldpc decode.
//   prevParityErrs   : Number of parity errors in previous iteration.
//   prevDecodeStatus : 0/1 is fail/pass of previously decoded block.
//   
// Outputs:
// (To the processor)
//   rdDataOut        : Read data bus.
// [Control registers:ctrl and elsewhere]
//   enable           : Master enable. Blip this low between decodes to init.
//   zEnum            : Select from `Z_SIZES in ldpcDecPkg.
//   rEnum            : Select code rate from R_SIZES in ldpcDecPkg.
//   bitsPerSymbol    : NCBPS*mStbc.
//   packetLen        : Number of blocks in packet.
//   packetBytes      : Number of bytes in packet AFTER by/byte padding
//                      is removed.
//   frameEndByte     : number of bytes to end of current MAC frame.
//   packetLen        : Number of blocks in packet.
//   packetBytes      : Number of bytes in packet AFTER by/byte padding
//                      is removed.
//   frameEndByte     : Byte number of last byte in current MAC frame.
//   nShrtFloor       : Floor of number of shorting bits.
//   shrtMod          : 1 when blkNum < nShrtMod.
//   nPuncFloor       : Floor of number of shorting bits.
//   puncMod          : 1 when blkNum < nPuncMod.
//   nRepFloor        : Floor of number of repetition bits.
//   repMod           : 1 when blkNum < nRepMod.
//   parityThresh     : Used for early termination if 'poor' progress is made.
//   nomInterations    : Max number of iterations allowed on average (x8)
//   earlyTestIterations : Controls iter at which early termination tested.
//   maxRunningCount  : Max number of iterations (x8) - peak.
//   endRunningCount  : Max number of iterations (x8) near end of packet.
//   
// 21 Apr 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecRegs
( 
  input                                   nReset,
  input                                   clk,
  // Processor interface inputs.
  input [`LDEC_BUS32:0]                   writeEn,
  input                                   readEn,
  input [`LDEC_NUM_ADDR_BITS-1:0]         addr,
  input [`LDEC_NUM_DATA_BITS-1:0]         wrData,

  // Read only registers that may be monitored by processor.
  input [numBits(`LDEC_MAX_BLKNUM)-1:0]   blkErrs,
  input [numBits(`LDEC_MAX_BLKNUM)-1:0]   curBlkNum,
  input [numBits(`LDEC_MAX_ITER)-1:0]     curIteration,
  input [numBits(`LDEC_PAR_ERRS_MAX)-1:0] prevParityErrs,
  input [numBits(`LDEC_MAX_ITER)-1:0]     prevIterations,
  input                                   prevDecodeStatus,
  input [`LDEC_VAR_BITS+5:0]              varMetricChkSum,
  input [`LDEC_CELL_RAM_W-1:0]            cellRdData,

  output                                  clkEnOut,
  output [`LDEC_NUM_DATA_BITS-1:0]        rdDataOut,
    
  // Control registers.
    
  // Master enable to control.
  output                                  enableOut,
  // Special Modes.
  output                                  enLostTimeOut,
  output                                  beatTimeLineOut,
  // To mem block to program the code cell information.
  output                                  cellAccessOut,
  output                                  cellReadOut,
  output                                  cellWriteOut,
  output reg [`LDEC_CELL_RAM_W-1:0]       cellWrDataOut,
  // Code characteristics to control.
  output [numBits(`LDEC_Z_ENUM_MAX)-1:0]  zEnumOut,
  output [numBits(`LDEC_R_ENUM_MAX)-1:0 ] rEnumOut,
  output [numBits(`LDEC_BPS_MAX)-1:0]     bitsPerSymbolOut,
  // Packet Management
  output [numBits(`LDEC_MAX_BLKNUM)-1:0]  packetLenOut,
  output [15:0]                           packetBytesLsOut,
  output [`LDEC_PB_MS_LEFT:0]             packetBytesMsOut,
  output [15:0]                           frameEndByteLsOut,
  output [`LDEC_PB_MS_LEFT+1:0]           frameEndByteMsOut,
  // Shortening etc parameters to control.
  output [numBits(`LDEC_N_MAX-1)-1:0]     nShrtFloorOut,
  output [numBits(`LDEC_MAX_BLKNUM)-1:0]  shrtModOut,
  output [numBits(`LDEC_M_MAX-1)-1:0]     nPuncFloorOut,
  output [numBits(`LDEC_MAX_BLKNUM)-1:0]  puncModOut,
  output [numBits(`LDEC_NREPS_MAX)-1:0]   nRepFloorOut,
  output [numBits(`LDEC_MAX_BLKNUM)-1:0]  repModOut,
  output [`LDEC_VAR_BITS-1+7:0]           targetLevelOut,
  output [numBits(`LDEC_K_MAX)-1:0]       parityThreshOut,
  output [numBits(8*`LDEC_MAX_ITER)-1:0]  nomIterationsOut,
  output [numBits(`LDEC_MAX_ITER)-1:0]    earlyTestIterationsOut,
  output [numBits(8*`LDEC_MAX_ITER)-1:0]  maxRunningCountOut,
  output [numBits(8*`LDEC_MAX_ITER)-1:0]  endRunningCountOut,
  output [`LDEC_CHK_BITS-2:0]             llrUnityRegOut);

`include "ldpcDecFuncs.vh"
 
  reg                                 enable;
  reg                                 enLostTime;
  reg                                 beatTimeLine;
  reg [numBits(`LDEC_MAX_BLKNUM)-1:0] packetLen;
  reg [15:0]                          packetBytesLs;
  reg [`LDEC_PB_MS_LEFT:0]            packetBytesMs;
  reg [15:0]                          frameEndByteLs;
  reg [`LDEC_PB_MS_LEFT+1:0]          frameEndByteMs;
  
  reg [numBits(`LDEC_Z_ENUM_MAX)-1:0] zEnum;
  reg [numBits(`LDEC_R_ENUM_MAX)-1:0] rEnum;
  reg [numBits(`LDEC_BPS_MAX)-1:0]    bitsPerSymbol;
  reg [numBits(`LDEC_N_MAX-1)-1:0]    nShrtFloor;
  reg [numBits(`LDEC_MAX_BLKNUM)-1:0] shrtMod;
  reg [numBits(`LDEC_M_MAX-1)-1:0]    nPuncFloor;
  reg [numBits(`LDEC_MAX_BLKNUM)-1:0] puncMod;
  reg [numBits(`LDEC_NREPS_MAX)-1:0]  nRepFloor;
  reg [numBits(`LDEC_MAX_BLKNUM)-1:0] repMod;
  
  reg [`LDEC_VAR_BITS-1+7:0]          targetLevel;
  reg [numBits(`LDEC_K_MAX)-1:0]      parityThresh;
  reg [numBits(8*`LDEC_MAX_ITER)-1:0] nomIterations;
  reg [numBits(`LDEC_MAX_ITER)-1:0]   earlyTestIterations;
  reg [numBits(8*`LDEC_MAX_ITER)-1:0] maxRunningCount;
  reg [numBits(8*`LDEC_MAX_ITER)-1:0] endRunningCount;
  reg [`LDEC_CHK_BITS-2:0]            llrUnityReg;
  
  wire                           rdSel0;
  wire                           rdSel1;
  wire                           writeEn0;
  wire                           writeEn1;
  wire [15:0]                    wrData0;
  wire [15:0]                    wrData1;
  reg  [15:0]                    rdData0;
  reg  [15:0]                    rdData1;
  wire [`LDEC_NUM_ADDR_BITS-2:0] addr2;
 
  reg                            cellAccess;
  reg                            cellWrite;

  generate
    if (`LDEC_BUS32==0) begin: gBus16
      // Support 16-bit. Odd && Even addresses are both written from
      // writeData[15:0]. LSB of adddress selects between odd/even.
      assign writeEn0 = writeEn[0] & !addr[0];
      assign writeEn1 = writeEn[0] & addr[0];  
      assign wrData0  = wrData[15:0];
      assign wrData1  = wrData[15:0];
      // Select odd/even read words according to address.
      assign rdSel0   = readEn & !addr[0];
      assign rdSel1   = readEn & addr[0];
      assign rdDataOut = (addr[0]) ? rdData1 : rdData0;
    end
  endgenerate // gBus16
  
  generate
    if (`LDEC_BUS32==1) begin: gBus32
      // Support for 16-bit plus 32-bit. Odd && Even addresses can
      // be written simultanously. LSB of address is ignored. 
      assign writeEn0 = writeEn[0];
      assign writeEn1 = writeEn[`LDEC_BUS32];  
      assign wrData0  = wrData[15:0];
      assign wrData1  = wrData[15+`LDEC_BUS32*16:`LDEC_BUS32*16];
      // Both words are presented
      assign rdSel0   = readEn;
      assign rdSel1   = readEn;
      assign rdDataOut = {rdData1, rdData0};
    end
  endgenerate // gBus32
  
  assign addr2 = addr[`LDEC_NUM_ADDR_BITS-1:1];  
  assign clkEnOut = (| writeEn) | cellWrite;

  //---------------------------------------------------------------------------
  // Write registers
  //---------------------------------------------------------------------------
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pWrite
    if (!nReset) begin
      // register bank reset states for even addresses.
      enable           <= 1'b0;
      enLostTime       <= 1'b0;
      beatTimeLine     <= 1'b0;
      cellAccess       <= 1'b0;
      llrUnityReg      <= `LDEC_PAD(1'b0, `LDEC_CHK_BITS-1);
      if (`LDEC_PAD_SUPPORT) begin
        packetBytesLs  <= `LDEC_PAD(1'b0, 16);
      end
      if (`LDEC_SPECIAL_AGG_SUPPORT) begin
        frameEndByteLs   <= `LDEC_PAD(1'b0, 16);
      end
      zEnum            <= 2'b0;
      rEnum            <= 2'b0;
      // Leave these tied off when SUPPORT is disabled.
      nShrtFloor       <= `LDEC_PAD(1'b0, numBits(`LDEC_N_MAX-1)-1);
      nPuncFloor       <= `LDEC_PAD(1'b0, numBits(`LDEC_M_MAX-1)-1);
      nRepFloor        <= `LDEC_PAD(1'b0, numBits(`LDEC_NREPS_MAX)-1);
      targetLevel      <= `LDEC_PAD(1'b0, `LDEC_VAR_BITS-1+7);
      endRunningCount  <= `LDEC_PAD(1'b0, 7);
      nomIterations    <= `LDEC_PAD(1'b0, 7);
      cellWrDataOut    <= `LDEC_PAD(1'b0, `LDEC_CELL_RAM_W-1);
      cellWrite        <= 1'b0;
      // register bank reset states for odd addresses.
      packetLen        <= `LDEC_PAD(1'b0, 7);
      if (`LDEC_PAD_SUPPORT) begin
        packetBytesMs  <= `LDEC_PAD(1'b0, `LDEC_PB_MS_LEFT);
      end
      if (`LDEC_SPECIAL_AGG_SUPPORT) begin
        frameEndByteMs <= `LDEC_PAD(1'b0, `LDEC_PB_MS_LEFT+1);
      end
      zEnum            <= `LDEC_PAD(1'b0, numBits(`LDEC_Z_ENUM_MAX)-1);
      rEnum            <= `LDEC_PAD(1'b0, numBits(`LDEC_R_ENUM_MAX)-1);
      bitsPerSymbol    <= `LDEC_PAD(1'b0, numBits(`LDEC_BPS_MAX)-1);
      // Leave these tied off when SUPPORT is disabled.
      shrtMod          <= `LDEC_PAD(1'b0, numBits(`LDEC_MAX_BLKNUM)-1);
      puncMod          <= `LDEC_PAD(1'b0, numBits(`LDEC_MAX_BLKNUM)-1);   
      repMod           <= `LDEC_PAD(1'b0, numBits(`LDEC_MAX_BLKNUM)-1);   
      if (`LDEC_EARLY_TERM_SUPPORT) begin
        parityThresh   <= `LDEC_PAD(1'b0, numBits(`LDEC_K_MAX)-1);
        earlyTestIterations <= `LDEC_PAD(1'b0, numBits(`LDEC_MAX_ITER)-1);
      end
      maxRunningCount  <= `LDEC_PAD(1'b0, numBits(8*`LDEC_MAX_ITER)-1);       
    end else begin
      //---------------------------------------------------------------------------
      // Write registers on even addresses
      //---------------------------------------------------------------------------
      cellWrite <= 1'b0;               
      if (writeEn0) begin
        case (addr2)
          `LDEC_ENABLE_ADDR/2 : begin
            enable <= wrData0[0];
            enLostTime <= wrData0[1];
            beatTimeLine <= wrData0[2];
            cellAccess <= wrData0[3];
            llrUnityReg <= wrData0[(`LDEC_CHK_BITS-1)+7:8];
          end
          `LDEC_PACKET_BYTES_LS_ADDR/2 : begin
            if (`LDEC_PAD_SUPPORT) begin
              packetBytesLs <= wrData0;
            end
          end
          `LDEC_FRAME_END_BYTE_LS_ADDR/2 : begin
            if (`LDEC_SPECIAL_AGG_SUPPORT) begin
              frameEndByteLs <= wrData0;
            end
          end
          `LDEC_CODE_ENUM_ADDR/2 : begin
            zEnum <= wrData0[1:0]; 
            rEnum <= wrData0[9:8]; 
          end
          `LDEC_SHRT_FLOOR_ADDR/2 : begin
            if (`LDEC_SHRT_SUPPORT) begin
              nShrtFloor <= wrData0[numBits(`LDEC_N_MAX-1)-1:0];
            end
          end
          `LDEC_PUNC_FLOOR_ADDR/2 : begin
            if (`LDEC_PUNC_SUPPORT) begin
              nPuncFloor <= wrData0[numBits(`LDEC_M_MAX-1)-1:0];
            end
          end
          `LDEC_REP_FLOOR_ADDR/2 : begin
            if (`LDEC_REP_SUPPORT) begin
              nRepFloor <= wrData0[numBits(`LDEC_NREPS_MAX)-1:0];
            end
          end
          `LDEC_TARGET_LEVEL_ADDR/2 : begin
            targetLevel <= wrData[`LDEC_VAR_BITS-1+7:0];
          end
          `LDEC_ITERATIONS1_ADDR/2 : begin
            nomIterations <= wrData0[7:0];
            endRunningCount <= wrData0[15:8];
          end
          `LDEC_CODE_ADDR/2 : begin
            cellWrDataOut <= wrData0[`LDEC_CELL_RAM_W-1:0];
            cellWrite <= cellAccess;
          end
          default : ;
        endcase
      end
      //---------------------------------------------------------------------------
      // Write registers on odd addresses
      //---------------------------------------------------------------------------
      if (writeEn1 == 1'b1) begin
        case (addr2)
          `LDEC_PACKET_LEN_ADDR/2 : begin
            packetLen <= wrData1[numBits(`LDEC_MAX_BLKNUM)-1:0];
          end
          `LDEC_PACKET_BYTES_MS_ADDR/2 : begin
            if (`LDEC_PAD_SUPPORT) begin
              packetBytesMs <= wrData1[`LDEC_PB_MS_LEFT:0];
            end
          end
          `LDEC_FRAME_END_BYTE_MS_ADDR/2 : begin
            if (`LDEC_SPECIAL_AGG_SUPPORT) begin
              frameEndByteMs <= wrData1[`LDEC_PB_MS_LEFT+1:0];
            end
          end
          `LDEC_BITS_PER_SYMBOL_ADDR/2 : begin
            bitsPerSymbol <= wrData1[numBits(`LDEC_BPS_MAX)-1:0]; 
          end
          `LDEC_SHRT_MOD_ADDR/2 : begin
            if (`LDEC_SHRT_SUPPORT) begin
              shrtMod <= wrData1[numBits(`LDEC_MAX_BLKNUM)-1:0];
            end
          end
          `LDEC_PUNC_MOD_ADDR/2 : begin
            if (`LDEC_PUNC_SUPPORT) begin
              puncMod <= wrData1[numBits(`LDEC_MAX_BLKNUM)-1:0];
            end
          end
          `LDEC_REP_MOD_ADDR/2 : begin
            if (`LDEC_REP_SUPPORT) begin
              repMod <= wrData1[numBits(`LDEC_MAX_BLKNUM)-1:0];
            end
          end
          `LDEC_PARITY_THRESH_ADDR/2 : begin
            if (`LDEC_EARLY_TERM_SUPPORT) begin
              parityThresh <= wrData1[numBits(`LDEC_K_MAX)-1:0];
            end
          end
          `LDEC_ITERATIONS2_ADDR/2 : begin
            maxRunningCount <= wrData1[7:0];
            if (`LDEC_EARLY_TERM_SUPPORT) begin
              earlyTestIterations <= wrData1[numBits(`LDEC_MAX_ITER)-1+8:8];
            end            
          end
          default : ;
        endcase
      end
    end
  end // pWrite
  
  //---------------------------------------------------------------------------
  // Read Mux
  //---------------------------------------------------------------------------

  always @(*) begin
    rdData0 = `LDEC_PAD(1'b0, 15);
    if (rdSel0) begin
      case (addr2) 
        `LDEC_BLK_VERSION_ADDR/2        : rdData0 = 
//                                `LDEC_NUM_DATA_BITS'`LDEC_BLK_ID_VER;
                                `LDEC_BLK_ID_VER;
        `LDEC_ENABLE_ADDR/2             :
          rdData0 = {llrUnityReg, 4'b0000, cellAccess, beatTimeLine, enLostTime, enable};
        `LDEC_PACKET_BYTES_LS_ADDR/2    : rdData0 = packetBytesLs;
        `LDEC_FRAME_END_BYTE_LS_ADDR/2  : rdData0 = frameEndByteLs;
        
        `LDEC_CODE_ENUM_ADDR/2          : rdData0 = {rEnum, 6'b000000, zEnum};
        `LDEC_SHRT_FLOOR_ADDR/2         : rdData0 = `LDEC_SHRT_SUPPORT ? nShrtFloor : `LDEC_PAD(1'b0, numBits(`LDEC_N_MAX-1)-1);
        `LDEC_PUNC_FLOOR_ADDR/2         : rdData0 = `LDEC_PUNC_SUPPORT ? nPuncFloor : `LDEC_PAD(1'b0, numBits(`LDEC_M_MAX-1)-1);
        `LDEC_REP_FLOOR_ADDR/2          : rdData0 = `LDEC_REP_SUPPORT ? nRepFloor : `LDEC_PAD(1'b0, numBits(`LDEC_NREPS_MAX)-1);
        `LDEC_TARGET_LEVEL_ADDR/2       : rdData0 = targetLevel;
        `LDEC_ITERATIONS1_ADDR/2        : rdData0 = {endRunningCount, nomIterations};
        // Status registers      
        `LDEC_CUR_BLK_NUM_ADDR/2        : rdData0 = curBlkNum;
        `LDEC_PREV_ITERATIONS_ADDR/2    : rdData0 = prevIterations;
        `LDEC_PREV_DECODE_STATUS_ADDR/2 : rdData0 = prevDecodeStatus;
        `LDEC_VAR_METRIC_CHKSUM_ADDR/2  : rdData0 = varMetricChkSum;
        default : ;
        
      endcase
    end
  end

   always @(*) begin
    rdData1 = `LDEC_PAD(1'b0, 16);
    if (rdSel1) begin
      case (addr2)
        
        `LDEC_PACKET_LEN_ADDR/2        : rdData1 = packetLen;
        `LDEC_PACKET_BYTES_MS_ADDR/2   : rdData1 = packetBytesMs;
        `LDEC_FRAME_END_BYTE_MS_ADDR/2 : rdData1 = frameEndByteMs;
        
        `LDEC_BITS_PER_SYMBOL_ADDR/2   : rdData1 = bitsPerSymbol;
        `LDEC_SHRT_MOD_ADDR/2          : rdData1 = `LDEC_SHRT_SUPPORT ? shrtMod : `LDEC_PAD(1'b0, numBits(`LDEC_MAX_BLKNUM)-1);
        `LDEC_PUNC_MOD_ADDR/2          : rdData1 = `LDEC_PUNC_SUPPORT ? puncMod : `LDEC_PAD(1'b0, numBits(`LDEC_MAX_BLKNUM)-1);
        `LDEC_REP_MOD_ADDR/2           : rdData1 = `LDEC_REP_SUPPORT ? repMod : `LDEC_PAD(1'b0, numBits(`LDEC_MAX_BLKNUM)-1);
        `LDEC_PARITY_THRESH_ADDR/2     : rdData1 = `LDEC_EARLY_TERM_SUPPORT ? parityThresh : `LDEC_PAD(1'b0, numBits(`LDEC_K_MAX)-1);
        `LDEC_ITERATIONS2_ADDR/2       : rdData1 = {earlyTestIterations, maxRunningCount};       
         // Status registers      
        `LDEC_BLK_ERRS_ADDR/2           : rdData1 = blkErrs;
        `LDEC_PREV_PARITY_ERRS_ADDR/2   : rdData1 = prevParityErrs;
        `LDEC_CUR_ITERATION_ADDR/2      : rdData1 = curIteration;
        `LDEC_CODE_ADDR/2               : rdData1 = cellRdData;
        default : ;
        
      endcase
    end
  end

 // Create a pulse when the code cell information is read. This is used to
  // increment the read address.
  assign cellReadOut = cellAccess & rdSel1;

  assign enableOut = enable;
  assign enLostTimeOut = enLostTime;
  assign beatTimeLineOut = beatTimeLine;
  assign zEnumOut = zEnum;
  assign rEnumOut = rEnum;
  assign bitsPerSymbolOut = bitsPerSymbol;
  assign nShrtFloorOut = nShrtFloor;
  assign shrtModOut = shrtMod;
  assign nPuncFloorOut = nPuncFloor;
  assign puncModOut = puncMod;
  assign nRepFloorOut = nRepFloor;
  assign repModOut = repMod;
  assign targetLevelOut = targetLevel;

  generate
    if (`LDEC_EARLY_TERM_SUPPORT) begin: gETS
    assign parityThreshOut = parityThresh;
    assign earlyTestIterationsOut = earlyTestIterations;
  end
  endgenerate // gETS
  
  assign nomIterationsOut = nomIterations;
  assign maxRunningCountOut = maxRunningCount;
  assign endRunningCountOut = endRunningCount;
  assign llrUnityRegOut = llrUnityReg;
  assign cellAccessOut = cellAccess;
  assign packetLenOut = packetLen;
  assign packetBytesLsOut = packetBytesLs;
  assign packetBytesMsOut = packetBytesMs;
  assign frameEndByteLsOut = frameEndByteLs;
  assign frameEndByteMsOut = frameEndByteMs;
  
  assign cellWriteOut = cellWrite;

endmodule
