//------------------------------------------------------------------------------
// ldcpDecShort.v
// 
// Description
//   Creates a mask identifying which bits are shortened.
//
// Parameters:
// Inputs:
// (Programmable config)
//   z              : The macro cells size for the current code.
//   k              : Data-part size for the code.
//   k2             : Data-part size without shortening zeros.
// (From control)
//   col            : The column number to process.
//   stall          : The datapath is stalled.
// Outputs:
//   haveShort      : Shortening is taking place on this cell.
//   shorted        : The mask indicating shortened positions.
//  
// 2 Apr 2010 M. Rumsey - Ported from C reference model.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecShort
  #(parameter
    Z_HI  = numBits(`LDEC_Z_MAX)-1,
    Z1_HI = numBits(`LDEC_Z_MAX-1)-1,
    K_HI  = numBits(`LDEC_K_MAX)-1)
(
  input                                  nReset,
  input                                  clk,
  input [numBits(`LDEC_Z_ENUM_MAX)-1:0]  zEnum,
  input [Z_HI:0]                         z,
  input [K_HI:0]                         k,
  input [K_HI:0]                         k2,
       // From control    
  input [numBits(`LDEC_NCOLS_RIGHT)-1:0] col_1,
  input                                  stall_1,
  input                                  stall,
  input [Z1_HI:0]                        cycShift_1, 
  input [Z1_HI:0]                        cycShift, 
        // Up to fwd || bwd blocks 
  output [`LDEC_Z_MAX-1:0]               shortedOut);

`include "ldpcDecFuncs.vh"

  localparam NC_HI = numBits(`LDEC_NCOLS_RIGHT)-1;
  localparam [K_HI:0] K_MAX  = `LDEC_K_MAX;

  // Tabulate the start/end micro column numbers for each macro column.
  function automatic [K_HI:0] getStartCol(input [1:0]     zE,
                                          input [NC_HI:0] col);    
    begin
      if (col[NC_HI:0] < PARITY_START_COL(`LDEC_R_ENUM_MAX)) begin
        getStartCol = $unsigned(col[NC_HI:0]) * Z_SIZES2(zE);
      end else begin
        // This value is outside 'K' region and shortening will
        // never actually be triggered. Cannot set it to (say)
        // 0 without making endCol calculation more complicated.
        getStartCol = K_MAX;
      end
    end
  endfunction // getStartCol
 
  // Local variables - no registers required
  wire [K_HI:0]                          startCol;
  wire [numBits(`LDEC_K_MAX+`LDEC_Z_MAX)-1:0] endCol;
  reg  [Z1_HI:0]                         startColZ;
  reg  [Z1_HI:0]                         endColZ;
  reg                                    crosses;
  reg                                    haveShort;

  // Pipeline registering
  reg  [K_HI:0]                          endShort_1DA;
  reg  [K_HI:0]                          startShort_1DA;
  wire [Z1_HI:0]                         cycShift_1DA;
  wire                                   stall_1DA;
  wire [K_HI:0]                          startColDelta;  // used when PIPE_A=0
  reg  [K_HI:0]                          startColDA;
  reg  [numBits(`LDEC_K_MAX+`LDEC_Z_MAX)-1:0] endColDA;
  reg                                    stall_1DB;
  reg  [Z1_HI:0]                         startColZDB;
  reg  [Z1_HI:0]                         endColZDB;
  reg                                    haveShortDB;
  reg                                    crossesDB;

  // Local registers 
  reg [`LDEC_Z_MAX-1:0]                  shorted;


  assign startCol = getStartCol(zEnum, col_1);
  assign endCol = startCol + `LDEC_PAD(z, K_HI-Z1_HI) - `LDEC_PAD(1'b1, K_HI);
  
  //---------------------------------------------------------------------------
  // Pipe A
  //---------------------------------------------------------------------------

  // Pipelining adds registers between the processes above && below.
  // The aim is not specifically to improve performance between these
  // processes : in this is just an economical way to add latency so that
  // shorted is delayed when PIPEA is enabled in the fwd block.

  generate
    if (`LDEC_PIPE_A == 1) begin : genPipeA
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pPipe
        if (nReset == 1'b0) begin
          endShort_1DA   <= K_MAX;
          startShort_1DA <= K_MAX;
          startColDA     <= K_MAX;
          endColDA       <= K_MAX;
        end else begin
          endShort_1DA   <= k-`LDEC_PAD(1'b1, K_HI);
          startShort_1DA <= k2;
          startColDA     <= startCol;
          endColDA       <= endCol;
        end
      end //pPipe
      assign stall_1DA    = stall;
      assign cycShift_1DA = cycShift;
      
    end else begin

      always @(*) begin
        if (k > `LDEC_PAD(1'b0, K_HI)) begin
          endShort_1DA = k-`LDEC_PAD(1'b1, K_HI);
        end else begin
          endShort_1DA = `LDEC_PAD(1'b0, K_HI);
        end
        startShort_1DA  = k2;
        startColDA = startColDelta;
        endColDA = endCol;
      end
      assign startColDelta = startCol;
      assign stall_1DA  = stall_1;
      assign cycShift_1DA = cycShift_1;
    end
  endgenerate //  genPipeA 

  //---------------------------------------------------------------------------
  // Work out start/end relative to current macro column.
  //---------------------------------------------------------------------------

  always @(endShort_1DA, startShort_1DA, stall_1DA,
           cycShift_1DA, startColDA, endColDA, z)
  begin : pShort

    reg [Z1_HI:0] startColZV;
    reg [Z1_HI:0] endColZV;

    haveShort = 1'b0;
    crosses = 1'b0;                     // crosses Z boundary
    startColZV = `LDEC_PAD(1'b0, Z1_HI);  // start of short rel to cur col.
    endColZV = `LDEC_PAD(1'b0, Z1_HI);

    // Don't do arithmetic when stalled to avoid spurious out-of-range errs.
    if (stall_1DA == 1'b0) begin
      // Check for an overlap between our col && the shorted region.
      if ((startColDA <= endShort_1DA) && (endColDA >= startShort_1DA)) begin
        haveShort = 1'b1;
        // Get start && end of shorted region relative to this block of z.
        
        if (startShort_1DA > startColDA) begin
          startColZV = startShort_1DA - startColDA;
        end
        else begin
          startColZV = `LDEC_PAD(1'b0, Z1_HI);
        end
        
        // cyclical shift
        if (cycShift_1DA > startColZV) begin
          startColZV = startColZV + z - cycShift_1DA;
          // crosses will indicate whether start > end so that the start
          // to end region becomes the non-shorted region.
          crosses = 1'b1;
        end else begin
          startColZV = startColZV - cycShift_1DA;
        end
        
        // Now do the end of region.
        if (endShort_1DA >= endColDA) begin
          endColZV = z - `LDEC_PAD(1'b1, Z1_HI);
        end else begin
          endColZV = endShort_1DA - startColDA;  // end rel to this blk of Z.
        end
        
        // cyclical shift.
        if (cycShift_1DA > endColZV) begin
          endColZV = endColZV - cycShift_1DA + z;
          crosses = 1'b0;                      // cancel 'crosses'.
        end else begin
          endColZV = endColZV - cycShift_1DA;        
        end

      end else begin
        haveShort = 1'b0;
      end
      
    end
    startColZ = startColZV;
    endColZ = endColZV;
  end //pShort
  
  //---------------------------------------------------------------------------
  // Pipe B
  //---------------------------------------------------------------------------

  generate
    if (`LDEC_PIPE_B == 1) begin: genPipeB
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pPipe
        if (nReset == 1'b0) begin
          haveShortDB <= 1'b0;
          crossesDB <= 1'b0;
          startColZDB <= `LDEC_PAD(1'b0, Z1_HI);
          endColZDB <= `LDEC_PAD(1'b0, Z1_HI);
          stall_1DB <= 1'b0; 
        end else begin
          if (stall_1DA == 1'b0) begin
            haveShortDB <= haveShort;
            crossesDB <= crosses;
            startColZDB <= startColZ;
            endColZDB <= endColZ;
          end
        stall_1DB <= stall_1DA;
        end
      end //pPipe
    end else begin
      always @(*) begin
        haveShortDB = haveShort;
        crossesDB = crosses;
        startColZDB = startColZ;
        endColZDB = endColZ;
        stall_1DB = stall_1DA;
      end
    end
  endgenerate //  genPipeB
  
  //---------------------------------------------------------------------------
  // Identify shortened micro-columns
  //---------------------------------------------------------------------------

  generate
    genvar zI;
    
    for (zI=0; zI <`LDEC_Z_MAX; zI=zI+1) begin: genShort
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pShort2
        if (nReset == 1'b0) begin
          shorted[zI] <= 1'b0;
        end else begin
          if (stall_1DB == 1'b0) begin
            shorted[zI] <= 1'b0;
            if (haveShortDB == 1'b1) begin
              if (crossesDB == 1'b0) begin
                if ((zI[Z_HI:0] >= startColZDB) && (zI[Z_HI:0] <= endColZDB)) begin
                  shorted[zI] <= 1'b1;
                end
              end else begin
                if ((zI[Z_HI:0] >= startColZDB) || (zI[Z_HI:0] <= endColZDB)) begin
                  shorted[zI] <= 1'b1;
                end
              end
            end
          end
        end
      end //pShort2
    end
  endgenerate //  genShort
  
  assign shortedOut  = shorted;

endmodule
