//------------------------------------------------------------------------------
// ldcpDec.v
// 
// Description
//   Top level wrapper of the LDPC decoder with registers LUT.
// 
//------------------------------------------------------------------------------

`include "ldpcDec.vh"
   
module ldpcDecTop
(
 input                                    nReset,
 // Inputs from sender
 input                                    inStrobe,
 input [numBits(`LDEC_IP_WIDTH)-1:0]      ipWidth,
 input [`LDEC_IP_WIDTH*`LDEC_IP_BITS-1:0] inDataWord,
 input [`LDEC_CHK_BITS-1:0]               llrUnity,
 input                                    iterationAbort,
 // Inputs from output interface
 input                                    clrToSend,
 // Outputs to sender
 output                                   rdyToRcvOut,
 output                                   lastIpSampleOut,
 // Outputs to downstream hardware
 output                                   decodeCompleteOut,
 output                                   decodeStatusOut,
 output                                   packetCompleteOut,
 output                                   packetOpCompleteOut,
 output                                   packetStatusOut,
 output                                   opStrobeOut,
 output [`LDEC_OP_WIDTH-1:0]              opDataWordOut,
 output [`LDEC_OP_WIDTH/8-1:0]            opByteEnOut,
 //-------------------------------------------------------------------------
 // Registers
 //-------------------------------------------------------------------------
 // Master enable to control.
 input                                    enable,
 input                                    enLostTime,
 input                                    beatTimeLine,
 // To mem block to program the code cell information.
 input                                    cellAccess,
 input                                    cellRead,
 input                                    cellWrite,
 input [`LDEC_CELL_RAM_W-1:0]             cellWrData,
 // Code characteristics to control.
 input [numBits(`LDEC_Z_ENUM_MAX)-1:0]    zEnum,
 input [numBits(`LDEC_R_ENUM_MAX)-1:0]    rEnum,
 input [numBits(`LDEC_BPS_MAX)-1:0]       bitsPerSymbol,
 // Packet Management
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    packetLen,
 input [15:0]                             packetBytesLs,
 input [`LDEC_PB_MS_LEFT:0]               packetBytesMs,
 input [15:0]                             frameEndByteLs,
 input [`LDEC_PB_MS_LEFT+1:0]             frameEndByteMs,
 // Shortening etc parameters to control.
 input [numBits(`LDEC_N_MAX-1)-1:0]       nShrtFloor,
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    shrtMod,
 input [numBits(`LDEC_M_MAX-1)-1:0]       nPuncFloor,
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    puncMod,
 input [numBits(`LDEC_NREPS_MAX)-1:0]     nRepFloor,
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    repMod,
 // Status registers
 output [numBits(`LDEC_MAX_BLKNUM)-1:0]   blkErrsOut,
 output [numBits(`LDEC_MAX_BLKNUM)-1:0]   curBlkNumOut,
 output [numBits(`LDEC_MAX_ITER)-1:0]     curIterationOut,
 output [numBits(`LDEC_PAR_ERRS_MAX)-1:0] prevParityErrsOut,
 output [numBits(`LDEC_MAX_ITER)-1:0]     prevIterationsOut,
 output                                   prevDecodeStatusOut,
 output [`LDEC_VAR_BITS+5:0]              varMetricChkSumOut,
 output [`LDEC_CELL_RAM_W-1:0]            cellRdDataOut,
 //-------------------------------------------------------------------------
 // Clocking
 //-------------------------------------------------------------------------
 input                                    regsClk,
 input                                    ipClk,
 input                                    opClk,
 input                                    decClk,
 input                                    cellClk,
 input                                    vmRam1Clk, 
 input                                    vmRam2Clk, 
 input                                    hdRamClk, 
 input                                    crRamClk, 
 input                                    ctrlClk,
 input                                    stbc,
 input  [2:0]                             qamEnum,  // 0:4 is BPSK, QPSK, 16QAM, 64QAM, 256QAM
 input  [1:0]                             sizeEnum, // zEnum
 input  [1:0]                             rateEnum, // rEnum
 input                                    ac,       // 0:11n 1:11ac
 input  [1:0]                             bw,       // 0:3 is 20/40/80/160MHz
 input  [6:0]                             mcs,      // 0-9 for 11ac, 0-72 for 11n
 input  [1:0]                             nssLess1, // e.g. NSS=4 is specified as a 3.
 input  [`LDEC_B_BITS-1:0]                nCw,      // packetLen in LDPC blocks
 input  [1:0]                             nSym,     // num OFDM symbols. clip this to 3.
 output [7:0]                             rxMean,
 output                                   ipClkEnOut,
 output                                   opClkEnOut, 
 output                                   decClkEnOut,
 output                                   cellClkEnOut,
 output                                   vmRam1ClkEnOut, 
 output                                   vmRam2ClkEnOut, 
 output                                   hdRamClkEnOut, 
 output                                   crRamClkEnOut, 
 output                                   ctrlClkEnOut,
`ifdef LDEC_RAMS_AT_TOP
 // Memories (when `LDEC_RAMS_AT_TOP)
`ifndef LDEC_USE_CELL_ROM
 input [`LDEC_CELL_RAM_O-1:0]             cellRamOps,
 output [`LDEC_CELL_RAM_I-1:0]            cellRamIps,
`endif
 input [`LDEC_VM_RAM_O-1:0]               vmRamOps,
 output [`LDEC_VM_RAM_I-1:0]              vmRamIps,
 input [`LDEC_HD_RAM_O-1:0]               hdRamOps,
 output [`LDEC_HD_RAM_I-1:0]              hdRamIps,
 // For the cr && vr RAMS there are 2 port && 1 port options with slightly
 // different signal lists, hence signals are provided for both cases though
 // only one will be used at a time.
`ifdef LDEC_2_PORTS
 input [`LDEC_CR_RAM_O-1:0]               crRamOps,
 output [`LDEC_CR_RAM_I-1:0]              crRamIps,
 input [`LDEC_VR_RAM_O-1:0]               vrRamOps,
 output [`LDEC_VR_RAM_I-1:0]              vrRamIps,
`else
 input [`LDEC_CR1_RAM_O-1:0]              cr1RamOps,
 output [`LDEC_CR1_RAM_I-1:0]             cr1RamIps,
 input [`LDEC_VR1_RAM_O-1:0]              vr1RamOps,
 output [`LDEC_VR1_RAM_I-1:0]             vr1RamIps,
`endif
`endif
 //----------------------------
 //  Debug Port
 //----------------------------
 output [`LDEC_DBG_PORT_WIDTH-1:0]        dbgIp1Out,
 output [`LDEC_DBG_PORT_WIDTH-1:0]        dbgIp2Out,
 output [`LDEC_DBG_PORT_WIDTH-1:0]        dbgOpOut,
 output [`LDEC_DBG_PORT_WIDTH-1:0]        dbgCtrlOut);

`include "ldpcDecFuncs.vh"

                      
                                  
  //---------------------------------------------------------------------------
  // Status signals readable by processor.
  //---------------------------------------------------------------------------

  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   blkErrs;
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   curBlkNum;
  wire [numBits(`LDEC_MAX_ITER)-1:0]     curIteration;
  wire [numBits(`LDEC_PAR_ERRS_MAX)-1:0] prevParityErrs;
  wire [numBits(`LDEC_MAX_ITER)-1:0]     prevIterations;
  wire                                   prevDecodeStatus;
  wire [`LDEC_VAR_BITS+5:0]              varMetricChkSum;
  wire [`LDEC_CELL_RAM_W-1:0]            cellRdData;
  
  // Debug port
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgIp1;
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgIp2;
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgOp;
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgCtrl;

 
  //---------------------------------------------------------------------------
  // Registers && Control
  //---------------------------------------------------------------------------
 wire [`LDEC_CHK_BITS-2:0]               llrUnityReg;
 wire [`LDEC_VAR_BITS-1+7:0]             targetLevel;
 wire [numBits(`LDEC_K_MAX)-1:0]         parityThresh;
 wire [numBits(8*`LDEC_MAX_ITER)-1:0]    nomIterations;
 wire [numBits(`LDEC_MAX_ITER)-1:0]      earlyTestIterations;
 wire [numBits(8*`LDEC_MAX_ITER)-1:0]    maxRunningCount;
 wire [numBits(8*`LDEC_MAX_ITER)-1:0]    endRunningCount;

   ldpcDecLuTop regsLut
     (
      .clk                 (ipClk),
      .rst_n               (nReset),
      
      // Control registers.
      .stbc                (stbc),
      .qamEnum             (qamEnum),
      .sizeEnum            (sizeEnum),
      .rateEnum            (rateEnum),    
      .ac                  (ac),
      .bw                  (bw),
      .mcs                 (mcs),
      .nssLess1            (nssLess1),
      .nCw                 (nCw),
      .nSym                (nSym),
      
      
      .rxMean              (rxMean),
      .llrUnityReg         (llrUnityReg),
      .targetLevel         (targetLevel),
      .parityThresh        (parityThresh),
      .nomIterations       (nomIterations),
      .earlyTestIterations (earlyTestIterations),
      .maxRunningCount     (maxRunningCount),
      .endRunningCount     (endRunningCount));

  ldpcDecCore core
    (
     .nReset              (nReset),
     // Inputs from sender
     .inStrobe            (inStrobe),
     .inDataWord          (inDataWord),
     .llrUnity            (llrUnity),
     .iterationAbort      (iterationAbort),
     // Inputs from output interface
     .clrToSend           (clrToSend),
     // Outputs to sender
     .rdyToRcvOut         (rdyToRcvOut),
     .lastIpSampleOut     (lastIpSampleOut),
     // Outputs to downstream hardware
     .decodeCompleteOut   (decodeCompleteOut),
     .decodeStatusOut     (decodeStatusOut),
     .packetCompleteOut   (packetCompleteOut),
     .packetOpCompleteOut (packetOpCompleteOut),
     .packetStatusOut     (packetStatusOut),
     .opStrobeOut         (opStrobeOut),
     .opDataWordOut       (opDataWordOut),
     .opByteEnOut         (opByteEnOut),
     //-----------------------------------------
     // Registers
     //-----------------------------------------
     // Master enable to control.
     .enable              (enable),
     .enLostTime          (enLostTime),
     .beatTimeLine        (beatTimeLine),     
     // To mem block to program the code cell information.
     .cellAccess          (cellAccess),
     .cellRead            (cellRead),
     .cellWrite           (cellWrite),
     .cellWrData          (cellWrData),
     // Code characteristics to control.
     .ipWidth             (ipWidth),
     .zEnum               (zEnum),
     .rEnum               (rEnum),
     .bitsPerSymbol       (bitsPerSymbol),
     .packetLen           (packetLen),
     .packetBytesLs       (packetBytesLs),
     .packetBytesMs       (packetBytesMs),
     .frameEndByteLs      (frameEndByteLs),
     .frameEndByteMs      (frameEndByteMs),    
     // Shortening etc parameters to control.
     .nShrtFloor          (nShrtFloor),
     .shrtMod             (shrtMod),
     .nPuncFloor          (nPuncFloor),
     .puncMod             (puncMod),
     .nRepFloor           (nRepFloor),
     .repMod              (repMod),
     .targetLevel         (targetLevel),
     .parityThresh        (parityThresh),
     .nomIterations       (nomIterations),
     .earlyTestIterations (earlyTestIterations),
     .maxRunningCount     (maxRunningCount),
     .endRunningCount     (endRunningCount),
     .llrUnityReg         (llrUnityReg),
     // Status registers
     .blkErrsOut          (blkErrs),
     .curBlkNumOut        (curBlkNum),
     .curIterationOut     (curIteration),
     .prevParityErrsOut   (prevParityErrs),
     .prevIterationsOut   (prevIterations),
     .prevDecodeStatusOut (prevDecodeStatus),
     .varMetricChkSumOut  (varMetricChkSum),
     .cellRdDataOut       (cellRdData),
     //-----------------------------------------
     // Clocking
     //-----------------------------------------
     .regsClk             (regsClk),
     .ipClk               (ipClk),
     .opClk               (opClk),
     .decClk              (decClk),
     .cellClk             (cellClk),
     .vmRam1Clk           (vmRam1Clk),
     .vmRam2Clk           (vmRam2Clk),
     .hdRamClk            (hdRamClk),
     .crRamClk            (crRamClk),
     .ctrlClk             (ctrlClk),
     .ipClkEnOut          (ipClkEnOut),
     .opClkEnOut          (opClkEnOut),
     .decClkEnOut         (decClkEnOut),
     .cellClkEnOut        (cellClkEnOut),
     .vmRam1ClkEnOut      (vmRam1ClkEnOut),
     .vmRam2ClkEnOut      (vmRam2ClkEnOut),
     .hdRamClkEnOut       (hdRamClkEnOut),
     .crRamClkEnOut       (crRamClkEnOut),
     .ctrlClkEnOut        (ctrlClkEnOut),
     //-----------------------------------------------------------------------
     // I/O busses to RAMS (when `LDEC_RAMS_AT_TOP == 1)
     //-----------------------------------------------------------------------
`ifdef LDEC_RAMS_AT_TOP
`ifndef LDEC_USE_CELL_ROM
     .cellRamOps          (cellRamOps),
     .cellRamIps          (cellRamIps),
`endif
     .vmRamOps            (vmRamOps),
     .vmRamIps            (vmRamIps),
     .hdRamOps            (hdRamOps),
     .hdRamIps            (hdRamIps),
`ifdef LDEC_2_PORTS
     .crRamOps            (crRamOps),
     .crRamIps            (crRamIps),
     .vrRamOps            (vrRamOps),
     .vrRamIps            (vrRamIps),
`else
     .cr1RamOps           (crRamOps),
     .cr1RamIps           (cr1RamIps),
     .vr1RamOps           (vrRamOps),
     .vr1RamIps           (vr1RamIps),
`endif
`endif
     //----------------------------
     //  Debug Port
     //----------------------------
     .dbgIp1Out           (dbgIp1),
     .dbgIp2Out           (dbgIp2),
     .dbgOpOut            (dbgOp),
     .dbgCtrlOut          (dbgCtrl));

  //assign decodeActiveOut  = dbgCtrl[0];

  
endmodule

