//------------------------------------------------------------------------------
// ldcpDecVmMem.v
// 
// Description:
//   Contains the following memories:
//   cellInfo:
//     Stores code information. Address gen is cellIdx from ctrl blk
//      for read during operation. Write control comes from regs blk.
//   varMetrics:
//     Stores metrics. Address gen is via cellInfo in this blk. On first
//     iteration reads are
//   hardDecision:
//     Duplicates varMetrics sign.
//
//   ipClkEn             : Enables varMetricsIpMem clock when input;
//                         being loaded by the Ip block.
//   vmIpRamClk          : Clock for varMetricsIpMem.
//   vmDecRamClk          : Clock for varMetricsDecMem.
//   cellClk             : Clock for Port A of CellMem.
//   decClk              : Clock for memories used during decode.
//   enable              : Enable register.
// (From Regs)
//   cellAccess          : The Cell Info RAM is in exernal access mode.
//   cellRead            : Read the Cell Info RAM (this cycle).
//   cellWrite           : Write cellWrData to the Cell Info RAM.
//   cellWrData          : Code info to load into the Cell Info RAM.
// (From Ip)             
//   wrData              : Write data of length $size(inData).
//   wrOffset            : Write is to the bits starting at
//                         wrOffset*$size(inData).
//   wrAddr              : Write address - actually the macro column number.
//   wrEnable            : Memory write enable.    
// (From Ctrl)           
//   fwdCellIdx          : Index into the cell info memory.
//   bwdCellIdx          : Index into the cell info memory.
//   bwdStall            : Disables the write to varmetrics.
//   holdFwdCell         : These cause the Cell RAM reading to be disabled so
//   holdBwdCell         : that previous fetched data remains on the RAM
//                         output latches.
//   blkPingPong         : Determines which Vm buffer is used for input versus
//                         decode. Must be toggled at the moment decode starts
//                         for a block. This is not the same as the moment that
//                         is available.
// (From Bwd)            
//   varMetricWr         : Write bus of signal metrics.
//   hardDecisionsWr     : Write bus of hard decisions.                         
// (To Clocking)         
//   vmIpClkEn           : Clock enable for the varMetricsIpMem. Active
//                         during load of input and the first iteration.
//   cellClkEn           : Clock enable for the cell memory.
//   decClkEn            : General clock enable fo mems used in decoding.
// (To Ctrl && Fwd/Bwd)
//   fwdFlag             : The flag bits for the current fwd cell.
//   bwdFlag             : The flag bits for the current bwd cell.
// (To Fwd)              
//   varMetricRd         : Read bus of signal metrics.
// (To Bwd)
//   hardDecisionsRd     : Read bus from hard decisions.
//   
// 21 Apr 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

`define LDEC_WR_OFFSET_LEFT numBits(maximum(ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH)-1, 1))-1

module ldpcDecVmMem
(  
   input                                             nReset,
   input                                             ipClkEn,
   input                                             vmRam1Clk,
   input                                             vmRam2Clk,
   input                                             hdRamClk,
   input                                             cellClk,
   input                                             decClk,
   input                                             freeClk, // RW modification
   input                                             enable,
   input                                             disabling, 
   input                                             decodeActive,
   input                                             decodeActive_1,
   input                                             decodeActive_2,
   // From regs to load Cell Info RAM
   input                                             cellAccess,
   input                                             cellRead,
   input                                             cellWrite,
   input [`LDEC_CELL_RAM_W-1:0]                      cellWrData,
   input [numBits(`LDEC_Z_ENUM_MAX)-1:0]             zEnum,
   input [numBits(`LDEC_R_ENUM_MAX)-1:0]             rEnum,
   // From Ip (Initialisation of varMetrics)
   input [`LDEC_DEC_RAM_IP_WIDTH*`LDEC_VAR_BITS-1:0] wrData,
   input [`LDEC_WR_OFFSET_LEFT:0]                    wrOffset,
   input [numBits(`LDEC_NCOLS_RIGHT)-1:0]            wrAddr,
   input                                             wrEnable, 
   input [numBits(`LDEC_NCOLS_RIGHT)-1:0]            rdAddr,
   input                                             rdEnable,
   // To memory
   output [`LDEC_VM_RAM_W-1:0]                       rdData,
   // From Ctrl          
   input [numBits(`LDEC_NCELLS-1)-1:0]               fwdCellIdx,
   input [numBits(`LDEC_NCELLS-1)-1:0]               bwdCellIdx,
   input                                             bwdCellPrime,
   input                                             bwdStall,
   input                                             bwdStall_1,
   input                                             fwdStall_1,
   input                                             bwdNoOp,
   input                                             fwdNoOp_1,
   input                                             holdFwdCell_1,
   input                                             holdBwdCell_1,
   input                                             metricVeryFirstUse, 
   input                                             blkPingPongS,
   input                                             blkPingPongE,
   input                                             newBlk,
   // From Bwd           
   input `LDEC_varSlvZType                           varMetricWr,
   input `LDEC_sgnZType                              hardDecisionsWr,
    // From Op
   input                                             opBufferSel,
   input [`LDEC_VM_RAM_A-1:0]                        opBufferAddr,
   // To clocking        
   output                                            cellClkEnOut,
   output                                            vmRam1ClkEnOut,
   output                                            vmRam2ClkEnOut,
   output                                            hdRamClkEnOut,
   // To Regs             
   output reg [`LDEC_CELL_RAM_W-1:0]                 cellRdDataOut,
   // To Ctrl and fwd/bwd
   output [`LDEC_FLAG_BITS-1:0]                      fwdFlagOut,
   output [`LDEC_FLAG_BITS-1:0]                      bwdFlagOut,
   output [`LDEC_FLAG_BITS-1:0]                      fwdFlagOut_1,
   output [`LDEC_FLAG_BITS-1:0]                      bwdFlagOut_1,
   output [numBits(`LDEC_NCELLS)-1:0]                numMacroCellsOut,
   // To Fwd             
   output `LDEC_varSlvZType                          varMetricRdOut,
   output [numBits(`LDEC_NCOLS_RIGHT)-1:0]           fwdColOut_1,
   output [numBits(`LDEC_Z_MAX-1)-1:0]               fwdCycShiftOut_1,
   output [numBits(`LDEC_NCOLS_RIGHT)-1:0]           fwdColOut,
   output [numBits(`LDEC_Z_MAX-1)-1:0]               fwdCycShiftOut,
   // To Bwd             
   output `LDEC_sgnZType                             hardDecisionsRdOut,
   output [numBits(`LDEC_NCOLS_RIGHT)-1:0]           bwdColOut_1,
   output [numBits(`LDEC_NCOLS_RIGHT)-1:0]           bwdColOut,
   output [numBits(`LDEC_Z_MAX-1)-1:0]               bwdCycShiftOut_1,
   // To Op
   output `LDEC_sgnZType                             opBufferDataOut
   // Memories (when `LDEC_RAMS_AT_TOP)
`ifdef LDEC_RAMS_AT_TOP
   ,
`ifndef LDEC_USE_CELL_ROM
   input [`LDEC_CELL_RAM_O-1:0]                      cellRamOps,
   output [`LDEC_CELL_RAM_I-1:0]                     cellRamIps,
`endif
   input [`LDEC_VM_RAM_O-1:0]                        vmRamOps,
   output [`LDEC_VM_RAM_I-1:0]                       vmRamIps,
   input [`LDEC_HD_RAM_O-1:0]                        hdRamOps,
   output [`LDEC_HD_RAM_I-1:0]                       hdRamIps
`endif
   );

`include "ldpcDecFuncs.vh"

  localparam LDEC_DRIWVB = `LDEC_DEC_RAM_IP_WIDTH*`LDEC_VAR_BITS;
  localparam VIW_HI = ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH)-1;
  localparam COL_HI = numBits(`LDEC_NCOLS_RIGHT)-1;
  localparam Z1_HI  = numBits(`LDEC_Z_MAX-1)-1;
  
  // Signals for the Cell info memory.
  reg [numBits(`LDEC_NCELLS-1)-1:0]            cellLoadIdx;
  reg                                          cellAccessActive;
  wire [`LDEC_CELL_RAM_A-1:0]                  cellAddrA;
  wire [`LDEC_CELL_RAM_A-1:0]                  cellAddrB;
  wire                                         cellRamEnA;
  wire                                         cellRamEnB;
  wire [`LDEC_CELL_RAM_W-1:0]                  cellRdDataA;
  wire [`LDEC_CELL_RAM_W-1:0]                  cellRdDataB;
  wire [numBits(`LDEC_NCELLS)-1:0]             numMacroCellsRom;
  
  // Signals for the varMetric RAMs            
  wire                                         vmIpRamClkEn;
  wire                                         vmDecRamClkEn;
  wire                                         vmSelA;
  wire                                         vmSelB;
  wire [`LDEC_VM_RAM_W-1:0]                    vmIpData;
  reg [VIW_HI:0]                               vmIpWe;
  wire [`LDEC_VM_RAM_W-1:0]                    varMetricRd;

  // Other
  wire                                         notBwdStall;
  wire                                         notBwdStall_1;
  wire                                         notFwdStall_1;
  
  // Local version of outputs
  
  reg [COL_HI:0]                              fwdCol_1;
  reg [Z1_HI:0]                               fwdCycShift_1;
  reg [Z1_HI:0]                               fwdCycShift;
  reg [`LDEC_FLAG_BITS-1:0]                   fwdFlag_1;
  reg [COL_HI:0]                              fwdCol;
  reg [`LDEC_FLAG_BITS-1:0]                   fwdFlag;
  reg [COL_HI:0]                              bwdCol_1;
  reg [Z1_HI:0]                               bwdCycShift_1;
  reg [`LDEC_FLAG_BITS-1:0]                   bwdFlag_1;
  reg [COL_HI:0]                              bwdCol;
  reg [`LDEC_FLAG_BITS-1:0]                   bwdFlag;

  // Pipeline delay signals
  reg                                         notBwdStallD1;
  reg [`LDEC_VM_RAM_A-1:0]                    bwdColD1;
  reg                                         vmSelBD1;

  assign vmIpRamClkEn  = ipClkEn | disabling;
  assign vmDecRamClkEn = decodeActive_1 | disabling;
  assign vmRam2ClkEnOut = (blkPingPongS == 1'b0) ? vmIpRamClkEn : vmDecRamClkEn;
  assign vmRam1ClkEnOut = (blkPingPongS == 1'b1) ? vmIpRamClkEn : vmDecRamClkEn;
  
  assign cellClkEnOut = cellAccess | cellAccessActive | decodeActive_2 | disabling;
  // Hard Decision RAM is ping ponged and clock is a superset of enables
  // to avoid clock muxing.
  assign hdRamClkEnOut = decodeActive | opBufferSel | disabling;
  
  //---------------------------------------------------------------------------
  // CellInfo Memory
  //---------------------------------------------------------------------------

  // This is loaded by the host processor via the Regs block && read throughout
  // decode using two read ports. With around 88 cells && 15 bits per cell,
  // this is around 1300 bits && worth implementing as a DP memory.

  assign cellAddrB  = bwdCellIdx;
  assign cellAddrA = (cellAccess == 1'b1) ? cellLoadIdx : fwdCellIdx;
  assign cellRamEnA = cellAccess | (enable & !holdFwdCell_1);
  assign cellRamEnB  = cellAccess | (enable & !holdBwdCell_1) | bwdCellPrime;
               
  // Address generator when loading Cell Info. A cellLoad bit is set in a
  // control register to enable load. Writes || reads to CELL_INFO_ADDR
  // are signalled with cellWrite/Read and data is passed on CellWrData and
  // cellRdData. Reading is registered. If a spare cycle is inserted after
  // setting cellAccess begin address 0 contents will be available for
  // reading at `LDEC_CODE_ADDR. Subsequent reads should always have a spare
  // cycle inserted to allow the fetch to take place.
`ifndef LDEC_USE_CELL_ROM
  reg [numBits(`LDEC_NCELLS)-1:0]  numMacroCellsRam;  
  always @(posedge(cellClk) `LDEC_RESET_STR)
  begin : pCellLoadIdx
    reg  [numBits(`LDEC_NCELLS-1)-1:0] cellLoadIdxV;
    reg                                latchRdDataV;
    if (nReset == 1'b0) begin
      numMacroCellsRam <= 0;
      cellLoadIdx <= 0;
      cellAccessActive <= 1'b0;
      cellRdDataOut <= '0;
      latchRdDataV = 1'b0;
    end else begin
      if (cellAccess == 1'b0) begin
        cellLoadIdx <= 0;
        // The active sig extends the clock to allow the sync clr.
        cellAccessActive <= 1'b0;       
      end
      else begin
        cellAccessActive <= 1'b1;       
        if (cellRead == 1'b1 || cellWrite == 1'b1) begin
          cellLoadIdxV = cellLoadIdx;
          if (cellLoadIdxV < `LDEC_NCELLS-1) begin
            cellLoadIdxV = cellLoadIdx + 1'd1;
          end
          cellLoadIdx <= cellLoadIdxV;
          // Capture the number of cells written
          if (cellWrite == 1'b1) begin
            numMacroCellsRam <= cellLoadIdx+1'd1;
          end
        end
        // Read data is registered because putting RAM output directly onto a
        // peripheral read bus will often cause timing issues.
        if (latchRdDataV == 1'b1) begin
          cellRdDataOut <= cellRdDataA;
        end
        latchRdDataV = cellRead;
      end
    end        
  end //pCellLoadIdx
`else
  wire cellAccessActiveDummy;
  assign cellAccessActiveDummy = 1'b0;
  always @(*)
  begin
    //assign numMacroCellsRam = 0;
   cellAccessActive = cellAccessActiveDummy;
   cellRdDataOut = `LDEC_PAD(1'b0, `LDEC_CELL_RAM_W-1);
   cellLoadIdx = `LDEC_PAD(1'b0, numBits(`LDEC_NCELLS-1)-1);
  end
`endif
  
`ifdef LDEC_USE_CELL_ROM
      
  ldpcDecCellRom cellRom (
         .nReset         (nReset),
         .zEnum          (zEnum),
         .rEnum          (rEnum),
         .enable         (enable),
         .numMacroCells  (numMacroCellsRom),
         // Load/read via regs && read Fwd Cell info.
         .ckA            (cellClk),
         .selA           (cellRamEnA),
         .addrA          (cellAddrA),
         .dOutA          (cellRdDataA),
         // Bwd          Cell info (this port is read only).
         .ckB            (decClk),
         .selB           (cellRamEnB),
         .addrB          (cellAddrB),
         .dOutB          (cellRdDataB));
  assign numMacroCellsOut  = numMacroCellsRom;
      
`else

`ifndef LDEC_RAMS_AT_TOP
      
  ldpcDecCellRam CellMem
        (
         // Load/read via regs and read Fwd Cell info.
         .ckA   (cellClk),
         .selA  (cellRamEnA),
         .addrA (cellAddrA),
         .dInA  (cellWrData),
         .weA   (cellWrite),
         .dOutA (cellRdDataA),
         // Bwd Cell info (this port is read only).
         .ckB   (decClk),
         .selB  (cellRamEnB),
         .addrB (cellAddrB),
         .dInB  ('0),
         .weB   (1'b0),
         .dOutB (cellRdDataB));
  assign numMacroCellsOut = numMacroCellsRam;
      
`else
  
  assign cellRamIps = {cellRamEnA, cellRamEnB, cellWrite, cellAddrA,
                       cellAddrB, cellWrData};
  assign {cellRdDataA, cellRdDataB} = cellRamOps;
  assign numMacroCellsOut = numMacroCellsRam;

`endif
`endif

  // Pick sections out of the cell read data.

  always @(cellRdDataA)
  begin : pCellRdA
    
    parameter cycBase = `LDEC_FLAG_BITS;
    parameter colBase = cycBase + numBits(`LDEC_Z_MAX-1);
    parameter colTop = colBase + numBits(`LDEC_NCOLS-1) - 1;
    
    fwdCol_1 = cellRdDataA[colTop:colBase];
    fwdCycShift_1 = cellRdDataA[colBase-1:cycBase];
    fwdFlag_1 = cellRdDataA[`LDEC_FLAG_BITS-1:0];
  end //pCellRdA
  
  always @(cellRdDataB)
  begin : pCellRdB
    
    parameter cycBase = `LDEC_FLAG_BITS;
    parameter colBase = cycBase + numBits(`LDEC_Z_MAX-1);
    parameter colTop = colBase + numBits(`LDEC_NCOLS-1) - 1;
    
    bwdCol_1 = cellRdDataB[colTop:colBase];
    bwdCycShift_1 = cellRdDataB[colBase-1:cycBase];
    bwdFlag_1 = cellRdDataB[`LDEC_FLAG_BITS-1:0];
  end //pCellRdB
  
  // Retime to T==0
  always @(posedge(decClk) `LDEC_RESET_STR) begin

    if (nReset == 1'b0) begin
      fwdFlag <= `LDEC_PAD(1'b0, `LDEC_FLAG_BITS-1);
      bwdFlag <= `LDEC_PAD(1'b0, `LDEC_FLAG_BITS-1);
      bwdCol <= `LDEC_PAD(1'b0, COL_HI);
      fwdCol <= `LDEC_PAD(1'b0, COL_HI);
      fwdCycShift <= `LDEC_PAD(1'b0, Z1_HI);
    end else begin
      fwdFlag <= fwdFlag_1;
      bwdFlag <= bwdFlag_1;
      bwdCol <= bwdCol_1;
      fwdCol <= fwdCol_1;
      fwdCycShift <= fwdCycShift_1;
    end
  end //pCellReTime
    
  assign fwdColOut_1 = fwdCol_1;
  assign fwdColOut = fwdCol;
  assign fwdCycShiftOut_1 = fwdCycShift_1;
  assign fwdCycShiftOut = fwdCycShift;
  assign fwdFlagOut = fwdFlag;
  assign fwdFlagOut_1 = fwdFlag_1;
  assign bwdColOut_1 = bwdCol_1;
  assign bwdColOut = bwdCol;
  assign bwdCycShiftOut_1 = bwdCycShift_1;
  assign bwdFlagOut = bwdFlag;
  assign bwdFlagOut_1 = bwdFlag_1;

  //---------------------------------------------------------------------------
  // varMetrics RAM
  //---------------------------------------------------------------------------

  assign notBwdStall   = !bwdStall;
  assign notBwdStall_1 = !bwdStall_1;
  assign notFwdStall_1 = !fwdStall_1;

  assign vmSelA = notFwdStall_1 & !fwdNoOp_1;
  assign vmSelB = notBwdStall & !bwdNoOp;

  generate
    if (`LDEC_PIPE_X == 1) begin: gPipeX
      always @(posedge(cellClk) `LDEC_RESET_STR)
      begin : pPipe
        if (nReset == 1'b0) begin
          notBwdStallD1 <= 1'b0;
          bwdColD1 <= `LDEC_PAD(1'b0, COL_HI);
          vmSelBD1 <= 1'b0;      
        end else begin
          notBwdStallD1 <= notBwdStall;
          bwdColD1 <= bwdCol;
          vmSelBD1 <= vmSelB;
          // Clear the pipeline when starting a new block
          if (newBlk == 1'b1) begin
            notBwdStallD1 <= 1'b0;
            vmSelBD1 <= 1'b0;
          end
        end
      end //pPipe
    end else begin
      always @(*)
      begin
        notBwdStallD1 = notBwdStall;
        bwdColD1 = bwdCol;
        vmSelBD1 = vmSelB;
      end
    end
  endgenerate //  gPipeX

  // The VM RAMs reside in here, unless moved to the top with `LDEC_RAMS_AT_TOP.
  ldpcDecVmRamMux vmRamMux
    (
     .pp       (blkPingPongS),
     // These are direct clocks to the two RAMS inside vmDecRam, they are not
     // 'swapped' by pingPong like the other signals. Instead the clock enables
     // are 'swapped' in ldpcDecVmMem.
     .ck1      (vmRam1Clk),
     .ck2      (vmRam2Clk),
     // Buffer 1 is used during decoding.
     // Fwd   processing reads.
     .selA1    (vmSelA),
     .addrA1   (fwdCol_1),
     .dOutA1   (varMetricRd),
     // Bwd processing writes.
     .addrB1   (bwdColD1),
     .dInB1    (varMetricWr),
     .weB1     (vmSelBD1),
     // Buffer 2 is used for loading of input data.
     // IP Buffer Writes.
     .addrB2   (wrAddr),
     .dInB2    (vmIpData),   
     .weB2     (vmIpWe),
     // IP Buffer Reads (for repeat combining).
     .selA2    (rdEnable),
     .addrA2   (rdAddr),
     .dOutA2   (rdData)
     // port to bring RAMS out to the top
`ifdef LDEC_RAMS_AT_TOP
    ,
     .vmRamOps (vmRamOps),
     .vmRamIps (vmRamIps)
`endif
     );
  
  // The VmIp RAM takes its input from the wrData bus. wrData is steered in
  // a hardcoded way so that it is repeated across the width of the RAM. The
  // vmIpWe (write Enable) is the only dynamic signal.
  generate
    genvar i;
    for (i=0; i<ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH); i=i+1)
    begin: vmIpMap
      assign vmIpData[(i+1)* LDEC_DRIWVB-1 : i* LDEC_DRIWVB] =
             wrData[ LDEC_DRIWVB-1 : 0];
    end
  endgenerate //  vmIpMap
  
  always @(wrOffset, wrEnable)
  begin : pVmIpWe
    vmIpWe = `LDEC_PAD(1'b0, VIW_HI);
    vmIpWe[wrOffset] = wrEnable;
  end //pVmIpWe

  // Use newly updated varMetric data when the bwd section;
  // processing the same col as the fwd section. This happens when
  // `LDEC_BWD_FWD_LATENCY == 0. When `LDEC_BWD_FWD_LATENCY>0 fwd processing will
  // be staggered `LDEC_BWD_FWD_LATENCY cycles after bwd processing, allowing
  // data to be passed solely through the memory.
  
  assign varMetricRdOut = ((metricVeryFirstUse == 1'b0 && fwdCol == bwdColD1 &&
                            `LDEC_BWD_FWD_LATENCY == 0)) ? varMetricWr : varMetricRd;
  
  //---------------------------------------------------------------------------
  // Hard Decision RAM
  //---------------------------------------------------------------------------

  // This is ping-ponged to allow output data to be fetched during the following
  // decode. Furthermore it is dual ported so that the bwd part can read previous
  // estimates and test if there has been a bit inversion. The first time a
  // metric is used in a decode the data is !read so no initialisation is required.

  ldpcDecHdRamMux hdRamMux 
    (
     .pp       (blkPingPongE),
     .ck       (hdRamClk),
     .freeck   (freeClk), // RW modification
     .nReset   (nReset),
     //Bwd processing reads.
     .selA     (notBwdStall_1),
     .addrA    (bwdCol_1),
     .dOutA    (hardDecisionsRdOut),
     // Bwd processing writes (same as varMetricsSgn)
     .addrB    (bwdColD1),
     .dInB     (hardDecisionsWr),
     .weB      (notBwdStallD1),
     // Output interface maps to port A of the 'other' bank.
     .selA2    (opBufferSel),
     .addrA2   (opBufferAddr),
     .dOutA2   (opBufferDataOut)
     // port to bring RAMS out to the top
`ifdef LDEC_RAMS_AT_TOP
    ,
     .hdRamOps (hdRamOps),
     .hdRamIps (hdRamIps)
`endif
     
     );

endmodule

    
     
    
