//------------------------------------------------------------------------------
// ldpcDecVrRamMux.v
// 
// Description:
//   RAM wrapper for the Variable Metrix Decode RAM. This is a double buffered
//   RAM with pp (Ping Pong) controlling the switching of the buffers. In
//   addition, each RAM is dual port.
//
//   One RAM port is always used for write && the other for read.
//
// 8 Jun 2010 M.Rumsey. Created.
// 4 Oct 2010 M.Rumsey. Now incorporates the input buffer.
// 5 Apr 2011 M.Rumsey. Alternate version created (see header).
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//
//-----------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecVmRamMux
  (  
     input                       pp,
     input                       ck1,
     input                       ck2,
     // Buffer 1 is used during decoding.
     // Fwd processing reads.    
     input                       selA1,
     input [`LDEC_VM_RAM_A-1:0]  addrA1,
     output [`LDEC_VM_RAM_W-1:0] dOutA1,
     // Bwd processing writes.
     input [`LDEC_VM_RAM_A-1:0]  addrB1,
     input [`LDEC_VM_RAM_W-1:0]  dInB1,
     input                       weB1,
     // Buffer 2 is used for loading of input data.
     // A port is used for readback to allow combining of repeats.
     input                       selA2,
     input [`LDEC_VM_RAM_A-1:0]  addrA2,
     output [`LDEC_VM_RAM_W-1:0] dOutA2,
     // IP Buffer Writes.       
     input [`LDEC_VM_RAM_A-1:0]  addrB2,
     input [`LDEC_VM_RAM_W-1:0]  dInB2,
     input [`LDEC_VM_RAM_WE-1:0] weB2
     // port to allow RAMs to be moved to top level.
`ifdef LDEC_RAMS_AT_TOP
     ,
     input [`LDEC_VM_RAM_O-1:0]  vmRamOps,
     output [`LDEC_VM_RAM_I-1:0] vmRamIps
`endif
     );
  

`include "ldpcDecFuncs.vh"

  wire                           selAX;
  wire [`LDEC_VM_RAM_A-1:0]      addrAX;

  wire [`LDEC_VM_RAM_A-1:0]      addrBX;
  wire [`LDEC_VM_RAM_W-1:0]      dInBX;
  wire [`LDEC_VM_RAM_WE-1:0]     weBX;

  wire                           selAY;
  wire [`LDEC_VM_RAM_A-1:0]      addrAY;

  wire [`LDEC_VM_RAM_A-1:0]      addrBY;
  wire [`LDEC_VM_RAM_W-1:0]      dInBY;
  wire [`LDEC_VM_RAM_WE-1:0]     weBY;

  wire [`LDEC_VM_RAM_W-1:0]      dOutAX;
  wire [`LDEC_VM_RAM_W-1:0]      dOutAY;

  //---------------------------------------------------------------------------
  // Muxing of inputs
  //---------------------------------------------------------------------------

  // X ram uses the 1'b1 inputs when pp==0, otherwise the '2' inputs.
  assign selAX  = (pp == 1'b0 ) ? selA1  : selA2;
  assign addrAX = (pp == 1'b0 ) ? addrA1 : addrA2;
  assign addrBX = (pp == 1'b0 ) ? addrB1 : addrB2;
  assign dInBX  = (pp == 1'b0 ) ? dInB1  : dInB2;
  // The B1 WE is for the whole write word but we assume the RAM write;
  // sectioned for the benefit of the B2 port so expand accordingly.
  assign weBX = (pp == 1'b0 ) ? {`LDEC_VM_RAM_WE{weB1}} : weB2;
  
  // Y ram uses the 1'b1 inputs when pp==1, otherwise the '2' inputs.
  assign selAY  = (pp == 1'b1 ) ? selA1  : selA2;
  assign addrAY = (pp == 1'b1 ) ? addrA1 : addrA2;
  assign addrBY = (pp == 1'b1 ) ? addrB1 : addrB2;
  assign dInBY  = (pp == 1'b1 ) ? dInB1  : dInB2;
  assign weBY = (pp == 1'b1 ) ? {`LDEC_VM_RAM_WE{weB1}} : weB2;

  //---------------------------------------------------------------------------
  // Muxing of Outputs
  //---------------------------------------------------------------------------

   assign dOutA1 = (pp == 1'b0 ) ? dOutAX : dOutAY;
   assign dOutA2 = (pp == 1'b1 ) ? dOutAX : dOutAY;
  
  //---------------------------------------------------------------------------
  // The RAM instances
  //---------------------------------------------------------------------------

`ifndef LDEC_RAMS_AT_TOP

      ldpcDecVmRam vmRamX (
                           .ck    (ck1),
                           // A port always read
                           .selA  (selAX),
                           .addrA (addrAX),
                           .dOutA (dOutAX),
                           // B port always write
                           .addrB (addrBX),
                           .dInB  (dInBX),
                           .weB   (weBX));
      
      ldpcDecVmRam vmRamY (
                           .ck    (ck2),
                           // A port always read
                           .selA  (selAY),
                           .addrA (addrAY),
                           .dOutA (dOutAY),
                           // B port always write
                           .addrB (addrBY),
                           .dInB  (dInBY),
                           .weB   (weBY));

`else

      assign vmRamIps  = {selAX, selAY ,
                          weBX, addrAX, addrBX, dInBX,
                          weBY, addrAY, addrBY, dInBY};
      assign {dOutAX, dOutAY} = vmRamOps;

`endif

endmodule

    
    

