//------------------------------------------------------------------------------
// ldcpDecVrFifo.sv
// 
// Description:
//   Variable Response FIFO. This version uses dual port memory, which gives
//   best performance but uses more area. `FPGA would generally use this verson.
//   The FIFO is designed assuming that the RAM output is registered. This
//   registering is in the ldpcDecBwd block to encourage layout to group it
//   with the datapath processing that it relates to.
//
// Inputs:
//   en          : Enable both reading and writing
//   wrRowStart  : Write data for the 1st entry of row is on dIn.
//   wrStall     : Blocks the write. Important when the FIFO is full to stop
//                 overwriting of existing data.
//   rdRowStart  : Read data required on dOut in 2 cycles time.
//
// 1 June 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//
//-----------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecVrFifo
( 
  input                       nReset,
  input                       clk,
  input                       wrRowStart,
  input                       wrStall,
  input                       rdRowStart_2,
  input                       rdStall_2,
  input [`LDEC_VR_RAM_W-1:0]  dIn,
  output [`LDEC_VR_RAM_W-1:0] dOut
`ifdef LDEC_RAMS_AT_TOP
  ,
  input [`LDEC_VR_RAM_O-1:0]  vrRamOps,
  output [`LDEC_VR_RAM_I-1:0] vrRamIps
`endif
  );        

`include "ldpcDecFuncs.vh"

  localparam RAM_HI = `LDEC_VR_RAM_A-1;
  localparam [RAM_HI:0] VR_RAM_D1 = `LDEC_VR_RAM_D-1;

  wire                      wrEn;
  wire                      rdEn;
  reg  [RAM_HI:0]           rdAddr;
  reg  [RAM_HI:0]           wrAddr;
  reg  [RAM_HI:0]           rdAddrReg;
  reg  [RAM_HI:0]           wrAddrReg;
  wire [`LDEC_VR_RAM_W-1:0] ramRdData;

  assign wrEn = !wrStall;
  assign rdEn = !rdStall_2;
  
  // Address generation.
  
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pAddrReg
    if (nReset == 1'b0) begin
      rdAddrReg <= `LDEC_PAD(1'b0, RAM_HI);
      wrAddrReg <= `LDEC_PAD(1'b0, RAM_HI);
    end else begin
      if (rdStall_2 == 1'b0) begin
        if (rdAddr < VR_RAM_D1) begin
          rdAddrReg <= rdAddr + `LDEC_PAD(1'b1, RAM_HI);
        end
      end        
      if (wrAddr < VR_RAM_D1 && wrEn == 1'b1) begin
        wrAddrReg <= wrAddr + `LDEC_PAD(1'b1, RAM_HI);
      end
    end
  end //pAddrReg

  // The address presented to the RAM must be combinatorially derived
  // to avoid having to bring the rdRowStart signal yet another cycle
  // earlier (it is already T-2).
  always @(rdAddrReg, rdRowStart_2)
  begin : pRdAddr
    if (rdRowStart_2 == 1'b1) begin
      rdAddr = `LDEC_PAD(1'b0, RAM_HI);
    end
    else begin
      rdAddr = rdAddrReg;
    end
  end //pRdAddr

  always @(wrAddrReg, wrRowStart)
  begin : pWrAddr
    if (wrRowStart == 1'b1) begin
      wrAddr = `LDEC_PAD(1'b0, RAM_HI);
    end
    else begin
      wrAddr = wrAddrReg;
    end
  end //pWrAddr
  
  // The RAM

`ifndef LDEC_RAMS_AT_TOP
  
  ldpcDecVrRam vrRam
    (
     .ck    (clk),
     // Write
     .addrA (wrAddr),
     .dInA  (dIn),
     .weA   (wrEn),
     // Read
     .selB  (rdEn),
     .addrB (rdAddr),
     .dOutB (ramRdData));
  
`else
  
  assign vrRamIps  = {wrEn, rdEn, wrAddr, dIn, rdAddr};
  assign ramRdData = vrRamOps;

`endif
  
  assign dOut  = ramRdData;
  
endmodule

    
    

