//------------------------------------------------------------------------------
// ldcpDecVrFifo1p.v
// 
// Description:
//   Variable Response FIFO. Requires a read-before-write single port RAM. Depth
//   of the FIFO is 2 more than the RAM size due to registering && the
//   read-before-write operation. The registering is in the ldpcDecBwd block to
//   encourage layout to group it with the datapath processing that it relates to.
//
//   en          : Enable FIFO. Can only be dropped at end of block.
//   stall       : Stall read && write. Note that in 1 port mode, fwd/bwd
//                 stalls due to PIPE ABC are in step, while `PIPE_C &
//                 `BWD_FWD_LATENCY add stalls but also stagger fwd later than bwd.
//                 Can only stall when both fwd/bwd are stalled, which means
//                 number of stalls is NUM_PIPES_ABC only.
//   wrRowStart  : Write data for the 1st entry of row is on dIn.
//   wrStall     : May be raised when there is a write stall at the end of a row
//                 if all rowshave stalls at the end (these are 'pipeline' stalls).
//   rdRowStart  : Read data required on dOut in 2 cycles time. This means 2
//                 unstalled cycles.
//
// 1 June 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//
//-----------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecVrFifo1p
( 
  input                        nReset,
  input                        clk,
  input                        en,
  input                        wrRowStart,
  input                        rdRowStart_2,
  input [`LDEC_VR_RAM_W-1:0]   dIn,
  output [`LDEC_VR_RAM_W-1:0]  dOut
`ifdef LDEC_RAMS_AT_TOP
  ,
  input [`LDEC_VR1_RAM_O-1:0]  vrRamOps,
  output [`LDEC_VR1_RAM_I-1:0] vrRamIps
`endif
  );    

`include "ldpcDecFuncs.vh"

  wire                                rdWrEn;
  reg [numBits(`LDEC_VR_RAM_D-1)-1:0] addr;
  reg [numBits(`LDEC_VR_RAM_D-1)-1:0] addrLast;
  reg [numBits(`LDEC_VR_RAM_D-1)-1:0] wrapAddr;
  reg [numBits(`LDEC_VR_RAM_D-1)-1:0] rowStartAddr;
  reg [numBits(`LDEC_VR_RAM_D)-1:0]   addrReg;
  wire [`LDEC_VR_RAM_W-1:0]           dRamOut;
  reg                                 firstReadSeen;
  reg                                 firstWriteSeen;

  assign rdWrEn  = en;
 
  // Address generation. This gets set back to start of row on rdRowStart where
  // start of row is defined as the address registered when wrRowStart was high.

  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pAddrReg
    
    if (nReset == 1'b0) begin
      addrReg <= 0;
      rowStartAddr <= 0;
      wrapAddr <= `LDEC_VR_RAM_D-1;
      firstReadSeen <= 1'b0;
      firstWriteSeen <= 1'b0;
    end else begin
      if (en == 1'b0) begin
        // synchronous reset at end of block
        addrReg <= 0;
        rowStartAddr <= 0;
        wrapAddr <= `LDEC_VR_RAM_D-1;
        firstReadSeen <= 1'b0;
        firstWriteSeen <= 1'b0;
      end
      else begin
        addrLast <= addr;
        if (wrRowStart == 1'b1) begin
          rowStartAddr <= addr;
          firstWriteSeen <= 1'b1;
        end
        if (addr == wrapAddr) begin
          addrReg <= 0;
        end
        else if (firstWriteSeen == 1'b1 || wrRowStart == 1'b1) begin
          addrReg <= addr + 1;
        end
        if ( (rdRowStart_2 && !firstReadSeen) == 1'b1) begin
          firstReadSeen <= 1'b1;
          wrapAddr <= addrLast;
        end
      end
    end
    
  end //pAddrReg

  // The address presented to the RAM must be combinatorially derived
  // to avoid having to bring the rdRowStart signal yet another cycle
  // earlier (it is already T-2).
  always @(addrReg, rowStartAddr, rdRowStart_2)
  begin : pAddr
    if (rdRowStart_2 == 1'b1) begin
      addr <= rowStartAddr;
    end
    else begin
      addr <= addrReg;
    end
  end //pAddr
  
  // The RAM

`ifndef LDEC_RAMS_AT_TOP
  ldpcDecVr1pRam vrRam
    (
     .ck   (clk),
     .sel  (rdWrEn),
     .addr (addr),
     .dIn  (dIn),
     .we   (rdWrEn),
     .dOut (dRamOut));
`else
  assign vrRamIps  = {rdWrEn, addr, dIn};
  assign dRamOut = vrRamOps;
`endif

  assign dOut  = dRamOut;
  
endmodule

    
    

