/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module rx_bd_ctrl_data_parser
(
  /* system */
  input  wire        rst_n,
  input  wire        clk,
  
  input  wire        enable,
  output wire        done,
 
  input  wire [19:0] length,
  input  wire        confirmed,
  
  /* byte stream */
  output wire        bd_ready,
  input  wire [ 7:0] bd_data,
  input  wire        bd_last,
  input  wire        bd_valid,
  
  /* service */
  output reg         service_captured,
  output reg  [15:0] service,
  
  /* mpif */
  input  wire        mpif_ready,
  output reg  [ 7:0] mpif_data,
  output reg         mpif_last,
  output reg         mpif_valid
  
);
  
  /*****************************************************************************
  * note: at first glance, having 'bd_last' and 'length' parameter might seem 
  *       redundant.
  *  1) bd_last indicates the last byte of the PSDU (length common to all user)
  *  2) while 'length' is the length provided to the MAC depednign of the user payload
  *     it is mandatory to exactly pushed the number of bytes indicated by 'length'
  *
  *  in SU, bd_last should correspond to the last byte of the sequence of 'length byte'
  *  in MU, 1) our user can receive a payload ('length') smaller than the corresponding data field ('bd_last')
  *         2) protect from deadlock if 'length' is greater than the corresponding data field
  *     
  *****************************************************************************/
  localparam     IDLE=3'd0,
                 SERVICE0=3'd1,
                 SERVICE1=3'd2,
                 DATA=3'd3,
                 DONE=3'd4;
                 
  reg  [ 2:0]    state;
  reg            mpif_done;
  reg  [ 2:0]    wr_ptr, rd_ptr;
  wire [ 2:0]    n_wr_ptr, n_rd_ptr;
  reg  [ 7:0]    d0, d1, d2, d3;
  reg  [19:0]    count;
  wire [19:0]    n_count;
  
  wire           fifo_full, fifo_empty;
  wire [ 7:0]    bd_scrambler_data;
  wire [ 6:0]    n_scrambler_state;
  reg  [ 6:0]    scrambler_state;
  
  /* service capture and 4 bytes fifo */
 
  assign fifo_full  = wr_ptr == {~rd_ptr[2],rd_ptr[1:0]};
  assign fifo_empty = wr_ptr == rd_ptr;
 
  assign bd_ready   = state==SERVICE0 || 
                      state==SERVICE1 || 
                      state==DATA  && !fifo_full || 
                      state==DONE; /* pipeline flush */
  
  assign n_wr_ptr = wr_ptr +  3'd1;
  assign n_rd_ptr = rd_ptr +  3'd1;
  
  assign n_count  = count  + 20'd1;
  
  
  assign bd_scrambler_data =
   {bd_data[7]^scrambler_state[4]^scrambler_state[1]^scrambler_state[0],
    bd_data[6]^scrambler_state[2]^scrambler_state[5]^scrambler_state[1],
    bd_data[5]^scrambler_state[3]^scrambler_state[6]^scrambler_state[2],
    bd_data[4]^scrambler_state[3]^scrambler_state[0],
    bd_data[3]^scrambler_state[4]^scrambler_state[1],
    bd_data[2]^scrambler_state[2]^scrambler_state[5],
    bd_data[1]^scrambler_state[3]^scrambler_state[6],
    bd_data[0]^scrambler_state[0]};
  
  assign n_scrambler_state =
   {scrambler_state[5]^scrambler_state[2],
    scrambler_state[4]^scrambler_state[1],
    scrambler_state[3]^scrambler_state[0],
    scrambler_state[3]^scrambler_state[6]^scrambler_state[2],
    scrambler_state[2]^scrambler_state[5]^scrambler_state[1],
    scrambler_state[4]^scrambler_state[1]^scrambler_state[0],
    scrambler_state[0]^scrambler_state[6]};

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      state            <= IDLE;
      scrambler_state  <= 7'b0;
      service          <= 16'b0;
      service_captured <= 1'b0;
      d0               <= 9'd0;
      d1               <= 9'd0;
      d2               <= 9'd0;
      d3               <= 9'd0;
      wr_ptr           <= 3'd0;
    end
    else if(!enable)
    begin
      state            <= IDLE;
      scrambler_state  <= 7'b0;
      service          <= 16'b0;
      service_captured <= 1'b0;
      d0               <= 9'd0;
      d1               <= 9'd0;
      d2               <= 9'd0;
      d3               <= 9'd0;
      wr_ptr           <= 3'd0;
    end
    else
    begin
      case(state)                       
        
        IDLE:                           
        begin                           
          state           <= SERVICE0;  
        end                             
       
        SERVICE0:                       
        begin                           
          if(bd_ready && bd_valid)
          begin
            scrambler_state  <= {bd_data[2],bd_data[3],bd_data[4],bd_data[5],
                                 bd_data[6],bd_data[0]^bd_data[3],bd_data[1]^bd_data[4]};
            service[ 7:0]    <= bd_data;   
            state            <= SERVICE1; 
          end 
        end                             
       
        SERVICE1:                       
        begin                           
          if(bd_ready && bd_valid)
          begin
            scrambler_state  <= n_scrambler_state;
            service_captured <= 1'b1;      
            service[15:8]    <= bd_scrambler_data;   
            state            <= DATA; 
          end     
        end                             
       
        DATA:                           
        begin                           
          if(bd_ready && bd_valid)
          begin
            scrambler_state  <= n_scrambler_state;
            /* 4 bytes fifo, needed to retain any nonht data during the BD_LDATA0_VHTSIGA1 state */
            case(wr_ptr[1:0])
              2'd0:    d0 <= bd_scrambler_data;
              2'd1:    d1 <= bd_scrambler_data;
              2'd2:    d2 <= bd_scrambler_data;
              default: d3 <= bd_scrambler_data;
            endcase
            wr_ptr <= n_wr_ptr;
            if(bd_last) 
              state <= DONE;
          end
        end
       
        default: ;
      
      endcase
    end
  end
  
  /* output slot */
  assign done = mpif_done;
  
  wire n_mpif_last;
  assign n_mpif_last = n_count == length;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      mpif_data   <= 8'b0;
      mpif_last   <= 8'b0;
      mpif_valid  <= 1'b0; 
      mpif_done   <= 1'b0;
      count       <= 20'd0;
      rd_ptr      <= 3'd0;  
    end
    else if(!enable)
    begin
      mpif_data   <= 8'b0;
      mpif_last   <= 8'b0;
      mpif_valid  <= 1'b0; 
      mpif_done   <= 1'b0;
      count       <= 20'd0;  
      rd_ptr      <= 3'd0;  
    end
    else
    begin
        
      if(!mpif_done && confirmed && service_captured)
      begin
        /* still data to push towards mac */
        if(mpif_ready && mpif_valid && mpif_last)
        begin
          mpif_done  <= 1'b1;
          mpif_data  <= 8'b0;
          mpif_last  <= 1'b0;
          mpif_valid <= 1'b0;
        end
        else
        begin
        
          /* slot consummed */
          if(mpif_ready)
            mpif_valid <= 1'b0;
      
          /* slot read */
          if(mpif_ready || !mpif_valid)
          begin
            /* slot is free */
            if(state==DONE && fifo_empty)
            begin
              /* no more byte from the data field, complete 'length' with zero bytes */
              mpif_data  <= 8'd0;
              mpif_last  <= n_mpif_last;
              mpif_valid <= 1'b1;
              count      <= n_count;
            end
            else
            begin
              /* still bytes from data field */
              if(!fifo_empty)
              begin
                case(rd_ptr[1:0])
                  2'd0:    mpif_data <= d0;
                  2'd1:    mpif_data <= d1;
                  2'd2:    mpif_data <= d2;
                  default: mpif_data <= d3;
                endcase
                
                mpif_last  <= n_mpif_last;
                mpif_valid <= 1'b1;
                
                rd_ptr     <= n_rd_ptr;
                count      <= n_count;
              end
            end
          end
        end
      end 
    end
  end
  
endmodule
`default_nettype wire
