/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module rx_bd_ctrl_fsm
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire         rst_n,
  input  wire         clk,
  
  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire         enable,

  /*****************************************************************************
  * conf
  *****************************************************************************/
  input  wire [ 2:0]  mdmcfg_conf_primary, 
  input  wire [ 1:0]  mdmcfg_conf_bw, 
   
   
  /*****************************************************************************
  * data path
  *****************************************************************************/
  /* buffer pointers and attributes */
  input  wire         bdfd_wrptr,
  output reg          bdfd_rdptr,
  input  wire [ 2:0]  bdfd_ch0_wrptr,
  output reg  [ 2:0]  bdfd_ch0_rdptr,
  input  wire [ 2:0]  bdfd_ch1_wrptr,
  output reg  [ 2:0]  bdfd_ch1_rdptr,
  input  wire         bdfd_qbpsk,
  
  /* symbol */
  output wire [ 4:0]  bd_symbol,
  
  /* datapath symbol control */
  output reg          bd_bcc_enable,
  output reg          bd_ldpc_enable,
  output reg  [ 1:0]  bd_ch0_sel,            
  output reg          bd_ch0_symbol_start,
  input  wire         bd_ch0_symbol_done,
  output reg  [ 1:0]  bd_ch1_sel,            
  output reg          bd_ch1_symbol_start,
  input  wire         bd_ch1_symbol_done,
  output reg          bd_field_start,
  
  output reg          bd_fec,             
  output reg          bd_dcm,             
  output reg          bd_sigen,             
  output reg          bd_qbpsk,           
  output reg  [ 1:0]  bd_nsd,             
  output reg          bd_ruen,             
  output reg  [ 2:0]  bd_rulen,             
  output reg          bd_nes,               
  output reg  [ 2:0]  bd_nss,               
  output reg  [ 2:0]  bd_nbpsc,           
  output reg          bd_stbc,              
  output reg  [ 1:0]  bd_cr, 
  output reg          bd_row_fifo_en,
  output reg  [23:0]  bd_field_length,    
  output reg          bd_field_stfix,     
  
  /* lsig */
  output reg          lsig_parser_enable,     
  input  wire         lsig_captured,     
  input  wire  [ 2:0] lsig_nbpsc,
  input  wire  [ 1:0] lsig_cr,
  input  wire  [ 1:0] lsig_length_mod3,
  input  wire  [11:0] lsig_length,
  output reg          lsig_ucpu_start,
  input  wire         lsig_ucpu_done,
  
  /* htsig */
  output reg          htsig_parser_enable,
  input  wire         htsig_captured,
  input  wire  [ 1:0] htsig_nsd,
  input  wire  [ 2:0] htsig_nss,
  input  wire  [ 2:0] htsig_nbpsc,
  input  wire  [ 1:0] htsig_cr,
  input  wire         htsig_fec,
  input  wire         htsig_stbc,
  input  wire  [15:0] htsig_length,
  output reg          htsig_ucpu_start,
  input  wire         htsig_ucpu_done,
  
  /* vhtsig */
  output reg          vhtsiga_parser_enable, 
  input  wire         vhtsiga_captured,
  output reg          vhtsigb_parser_enable,     
  input  wire         vhtsigb_captured,
  input  wire  [ 1:0] vhtsig_nsd,
  input  wire  [ 2:0] vhtsig_nss,
  input  wire         vhtsig_nes,
  input  wire  [ 2:0] vhtsig_nbpsc,
  input  wire  [ 1:0] vhtsig_cr,
  input  wire         vhtsig_fec,
  input  wire         vhtsig_stbc,
  output reg          vhtsig_ucpu_start,
  input  wire         vhtsig_ucpu_done,
  
  /* herlsig */
  output reg          herlsig_parser_enable,     
  input  wire         herlsig_captured,
  input  wire         herlsig_eq_lsig,
 
  /* hesig */
  output reg          hesiga2_is_qbpsk,
  output reg          hesiga_parser_enable,     
  input  wire         hesiga_captured,     
  
  output reg          hesigb_parser_enable,     
  input  wire         hesigb_parser_ch0_done,
  input  wire         hesigb_parser_ch1_done,
  
  input  wire  [ 2:0] hesig_bandwidth,
  input  wire         hesig_fec,
  input  wire  [ 2:0] hesig_nbpsc,
  input  wire  [ 1:0] hesig_cr,
  input  wire         hesig_dcm,
  input  wire  [ 2:0] hesig_rulen,
  input  wire         hesig_stbc,
  input  wire  [ 2:0] hesig_nsts,
  input  wire  [ 2:0] hesig_sigb_nbpsc,
  input  wire  [ 1:0] hesig_sigb_cr,
  input  wire         hesig_sigb_dcm,
  
  output reg          hesig_ucpu_start,
  input  wire         hesig_ucpu_done,
  
  /* data */
  output reg          data_parser_enable,
  output reg          data_confirmed,
  
  /* common parameters */
  input  wire  [23:0] npld,
  input  wire  [15:0] nsym,
  input  wire  [ 7:0] nsym_hesigb,
  
  /* diags */
  output wire  [ 9:0] diag
  
);
  /*****************************************************************************
  * declaration
  *****************************************************************************/
  localparam  HE_SU=4'd5,
              HE_MU=4'd6,
              HE_ER_SU=4'd7,
              HE_TB=4'd8;

  localparam  BD_IDLE=5'd0,
              BD_LSIG=5'd1,
              BD_LDATA0_VHTSIGA1=5'd2,
              BD_LDATA=5'd3,
              BD_HTSIG1=5'd4,
              BD_HTSIG2=5'd5,
              BD_HTDATA=5'd6,
              BD_VHTSIGA2=5'd7,
              BD_VHTSIGB=5'd8,
              BD_VHTDATA=5'd9,
              BD_HERLSIG=5'd10,
              BD_HESIGA1=5'd11,
              BD_HESIGA2=5'd12,
              BD_HEERSUSIGA2=5'd13,
              BD_HEERSUSIGA3=5'd14,
              BD_HEERSUSIGA4=5'd15,
              BD_HESIGB=5'd16,
              BD_HEDATA=5'd17;

  localparam  BD_WAIT_SYMBOL=3'd0,
              BD_WAIT_PARSER=3'd1,
              BD_WAIT_FIFO=3'd2,
              BD_WAIT_UCPU=3'd3,
              BD_REPLAY=3'd4,
              BD_ROW_FIFO=3'd5;

  localparam  CR_12=2'd0,
              CR_23=2'd1,
              CR_34=2'd2,
              CR_56=2'd3;  
 
  localparam  NBPSC_1=3'd0,
              NBPSC_2=3'd1,
              NBPSC_4=3'd2,
              NBPSC_6=3'd3,
              NBPSC_8=3'd4,
              NBPSC_10=3'd5;
  
  localparam  NSD_48=2'd0,
              NSD_52=2'd1,
              NSD_108=2'd2,
              NSD_234=2'd3;  

  wire        n_bdfd_rdptr;
  wire [ 2:0] n_bdfd_ch0_rdptr;
  wire [ 2:0] n_bdfd_ch1_rdptr;
  wire        fifo_empty;
  wire        fifo_ch0_empty;
  wire        fifo_ch1_empty;
  
  reg  [ 4:0] bd_symbol_state; 
  reg  [ 2:0] bd_step_state;   
  reg  [15:0] sym_index;       
  reg  [ 7:0] ch0_sym_index;       
  reg  [ 7:0] ch1_sym_index;       
  wire [15:0] n_sym_index;     
  wire [ 7:0] n_ch0_sym_index;     
  wire [ 7:0] n_ch1_sym_index;  
  reg         ch0_state;
  reg         ch1_state;   
  
  /*****************************************************************************
  * buffer managment
  *****************************************************************************/
  assign fifo_empty       = bdfd_rdptr==bdfd_wrptr;
  assign fifo_ch0_empty   = bdfd_ch0_rdptr==bdfd_ch0_wrptr;
  assign fifo_ch1_empty   = bdfd_ch1_rdptr==bdfd_ch1_wrptr;
  
  assign n_bdfd_rdptr     = ~bdfd_rdptr;
  assign n_bdfd_ch0_rdptr = bdfd_ch0_rdptr + 3'd1;
  assign n_bdfd_ch1_rdptr = bdfd_ch1_rdptr + 3'd1;
  
  assign n_ch0_sym_index  = ch0_sym_index  + 8'd1;
  assign n_ch1_sym_index  = ch1_sym_index  + 8'd1;
 
  /*****************************************************************************
  * fsm
  *****************************************************************************/
  /* he_mu channel content availability */
  reg  ch0_available,ch1_available;
  always @(*)
  begin
    if(hesig_bandwidth==2'd0)
    begin
      /* HEMU20, only ch0 channel content */
      ch0_available = 1'b1;
      ch1_available = 1'b0;
    end
    else if(mdmcfg_conf_bw==2'd0)
    begin
      /* HEMU>20 with CONF20 */
      if(!mdmcfg_conf_primary[0])
      begin
        /* only ch0 available */
        ch0_available = 1'b1;
        ch1_available = 1'b0;
      end
      else
      begin
        /* only ch1 available */
        ch0_available = 1'b0;
        ch1_available = 1'b1;
      end
    end
    else
    begin
      /* CONF>20 and HEMU>20, both channel contents available */
      ch0_available = 1'b1;
      ch1_available = 1'b1;
    end
  end
  
  assign bd_symbol   = bd_symbol_state;
  assign n_sym_index = sym_index + 16'd1;
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      bd_symbol_state            <= BD_IDLE;
      bd_step_state              <= BD_WAIT_FIFO;
      sym_index                  <= 16'd0;
      bdfd_rdptr                 <= 1'd0;
      bdfd_ch0_rdptr             <= 3'd0;
      bdfd_ch1_rdptr             <= 3'd0;
      bd_bcc_enable              <= 1'b0;        
      bd_ldpc_enable             <= 1'b0;        
      bd_ch0_sel                 <= 2'd0;
      bd_ch0_symbol_start        <= 1'b0;         
      bd_ch1_sel                 <= 2'd0;
      bd_ch1_symbol_start        <= 1'b0;
      bd_field_start             <= 1'b0;
      bd_fec                     <= 1'b0;
      bd_dcm                     <= 1'b0;             
      bd_sigen                   <= 1'b0;             
      bd_qbpsk                   <= 1'b0;           
      bd_nsd                     <= 2'b0;             
      bd_ruen                    <= 1'b0;             
      bd_rulen                   <= 3'b0;             
      bd_nes                     <= 1'b0;               
      bd_nss                     <= 3'b0;               
      bd_nbpsc                   <= 3'b0;           
      bd_stbc                    <= 1'b0;              
      bd_cr                      <= 2'b0;                
      bd_row_fifo_en             <= 1'b0;
      bd_field_length            <= 24'b0;    
      bd_field_stfix             <= 1'b0;         
      hesiga2_is_qbpsk           <= 1'b0;
      lsig_parser_enable         <= 1'b0;     
      lsig_ucpu_start            <= 1'b0;     
      htsig_parser_enable        <= 1'b0;     
      htsig_ucpu_start           <= 1'b0;     
      vhtsiga_parser_enable      <= 1'b0;     
      vhtsigb_parser_enable      <= 1'b0;     
      vhtsig_ucpu_start          <= 1'b0;     
      herlsig_parser_enable      <= 1'b0;     
      hesiga_parser_enable       <= 1'b0;     
      hesigb_parser_enable       <= 1'b0;     
      hesig_ucpu_start           <= 1'b0;     
      data_parser_enable         <= 1'b0;     
      data_confirmed             <= 1'b0;
      ch0_state                  <= 1'b0;
      ch0_sym_index              <= 8'b0;
      ch1_state                  <= 1'b0;
      ch1_sym_index              <= 8'b0;
    end
    else if(!enable)
    begin
      bd_symbol_state            <= BD_IDLE;
      bd_step_state              <= BD_WAIT_FIFO;
      sym_index                  <= 16'd0;
      bdfd_rdptr                 <= 1'd0;
      bdfd_ch0_rdptr             <= 3'd0;
      bdfd_ch1_rdptr             <= 3'd0;
      bd_bcc_enable              <= 1'b0;        
      bd_ldpc_enable             <= 1'b0;        
      bd_ch0_sel                 <= 2'd0;
      bd_ch0_symbol_start        <= 1'b0;         
      bd_ch1_sel                 <= 2'd0;
      bd_ch1_symbol_start        <= 1'b0;
      bd_field_start             <= 1'b0;
      bd_fec                     <= 1'b0;
      bd_dcm                     <= 1'b0;             
      bd_sigen                   <= 1'b0;             
      bd_qbpsk                   <= 1'b0;           
      bd_nsd                     <= 2'b0;             
      bd_ruen                    <= 1'b0;             
      bd_rulen                   <= 3'b0;             
      bd_nes                     <= 1'b0;               
      bd_nss                     <= 3'b0;               
      bd_nbpsc                   <= 3'b0;           
      bd_stbc                    <= 1'b0;              
      bd_cr                      <= 2'b0;                
      bd_row_fifo_en             <= 1'b0;
      bd_field_length            <= 24'b0;    
      bd_field_stfix             <= 1'b0;         
      hesiga2_is_qbpsk           <= 1'b0;
      lsig_parser_enable         <= 1'b0;     
      lsig_ucpu_start            <= 1'b0;     
      htsig_parser_enable        <= 1'b0;     
      htsig_ucpu_start           <= 1'b0;     
      vhtsiga_parser_enable      <= 1'b0;     
      vhtsigb_parser_enable      <= 1'b0;     
      vhtsig_ucpu_start          <= 1'b0;     
      herlsig_parser_enable      <= 1'b0;     
      hesiga_parser_enable       <= 1'b0;     
      hesigb_parser_enable       <= 1'b0;     
      hesig_ucpu_start           <= 1'b0;     
      data_parser_enable         <= 1'b0;     
      data_confirmed             <= 1'b0;
      ch0_state                  <= 1'b0;
      ch0_sym_index              <= 8'b0;
      ch1_state                  <= 1'b0;
      ch1_sym_index              <= 8'b0;
    end
    else
    begin
      /* RTZ */
      bd_ch0_symbol_start <= 1'b0;
      bd_ch1_symbol_start <= 1'b0;
      bd_field_start      <= 1'b0;
      
      case(bd_symbol_state)
        /***********************************************************************
        * BD_IDLE -> BD_LSIG,BD_HTSIG1
        ***********************************************************************/
        BD_IDLE:
        begin
          /* initial state */
          bdfd_rdptr          <= 1'b0;
          bdfd_ch0_rdptr      <= 3'd0;
          bdfd_ch1_rdptr      <= 3'd0;
          bd_bcc_enable       <= 1'b0;        
          bd_ldpc_enable       <= 1'b0;        
          bd_ch0_sel          <= 2'd0;
          bd_ch0_symbol_start <= 1'b0;         
          bd_ch1_sel          <= 2'd0;
          bd_ch1_symbol_start <= 1'b0;
          
          bd_fec              <= 1'b0;
          bd_dcm              <= 1'b0;             
          bd_sigen            <= 1'b0;             
          bd_qbpsk            <= 1'b0;           
          bd_nsd              <= 2'b0;             
          bd_ruen             <= 1'b0;             
          bd_rulen            <= 3'b0;             
          bd_nes              <= 1'b0;               
          bd_nss              <= 3'b0;               
          bd_nbpsc            <= 3'b0;           
          bd_stbc             <= 1'b0;              
          bd_cr               <= 2'b0;                
          bd_row_fifo_en      <= 1'b0;
          bd_field_length     <= 24'b0;    
          bd_field_stfix      <= 1'b0;         
          
          hesiga2_is_qbpsk    <= 1'b0;
          data_confirmed      <= 1'b0;
          sym_index           <= 16'd0;  
         
          /* detect buffer update */
          if(!fifo_empty)
          begin
            bd_bcc_enable       <= 1'b1; 
            bd_ch0_symbol_start <= 1'b1; 
            bd_cr               <= CR_12;
            bd_nbpsc            <= NBPSC_1;
            bd_sigen            <= 1'b1;
            bd_field_start      <= 1'b1;       
            
            if(!bdfd_qbpsk)
            begin
              bd_field_length     <= 24'd24;    
              bd_field_stfix      <= 1'b0;
              lsig_parser_enable  <= 1'b1;
              bd_symbol_state     <= BD_LSIG;       
              bd_step_state       <= BD_WAIT_PARSER;       
            end
            else
            begin
              bd_qbpsk            <= 1'b1;
              bd_field_length     <= 24'd48;    
              bd_field_stfix      <= 1'b0;
              htsig_parser_enable <= 1'b1;
              bd_symbol_state     <= BD_HTSIG1;       
              bd_step_state       <= BD_WAIT_SYMBOL;       
            end
          end
        end
       
        /***********************************************************************
        * BD_LSIG -> BD_LSIG  (frame bandwidth estimation for nonht)
        *         -> HERL_SIG (speculation HE)
        ***********************************************************************/
        BD_LSIG:
        begin
          case(bd_step_state)
            BD_WAIT_PARSER:
            begin
              if(lsig_captured)
              begin
                lsig_ucpu_start <= 1'b1;
                bd_step_state   <= BD_WAIT_UCPU;
              end
            end
            BD_WAIT_UCPU:
            begin
              if(lsig_ucpu_done)
              begin
                /* always reset bit domain once captured a symbol with a tail field  */
                bd_bcc_enable    <= 1'b0;
                /* update buffer rdptr */
                bdfd_rdptr       <= n_bdfd_rdptr;
                bd_step_state    <= BD_WAIT_FIFO;
              end
            end
            BD_WAIT_FIFO:
            begin
              if(!fifo_empty)
              begin
                bd_bcc_enable       <= 1'b1;
                bd_ch0_symbol_start <= 1'b1;
                bd_field_start      <= 1'b1;       
                if(lsig_cr==CR_12 && lsig_nbpsc==NBPSC_1) /* 6Mb/s */                                                                     
                begin                                                                                        
                  if(!bdfd_qbpsk)                                                                        
                  begin                                                                                      
                    if(lsig_length_mod3==2'd0)
                    begin
                      /* l_length mod 3 is 0, can't be HE */
                      bd_field_length       <= 24'd0; 
                      bd_field_stfix        <= 1'd0; 
                      data_parser_enable    <= 1'b1;
                      vhtsiga_parser_enable <= 1'b1;
                      bd_symbol_state       <= BD_LDATA0_VHTSIGA1;
                      bd_step_state         <= BD_WAIT_SYMBOL;
                    end
                    else
                    begin
                      /* speculate the next symbol is HERLSIG   */                                             
                      bd_field_length       <= 24'd24;
                      bd_field_stfix        <= 1'b0;
                      herlsig_parser_enable <= 1'b1;
                      bd_symbol_state       <= BD_HERLSIG;
                      bd_step_state         <= BD_WAIT_PARSER;
                    end
                  end                                                                                        
                  else                                                                                       
                  begin                                                                                      
                    /* HTMM */                                                                               
                    /* refer 802.11n-2012 9.23.4 and 20.2.2 table 20-1, HTMM shall have L_DATARATE=6Mb/s */  
                    bd_qbpsk            <= 1'b1;
                    bd_field_length     <= 24'd48;
                    bd_field_stfix      <= 1'b0;
                    htsig_parser_enable <= 1'b1;
                    bd_symbol_state     <= BD_HTSIG1;                                                           
                    bd_step_state       <= BD_WAIT_SYMBOL;
                  end                                                                                        
                end                                                                                          
                else                                                                                         
                begin                                                                                        
                  /* NONHT */                                                                                
                  bd_nbpsc           <= lsig_nbpsc;
                  bd_cr              <= lsig_cr;             
                  bd_field_length    <= {9'b0, lsig_length, 3'b0};
                  bd_field_stfix     <= 1'b1;
                  data_parser_enable <= 1'b1;
                  data_confirmed     <= 1'b1;
                  bd_symbol_state    <= BD_LDATA;                                                            
                  bd_step_state      <= BD_WAIT_SYMBOL;
                end                                                                                          
              end
            end
          endcase
        end
        
        /***********************************************************************
        * BD_HERLSIG -> BD_HESIGA1, BD_LDATA0_VHTSIGA1
        ***********************************************************************/
        BD_HERLSIG:
        begin
          case(bd_step_state)
            BD_WAIT_PARSER:
            begin
              if(herlsig_captured)
              begin
                /* always reset bit domain once captured a symbol with a tail field  */
                bd_bcc_enable    <= 1'b0;
                if(herlsig_eq_lsig)
                begin
                  /* HE confirmed */
                  bdfd_rdptr     <= n_bdfd_rdptr;
                  bd_step_state  <= BD_WAIT_FIFO;
                end
                else
                begin
                  bd_step_state  <= BD_REPLAY;
                end
              end
            end
            BD_REPLAY:
            begin
              /* speculation failed, retry current symbol for NONHT/VHT discrimination */
              bd_bcc_enable         <= 1'b1;
              bd_ch0_symbol_start   <= 1'b1;
              bd_field_start        <= 1'b1;
              bd_field_length       <= 24'd0; /* 0 indicates that the field length is not yet known */
              bd_field_stfix        <= 1'd0; 
              data_parser_enable    <= 1'b1;
              vhtsiga_parser_enable <= 1'b1;
              bd_symbol_state       <= BD_LDATA0_VHTSIGA1;
              bd_step_state         <= BD_WAIT_SYMBOL;
            end
            BD_WAIT_FIFO:
            begin
              if(!fifo_empty)
              begin
                bd_bcc_enable        <= 1'b1;
                bd_ch0_symbol_start  <= 1'b1;
                bd_nsd               <= NSD_52;
                bd_field_start       <= 1'b1;
                bd_field_length      <= 24'd52;
                bd_field_stfix       <= 1'b0;
                hesiga_parser_enable <= 1'b1;
                bd_symbol_state      <= BD_HESIGA1;
                bd_step_state        <= BD_WAIT_SYMBOL;
              end
            end
          endcase
        end
       
        /***********************************************************************
        * BD_HTSIG1 -> BD_HTSIG2
        ***********************************************************************/
        BD_HTSIG1:
        begin
          case(bd_step_state)
            BD_WAIT_SYMBOL:
            begin
              if(bd_ch0_symbol_done)
              begin
                /* update buffer rdptr */
                bdfd_rdptr     <= n_bdfd_rdptr;
                bd_step_state  <= BD_WAIT_FIFO;
              end
            end
            BD_WAIT_FIFO:
            begin
              /* detect buffer update */   
              if(!fifo_empty)               
              begin                        
                bd_ch0_symbol_start <= 1'b1;    
                bd_qbpsk            <= 1'b1;
                bd_symbol_state     <= BD_HTSIG2;
                bd_step_state       <= BD_WAIT_PARSER;  
              end  
            end                        
          endcase                            
        end
       
        /***********************************************************************
        * BD_HTSIG2 -> BD_HTDATA
        ***********************************************************************/
        BD_HTSIG2:
        begin      
          case(bd_step_state)
            BD_WAIT_PARSER:
            begin
              /* wait until processing end of current symbol */
              if(htsig_captured)
              begin
                htsig_ucpu_start <= 1'b1;
                bd_step_state    <= BD_WAIT_UCPU;
              end
            end
            BD_WAIT_UCPU:
            begin
              if(htsig_ucpu_done)
              begin
                /* always reset bit domain once captured a symbol with a tail field */
                bd_bcc_enable    <= 1'b0;
                /* update buffer rdptr */
                bdfd_rdptr       <= n_bdfd_rdptr;
              
                /* */
                bd_step_state       <= BD_WAIT_FIFO;   
              end
            end
            
            BD_WAIT_FIFO:
            begin
              bd_row_fifo_en      <= 1'b1;
              bd_sigen            <= 1'b0;
              if(!htsig_fec)
              begin
                bd_bcc_enable       <= 1'b1;                                                  
                bd_ch0_symbol_start <= 1'b1;
              end
              else
              begin
                bd_bcc_enable       <= 1'b0;
                bd_ldpc_enable      <= 1'b1;
              end
              bd_fec              <= htsig_fec;   
              bd_qbpsk            <= 1'b0;  
              bd_nsd              <= htsig_nsd;    
              bd_nss              <= htsig_nss;   
              bd_stbc             <= htsig_stbc;
              bd_nbpsc            <= htsig_nbpsc;
              bd_cr               <= htsig_cr;    
              bd_field_start      <= 1'b1;
              bd_field_length     <= {5'b0,htsig_length,3'b0};
              bd_field_stfix      <= 1'b1;
              data_parser_enable  <= 1'b1;
              data_confirmed      <= 1'b1;
              bd_symbol_state     <= BD_HTDATA;      /* note: fsm is parked in BD_HTDATA for NDP too */    
              bd_step_state       <= BD_ROW_FIFO;   
            end
          endcase
        end
       
        /***********************************************************************
        * BD_DATA0_VHTSIGA1 -> BD_LDATA, BD_VHTSIGA2
        ***********************************************************************/
        BD_LDATA0_VHTSIGA1:
        begin      
          case(bd_step_state)
            BD_WAIT_SYMBOL:
            begin
              if(bd_ch0_symbol_done)
              begin
                /* update buffer rdptr */
                bdfd_rdptr    <= n_bdfd_rdptr;
                bd_step_state <= BD_WAIT_FIFO;
              end
            end
            
            BD_WAIT_FIFO:
            begin
              /* detect buffer update */                
              if(!fifo_empty)                            
              begin                                     
                bd_ch0_symbol_start <= 1'b1;                     
                if(!bdfd_qbpsk)                          
                begin                                   
                  /* NONHT */                           
                  bd_qbpsk              <= 1'b0;
                  bd_field_length       <= {9'b0,lsig_length,3'b0};
                  bd_field_stfix        <= 1'b1;
                  data_confirmed        <= 1'b1;
                  vhtsiga_parser_enable <= 1'b0;
                  sym_index             <= 16'd1;
                  bd_symbol_state       <= BD_LDATA;
                  bd_step_state         <= BD_WAIT_SYMBOL;              
                end                                     
                else                                    
                begin                                   
                  /* VHT */                             
                  bd_qbpsk              <= 1'b1;
                  bd_field_length       <= 24'd48;
                  data_parser_enable    <= 1'b0;   /* vht confirmed, reset data parser */
                  bd_symbol_state       <= BD_VHTSIGA2;
                  bd_step_state         <= BD_WAIT_PARSER;           
                end                                     
              end
            end                                       
          endcase                                         
        end
      
        /***********************************************************************
        * BD_VHTSIGA2 -> BD_VHTSIGB
        ***********************************************************************/
        BD_VHTSIGA2:
        begin
          case(bd_step_state)
            BD_WAIT_PARSER:
            begin
              if(vhtsiga_captured)
              begin
                /* always reset bit domain once captured a symbol with a tail field  */
                bd_bcc_enable    <= 1'b0;
                /* update buffer rdptr */
                bdfd_rdptr       <= n_bdfd_rdptr;
                bd_step_state    <= BD_ROW_FIFO;
              end
            end
            
            BD_ROW_FIFO:
            begin
              bd_bcc_enable         <= 1'b1;
              bd_sigen              <= 1'b0;
              bd_row_fifo_en        <= 1'b1;
              bd_ch0_symbol_start   <= 1'b1;
              bd_qbpsk              <= 1'b0;
              bd_nsd                <= vhtsig_nsd;
              bd_field_start        <= 1'b1;
              case(vhtsig_nsd)
                 NSD_52: bd_field_length <= 24'd26;  /* 20M */
                NSD_108: bd_field_length <= 24'd54;  /* 40M */
                default: bd_field_length <= 24'd117; /* 80M */
              endcase
              bd_field_stfix        <= 1'b0;
              vhtsigb_parser_enable <= 1'b1;
              bd_symbol_state       <= BD_VHTSIGB;
              bd_step_state         <= BD_WAIT_PARSER;   
            end
          endcase                           
        end
        
        /***********************************************************************
        * BD_VHTSIGB -> BD_VHTDATA
        ***********************************************************************/
        BD_VHTSIGB:
        begin
          case(bd_step_state)
            BD_WAIT_PARSER:
            begin
              if(vhtsigb_captured)
              begin
                vhtsig_ucpu_start <= 1'b1;
                bd_step_state     <= BD_WAIT_UCPU;
              end
            end
            BD_WAIT_UCPU:
            begin
              if(vhtsig_ucpu_done)
              begin
                /* always reset bit domain once captured a symbol with a tail field  */
                bd_bcc_enable  <= 1'b0;  
                bd_step_state  <= BD_ROW_FIFO;
              end
            end
            BD_ROW_FIFO:
            begin
              bd_row_fifo_en      <= 1'b1;
              if(!vhtsig_fec)
              begin
                bd_bcc_enable       <= 1'b1;  
                bd_ch0_symbol_start <= 1'b1;
              end
              else
              begin
                bd_bcc_enable       <= 1'b0;  
                bd_ldpc_enable      <= 1'b1;  
              end                                                      
              bd_fec              <= vhtsig_fec;            
              bd_nes              <= vhtsig_nes;               
              bd_nss              <= vhtsig_nss;             
              bd_nbpsc            <= vhtsig_nbpsc;           
              bd_cr               <= vhtsig_cr;           
              bd_stbc             <= vhtsig_stbc;           
              bd_field_start      <= 1'b1;
              bd_field_length     <= npld;
              bd_field_stfix      <= 1'b0;
              data_parser_enable  <= 1'b1;
              data_confirmed      <= 1'b1;
              bd_symbol_state     <= BD_VHTDATA; /* note: fsm is parked in BD_VHTDATA for NDP too  */
              bd_step_state       <= BD_ROW_FIFO;       
            end                                                                        
          endcase                                                                                 
        end
        
        /***********************************************************************
        * BD_HESIGA1 -> BD_HESUSIGA2, BD_HEERSUSIGA2
        ***********************************************************************/
        BD_HESIGA1:
        begin
          case(bd_step_state)
            BD_WAIT_SYMBOL:
            begin
              if(bd_ch0_symbol_done)
              begin
                /* update buffer rdptr */
                bdfd_rdptr      <= n_bdfd_rdptr;
                bd_step_state   <= BD_WAIT_FIFO;
              end
            end
            BD_WAIT_FIFO:
            begin
              /* detect buffer update */                
              if(!fifo_empty)                            
              begin                                     
                if(!bdfd_qbpsk)                          
                begin                                   
                  /* HESIGA2 */                           
                  bd_ch0_symbol_start <= 1'b1;                     
                  bd_symbol_state     <= BD_HESIGA2; 
                  bd_step_state       <= BD_WAIT_PARSER;             
                end                                     
                else                                    
                begin
                  /* HEERSUSIGA2 , do nothing , skipped */
                  hesiga2_is_qbpsk    <= 1'b1;
                  bdfd_rdptr          <= n_bdfd_rdptr;
                  bd_symbol_state     <= BD_HEERSUSIGA2;
                  bd_step_state       <= BD_WAIT_FIFO;            
                end  
              end 
            end
          endcase                                
        end

        /***********************************************************************
        * BD_HESIGA2 -> BD_HESIGB, BD_HEDATA
        ***********************************************************************/
        BD_HESIGA2:
        begin
          case(bd_step_state)
            BD_WAIT_PARSER:
            begin
              if(hesiga_captured)
              begin
                if(lsig_length_mod3==2'd2)
                begin
                  /* HE_MU */
                  /* always reset bit domain once captured a symbol with a tail field */
                  bd_bcc_enable    <= 1'b0;
                  /* update buffer rdptr */
                  bdfd_rdptr       <= n_bdfd_rdptr;
                  bd_step_state    <= BD_WAIT_FIFO;
                end
                else
                begin
                  /* not a HE_MU */
                  hesig_ucpu_start <= 1'b1;
                  bd_step_state    <= BD_WAIT_UCPU;
                end
              end
            end
            BD_WAIT_UCPU:                                                               
            begin                                                                       
              if(hesig_ucpu_done)                                                       
              begin                                                                     
                /* always reset bit domain once captured a symbol with a tail field */  
                bd_bcc_enable    <= 1'b0;                                               
                /* update buffer rdptr */                                               
                bdfd_rdptr       <= n_bdfd_rdptr;
                bd_step_state    <= BD_WAIT_FIFO;                                       
              end                                                                       
            end                                                                         
            BD_WAIT_FIFO:                                                                             
            begin                                                                                     
              if(lsig_length_mod3==2'd2)                                                            
              begin                                                                                 
                /* HEMU */                                                                        
                bd_bcc_enable        <= 1'b1;                                                        
                /* note: bd_field_start handled by rx_bd_ctrl_hesigb_parser */                    
                bd_qbpsk             <= 1'b0;                                                     
                bd_nbpsc             <= hesig_sigb_nbpsc;                                         
                bd_cr                <= hesig_sigb_cr;                                            
                bd_dcm               <= hesig_sigb_dcm;                                           
                bd_field_length      <= 24'd0;                                                     
                bd_field_stfix       <= 1'b0;                                                     
                hesigb_parser_enable <= 1'b1;                                                     
                bd_symbol_state      <= BD_HESIGB;                                                
                bd_step_state        <= BD_WAIT_SYMBOL;                                           
              end
              else
              begin
                /* HESU */                                                                          
                bd_row_fifo_en      <= 1'b1;
                bd_sigen            <= 1'b0;
                if(!hesig_fec)
                begin
                  bd_bcc_enable       <= 1'b1;                                                        
                  bd_ch0_symbol_start <= 1'b1;
                end
                else
                begin
                  bd_bcc_enable       <= 1'b0;                                                        
                  bd_ldpc_enable      <= 1'b1;                                                        
                end                                                        
                bd_fec              <= hesig_fec;                                                  
                bd_dcm              <= hesig_dcm;                                                  
                bd_qbpsk            <= 1'b0;                                                       
                bd_ruen             <= 1'b1;                                                       
                bd_rulen            <= hesig_rulen;                                                
                bd_stbc             <= hesig_stbc;                                                 
                bd_nss              <= hesig_stbc?{1'b0,hesig_nsts[2:1]}:hesig_nsts;               
                bd_nbpsc            <= hesig_nbpsc;                                                
                bd_cr               <= hesig_cr;                                                   
                bd_field_start      <= 1'b1;                                                       
                bd_field_length     <= npld;                                         
                bd_field_stfix      <= 1'b0;                                                       
                data_parser_enable  <= 1'b1;                                                       
                data_confirmed      <= 1'b1;                                                       
                bd_symbol_state     <= BD_HEDATA;                                                  
                bd_step_state       <= BD_ROW_FIFO;                                             
              end                                                                                     
            end                                                                                       
          endcase
        end

        /***********************************************************************
        * BD_HEERSUSIGA2 -> BD_HEERSUSIGA3
        ***********************************************************************/
        BD_HEERSUSIGA2:
        begin
          /* HE_ER_SU not implemented, anyway premable is decoded 
             HE_ER_SU SIGA1 and SIGA3 are gathered to get the HE_SU SIG fields
             HE_ER_SU SIGA2 and SIGA4 are dropped
          */
          if(!fifo_empty)
          begin
            bd_ch0_symbol_start <= 1'b1;                     
            bd_qbpsk            <= 1'b0;
            bd_symbol_state     <= BD_HEERSUSIGA3;
            bd_step_state       <= BD_WAIT_PARSER; 
          end
        end
       
        /***********************************************************************
        * BD_HEERSUSIGA3 -> BD_HEERSUSIGA4
        ***********************************************************************/
        BD_HEERSUSIGA3:
        begin
          case(bd_step_state)
            BD_WAIT_PARSER:
            begin
              if(hesiga_captured)
              begin
                hesig_ucpu_start <= 1'b1;
                bd_step_state    <= BD_WAIT_UCPU;
              end
            end
            
            BD_WAIT_UCPU:
            begin
              if(hesig_ucpu_done)
              begin
                /* update buffer rdptr */
                bdfd_rdptr       <= n_bdfd_rdptr;
                bd_step_state  <= BD_WAIT_FIFO;
              end
            end
            BD_WAIT_FIFO:
            begin
              if(!fifo_empty)
              begin
                /* HEERSUSIGA4, do nothing skipped */                         
                /* update buffer rdptr */
                bdfd_rdptr       <= n_bdfd_rdptr;
                bd_symbol_state <= BD_HEERSUSIGA4;
                bd_step_state   <= BD_WAIT_FIFO;
              end
            end             
          endcase
        end
       
        /***********************************************************************
        * BD_HEERSUSIGA4 -> BD_HEDATA
        ***********************************************************************/
        BD_HEERSUSIGA4:
        begin
          if(!fifo_empty)
          begin
            bd_sigen            <= 1'b0;
            bd_row_fifo_en      <= 1'b1;
            
            if(!hesig_fec)
            begin
              bd_bcc_enable       <= 1'b1;
              bd_ch0_symbol_start <= 1'b1;
            end
            else
            begin
              bd_bcc_enable       <= 1'b0;
              bd_ldpc_enable      <= 1'b1;
            end
            bd_fec              <= hesig_fec;             
            bd_dcm              <= hesig_dcm;      
            bd_qbpsk            <= 1'b0;
            bd_ruen             <= 1'b1;             
            bd_rulen            <= hesig_rulen;
            bd_stbc             <= hesig_stbc;
            bd_nss              <= hesig_stbc?{1'b0,hesig_nsts[2:1]}:hesig_nsts;             
            bd_nbpsc            <= hesig_nbpsc;           
            bd_cr               <= hesig_cr;               
            
            bd_field_start      <= 1'b1;                     
            bd_field_length     <= npld;
            bd_field_stfix      <= 1'b0;    
            
            data_parser_enable  <= 1'b1;
            data_confirmed      <= 1'b1;
            bd_symbol_state     <= BD_HEDATA;   
            bd_step_state       <= BD_WAIT_SYMBOL;   
          end
        end

        /***********************************************************************
        * BD_HESIGB -> BD_HESIGB, BD_HEDATA
        ***********************************************************************/
        BD_HESIGB:
        begin
          case(bd_step_state)
            BD_WAIT_SYMBOL:
            begin
              /* channel content 0 */
              if(ch0_available)
              begin
                if(!ch0_state)
                begin
                  if(!fifo_ch0_empty)
                  begin
                    bd_ch0_symbol_start <= ~hesigb_parser_ch0_done;
                    ch0_state           <= 1'b1;
                  end
                end
                else
                begin
                  if(bd_ch0_symbol_done || hesigb_parser_ch0_done)
                  begin
                    if(n_ch0_sym_index!=nsym_hesigb)
                    begin
                      ch0_state      <= 1'b0;
                      ch0_sym_index  <= n_ch0_sym_index;
                      bdfd_ch0_rdptr <= n_bdfd_ch0_rdptr;
                      bd_ch0_sel     <= n_bdfd_ch0_rdptr[1:0];
                    end
                  end
                end
              end
              else
              begin
                bdfd_ch0_rdptr <= bdfd_ch0_wrptr;
              end
              
              /* channel content 1 */
              if(ch1_available)
              begin
                if(!ch1_state)
                begin
                  if(!fifo_ch1_empty)
                  begin
                    bd_ch1_symbol_start <= ~hesigb_parser_ch1_done;
                    ch1_state           <= 1'b1;
                  end
                end
                else
                begin
                  if(bd_ch1_symbol_done || hesigb_parser_ch1_done)
                  begin
                    if(n_ch1_sym_index!=nsym_hesigb)
                    begin
                      ch1_state      <= 1'b0;
                      ch1_sym_index  <= n_ch1_sym_index;
                      bdfd_ch1_rdptr <= n_bdfd_ch1_rdptr;
                      bd_ch1_sel     <= n_bdfd_ch1_rdptr[1:0];
                    end
                  end
                end
              end
              else
              begin
                bdfd_ch1_rdptr <= bdfd_ch1_wrptr;
              end
              
              /* stop condition */
              if( (!ch0_available || hesigb_parser_ch0_done && (n_ch0_sym_index==nsym_hesigb)) &&
                  (!ch1_available || hesigb_parser_ch1_done && (n_ch1_sym_index==nsym_hesigb)))
              begin
                bd_bcc_enable    <= 1'b0;
                hesig_ucpu_start <= 1'b1;
                bd_step_state    <= BD_WAIT_UCPU;
              end
            end
            
            BD_WAIT_UCPU:
            begin
              if(hesig_ucpu_done)
              begin
                bd_step_state       <= BD_WAIT_FIFO;
              end
            end
            
            BD_WAIT_FIFO:
            begin
              bd_ch0_sel          <= 2'd0;
              bd_ch1_sel          <= 2'd0;
              if(!hesig_fec)
              begin
                bd_bcc_enable       <= 1'b1;
                bd_ch0_symbol_start <= 1'b1;
              end
              else
              begin
                bd_bcc_enable       <= 1'b0;
                bd_ldpc_enable      <= 1'b1;
              end
              
              bd_fec              <= hesig_fec;             
              bd_dcm              <= hesig_dcm;                  
              bd_ruen             <= 1'b1;
              bd_rulen            <= hesig_rulen;                  
              bd_nss              <= hesig_nsts; /* in mu, no stbc so nss=nsts */              
              bd_nbpsc            <= hesig_nbpsc;                
              bd_cr               <= hesig_cr;                    
              bd_sigen            <= 1'b0;
              bd_row_fifo_en      <= 1'b1;                                          

              bd_field_start      <= 1'b1;
              bd_field_length     <= npld;
              bd_field_stfix      <= 1'b0;
              
              data_parser_enable  <= 1'b1;
              data_confirmed      <= 1'b1;
              
              sym_index           <= 16'd0;
              
              bd_symbol_state     <= BD_HEDATA;
              bd_step_state       <= BD_ROW_FIFO;
            end
          endcase
        end
        
        /***********************************************************************
        * BD_HTDATA,BD_VHTDATA,BD_HEDATA
        ***********************************************************************/
        BD_HTDATA,BD_VHTDATA,BD_HEDATA:
        begin
          /* wait until processing of current symbol done */
          if(bd_ch0_symbol_done)
          begin
            bd_ch0_symbol_start <= 1'b1;
          end
        end
        
        /***********************************************************************
        * BD_LDATA,BD_HTDATA,BD_VHTDATA,BD_HEDATA
        ***********************************************************************/
        BD_LDATA:
        begin
          case(bd_step_state)
            BD_WAIT_SYMBOL:
            begin
              /* wait until processing of current symbol done */
              if(bd_ch0_symbol_done)
              begin
                /* update buffer rdptr */
                bdfd_rdptr     <= n_bdfd_rdptr;
                bd_step_state  <= BD_WAIT_FIFO;
              end
            end
            
            BD_WAIT_FIFO:
            begin
              if(!fifo_empty)
              begin
                bd_ch0_symbol_start <= 1'b1;
                bd_step_state       <= BD_WAIT_SYMBOL;
              end
            end
          endcase
        end
        
        /***********************************************************************
        * DONE
        ***********************************************************************/
        default: ;
        
      endcase
    end
  end

  /*****************************************************************************
  * diag
  *****************************************************************************/
  assign diag[9]   = ch1_state;
  assign diag[8]   = ch0_state;
  assign diag[7:3] = bd_symbol_state;
  assign diag[2:0] = bd_step_state;
  
`ifdef RW_SIMU_ON
  reg [ 32*8-1:0] s_bd_symbol_state;
  always @(*)
  begin
    case(bd_symbol_state)
      BD_IDLE:            s_bd_symbol_state = "BD_IDLE";
      BD_LSIG:            s_bd_symbol_state = "BD_LSIG";
      BD_LDATA0_VHTSIGA1: s_bd_symbol_state = "BD_LDATA0_VHTSIGA1";
      BD_LDATA:           s_bd_symbol_state = "BD_LDATA";
      BD_HTSIG1:          s_bd_symbol_state = "BD_HTSIG1";
      BD_HTSIG2:          s_bd_symbol_state = "BD_HTSIG2";
      BD_HTDATA:          s_bd_symbol_state = "BD_HTDATA";
      BD_VHTSIGA2:        s_bd_symbol_state = "BD_VHTSIGA2";
      BD_VHTSIGB:         s_bd_symbol_state = "BD_VHTSIGB";
      BD_VHTDATA:         s_bd_symbol_state = "BD_VHTDATA";
      BD_HERLSIG:         s_bd_symbol_state = "BD_HERLSIG";
      BD_HESIGA1:         s_bd_symbol_state = "BD_HESIGA1";
      BD_HESIGA2:         s_bd_symbol_state = "BD_HESIGA2";
      BD_HEERSUSIGA2:     s_bd_symbol_state = "BD_HEERSUSIGA2";
      BD_HEERSUSIGA3:     s_bd_symbol_state = "BD_HEERSUSIGA3";
      BD_HEERSUSIGA4:     s_bd_symbol_state = "BD_HEERSUSIGA4";
      BD_HESIGB:          s_bd_symbol_state = "BD_HESIGB";
      BD_HEDATA:          s_bd_symbol_state = "BD_HEDATA";
      default:            s_bd_symbol_state = "XXXXXXXXX";
    endcase
  end
 
  reg [ 32*8-1:0] s_bd_step_state;
  always @(*)
  begin
    case(bd_step_state)
      BD_WAIT_SYMBOL: s_bd_step_state = "BD_WAIT_SYMBOL";
      BD_WAIT_PARSER: s_bd_step_state = "BD_WAIT_PARSER";
      BD_WAIT_FIFO:   s_bd_step_state = "BD_WAIT_FIFO";
      BD_WAIT_UCPU:   s_bd_step_state = "BD_WAIT_UCPU";
      BD_REPLAY:      s_bd_step_state = "BD_REPLAY";
      BD_ROW_FIFO:    s_bd_step_state = "BD_ROW_FIFO";
      default:        s_bd_step_state = "XXXXXXXXX";
    endcase
  end
`endif  
  
endmodule
`default_nettype wire

   
