/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module rx_bd_ctrl_ucpu
(
  /* system */
  input  wire         clk,
  input  wire         rst_n,
  
  /* control */
  input  wire         enable,

  /* done */
  output reg          ndbps_done,
  output reg          l_done,
  output reg          ht_done,
  output reg          vht_done,
  output reg          he_done,
  
  /* lsig parameters */
  input  wire         start_lsig,
  input  wire [11:0]  lsig_length,
  input  wire [ 2:0]  lsig_nbpsc,
  input  wire [ 1:0]  lsig_cr,
  
  /* htsig parameters */ 
  input  wire         start_htsig,
  input  wire         htsig_stbc,
  input  wire [ 2:0]  htsig_nss,
  input  wire [ 1:0]  htsig_cr,
  input  wire [ 1:0]  htsig_nsd,
  input  wire [ 2:0]  htsig_nbpsc,
  input  wire [15:0]  htsig_length,
  input  wire         htsig_fec,

  /* vhtsig parameters */
  input  wire         start_vhtsig,
  input  wire [ 1:0]  vhtsig_gi_type,
  input  wire         vhtsig_sgidisamb,
  input  wire         vhtsig_stbc,
  input  wire [ 2:0]  vhtsig_bandwidth,
  input  wire [ 2:0]  vhtsig_nsts,
  input  wire [ 2:0]  vhtsig_nsts_total,
  input  wire [ 1:0]  vhtsig_cr,
  input  wire [ 2:0]  vhtsig_nbpsc,
  input  wire         vhtsig_fec,
  input  wire         vhtsig_extra,

  /* hesig parameters */
  input  wire         start_hesig,
  input  wire [ 3:0]  hesig_format,
  input  wire [ 1:0]  hesig_a,
  input  wire [ 2:0]  hesig_nsts,
  input  wire [ 2:0]  hesig_nheltf,
  input  wire [ 1:0]  hesig_gi_type,
  input  wire [ 1:0]  hesig_heltf_type,
  input  wire         hesig_pedisamb,
  input  wire         hesig_dcm,
  input  wire [ 2:0]  hesig_rulen,
  input  wire [ 1:0]  hesig_cr,
  input  wire [ 2:0]  hesig_nbpsc,
  input  wire         hesig_stbc,
  input  wire         hesig_fec,
  input  wire         hesig_extra,
  input  wire         hesig_doppler,
  input  wire         hesig_midamble,
  input  wire [ 7:0]  hesig_sigb_nsym,
  
  /*  results */
  output reg  [15:0]  ndbps,
  output reg  [15:0]  ncbps,
  output reg          inconsistent,
  output reg  [ 1:0]  l_length_mod3, 
  output reg  [15:0]  nsym,
  output reg  [25:0]  psdulen,
  output reg  [12:0]  rxtime,    
  output reg  [ 2:0]  tpe,
  output reg  [ 8:0]  nma,
  output reg  [ 1:0]  nes,
  output reg  [23:0]  npld,
  
  /* LDPC results */
  output reg  [11:0]  ncw,
  output reg  [ 1:0]  lldpc,
  output reg  [11:0]  nshrtq,
  output reg  [11:0]  nshrtr,
  output reg  [11:0]  npuncq,
  output reg  [11:0]  npuncr,
  output reg  [13:0]  nrepq,
  output reg  [11:0]  nrepr
);
  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam  NSD_48=2'd0, NSD_52=2'd1;
  localparam  CR_12=2'd0,CR_23=2'd1,CR_34=2'd2,CR_56=2'd3; 
  localparam  CBW_20=2'd0,CBW_40=2'd1,CBW_80=2'd2;
  localparam  NSTS_1=3'd0,NSTS_2=3'd1,NSTS_3=3'd2,NSTS_4=3'd3,NSTS_5=3'd4,NSTS_6=3'd5,NSTS_7=3'd6,NSTS_8=3'd7;
  localparam  NSS_1=3'd0,NSS_2=3'd1,NSS_3=3'd2,NSS_4=3'd3,NSS_5=3'd4,NSS_6=3'd5,NSS_7=3'd6,NSS_8=3'd7;
  localparam  NBPSC_1=3'd0,NBPSC_2=3'd1,NBPSC_4=3'd2,NBPSC_6=3'd3,NBPSC_8=3'd4,NBPSC_10=3'd5;
  localparam  RU_26=3'd0,RU_52=3'd1,RU_106=3'd2,RU_242=3'd3,RU_484=3'd4,RU_996=3'd5,RU_2X996=3'd6;
  localparam  NON_HT=4'd0,NON_HT_DUP=4'd1,HT_MM=4'd2,HT_GF=4'd3,VHT=4'd4,
              HE_SU=4'd5,HE_MU=4'd6,HE_ER_SU=4'd7,HE_TB=4'd8;

  localparam  GI_400=2'd0,GI_800=2'd1,GI_1600=2'd2,GI_3200=2'd3;
  localparam  HELTF_1X=2'd0,HELTF_2X=2'd1,HELTF_4X=2'd2;

  wire  [63:0] cc;
  wire  [ 8:0] pr_addr;
  wire  [15:0] pr_data;
  wire  [ 5:0] rb_addr;
  reg   [31:0] rb_data;
  wire  [ 5:0] wb_addr;
  wire  [31:0] wb_x,wb_y;
  
  /* wire  */
  reg   [ 1:0] cr;
  reg   [10:0] nsd,nsd_short;
  reg   [ 4:0] nbpsc;
  reg   [ 3:0] nss;
  reg          stbc;
  reg          fec;
  reg   [ 2:0] ainit;
  
  /* flops */
  reg   [31:0] r0,r1,r3;
  reg          extra;
  reg   [23:0] navbits;
  reg   [23:0] nshrt,npunc;
  reg   [15:0] nsyminit;
  reg   [15:0] ncbps_short;
  reg   [15:0] ndbps_short;
 
  /*****************************************************************************
  * rw_ucpu
  *****************************************************************************/
  wire   ucpu_done;
  reg    ucpu_enable,lsig_state,htsig_state,vhtsig_state,hesig_state;
 
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      ucpu_enable     <= 1'b0;
      lsig_state      <= 1'b0;
      htsig_state     <= 1'b0;
      vhtsig_state    <= 1'b0;
      hesig_state     <= 1'b0;
    end
    else if(!enable)
    begin
      ucpu_enable     <= 1'b0;
      lsig_state      <= 1'b0;
      htsig_state     <= 1'b0;
      vhtsig_state    <= 1'b0;
      hesig_state     <= 1'b0;
    end
    else
    begin
      if(!lsig_state && start_lsig)     {ucpu_enable,   lsig_state} <= 2'b11;
      if(!htsig_state && start_htsig)   {ucpu_enable,  htsig_state} <= 2'b11;
      if(!vhtsig_state && start_vhtsig) {ucpu_enable, vhtsig_state} <= 2'b11;
      if(!hesig_state && start_hesig)   {ucpu_enable,  hesig_state} <= 2'b11;
      if(ucpu_done)    ucpu_enable <= 1'b0;
    end
  end
  
  rw_ucpu u_ucpu
  (
    /* system */
    .rst_n(       rst_n),
    .clk(         clk),
  
    /* control */
    .enable(      ucpu_enable),
    .done(        ucpu_done),
  
    /* prog */
    .pr_addr(     pr_addr),
    .pr_data(     pr_data),

    /* cc */
    .cc(          cc),
  
    /* rb */
    .rb_addr(     rb_addr),
    .rb_data(     rb_data),
  
    /* wb */
    .wb_addr(     wb_addr),
    .wb_x(        wb_x),
    .wb_y(        wb_y)
  );

  /*****************************************************************************
  * condition codes
  *****************************************************************************/
  localparam CC_TRUE             = 6'd0,
             CC_X_ZERO           = 6'd1,
             CC_HT               = 6'd2,
             CC_VHT              = 6'd3,
             CC_HE               = 6'd4,
             CC_LDPC             = 6'd5,
             CC_BCC              = 6'd6,
             CC_NE               = 6'd7,
             CC_LE               = 6'd8,
             CC_GE               = 6'd9,
             CC_LT               = 6'd10,
             CC_GT               = 6'd11,
             CC_NOEXTRA          = 6'd12,
             CC_VHTSIGA_EXTRA    = 6'd13,
             CC_HESIGA_EXTRA     = 6'd14,
             CC_AINIT_EQ1        = 6'd15,
             CC_AINIT_EQ2        = 6'd16,
             CC_AINIT_EQ3        = 6'd17,
             CC_AINIT_EQ4        = 6'd18,
             CC_NOT_PEDISAMB     = 6'd19,
             CC_NOT_A_EQ1_EXTRA  = 6'd20,
             CC_NMA_TEST         = 6'd21;
           
  assign cc[CC_TRUE]            = 1'b1;
  assign cc[CC_X_ZERO]          = wb_x[31:0]==32'd0;
  assign cc[CC_HT]              = htsig_state;
  assign cc[CC_VHT]             = vhtsig_state;
  assign cc[CC_HE]              = hesig_state;
  assign cc[CC_LDPC]            = hesig_state &&  hesig_fec || vhtsig_state &&  vhtsig_fec || htsig_state &&  htsig_fec ;
  assign cc[CC_BCC]             = hesig_state && !hesig_fec || vhtsig_state && !vhtsig_fec || htsig_state && !htsig_fec ;
  assign cc[CC_NE]              = wb_y[31:0]!=32'd0; 
  assign cc[CC_LE]              = wb_y[31:0]==32'd0 || wb_y[31]; 
  assign cc[CC_GE]              = !wb_y[31]; 
  assign cc[CC_LT]              = wb_y[31];
  assign cc[CC_GT]              = !wb_y[31] && wb_y[30:0]!=31'd0; 
  assign cc[CC_NOEXTRA]         = !extra;
  assign cc[CC_VHTSIGA_EXTRA]   = vhtsig_state && vhtsig_fec && vhtsig_extra;
  assign cc[CC_HESIGA_EXTRA]    = hesig_state && hesig_fec && hesig_extra;
  assign cc[CC_AINIT_EQ1]       = ainit==3'd1;
  assign cc[CC_AINIT_EQ2]       = ainit==3'd2;
  assign cc[CC_AINIT_EQ3]       = ainit==3'd3;
  assign cc[CC_AINIT_EQ4]       = ainit==3'd4;
  assign cc[CC_NOT_PEDISAMB]    = !hesig_pedisamb;
  assign cc[CC_NOT_A_EQ1_EXTRA] = !(hesig_fec && hesig_a==2'd1 && hesig_extra);
  assign cc[CC_NMA_TEST]        = hesig_doppler && !wb_y[31];
  assign cc[63:22]              = 46'b0;
  
  /*****************************************************************************
  * decoding
  *****************************************************************************/
  always @(*)
  begin
    if(hesig_state)
    begin
      cr   = hesig_cr;
      fec  = hesig_fec; 
      stbc = hesig_stbc;
      case(hesig_nbpsc)
        NBPSC_1:  nbpsc = 5'd1;
        NBPSC_2:  nbpsc = 5'd2;
        NBPSC_4:  nbpsc = 5'd4;
        NBPSC_6:  nbpsc = 5'd6;
        NBPSC_8:  nbpsc = 5'd8;
       default:   nbpsc = 5'd10;
      endcase
      case(hesig_nsts)
        NSTS_1:   nss = 4'd1;
        NSTS_2:   nss = (hesig_stbc)?4'd1:4'd2;
        NSTS_3:   nss = 4'd3;
        NSTS_4:   nss = (hesig_stbc)?4'd2:4'd4;
        NSTS_5:   nss = 4'd5;
        NSTS_6:   nss = (hesig_stbc)?4'd3:4'd6;
        NSTS_7:   nss = 4'd7;
        default:  nss = (hesig_stbc)?4'd4:4'd8;
      endcase
      case(hesig_rulen)
        RU_26:    {nsd,nsd_short} = (!hesig_dcm)?{  11'd24,   11'd6}:{  11'd12,   11'd2};
        RU_52:    {nsd,nsd_short} = (!hesig_dcm)?{  11'd48,  11'd12}:{  11'd24,   11'd6};
        RU_106:   {nsd,nsd_short} = (!hesig_dcm)?{ 11'd102,  11'd24}:{  11'd51,  11'd12};
        RU_242:   {nsd,nsd_short} = (!hesig_dcm)?{ 11'd234,  11'd60}:{ 11'd117,  11'd30};
        RU_484:   {nsd,nsd_short} = (!hesig_dcm)?{ 11'd468, 11'd120}:{ 11'd234,  11'd60};
        RU_996:   {nsd,nsd_short} = (!hesig_dcm)?{ 11'd980, 11'd240}:{ 11'd490, 11'd120};
        RU_2X996: {nsd,nsd_short} = (!hesig_dcm)?{11'd1960, 11'd492}:{ 11'd980, 11'd246};
        default:  {nsd,nsd_short} = {11'd0,11'd0};
      endcase 
    end
    else if(vhtsig_state)
    begin
      nsd_short = 11'd0;
      cr        = vhtsig_cr;
      stbc      = vhtsig_stbc;
      fec       = vhtsig_fec;
      case(vhtsig_bandwidth)
        3'd0:    nsd = 11'd52;
        3'd1:    nsd = 11'd108;
        default: nsd = 11'd234;
      endcase
      case(vhtsig_nsts)
          NSTS_1:  nss = 4'd1;
          NSTS_2:  nss = (vhtsig_stbc)?4'd1:4'd2;
          NSTS_3:  nss = 4'd3;
          NSTS_4:  nss = (vhtsig_stbc)?4'd2:4'd4;
          NSTS_5:  nss = 4'd5;
          NSTS_6:  nss = (vhtsig_stbc)?4'd3:4'd6;
          NSTS_7:  nss = 4'd7;
          default: nss = (vhtsig_stbc)?4'd4:4'd4;
      endcase
      case(vhtsig_nbpsc)        
        NBPSC_1:  nbpsc = 5'd1;  
        NBPSC_2:  nbpsc = 5'd2;  
        NBPSC_4:  nbpsc = 5'd4;  
        NBPSC_6:  nbpsc = 5'd6;  
        default:  nbpsc = 5'd8;  
      endcase                    
    end
    else if(htsig_state)
    begin
      nsd_short = 11'd0;
      cr        = htsig_cr;
      stbc      = htsig_stbc;
      fec       = htsig_fec;
      case(htsig_nsd)
        NSD_48:  nsd = 11'd48;
        NSD_52:  nsd = 11'd52;
        default: nsd = 11'd108;
      endcase
      case(htsig_nbpsc)
        NBPSC_1:  nbpsc = 5'd1;
        NBPSC_2:  nbpsc = 5'd2;
        NBPSC_4:  nbpsc = 5'd4;
        default:  nbpsc = 5'd6;
      endcase
      case(htsig_nss)
        NSS_1:    nss   = 4'd1;
        NSS_2:    nss   = 4'd2;
        NSS_3:    nss   = 4'd3;
        default:  nss   = 4'd4;
      endcase
    end
    else
    begin
      nsd       = 11'd48;
      nsd_short = 11'd0;
      cr        = lsig_cr;
      stbc      = 1'b0;
      fec       = 1'b0;
      nss       = 4'd1;
      case(lsig_nbpsc)
        NBPSC_1:  nbpsc = 5'd1;
        NBPSC_2:  nbpsc = 5'd2;
        NBPSC_4:  nbpsc = 5'd4;
        default:  nbpsc = 5'd6;
      endcase
    end
    
    if(fec)
    begin           
      nes = 2'd0;
    end
    else if(!htsig_state && 
             vhtsig_bandwidth==3'd2  && 
             vhtsig_nsts==NSTS_2 && 
             !vhtsig_stbc && 
             (vhtsig_nbpsc==NBPSC_8 || vhtsig_nbpsc==NBPSC_6 && vhtsig_cr==CR_56)) 
    begin
      nes = 2'd2; 
    end
    else
    begin              
      nes = 2'd1; 
    end

    if(hesig_fec && hesig_extra)
    begin
      case(hesig_a)
        2'd1:    ainit=3'd4;
        2'd2:    ainit=3'd1;
        2'd3:    ainit=3'd2; 
        default: ainit=3'd3; 
      endcase
    end
    else
    begin
      case(hesig_a)
        2'd1:    ainit=3'd1;
        2'd2:    ainit=3'd2;
        2'd3:    ainit=3'd3; 
        default: ainit=3'd4; 
      endcase
    end
  end
 
  /*****************************************************************************
  * operands
  *****************************************************************************/
  localparam  R_NULL        = 6'd0,  R_1           = 6'd1,
              R_3           = 6'd2,  R_5           = 6'd3,           
              R_10          = 6'd4,  R_22          = 6'd5,          
              R_648         = 6'd6,  R_R0X4        = 6'd7,           
              R_R0          = 6'd8,  R_R3          = 6'd9,
              R_R1          = 6'd10, R_L_LENGTHX8  = 6'd11,           
              R_L_LENGTH    = 6'd12, R_NSD         = 6'd13,     
              R_HT_LENGTHX8 = 6'd14, R_CRN         = 6'd15,          
              R_NBPSC       = 6'd16, R_12XCRN      = 6'd17,          
              R_CRD         = 6'd18, R_2916DIVCRD  = 6'd19,       
              R_1944XCR     = 6'd20, R_912DIVCRD   = 6'd21,     
              R_1464DIVCRD  = 6'd22, R_NAVBITS     = 6'd23,    
              R_NPLD        = 6'd24, R_NPUNC       = 6'd25,      
              R_NSHRT       = 6'd26, R_LLDPC       = 6'd27,        
              R_NCW         = 6'd28, R_MXNCBPS     = 6'd29,        
              R_NSYMINIT    = 6'd30, R_MSTBC       = 6'd31,      
              R_MXNDBPS     = 6'd32, R_TSYM        = 6'd33,        
              R_SPT         = 6'd34, R_RXTIME      = 6'd35,         
              R_SGIDIS      = 6'd36, R_NDBPS       = 6'd37,       
              R_36P4XNVHTLTF= 6'd38, R_EXTRAFIX    = 6'd39,        
              R_NCBPS       = 6'd40, R_HEMPLUS3    = 6'd41,     
              R_THELTF      = 6'd42, R_NHELTF      = 6'd43,
              R_NSS         = 6'd44, R_PEDISAMB    = 6'd45, 
              R_MXNDBPSSHORT= 6'd46, R_120         = 6'd47, 
              R_NSYM        = 6'd48, R_NSDSHORT    = 6'd49,
              R_480         = 6'd50, R_HESIGBNSYM  = 6'd51,
              R_AINIT       = 6'd52, R_MXNCBPSSHORT= 6'd53,
              R_MMAXTSYM    = 6'd54, R_PEDISP2XTSYM= 6'd55,
              R_NMA         = 6'd56;            
  
  always @(*)
  begin
    case(rb_addr)
      R_1:             rb_data = 32'd1;
      R_3:             rb_data = 32'd3;
      R_5:             rb_data = 32'd5;
      R_10:            rb_data = 32'd10;
      R_480:           rb_data = 32'd480;
      R_22:            rb_data = 32'd22;
      R_648:           rb_data = 32'd648;
      R_R0:            rb_data = r0;
      R_R0X4:          rb_data = {r0[29:0],2'b0};
      R_R1:            rb_data = r1;
      R_R3:            rb_data = r3;
      R_L_LENGTH:      rb_data = {20'b0,lsig_length};
      R_L_LENGTHX8:    rb_data = {17'b0,lsig_length,3'b0};
      R_HT_LENGTHX8:   rb_data = {13'b0,htsig_length,3'b0};
      R_NBPSC:         rb_data = {27'b0,nbpsc};
      R_NSD:           rb_data = {21'b0,nsd};
      R_NSDSHORT:      rb_data = {21'b0,nsd_short};
      R_CRN:           case(cr)
                         CR_12:    rb_data = 32'd1; 
                         CR_23:    rb_data = 32'd2; 
                         CR_34:    rb_data = 32'd3; 
                         default:  rb_data = 32'd5; 
                       endcase                      
      R_CRD:           case(cr)
                         CR_12:    rb_data = 32'd2;
                         CR_23:    rb_data = 32'd3;
                         CR_34:    rb_data = 32'd4;
                         default   rb_data = 32'd6;
                       endcase
      R_12XCRN:        case(cr)
                         CR_12:    rb_data = 32'd12;
                         CR_23:    rb_data = 32'd24;
                         CR_34:    rb_data = 32'd36;
                         default:  rb_data = 32'd60;
                       endcase      
      R_1944XCR:       case(cr)
                         CR_12:    rb_data = 32'd972;   // 1944 * 1/2
                         CR_23:    rb_data = 32'd1296;  // 1944 * 2/3
                         CR_34:    rb_data = 32'd1458;  // 1944 * 3/4 
                         default:  rb_data = 32'd1620;  // 1944 * 5/6 
                       endcase       
      R_2916DIVCRD:    case(cr)
                         CR_12:    rb_data = 32'd1458;
                         CR_23:    rb_data = 32'd972;
                         CR_34:    rb_data = 32'd729;
                         default:  rb_data = 32'd486;
                       endcase  
      R_1464DIVCRD:    case(cr)            
                         CR_12:    rb_data = 32'd732; 
                         CR_23:    rb_data = 32'd488; 
                         CR_34:    rb_data = 32'd366; 
                         default:  rb_data = 32'd244; 
                       endcase              
      R_912DIVCRD:     case(cr)            
                         CR_12:    rb_data = 32'd456; 
                         CR_23:    rb_data = 32'd304; 
                         CR_34:    rb_data = 32'd228; 
                         default:  rb_data = 32'd152; 
                       endcase 
      R_NPLD:          rb_data = {8'b0,npld};
      R_NAVBITS:       rb_data = {8'b0,navbits};
      R_NSHRT:         rb_data = {8'b0,nshrt};
      R_NPUNC:         rb_data = {8'b0,npunc};
      R_NCW:           rb_data = {20'd0,ncw};
      R_LLDPC:         case(lldpc)
                         2'd0:    rb_data = 32'd648;
                         2'd1:    rb_data = 32'd1296;
                         default: rb_data = 32'd1944;
                       endcase
      R_NSYMINIT:      rb_data = {16'b0,nsyminit};
      R_NSYM:          rb_data = {16'b0,    nsym};
      R_MXNCBPS:       rb_data = (stbc)?{15'd0,ncbps,1'b0}:{16'd0,ncbps};
      R_MXNDBPS:       rb_data = (stbc)?{15'd0,ndbps,1'b0}:{16'd0,ndbps};
      R_MXNDBPSSHORT:  rb_data = (stbc)?{15'd0,ndbps_short,1'b0}:{16'd0,ndbps_short};
      R_MXNCBPSSHORT:  rb_data = (stbc)?{15'd0,ncbps_short,1'b0}:{16'd0,ncbps_short};
      R_MSTBC:         rb_data = (stbc)?32'd2:32'd1;
      R_SPT:           case(nes)
                         2'd0:    rb_data = 32'd16;
                         2'd1:    rb_data = 32'd22;
                         default: rb_data = 32'd28;
                       endcase
      R_TSYM:          if(vhtsig_state)
                         rb_data = (vhtsig_gi_type==2'd0)?32'd18:32'd20; /* x10  */
                       else
                         case(hesig_gi_type)                   /* x30 */
                           GI_800:  rb_data = 32'd408;
                           GI_1600: rb_data = 32'd432;
                           default: rb_data = 32'd480;
                         endcase
      
      R_SGIDIS:        rb_data = {31'b0,vhtsig_sgidisamb & vhtsig_gi_type==2'd0};
      R_RXTIME:        rb_data = {19'b0,rxtime};
      R_36P4XNVHTLTF:  case(vhtsig_nsts_total)
                         NSTS_1:  rb_data = 32'd40; 
                         NSTS_2:  rb_data = 32'd44; 
                         NSTS_3:  rb_data = 32'd52; 
                         NSTS_4:  rb_data = 32'd52; 
                         NSTS_5:  rb_data = 32'd60;  
                         NSTS_6:  rb_data = 32'd60;  
                         NSTS_7:  rb_data = 32'd68;  
                         default: rb_data = 32'd68;  
                       endcase
      R_NDBPS:         rb_data = {16'b0,ndbps};
      R_NCBPS:         rb_data = {16'b0,ncbps};
      R_NSS:           rb_data = {28'b0,  nss};
      R_EXTRAFIX:      
      begin
        if(!vhtsig_fec || !vhtsig_extra) 
          rb_data=32'd0; /* BCC or LDPC NOEXTRA   */
        else if(!vhtsig_stbc)            
          rb_data=32'd1; /* LDPC EXTRA and NOSTBC */
        else                          
          rb_data=32'd2; /* LDPC EXTRA and STBC   */
      end
      R_NHELTF:
      begin
        case(hesig_nheltf)
          3'd0:     rb_data = 32'd1;
          3'd1:     rb_data = 32'd2;
          3'd3:     rb_data = 32'd4;
          3'd5:     rb_data = 32'd6;
          default:  rb_data = 32'd8;
        endcase
      end
      R_THELTF:
      begin
        case({hesig_gi_type,hesig_heltf_type})
          { GI_800, HELTF_1X}:  rb_data = 32'd120; // (0.8+1*3.2)*30
          { GI_800, HELTF_2X}:  rb_data = 32'd216; // (0.8+2*3.2)*30 
          {GI_1600, HELTF_2X}:  rb_data = 32'd240; // (1.6+2*3.2)*30 
          { GI_800, HELTF_4X}:  rb_data = 32'd408; // (0.8+4*3.2)*30 
          {GI_3200, HELTF_4X}:  rb_data = 32'd480; // (0.8+1*3.2)*30 
          default:              rb_data = 32'd0;
        endcase
      end

      /* l_length + m + 3 
       how to determine m ?
      
       l_length%3 = 2 for ERSU or MU
                  = 1 for SU or TB
      
       m has to be determined in such way that (l_length + m + 3)%3 =0
       so  m = 2 for TB or SU
           m = 1 for ERSU or MU */

      R_HEMPLUS3:      case(hesig_format)
                         HE_ER_SU,HE_MU: rb_data = 32'd4; // 3+1 (m=1)
                         default:        rb_data = 32'd5; // 3+2 (m=2)
                       endcase
     
      R_PEDISAMB:      rb_data = {31'd0,hesig_pedisamb};
      R_120:           rb_data = 32'd120;
      R_HESIGBNSYM:    rb_data = {24'd0,hesig_sigb_nsym};
      R_AINIT:         rb_data = {29'd0,ainit};

      R_MMAXTSYM:
      begin
        if(!hesig_midamble)
        begin
          case(hesig_gi_type)                   /* x30 to prevent rationnal */
            GI_800:  rb_data = 32'd4080;
            GI_1600: rb_data = 32'd4320;
            default: rb_data = 32'd4800;
          endcase
        end
        else
        begin
          case(hesig_gi_type)                   /* x30 to prevent rationnal */
            GI_800:  rb_data = 32'd8160;
            GI_1600: rb_data = 32'd8640;
            default: rb_data = 32'd9600;
          endcase
        end
      end
      
      R_PEDISP2XTSYM:
      begin
        if(!hesig_pedisamb)
        begin
          case(hesig_gi_type)                  
            GI_800:  rb_data = 32'd816; /* (0+2)*13.6*30 */
            GI_1600: rb_data = 32'd864; /* (0+2)*14.4*30 */
            default: rb_data = 32'd960; /* (0+2)*16.0*30 */
          endcase
        end
        else
        begin
          case(hesig_gi_type)                   
            GI_800:  rb_data = 32'd1224; /* (1+2)*13.6*30 */
            GI_1600: rb_data = 32'd1296; /* (1+2)*14.4*30 */
            default: rb_data = 32'd1440; /* (1+2)*16.0*30 */
          endcase
        end
      end
      
      R_NMA:   rb_data = {23'd0,nma};
      
      default: rb_data = 32'd0;
    endcase
  end

  /*****************************************************************************
  * register file
  *****************************************************************************/
  localparam W_NULL         = 6'd0,  W_R0         = 6'd1,         
             W_R1           = 6'd2,  W_R3         = 6'd3,         
             W_NCBPS        = 6'd4,  W_NDBPS      = 6'd5,      
             W_NSYM         = 6'd6,  W_RXTIME     = 6'd7,     
             W_PSDULEN      = 6'd8,  W_NSYMINIT   = 6'd9,   
             W_NPLD         = 6'd10, W_INCONS_LT  = 6'd11,   
             W_NAVBITS      = 6'd12, W_LLDPC648   = 6'd13,    
             W_LLDPC1296    = 6'd14, W_LLDPC1944  = 6'd15,   
             W_NCW          = 6'd16, W_NCW1       = 6'd17,        
             W_NCW2         = 6'd18, W_NSHRT      = 6'd19,       
             W_NPUNC        = 6'd20, W_NREPQR     = 6'd21,      
             W_NPUNCQR      = 6'd22, W_NSHRTQR    = 6'd23,     
             W_LLENGTHMOD3  = 6'd24, W_EXTRA      = 6'd25,       
             W_NCBPSSHORT   = 6'd26, W_TPE        = 6'd27,         
             W_NDBPSSHORT   = 6'd28, W_DONE       = 6'd29,
             W_NMA          = 6'd30;  
                              
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
       r0                <= 32'b0;              
       r1                <= 32'b0;                
       r3                <= 32'b0;               
       extra             <= 1'b0;
       nsyminit          <= 16'd0;            
       
       npld              <= 24'd0;          
       navbits           <= 24'd0;            
       lldpc             <= 2'd0;
       ncw               <= 12'd0;
       nshrt             <= 24'b0;
       npunc             <= 24'b0;            
      {nshrtq,nshrtr}    <= {12'b0,12'b0};
      {npuncq,npuncr}    <= {12'b0,12'b0};
      { nrepq, nrepr}    <= {14'b0,12'b0};
      
      inconsistent       <= 1'b0;
      ncbps              <= 16'b0;
      ndbps              <= 16'b0;
      ncbps_short        <= 16'b0;
      ndbps_short        <= 16'b0;
      nsym               <= 16'b0;
      psdulen            <= 26'b0;
      tpe                <= 3'b0;
      rxtime             <= 13'b0;
      l_length_mod3      <= 2'd0;
      nma                <= 9'd0;
      
      ndbps_done         <= 1'b0;
      l_done             <= 1'b0;
      ht_done            <= 1'b0;
      vht_done           <= 1'b0;
      he_done            <= 1'b0;
    end
    else if(!enable)
    begin
       r0                <= 32'b0;              
       r1                <= 32'b0;                
       r3                <= 32'b0;               
       extra             <= 1'b0;
       nsyminit          <= 16'd0;            
       
       npld              <= 24'd0;          
       navbits           <= 24'd0;            
       lldpc             <= 2'd0;
       ncw               <= 12'd0;
       nshrt             <= 24'b0;
       npunc             <= 24'b0;            
      {nshrtq,nshrtr}    <= {12'b0,12'b0};
      {npuncq,npuncr}    <= {12'b0,12'b0};
      { nrepq, nrepr}    <= {14'b0,12'b0};
      
      inconsistent       <= 1'b0;
      ncbps              <= 16'b0;
      ndbps              <= 16'b0;
      ncbps_short        <= 16'b0;
      ndbps_short        <= 16'b0;
      nsym               <= 16'b0;
      psdulen            <= 26'b0;
      tpe                <= 3'b0;
      rxtime             <= 13'b0;
      l_length_mod3      <= 2'd0;
      nma                <= 9'd0;
      
      ndbps_done         <= 1'b0;
      l_done             <= 1'b0;
      ht_done            <= 1'b0;
      vht_done           <= 1'b0;
      he_done            <= 1'b0;
    end
    else
    begin
      case(wb_addr)
        W_R0:    r0      <= wb_y;                 
        W_R1:    r1      <= wb_y;                 
        W_R3:    r3      <= wb_y;                 
        W_NCBPS: ncbps   <= wb_y[15:0]; 
        W_NDBPS:
        begin
         ndbps           <= wb_y[15:0];
         ndbps_done      <= 1'b1;
        end
        W_NSYM:
        begin
           nsym   <= wb_y[15:0];
           if(hesig_state && hesig_format!=HE_SU)
             inconsistent <= inconsistent | wb_y[31:0]==32'd0;
        end
        W_RXTIME: rxtime <= {wb_y[10:0],2'b0};
        W_NSYMINIT:
        begin
          if(htsig_state)
          begin
            if(htsig_stbc)
            begin
              /* mstbc=2 */
              nsyminit <= {wb_y[14:0],1'b0};
              nsym     <= {wb_y[14:0],1'b0};
            end
            else
            begin
              /* mstbc=1 */        
              nsyminit <= wb_y[15:0];
              nsym     <= wb_y[15:0];
            end
          end
          else
          begin
            nsyminit     <= wb_y[ 15:0];
          end
        end
        W_PSDULEN:
        begin
          inconsistent    <= inconsistent | wb_y[31] | wb_y[31:3]==29'd0;
          psdulen         <= wb_y[28:3];
        end
        W_INCONS_LT:   inconsistent    <= inconsistent | wb_y[31];
        W_NPLD:        npld            <= wb_y[23:0];         
        W_NAVBITS:     navbits         <= wb_y[23:0];             
        W_LLDPC648:    lldpc           <= 2'd0;
        W_LLDPC1296:   lldpc           <= 2'd1;
        W_LLDPC1944:   lldpc           <= 2'd2;
        W_NCW1:        ncw             <= 12'd1;
        W_NCW2:        ncw             <= 12'd2;
        W_NCW:         ncw             <= wb_y[11:0];
        W_NSHRT:       nshrt           <= wb_y[23:0];
        W_NPUNC:       npunc           <= wb_y[23:0];
        W_NSHRTQR:     {nshrtq,nshrtr} <= {wb_y[11:0],wb_x[11:0]};
        W_NPUNCQR:     {npuncq,npuncr} <= {wb_y[11:0],wb_x[11:0]};
        W_NREPQR:      { nrepq, nrepr} <= {wb_y[13:0],wb_x[11:0]};
        W_EXTRA:       extra           <= 1'b1;
        W_LLENGTHMOD3: l_length_mod3   <= wb_x[1:0];
        W_TPE: 
        begin
          inconsistent  <= inconsistent | wb_y[ 2:0]>=3'd5;
          tpe           <= wb_y[ 2:0];
        end
        W_NCBPSSHORT: ncbps_short <= wb_y[15:0];
        W_NDBPSSHORT: ndbps_short <= wb_y[15:0];
        W_NMA:        nma         <= wb_y[8:0];
        W_DONE:
        begin
          l_done        <= lsig_state;
          ht_done       <= htsig_state;
          vht_done      <= vhtsig_state;
          he_done       <= hesig_state;
        end
        default: /* do nothing */;
      endcase
    end
  end
 
  /*****************************************************************************
  * program
  *****************************************************************************/
  localparam   NOP  = 3'd0,  CLRXY  = 3'd1,
               ADD  = 3'd2,  SUB    = 3'd3,
               MUL  = 3'd4,  DIV    = 3'd5,
               TERM = 3'd7;
  
  localparam   I=1'b0, J=1'b1;
  
  wire [15:0]  p[0:310];

  localparam  L_NONHT          = 9'd3,
              L_NONHT_0        = 9'd18,
              L_NONHT_1        = 9'd24,
              L_HT             = 9'd28,
              L_HT_1           = 9'd41,
              L_VHT            = 9'd47,
              L_VHT_1          = 9'd70,
              L_HE             = 9'd82,
              L_HE_X1          = 9'd123,
              L_HE_X2          = 9'd125,
              L_HE_2           = 9'd137,
              L_HE_3           = 9'd142,
              L_HE_4           = 9'd155,
              L_HE_5           = 9'd165,
              L_HE_6           = 9'd180,
              L_NCW            = 9'd182,
              NCW_4            = 9'd191,
              NCW_41           = 9'd196,
              NCW_3            = 9'd198,
              NCW_2            = 9'd205,
              NCW_1            = 9'd208,
              NCW_0            = 9'd215,
              L_NSHR           = 9'd221,
              NSHR_0           = 9'd232,
              NSHR_1           = 9'd239,
              L_HT_DETEXTRA    = 9'd244,
              L_HE_ADDEXTRA    = 9'd264,
              L_HE_ADDEXTRA_AINITEQ3 = 9'd269,
              L_HT_ADDEXTRA    = 9'd275,
              L_NAVBITS_WRITE  = 9'd278,
              UPD_0            = 9'd285,
              UPD_1            = 9'd286,
              UPD_2            = 9'd292,
              L_END            = 9'd308;

  /***************************************************************************
  * 
  ***************************************************************************/
  assign p[              0] = {J,               CC_HT,           L_HT};
  assign p[              1] = {J,              CC_VHT,          L_VHT};
  assign p[              2] = {J,               CC_HE,           L_HE};
  /***************************************************************************
  * NONHT
  ***************************************************************************/
  assign p[        L_NONHT] = {I,      W_NULL,  CLRXY,     R_L_LENGTH};               
  assign p[              4] = {I,      W_NULL,    ADD,            R_3}; 
  assign p[              5] = {I,      W_NULL,    DIV,         R_NULL}; 
  assign p[              6] = {I,W_LLENGTHMOD3, CLRXY,        R_NBPSC}; // l_length_mod3 = l_length%3         
  assign p[              7] = {I,      W_NULL,    ADD,          R_NSD}; 
  assign p[              8] = {I,      W_NULL,    MUL,         R_NULL}; 
  assign p[              9] = {I,     W_NCBPS,    NOP,          R_CRN}; // ncbps = 48*nbpsc
  assign p[             10] = {I,      W_NULL,    MUL,          R_CRD};
  assign p[             11] = {I,      W_NULL,    DIV,         R_NULL};
  assign p[             12] = {I,     W_NDBPS,  CLRXY,   R_L_LENGTHX8}; // ndbps = ncbps*cr_n/cr_d
  assign p[             13] = {I,      W_NULL,    ADD,           R_22};
  assign p[             14] = {I,      W_NULL,    ADD,        R_NDBPS};
  assign p[             15] = {I,      W_NULL,    DIV,            R_1};
  assign p[             16] = {J,           CC_X_ZERO,      L_NONHT_0};
  assign p[             17] = {I,      W_NULL,    ADD,         R_NULL};
  assign p[      L_NONHT_0] = {I,      W_NSYM,  CLRXY,     R_L_LENGTH}; // nsym = ceil( (16+8*l_length+6) / ndbps ) 
  assign p[             19] = {I,      W_NULL,    ADD,            R_3}; 
  assign p[             20] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[             21] = {I,      W_NULL,    DIV,            R_1}; 
  assign p[             22] = {J,           CC_X_ZERO,      L_NONHT_1}; 
  assign p[             23] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[      L_NONHT_1] = {I,      W_NULL,    NOP,            R_5}; 
  assign p[             25] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[             26] = {I,    W_RXTIME,    NOP,         R_NULL}; // rxtime= 4*(ceil( (l_length+3)/3) + 5)
  assign p[             27] = {I,      W_DONE,   TERM,         R_NULL}; 
  /***************************************************************************
  * HT
  ***************************************************************************/
  assign p[           L_HT] = {I,      W_NULL,  CLRXY,        R_NBPSC};  
  assign p[             29] = {I,      W_NULL,    ADD,          R_NSD};
  assign p[             30] = {I,      W_NULL,    MUL,          R_NSS};
  assign p[             31] = {I,      W_NULL,    MUL,         R_NULL};
  assign p[             32] = {I,     W_NCBPS,    NOP,          R_CRN}; // ncbps = nbpsc*nsd*nss
  assign p[             33] = {I,      W_NULL,    MUL,          R_CRD}; 
  assign p[             34] = {I,      W_NULL,    DIV,         R_NULL}; 
  assign p[             35] = {I,     W_NDBPS,  CLRXY,  R_HT_LENGTHX8}; // ndbps = ncbps*crn/crd
  assign p[             36] = {I,      W_NULL,    ADD,          R_SPT};
  assign p[             37] = {I,      W_NULL,    ADD,      R_MXNDBPS}; 
  assign p[             38] = {I,      W_NPLD,    DIV,            R_1}; // npld = htlength*8+16+tail
  assign p[             39] = {J,           CC_X_ZERO,         L_HT_1}; 
  assign p[             40] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[         L_HT_1] = {I,  W_NSYMINIT,    NOP,         R_NULL}; // nsym = ceil(  npld/(mstbc*ndbps) )
  assign p[             42] = {J,              CC_BCC,          L_END}; 
  assign p[             43] = {I,      W_NULL,    NOP,      R_MXNCBPS}; 
  assign p[             44] = {I,      W_NULL,    MUL,         R_NULL}; 
  assign p[             45] = {I,   W_NAVBITS,    NOP,         R_NULL}; // navbits = npld*mstbc*mcbps
  assign p[             46] = {J,     CC_TRUE,                  L_NCW}; 
  /***************************************************************************
  * SUBPROG2: VHT NDBPS,NCBPS,NSYM
  ***************************************************************************/  
  assign p[          L_VHT] = {I,      W_NULL,  CLRXY,        R_NBPSC};
  assign p[             48] = {I,      W_NULL,    ADD,          R_NSD};
  assign p[             49] = {I,      W_NULL,    MUL,          R_NSS}; 
  assign p[             50] = {I,      W_NULL,    MUL,         R_NULL};
  assign p[             51] = {I,     W_NCBPS,    NOP,          R_CRN}; // ncbps = nbpsc*nsd*nss
  assign p[             52] = {I,      W_NULL,    MUL,          R_CRD};
  assign p[             53] = {I,      W_NULL,    DIV,         R_NULL};
  assign p[             54] = {I,     W_NDBPS,  CLRXY,       R_RXTIME}; // ndbps = ncbps*crn/crd
  assign p[             55] = {I,      W_NULL,    ADD, R_36P4XNVHTLTF};     
  assign p[             56] = {I,      W_NULL,    SUB,         R_NULL};     
  assign p[             57] = {I, W_INCONS_LT,    NOP,         R_NULL};       
  assign p[             58] = {I,        W_R0,    NOP,         R_NULL}; // r0 =  rxtime - (36+4*nvhtltf)             
  assign p[             59] = {I,      W_NULL,    NOP,         R_R0X4};    
  assign p[             60] = {I,      W_NULL,    ADD,         R_TSYM}; // y = 5*r0     
  assign p[             61] = {I,      W_NULL,    DIV,       R_SGIDIS}; // y = 5*(rxtime - (36+4*nvhtltf)) / (5*tsym)   
  assign p[             62] = {I,      W_NULL,    SUB,         R_NULL};
  assign p[             63] = {I, W_INCONS_LT,    NOP,         R_NULL};
  assign p[             64] = {I,      W_NSYM,    NOP,         R_NULL}; // nsym = floor( ( rxtime - (36+4*nvhtltf) ) / tsym ) - sgidisamb
  assign p[             65] = {J,               CC_NE,        L_VHT_1};       
  assign p[             66] = {I,      W_NULL,  CLRXY,         R_NULL}; // NDP
  assign p[             67] = {I,      W_NSYM,    NOP,         R_NULL}; 
  assign p[             68] = {I,  W_NSYMINIT,    NOP,         R_NULL}; 
  assign p[             69] = {J,             CC_TRUE,          L_END}; 
  assign p[        L_VHT_1] = {I,      W_NULL,    NOP,     R_EXTRAFIX}; // nsym = floor( ( rxtime - (36+4*nvhtltf) ) / tsym ) - sgidisamb
  assign p[             71] = {I,      W_NULL,    SUB,         R_NULL}; 
  assign p[             72] = {I, W_INCONS_LT,    NOP,        R_NDBPS};
  assign p[             73] = {I,  W_NSYMINIT,    MUL,          R_SPT}; // nsyminit = nsym - extrafix(0,1,2) 
  assign p[             74] = {I,      W_NPLD,    SUB,         R_NULL}; // npld     = nsyminit*ndbps
  assign p[             75] = {I,   W_PSDULEN,    NOP ,        R_NULL}; // psdulen = floor( (nsyminit*ndbps-(16+6*nes))/8 )
  assign p[             76] = {J,      CC_BCC,                  L_END}; 
  assign p[             77] = {I,      W_NULL,  CLRXY,     R_NSYMINIT}; 
  assign p[             78] = {I,      W_NULL,    ADD,        R_NCBPS}; 
  assign p[             79] = {I,      W_NULL,    MUL,         R_NULL}; 
  assign p[             80] = {I,   W_NAVBITS,    NOP,         R_NULL}; // navbits = nsyminit*ncbps
  assign p[             81] = {J,     CC_TRUE,                  L_NCW}; 
  /***************************************************************************
  * SUBPROG3: HE NDBPS,NCBPS,NSYM
  ***************************************************************************/
  assign p[           L_HE] = {I,      W_NULL,  CLRXY,        R_NBPSC}; // nes=1
  assign p[             83] = {I,      W_NULL,    ADD,          R_NSS}; 
  assign p[             84] = {I,      W_NULL,    MUL,          R_NSD}; 
  assign p[             85] = {I,      W_NULL,    MUL,         R_NULL}; 
  assign p[             86] = {I,     W_NCBPS,    NOP,          R_CRN}; // ncbps=nbpsc*nss*nsd
  assign p[             87] = {I,      W_NULL,    MUL,          R_CRD}; 
  assign p[             88] = {I,      W_NULL,    DIV,         R_NULL}; 
  assign p[             89] = {I,     W_NDBPS,  CLRXY,        R_NBPSC}; // ndbps=floor(ncbps*crn/crd)
  assign p[             90] = {I,      W_NULL,    ADD,          R_NSS};
  assign p[             91] = {I,      W_NULL,    MUL,     R_NSDSHORT}; 
  assign p[             92] = {I,      W_NULL,    MUL,         R_NULL}; 
  assign p[             93] = {I,W_NCBPSSHORT,    NOP,          R_CRN}; // ncbps_short=nbpsc*nss*nsd_short
  assign p[             94] = {I,      W_NULL,    MUL,          R_CRD};
  assign p[             95] = {I,      W_NULL,    DIV,         R_NULL}; 
  assign p[             96] = {I,W_NDBPSSHORT,    NOP,         R_NULL}; // ndbps_short=floor(ncbps_short*crn/crd)
  assign p[             97] = {I,      W_NULL,  CLRXY,       R_NHELTF}; 
  assign p[             98] = {I,      W_NULL,    ADD,       R_THELTF}; 
  assign p[             99] = {I,      W_NULL,    MUL,          R_480}; 
  assign p[            100] = {I,        W_R3,    ADD,         R_NULL}; // r3 = nheltf*theltf
  assign p[            101] = {I,        W_R1,  CLRXY,   R_HESIGBNSYM}; // r1 = nheltf*theltf + 16  (x30) 
  assign p[            102] = {I,      W_NULL,    ADD,          R_120}; 
  assign p[            103] = {I,      W_NULL,    MUL,           R_R1}; 
  assign p[            104] = {I,      W_NULL,    ADD,         R_NULL}; // y  = hesigb_nsym*4us
  assign p[            105] = {I,        W_R1,  CLRXY,           R_10}; // r1 = t_he_preamble + hesigb_nsym*4us;
  assign p[            106] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[            107] = {I,        W_R0,  CLRXY,     R_L_LENGTH}; // r0 = 10
  assign p[            108] = {I,      W_NULL,    ADD,     R_HEMPLUS3}; 
  assign p[            109] = {I,      W_NULL,    ADD,         R_R0X4};
  assign p[            110] = {I,      W_NULL,    MUL,           R_R1}; // y  = (l_length+m+3)*40
  assign p[            111] = {I,      W_NULL,    SUB,         R_NULL}; 
  assign p[            112] = {I,        W_R1,    NOP,         R_NULL}; // r1 = (l_length+m+3)*40 - t_he_preamble*30
  assign p[            113] = {I,      W_NULL,  CLRXY,           R_R3};
  assign p[            114] = {I,      W_NULL,    ADD,     R_MMAXTSYM};
  assign p[            115] = {I,      W_NULL,    ADD,         R_NULL};
  assign p[            116] = {I,        W_R0,  CLRXY,           R_R1}; // r0  = t_ma = mma*t_sym + nheltf*theltf
  assign p[            117] = {I,      W_NULL,    ADD, R_PEDISP2XTSYM};
  assign p[            118] = {I,      W_NULL,    SUB,           R_R0}; // y = (l_length+m+3)*40 - t_he_preamble*30 - (bpedis+2)*tsym*30
  assign p[            119] = {J,         CC_NMA_TEST,        L_HE_X1}; 
  assign p[            120] = {I,      W_NULL,  CLRXY,         R_NULL}; 
  assign p[            121] = {I,       W_NMA,    MUL,         R_NULL}; // nma = 0 if( !doppler || y<0)
  assign p[            122] = {J,             CC_TRUE,        L_HE_X2};
  assign p[        L_HE_X1] = {I,      W_NULL,    DIV,         R_NULL};
  assign p[            124] = {I,       W_NMA,    MUL,         R_NULL}; // nma = floor( [ (l_length+m+3)*40 - t_he_preamble*30 - (bpedis+2)*tsym*30 ] / t_ma)
  assign p[        L_HE_X2] = {I,      W_NULL,  CLRXY,           R_R3}; 
  assign p[            126] = {I,      W_NULL,    ADD,          R_NMA}; 
  assign p[            127] = {I,      W_NULL,    MUL,         R_NULL}; 
  assign p[            128] = {I,        W_R3,    NOP,         R_NULL}; 
  assign p[            129] = {I,      W_NULL,  CLRXY,           R_R1}; // r3 = nma*nheltf*theltf
  assign p[            130] = {I,      W_NULL,    ADD,           R_R3}; 
  assign p[            131] = {I,      W_NULL,    SUB,         R_NULL}; 
  assign p[            132] = {I,        W_R1,    NOP,         R_TSYM}; // r1 = (l_length+m+3)*40 - t_he_preamble*30 - nma*nheltf*theltf 
  assign p[            133] = {I, W_INCONS_LT,    DIV,            R_1}; 
  assign p[            134] = {J,     CC_NOT_PEDISAMB,         L_HE_2}; 
  assign p[            135] = {I,      W_NULL,    SUB,         R_NULL};
  assign p[            136] = {I, W_INCONS_LT,    NOP,         R_NULL}; // y = floor[ ( (l_length+m+3)*40 - t_he_preamble*30 ) / t_sym*30 ] - bpedis
  assign p[         L_HE_2] = {J,       CC_NE,                 L_HE_3}; 
  assign p[            138] = {I,      W_NULL,  CLRXY,            R_1}; // NDP
  assign p[            139] = {I,      W_NSYM,    ADD,         R_NULL}; // nsym = 0
  assign p[            140] = {I,       W_TPE,    NOP,         R_NULL}; // tpe  = 1 (4us)
  assign p[            141] = {I,      W_DONE,   TERM,         R_NULL}; // DONE
  assign p[         L_HE_3] = {I,      W_NSYM,    NOP,         R_TSYM}; // nsym = floor[ ( (l_length+m+3)*40 - t_he_preamble*30 ) / t_sym*30 ] - bpedis
  assign p[            143] = {I,      W_NULL,    MUL,         R_NULL}; 
  assign p[            144] = {I,        W_R0,    NOP,         R_NULL}; // r0   = nsym*t_sym*30
  assign p[            145] = {I,      W_NULL,  CLRXY,           R_R1}; 
  assign p[            146] = {I,      W_NULL,    ADD,           R_R0}; 
  assign p[            147] = {I,      W_NULL,    SUB,          R_120}; 
  assign p[            148] = {I, W_INCONS_LT,    NOP,         R_NULL}; // y    = (l_length+m+3)*40 - t_he_preamble*30 - nma*nheltf*theltf - nsym*t_sym*30
  assign p[            149] = {I,      W_NULL,    DIV,         R_NULL}; 
  assign p[            150] = {I,       W_TPE,    NOP,         R_NULL}; // tpe  = floor[ ( (l_length+m+3)*40 - t_he_preamble*30  - nma*nheltf*theltf - nsym*t_sym*30 ) / 120 ]
/* NSYMINIT */
  assign p[            151] = {I,      W_NULL,  CLRXY,         R_NSYM}; 
  assign p[            152] = {I,      W_NULL,    ADD,        R_MSTBC}; 
  assign p[            153] = {J, CC_NOT_A_EQ1_EXTRA,          L_HE_4};
  assign p[            154] = {I,      W_NULL,    SUB,         R_NULL}; 
  assign p[         L_HE_4] = {I,  W_NSYMINIT,    NOP,        R_NDBPS}; // nsyminit = nsym-mstbc if(extra && fec && a==1)
/* NPLD, PSDULEN */
  assign p[            156] = {I,      W_NULL,    MUL,      R_MXNDBPS}; // y    = nsym * ndbps 
  assign p[            157] = {J,CC_AINIT_EQ4,                 L_HE_5};
  assign p[            158] = {I,      W_NULL,    SUB,         R_NULL}; 
  assign p[            159] = {I,      W_NULL,    NOP, R_MXNDBPSSHORT}; 
  assign p[            160] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[            161] = {J,CC_AINIT_EQ1,                 L_HE_5};
  assign p[            162] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[            163] = {J,CC_AINIT_EQ2,                 L_HE_5};
  assign p[            164] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[         L_HE_5] = {I,      W_NPLD,    NOP,          R_SPT}; // if(a==4) npld = nsyminit*ndbps  else npld = (nsyminit-mstbc)*ndbps+mstbc*ainit*ndbps_short
  assign p[            166] = {I,      W_NULL,    SUB,         R_NULL}; 
  assign p[            167] = {I,   W_PSDULEN,    NOP,         R_NULL}; // psdulen = floor( (fieldlen-16-tail)/8); note: round done at writeback
  assign p[            168] = {J,      CC_BCC,                  L_END}; 
/* NAVBITS */
  assign p[            169] = {I,      W_NULL,  CLRXY,     R_NSYMINIT}; 
  assign p[            170] = {I,      W_NULL,    ADD,        R_NCBPS}; 
  assign p[            171] = {I,      W_NULL,    MUL,      R_MXNCBPS};
  assign p[            172] = {J,CC_AINIT_EQ4,                 L_HE_6};
  assign p[            173] = {I,      W_NULL,    SUB,         R_NULL}; 
  assign p[            174] = {I,      W_NULL,    NOP, R_MXNCBPSSHORT}; 
  assign p[            175] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[            176] = {J,CC_AINIT_EQ1,                 L_HE_6};
  assign p[            177] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[            178] = {J,CC_AINIT_EQ2,                 L_HE_6};
  assign p[            179] = {I,      W_NULL,    ADD,         R_NULL}; 
  assign p[         L_HE_6] = {I,   W_NAVBITS,    NOP,         R_NULL}; //if(a==4) navbits = nsyminit*ncbps  else navbits = (nsyminit-mstbc)*ncbps+mstbc*ainit*ncbps_short
  assign p[            181] = {I, W_INCONS_LT,    NOP,         R_NULL}; 
  /***************************************************************************
  * HT/VHT/HE LDPC COMMON
  ****************************************************************************
  * ncw, lldpc, nshrt, npunc(1st pass) ,extrau(final) and nsymu(final)
  * note: navbits and npunc are **not** updated by extra, this is done into
  *       another sub-progr.
  ***************************************************************************/
  /* NCW, LLDPC */
  assign p[          L_NCW] = {I,      W_NULL,    NOP,          R_648}; 
  assign p[            183] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-648  
  assign p[            184] = {J,               CC_LE,          NCW_0}; // JMP NCW0 IF NAVBITS<=648
  assign p[            185] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-1296    
  assign p[            186] = {J,               CC_LE,          NCW_1}; // JMP NCW1 IF NAVBITS<=1296
  assign p[            187] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-1944     
  assign p[            188] = {J,               CC_LE,          NCW_2}; // JMP NCW2 IF NAVBITS<=1944
  assign p[            189] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-2592      
  assign p[            190] = {J,               CC_LE,          NCW_3}; // JMP NCW3 IF NAVBITS<=2592
  /* IF 2592 < NAVBITS */
  assign p[          NCW_4] = {I, W_LLDPC1944,  CLRXY,         R_NPLD}; // LLDPC=1944 ;   X=Y=0
  assign p[            192] = {I,      W_NULL,    ADD,      R_1944XCR}; // Y=NPLD;        B=1944*CR
  assign p[            193] = {I,      W_NULL,    DIV,            R_1}; // Y=NPLD/1944/R; B=1
  assign p[            194] = {J,           CC_X_ZERO,         NCW_41}; // JMP  IF X==0
  assign p[            195] = {I,      W_NULL,    ADD,         R_NULL}; // Y=NPLD/1944/R+1 
  assign p[         NCW_41] = {I,       W_NCW,    NOP,         R_NULL}; // NCW=Y;
  assign p[            197] = {J,             CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF 1944 < NAVBITS <= 2592 */
  assign p[          NCW_3] = {I,      W_NCW2,  CLRXY,      R_NAVBITS}; // NCW=2; X=Y=0 ; B=NAVBITS
  assign p[            199] = {I, W_LLDPC1944,    ADD,         R_NPLD}; // LLDPC=1944; Y=NAVBITS;  B=NPLD
  assign p[            200] = {I,      W_NULL,    SUB,   R_2916DIVCRD}; // Y=NAVBITS-NPLD;         B=2916*(1-CR)
  assign p[            201] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-NPLD-2616*(1-CR)
  assign p[            202] = {J,               CC_GE,         L_NSHR}; // JMP L_NSHR IF NAVBITS >= (NPLD+2616*(1-CR))     
  assign p[            203] = {I, W_LLDPC1296,    NOP,         R_NULL}; // LLDPC=1296  
  assign p[            204] = {J,             CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF 1296 < NAVBITS <= 1944 */
  assign p[          NCW_2] = {I,      W_NCW1,    NOP,         R_NULL}; // NCW=1 ;
  assign p[            206] = {I, W_LLDPC1944,    NOP,         R_NULL}; // LLDPC=2 [1944] ;
  assign p[            207] = {J,             CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF 648  < NAVBITS <= 1296 */
  assign p[          NCW_1] = {I,      W_NCW1,  CLRXY,      R_NAVBITS}; // NCW=1 ; X=Y=0 ; B=NAVBITS
  assign p[            209] = {I, W_LLDPC1944,    ADD,         R_NPLD}; // LLDPC=1944; Y=NAVBITS; B=NPLD 
  assign p[            210] = {I,      W_NULL,    SUB,   R_1464DIVCRD}; // Y=NAVBITS-NPLD;        B=1464*(1-CR)
  assign p[            211] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-NPLD-1464*(1-R)    
  assign p[            212] = {J,               CC_GE,         L_NSHR}; // JMP L_NSHR IF NAVBITS >= (NPLD+1464*(1-CR)) 
  assign p[            213] = {I, W_LLDPC1296,    NOP,         R_NULL}; // LLDPC=1296   
  assign p[            214] = {J,             CC_TRUE,         L_NSHR}; // JMP L_NSHR
  /* IF NAVBITS <= 648  */
  assign p[          NCW_0] = {I,      W_NCW1,  CLRXY,      R_NAVBITS}; // NCW=1 ; X=Y=0 ;          B=NAVBITS
  assign p[            216] = {I, W_LLDPC1296,    ADD,         R_NPLD}; // LLDPC=1296; Y=NAVBITS;   B=NPLD
  assign p[            217] = {I,      W_NULL,    SUB,    R_912DIVCRD}; // Y=NAVBITS-NPLD;          B=912*(1-CR)
  assign p[            218] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-NPLD-912*(1-CR)
  assign p[            219] = {J,               CC_GE,         L_NSHR}; // JMP L_NSHR   
  assign p[            220] = {I,  W_LLDPC648,    NOP,         R_NULL}; // LLDPC=648  
  /* NSHRT */
  assign p[         L_NSHR] = {I,      W_NULL,  CLRXY,          R_NCW}; // X=Y=0;      B=NCW
  assign p[            222] = {I,      W_NULL,    ADD,        R_LLDPC}; // Y=NCW;      B=LLDPC
  assign p[            223] = {I,      W_NULL,    MUL,         R_NULL}; // Y=NCW*LLDPC
  /*    R0=NCW*LLDPC                        [ used several times afterwards ] */
  assign p[            224] = {I,        W_R0,    NOP,          R_CRD}; // R0=Y; B=CRD
  /*    R1=NCW*LLDPC*(1-CR)=NCW*LLDPC/CRD   [ used several times afterwards ] */
  assign p[            225] = {I,      W_NULL,    DIV,         R_NULL}; // Y=NCW*LLDPC/CRD
  assign p[            226] = {I,        W_R1,    NOP,          R_CRN}; // R1=Y; B=CRN
  /* NSHRT=MAX(0,NCW*LLDPC*CR-NPLD) */
  assign p[            227] = {I,      W_NULL,    MUL,         R_NULL}; // Y=NCW*LLDPC/CRD
  assign p[            228] = {I,      W_NULL,    NOP,         R_NPLD}; // Y=NCW*LLDPC/CRD*CRN;  B=NPLD
  assign p[            229] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NCW*LLDPC/CRD*CRN-NPLD
  assign p[            230] = {J,               CC_GE,         NSHR_0}; // JMP LA_9 NCW*LLDPC/CRD*CRN-NPLD >= 0
  assign p[            231] = {I,      W_NULL,  CLRXY,         R_NULL}; // X=Y=0
  assign p[         NSHR_0] = {I,     W_NSHRT,    NOP,         R_NULL}; // NSHRT=Y
  /* NPUNC=MAX(0,NCW*LLDPC-NAVBITS-NSHRT) */
  assign p[            233] = {I,      W_NULL,  CLRXY,           R_R0}; // B=NCW*LLDPC;
  assign p[            234] = {I,      W_NULL,    ADD,      R_NAVBITS}; // Y=NCW*LLDPC;  B=NAVBITS
  assign p[            235] = {I,      W_NULL,    SUB,        R_NSHRT}; // Y=NCW*LLDPC-NAVBITS;  B=NSHRT
  assign p[            236] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NCW*LLDPC-NAVBITS-NSHRT
  assign p[            237] = {J,               CC_GE,         NSHR_1}; // JMP LA_10 NCW*LLDPC-NAVBITS-NSHRT >= 0
  assign p[            238] = {I,      W_NULL,  CLRXY,         R_NULL}; // X=Y=0;
  /* R3=10*NPUNC                            [ used several times afterwards ] */
  assign p[         NSHR_1] = {I,     W_NPUNC,    NOP,           R_10}; // NPUNC=Y;  B=10 ;
  /***************************************************************************
  * HT extra bit generation: 
  *  for HT,  determine the extra bit
  *  for VHT, the bit is already provided by VHTSIGA                  
  *  for HE,  the bit is already provided by VHTSIGA                  
  ***************************************************************************/  
  assign p[            240] = {J,               CC_HT,  L_HT_DETEXTRA}; // JMP L_HT_DETEXTRA IF HT
  assign p[            241] = {J,    CC_VHTSIGA_EXTRA,  L_HT_ADDEXTRA}; // JMP L_HT_ADDEXTRA IF CC_VHTSIGA_EXTRA
  assign p[            242] = {J,     CC_HESIGA_EXTRA,  L_HE_ADDEXTRA}; // JMP L_HE_ADDEXTRA IF CC_HESIGA_EXTRA
  assign p[            243] = {J,             CC_TRUE,          UPD_1}; // JMP UPD_1 
  /* IF NPUNC > 0.3*NCW*LLDPC*(1-CR) THEN ADD EXTRAU */
  assign p[  L_HT_DETEXTRA] = {I,      W_NULL,    MUL,         R_NULL}; // Y=NPUNC*10;
  assign p[            245] = {I,        W_R3,  CLRXY,           R_R1}; // R3=10*NPUNC ; X=Y=0; B=NCW*LLDPC*(1-CR) 
  assign p[            246] = {I,      W_NULL,    ADD,         R_NULL}; // Y=NCW*LLDPC*(1-CR)
  assign p[            247] = {I,      W_NULL,    ADD,         R_NULL}; // Y=2*NCW*LLDPC*(1-CR)
  assign p[            248] = {I,      W_NULL,    ADD,           R_R3}; // Y=2*NCW*LLDPC*(1-CR); B=10*NPUNC
  assign p[            249] = {I,      W_NULL,    SUB,         R_NULL}; // Y=3*NCW*LLDPC*(1-CR)-10*NPUNC
  assign p[            250] = {J,               CC_LT,  L_HT_ADDEXTRA}; // JMP ADDEXT IF 10*NPUNC > 3*NCW*LLDPC*(1-CR)
  /* IF !(NPUNC > 0.1*NCW*LLDPC(1-CR)) THEN NO EXTRAU  IS REWRITTEN AS */
  /* IF NPUNC <= 0.1*NCW*LLDPC(1-CR) THEN NO EXTRAU   */
  assign p[            251] = {I,      W_NULL,  CLRXY,           R_R1}; // X=Y=0;  B=NCW*LLDPC*(1-CR)
  assign p[            252] = {I,      W_NULL,    ADD,           R_R3}; // Y=NCW*LLDPC*(1-CR); B=10*NPUNC  
  assign p[            253] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NCW*LLDPC*(1-CR)-10*NPUNC
  assign p[            254] = {J,               CC_GE,          UPD_1}; // JMP NOEXT [NOEXTRAU]
  assign p[            255] = {I,      W_NULL,  CLRXY,        R_NSHRT}; // X=Y=0 ; B=NSHRT 
  assign p[            256] = {I,      W_NULL,    ADD,           R_10}; // Y=NSHRT ; B=10
  assign p[            257] = {I,      W_NULL,    MUL,         R_NULL}; // Y=NSHRT*10;
  /*   R3 = 10*NSHRT   */
  assign p[            258] = {I,        W_R3,  CLRXY,        R_NPUNC}; // R3=10*NSHRT ; X=Y=0; B=NPUNC
  assign p[            259] = {I,      W_NULL,    ADD,       R_12XCRN}; // Y=NPUNC ; B=12*CR/(1-CR)
  assign p[            260] = {I,      W_NULL,    MUL,           R_R3}; // Y=12*NPUNC*R/(1-R) ; B=10*NSHRT
  assign p[            261] = {I,      W_NULL,    SUB,         R_NULL}; // Y=12*NPUNC*R/(1-R)-10*NSHRT
  /* IF NSHRT < 1.2*NPUNC*R/(1-R) THEN ADD EXTRAU  */
  assign p[            262] = {J,               CC_GT,  L_HT_ADDEXTRA}; // JMP L_HT_ADDEXTRA
  assign p[            263] = {J,             CC_TRUE,          UPD_1}; // JMP UPD_1
  /***************************************************************************
  * UPDATE NAVBITS AND NBPUNC (HE ONLY)
  ***************************************************************************/
  assign p[  L_HE_ADDEXTRA] = {I,      W_NULL,  CLRXY,      R_NAVBITS}; // X=Y=0;        B=NAVBITS
  assign p[            265] = {J,CC_AINIT_EQ3, L_HE_ADDEXTRA_AINITEQ3};
  assign p[            266] = {I,      W_NULL,  ADD,   R_MXNCBPSSHORT};
  assign p[            267] = {I,      W_NULL,  ADD,           R_NULL};
  assign p[            268] = {J,           CC_TRUE,  L_NAVBITS_WRITE}; // JMP UPD_1
  assign p[L_HE_ADDEXTRA_AINITEQ3] = {I,      W_NULL,  ADD,        R_MXNCBPS};
  assign p[            270] = {I,      W_NULL,  ADD,   R_MXNCBPSSHORT};
  assign p[            271] = {I,      W_NULL,  SUB,           R_NULL};
  assign p[            272] = {I,      W_NULL,  SUB,           R_NULL};
  assign p[            273] = {I,      W_NULL,  SUB,           R_NULL}; // y=navbits + mstbc*(ncbps-3*ncbps_short)
  assign p[            274] = {J,           CC_TRUE,  L_NAVBITS_WRITE};
  /***************************************************************************
  * UPDATE NAVBITS AND NBPUNC (HT AND VHT ONLY)
  ***************************************************************************/
  /* EXTRA SYMBLOL, UPDATE NAVBITS AND NPUNC (HT AND VHT ONLY) */
  assign p[  L_HT_ADDEXTRA] = {I,     W_EXTRA,  CLRXY,      R_NAVBITS}; // X=Y=0;        B=NAVBITS
  assign p[            276] = {I,      W_NULL,    ADD,      R_MXNCBPS}; // Y=NAVBITS;    B=NCBPS*MSTBC
  assign p[            277] = {I,      W_NULL,    ADD,         R_NULL}; // Y=NAVBITS+NCBPS*MSTBC
  /* NAVBITS */
  assign p[L_NAVBITS_WRITE] = {I,   W_NAVBITS,    NOP,         R_NULL}; // NAVBITS=Y
  assign p[            279] = {I,      W_NULL,  CLRXY,           R_R0}; // X=Y=0; B=NCW*LLDPC;
  assign p[            280] = {I,      W_NULL,    ADD,      R_NAVBITS}; // Y=NCW*LLDPC;         B=NAVBITS
  assign p[            281] = {I,      W_NULL,    SUB,        R_NSHRT}; // Y=NCW*LLDPC-NAVBITS; B=NSHRT
  assign p[            282] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NCW*LLDPC-NAVBITS-NSHRT
  assign p[            283] = {J,               CC_GE,          UPD_0}; // JMP SP3_0 IF NCW*LLDPC-NAVBITS-NSHRT >=0 
  assign p[            284] = {I,      W_NULL,  CLRXY,         R_NULL}; // X=Y=0;
  assign p[          UPD_0] = {I,     W_NPUNC,    NOP,         R_NULL}; // NPUNC=Y;
  /***************************************************************************
  * NREPQ,NREPR, NPUNCQ,NPUNCR, NSHRTQ,NSHRTR
  ***************************************************************************/
  /* NREP */
  assign p[          UPD_1] = {I,      W_NULL,  CLRXY,      R_NAVBITS}; // X=Y=0; B=NAVBITS
  assign p[            287] = {I,      W_NULL,    ADD,           R_R1}; // Y=NAVBITS;                  B=NCW*LLDPC*(1-CR)
  assign p[            288] = {I,      W_NULL,    SUB,         R_NPLD}; // Y=NAVBITS-NCW*LLDPC*(1-CR); B=NPLD
  assign p[            289] = {I,      W_NULL,    SUB,         R_NULL}; // Y=NAVBITS-NCW*LLDPC*(1-CR)-NPLD
  assign p[            290] = {J,               CC_GE,          UPD_2}; // JMP SP4_0 IF NAVBITS-NCW*LLDPC*(1-CR)-NPLD>=0     
  assign p[            291] = {I,      W_NULL,  CLRXY,         R_NULL}; // X=Y=0
  assign p[          UPD_2] = {I,      W_NULL,    NOP,          R_NCW}; // B=NCW
  assign p[            293] = {I,      W_NULL,    DIV,         R_NULL}; // Y=NREP/NCW
  /* NREPQ,  NREPR */
  assign p[            294] = {I,    W_NREPQR,  CLRXY,        R_NPUNC}; // {NREPQ,NREPR}={X,Y}; X=Y=0; B=NPUNC
  assign p[            295] = {I,      W_NULL,    ADD,          R_NCW}; // Y=NPUNC; B=NCW
  assign p[            296] = {I,      W_NULL,    DIV,         R_NULL}; // Y=NPUNC/NCW
  /* NPUNCQ, NPUNCR */
  assign p[            297] = {I,   W_NPUNCQR,  CLRXY,        R_NSHRT}; // {NPUNCQ,NPUNCR}={X,Y}; X=Y=0; B=NSHRT
  assign p[            298] = {I,      W_NULL,    ADD,          R_NCW}; // Y=NSHRT; B=NCW
  assign p[            299] = {I,      W_NULL,    DIV,         R_NULL}; // Y=NSHRT/NCW
  /* NSHRTQ, NSHRTR */
  assign p[            300] = {I,   W_NSHRTQR,  CLRXY,         R_NULL}; // {NSHRTQ,NSHRTR}={X,Y}; X=Y=0; 
  assign p[            301] = {J,              CC_VHT,          L_END}; // JUMP TO L_END IF VHT
  assign p[            302] = {J,               CC_HE,          L_END}; // JUMP TO L_END IF HE
  /* UPDATE HT_NSYM */
  assign p[            303] = {J,          CC_NOEXTRA,          L_END}; // JMP H_END IF NOEXTRA
  assign p[            304] = {I,     W_NULL,   CLRXY,     R_NSYMINIT}; // Y=0; B=NSYMINIT
  assign p[            305] = {I,     W_NULL,     ADD,        R_MSTBC}; // Y=NSYMINIT;  B=MSTBC
  assign p[            306] = {I,     W_NULL,     ADD,         R_NULL}; // Y=MSTBC+NSYMINIT
  /* HT_NSYM */
  assign p[            307] = {I,     W_NSYM,     NOP,         R_NULL}; // HT_NSYM=MSTBC+NSYMINIT
  assign p[          L_END] = {I,     W_DONE,    TERM,         R_NULL}; // HT_DONE=1
  /***************************************************************************
  * NULL line because of prefecth
  ***************************************************************************/
  assign p[            309] = {I,     W_NULL,   TERM,          R_NULL}; //
  assign p[            310] = {I,     W_NULL,   TERM,          R_NULL}; //


  assign pr_data  = p[pr_addr];
  
  
`ifdef RW_SIMU_ON
  /*****************************************************************************
  * for debug/development
  *****************************************************************************/
  // pragma coverage block = off, expr = off, toggle = off
  reg [8:0] op_pc_s;
  reg       op_drop_s;

  always @(posedge clk,negedge rst_n)
    if(!rst_n)
    begin
      op_pc_s   <= 'd0;
      op_drop_s <= 'b0;
    end
    else if(!u_ucpu.enable || u_ucpu.done)
    begin
      op_pc_s   <= 'd0;
      op_drop_s <= 'b0;
    end
    else
    begin
      op_drop_s <= u_ucpu.op_jmp && u_ucpu.cc_flag;
      op_pc_s   <= u_ucpu.pc;
    end

  reg [32*8-1:0] op_wb_s;
  reg [32*8-1:0] op_rb_s;
  reg [32*8-1:0] op_alu_s;
  reg [32*8-1:0] op_cond_s;
  reg [ 8:0]     op_target_s;

  always @(*)
  begin
    op_wb_s     = "XXXXXXXX";
    op_alu_s    = "XXXXXXXX";
    op_rb_s     = "XXXXXXXX";
    op_cond_s   = "XXXXXXXX";
    op_target_s = 0;

    if(!u_ucpu.done)
    begin
      if(op_drop_s)
      begin
        op_wb_s     = " ";
        op_alu_s    = "DROP";
        op_rb_s     = " ";
        op_target_s = 0;
        op_cond_s   = " ";
      end
      else if(u_ucpu.op_jmp)
      begin
        op_wb_s     = " ";
        op_alu_s    = " ";
        op_rb_s     = " ";
        op_target_s = u_ucpu.op_target;
        case(u_ucpu.op_cond)
          CC_LE: op_cond_s = "CC_LE";
          CC_X_ZERO: op_cond_s = "CC_X_ZERO";
          CC_VHTSIGA_EXTRA: op_cond_s = "CC_VHTSIGA_EXTRA";
          CC_HT: op_cond_s = "CC_HT";
          CC_HESIGA_EXTRA: op_cond_s = "CC_HESIGA_EXTRA";
          CC_NMA_TEST: op_cond_s = "CC_NMA_TEST";
          CC_VHT: op_cond_s = "CC_VHT";
          CC_BCC: op_cond_s = "CC_BCC";
          CC_NOEXTRA: op_cond_s = "CC_NOEXTRA";
          CC_AINIT_EQ3: op_cond_s = "CC_AINIT_EQ3";
          CC_GT: op_cond_s = "CC_GT";
          CC_HE: op_cond_s = "CC_HE";
          CC_AINIT_EQ2: op_cond_s = "CC_AINIT_EQ2";
          CC_TRUE: op_cond_s = "CC_TRUE";
          CC_NOT_A_EQ1_EXTRA: op_cond_s = "CC_NOT_A_EQ1_EXTRA";
          CC_NE: op_cond_s = "CC_NE";
          CC_NOT_PEDISAMB: op_cond_s = "CC_NOT_PEDISAMB";
          CC_AINIT_EQ4: op_cond_s = "CC_AINIT_EQ4";
          CC_AINIT_EQ1: op_cond_s = "CC_AINIT_EQ1";
          CC_LT: op_cond_s = "CC_LT";
          CC_GE: op_cond_s = "CC_GE";
          default: op_cond_s = "XXXXXX";
        endcase
      end
      else
      begin
        op_cond_s = " ";
        case (u_ucpu.op_wb)
          W_TPE: op_wb_s = "W_TPE";
          W_NMA: op_wb_s = "W_NMA";
          W_R3: op_wb_s = "W_R3";
          W_NULL: op_wb_s = "W_NULL";
          W_NCW: op_wb_s = "W_NCW";
          W_LLDPC648: op_wb_s = "W_LLDPC648";
          W_R1: op_wb_s = "W_R1";
          W_NPUNCQR: op_wb_s = "W_NPUNCQR";
          W_NREPQR: op_wb_s = "W_NREPQR";
          W_EXTRA: op_wb_s = "W_EXTRA";
          W_NPUNC: op_wb_s = "W_NPUNC";
          W_NDBPSSHORT: op_wb_s = "W_NDBPSSHORT";
          W_NSYM: op_wb_s = "W_NSYM";
          W_NSHRTQR: op_wb_s = "W_NSHRTQR";
          W_NDBPS: op_wb_s = "W_NDBPS";
          W_NSHRT: op_wb_s = "W_NSHRT";
          W_INCONS_LT: op_wb_s = "W_INCONS_LT";
          W_NPLD: op_wb_s = "W_NPLD";
          W_NCW1: op_wb_s = "W_NCW1";
          W_RXTIME: op_wb_s = "W_RXTIME";
          W_LLDPC1944: op_wb_s = "W_LLDPC1944";
          W_R0: op_wb_s = "W_R0";
          W_NSYMINIT: op_wb_s = "W_NSYMINIT";
          W_PSDULEN: op_wb_s = "W_PSDULEN";
          W_NCBPSSHORT: op_wb_s = "W_NCBPSSHORT";
          W_NCW2: op_wb_s = "W_NCW2";
          W_DONE: op_wb_s = "W_DONE";
          W_NCBPS: op_wb_s = "W_NCBPS";
          W_NAVBITS: op_wb_s = "W_NAVBITS";
          W_LLENGTHMOD3: op_wb_s = "W_LLENGTHMOD3";
          W_LLDPC1296: op_wb_s = "W_LLDPC1296";
          default:             op_wb_s = "XXXXXX";
        endcase
        case(u_ucpu.op_rb)
          R_NSYM: op_rb_s = "R_NSYM";
          R_CRD: op_rb_s = "R_CRD";
          R_912DIVCRD: op_rb_s = "R_912DIVCRD";
          R_MSTBC: op_rb_s = "R_MSTBC";
          R_480: op_rb_s = "R_480";
          R_5: op_rb_s = "R_5";
          R_648: op_rb_s = "R_648";
          R_NSDSHORT: op_rb_s = "R_NSDSHORT";
          R_1: op_rb_s = "R_1";
          R_10: op_rb_s = "R_10";
          R_2916DIVCRD: op_rb_s = "R_2916DIVCRD";
          R_SGIDIS: op_rb_s = "R_SGIDIS";
          R_RXTIME: op_rb_s = "R_RXTIME";
          R_MXNCBPS: op_rb_s = "R_MXNCBPS";
          R_MXNDBPS: op_rb_s = "R_MXNDBPS";
          R_CRN: op_rb_s = "R_CRN";
          R_L_LENGTHX8: op_rb_s = "R_L_LENGTHX8";
          R_MXNDBPSSHORT: op_rb_s = "R_MXNDBPSSHORT";
          R_HESIGBNSYM: op_rb_s = "R_HESIGBNSYM";
          R_22: op_rb_s = "R_22";
          R_MMAXTSYM: op_rb_s = "R_MMAXTSYM";
          R_R1: op_rb_s = "R_R1";
          R_NSHRT: op_rb_s = "R_NSHRT";
          R_36P4XNVHTLTF: op_rb_s = "R_36P4XNVHTLTF";
          R_LLDPC: op_rb_s = "R_LLDPC";
          R_MXNCBPSSHORT: op_rb_s = "R_MXNCBPSSHORT";
          R_NULL: op_rb_s = "R_NULL";
          R_NMA: op_rb_s = "R_NMA";
          R_NSD: op_rb_s = "R_NSD";
          R_1464DIVCRD: op_rb_s = "R_1464DIVCRD";
          R_SPT: op_rb_s = "R_SPT";
          R_NCW: op_rb_s = "R_NCW";
          R_NPLD: op_rb_s = "R_NPLD";
          R_R0: op_rb_s = "R_R0";
          R_L_LENGTH: op_rb_s = "R_L_LENGTH";
          R_NAVBITS: op_rb_s = "R_NAVBITS";
          R_NSYMINIT: op_rb_s = "R_NSYMINIT";
          R_R3: op_rb_s = "R_R3";
          R_NCBPS: op_rb_s = "R_NCBPS";
          R_TSYM: op_rb_s = "R_TSYM";
          R_NHELTF: op_rb_s = "R_NHELTF";
          R_12XCRN: op_rb_s = "R_12XCRN";
          R_HT_LENGTHX8: op_rb_s = "R_HT_LENGTHX8";
          R_NBPSC: op_rb_s = "R_NBPSC";
          R_THELTF: op_rb_s = "R_THELTF";
          R_PEDISP2XTSYM: op_rb_s = "R_PEDISP2XTSYM";
          R_NPUNC: op_rb_s = "R_NPUNC";
          R_1944XCR: op_rb_s = "R_1944XCR";
          R_120: op_rb_s = "R_120";
          R_NSS: op_rb_s = "R_NSS";
          R_EXTRAFIX: op_rb_s = "R_EXTRAFIX";
          R_HEMPLUS3: op_rb_s = "R_HEMPLUS3";
          R_3: op_rb_s = "R_3";
          R_R0X4: op_rb_s = "R_R0X4";
          R_NDBPS: op_rb_s = "R_NDBPS";
          default:        op_rb_s = "XXXXXX";
        endcase
        case (u_ucpu.op_alu)
          NOP:     op_alu_s = "NOP";
          CLRXY:   op_alu_s = "CLRXY";
          ADD:     op_alu_s = "ADD";
          SUB:     op_alu_s = "SUB";
          MUL:     op_alu_s = "MUL";
          DIV:     op_alu_s = "DIV";
          TERM:    op_alu_s = "TERM";
        endcase
      end
    end
  end
  // pragma coverage block = on, expr = on, toggle = on

`endif
endmodule

