/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module rx_bd_deinterleaver
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire        rst_n,
  input  wire        clk,

  /*****************************************************************************
  * symbol/control
  *****************************************************************************/
  input  wire        enable,
  
  input  wire [ 2:0] nss,
  input  wire [ 1:0] nsd,
  input  wire [ 2:0] nbpsc,
  input  wire        ruen,
  input  wire [ 2:0] rulen,
  input  wire        dcm,
  input  wire        fec,
  input  wire        qbpsk,
  input  wire        sigben,
  input  wire        sel,
  
  /*****************************************************************************
  *
  *****************************************************************************/
  input  wire [ 1:0] chan0_sel,
  input  wire        chan0_start,
  output reg         chan0_busy,
  input  wire [ 1:0] chan1_sel,
  input  wire        chan1_start,
  output reg         chan1_busy,
 
  input  wire        ready,
  input  wire        chansel,
  output wire [ 4:0] sb0,  
  output wire [ 4:0] sb1,  
  output wire [ 4:0] sb2,  
  output wire [ 4:0] sb3,
  output wire [ 4:0] sb4,
  output wire [ 4:0] sb5,
  output wire [ 4:0] sb6,
  output wire [ 4:0] sb7,
  output wire [ 4:0] sb8,
  output wire [ 4:0] sb9,
  output wire        chan,
  output wire        last,
  output wire [ 3:0] len,
  output wire        valid,
  
  /*****************************************************************************
  * sram interface
  *
  * note: when sigb=0; raddr[9] is 'sel'
  *       when sigb=1, raddr[9] 0 = channel0 content area
  *                    raddr[9] 1 = channel1 content area
  *
  *****************************************************************************/
  output wire        ren,
  output wire [ 9:0] raddr,
  input  wire [49:0] rdata
);
  /* constants */
  localparam  NSD_48=2'd0, 
              NSD_52=2'd1, 
              NSD_108=2'd2, 
              NSD_234=2'd3;

  localparam  RU_26=3'd0,
              RU_52=3'd1,
              RU_106=3'd2,
              RU_242=3'd3,
              RU_484=3'd4,
              RU_996=3'd5;
  
  localparam  NBPSC_1=3'd0,
              NBPSC_2=3'd1,
              NBPSC_4=3'd2,
              NBPSC_6=3'd3,
              NBPSC_8=3'd4,
              NBPSC_10=3'd5;

  localparam  NSS_1=3'd0,
              NSS_2=3'd1,
              NSS_3=3'd2,
              NSS_4=3'd3;

`ifdef RW_NX_DERIV_EQU_1024QAM
  localparam DWIDTH=50;
`else
  localparam DWIDTH=40;
`endif // RW_NX_DERIV_EQU_1024QAM

  /*****************************************************************************
  * parameters decoding
  *****************************************************************************/
  reg  [ 2:0] s;    /* max(1,0.5*nbpsc) */
  reg  [ 4:0] ncol;
  reg  [ 3:0] nrow;
  reg  [ 3:0] nbpsc_dec;
  reg         odd_ncbps;
  
  always @(*)
  begin
    case(nbpsc)                               
      NBPSC_1: {s,nbpsc_dec} = {3'd1,4'd1};   
      NBPSC_2: {s,nbpsc_dec} = {3'd1,4'd2};   
      NBPSC_4: {s,nbpsc_dec} = {3'd2,4'd4};   
      NBPSC_6: {s,nbpsc_dec} = {3'd3,4'd6};   
      NBPSC_8: {s,nbpsc_dec} = {3'd4,4'd8};   
      default: {s,nbpsc_dec} = {3'd5,4'd10};   
    endcase                                   
    
    if(!ruen)
      case(nsd)
        NSD_48:    {ncol, nrow} = {5'd16, 4'd3};
        NSD_52:
          if(!dcm)
                   {ncol, nrow} = {5'd13, 4'd4};
          else
                   {ncol, nrow} = {5'd13, 4'd2};
       
        NSD_108:   {ncol, nrow} = {5'd18, 4'd6};
        default:   {ncol, nrow} = {5'd26, 4'd9};
      endcase
    else
    begin
      if(!dcm)
        case(rulen)
          RU_26:   {ncol, nrow} = { 5'd8, 4'd3};
          RU_52:   {ncol, nrow} = {5'd16, 4'd3};
          RU_106:  {ncol, nrow} = {5'd17, 4'd6};
          default: {ncol, nrow} = {5'd26, 4'd9};
        endcase 
      else
        case(rulen)
          RU_26:   {ncol, nrow} = { 5'd4, 4'd3};
          RU_52:   {ncol, nrow} = { 5'd8, 4'd3};
          RU_106:  {ncol, nrow} = {5'd17, 4'd3};
          default: {ncol, nrow} = {5'd13, 4'd9};
        endcase 
    end
    
    odd_ncbps = ruen && dcm && nss==NSS_1 && nbpsc==NBPSC_1 && (rulen==RU_106 || rulen==RU_242);
    
  end

  /*****************************************************************************
  * S0: address, offset, nsb generation
  *****************************************************************************/
  wire        bcc_enable;
`ifdef RW_NX_LDPC_DEC
  wire        ldpc_ready;
  wire        ldpc_enable;
  
  assign ldpc_ready  = ready  && fec; 
  assign ldpc_enable = enable && fec;
  assign bcc_enable  = enable && !fec;

  wire        s0_ldpc_valid;
  wire        s0_ldpc_last;
  wire [8:0]  s0_ldpc_addr;
  
  rx_bd_deinterleaver_genaddr_ldpc u_genaddr_ldpc
  (
    .rst_n(      rst_n),
    .clk(        clk),

    .enable(     ldpc_enable),
    .start(      chan0_start),
    .ruen(       ruen),
    .rulen(      rulen),
    .nsd(        nsd),
  
    .ready(      ldpc_ready),
    .valid(      s0_ldpc_valid),
    .last(       s0_ldpc_last),
    .addr(       s0_ldpc_addr)
  );
`else
  assign bcc_enable  = enable;
`endif // RW_NX_LDPC_DEC

  wire        s0_chan0_valid;
  wire [7:0]  s0_chan0_addr;
  wire [3:0]  s0_chan0_nsb;
  wire [2:0]  s0_chan0_offset;
  wire        s0_chan0_last;
  wire        ch0_ready;
  
  assign ch0_ready = ready & !chansel && !fec; 
  
  rx_bd_deinterleaver_genaddr u_genaddr_ch0
  (
    .rst_n(      rst_n),
    .clk(        clk),

    .enable(     bcc_enable),
    .start(      chan0_start),
  
    .b_init(     qbpsk),
    .nbpsc_dec(  nbpsc_dec),
    .ruen(       ruen),
    .nrow(       nrow),
    .ncol(       ncol),
    .sel(        chan0_sel),
  
    .ready(      ch0_ready),
    .valid(      s0_chan0_valid),
    .addr(       s0_chan0_addr),
    .nsb(        s0_chan0_nsb),
    .offset(     s0_chan0_offset),
    .last(       s0_chan0_last)
  );

  wire        s0_chan1_valid;
  wire [7:0]  s0_chan1_addr;
  wire [3:0]  s0_chan1_nsb;
  wire [2:0]  s0_chan1_offset;
  wire        s0_chan1_last;
  wire        ch1_ready;
  
  assign ch1_ready = ready & chansel && !fec; 
  
  rx_bd_deinterleaver_genaddr u_genaddr_ch1
  (
    .rst_n(      rst_n),
    .clk(        clk),

    .enable(     bcc_enable),
    .start(      chan1_start),
  
    .b_init(     qbpsk),
    .nbpsc_dec(  nbpsc_dec),
    .ruen(       ruen),
    .nrow(       nrow),
    .ncol(       ncol),
    .sel(        chan1_sel),
  
    .ready(      ch1_ready),
    .valid(      s0_chan1_valid),
    .addr(       s0_chan1_addr),
    .nsb(        s0_chan1_nsb),
    .offset(     s0_chan1_offset),
    .last(       s0_chan1_last)
  );

  /*****************************************************************************
  * S1: memory address cycle
  *****************************************************************************/
  reg         s1_en;
  reg  [ 9:0] s1_addr;
  reg  [ 3:0] s1_nsb;
  reg  [ 2:0] s1_offset;
  reg         s1_last;
  reg         s1_chan;
 
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1_en     <= 1'b0;
      s1_addr   <= 10'b0;
      s1_nsb    <= 4'd0;
      s1_offset <= 3'd0;
      s1_chan   <= 1'b0;
      s1_last   <= 1'b0;
    end
    else if(!enable)
    begin
      s1_en     <= 1'b0;
      s1_addr   <= 10'b0;
      s1_nsb    <= 4'd0;
      s1_offset <= 3'd0;
      s1_chan   <= 1'b0;
      s1_last   <= 1'b0;
    end
    else
    begin
      if(ready)
      begin
`ifdef RW_NX_LDPC_DEC
        if (fec)
        begin
          s1_en     <= s0_ldpc_valid;
          s1_addr   <= {sel,s0_ldpc_addr};
          s1_nsb    <= nbpsc_dec;
          s1_offset <= 4'd0;
          s1_last   <= s0_ldpc_last;
          s1_chan   <= 1'b0;
        end 
        else
`endif // RW_NX_LDPC_DEC
        if(!chansel)
        begin
          s1_en     <= s0_chan0_valid;
          s1_addr   <= {~sigben & sel ,1'b0,s0_chan0_addr};
          s1_nsb    <= s0_chan0_nsb;
          s1_offset <= s0_chan0_offset;
          s1_last   <= s0_chan0_last;
          s1_chan   <= 1'b0;
        end
        else
        begin
          s1_en     <= s0_chan1_valid;
          s1_addr   <= {1'b1,1'b0,s0_chan1_addr};
          s1_nsb    <= s0_chan1_nsb;
          s1_offset <= s0_chan1_offset;
          s1_last   <= s0_chan1_last;
          s1_chan   <= 1'b1;
        end
      end
      else
      begin
        /* sig:   request mode    */
        /* others: continuous mode */
        if(sigben)
          s1_en <= 1'b0; 
      end
    end
  end
 
  assign ren   = s1_en;
  assign raddr = s1_addr;
 
  /* fix nsb for odd ncbps */
  wire [ 3:0] s1_nsb_minus_1;
  assign s1_nsb_minus_1 = s1_nsb - 3'd1;
 
  /*****************************************************************************
  * S2: memory access cycle
  *****************************************************************************/
  reg        s2_en;
  reg [ 3:0] s2_nsb;
  reg [ 2:0] s2_offset;
  reg        s2_last;
  reg        s2_chan;

  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2_en     <= 1'b0;
      s2_nsb    <= 4'd0;
      s2_offset <= 3'd0;
      s2_last   <= 1'b0;
      s2_chan   <= 1'b0;
    end
    else if(!enable)
    begin
      s2_en     <= 1'b0;
      s2_nsb    <= 4'd0;
      s2_offset <= 3'd0;
      s2_last   <= 1'b0;
      s2_chan   <= 1'b0;
    end
    else
    begin
      if(ready || sigben)
      begin
        s2_en     <= s1_en;
        if(s1_last && odd_ncbps)
          s2_nsb  <= s1_nsb_minus_1;
        else
          s2_nsb  <= s1_nsb;
        s2_offset <= s1_offset;
        s2_last   <= s1_last;
        s2_chan   <= s1_chan;
      end
    end
  end
 
  // decode rdata to wires to ease debug
  wire [ 4:0]  s2_sb7, s2_sb6, s2_sb5, s2_sb4, s2_sb3, s2_sb2, s2_sb1, s2_sb0; 
  assign { s2_sb7, s2_sb6, s2_sb5, s2_sb4, s2_sb3, s2_sb2, s2_sb1, s2_sb0} = rdata[39:0]; 
`ifdef RW_NX_DERIV_EQU_1024QAM
  wire [ 4:0]  s2_sb9, s2_sb8; 
  assign { s2_sb9, s2_sb8} = rdata[49:40]; 
`endif // RW_NX_DERIV_EQU_1024QAM
  
  /*****************************************************************************
  * S3: data capture (S3A: capture stage, S3B: stall buffer)
  *****************************************************************************/
  reg          ready_1t;
  reg          s3a_en,     s3b_en;
  reg   [ 3:0] s3a_nsb,    s3b_nsb;
  reg   [ 2:0] s3a_offset, s3b_offset;
  reg   [DWIDTH-1:0] s3a_data,   s3b_data;
  reg          s3a_last,   s3b_last;
  reg          s3a_chan,   s3b_chan;

  wire         s3_en;
  wire  [ 3:0] s3_nsb;
  wire  [ 2:0] s3_offset;
  wire  [DWIDTH-1:0] s3_data;
  wire         s3_last;
  wire         s3_chan;
   
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s3a_en     <= 1'b0;
      s3a_data   <= {DWIDTH{1'b0}};
      s3a_nsb    <= 4'd0;
      s3a_offset <= 3'd0;
      s3a_last   <= 1'b0;
      s3a_chan   <= 1'b0;
    
      s3b_en     <= 1'b0;
      s3b_data   <= {DWIDTH{1'b0}};
      s3b_nsb    <= 4'd0;
      s3b_offset <= 3'd0;
      s3b_last   <= 1'b0;
      s3b_chan   <= 1'b0;
      ready_1t   <= 1'b0;
    end
    else if(!enable)
    begin
      /* note: no need to clear s3_data here, avoid feedback logic on mem rdata path */
      s3a_en     <= 1'b0;
      s3b_en     <= 1'b0;
    end
    else
    begin
      ready_1t <= ready;
      if(sigben)
      begin
        /***********************************************************************
        * sig: request mode
        ***********************************************************************/ 
        s3a_data[DWIDTH-1:0] <= rdata[DWIDTH-1:0];
        s3a_en     <= s2_en;
        s3a_nsb    <= s2_nsb;
        s3a_offset <= s2_offset;
        s3a_last   <= s2_last;
        s3a_chan   <= s2_chan;
      end
      else
      begin
        /***********************************************************************
        * others: continuous mode
        ***********************************************************************/ 
        /* s3a: capture stage */
        if(ready_1t)
        begin
          s3a_en     <= s2_en;
          s3a_data   <= rdata[39:0];
`ifdef RW_NX_DERIV_EQU_1024QAM
          s3a_data[49:40]   <= rdata[49:40];
`endif // RW_NX_DERIV_EQU_1024QAM
          s3a_nsb    <= s2_nsb;
          s3a_offset <= s2_offset;
          s3a_last   <= s2_last;
          s3a_chan   <= s2_chan;
        end
        else if(ready && !s3b_en)
        begin
          s3a_en     <= 1'b0;
          s3a_data   <= {DWIDTH{1'b0}};
          s3a_nsb    <= 4'd0;
          s3a_offset <= 3'd0;
          s3a_last   <= 1'b0;
          s3a_chan   <= 1'b0;
        end
       
        /* s3b: stall buffer stage */
        if(!ready && ready_1t)
        begin
          if(s3a_en)
          begin
            s3b_en     <= s3a_en;
            s3b_data   <= s3a_data;
            s3b_nsb    <= s3a_nsb;
            s3b_offset <= s3a_offset;
            s3b_last   <= s3a_last;
            s3b_chan   <= s3a_chan;
          end
        end
        else if(ready && !ready_1t)
        begin
          s3b_en <= 1'b0;
        end
      end 
    end
  end
 
  /* s3/s3b mux */
  assign s3_en     = s3b_en | s3a_en;
  assign s3_data   = s3b_en ? s3b_data   : s3a_data;
  assign s3_nsb    = s3b_en ? s3b_nsb    : s3a_nsb;
  assign s3_offset = s3b_en ? s3b_offset : s3a_offset;
  assign s3_last   = s3b_en ? s3b_last   : s3a_last;
  assign s3_chan   = s3b_en ? s3b_chan   : s3a_chan;  
 
  /*  remove offset and mask unwanted sb */
  wire   [DWIDTH-1:0]  s3_in_sh1,s3_in_sh2,s3_in_sh3;
  wire   [ 3:0]        s3_im_shsel;
  wire   [DWIDTH-1:0]  s3_im_sh1,s3_im_sh2,s3_im_sh3,s3_im_sh4;
  wire   [DWIDTH-1:0]  s3_in_data;
  
  assign s3_in_sh1   = s3_offset[0]?{ 5'b0,   s3_data[DWIDTH-1:5]}:s3_data;
  assign s3_in_sh2   = s3_offset[1]?{10'b0, s3_in_sh1[DWIDTH-1:10]}:s3_in_sh1;
  assign s3_in_sh3   = s3_offset[2]?{20'b0, s3_in_sh2[DWIDTH-1:20]}:s3_in_sh2;

`ifdef RW_NX_DERIV_EQU_1024QAM
  assign s3_im_shsel = 4'd10 - s3_nsb; 
`else
  assign s3_im_shsel = 4'd8 - s3_nsb; 
`endif // RW_NX_DERIV_EQU_1024QAM

  assign s3_im_sh1   = s3_im_shsel[0]?    { 5'b0,{(DWIDTH-5){1'b1}}}:{DWIDTH{1'b1}};
  assign s3_im_sh2   = s3_im_shsel[1]?{10'b0,s3_im_sh1[DWIDTH-1:10]}:s3_im_sh1;
  assign s3_im_sh3   = s3_im_shsel[2]?{20'b0,s3_im_sh2[DWIDTH-1:20]}:s3_im_sh2;
`ifdef RW_NX_DERIV_EQU_1024QAM
  assign s3_im_sh4   = s3_im_shsel[3]?{40'b0,s3_im_sh3[DWIDTH-1:40]}:s3_im_sh3;
`else
  assign s3_im_sh4   = s3_im_shsel[3]?                         40'd0:s3_im_sh3;
`endif // RW_NX_DERIV_EQU_1024QAM
  
  assign s3_in_data  = s3_in_sh3 & s3_im_sh4;


  wire [ 4:0] s3a_sb7,s3a_sb6,s3a_sb5,s3a_sb4,s3a_sb3,s3a_sb2,s3a_sb1,s3a_sb0;
  wire [ 4:0] s3b_sb7,s3b_sb6,s3b_sb5,s3b_sb4,s3b_sb3,s3b_sb2,s3b_sb1,s3b_sb0; 
  wire [ 4:0]  s3_sb7, s3_sb6, s3_sb5, s3_sb4, s3_sb3, s3_sb2, s3_sb1, s3_sb0; 

  assign {s3a_sb7,s3a_sb6,s3a_sb5,s3a_sb4,s3a_sb3,s3a_sb2,s3a_sb1,s3a_sb0} = s3a_data;
  assign {s3b_sb7,s3b_sb6,s3b_sb5,s3b_sb4,s3b_sb3,s3b_sb2,s3b_sb1,s3b_sb0} = s3b_data;
  assign { s3_sb7, s3_sb6, s3_sb5, s3_sb4, s3_sb3, s3_sb2, s3_sb1, s3_sb0} = s3_in_data;

 
  /*****************************************************************************
  * S4: output slot
  *****************************************************************************/
  reg   [DWIDTH-1:0] s4_data;
  reg   [ 3:0] s4_len;
  reg          s4_last;
  reg          s4_valid;
  reg          s4_chan;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s4_data    <= {DWIDTH{1'b0}};
      s4_len     <= 4'b0;
      s4_last    <= 1'b0;
      s4_valid   <= 1'b0;
      s4_chan    <= 1'b0;
      chan0_busy <= 1'b0;
      chan1_busy <= 1'b0;
    end
    else if(!enable)
    begin
      s4_data    <= {DWIDTH{1'b0}};
      s4_len     <= 4'b0;
      s4_last    <= 1'b0;
      s4_valid   <= 1'b0;
      s4_chan    <= 1'b0;
      chan0_busy <= 1'b0;
      chan1_busy <= 1'b0;
    end
    else
    begin
      if(chan0_start)
        chan0_busy <= 1'b1;
        
      if(chan1_start)
        chan1_busy <= 1'b1;
        
      if(ready || sigben)
      begin
        if(s4_valid && s4_last)
        begin
          s4_data  <= {DWIDTH{1'b0}};
          s4_len   <= 4'b0;
          s4_last  <= 1'b0;
          s4_valid <= 1'b0;
          s4_chan  <= 1'b0;
          if(!s4_chan)
            chan0_busy <= 1'b0;
          else
            chan1_busy <= 1'b0;
        end
        else if(s3_en)
        begin
          s4_len   <= s3_nsb;
          s4_data  <= s3_in_data;
          s4_last  <= s3_last;
          s4_chan  <= s3_chan;
          s4_valid <= 1'b1;
        end
        else
        begin
          s4_valid <= 4'b0;
        end
      end
    end
  end
 
  assign {sb7,sb6,sb5,sb4,sb3,sb2,sb1,sb0} = s4_data[39:0];
`ifdef RW_NX_DERIV_EQU_1024QAM
  assign {sb9,sb8} = s4_data[49:40];
`else
  assign {sb9,sb8} = 10'd0;
`endif // RW_NX_DERIV_EQU_1024QAM

  assign len                               = s4_len;
  assign last                              = s4_last;
  assign chan                              = s4_chan;
  assign valid                             = s4_valid;
 
endmodule
`default_nettype wire
