`default_nettype none
module rx_bd_deinterleaver_genaddr
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire        rst_n,
  input  wire        clk,

  input  wire        enable,
  input  wire        start,
  
  input  wire        b_init,
  input  wire [3:0]  nbpsc_dec,
  input  wire        ruen,
  input  wire [3:0]  nrow,
  input  wire [4:0]  ncol,
  input  wire [1:0]  sel,
  
  input  wire        ready,
  output wire        valid,
  output wire [7:0]  addr,
  output wire [3:0]  nsb,
  output wire [2:0]  offset,
  output wire        last
);

  /*****************************************************************************
  * S0: address, offset, nsb generation
  *****************************************************************************/
  /* sequence state row,b,col */
  wire [ 3:0] n_s0_row;
  wire [ 3:0] n_s0_b;
  wire [ 5:0] n_s0_col;
  reg  [ 3:0] s0_row;
  reg  [ 3:0] s0_b;
  reg  [ 2:0] s0_ba;
  reg  [ 4:0] s0_col;
  wire        s0_row_last,s0_b_last,s0_col_last;
  wire [ 4:0] s0_remaining_col;
  wire        s0_partial;
  wire [ 3:0] s0_nsb_required;
  wire [ 3:0] s0_nsb_available;
  reg  [ 3:0] s0_nsb_available_1t;
  wire [ 2:0] s0_offset;
  wire        s0_incomplete;
  wire [ 7:0] s0_row_x_26,s0_row_x_13,s0_row_x_16,s0_row_x_k;
  wire [ 7:0] s0_rem_x_ba;

  reg         s0_state;
  reg         s0_incomplete_done;
  wire [ 7:0] s0_full_addr;
  wire [ 7:0] s0_partial_addr;
  wire [ 7:0] s0_addr;
  wire        s0_last;

  reg  [ 7:0] sig_offset;
  
  always @(*)
  begin
    case({nbpsc_dec,s0_b})
      {4'd1,4'd0}: s0_ba = 3'd0; /* nbpsc==1 || !qbpskdet */
      {4'd1,4'd1}: s0_ba = 3'd4; /*              qbpskdet */
      {4'd2,4'd0}: s0_ba = 3'd0;
      {4'd2,4'd1}: s0_ba = 3'd4;
      {4'd4,4'd0}: s0_ba = 3'd0;
      {4'd4,4'd1}: s0_ba = 3'd1;
      {4'd4,4'd2}: s0_ba = 3'd4;
      {4'd4,4'd3}: s0_ba = 3'd5;
      {4'd6,4'd0}: s0_ba = 3'd0;
      {4'd6,4'd1}: s0_ba = 3'd1;
      {4'd6,4'd2}: s0_ba = 3'd2;
      {4'd6,4'd3}: s0_ba = 3'd4;
      {4'd6,4'd4}: s0_ba = 3'd5;
      {4'd6,4'd5}: s0_ba = 3'd6;
      {4'd8,4'd0}: s0_ba = 3'd0;
      {4'd8,4'd1}: s0_ba = 3'd1;
      {4'd8,4'd2}: s0_ba = 3'd2;
      {4'd8,4'd3}: s0_ba = 3'd3;
      {4'd8,4'd4}: s0_ba = 3'd4;
      {4'd8,4'd5}: s0_ba = 3'd5;
      {4'd8,4'd6}: s0_ba = 3'd6;
      default:     s0_ba = 3'd7;
    endcase
    
    if(!ruen)
      if(ncol==5'd16)
        /* NSD48 */
        case(sel[1:0])
          2'd0:    sig_offset = 8'd0;
          2'd1:    sig_offset = 8'd48;
          2'd2:    sig_offset = 8'd96;
          default: sig_offset = 8'd144;
        endcase
      else
        /* NSD52, for others nsd, sel must be zero*/
        case(sel[1:0])
          2'd0:    sig_offset = 8'd0;
          2'd1:    sig_offset = 8'd52;
          2'd2:    sig_offset = 8'd104;
          default: sig_offset = 8'd156;
        endcase
    else
      sig_offset = 8'd0;
  end

  assign n_s0_row           = s0_row        +  4'd1;
  assign n_s0_b             = s0_b          +  4'd1;
  assign n_s0_col           = {1'b0,s0_col} +  5'd8; 
 
  assign s0_row_last        = n_s0_row == nrow;
  assign s0_b_last          = n_s0_b   >= nbpsc_dec;
  assign s0_col_last        = n_s0_col >= {1'b0,ncol};
  
  assign s0_remaining_col   = ncol - s0_col; 
  assign s0_partial         = s0_remaining_col < 5'd8;
  assign s0_nsb_required    = s0_partial?s0_remaining_col:4'd8;
  assign s0_offset          = (s0_partial&&!s0_incomplete_done)?s0_rem_x_ba[2:0]:3'd0;
  assign s0_incomplete      = s0_partial & ( (s0_nsb_required + {1'b0,s0_offset}) > 8'd8 );
  assign s0_nsb_available   = s0_incomplete_done?(s0_nsb_required-s0_nsb_available_1t):
                              s0_incomplete?(4'd8-{1'b0,s0_offset}):s0_nsb_required;

  /* non-sig symbols */
  assign s0_row_x_26        = {3'b0,     s0_row,1'b0} + 
                              {1'b0,     s0_row,3'b0} + 
                              {          s0_row,4'b0};
 
  /* nsd=52 sig symbols */
  assign s0_row_x_13        = {1'b0,s0_row_x_26[7:1]};
  
  /* nsd=48 sig symbols */
  assign s0_row_x_16        = {         s0_row,4'b0};
                              
                              
  assign s0_row_x_k         = (!ruen && ncol==5'd16)?s0_row_x_16:
                              (!ruen && ncol==5'd13)?s0_row_x_13:
                                                     s0_row_x_26;


  assign s0_rem_x_ba        = ({3'b0,     s0_remaining_col} & {8{s0_ba[0]}} )+
                              ({2'b0,s0_remaining_col,1'b0} & {8{s0_ba[1]}} )+
                              ({1'b0,s0_remaining_col,2'b0} & {8{s0_ba[2]}} );


  assign s0_full_addr       =                s0_row_x_k +
                              {3'b0,  s0_col[4:3],3'b0} +
                              {          5'b0,   s0_ba} +
                              sig_offset;

  assign s0_partial_addr    =                s0_row_x_k +
                              {3'b0,    ncol[4:3],3'b0} +
                              {3'b0,  s0_rem_x_ba[7:3]} +
                              {7'b0,s0_incomplete_done} +
                              sig_offset;
  
  assign s0_addr            = { s0_partial?s0_partial_addr:s0_full_addr};

  assign s0_last            = (!s0_incomplete | s0_incomplete_done) & s0_col_last & s0_b_last & s0_row_last;

  /* fsm */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s0_incomplete_done <= 1'b1;
      s0_nsb_available_1t<= 4'd0;
      s0_state           <= 1'b0;
      s0_row             <= 4'd0;
      s0_col             <= 5'd0;
      s0_b               <= 4'd0;
    end
    else if(!enable)
    begin
      s0_incomplete_done <= 1'b1;
      s0_nsb_available_1t<= 4'd0;
      s0_state           <= 1'b0;
      s0_row             <= 4'd0;
      s0_col             <= 5'd0;
      s0_b               <= 4'd0;
    end
    else
    begin
      if(!s0_state)
      begin
        s0_row   <= 4'd0;
        s0_col   <= 5'd0;
        s0_b     <= {3'b0,b_init};
        if(start)
          s0_state <= 1'b1;
      end
      else
      begin
        if(ready)
        begin
          s0_incomplete_done  <= 1'b0;
          s0_nsb_available_1t <= 4'd0;
          if(!s0_incomplete || s0_incomplete_done)
          begin
            /* the memory line contains all the consecutive requried soft bits */
            if(s0_col_last)
            begin
              s0_col  <= 5'd0;
              if(s0_b_last)
              begin
                s0_b <= {3'b0,b_init};
                if(s0_row_last)
                begin
                  s0_row   <= 4'd0;
                  s0_state <= 1'b0;
                end
                else
                begin
                  s0_row   <= n_s0_row;
                end
              end
              else
              begin
                s0_b <= n_s0_b;
              end
            end
            else
            begin
              s0_col  <= n_s0_col[4:0];
            end
          end
          else
          begin
            /* the memory line does not have all consecutive soft-bit, need a new access */  
            s0_incomplete_done  <=1'b1;
            s0_nsb_available_1t <= s0_nsb_available;
          end
        end
      end
    end
  end

  assign valid  = s0_state;
  assign addr   = s0_addr;  
  assign nsb    = s0_nsb_available;   
  assign offset = s0_offset;
  assign last   = s0_last;  

endmodule
`default_nettype wire
