`default_nettype none
module rx_bd_deinterleaver_genaddr2
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire        rst_n,
  input  wire        clk,

  input  wire        enable,
  input  wire        start,
  
  input  wire        odd_ndbps,
  input  wire        b_init,
  input  wire [3:0]  nbpsc_dec,
  input  wire        sigen,
  input  wire [3:0]  nrow,
  input  wire [4:0]  ncol,
  input  wire [1:0]  sel,
  input  wire        row_fifo,
  input  wire [2:0]  row_fifo_wrptr,
  output wire [2:0]  row_fifo_rdptr,
  
  input  wire        ready,
  output wire        valid,
  output wire [7:0]  addr,
  output wire [3:0]  nsb,
  output wire [2:0]  offset,
  output wire        last
);

  /*****************************************************************************
  * S0: address, offset, nsb generation
  *****************************************************************************/
  /* sequence state row,b,col */
  wire [ 3:0] dlen;   
  wire [ 4:0] ncol_fix;
  wire [ 3:0] n_s0_row;
  wire [ 3:0] n_s0_b;
  wire [ 5:0] n_s0_col;
  reg  [ 3:0] s0_row;
  reg  [ 3:0] s0_b;
  reg  [ 2:0] s0_ba;
  reg  [ 4:0] s0_col;
  wire        s0_row_last,s0_b_last,s0_col_last;
  wire [ 4:0] s0_remaining_col;
  wire        s0_partial;
  wire [ 3:0] s0_nsb_required;
  wire [ 3:0] s0_nsb_available;
  reg  [ 3:0] s0_nsb_available_1t;
  wire [ 2:0] s0_offset;
  wire        s0_incomplete;
  wire [ 7:0] s0_row_fifosel_x_26;
  wire [ 7:0] s0_row_x_26,s0_row_x_13,s0_row_x_16,s0_row_x_k;
  wire [ 7:0] s0_rem_x_ba;
  reg  [ 2:0] s0_rem_x_ba_mod_dlen;
  reg  [ 2:0] s0_rem_x_ba_div_dlen; 
  
  reg  [ 2:0] s0_row_fifo_rdptr;

  localparam  IDLE=2'd0, WAIT_ROW_FIFO=2'd1,READ_ROW=2'd2;
  
  reg  [ 1:0] s0_state;
  reg         s0_incomplete_done;
  wire [ 7:0] s0_full_addr;
  wire [ 7:0] s0_partial_addr;
  wire [ 7:0] s0_addr;
  wire        s0_last;
  
  reg  [ 7:0] sig_offset;
  
  always @(*)
  begin
    if(!sigen)
      case({nbpsc_dec,s0_b})
        {4'd1,4'd0}: s0_ba = 3'd0;
        {4'd1,4'd1}: s0_ba = 3'd4;
        
        {4'd2,4'd0}: s0_ba = 3'd0;
        {4'd2,4'd1}: s0_ba = 3'd4;
        
        {4'd4,4'd0}: s0_ba = 3'd0;
        {4'd4,4'd1}: s0_ba = 3'd1;
        {4'd4,4'd2}: s0_ba = 3'd4;
        {4'd4,4'd3}: s0_ba = 3'd5;
        
        {4'd6,4'd0}: s0_ba = 3'd0;
        {4'd6,4'd1}: s0_ba = 3'd1;
        {4'd6,4'd2}: s0_ba = 3'd2;
        {4'd6,4'd3}: s0_ba = 3'd4;
        {4'd6,4'd4}: s0_ba = 3'd5;
        {4'd6,4'd5}: s0_ba = 3'd6;
        
        {4'd8,4'd0}: s0_ba = 3'd0;
        {4'd8,4'd1}: s0_ba = 3'd1;
        {4'd8,4'd2}: s0_ba = 3'd2;
        {4'd8,4'd3}: s0_ba = 3'd3;
        {4'd8,4'd4}: s0_ba = 3'd4;
        {4'd8,4'd5}: s0_ba = 3'd5;
        {4'd8,4'd6}: s0_ba = 3'd6;
        default:     s0_ba = 3'd7;
      endcase
    else
      case({nbpsc_dec,s0_b})
        {4'd1,4'd0}: s0_ba = 3'd0; /* nbpsc==1 || !qbpskdet */
        {4'd1,4'd1}: s0_ba = 3'd3; /*              qbpskdet */
        
        {4'd2,4'd0}: s0_ba = 3'd0;
        {4'd2,4'd1}: s0_ba = 3'd3;
        
        {4'd4,4'd0}: s0_ba = 3'd0;
        {4'd4,4'd1}: s0_ba = 3'd1;
        {4'd4,4'd2}: s0_ba = 3'd3;
        {4'd4,4'd3}: s0_ba = 3'd4;
        
        {4'd6,4'd0}: s0_ba = 3'd0;
        {4'd6,4'd1}: s0_ba = 3'd1;
        {4'd6,4'd2}: s0_ba = 3'd2;
        {4'd6,4'd3}: s0_ba = 3'd3;
        {4'd6,4'd4}: s0_ba = 3'd4;
        default:     s0_ba = 3'd5;
      endcase
    
    if(sigen)
      if(ncol==5'd16)
        /* NSD48 */
        case(sel[1:0])
          2'd0:    sig_offset = 8'd0;
          2'd1:    sig_offset = 8'd48;
          2'd2:    sig_offset = 8'd96;
          default: sig_offset = 8'd144;
        endcase
      else
        /* NSD52, for others nsd, sel must be zero*/
        case(sel[1:0])
          2'd0:    sig_offset = 8'd0;
          2'd1:    sig_offset = 8'd52;
          2'd2:    sig_offset = 8'd104;
          default: sig_offset = 8'd156;
        endcase
    else
      sig_offset = 8'd0;
  end
  
  
  /* compute div8,mod8 or div6,mod6 depeding of memory line length */
  always @(*)
  begin:b_div_mod
    reg [ 2:0] v;
    if(!sigen)
    begin
      /* memory len 8 sb */
      s0_rem_x_ba_mod_dlen = s0_rem_x_ba[2:0];
      s0_rem_x_ba_div_dlen = s0_rem_x_ba[5:3];
      v = 0;
    end
    else
    begin
      /* memory len 6 sb */
      case(s0_rem_x_ba[5:0])
         6'd0, 6'd1, 6'd2, 6'd3, 6'd4, 6'd5: {s0_rem_x_ba_div_dlen,v} = {3'd0,3'd0};
         6'd6, 6'd7, 6'd8, 6'd9,6'd10,6'd11: {s0_rem_x_ba_div_dlen,v} = {3'd1,3'd6};
        6'd12,6'd13,6'd14,6'd15,6'd16,6'd17: {s0_rem_x_ba_div_dlen,v} = {3'd2,3'd4};
        6'd18,6'd19,6'd20,6'd21,6'd22,6'd23: {s0_rem_x_ba_div_dlen,v} = {3'd3,3'd2};
        6'd24,6'd25,6'd26,6'd27,6'd28,6'd29: {s0_rem_x_ba_div_dlen,v} = {3'd4,3'd0};
        default:                             {s0_rem_x_ba_div_dlen,v} = {3'd5,3'd6};
      endcase
      s0_rem_x_ba_mod_dlen = s0_rem_x_ba[2:0] - v;
    end
  end
  
  /* partial base */
  reg [4:0] partial_base;
  
  always @(*)
  begin:b_partial_base
    if(!sigen)
    begin
      /* partial base = 8*floor(ncol/8) */
      partial_base = {ncol[4:3],3'b0};
    end
    else
    begin
      /* partial base = 6*floor(ncol/6) */
      case(ncol)
          5'd26: partial_base = 5'd24;
          5'd18: partial_base = 5'd18;
          5'd17: partial_base = 5'd12;
          5'd16: partial_base = 5'd12;
          5'd13: partial_base = 5'd12;
           5'd8: partial_base = 5'd6;
        default: partial_base = 5'd0;  /* 5'd4 */
      endcase
    end
  end
  
  wire [2:0] n_s0_row_fifo_rdptr;
  assign n_s0_row_fifo_rdptr = s0_row_fifo_rdptr + 3'd1;
  
  wire       row_fifo_empty;
  assign     row_fifo_empty = s0_row_fifo_rdptr==row_fifo_wrptr;
  
  assign dlen               = sigen?4'd6:4'd8;
  assign ncol_fix           = (odd_ndbps & s0_row_last)?(ncol-5'd1):ncol;

  assign n_s0_row           = s0_row        +  4'd1;
  assign n_s0_b             = s0_b          +  4'd1;
  assign n_s0_col           = {1'b0,s0_col} +  {2'b0,dlen}; 
 
  assign s0_row_last        = n_s0_row == nrow;
  assign s0_b_last          = n_s0_b   >= nbpsc_dec;


  assign s0_col_last        = n_s0_col >= {1'b0,ncol_fix};
  assign s0_remaining_col   = ncol_fix - s0_col; 
  
  assign s0_partial         = s0_remaining_col < {1'b0,dlen};
  assign s0_nsb_required    = s0_partial?s0_remaining_col[3:0]:dlen;
  assign s0_offset          = (s0_partial&&!s0_incomplete_done)?s0_rem_x_ba_mod_dlen:3'd0;
  assign s0_incomplete      = s0_partial & ( (s0_nsb_required + {1'b0,s0_offset}) > dlen );
  assign s0_nsb_available   = s0_incomplete_done?(s0_nsb_required-s0_nsb_available_1t):
                              s0_incomplete?(dlen-{1'b0,s0_offset}):s0_nsb_required;

  /* non-sig symbols */ 
  assign s0_row_x_26        = {3'b0,   s0_row,1'b0} + 
                              {1'b0,   s0_row,3'b0} + 
                              {        s0_row,4'b0};
 
  assign s0_row_fifosel_x_26= {5'b0, s0_row_fifo_rdptr[1:0],1'b0} + 
                              {3'b0, s0_row_fifo_rdptr[1:0],3'b0} + 
                              {2'b0, s0_row_fifo_rdptr[1:0],4'b0};
 
  /* nsd=52 sig symbols */
  assign s0_row_x_13        = {1'b0,s0_row_x_26[7:1]};
  
  /* nsd=48 sig symbols */
  assign s0_row_x_16        = {         s0_row,4'b0};
                              
                              
  /* nsd=others */
  assign s0_row_x_k         = (sigen && ncol==5'd16)?s0_row_x_16:
                              (sigen && ncol==5'd13)?s0_row_x_13:
                              (row_fifo)            ?s0_row_fifosel_x_26:
                                                     s0_row_x_26;

  
  assign s0_rem_x_ba        = ({3'b0,     s0_remaining_col} & {8{s0_ba[0]}} )+
                              ({2'b0,s0_remaining_col,1'b0} & {8{s0_ba[1]}} )+
                              ({1'b0,s0_remaining_col,2'b0} & {8{s0_ba[2]}} );


  assign s0_full_addr       =                    s0_row_x_k +
                              {3'b0,                s0_col} +
                              {              5'b0,   s0_ba} +
                                                 sig_offset;

  assign s0_partial_addr    =                    s0_row_x_k +
                              {3'b0,          partial_base} +
                              {5'b0,  s0_rem_x_ba_div_dlen} +
                              {7'b0,    s0_incomplete_done} +
                                                 sig_offset;
  
  assign s0_addr            = { s0_partial?s0_partial_addr:s0_full_addr};

  assign s0_last            = (!s0_incomplete | s0_incomplete_done) & s0_col_last & s0_b_last  & s0_row_last & (s0_state==READ_ROW);
  
  /* fsm */
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s0_state                 <= IDLE;
      s0_incomplete_done       <= 1'b0;
      s0_nsb_available_1t      <= 4'd0;
      s0_row                   <= 4'd0;
      s0_col                   <= 5'd0;
      s0_b                     <= 4'd0;
      s0_row_fifo_rdptr        <= 3'd0;
    end
    else if(!enable)
    begin
      s0_state                 <= IDLE;
      s0_incomplete_done       <= 1'b0;
      s0_nsb_available_1t      <= 4'd0;
      s0_row                   <= 4'd0;
      s0_col                   <= 5'd0;
      s0_b                     <= 4'd0;
      if(!row_fifo)
        s0_row_fifo_rdptr        <= 3'd0;
    end
    else
    begin
      case(s0_state)
        IDLE:
        begin
          s0_col <= 5'd0;
          s0_b   <= {3'b0,b_init};
          if(start)
          begin
            if(row_fifo)
              s0_state <= WAIT_ROW_FIFO;
            else
              s0_state <= READ_ROW;
          end
        end
        
        WAIT_ROW_FIFO:
        begin
          if(!row_fifo_empty)
            s0_state <= READ_ROW;
        end
        
        default: /* READ_ROW */
        begin
          if(ready)
          begin
            s0_incomplete_done  <= 1'b0;
            s0_nsb_available_1t <= 4'd0;
            if(!s0_incomplete || s0_incomplete_done)
            begin
              /* the memory line contains all the consecutive requried soft bits */
              if(s0_col_last)
              begin
                s0_col  <= 5'd0;
                if(s0_b_last)
                begin
                  s0_b  <= {3'b0,b_init};
                  if(row_fifo)
                    s0_row_fifo_rdptr <= n_s0_row_fifo_rdptr;
                  
                  if(s0_row_last)
                  begin
                    s0_row   <= 4'd0;
                    s0_state <= IDLE;
                  end
                  else
                  begin
                    s0_row   <= n_s0_row;
                    if(row_fifo)
                      s0_state          <= WAIT_ROW_FIFO;
                  end
                end
                else
                begin
                  s0_b <= n_s0_b;
                end
              end
              else
              begin
                s0_col  <= n_s0_col[4:0];
              end
            end
            else
            begin
              /* the memory line does not have all consecutive soft-bit, need a new access */  
              s0_incomplete_done  <=1'b1;
              s0_nsb_available_1t <= s0_nsb_available;
            end
          end
        end
      endcase
    end
  end

  assign row_fifo_rdptr = s0_row_fifo_rdptr;
  assign valid          = s0_state==READ_ROW;
  assign addr           = s0_addr;  
  assign nsb            = s0_nsb_available;   
  assign offset         = s0_offset;
  assign last           = s0_last;  

`ifdef RW_SIMU_ON
  reg [ 32*8-1:0] s_s0_state;
  always @(*)
  begin
    case(s0_state)
      IDLE:          s_s0_state = "IDLE";
      WAIT_ROW_FIFO: s_s0_state = "WAIT_ROW_FIFO";
      READ_ROW:      s_s0_state = "READ_ROW";
      default:       s_s0_state = "XXXXXXXXX";
    endcase
  end
`endif


endmodule
`default_nettype wire
