`default_nettype none
module rx_bd_ldpcin_resync
(
  /*****************************************************************************
  * regular domain
  *****************************************************************************/
  /* system */
  input  wire             clk,
  input  wire             rst_n,

  /* clken */
  output wire             ldpc_clken,

  /* global enable */
  input  wire             global_enable,

  output wire             ready,
  input  wire  [ 4:0]     sb0,
  input  wire  [ 4:0]     sb1,
  input  wire  [ 4:0]     sb2,
  input  wire  [ 4:0]     sb3,
  input  wire  [ 4:0]     sb4,
  input  wire  [ 4:0]     sb5,
  input  wire  [ 4:0]     sb6,
  input  wire  [ 4:0]     sb7,
  input  wire  [ 4:0]     sb8,
  input  wire  [ 4:0]     sb9,
  input  wire  [ 4:0]     sb10,
  input  wire  [ 4:0]     sb11,
  input  wire  [ 3:0]     len,
  input  wire             valid,

  /* parameters */
  input  wire             update,
  input  wire  [ 3:0]     format,
  input  wire  [ 2:0]     bandwidth,
  input  wire  [ 6:0]     mcs,
  input  wire  [ 1:0]     cr,
  input  wire  [ 2:0]     nbpsc,
  input  wire  [ 2:0]     nss,
  input  wire             stbc,
  input  wire  [ 2:0]     rulen,
  input  wire  [15:0]     nsym,
  input  wire  [15:0]     ncbps,
  input  wire  [11:0]     ncw,
  input  wire  [ 1:0]     lldpc,
  input  wire  [11:0]     nshrtq,
  input  wire  [11:0]     nshrtr,
  input  wire  [11:0]     npuncq,
  input  wire  [11:0]     npuncr,
  input  wire  [13:0]     nrepq,
  input  wire  [11:0]     nrepr,

  /*****************************************************************************
  * ldpc domain
  *****************************************************************************/
  /* system */
  input wire              ldpc_clk,
  input wire              ldpc_rst_n,

  /* global enable */
  input wire              ldpc_global_enable,

  /* input stream */
  input  wire  [ 3:0]     ldpc_ready,
  output wire  [ 4:0]     ldpc_sb0,
  output wire  [ 4:0]     ldpc_sb1,
  output wire  [ 4:0]     ldpc_sb2,
  output wire  [ 4:0]     ldpc_sb3,
  output wire  [ 4:0]     ldpc_sb4,
  output wire  [ 4:0]     ldpc_sb5,
  output wire  [ 4:0]     ldpc_sb6,
  output wire  [ 4:0]     ldpc_sb7,
  output wire  [ 4:0]     ldpc_sb8,
  output wire  [ 4:0]     ldpc_sb9,
  output wire  [ 3:0]     ldpc_valid,
 
  /* parameters */
  output reg              ldpc_update,
  output reg   [ 3:0]     ldpc_format,
  output reg   [ 2:0]     ldpc_bandwidth,
  output reg   [ 6:0]     ldpc_mcs,
  output reg   [ 1:0]     ldpc_cr,
  output reg   [ 2:0]     ldpc_nbpsc,
  output reg   [ 2:0]     ldpc_nss,
  output reg              ldpc_stbc,
  output reg   [ 2:0]     ldpc_rulen,
  output reg   [15:0]     ldpc_nsym,
  output reg   [15:0]     ldpc_ncbps,
  output reg   [11:0]     ldpc_ncw,
  output reg   [ 1:0]     ldpc_lldpc,
  output reg   [11:0]     ldpc_nshrtq,
  output reg   [11:0]     ldpc_nshrtr,
  output reg   [11:0]     ldpc_npuncq,
  output reg   [11:0]     ldpc_npuncr,
  output reg   [13:0]     ldpc_nrepq,
  output reg   [11:0]     ldpc_nrepr
);
 
  /*****************************************************************************
  * clock/block enable control
  *****************************************************************************/
  wire       ldpc_en;
  reg [5:0]  ldpc_clken_sr;
  reg [5:0]  ldpc_en_sr;
  reg        ldpc_state;
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      ldpc_clken_sr <= 6'b0;
      ldpc_en_sr    <= 6'b0;
      ldpc_state    <= 1'b0;
    end
    else
    begin
      ldpc_clken_sr <= {ldpc_clken_sr[5],ldpc_clken_sr[5:1]};
      ldpc_en_sr    <= {   ldpc_en_sr[5],   ldpc_en_sr[5:1]};
      if(!ldpc_state)
      begin
        if(update && global_enable)
        begin
          ldpc_clken_sr <= 6'b111111;      
          ldpc_en_sr    <= 6'b100000;      
          ldpc_state    <= 1'b1;
        end
      end
      else
      begin
        if(!update || !global_enable)
        begin
          ldpc_clken_sr <= 6'b011111;      
          ldpc_en_sr    <= 6'b000000;      
          ldpc_state    <= 1'b0;
        end
      end
    end
  end
  
  assign ldpc_clken = ldpc_clken_sr[0];
  assign ldpc_en    = ldpc_en_sr[0];

  /*****************************************************************************
  * S0 asynchronous fifo
  *****************************************************************************/
  wire         s0_ready;
  wire  [ 3:0] s0_len;
  wire  [59:0] s0_data;
  wire  [59:0] s0_mask1,s0_mask2,s0_mask3,s0_mask4,s0_data_mask;
  wire         s0_valid;
  
  wire async_fifo_ready;
  assign ready            = async_fifo_ready & update;
  
  wire async_fifo_valid;
  assign async_fifo_valid = ready & valid;
  
  
  async_fifo2
  #(
    .g_width(      32'd64),
    .g_log_depth(  32'd2)
  )
  u_async_fifo2
  (
    /* control (async) */
    
    /* write domain (regular clk) */
    .w_rst_n(        rst_n),
    .w_clk(          clk),
    
    .w_enable(       global_enable),
    .w_ready(        async_fifo_ready),
    .w_data(         {len,
                      sb11,sb10,sb9,sb8,sb7,sb6,
                      sb5,sb4,sb3,sb2,sb1,sb0}),
    .w_valid(        async_fifo_valid),

    /* read domain (ldpc clk) */
    .r_rst_n(        ldpc_rst_n),
    .r_clk(          ldpc_clk),
    
    .r_enable(       ldpc_global_enable),
    .r_ready(        s0_ready),
    .r_data(         {s0_len,s0_data}),
    .r_valid(        s0_valid)
  );

  assign s0_mask1        = (s0_valid & s0_len[0])?{         55'b0,{ 5{1'b1}}}:60'd0;
  assign s0_mask2        = (s0_valid & s0_len[1])?{s0_mask1[49:0],{10{1'b1}}}:s0_mask1;
  assign s0_mask3        = (s0_valid & s0_len[2])?{s0_mask2[39:0],{20{1'b1}}}:s0_mask2;
  assign s0_mask4        = (s0_valid & s0_len[3])?{s0_mask3[19:0],{40{1'b1}}}:s0_mask3;
  assign s0_data_mask    = s0_data & s0_mask4;
 
  /*****************************************************************************
  * s1 output fifo
  *****************************************************************************/
  reg  [109:0] s1_fifo;
  wire [109:0] n_s1_fifo;
  reg  [  4:0] s1_fifo_count;
  wire [  4:0] n_s1_fifo_count;
  
  wire [ 3:0]  s1_fifo_wrlen;
  wire [ 3:0]  s1_fifo_rdlen;
  wire [ 4:0]  s1_fifo_in_shsel;
  
  wire [109:0] s1_fifo_in_sh1,s1_fifo_in_sh2,s1_fifo_in_sh3,s1_fifo_in_sh4,s1_fifo_in_sh5;
  wire [109:0] s1_fifo_out_sh1,s1_fifo_out_sh2,s1_fifo_out_sh3,s1_fifo_out_sh4;
      
  assign s0_ready        = s1_fifo_count<=5'd10;

  assign s1_fifo_wrlen   = s0_len & {4{s0_ready & s0_valid}};
  assign s1_fifo_rdlen   = (ldpc_valid>ldpc_ready)?ldpc_ready:ldpc_valid;
  
  assign s1_fifo_in_shsel= s1_fifo_count - {1'b0,s1_fifo_rdlen};
  
  assign s1_fifo_in_sh1  = s1_fifo_in_shsel[0]?{ 45'b0,s0_data_mask, 5'b0}:{50'b0,s0_data_mask};
  assign s1_fifo_in_sh2  = s1_fifo_in_shsel[1]?{s1_fifo_in_sh1[99:0],10'b0}:s1_fifo_in_sh1;
  assign s1_fifo_in_sh3  = s1_fifo_in_shsel[2]?{s1_fifo_in_sh2[89:0],20'b0}:s1_fifo_in_sh2;
  assign s1_fifo_in_sh4  = s1_fifo_in_shsel[3]?{s1_fifo_in_sh3[69:0],40'b0}:s1_fifo_in_sh3;
  assign s1_fifo_in_sh5  = s1_fifo_in_shsel[4]?{s1_fifo_in_sh4[29:0],80'b0}:s1_fifo_in_sh4;
  
  assign s1_fifo_out_sh1 = s1_fifo_rdlen[0]?{ 5'b0,        s1_fifo[109: 5]}:s1_fifo;
  assign s1_fifo_out_sh2 = s1_fifo_rdlen[1]?{10'b0,s1_fifo_out_sh1[109:10]}:s1_fifo_out_sh1;
  assign s1_fifo_out_sh3 = s1_fifo_rdlen[2]?{20'b0,s1_fifo_out_sh2[109:20]}:s1_fifo_out_sh2;
  assign s1_fifo_out_sh4 = s1_fifo_rdlen[3]?{40'b0,s1_fifo_out_sh3[109:40]}:s1_fifo_out_sh3;
  
  assign n_s1_fifo       = s1_fifo_in_sh5 | s1_fifo_out_sh4;
  assign n_s1_fifo_count = s1_fifo_count + {1'b0,s1_fifo_wrlen} - {1'b0,s1_fifo_rdlen};

  always @(posedge ldpc_clk,negedge ldpc_rst_n)
  begin
    if(!ldpc_rst_n)
    begin
      s1_fifo       <= 110'd0;
      s1_fifo_count <= 5'd0;
    end
    else if(!ldpc_global_enable)
    begin
      s1_fifo       <= 110'd0;
      s1_fifo_count <= 5'd0;
    end
    else
    begin
      s1_fifo       <= n_s1_fifo;
      s1_fifo_count <= n_s1_fifo_count;
    end
  end
 
  wire [49:0] s1_fifo_mask1,s1_fifo_mask2,s1_fifo_mask3,s1_fifo_mask4;
 
  assign s1_fifo_mask1   = s1_fifo_rdlen[0]?{45'b0,{5{1'b1}}}:50'b0;
  assign s1_fifo_mask2   = s1_fifo_rdlen[1]?{s1_fifo_mask1[39:0],{10{1'b1}}}:s1_fifo_mask1;
  assign s1_fifo_mask3   = s1_fifo_rdlen[2]?{s1_fifo_mask2[29:0],{20{1'b1}}}:s1_fifo_mask2;
  assign s1_fifo_mask4   = s1_fifo_rdlen[3]?{s1_fifo_mask3[ 9:0],{40{1'b1}}}:s1_fifo_mask3;
 
  assign ldpc_valid      = (s1_fifo_count>=5'd10)?4'd10:s1_fifo_count[3:0];
  assign {ldpc_sb9,ldpc_sb8,ldpc_sb7,ldpc_sb6,ldpc_sb5,ldpc_sb4,ldpc_sb3,ldpc_sb2,ldpc_sb1,ldpc_sb0} = s1_fifo[49:0] & s1_fifo_mask4;

  /*****************************************************************************
  * paramters resynchronization
  *****************************************************************************/
  wire  ldpc_update_m1t;
  ClkSyncSimple u_parameters_resync
  ( 
    .dstclk(          ldpc_clk),
    .dstresetn(       ldpc_rst_n),
    .srcdata(         ldpc_en),
    .dstdata(         ldpc_update_m1t)
  );

  always @(posedge ldpc_clk,negedge ldpc_rst_n)
  begin
    if(!ldpc_rst_n)
    begin
      ldpc_update    <= 1'b0;
      ldpc_format    <= 4'd0;
      ldpc_bandwidth <= 3'd0;
      ldpc_mcs       <= 7'd0;
      ldpc_cr        <= 2'b0;
      ldpc_nbpsc     <= 3'b0;
      ldpc_nss       <= 3'b0;
      ldpc_stbc      <= 1'b0;
      ldpc_rulen     <= 3'd0;
      ldpc_nsym      <= 16'd0;
      ldpc_ncbps     <= 16'd0;
      ldpc_ncw       <= 12'd0;
      ldpc_lldpc     <= 2'd0;
      ldpc_nshrtq    <= 12'd0; 
      ldpc_nshrtr    <= 12'd0; 
      ldpc_npuncq    <= 12'd0; 
      ldpc_npuncr    <= 12'd0; 
      ldpc_nrepq     <= 14'd0; 
      ldpc_nrepr     <= 12'd0; 
    end
    else if(!ldpc_global_enable)
    begin
      ldpc_update    <= 1'b0;
      ldpc_format    <= 4'd0;
      ldpc_bandwidth <= 3'd0;
      ldpc_mcs       <= 7'd0;
      ldpc_cr        <= 2'b0;
      ldpc_nbpsc     <= 3'b0;
      ldpc_nss       <= 3'b0;
      ldpc_stbc      <= 1'b0;
      ldpc_rulen     <= 3'd0;
      ldpc_nsym      <= 16'd0;
      ldpc_ncbps     <= 16'd0;
      ldpc_ncw       <= 12'd0;
      ldpc_lldpc     <= 2'd0;
      ldpc_nshrtq    <= 12'd0; 
      ldpc_nshrtr    <= 12'd0; 
      ldpc_npuncq    <= 12'd0; 
      ldpc_npuncr    <= 12'd0; 
      ldpc_nrepq     <= 14'd0; 
      ldpc_nrepr     <= 12'd0; 
    end
    else
    begin
      ldpc_update <= ldpc_update_m1t;
      if(ldpc_update_m1t && !ldpc_update)
      begin
        ldpc_format    <= format;
        ldpc_bandwidth <= bandwidth;
        ldpc_mcs       <= mcs;
        ldpc_cr        <= cr;
        ldpc_nbpsc     <= nbpsc;
        ldpc_nss       <= nss;
        ldpc_stbc      <= stbc;
        ldpc_rulen     <= rulen;
        ldpc_nsym      <= nsym;
        ldpc_ncbps     <= ncbps;
        ldpc_ncw       <= ncw;
        ldpc_lldpc     <= lldpc;
        ldpc_nshrtq    <= nshrtq;
        ldpc_nshrtr    <= nshrtr;
        ldpc_npuncq    <= npuncq;
        ldpc_npuncr    <= npuncr;
        ldpc_nrepq     <= nrepq;
        ldpc_nrepr     <= nrepr;
      end
    end
  end

endmodule
`default_nettype wire
