/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
module ViterbiController #(parameter REGLENGTHG = 120,
                           parameter CNTRSIZEG  = 8 // Indicated out of 8 how many are valid bits (during flush, parallel
                                                    // reading). To represent 8  4 bits are required.
                          )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            input   wire                                   nHrdRst,
            input   wire                                   ViterbiClk,
            input   wire                                   Enable,
            input   wire                                   FlushEop,     // At the end of valid fields for parallel read from Reg0
            input   wire                                   StartFieldIn, // Beginning of Valid fields
            input   wire                                   DataValidIn,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output  wire                                   VtrbInValid,
            output  wire [CNTRSIZEG-1:0]                   CntrVtrbRd,  // INDEX for parallel read
                                                                        // from Reg0
            output  wire                                   FlushVtrb,   // Do parallel reading from
                                                                        // Reg0 when high
            output  wire                                   InitViterbi, // Initialize PM,
                                                                        // TB Reg in IDLE state
            output  wire                                   ValidOut,
            output  wire                                   FlushDoneP
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
//-- Controller state machine state variables
localparam IDLE       = 4'b0001;
localparam TB_FILLING = 4'b0010;
localparam TB_FULL    = 4'b0100;
localparam FLUSH_TB   = 4'b1000;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg [3:0]              CntlCurrSt; // Current state
                                   //register for FSM
reg [3:0]              CntlNxtSt;  // Next state for FSM
reg [CNTRSIZEG-1:0]    PrllRdIndx; //Read index for
                                     // parallel reading
                                     //during flush
reg                    InitPath;
reg                    VtrbDataOutValid;
reg                    CounterEn;
reg                    CounterRst;
reg                    FlushFrVtrb;

reg [CNTRSIZEG-1:0]    CntrVal; // counter
reg [6:0]              VtrbVldOutDly; // 7 pipeline for no. of
                                      //valid out bits
reg [6:0]              DataInValidDly;
reg [CNTRSIZEG-1:0]    PrllRdDly[5:0];
reg [5:0]              FlushVtrbDly; // Delayed Flush signal for viterbi
reg [6:0]              FlushDoneDly; // Delayed Flush  Done signal from viterbi


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
wire        VtrbInValidInt;
wire        FlushDonePInt;
integer     i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign FlushVtrb     = FlushVtrbDly[5];
assign CntrVtrbRd    = PrllRdDly[5] ;
assign InitViterbi   = InitPath;
assign ValidOut      = VtrbVldOutDly[6];
assign FlushDoneP    = FlushDoneDly[6];

always @(posedge ViterbiClk or negedge nHrdRst)
begin
   if(!nHrdRst) 
   begin
     DataInValidDly      <=  7'b0; // {(6){1'b0}};
   end
   else if(!Enable)
   begin
     DataInValidDly      <=  7'b0; // {(6){1'b0}};
   end
   else 
   begin
     DataInValidDly[6:0] <= {DataInValidDly[5:0],VtrbInValidInt};
   end // else: !if(nHrdRst == 1'b0)
end // always


//---------------------------------------------------------
//        Registering & delaying control signal(pipeline in Viterbi)
//---------------------------------------------------------

// Delay of 6 clocks are due to 2 stage registering in BMU,
// 1 registering for survivor path selection, 3 stages of pipelining
// in adderlogic for viterbi o/p generation .
//  Flush for Viterbi has one less index for compensating pipelining
// delay because the signal assignment is in other state which
// causes one clock delay hence total delay becomes same (6 cycles).

always@(posedge ViterbiClk or negedge nHrdRst)
begin:pCntrSigDlySeq
 if(nHrdRst == 1'b0) 
 begin
   FlushVtrbDly   <= 6'b0;
   FlushDoneDly   <= 7'b0;
   VtrbVldOutDly  <= 7'b0;
   for(i=0; i<6;i=i+1)
     PrllRdDly[i] <= {(CNTRSIZEG){1'b0}};
 end
 else if(!Enable)
 begin
   FlushVtrbDly   <= 6'b0;
   FlushDoneDly   <= 7'b0;
   VtrbVldOutDly  <= 7'b0;
   for(i=0; i<6;i=i+1)
     PrllRdDly[i] <= {(CNTRSIZEG){1'b0}};
 end
 else 
 begin

    // Delaying the Flushing signal for Viterbi to clear the  
    // pipelining and start parallel reading( 1 less because  
    // parallel read is done without delay                    

    FlushVtrbDly[5:0] <= {FlushVtrbDly[4:0],FlushFrVtrb};

    FlushDoneDly[6:0] <= {FlushDoneDly[5:0],FlushDonePInt};

    // Delayed signal for mux index during viterbi flushing   
    // (parallel read)
    PrllRdDly[0] <= PrllRdIndx;
    PrllRdDly[1] <= PrllRdDly[0];
    PrllRdDly[2] <= PrllRdDly[1];
    PrllRdDly[3] <= PrllRdDly[2];
    PrllRdDly[4] <= PrllRdDly[3];
    PrllRdDly[5] <= PrllRdDly[4];

    // Delayed signal to tell how many bits are valid in the  
    // out data bus                                           
    VtrbVldOutDly[6:0] <= {VtrbVldOutDly[5:0],VtrbDataOutValid};
  end // !nHrdRst
end // always


 //* The Valid data out signal obtained from the Stream
 //  Mux and Depuncturer block is directly passed to the
 //  Viterbi Decoder block
 assign VtrbInValidInt  = DataValidIn;
 assign VtrbInValid     = VtrbInValidInt;

//---------------------------------------------------------
//        Counter logic for o/p Parallel read during Flush
//---------------------------------------------------------

always@(posedge ViterbiClk or negedge nHrdRst)
begin:pCntrSeq
  if(nHrdRst == 1'b0) 
  begin
    CntrVal    <= {{CNTRSIZEG}{1'b0}};
    PrllRdIndx <= {{CNTRSIZEG}{1'b0}};
  end
  else if(!Enable)
  begin
    CntrVal    <= {{CNTRSIZEG}{1'b0}};
    PrllRdIndx <= {{CNTRSIZEG}{1'b0}};
  end
  else
  begin
    if(CounterRst == 1'b1) 
    begin                                             
       CntrVal    <= {{CNTRSIZEG}{1'b0}};                                    
       PrllRdIndx <= {{CNTRSIZEG}{1'b0}};                                    
    end                                                                      
    else 
    begin                                                               
      // Increment the cntr till TB reg is full  Increment the               
      // cntr for every 4 soft bit valid in till TB reg is full.             
      // Every cnt val represents 2 decoded o/p bits                         

      if((CounterEn ==1'b1) && (VtrbInValidInt == 1'b1)) 
      begin               
        CntrVal <= CntrVal + {{(CNTRSIZEG-1){1'b0}}, 1'b1};
      end                                                                    

      // Start decrementing mux index for parallel read during               
      // and register the value when next state is flush_tb                  
      if(CntlCurrSt !=FLUSH_TB) 
      begin                                        
        PrllRdIndx <= CntrVal;                                               
      end                                                                    
      else 
      begin                                                             
        PrllRdIndx <= PrllRdIndx - {{(CNTRSIZEG-1){1'b0}}, 1'b1};
      end                                                                    
    end                                     
  end // else : !if((nHrdRst == 1'b0)
end //always : pCntrSeq

always@(posedge ViterbiClk or negedge nHrdRst)
begin:pInitPathSeq
  if(nHrdRst == 1'b0)
  begin
    InitPath <= 1'b0;
  end
  else if(!Enable || FlushDoneDly[5])
  begin
    InitPath <= 1'b1;
  end
  else
  begin
    InitPath <= 1'b0;
  end // else : !if((nHrdRst == 1'b0)
end //always : pInitPathSeq

//---------------------------------------------------------
//        Viterbi controller State machine
//---------------------------------------------------------

//---------Current state assignment-----------
always@(posedge ViterbiClk or negedge nHrdRst)
begin:pCntlCurrStSeq
  if(nHrdRst == 1'b0)
  begin
    CntlCurrSt <= IDLE;
  end
  else if(!Enable)
  begin
    CntlCurrSt <= IDLE;
  end
  else
  begin
    CntlCurrSt <= CntlNxtSt;
  end // else : !if((nHrdRst == 1'b0)
end //always : pCntlCurrStSeq


assign FlushDonePInt=(CntlNxtSt==IDLE) & (CntlCurrSt==FLUSH_TB);

//---------Next state assignment-----------

always @(*)
begin :pCntlNxtStCom

 case(CntlCurrSt)

  IDLE:
  begin
    VtrbDataOutValid = 1'b0; //Indicates out of 8 how none are valid decoded bits
    CounterEn        = 1'b0;
    CounterRst       = 1'b1;
    FlushFrVtrb      = 1'b0; //Flushing signal for viterbi to start parallel rd

    if(StartFieldIn == 1'b1) begin
      CntlNxtSt     = TB_FILLING;
    end
    else begin
      CntlNxtSt     = IDLE;
    end
  end //case:IDLE

  TB_FILLING:
  begin
    VtrbDataOutValid = 1'b0; //Indicates out of 8 none  are valid decoded bits

    CounterEn        = 1'b1; // Enable the counter to count valid decoder i/p
    CounterRst       = 1'b0;
    FlushFrVtrb      = 1'b0;  //Flushing signal for viterbi to start parallel rd

    if((CntrVal >= (REGLENGTHG/2)-1) && (VtrbInValidInt==1'b1)) begin // Every clk 2 decoded bits out
                                                                      //so factor of 2
      CntlNxtSt = TB_FULL;
    end
    else begin
      if(FlushEop ==1'b1) begin
        CntlNxtSt = FLUSH_TB;
      end
      else begin
        CntlNxtSt = TB_FILLING;
      end //else:!(FlushEop)
    end //else:!(CntrVal == REGLENGTHG/2)

  end //case:TB_FILLING

  TB_FULL:
  begin
    if(VtrbInValidInt == 1'b1)
      VtrbDataOutValid = 1'b1; //2 bits are valid once TB full
    else
      VtrbDataOutValid = 1'b0;

    CounterEn       = 1'b0; // Stop the counter as TB reg is full
    CounterRst      = 1'b0;
    FlushFrVtrb     = 1'b0;

    if(FlushEop ==1'b1) begin
      CntlNxtSt     = FLUSH_TB;
    end
    else begin
      CntlNxtSt     = TB_FULL;
    end // else: !(FlushEop)
  end //case:TB_FULL

  FLUSH_TB:
  begin
    CounterEn       = 1'b0;
    CounterRst      = 1'b0;
    FlushFrVtrb     = 1'b1;

    //If the parallel read index is non-zero, there
    //there are bits remaining in the TB register
    if(PrllRdIndx != {CNTRSIZEG{1'b0}}) begin
      VtrbDataOutValid   =  1'b1;
    end
    else begin
      VtrbDataOutValid   = 1'b0;
    end

    if(PrllRdIndx == {CNTRSIZEG{1'b0}}) begin
      CntlNxtSt     = IDLE;
    end
    else begin
      CntlNxtSt     = FLUSH_TB;
    end
  end //case:FLUSH_TB

  default:
  begin
    VtrbDataOutValid = 1'b0;

    CounterEn        = 1'b0; // Counter is decremented during flushing
    CounterRst       = 1'b1;
    FlushFrVtrb      = 1'b0;
    CntlNxtSt = IDLE;
  end //case:case:default

 endcase //case(CntlCurrSt)
end //always :pCntlNxtStCom

`ifdef RW_SIMU_ON
// assigning Strings to States so that it would be easy
// to debug in simulation
  reg [16*8-1:0] VtbCntlCurrStStr;
  
  always @(*)
  begin
    case (CntlCurrSt)
      IDLE       : VtbCntlCurrStStr = "IDLE";
      TB_FILLING : VtbCntlCurrStStr = "TB_FILLING";
      TB_FULL    : VtbCntlCurrStStr = "TB_FULL";
      FLUSH_TB   : VtbCntlCurrStStr = "FLUSH_TB";
      default    : VtbCntlCurrStStr = "UNKNOWN";
    endcase
  end
`endif

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
