////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 38881 $
// $Date: 2019-05-21 17:42:07 +0200 (Tue, 21 May 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Computes channel estimate
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/ChannelEstimStream.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module ChannelEstimStream #(parameter FFTBUFWIDTH   = 13, // Data width of the FFT memory
                            parameter HBUFDATAWIDTH = 13  // Data width of one channel coefficient inside H memory
                           )(

            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input   wire                            nPhyRst,          // Active Low Reset
            input   wire                            PhyClk,           // Phy Clock

            ///////////////////////////////////////////////
            // Control Signals
            ///////////////////////////////////////////////
            input   wire                            ReadFromH,        // Read data from H

            ///////////////////////////////////////////////
            // Data Related
            ///////////////////////////////////////////////
            input  wire        [1:0]                HDataMult,        // Shifting
            input  wire        [1:0]                RoundLSB,         // Rounding
            input  wire                             PtCoeff,          // Pt Matrix
            input  wire                             PreambleCoeff,    // Preamble Values
            //
            input  wire signed [FFTBUFWIDTH-1:0]    FFTBufReadDataRe, // Real Component
            input  wire signed [FFTBUFWIDTH-1:0]    FFTBufReadDataIm, // Imag Component
            //
            input  wire signed [HBUFDATAWIDTH-1:0]  HBufReadDataRe,   // Real Component
            input  wire signed [HBUFDATAWIDTH-1:0]  HBufReadDataIm,   // Imag Component
            input  wire                             HBufReadDataEnIn, // Qualifies Rd Data
            //
            output wire signed [HBUFDATAWIDTH-1:0]  EstDataOutRe,     // Real Comp of Output
            output wire signed [HBUFDATAWIDTH-1:0]  EstDataOutIm      // Imag Comp of Output
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed  [HBUFDATAWIDTH-1:0]     CONST_ZERO_HBUFDATAWIDTH    = {{HBUFDATAWIDTH}{1'b0}};
localparam signed  [HBUFDATAWIDTH+1:0]     CONST_ZERO_HBUFDATAWIDTH_P2 = {{HBUFDATAWIDTH+2}{1'b0}};
localparam signed  [FFTBUFWIDTH-1:0]       CONST_ZERO_FFTBUFWIDTH      = {{FFTBUFWIDTH}{1'b0}};
localparam signed  [HBUFDATAWIDTH-1:0]     CONST_1_HBUFDATAWIDTH       = {{{(HBUFDATAWIDTH-1)}{1'b0}},1'b1};  // 'd1
localparam signed  [HBUFDATAWIDTH-1:0]     CONST_2_HBUFDATAWIDTH       = {{{(HBUFDATAWIDTH-2)}{1'b0}},2'b10}; // 'd2

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire                                   FFTBufDataSel;
wire    signed     [FFTBUFWIDTH:0]     IntFFTRd1Re;
wire    signed     [FFTBUFWIDTH:0]     IntFFTRd1Im;
wire    signed     [FFTBUFWIDTH-1:0]   IntFFTRd1ReSat;
wire    signed     [FFTBUFWIDTH-1:0]   IntFFTRd1ImSat;
wire    signed     [HBUFDATAWIDTH+1:0] IntSumRe;
wire    signed     [HBUFDATAWIDTH+1:0] IntSumIm;
wire    signed     [HBUFDATAWIDTH-1:0] EstDataRe;
wire    signed     [HBUFDATAWIDTH-1:0] EstDataIm;
wire    signed     [HBUFDATAWIDTH-1:0] RndSum1Re;
wire    signed     [HBUFDATAWIDTH-1:0] RndSum2Re;
wire    signed     [HBUFDATAWIDTH-1:0] RndSum1Im;
wire    signed     [HBUFDATAWIDTH-1:0] RndSum2Im;


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [HBUFDATAWIDTH+1:0] IntHData1Re;
reg     signed     [HBUFDATAWIDTH+1:0] IntHData1Im;
reg     signed     [HBUFDATAWIDTH-1:0] RndSumRe;
reg     signed     [HBUFDATAWIDTH-1:0] RndSumIm;
reg     signed     [HBUFDATAWIDTH+1:0] IntSumRe_1t;
reg     signed     [HBUFDATAWIDTH+1:0] IntSumIm_1t;
reg     signed     [FFTBUFWIDTH-1:0]   IntFFTRd1Re_1t;
reg     signed     [FFTBUFWIDTH-1:0]   IntFFTRd1Im_1t;
reg     signed     [HBUFDATAWIDTH-1:0] HBufReadDataRe_1t;
reg     signed     [HBUFDATAWIDTH-1:0] HBufReadDataIm_1t;
reg                                    HBufReadDataEnIn_1t;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module calculates one channel estimate value for a sub carrier.

// Step 1 - Xor the incoming PtCoeff Value and the Preamble Value
assign FFTBufDataSel = PtCoeff ^ PreambleCoeff;

// Step 2 - Choose the positive or negative value of FFTBufReadData
assign IntFFTRd1Re = (FFTBufDataSel) ? -$signed({FFTBufReadDataRe[FFTBUFWIDTH-1],FFTBufReadDataRe}) : {FFTBufReadDataRe[FFTBUFWIDTH-1],FFTBufReadDataRe};
assign IntFFTRd1Im = (FFTBufDataSel) ? -$signed({FFTBufReadDataIm[FFTBUFWIDTH-1],FFTBufReadDataIm}) : {FFTBufReadDataIm[FFTBUFWIDTH-1],FFTBufReadDataIm};

// Step 3 - Multiply HBufReadData by 1/2/4 (signed left shift by HDataMult)
always @ (*)
   begin: IntHData1Re_Blk
      case (HDataMult)
         2'b00  : IntHData1Re = {HBufReadDataRe[HBUFDATAWIDTH-1],HBufReadDataRe[HBUFDATAWIDTH-1],HBufReadDataRe};
         2'b01  : IntHData1Re = {HBufReadDataRe[HBUFDATAWIDTH-1],HBufReadDataRe,1'b0};
         default: IntHData1Re = {HBufReadDataRe,2'b0};
      endcase
   end //IntHData1Re_Blk

always @ (*)
   begin: IntHData1Im_Blk
      case (HDataMult)
         2'b00  : IntHData1Im = {HBufReadDataIm[HBUFDATAWIDTH-1],HBufReadDataIm[HBUFDATAWIDTH-1],HBufReadDataIm};
         2'b01  : IntHData1Im = {HBufReadDataIm[HBUFDATAWIDTH-1],HBufReadDataIm,1'b0};
         default: IntHData1Im = {HBufReadDataIm,2'b0};
      endcase
   end //IntHData1Im_Blk

SatSymSigned #(
               .INPUT_WIDTH(FFTBUFWIDTH+1),
               .OUTPUT_WIDTH(FFTBUFWIDTH)
              )
               U_SATFFTRE(
                        .InputData(IntFFTRd1Re),
                        .SatSymData(IntFFTRd1ReSat)
                        );

SatSymSigned #(
               .INPUT_WIDTH(FFTBUFWIDTH+1),
               .OUTPUT_WIDTH(FFTBUFWIDTH)
              )
               U_SATFFTIM(
                        .InputData(IntFFTRd1Im),
                        .SatSymData(IntFFTRd1ImSat)
                       );

// Step 4a - Add IntHData1 and IntFFTRd1
// assign IntSumRe = $signed({IntFFTRd1Re[FFTBUFWIDTH-1],IntFFTRd1Re}) + IntHData1Re;
// assign IntSumIm = $signed({IntFFTRd1Im[FFTBUFWIDTH-1],IntFFTRd1Im}) + IntHData1Im;
assign IntSumRe = $signed({{2{IntFFTRd1ReSat[FFTBUFWIDTH-1]}},IntFFTRd1ReSat}) + IntHData1Re;
assign IntSumIm = $signed({{2{IntFFTRd1ImSat[FFTBUFWIDTH-1]}},IntFFTRd1ImSat}) + IntHData1Im;

// Step 4b - Register the Sum
always @ (posedge PhyClk or negedge nPhyRst)
   begin: IntSumReg_Blk
      if (nPhyRst == 1'b0) begin
         IntSumRe_1t <= CONST_ZERO_HBUFDATAWIDTH_P2;
         IntSumIm_1t <= CONST_ZERO_HBUFDATAWIDTH_P2;
      end
      else begin
         IntSumRe_1t <= IntSumRe;
         IntSumIm_1t <= IntSumIm;
      end
   end //IntSumReg_Blk

// Step 5 - Round IntSum by RoundLSB (0/1/2 bits)
// Round IntSumRe by 1 bit
Round #(.INPUT_WIDTH(HBUFDATAWIDTH+1),
        .OUTPUT_WIDTH(HBUFDATAWIDTH)
       ) U_ROUND1(.InputData($signed(IntSumRe_1t[HBUFDATAWIDTH:0])),
                  .RoundData(RndSum1Re[HBUFDATAWIDTH-1:0]));

// Round IntSumRe by 2 bits
Round #(.INPUT_WIDTH(HBUFDATAWIDTH+2),
        .OUTPUT_WIDTH(HBUFDATAWIDTH)
       ) U_ROUND2(.InputData($signed(IntSumRe_1t[HBUFDATAWIDTH+1:0])),
                  .RoundData(RndSum2Re[HBUFDATAWIDTH-1:0]));

// Round IntSumIm by 1 bit
Round #(.INPUT_WIDTH(HBUFDATAWIDTH+1),
        .OUTPUT_WIDTH(HBUFDATAWIDTH)
       ) U_ROUND3(.InputData($signed(IntSumIm_1t[HBUFDATAWIDTH:0])),
                  .RoundData(RndSum1Im[HBUFDATAWIDTH-1:0]));

// Round IntSumIm by 2 bits
Round #(.INPUT_WIDTH(HBUFDATAWIDTH+2),
        .OUTPUT_WIDTH(HBUFDATAWIDTH)
       ) U_ROUND4(.InputData($signed(IntSumIm_1t[HBUFDATAWIDTH+1:0])),
                  .RoundData(RndSum2Im[HBUFDATAWIDTH-1:0]));

always @ (*)
   begin: RndSumRe_Blk
      case (RoundLSB)

         2'b00  : RndSumRe = $signed(IntSumRe_1t[HBUFDATAWIDTH-1:0]);
         2'b01  : RndSumRe = $signed(RndSum1Re[HBUFDATAWIDTH-1:0]);
         2'b10  : RndSumRe = $signed(RndSum2Re[HBUFDATAWIDTH-1:0]);
         default: RndSumRe = $signed(IntSumRe_1t[HBUFDATAWIDTH-1:0]);

      endcase
   end //RndSumRe_Blk

always @ (*)
   begin: RndSumIm_Blk
      case (RoundLSB)

         2'b00  : RndSumIm = $signed(IntSumIm_1t[HBUFDATAWIDTH-1:0]);
         2'b01  : RndSumIm = $signed(RndSum1Im[HBUFDATAWIDTH-1:0]);
         2'b10  : RndSumIm = $signed(RndSum2Im[HBUFDATAWIDTH-1:0]);
         default: RndSumIm = $signed(IntSumIm_1t[HBUFDATAWIDTH-1:0]);

      endcase
   end //RndSumIm_Blk

// Step 6 - Select result of Step 2 or Step 5 depending on HBufReadDataEnIn
// Register the output of Step 2 so that the delay is the same
always @ (posedge PhyClk or negedge nPhyRst)
   begin: IntFFTRdReg_Blk
      if (nPhyRst == 1'b0) begin
         IntFFTRd1Re_1t <= CONST_ZERO_FFTBUFWIDTH;
         IntFFTRd1Im_1t <= CONST_ZERO_FFTBUFWIDTH;
      end
      else begin
         IntFFTRd1Re_1t <= IntFFTRd1ReSat;
         IntFFTRd1Im_1t <= IntFFTRd1ImSat;
      end
   end //IntFFTRdReg_Blk

// Delay HBufReadDataEnIn_1t also by one clock to compensate for
// computational delay in this module
always @ (posedge PhyClk or negedge nPhyRst)
   begin: HBufReadDataEnIn_1t_Blk
      if (nPhyRst == 1'b0)
         HBufReadDataEnIn_1t <= 1'b0;
      else
         HBufReadDataEnIn_1t <= HBufReadDataEnIn;
   end //HBufReadDataEnIn_1t_Blk

assign EstDataRe = (HBufReadDataEnIn_1t) ? RndSumRe : IntFFTRd1Re_1t;
assign EstDataIm = (HBufReadDataEnIn_1t) ? RndSumIm : IntFFTRd1Im_1t;

// Registered values of the input HBuf data
// Used only in ReadFromH case
always @ (posedge PhyClk or negedge nPhyRst)
   begin: HBufReadDataReg_Blk
      if (nPhyRst == 1'b0) begin
         HBufReadDataRe_1t <= CONST_ZERO_HBUFDATAWIDTH;
         HBufReadDataIm_1t <= CONST_ZERO_HBUFDATAWIDTH;
      end
      else begin
         HBufReadDataRe_1t <= HBufReadDataRe;
         HBufReadDataIm_1t <= HBufReadDataIm;
      end
   end //HBufReadDataReg_Blk


// Step 7 - Select module output depending on the following criterion:
// a)If ReadFromH is high, send out the H Mem data after registering
// b)Else send out the result of Step 6
assign EstDataOutRe = (ReadFromH) ? HBufReadDataRe_1t : EstDataRe;
assign EstDataOutIm = (ReadFromH) ? HBufReadDataIm_1t : EstDataIm;


endmodule // ChannelEstimStream

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
