//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 33993 $
// $Date: 2018-05-16 17:34:14 +0200 (Wed, 16 May 2018) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Main CORDIC Module. Rotation by Delta.
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/CordicIndRot.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module CordicIndRot #(parameter ITERATION = 13,
                      parameter DATAWIDTH = 13
                     )(

            ///////////////////////////////////////////////
            // Clock and reset
            ///////////////////////////////////////////////
            input  wire                                    PhyClk,   // Active LOW Reset
            input  wire                                    nPhyRst,  // PHY Clock

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Control Signals
            input  wire                                    Enable,  //Active HIGH Enable
            input  wire            [ITERATION-1:0]         DeltaIn, //Delta Value from ROM
            //Data
            input  wire   signed   [DATAWIDTH-1:0]         ReDataIn, //Real Component of Data
            input  wire   signed   [DATAWIDTH-1:0]         ImDataIn, //Imag Component of Data

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output wire signed     [DATAWIDTH-1:0]         ReDataOut, //Real Component of Data
            output wire signed     [DATAWIDTH-1:0]         ImDataOut //Imag Component of Dat
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Var Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [DATAWIDTH:0]       ReDataOutInt;
reg     signed     [DATAWIDTH:0]       ImDataOutInt;
reg     signed     [DATAWIDTH+1:0]     XArray[ITERATION:0];
reg     signed     [DATAWIDTH+1:0]     YArray[ITERATION:0];
reg                [ITERATION-1:0]     DeltaInt[ITERATION:0];

integer i,k;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Read in the inputs to the first slot of the arrays
always @ (*)
   begin: Input_Blk
      XArray[0]   = {ReDataIn[DATAWIDTH-1],ReDataIn[DATAWIDTH-1],ReDataIn};
      YArray[0]   = {ImDataIn[DATAWIDTH-1],ImDataIn[DATAWIDTH-1],ImDataIn};
      DeltaInt[0] = DeltaIn;
   end //Input_Blk


//Function to generate the Normalized Value of the output
//Multiply by normalization factor 1244/2048

function           [DATAWIDTH:0]                  NormalizeFnct;

//Inputs
input    signed    [DATAWIDTH+1:0]                DataIn;

//Internal Registers
reg      signed    [DATAWIDTH+12:0]               Normal;

   begin
      //Multiply by 1244
      //1244 = 1024 + 128 + 64 + 16 + 8 + 4
      Normal = (DataIn <<< 'sd10) + (DataIn <<< 'sd7) + (DataIn <<< 'sd6)
               + (DataIn <<< 'sd4) + (DataIn <<< 'sd3) + (DataIn <<< 'sd2);

      // Divide by 2048 and round
      NormalizeFnct = Normal[DATAWIDTH+11:11]+{{DATAWIDTH{1'b0}},Normal[10]};
   end
endfunction  //NormalizeFnct;


//Perform Rotation
always @(posedge PhyClk or negedge nPhyRst)
   begin: Rotation_Blk
      if (nPhyRst == 1'b0) begin
         for (i = 1; i <= ITERATION; i=i+1) begin
            XArray[i]   <= $signed({{DATAWIDTH+2}{1'b0}});
            YArray[i]   <= $signed({{DATAWIDTH+2}{1'b0}});
            DeltaInt[i] <= {{ITERATION}{1'b0}};
         end
      end
      else if (Enable == 1'b1) begin
         for (k = 1; k <= ITERATION; k=k+1) begin
            //When enabled iterate using the appropriate Delta
            if (DeltaInt[k-1][k-1] == 1'b1) begin
               if (k>1) begin
                 XArray[k]<= XArray[k-1]-((YArray[k-1] >>> (k-1)) + $signed({{DATAWIDTH+1{1'b0}},YArray[k-1][k-2]}));
                 YArray[k]<= YArray[k-1]+((XArray[k-1] >>> (k-1)) + $signed({{DATAWIDTH+1{1'b0}},XArray[k-1][k-2]}));
               end
               else begin
                 XArray[k]<= XArray[k-1]-((YArray[k-1] >>> (k-1)));
                 YArray[k]<= YArray[k-1]+((XArray[k-1] >>> (k-1)));
               end
            end
            else begin
               if (k>1) begin
                 XArray[k]<= XArray[k-1]+((YArray[k-1] >>> (k-1)) + $signed({{DATAWIDTH+1{1'b0}},YArray[k-1][k-2]}));
                 YArray[k]<= YArray[k-1]-((XArray[k-1] >>> (k-1)) + $signed({{DATAWIDTH+1{1'b0}},XArray[k-1][k-2]}));
               end
               else begin
                 XArray[k]<= XArray[k-1]+((YArray[k-1] >>> (k-1)));
                 YArray[k]<= YArray[k-1]-((XArray[k-1] >>> (k-1)));
               end
            end
            DeltaInt[k] <= DeltaInt[k-1];
         end //for loop
      end //else if
   end //Rotation_Blk


//Normalize Seq Process
always @(posedge PhyClk or negedge nPhyRst)
   begin: Normalize_Blk
      if (nPhyRst == 1'b0) begin
         ReDataOutInt <= $signed({{DATAWIDTH+1}{1'b0}});
         ImDataOutInt <= $signed({{DATAWIDTH+1}{1'b0}});
      end
      else if (Enable == 1'b1) begin
         //Normalize and send
         ReDataOutInt <= $signed(NormalizeFnct(XArray[ITERATION]));
         ImDataOutInt <= $signed(NormalizeFnct(YArray[ITERATION]));
      end
   end //Normalize_Blk


//Generate Outputs
//Saturation to DATAWIDTH
SatSymSigned #(
               .INPUT_WIDTH(DATAWIDTH+1),
               .OUTPUT_WIDTH(DATAWIDTH)
              )
               U_SATBLK1(
                        .InputData(ReDataOutInt),
                        .SatSymData(ReDataOut)
                        );

SatSymSigned #(
               .INPUT_WIDTH(DATAWIDTH+1),
               .OUTPUT_WIDTH(DATAWIDTH)
              )
               U_SATBLK2(
                        .InputData(ImDataOutInt),
                        .SatSymData(ImDataOut)
                        );


endmodule //CordicIndRot

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
