//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39801 $
// $Date: 2019-09-16 18:22:03 +0200 (Mon, 16 Sep 2019) $
// ---------------------------------------------------------------------------
// Dependencies     :
// Description      : Cordic, Filter and Memory Read Controller
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/EstimRotCtrl.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module EstimRotCtrl #(
            parameter CORDICLAT = 14 // Cordic Latency Value - 14 clocks
                         )(
            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input  wire                                   nPhyRst,         // Active Low Reset
            input  wire                                   PhyClk,          // Phy Clock

            ///////////////////////////////////////////////
            // Register
            ///////////////////////////////////////////////
            input  wire        [1:0]                      CfgSmoothFilterSet, // Filter Set

            ///////////////////////////////////////////////
            // RX FD FSM Interface
            ///////////////////////////////////////////////
            input  wire        [1:0]                      RxNsts,          // No. of Space Time Steams
            input  wire        [3:0]                      NShiftLTF,       // No. of LTF to shift in case of Mu-MIMO
            input  wire        [3:0]                      ReceptionMode,   // Mode of Current Pkt
            input  wire        [2:0]                      SymbolType,      // Type of Symbol
            input  wire        [2:0]                      RUType,          // Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)
            input  wire        [5:0]                      RUIndex,         // RU allocation index (up to 37)
            input  wire                                   FirstLTF,        // Indicates First LTF
            input  wire                                   SmoothEn,        // Enable smoothing
            input  wire                                   StartSignalP,    // Start of a symbol
            input  wire                                   SmoothfromHP,    // Start Smoothing Only
            input  wire                                   EstimEn,         // Enable Ch Est
            input  wire                                   FrameNDP,        // Indicates frame is NDP
`ifdef RW_MUMIMO_RX_EN
            input  wire                                   HEMuMIMOEn,      // Indicates frame is HE Mu-MIMO
`endif // RW_MUMIMO_RX_EN
            //
            output reg                                    EstimDoneP,      // Estimation Done
            output reg                                    SmthDoneP,       // Smoothing Done
            
            ///////////////////////////////////////////////
            // HWriteCtrl Interface
            ///////////////////////////////////////////////
            input  wire                                   LastWriteP,      // H memory last write
            //
            output reg  signed [9:0]                      LastPosIndex,    // Last index subcarrier
            output reg                                    HWriteI,         // Write I phase
            output reg                                    HWriteQ,         // Write Q phase
            //
            output wire signed [9:0]                      HWriteSCInd,
            output wire                                   HWriteDataEnOut,
        
            ///////////////////////////////////////////////
            // Controls
            ///////////////////////////////////////////////
            output reg         [3:0]                      BlockEn,           // Block enable

            ///////////////////////////////////////////////
            // FFT Memory Read Interface
            ///////////////////////////////////////////////
            output wire        [9:0]                      FFTBufRdAddrOut,   // Address
            output wire                                   FFTBufRdAddrEnOut, // Qualifies Address

            ///////////////////////////////////////////////
            // H memory Read Interface
            ///////////////////////////////////////////////
            output wire        [9:0]                      HBufRdAddrOut,     // Address
            output wire                                   HBufRdAddrEnOut,   // Qualifies Address

            ///////////////////////////////////////////////
            // Equalyzer Interface
            ///////////////////////////////////////////////
            output reg                                    DataChannelEnOut,  // Enable G Matrix Comp

            ///////////////////////////////////////////////
            // Interface to ChannelEstStream Module
            ///////////////////////////////////////////////
            output reg         [1:0]                      HDataMult,         // Shifting
            output reg         [1:0]                      RoundLSB,          // Rounding 
            output wire                                   PreambleCoeff,     // Preamble Value
            output reg         [3:0]                      PtCoeff,           // Pt Matrix Value
            output reg                                    ReadFromH,         // Control memory read data muxing
            output wire                                   DCClear,           // Clear DC value

            ///////////////////////////////////////////////
            // Interface to SmoothTop Module
            ///////////////////////////////////////////////
            output reg         [2:0]                      SmoothStep,        // Smooth steps
            output reg                                    SelIQPhase,        // Select I or Q phase for filter
            output wire                                   ShiftIn,           // Enables input to filter
            output wire                                   DivByJ,            // Division by j enabled when 1
            output wire                                   MultByJ,           // Multiplication by j
            //
            output wire        [3:0]                      RotIndex0,         // Cordic0 Index
            output wire        [3:0]                      RotIndex1,         // Cordic1 Index
            output wire        [3:0]                      RotIndex2,         // Cordic2 Index
            output wire        [3:0]                      RotIndex3,         // Cordic3 Index
            //
            output wire        [1:0]                      RotQuad0,          // Cordic0 Quad
            output wire        [1:0]                      RotQuad1,          // Cordic1 Quad
            output wire        [1:0]                      RotQuad2,          // Cordic2 Quad
            output wire        [1:0]                      RotQuad3,          // Cordic3 Quad
            //
            output wire                                   RotDataEnIn,       // Qualifies Data In
            output wire                                   RotDataEnOut,      // Qualifies Data Out
            //
            output wire                                   PilotAvEn,          // Enable Pilot Averaging
            output wire                                   PilotAvDataInValid, // Enable Pilot Averaging input
            output wire        [9:0]                      PilotAvSCIndex,     // SC Index at Pilot Averaging input
            //
            output wire                                   PilotRead,          // Indicates that the current SC is a pilot to store

            ///////////////////////////////////////////////
            // Interface to Smoothing Filter
            ///////////////////////////////////////////////
            output wire                                   SmoothFilterEn,   // Enable smoothing filter
            //
            output wire signed [9:0]                      SmoothCoeff0,  // Smoothing Coeff 0
            output wire signed [9:0]                      SmoothCoeff1,  // Smoothing Coeff 1
            output wire signed [9:0]                      SmoothCoeff2,  // Smoothing Coeff 2
            output wire signed [9:0]                      SmoothCoeff3,  // Smoothing Coeff 3
            output wire signed [9:0]                      SmoothCoeff4,  // Smoothing Coeff 4
            output wire signed [9:0]                      SmoothCoeff5,  // Smoothing Coeff 5
            output wire signed [9:0]                      SmoothCoeff6,  // Smoothing Coeff 6
            output wire signed [9:0]                      SmoothCoeff7,  // Smoothing Coeff 7
            output wire signed [9:0]                      SmoothCoeff8,  // Smoothing Coeff 8
            output wire signed [9:0]                      SmoothCoeff9,  // Smoothing Coeff 9
            output wire signed [9:0]                      SmoothCoeff10, // Smoothing Coeff 10
            output wire signed [9:0]                      SmoothCoeff11, // Smoothing Coeff 11
            output wire        [1:0]                      SmoothCoeffRS  // Smoothing Right Shift
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
// FSM Size
localparam       FSM_SIZE       = 32'd4;

localparam [3:0] CORDICLAT_SIZE = CORDICLAT;

// Address generator FSM States
localparam [FSM_SIZE-1:0] IDLE        = 4'd0,
                          COUNTNEG1   = 4'd1,
                          WAITFIRST   = 4'd2,
                          COUNTNEG2   = 4'd3,
                          COUNTPOS1   = 4'd4,
                          COUNTPOS2   = 4'd5,
                          EMPTYCORDIC = 4'd6,
                          DC_MODE     = 4'd7,
                          WAITLASTWR  = 4'd8,
                          DONE        = 4'd9;

// Smooth step FSM States
localparam [FSM_SIZE-1:0] IDLE_ST        = 4'd0,
                          CH_ESTIM_ST    = 4'd1,
                          PRE_CORDIC_ST  = 4'd2,
                          DC_I_ST        = 4'd3,
`ifdef RW_NX_DERIV_CHBW20ONLY
                          DC_Q_ST        = 4'd4,
                          VIRTUAL_Q_ST   = 4'd6,
                          SMOOTH_Q_ST    = 4'd8,
                          CH_INTERP_Q_ST = 4'd10,
                          PILOT_Q_ST     = 4'd13,
`endif
                          VIRTUAL_I_ST   = 4'd5,
                          SMOOTH_I_ST    = 4'd7,
                          CH_INTERP_I_ST = 4'd9,
                          POST_CORDIC_ST = 4'd11,
                          END_ST         = 4'd12,
                          PILOT_I_ST     = 4'd14;

// ReceptionMode definition
localparam [3:0] MODE_20MHZ_L     = 4'd0,
                 MODE_20MHZ_HT    = 4'd2,
                 MODE_40MHZ_HT    = 4'd3,
                 MODE_20MHZ_HE_SU = 4'd6,
                 MODE_20MHZ_HE_MU = 4'd7,
                 MODE_40MHZ_HE_SU = 4'd8,
                 MODE_40MHZ_HE_MU = 4'd9,
                 MODE_80MHZ_HE_SU = 4'd10,
                 MODE_80MHZ_HE_MU = 4'd11;

localparam       MODE_BW_IS_20MHZ  = 2'd0,
                 MODE_BW_IS_40MHZ  = 2'd1,
                 MODE_BW_IS_80MHZ  = 2'd2;

   
// SymbolType definition
localparam [2:0] L_LTF    = 3'd0,
                 VHT_LTF  = 3'd1,
                 HT_LTF   = 3'd2,
                 HE_1xLTF = 3'd4,
                 HE_2xLTF = 3'd5,
                 HE_4xLTF = 3'd6,
                 HE_LSIG  = 3'd7;

// SmoothStep definition
localparam [2:0] IDLE_STEP      = 3'd0,
                 SMOOTHING      = 3'd1,
                 DC_INTERP      = 3'd2,
                 VIRTUAL_INTERP = 3'd3,
                 CHAN_INTERP    = 3'd4,
                 POST_CORDIC    = 3'd5,
                 PILOT_INTERP   = 3'd6;

// RUType definition
localparam [2:0] RU26  = 3'd0,
                 RU52  = 3'd1,
                 RU106 = 3'd2,
                 RU242 = 3'd3,
                 RU484 = 3'd4;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [FSM_SIZE-1:0]      AddrGenCS;
reg                [FSM_SIZE-1:0]      AddrGenNS;
reg                [FSM_SIZE-1:0]      SmoothStepCS;

reg                [2:0]               SmoothStep_1t;
reg                [3:0]               ReceptionMode_1t;
reg                [1:0]               ReceptionMode_BW_1t;
reg                [2:0]               SymbolType_1t;
reg                [5:0]               RUIndex_1t;
reg                [2:0]               RUType_1t;
reg  signed        [9:0]               FirstNegIndex;
reg  signed        [9:0]               FirstNegIndex_1t;
reg  signed        [9:0]               LastNegIndex;
reg  signed        [9:0]               LastNegIndex_1t;
reg  signed        [9:0]               FirstPosIndex;
reg  signed        [9:0]               FirstPosIndex_1t;
reg  signed        [9:0]               LastPosIndex_1t;
reg  signed        [9:0]               VirtualIndex;
reg  signed        [9:0]               PilotIndex;
reg  signed        [9:0]               PilotIndex_1t;
reg                [3:0]               WaitCount;
reg                [1:0]               SymbolCount;
reg                                    RdAddrValidSC;
reg                                    RdAddrValidSC_1t;
reg                                    RdAddrValidSC_2t;
reg                                    FFTBufRdAddrEn;
reg                                    HBufRdAddrEn;
reg                [9:0]               RdAddr;
reg                [9:0]               RdAddr_1t;
reg                [9:0]               RdAddr_2t;
reg                [9:0]               RdAddrInc_1t;
reg                                    PreambleCoeffInt;
reg                                    PreambleCoeffInt_1t;
reg                                    AddrGenStartP;
reg                                    PreRotBypass;  // Bypass Pre Cordic delay line
reg                                    PostRotBypass; // Bypass Post Cordic delay line
reg                                    SmoothBypass;  // Bypass smoothing filter delay line
reg                                    CounterSkip;   // Skip index increment for DC or Virtual with only 1 SC interpolation
reg                                    FilterOutValid;
reg                                    FilterOutValid_1t;
reg                                    FilterOutValid_2t;
reg                                    FilterOutValid_3t;
reg                                    SCCordicEn_1t;
reg                                    VirtualHigh;
reg                [9:0]               CoeffVirtual;
reg                [9:0]               CoeffPilot;
reg                                    CurrentSCIsPilot;
reg                                    SmoothingDone;
reg                                    SpecialRU26Idx5;

wire                                   No_Bypass;
wire    signed     [9:0]               SCIndChEstOut;
wire    signed     [9:0]               SCIndCordic;
wire    signed     [9:0]               SCIndCordicOut;
wire                                   SCCordicEn;
wire                                   ShiftInInt_1t;
wire                                   ShiftInInt_3t;
wire               [9:0]               RdAddrOffset;
wire               [9:0]               RdAddrInc;
wire               [9:0]               RdAddrIncOffset;
wire               [9:0]               WrAddrInc;
wire               [9:0]               WrAddrIncOffset;
wire                                   RotDataEnOutInt;
wire               [5:0]               RUIndexSel;
wire               [2:0]               RUTypeSel;
wire               [1:0]               ReceptionMode_BW;

// From post
wire                                   SCIndSmoothFiltProcEn;
wire    signed     [9:0]               SCIndSmoothFiltProc;
wire    signed     [9:0]               SCIndSmoothFiltProcMux;
wire    signed     [9:0]               SCIndFilterCoeff;
wire    signed     [9:0]               SCIndSmoothFiltPreOut;
wire    signed     [9:0]               SCIndSmoothFiltPreOutMux;
wire    signed     [9:0]               SCIndSmoothFiltOut;
wire                                   LUTEnable;
wire                                   LUTEnableMask;
wire                                   LUTSel;
wire               [5:0]               LUTAddr;
wire               [5:0]               RdAddr2s_2t;
wire               [5:0]               RdAddrSel;
wire               [9:0]               RdAddrMux;
wire    signed     [9:0]               DLDataInTap2;
wire    signed     [9:0]               DLDataInTap4;
wire    signed     [9:0]               DLDataInTap4Mask;
wire                                   PilotAvDataEnIn;
wire                                   PilotAvDataEnOut;
wire                                   PilotAvDataEnOutMux;
wire                                   PilotAvDataEnOutMask;
wire                                   PilotAvDataEnOutTap8;
wire                                   PilotAvDataEnOutTap10;
wire                                   PilotAvDataEnOutTap10Mask;
wire                                   PilotAvDataEnOutTap18;
wire                                   SCSmoothFiltOut;
wire               [1:0]               ChanInterpCountCoeff;
wire    signed     [9:0]               CurrentSCInd;

reg     signed     [9:0]               SCIndSmoothFiltProcInt;
reg     signed     [9:0]               SCIndSmoothFiltProcInt_1t;
reg                                    PilotAvDataEnOut_1t;
reg                                    PilotAvDataEnOut_2t;
reg                                    PilotAvDataEnOut_3t;
reg                                    PilotAvDataEnOut_4t;
reg                                    PilotAvDataEnOut_5t;
reg                                    CountDone;
reg                [1:0]               ChanInterpCount;
reg                                    ChanInterpValidMask;
reg                [2:0]               LTFIndex;
reg                [3:0]               PilotCount;
reg                [3:0]               PilotCountSel;

// lut according to                          SmoothStep
//                                           |    Mode Bandwidth
//                                           |    |
reg [39:0]                   Index__L_LTF  [0:6];
reg [39:0]                   Index__VHT_LTF[0:6][0:1];
reg [39:0]                   Index__HT_LTF [0:6][0:1];

// lut according to                                    SmoothStep
//                                                     |    Mode Bandwidth
//                                                     |    |    RUIndex
//                                                     |    |    |    PilotCount
//                                                     |    |    |     |
reg [39:0]                   Index__HE_1xLTF         [0:6][0:1][0:1] [0:7];
reg [39:0]                   Index__HE_2xLTF_SU      [0:6][0:1]      [0:15];
reg [39:0]                   Index__HE_2xLTF_MU_RU26 [0:6][0:2][0:37][0:1];
reg [39:0]                   Index__HE_2xLTF_MU_RU52 [0:6][0:2][0:16] [0:3];
reg [39:0]                   Index__HE_2xLTF_MU_RU106[0:6][0:2][0:8] [0:3];
reg [39:0]                   Index__HE_2xLTF_MU_RU242[0:6][0:1][0:2] [0:7];
reg [39:0]                   Index__HE_2xLTF_MU_RU484[0:6]           [0:15];
reg [39:0]                   Index__HE_4xLTF_SU      [0:6][0:1];
reg [39:0]                   Index__HE_4xLTF_MU_RU26 [0:6][0:2][0:37];
reg [39:0]                   Index__HE_4xLTF_MU_RU52 [0:6][0:2][0:16];
reg [39:0]                   Index__HE_4xLTF_MU_RU106[0:6][0:2][0:8];
reg [39:0]                   Index__HE_4xLTF_MU_RU242[0:6][0:1][0:2];
reg [39:0]                   Index__HE_4xLTF_MU_RU484[0:6][0:1];
reg [39:0]                   Index__HE_LSIG          [0:6];


// lut according to                                  Mode Bandwidth
//                                                   |    RUIndex
//                                                   |    |   
reg [19:0]                IndexVirt__HE_1xLTF      [0:1];
reg [19:0]                IndexVirt__HE_2xLTF_RU26 [0:2][0:37]; // could be split as Index_HE between SU and MU
reg [19:0]                IndexVirt__HE_2xLTF_RU52 [0:2][0:16]; // could be split as Index_HE between SU and MU
reg [19:0]                IndexVirt__HE_2xLTF_RU106[0:2][0:8];  // could be split as Index_HE between SU and MU
reg [19:0]                IndexVirt__HE_2xLTF_RU242[0:1][0:2];  // could be split as Index_HE between SU and MU

   
// lut according to                             Mode Bandwidth
//                                              |    PilotCount
//                                              |    |
reg [19:0]                IndexPilot__HE_1xLTF[0:1][0:15];

// lut according to                                 
//                                                    Mode Bandwidth
//                                                    |    RUIndex
//                                                    |    |     PilotCount
//                                                    |    |     |
reg [19:0]                IndexPilot__HE_2xLTF_SU   [0:1]      [0:15];
reg [19:0]                IndexPilot__HE_2xLTF_RU26 [0:2][0:37][0:3];
reg [19:0]                IndexPilot__HE_2xLTF_RU52 [0:2][0:16][0:3];
reg [19:0]                IndexPilot__HE_2xLTF_RU106[0:2][0:8] [0:3];
reg [19:0]                IndexPilot__HE_2xLTF_RU242[0:1][0:2] [0:7];
reg [19:0]                IndexPilot__HE_2xLTF_RU484           [0:15];


// The following 5 strings are useful during simulations
// to view States using strings instead of numbers
`ifdef RW_SIMU_ON
  reg [25*8:0] SymbolTypeStr;
  reg [25*8:0] ReceptionModeStr;
  reg [25*8:0] AddrGenCSStr;
  reg [25*8:0] AddrGenNSStr;
  //
  reg [25*8:0] SmoothStepCSStr;
  reg [25*8:0] SmoothStepStr;
`endif
  
  integer i,j,k,m;
  
//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// EstimSmoothRx Block Enable Generation
always @ (*)
   begin: Rx_Enable_Blk
      //Default Value
      BlockEn = 4'b0000;
      case (RxNsts[1:0])
         2'b01   : BlockEn = 4'b0011; // HT/VHT/HE-LFT
         2'b10   : BlockEn = 4'b0111;
         2'b11   : BlockEn = 4'b1111;
         default : BlockEn = 4'b0001; // LFT
      endcase //case
`ifdef RW_MUMIMO_RX_EN
      if (HEMuMIMOEn)
        BlockEn = 4'b1111; // Force all block enable for HE-Mu-MIMO as RxNsts=0
`endif // RW_MUMIMO_RX_EN
  end //Rx_Enable_Blk

   wire [19:0] AllZeros = 20'd0;
   
always @(*)
  begin
     for (i=0; i<7;i=i+1) // SmoothStep
     begin
       for (j=0;j<2;j=j+1) // Mode Bandwidth : 20 / 40 MHz
         begin
            Index__L_LTF  [i]             = {AllZeros, AllZeros};
            Index__VHT_LTF[i][j]          = {AllZeros, AllZeros};
            Index__HT_LTF [i][j]          = {AllZeros, AllZeros};
            Index__HE_4xLTF_SU[i][j]      = {AllZeros, AllZeros};
            Index__HE_LSIG[i]             = {AllZeros, AllZeros};
            IndexVirt__HE_1xLTF[j]         = AllZeros;
            Index__HE_4xLTF_MU_RU484[i][j] = {AllZeros, AllZeros};
// lut according to                          SmoothStep
//                                           |   Mode Bandwidth
//                                           |   |   RUIndex
//                                           |   |   |   PilotCount
//                                           |   |   |   |
            for (k=0;k<3;k=k+1) // RU Index
              for (m=0; m<8; m=m+1) // PilotCount
                begin
                   Index__HE_2xLTF_MU_RU242 [i] [j] [k] [m] = {AllZeros, AllZeros};
                   Index__HE_4xLTF_MU_RU242 [i] [j] [k]     = {AllZeros, AllZeros};
                end
            for (k=0;k<2;k=k+1) // RU Index
              for (m=0; m<8; m=m+1) // PilotCount
                begin
                   Index__HE_1xLTF          [i] [j] [k] [m] = {AllZeros, AllZeros};
                end
            for (m=0; m<16; m=m+1) // PilotCount
              Index__HE_2xLTF_SU            [i] [j]     [m] = {AllZeros, AllZeros};
            
            for (m=0; m<16; m=m+1) // PilotCount
              Index__HE_2xLTF_MU_RU484      [i]         [m] = {AllZeros, AllZeros};
         end

       for (j=0;j<3;j=j+1) // Mode Bandwidth : 20 / 40 / 80 MHz
         begin
            for (k=0;k<38;k=k+1) // RU Index
              for (m=0; m<2; m=m+1) // PilotCount
                begin
// lut according to                        SmoothStep
//                                         |    Mode Bandwidth
//                                         |    |   RUIndex
//                                         |    |   |   PilotCount
//                                         |    |   |   |
                   Index__HE_2xLTF_MU_RU26 [i] [j] [k] [m] = {AllZeros, AllZeros};
                   Index__HE_4xLTF_MU_RU26 [i] [j] [k]     = {AllZeros, AllZeros};
                end
            for (k=0;k<17;k=k+1) // RU Index
              for (m=0; m<4; m=m+1) // PilotCount
                begin
                   Index__HE_2xLTF_MU_RU52 [i] [j] [k] [m] = {AllZeros, AllZeros};
                   Index__HE_4xLTF_MU_RU52 [i] [j] [k]     = {AllZeros, AllZeros};
                end
            for (k=0;k<9;k=k+1) // RU Index
              for (m=0; m<4; m=m+1) // PilotCount
                begin
                   Index__HE_2xLTF_MU_RU106 [i] [j] [k] [m] = {AllZeros, AllZeros};
                   Index__HE_4xLTF_MU_RU106 [i] [j] [k]     = {AllZeros, AllZeros};
                end
         end
     end
   
     for (i=0; i<2;i=i+1) // Mode Bandwidth : 20 / 40 MHz
       begin
          for (k=0;k<3;k=k+1) // RU Index
            IndexVirt__HE_2xLTF_RU242       [i]     [k] = AllZeros;

          for (k=0;k<16;k=k+1) // PilotCount
            begin
               IndexPilot__HE_1xLTF         [i]     [k] = AllZeros;
               IndexPilot__HE_2xLTF_SU      [i]     [k] = AllZeros;
            end
          for (k=0;k<3;k=k+1) // RU Index
            for (m=0; m<8; m=m+1) // PilotCount
              IndexPilot__HE_2xLTF_RU242    [i] [k] [m] = AllZeros;
       end

     for (i=0; i<3;i=i+1) // Mode Bandwidth : 20 / 40 / 80 MHz
       begin
          for (k=0;k<38;k=k+1) // RU Index
            IndexVirt__HE_2xLTF_RU26        [i]     [k] = AllZeros; 
          for (k=0;k<17;k=k+1) // RU Index
            IndexVirt__HE_2xLTF_RU52        [i]     [k] = AllZeros;
          for (k=0;k<9;k=k+1) // RU Index
            IndexVirt__HE_2xLTF_RU106       [i]     [k] = AllZeros;

          for (m=0; m<4; m=m+1) // PilotCount
            begin
               for (k=0;k<38;k=k+1) // RU Index
                 IndexPilot__HE_2xLTF_RU26  [i] [k] [m] = AllZeros;
               for (k=0;k<17;k=k+1) // RU Index
                 IndexPilot__HE_2xLTF_RU52  [i] [k] [m] = AllZeros;
               for (k=0;k<9;k=k+1) // RU Index
                 IndexPilot__HE_2xLTF_RU106 [i] [k] [m] = AllZeros;
            end
       end

     for (k=0;k<16;k=k+1) // RU Index
       IndexPilot__HE_2xLTF_RU484 [k] = AllZeros;

  // And now set the correct values 
   
  //DC Interpolation values
  //                                       Mode_BW,    RUIndex  PilotCount
  //                                           |          |      |
  Index__L_LTF            [DC_INTERP]                               = {10'd1, 10'd6, -10'sd6, -10'sd1};
  Index__HE_LSIG          [DC_INTERP]                               = Index__L_LTF [DC_INTERP];
  Index__VHT_LTF          [DC_INTERP] [MODE_BW_IS_20MHZ]            = Index__L_LTF [DC_INTERP];
  Index__HT_LTF           [DC_INTERP] [MODE_BW_IS_20MHZ]            = Index__L_LTF [DC_INTERP];

for (i=0; i<38; i=i+1) begin : Index_gen

 Index__HE_4xLTF_SU       [DC_INTERP] [MODE_BW_IS_20MHZ]            = { 10'd0,  10'd1,   -10'sd2,   -10'sd1};
 Index__HE_2xLTF_SU       [DC_INTERP] [MODE_BW_IS_20MHZ] [0]        = { 10'd2,  10'd12,  -10'sd12,  -10'sd2};
 Index__HE_1xLTF          [DC_INTERP] [MODE_BW_IS_20MHZ] [i%2]  [0] = { 10'd4,  10'd24,  -10'sd24,  -10'sd4};

 Index__HE_4xLTF_MU_RU26  [DC_INTERP] [MODE_BW_IS_20MHZ] [i]        = { 10'd0,  10'd3,  -10'sd4,  -10'sd3};
 Index__HE_4xLTF_MU_RU52  [DC_INTERP] [MODE_BW_IS_20MHZ] [i%17]     = { 10'd0,  10'd3,  -10'sd4,  -10'sd3};
 Index__HE_4xLTF_MU_RU106 [DC_INTERP] [MODE_BW_IS_20MHZ] [i%9]      = { 10'd0,  10'd3,  -10'sd4,  -10'sd3};
 Index__HE_4xLTF_MU_RU242 [DC_INTERP] [MODE_BW_IS_20MHZ] [i%3]      = { 10'd0,  10'd1,  -10'sd2,  -10'sd1};

 Index__HE_2xLTF_MU_RU26  [DC_INTERP] [MODE_BW_IS_20MHZ] [i]    [0] = { 10'd0,  10'd2,   -10'sd4,  -10'sd2};
 Index__HE_2xLTF_MU_RU52  [DC_INTERP] [MODE_BW_IS_20MHZ] [i%17] [0] = { 10'd0,  10'd2,   -10'sd4,   -10'sd2};
 Index__HE_2xLTF_MU_RU106 [DC_INTERP] [MODE_BW_IS_20MHZ] [i%9]  [0] = { 10'd0,  10'd2,   -10'sd4,   -10'sd2};
 Index__HE_2xLTF_MU_RU242 [DC_INTERP] [MODE_BW_IS_20MHZ] [i%3]  [0] = { 10'd2,  10'd12,  -10'sd12,  -10'sd2};
   
 Index__VHT_LTF           [DC_INTERP] [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd1,  -10'sd2,  -10'sd1};

 Index__HT_LTF            [DC_INTERP] [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd1,  -10'sd2,  -10'sd1};

 Index__HE_4xLTF_SU       [DC_INTERP] [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};
 Index__HE_2xLTF_SU       [DC_INTERP] [MODE_BW_IS_40MHZ]        [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};
 Index__HE_1xLTF          [DC_INTERP] [MODE_BW_IS_40MHZ] [i%2]  [0] = { 10'd4,  10'd24,  -10'sd24,  -10'sd4};

 Index__HE_4xLTF_MU_RU26  [DC_INTERP] [MODE_BW_IS_40MHZ] [i]        = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};
 Index__HE_4xLTF_MU_RU52  [DC_INTERP] [MODE_BW_IS_40MHZ] [i%17]     = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};
 Index__HE_4xLTF_MU_RU106 [DC_INTERP] [MODE_BW_IS_40MHZ] [i%9]      = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};
 Index__HE_4xLTF_MU_RU242 [DC_INTERP] [MODE_BW_IS_40MHZ] [i%3]      = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};
 Index__HE_4xLTF_MU_RU484 [DC_INTERP] [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};

 Index__HE_2xLTF_MU_RU26  [DC_INTERP] [MODE_BW_IS_40MHZ] [i]    [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};
 Index__HE_2xLTF_MU_RU52  [DC_INTERP] [MODE_BW_IS_40MHZ] [i%17] [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};
 Index__HE_2xLTF_MU_RU106 [DC_INTERP] [MODE_BW_IS_40MHZ] [i%9]  [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};
 Index__HE_2xLTF_MU_RU242 [DC_INTERP] [MODE_BW_IS_40MHZ] [i%3]  [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};
 Index__HE_2xLTF_MU_RU484 [DC_INTERP]                    [i%16]     = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};

 Index__HE_4xLTF_MU_RU26  [DC_INTERP] [MODE_BW_IS_80MHZ] [i]        = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};
 Index__HE_4xLTF_MU_RU52  [DC_INTERP] [MODE_BW_IS_80MHZ] [i%17]     = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};
 Index__HE_4xLTF_MU_RU106 [DC_INTERP] [MODE_BW_IS_80MHZ] [i%9]      = { 10'd0,  10'd2,  -10'sd3,  -10'sd2};

 Index__HE_2xLTF_MU_RU26  [DC_INTERP] [MODE_BW_IS_80MHZ] [i]    [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};
 Index__HE_2xLTF_MU_RU52  [DC_INTERP] [MODE_BW_IS_80MHZ] [i%17] [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};
 Index__HE_2xLTF_MU_RU106 [DC_INTERP] [MODE_BW_IS_80MHZ] [i%9]  [0] = { 10'd0,  10'd2,  -10'sd4,  -10'sd2};

// FOR PILOTS INTERPOLATION
//                                            Mode_BW,     RUIndex  PilotCount and 2 indexes
//                                               |            |      |
//                                                                        {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex}
//                                                                        {PilotIndex , CoeffPilot}
 Index__HE_1xLTF [PILOT_INTERP]          [MODE_BW_IS_20MHZ] [i%2]   [3] = { 10'd112,  10'd120,  10'd96,  10'd108};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_20MHZ]         [3] = { 10'd116,  10'd1};
 Index__HE_1xLTF [PILOT_INTERP]          [MODE_BW_IS_20MHZ] [i%2]   [2] = { 10'd52,  10'd60,  10'd36,  10'd44};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_20MHZ]         [2] = { 10'd48,  10'd0};
 Index__HE_1xLTF [PILOT_INTERP]          [MODE_BW_IS_20MHZ] [i%2]   [1] = { -10'sd44,  -10'sd36,  -10'sd60,  -10'sd52};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_20MHZ]         [1] = { -10'sd48,  10'd0};
 Index__HE_1xLTF [PILOT_INTERP]          [MODE_BW_IS_20MHZ] [i%2]   [0] = { -10'sd108,  -10'sd96,  -10'sd120,  -10'sd112};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_20MHZ]         [0] = { -10'sd116,  -10'sd1};

 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [7] = { 10'd114,  10'd118,  10'd106,  10'd112};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [7] = { 10'd116,  10'd1};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [6] = { 10'd92,  10'd96,  10'd84,  10'd88};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [6] = { 10'd90,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [5] = { 10'd50,  10'd54,  10'd42,  10'd46};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [5] = { 10'd48,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [4] = { 10'd24,  10'd28,  10'd16,  10'd20};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [4] = { 10'd22,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [3] = { -10'sd20,  -10'sd16,  -10'sd28,  -10'sd24};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [3] = { -10'sd22,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [2] = { -10'sd46,  -10'sd42,  -10'sd54,  -10'sd50};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [2] = { -10'sd48,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [1] = { -10'sd88,  -10'sd84,  -10'sd96,  -10'sd92};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [1] = { -'sd90,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_20MHZ]         [0] = { -10'sd112,  -10'sd106,  -10'sd118,  -10'sd114};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_20MHZ]         [0] = { -10'sd116,  -10'sd1};

 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [9]     [1] = { 10'd114,  10'd118,  10'd106,  10'd112};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [9]     [1] = { 10'd116,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [9]     [0] = { 10'd106,  10'd112,  10'd100,  10'd104};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [9]     [0] = { 10'd102,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [8]     [1] = { 10'd88,  10'd92,  10'd80,  10'd86};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [8]     [1] = { 10'd90,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [8]     [0] = { 10'd80,  10'd86,  10'd74,  10'd78};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [8]     [0] = { 10'd76,  -10'sd1};                   
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [7]     [1] = { 10'd60,  10'd64,  10'd52,  10'd58};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [7]     [1] = { 10'd62,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [7]     [0] = { 10'd52,  10'd58,  10'd46,  10'd50};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [7]     [0] = { 10'd48,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [6]     [1] = { 10'd34,  10'd38,  10'd26,  10'd32};    
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [6]     [1] = { 10'd36,  10'd1};                     
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [6]     [0] = { 10'd26,  10'd32,  10'd20,  10'd24};    
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [6]     [0] = { 10'd22,  -10'sd1};                   
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [5]     [1] = { 10'd12,  10'd16,  10'd4,  10'd8};      
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [5]     [1] = { 10'd10,  10'd0};                     
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [5]     [0] = { -10'sd8,  -10'sd4,  -10'sd16,  -10'sd12};                                
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [5]     [0] = { -10'sd10,  10'd0};                                               
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [4]     [1] = { -10'sd24,  -10'sd20,  -10'sd32,  -10'sd26}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [4]     [1] = { -10'sd22,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [4]     [0] = { -10'sd28,  -10'sd26,  -10'sd38,  -10'sd34};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [4]     [0] = { -10'sd36,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [3]     [1] = { -10'sd50,  -10'sd46,  -10'sd58,  -10'sd52};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [3]     [1] = { -10'sd48,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [3]     [0] = { -10'sd56,  -10'sd52,  -10'sd64,  -10'sd60};                                            
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [3]     [0] = { -10'sd62,  -10'sd1};                                                                   
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [2]     [1] = { -10'sd78,  -10'sd74,  -10'sd86,  -10'sd82};      
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [2]     [1] = { -10'sd76,  10'd1};                             
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [2]     [0] = { -10'sd86,  -10'sd80,  -10'sd92,  -10'sd88};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [2]     [0] = { -'sd90,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [1]     [1] = { -10'sd104,  -10'sd100,  -10'sd112,  -10'sd106};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [1]     [1] = { -'sd102,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd112,  -10'sd106,  -10'sd118,  -10'sd114};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd116,  -10'sd1};

 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [4]     [3] = { 10'd114,  10'd118,  10'd106,  10'd112};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [4]     [3] = { 10'd116,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [4]     [2] = { 10'd104,  10'd108,  10'd96,  10'd100};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [4]     [2] = { 10'd102,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [4]     [1] = { 10'd92,  10'd96,  10'd84,  10'd88};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [4]     [1] = { 10'd90,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [4]     [0] = { 10'd80,  10'd86,  10'd74,  10'd78};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [4]     [0] = { 10'd76,  -10'sd1};                       
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [3]     [3] = { 10'd60,  10'd64,  10'd52,  10'd58};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [3]     [3] = { 10'd62,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [3]     [2] = { 10'd50,  10'd54,  10'd42,  10'd46};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [3]     [2] = { 10'd48,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [3]     [1] = { 10'd38,  10'd42,  10'd30,  10'd34};                               
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [3]     [1] = { 10'd36,  10'd0};                                              
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [3]     [0] = { 10'd26,  10'd32,  10'd20,  10'd24};                                
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [3]     [0] = { 10'd22,  -10'sd1};                                               
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [2]     [3] = { -10'sd24,  -10'sd20,  -10'sd32,  -10'sd26}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [2]     [3] = { -10'sd22,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [2]     [2] = { -10'sd34,  -10'sd30,  -10'sd42,  -10'sd38};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [2]     [2] = { -10'sd36,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [2]     [1] = { -10'sd46,  -10'sd42,  -10'sd54,  -10'sd50};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [2]     [1] = { -10'sd48,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [2]     [0] = { -10'sd56,  -10'sd52,  -10'sd64,  -10'sd60};                                            
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [2]     [0] = { -10'sd62,  -10'sd1};                                                                   
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [1]     [3] = { -10'sd78,  -10'sd74,  -10'sd86,  -10'sd80};      
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [1]     [3] = { -10'sd76,  10'd1};                             
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [1]     [2] = { -10'sd88,  -10'sd84,  -10'sd96,  -10'sd92};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [1]     [2] = { -'sd90,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [1]     [1] = { -10'sd100,  -10'sd96,  -10'sd108,  -10'sd104};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [1]     [1] = { -'sd102,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd112,  -10'sd106,  -10'sd118,  -10'sd114};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd116,  -10'sd1};

 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [3] = { 10'd114,  10'd118,  10'd106,  10'd112};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [2]     [3] = { 10'd116,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [2] = { 10'd92,  10'd96,  10'd84,  10'd88};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [2]     [2] = { 10'd90,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [1] = { 10'd50,  10'd54,  10'd42,  10'd46};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [2]     [1] = { 10'd48,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [0] = { 10'd26,  10'd32,  10'd20,  10'd24};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [2]     [0] = { 10'd22,  -10'sd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [3] = { -10'sd24,  -10'sd20,  -10'sd32,  -10'sd26};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [1]     [3] = { -10'sd22,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [2] = { -10'sd46,  -10'sd42,  -10'sd54,  -10'sd50};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [1]     [2] = { -10'sd48,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [1] = { -10'sd88,  -10'sd84,  -10'sd96,  -10'sd92};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [1]     [1] = { -'sd90,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd112,  -10'sd106,  -10'sd118,  -10'sd114};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd116,  -10'sd1};

 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [7] = { 10'd114,  10'd118,  10'd106,  10'd112};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [7] = { 10'd116,  10'd1};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [6] = { 10'd92,  10'd96,  10'd84,  10'd88};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [6] = { 10'd90,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [5] = { 10'd50,  10'd54,  10'd42,  10'd46};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [5] = { 10'd48,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [4] = { 10'd24,  10'd28,  10'd16,  10'd20};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [4] = { 10'd22,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [3] = { -10'sd20,  -10'sd16,  -10'sd28,  -10'sd24};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [3] = { -10'sd22,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [2] = { -10'sd46,  -10'sd42,  -10'sd54,  -10'sd50};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [2] = { -10'sd48,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [1] = { -10'sd88,  -10'sd84,  -10'sd96,  -10'sd92};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [1] = { -'sd90,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [2]     [0] = { -10'sd112,  -10'sd106,  -10'sd118,  -10'sd114};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [2]     [0] = { -10'sd116,  -10'sd1};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [7] = { 10'd114,  10'd118,  10'd106,  10'd112};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [7] = { 10'd116,  10'd1};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [6] = { 10'd92,  10'd96,  10'd84,  10'd88};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [6] = { 10'd90,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [5] = { 10'd50,  10'd54,  10'd42,  10'd46};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [5] = { 10'd48,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [4] = { 10'd24,  10'd28,  10'd16,  10'd20};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [4] = { 10'd22,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [3] = { -10'sd20,  -10'sd16,  -10'sd28,  -10'sd24};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [3] = { -10'sd22,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [2] = { -10'sd46,  -10'sd42,  -10'sd54,  -10'sd50};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [2] = { -10'sd48,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [1] = { -10'sd88,  -10'sd84,  -10'sd96,  -10'sd92};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [1] = { -'sd90,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd112,  -10'sd106,  -10'sd118,  -10'sd114};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_20MHZ] [1]     [0] = { -10'sd116,  -10'sd1};

 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [7] = { 10'd208,  10'd216,  10'd192,  10'd204};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [7] = { 10'd212,  10'd1};
 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [6] = { 10'd148,  10'd156,  10'd132,  10'd140};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [6] = { 10'd144,  10'd0};
 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [5] = { 10'd108,  10'd116,  10'd92,  10'd100};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [5] = { 10'd104,  10'd0};
 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [4] = { 10'd40,  10'd48,  10'd24,  10'd32};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [4] = { 10'd36,  10'd0};
 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [3] = { -10'sd32,  -10'sd24,  -10'sd48,  -10'sd40};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [3] = { -10'sd36,  10'd0};
 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [2] = { -10'sd100,  -10'sd92,  -10'sd116,  -10'sd108};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [2] = { -10'sd104,  10'd0};
 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [1] = { -10'sd140,  -10'sd132,  -10'sd156,  -10'sd148};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [1] = { -'sd144,  10'd0};
 Index__HE_1xLTF    [PILOT_INTERP]       [MODE_BW_IS_40MHZ] [i%2]   [0] = { -10'sd204,  -10'sd192,  -10'sd216,  -10'sd208};
 IndexPilot__HE_1xLTF                    [MODE_BW_IS_40MHZ]         [0] = { -10'sd212,  -10'sd1};

 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]        [15] = { 10'd236,  10'd240,  10'd228,  10'd234};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]        [15] = { 10'd238,  10'd1};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]        [14] = { 10'd214,  10'd218,  10'd206,  10'd210};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]        [14] = { 10'd212,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]        [13] = { 10'd172,  10'd176,  10'd164,  10'd168};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]        [13] = { 10'd170,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]        [12] = { 10'd146,  10'd150,  10'd138,  10'd142};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]        [12] = { 10'd144,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]        [11] = { 10'd106,  10'd110,  10'd98,  10'd102};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]        [11] = { 10'd104,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]        [10] = { 10'd80,  10'd84,  10'd72,  10'd76};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]        [10] = { 10'd78,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [9] = { 10'd38,  10'd42,  10'd30,  10'd34};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [9] = { 10'd36,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [8] = { 10'd12,  10'd16,  10'd4,  10'd8};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [8] = { 10'd10,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [7] = { -10'sd8,  -10'sd4,  -10'sd16,  -10'sd12};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [7] = { -10'sd10,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [6] = { -10'sd34,  -10'sd30,  -10'sd42,  -10'sd38};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [6] = { -10'sd36,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [5] = { -10'sd76,  -10'sd72,  -10'sd84,  -10'sd80};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [5] = { -10'sd78,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [4] = { -10'sd102,  -10'sd98,  -10'sd110,  -10'sd106};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [4] = { -10'sd104,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [3] = { -10'sd142,  -10'sd138,  -10'sd150,  -10'sd146};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [3] = { -10'sd144,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [2] = { -10'sd168,  -10'sd164,  -10'sd176,  -10'sd172};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [2] = { -10'sd170,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [1] = { -10'sd210,  -10'sd206,  -10'sd218,  -10'sd214};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [1] = { -'sd212,  10'd0};
 Index__HE_2xLTF_SU [PILOT_INTERP]       [MODE_BW_IS_40MHZ]         [0] = { -10'sd234,  -10'sd228,  -10'sd240,  -10'sd236};
 IndexPilot__HE_2xLTF_SU                 [MODE_BW_IS_40MHZ]         [0] = { -10'sd238,  -10'sd1};

 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [18]    [1] = { 10'd236,  10'd240,  10'd228,  10'd234};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [18]    [1] = { 10'd238,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [18]    [0] = { 10'd232,  10'd234,  10'd222,  10'd226};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [18]    [0] = { 10'd224,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [17]    [1] = { 10'd210,  10'd214,  10'd202,  10'd208};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [17]    [1] = { 10'd212,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [17]    [0] = { 10'd202,  10'd208,  10'd196,  10'd200};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [17]    [0] = { 10'd198,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [16]    [1] = { 10'd182,  10'd186,  10'd174,  10'd180};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [16]    [1] = { 10'd184,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [16]    [0] = { 10'd178,  10'd180,  10'd168,  10'd172};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [16]    [0] = { 10'd170,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [15]    [1] = { 10'd156,  10'd160,  10'd148,  10'd154};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [15]    [1] = { 10'd158,  10'd1};                    
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [15]    [0] = { 10'd148,  10'd154,  10'd142,  10'd146};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [15]    [0] = { 10'd144,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [14]    [1] = { 10'd128,  10'd132,  10'd120,  10'd126};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [14]    [1] = { 10'd130,  10'd1};                    
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [14]    [0] = { 10'd120,  10'd126,  10'd114,  10'd118};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [14]    [0] = { 10'd116,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [13]    [1] = { 10'd102,  10'd106,  10'd94,  10'd100}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [13]    [1] = { 10'd104,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [13]    [0] = { 10'd98,  10'd100,  10'd88,  10'd92};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [13]    [0] = { 10'd90,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [12]    [1] = { 10'd76,  10'd80,  10'd68,  10'd74};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [12]    [1] = { 10'd78,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [12]    [0] = { 10'd68,  10'd74,  10'd62,  10'd66};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [12]    [0] = { 10'd64,  -10'sd1};                
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [11]    [1] = { 10'd48,  10'd52,  10'd40,  10'd46};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [11]    [1] = { 10'd50,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [11]    [0] = { 10'd40,  10'd46,  10'd34,  10'd38};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [11]    [0] = { 10'd36,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [10]    [1] = { 10'd22,  10'd26,  10'd14,  10'd20};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [10]    [1] = { 10'd24,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [10]    [0] = { 10'd14,  10'd20,  10'd8,  10'd12};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [10]    [0] = { 10'd10,  -10'sd1};                     
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [9]     [1] = { -10'sd12,  -10'sd8,  -10'sd20,  -10'sd14};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [9]     [1] = { -10'sd10,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [9]     [0] = { -10'sd16,  -10'sd14,  -10'sd26,  -10'sd22};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [9]     [0] = { -10'sd24,  -10'sd1};                       
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [8]     [1] = { -10'sd38,  -10'sd34,  -10'sd46,  -10'sd40};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [8]     [1] = { -10'sd36,  10'd1};                       
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [8]     [0] = { -10'sd46,  -10'sd40,  -10'sd52,  -10'sd48}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [8]     [0] = { -10'sd50,  -10'sd1};                        
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [7]     [1] = { -10'sd66,  -10'sd62,  -10'sd74,  -10'sd68};   
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [7]     [1] = { -10'sd64,  10'd1};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [7]     [0] = { -10'sd70,  -10'sd68,  -10'sd80,  -10'sd76};    
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [7]     [0] = { -10'sd78,  -10'sd1};                           
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [6]     [1] = { -10'sd92,  -10'sd88,  -10'sd100,  -10'sd94};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [6]     [1] = { -10'sd90,  10'd1};                       
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [6]     [0] = { -10'sd100,  -10'sd94,  -10'sd106,  -10'sd102}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [6]     [0] = { -10'sd104,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [5]     [1] = { -10'sd118,  -10'sd114,  -10'sd126,  -10'sd120};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [5]     [1] = { -10'sd116,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [5]     [0] = { -10'sd122,  -10'sd120,  -10'sd132,  -10'sd128};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [5]     [0] = { -10'sd130,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [4]     [1] = { -10'sd146,  -10'sd142,  -10'sd154,  -10'sd148};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [4]     [1] = { -10'sd144,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [4]     [0] = { -10'sd150,  -10'sd148,  -10'sd160,  -10'sd156};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [4]     [0] = { -10'sd158,  -10'sd1};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [3]     [1] = { -10'sd172,  -10'sd168,  -10'sd180,  -10'sd174};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [3]     [1] = { -10'sd170,  10'd1};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [3]     [0] = { -10'sd180,  -10'sd174,  -10'sd186,  -10'sd182};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [3]     [0] = { -10'sd184,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [2]     [1] = { -10'sd200,  -10'sd196,  -10'sd208,  -10'sd202};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [2]     [1] = { -10'sd198,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [2]     [0] = { -10'sd204,  -10'sd202,  -10'sd214,  -10'sd210};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [2]     [0] = { -10'sd212,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [1]     [1] = { -10'sd226,  -10'sd222,  -10'sd234,  -10'sd228};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [1]     [1] = { -10'sd224,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd234,  -10'sd228,  -10'sd240,  -10'sd236};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd238,  -10'sd1};

 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [8]     [3] = { 10'd236,  10'd240,  10'd228,  10'd234};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [8]     [3] = { 10'd238,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [8]     [2] = { 10'd226,  10'd230,  10'd218,  10'd222};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [8]     [2] = { 10'd224,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [8]     [1] = { 10'd214,  10'd218,  10'd206,  10'd210};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [8]     [1] = { 10'd212,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [8]     [0] = { 10'd202,  10'd208,  10'd196,  10'd200};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [8]     [0] = { 10'd198,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [7]     [3] = { 10'd182,  10'd186,  10'd174,  10'd180};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [7]     [3] = { 10'd184,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [7]     [2] = { 10'd172,  10'd176,  10'd164,  10'd168};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [7]     [2] = { 10'd170,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [7]     [1] = { 10'd160,  10'd164,  10'd152,  10'd156};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [7]     [1] = { 10'd158,  10'd0};                    
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [7]     [0] = { 10'd148,  10'd154,  10'd142,  10'd146};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [7]     [0] = { 10'd144,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [6]     [3] = { 10'd102,  10'd106,  10'd94,  10'd100}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [6]     [3] = { 10'd104,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [6]     [2] = { 10'd92,  10'd96,  10'd84,  10'd88};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [6]     [2] = { 10'd90,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [6]     [1] = { 10'd80,  10'd84,  10'd72,  10'd76};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [6]     [1] = { 10'd78,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [6]     [0] = { 10'd68,  10'd74,  10'd62,  10'd66};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [6]     [0] = { 10'd64,  -10'sd1};                
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [5]     [3] = { 10'd48,  10'd52,  10'd40,  10'd46};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [5]     [3] = { 10'd50,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [5]     [2] = { 10'd38,  10'd42,  10'd30,  10'd34};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [5]     [2] = { 10'd36,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [5]     [1] = { 10'd26,  10'd30,  10'd18,  10'd22};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [5]     [1] = { 10'd24,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [5]     [0] = { 10'd14,  10'd20,  10'd8,  10'd12};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [5]     [0] = { 10'd10,  -10'sd1};                     
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [4]     [3] = { -10'sd12,  -10'sd8,  -10'sd20,  -10'sd14};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [4]     [3] = { -10'sd10,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [4]     [2] = { -10'sd22,  -10'sd18,  -10'sd30,  -10'sd26};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [4]     [2] = { -10'sd24,  10'd0};                       
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [4]     [1] = { -10'sd34,  -10'sd30,  -10'sd42,  -10'sd38};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [4]     [1] = { -10'sd36,  10'd0};                       
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [4]     [0] = { -10'sd46,  -10'sd40,  -10'sd52,  -10'sd48}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [4]     [0] = { -10'sd50,  -10'sd1};                        
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [3]     [3] = { -10'sd66,  -10'sd62,  -10'sd74,  -10'sd68};   
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [3]     [3] = { -10'sd64,  10'd1};                          
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [3]     [2] = { -10'sd76,  -10'sd72,  -10'sd84,  -10'sd80};    
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [3]     [2] = { -10'sd78,  10'd0};                           
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [3]     [1] = { -10'sd88,  -10'sd84,  -10'sd96,  -10'sd92};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [3]     [1] = { -10'sd90,  10'd0};                       
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [3]     [0] = { -10'sd100,  -10'sd94,  -10'sd106,  -10'sd102}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [3]     [0] = { -10'sd104,  -10'sd1};                        
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [2]     [3] = { -10'sd146,  -10'sd142,  -10'sd154,  -10'sd148};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [2]     [3] = { -10'sd144,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [2]     [2] = { -10'sd156,  -10'sd152,  -10'sd164,  -10'sd160};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [2]     [2] = { -10'sd158,  10'd0};                          
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [2]     [1] = { -10'sd168,  -10'sd164,  -10'sd176,  -10'sd172};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [2]     [1] = { -10'sd170,  10'd0};                          
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [2]     [0] = { -10'sd180,  -10'sd174,  -10'sd186,  -10'sd182};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [2]     [0] = { -10'sd184,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [1]     [3] = { -10'sd200,  -10'sd196,  -10'sd208,  -10'sd202};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [1]     [3] = { -10'sd198,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [1]     [2] = { -10'sd210,  -10'sd206,  -10'sd218,  -10'sd214};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [1]     [2] = { -10'sd212,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [1]     [1] = { -10'sd222,  -10'sd218,  -10'sd230,  -10'sd226};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [1]     [1] = { -10'sd224,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd234,  -10'sd228,  -10'sd240,  -10'sd236};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd238,  -10'sd1};

 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [4]     [3] = { 10'd236,  10'd240,  10'd228,  10'd234};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [4]     [3] = { 10'd238,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [4]     [2] = { 10'd214,  10'd218,  10'd206,  10'd210};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [4]     [2] = { 10'd212,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [4]     [1] = { 10'd172,  10'd176,  10'd164,  10'd168};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [4]     [1] = { 10'd170,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [4]     [0] = { 10'd148,  10'd154,  10'd142,  10'd146};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [4]     [0] = { 10'd144,  -10'sd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [3]     [3] = { 10'd102,  10'd106,  10'd94,  10'd100};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [3]     [3] = { 10'd104,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [3]     [2] = { 10'd80,  10'd84,  10'd72,  10'd76};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [3]     [2] = { 10'd78,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [3]     [1] = { 10'd38,  10'd42,  10'd30,  10'd34};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [3]     [1] = { 10'd36,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [3]     [0] = { 10'd14,  10'd20,  10'd8,  10'd12};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [3]     [0] = { 10'd10,  -10'sd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [3] = { -10'sd12,  -10'sd8,  -10'sd20,  -10'sd14};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [2]     [3] = { -10'sd10,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [2] = { -10'sd34,  -10'sd30,  -10'sd42,  -10'sd38};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [2]     [2] = { -10'sd36,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [1] = { -10'sd76,  -10'sd72,  -10'sd84,  -10'sd80};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [2]     [1] = { -10'sd78,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [0] = { -10'sd100,  -10'sd94,  -10'sd106,  -10'sd102};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [2]     [0] = { -10'sd104,  -10'sd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [3] = { -10'sd146,  -10'sd142,  -10'sd154,  -10'sd148};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [1]     [3] = { -10'sd144,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [2] = { -10'sd168,  -10'sd164,  -10'sd176,  -10'sd172};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [1]     [2] = { -10'sd170,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [1] = { -10'sd210,  -10'sd206,  -10'sd218,  -10'sd214};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [1]     [1] = { -10'sd212,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd234,  -10'sd228,  -10'sd240,  -10'sd236};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd238,  -10'sd1};

 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [7] = { 10'd236,  10'd240,  10'd228,  10'd234};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [7] = { 10'd238,  10'd1};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [6] = { 10'd214,  10'd218,  10'd206,  10'd210};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [6] = { 10'd212,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [5] = { 10'd172,  10'd176,  10'd164,  10'd168};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [5] = { 10'd170,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [4] = { 10'd146,  10'd150,  10'd138,  10'd142};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [4] = { 10'd144,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [3] = { 10'd106,  10'd110,  10'd98,  10'd102};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [3] = { 10'd104,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [2] = { 10'd80,  10'd84,  10'd72,  10'd76};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [2] = { 10'd78,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [1] = { 10'd38,  10'd42,  10'd30,  10'd34};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [1] = { 10'd36,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [2]     [0] = { 10'd18,  10'd20,  10'd8,  10'd12};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [2]     [0] = { 10'd10,  -10'd1};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [7] = { -10'sd12,  -10'sd8,  -10'sd20,  -10'sd14};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [7] = { -10'sd10,  10'd1};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [6] = { -10'sd34,  -10'sd30,  -10'sd42,  -10'sd38};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [6] = { -10'sd36,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [5] = { -10'sd76,  -10'sd72,  -10'sd84,  -10'sd80};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [5] = { -10'sd78,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [4] = { -10'sd102,  -10'sd98,  -10'sd110,  -10'sd106};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [4] = { -10'sd104,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [3] = { -10'sd142,  -10'sd138,  -10'sd150,  -10'sd146};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [3] = { -10'sd144,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [2] = { -10'sd168,  -10'sd164,  -10'sd176,  -10'sd172};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [2] = { -10'sd170,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [1] = { -10'sd210,  -10'sd206,  -10'sd218,  -10'sd214};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [1] = { -'sd212,  10'd0};
 Index__HE_2xLTF_MU_RU242 [PILOT_INTERP] [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd234,  -10'sd228,  -10'sd240,  -10'sd236};
 IndexPilot__HE_2xLTF_RU242              [MODE_BW_IS_40MHZ] [1]     [0] = { -10'sd238,  -10'sd1};

 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [15] = { 10'd236,  10'd240,  10'd228,  10'd234};
 IndexPilot__HE_2xLTF_RU484                                        [15] = { 10'd238,  10'd1};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [14] = { 10'd214,  10'd218,  10'd206,  10'd210};
 IndexPilot__HE_2xLTF_RU484                                        [14] = { 10'd212,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [13] = { 10'd172,  10'd176,  10'd164,  10'd168};
 IndexPilot__HE_2xLTF_RU484                                        [13] = { 10'd170,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [12] = { 10'd146,  10'd150,  10'd138,  10'd142};
 IndexPilot__HE_2xLTF_RU484                                        [12] = { 10'd144,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [11] = { 10'd106,  10'd110,  10'd98,  10'd102};
 IndexPilot__HE_2xLTF_RU484                                        [11] = { 10'd104,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [10] = { 10'd80,  10'd84,  10'd72,  10'd76};
 IndexPilot__HE_2xLTF_RU484                                        [10] = { 10'd78,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [9]  = { 10'd38,  10'd42,  10'd30,  10'd34};
 IndexPilot__HE_2xLTF_RU484                                        [9]  = { 10'd36,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [8]  = { 10'd12,  10'd16,  10'd4,  10'd8};
 IndexPilot__HE_2xLTF_RU484                                        [8]  = { 10'd10,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [7]  = { -10'sd8,  -10'sd4,  -10'sd16,  -10'sd12};
 IndexPilot__HE_2xLTF_RU484                                        [7]  = { -10'sd10,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [6]  = { -10'sd34,  -10'sd30,  -10'sd42,  -10'sd38};
 IndexPilot__HE_2xLTF_RU484                                        [6]  = { -10'sd36,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [5]  = { -10'sd76,  -10'sd72,  -10'sd84,  -10'sd80};
 IndexPilot__HE_2xLTF_RU484                                        [5]  = { -10'sd78,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [4]  = { -10'sd102,  -10'sd98,  -10'sd110,  -10'sd106};
 IndexPilot__HE_2xLTF_RU484                                        [4]  = { -10'sd104,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [3]  = { -10'sd142,  -10'sd138,  -10'sd150,  -10'sd146};
 IndexPilot__HE_2xLTF_RU484                                        [3]  = { -10'sd144,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [2]  = { -10'sd168,  -10'sd164,  -10'sd176,  -10'sd172};
 IndexPilot__HE_2xLTF_RU484                                        [2]  = { -10'sd170,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [1]  = { -10'sd210,  -10'sd206,  -10'sd218,  -10'sd214};
 IndexPilot__HE_2xLTF_RU484                                        [1]  = { -'sd212,  10'd0};
 Index__HE_2xLTF_MU_RU484 [PILOT_INTERP]                           [0]  = { -10'sd234,  -10'sd228,  -10'sd240,  -10'sd236};
 IndexPilot__HE_2xLTF_RU484                                        [0]  = { -10'sd238,  -10'sd1};

 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [37]    [1] = { 10'd492,  10'd496,  10'd484,  10'd490};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [37]    [1] = { 10'd494,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [37]    [0] = { 10'd488,  10'd490,  10'd478,  10'd482};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [37]    [0] = { 10'd480,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [36]    [1] = { 10'd466,  10'd470,  10'd458,  10'd464};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [36]    [1] = { 10'd468,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [36]    [0] = { 10'd462,  10'd464,  10'd452,  10'd456};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [36]    [0] = { 10'd454,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [35]    [1] = { 10'd438,  10'd442,  10'd430,  10'd436};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [35]    [1] = { 10'd440,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [35]    [0] = { 10'd434,  10'd436,  10'd424,  10'd428};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [35]    [0] = { 10'd426,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [34]    [1] = { 10'd412,  10'd416,  10'd404,  10'd410};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [34]    [1] = { 10'd414,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [34]    [0] = { 10'd408,  10'd410,  10'd398,  10'd402};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [34]    [0] = { 10'd400,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [33]    [1] = { 10'd384,  10'd388,  10'd376,  10'd382};   
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [33]    [1] = { 10'd386,  10'd1};                    
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [33]    [0] = { 10'd380,  10'd382,  10'd370,  10'd374};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [33]    [0] = { 10'd372,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [32]    [1] = { 10'd358,  10'd362,  10'd350,  10'd356};     
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [32]    [1] = { 10'd360,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [32]    [0] = { 10'd354,  10'd356,  10'd344,  10'd348};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [32]    [0] = { 10'd346,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [31]    [1] = { 10'd332,  10'd336,  10'd324,  10'd330};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [31]    [1] = { 10'd334,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [31]    [0] = { 10'd328,  10'd330,  10'd318,  10'd322};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [31]    [0] = { 10'd320,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [30]    [1] = { 10'd304,  10'd308,  10'd296,  10'd302};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [30]    [1] = { 10'd306,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [30]    [0] = { 10'd300,  10'd302,  10'd290,  10'd294};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [30]    [0] = { 10'd292,  -10'sd1};                
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [29]    [1] = { 10'd278,  10'd282,  10'd270,  10'd276};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [29]    [1] = { 10'd280,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [29]    [0] = { 10'd274,  10'd276,  10'd264,  10'd268};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [29]    [0] = { 10'd266,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [28]    [1] = { 10'd250,  10'd254,  10'd242,  10'd248};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [28]    [1] = { 10'd252,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [28]    [0] = { 10'd246,  10'd248,  10'd236,  10'd240};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [28]    [0] = { 10'd238,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [27]    [1] = { 10'd226,  10'd228,  10'd216,  10'd224};   
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [27]    [1] = { 10'd226,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [27]    [0] = { 10'd220,  10'd222,  10'd210,  10'd214};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [27]    [0] = { 10'd212,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [26]    [1] = { 10'd196,  10'd200,  10'd188,  10'd194};   
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [26]    [1] = { 10'd198,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [26]    [0] = { 10'd192,  10'd194,  10'd182,  10'd186};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [26]    [0] = { 10'd184,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [25]    [1] = { 10'd170,  10'd174,  10'd162,  10'd168};  
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [25]    [1] = { 10'd172,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [25]    [0] = { 10'd166,  10'd168,  10'd156,  10'd160};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [25]    [0] = { 10'd158,  -10'sd1};                   
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [24]    [1] = { 10'd142,  10'd146,  10'd134,  10'd140};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [24]    [1] = { 10'd144,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [24]    [0] = { 10'd138,  10'd140,  10'd128,  10'd132};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [24]    [0] = { 10'd130,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [23]    [1] = { 10'd116,  10'd120,  10'd108,  10'd114};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [23]    [1] = { 10'd118,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [23]    [0] = { 10'd112,  10'd114,  10'd102,  10'd106};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [23]    [0] = { 10'd104,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [22]    [1] = { 10'd90,  10'd94,  10'd82,  10'd88};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [22]    [1] = { 10'd92,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [22]    [0] = { 10'd86,  10'd88,  10'd76,  10'd80};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [22]    [0] = { 10'd78,  -10'sd1};                
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [21]    [1] = { 10'd62,  10'd66,  10'd54,  10'd60};    
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [21]    [1] = { 10'd64,  10'd1};               
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [21]    [0] = { 10'd58,  10'd60,  10'd48,  10'd52};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [21]    [0] = { 10'd50,  -10'sd1}; 
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [20]    [1] = { 10'd36,  10'd40,  10'd28,  10'd34};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [20]    [1] = { 10'd38,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [20]    [0] = { 10'd32,  10'd34,  10'd22,  10'd26};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [20]    [0] = { 10'd24,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [19]    [1] = { 10'd12,  10'd16,  10'd4,  10'd8};  
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [19]    [1] = { 10'd10,    10'd0};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [19]    [0] = { -10'sd8,  -10'sd4,  -10'sd16,  -10'sd12};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [19]    [0] = { -10'sd10,  10'd0};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [18]    [1] = { -10'sd26,  -10'sd22,  -10'sd34,  -10'sd28};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [18]    [1] = { -10'sd24,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [18]    [0] = { -10'sd30,  -10'sd28,  -10'sd40,  -10'sd36};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [18]    [0] = { -10'sd38,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [17]    [1] = { -10'sd52,  -10'sd48,  -10'sd60,  -10'sd54}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [17]    [1] = { -10'sd50,  10'd1};                        
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [17]    [0] = { -10'sd56,  -10'sd54,  -10'sd66,  -10'sd62};   
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [17]    [0] = { -10'sd64,  -10'sd1};                        
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [16]    [1] = { -10'sd80,  -10'sd76,  -10'sd88,  -10'sd82};    
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [16]    [1] = { -10'sd78,   10'd1};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [16]    [0] = { -10'sd88,  -10'sd82,  -10'sd94,  -10'sd90};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [16]    [0] = { -10'sd92,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [15]    [1] = { -10'sd106,  -10'sd102,  -10'sd114,  -10'sd112}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [15]    [1] = { -10'sd104,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [15]    [0] = { -10'sd110,  -10'sd108,  -10'sd120,  -10'sd116};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [15]    [0] = { -10'sd118,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [14]    [1] = { -10'sd132,  -10'sd128,  -10'sd140,  -10'sd138}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [14]    [1] = { -10'sd130,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [14]    [0] = { -10'sd136,  -10'sd134,  -10'sd146,  -10'sd142};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [14]    [0] = { -10'sd144,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [13]    [1] = { -10'sd160,  -10'sd156,  -10'sd168,  -10'sd166}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [13]    [1] = { -10'sd158,  10'd1};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [13]    [0] = { -10'sd164,  -10'sd162,  -10'sd174,  -10'sd170};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [13]    [0] = { -10'sd172,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [12]    [1] = { -10'sd186,  -10'sd182,  -10'sd194,  -10'sd188};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [12]    [1] = { -10'sd184,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [12]    [0] = { -10'sd194,  -10'sd188,  -10'sd200,  -10'sd196};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [12]    [0] = { -10'sd198,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [11]    [1] = { -10'sd214,  -10'sd210,  -10'sd222,  -10'sd216};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [11]    [1] = { -10'sd212,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [11]    [0] = { -10'sd218,  -10'sd216,  -10'sd228,  -10'sd224};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [11]    [0] = { -10'sd226,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [10]    [1] = { -10'sd240,  -10'sd236,  -10'sd248,  -10'sd242};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [10]    [1] = { -10'sd238,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [10]    [0] = { -10'sd248,  -10'sd242,  -10'sd254,  -10'sd250};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [10]    [0] = { -10'sd252,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [9]     [1] = { -10'sd268,  -10'sd264,  -10'sd276,  -10'sd270}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [9]     [1] = { -10'sd266,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [9]     [0] = { -10'sd272,  -10'sd270,  -10'sd282,  -10'sd278};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [9]     [0] = { -10'sd280,  -10'sd1};                       
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [8]     [1] = { -10'sd294,  -10'sd290,  -10'sd302,  -10'sd296};  
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [8]     [1] = { -10'sd292,  10'd1};                       
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [8]     [0] = { -10'sd298,  -10'sd296,  -10'sd308,  -10'sd304}; 
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [8]     [0] = { -10'sd306,  -10'sd1};                           
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [7]     [1] = { -10'sd322,  -10'sd318,  -10'sd330,  -10'sd324};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [7]     [1] = { -10'sd320,  10'd1};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [7]     [0] = { -10'sd332,  -10'sd328,  -10'sd340,  -10'sd336};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [7]     [0] = { -10'sd334,  10'd0};                             
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [6]     [1] = { -10'sd344,  -10'sd340,  -10'sd352,  -10'sd348};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [6]     [1] = { -10'sd346,  10'd0};                       
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [6]     [0] = { -10'sd356,  -10'sd350,  -10'sd362,  -10'sd358};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [6]     [0] = { -10'sd360,  -10'sd1};                        
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [5]     [1] = { -10'sd374,  -10'sd370,  -10'sd382,  -10'sd376};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [5]     [1] = { -10'sd372,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [5]     [0] = { -10'sd382,  -10'sd376,  -10'sd388,  -10'sd384};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [5]     [0] = { -10'sd386,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [4]     [1] = { -10'sd402,  -10'sd398,  -10'sd410,  -10'sd404};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [4]     [1] = { -10'sd400,  10'd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [4]     [0] = { -10'sd412,  -10'sd408,  -10'sd420,  -10'sd416};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [4]     [0] = { -10'sd414,  10'd0};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [3]     [1] = { -10'sd424,  -10'sd420,  -10'sd432,  -10'sd428};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [3]     [1] = { -10'sd426,  10'd0};                          
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [3]     [0] = { -10'sd436,  -10'sd430,  -10'sd442,  -10'sd438};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [3]     [0] = { -10'sd440,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [2]     [1] = { -10'sd452,  -10'sd448,  -10'sd460,  -10'sd456};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [2]     [1] = { -10'sd454,  10'd0};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [2]     [0] = { -10'sd464,  -10'sd458,  -10'sd470,  -10'sd466};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [2]     [0] = { -10'sd468,  -10'sd1};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [1]     [1] = { -10'sd478,  -10'sd474,  -10'sd486,  -10'sd482};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [1]     [1] = { -10'sd480,  10'd0};
 Index__HE_2xLTF_MU_RU26 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [1]     [0] = { -10'sd490,  -10'sd484,  -10'sd496,  -10'sd492};
 IndexPilot__HE_2xLTF_RU26               [MODE_BW_IS_80MHZ] [1]     [0] = { -10'sd494,  -10'sd1};

 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [16]    [3] = { 10'd492,  10'd496,  10'd484,  10'd490};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [16]    [3] = { 10'd494,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [16]    [2] = { 10'd482,  10'd486,  10'd474,  10'd478};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [16]    [2] = { 10'd480,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [16]    [1] = { 10'd470,  10'd474,  10'd462,  10'd466};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [16]    [1] = { 10'd468,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [16]    [0] = { 10'd462,  10'd464,  10'd452,  10'd456};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [16]    [0] = { 10'd454,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [15]    [3] = { 10'd438,  10'd442,  10'd430,  10'd436};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [15]    [3] = { 10'd440,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [15]    [2] = { 10'd428,  10'd432,  10'd420,  10'd424};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [15]    [2] = { 10'd426,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [15]    [1] = { 10'd416,  10'd420,  10'd408,  10'd412};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [15]    [1] = { 10'd414,  10'd0};                    
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [15]    [0] = { 10'd408,  10'd410,  10'd398,  10'd402};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [15]    [0] = { 10'd400,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [14]    [3] = { 10'd358,  10'd362,  10'd350,  10'd356}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [14]    [3] = { 10'd360,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [14]    [2] = { 10'd348,  10'd352,  10'd340,  10'd344};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [14]    [2] = { 10'd346,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [14]    [1] = { 10'd336,  10'd340,  10'd328,  10'd332};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [14]    [1] = { 10'd334,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [14]    [0] = { 10'd328,  10'd330,  10'd318,  10'd322};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [14]    [0] = { 10'd320,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [13]    [3] = { 10'd304,  10'd308,  10'd296,  10'd302};   
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [13]    [3] = { 10'd306,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [13]    [2] = { 10'd294,  10'd298,  10'd286,  10'd290};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [13]    [2] = { 10'd292,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [13]    [1] = { 10'd282,  10'd286,  10'd274,  10'd278};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [13]    [1] = { 10'd280,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [13]    [0] = { 10'd274,  10'd276,  10'd264,  10'd268};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [13]    [0] = { 10'd266,  -10'sd1};                    
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [12]    [3] = { 10'd250,  10'd254,  10'd242,  10'd248};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [12]    [3] = { 10'd252,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [12]    [2] = { 10'd240,  10'd244,  10'd232,  10'd236};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [12]    [2] = { 10'd238,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [12]    [1] = { 10'd228,  10'd232,  10'd220,  10'd224};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [12]    [1] = { 10'd226,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [12]    [0] = { 10'd220,  10'd222,  10'd210,  10'd214};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [12]    [0] = { 10'd212,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [11]    [3] = { 10'd196,  10'd200,  10'd188,  10'd194};  
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [11]    [3] = { 10'd198,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [11]    [2] = { 10'd186,  10'd190,  10'd178,  10'd182};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [11]    [2] = { 10'd184,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [11]    [1] = { 10'd174,  10'd178,  10'd166,  10'd170};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [11]    [1] = { 10'd172,  10'd0};                    
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [11]    [0] = { 10'd166,  10'd168,  10'd156,  10'd160};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [11]    [0] = { 10'd158,  -10'sd1};                   
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [10]    [3] = { 10'd116,  10'd120,  10'd108,  10'd114}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [10]    [3] = { 10'd118,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [10]    [2] = { 10'd106,  10'd110,  10'd98,  10'd102};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [10]    [2] = { 10'd104,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [10]    [1] = { 10'd94,  10'd98,  10'd86,  10'd90};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [10]    [1] = { 10'd92,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [10]    [0] = { 10'd86,  10'd88,  10'd76,  10'd80};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [10]    [0] = { 10'd78,  -10'sd1};                
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [9]     [3] = { 10'd62,  10'd66,  10'd54,  10'd60};  
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [9]     [3] = { 10'd64,  10'd1};               
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [9]     [2] = { 10'd52,  10'd56,  10'd44,  10'd48};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [9]     [2] = { 10'd50,  10'd0}; 
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [9]     [1] = { 10'd40,  10'd44,  10'd32,  10'd36};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [9]     [1] = { 10'd38,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [9]     [0] = { 10'd32,  10'd34,  10'd22,  10'd26};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [9]     [0] = { 10'd24,  -10'sd1};                   
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [8]     [3] = { -10'sd26,  -10'sd22,  -10'sd34,  -10'sd28};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [8]     [3] = { -10'sd24,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [8]     [2] = { -10'sd36,  -10'sd32,  -10'sd44,  -10'sd40};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [8]     [2] = { -10'sd38,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [8]     [1] = { -10'sd48,  -10'sd44,  -10'sd56,  -10'sd52};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [8]     [1] = { -10'sd50,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [8]     [0] = { -10'sd56,  -10'sd54,  -10'sd66,  -10'sd62};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [8]     [0] = { -10'sd64,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [7]     [3] = { -10'sd80,  -10'sd76,  -10'sd88,  -10'sd82};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [7]     [3] = { -10'sd78,   10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [7]     [2] = { -10'sd90,  -10'sd86,  -10'sd98,  -10'sd94};     
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [7]     [2] = { -10'sd92,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [7]     [1] = { -10'sd102,  -10'sd98,  -10'sd110,  -10'sd106};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [7]     [1] = { -10'sd104,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [7]     [0] = { -10'sd110,  -10'sd108,  -10'sd120,  -10'sd116}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [7]     [0] = { -10'sd118,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [6]     [3] = { -10'sd160,  -10'sd156,  -10'sd168,  -10'sd162}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [6]     [3] = { -10'sd158,  10'd1};                          
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [6]     [2] = { -10'sd170,  -10'sd166,  -10'sd178,  -10'sd174};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [6]     [2] = { -10'sd172,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [6]     [1] = { -10'sd182,  -10'sd178,  -10'sd190,  -10'sd186};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [6]     [1] = { -10'sd184,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [6]     [0] = { -10'sd194,  -10'sd188,  -10'sd200,  -10'sd196};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [6]     [0] = { -10'sd198,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [5]     [3] = { -10'sd214,  -10'sd210,  -10'sd222,  -10'sd216};  
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [5]     [3] = { -10'sd212,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [5]     [2] = { -10'sd224,  -10'sd220,  -10'sd232,  -10'sd228};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [5]     [2] = { -10'sd226,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [5]     [1] = { -10'sd236,  -10'sd232,  -10'sd244,  -10'sd240};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [5]     [1] = { -10'sd238,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [5]     [0] = { -10'sd248,  -10'sd242,  -10'sd254,  -10'sd250};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [5]     [0] = { -10'sd252,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [4]     [3] = { -10'sd268,  -10'sd264,  -10'sd276,  -10'sd270}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [4]     [3] = { -10'sd266,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [4]     [2] = { -10'sd278,  -10'sd274,  -10'sd286,  -10'sd282};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [4]     [2] = { -10'sd280,  10'd0};                       
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [4]     [1] = { -10'sd290,  -10'sd286,  -10'sd298,  -10'sd294};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [4]     [1] = { -10'sd292,  10'd0};                       
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [4]     [0] = { -10'sd298,  -10'sd296,  -10'sd308,  -10'sd304}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [4]     [0] = { -10'sd306,  -10'sd1};                             
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [3]     [3] = { -10'sd322,  -10'sd318,  -10'sd330,  -10'sd324};   
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [3]     [3] = { -10'sd320,  10'd1};                          
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [3]     [2] = { -10'sd332,  -10'sd328,  -10'sd340,  -10'sd336};    
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [3]     [2] = { -10'sd334,  10'd0};                           
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [3]     [1] = { -10'sd344,  -10'sd340,  -10'sd352,  -10'sd348};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [3]     [1] = { -10'sd346,  10'd0};                       
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [3]     [0] = { -10'sd356,  -10'sd350,  -10'sd362,  -10'sd358}; 
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [3]     [0] = { -10'sd360,  -10'sd1};                        
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [2]     [3] = { -10'sd402,  -10'sd398,  -10'sd410,  -10'sd404};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [2]     [3] = { -10'sd400,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [2]     [2] = { -10'sd412,  -10'sd408,  -10'sd420,  -10'sd416};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [2]     [2] = { -10'sd414,  10'd0};                          
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [2]     [1] = { -10'sd424,  -10'sd420,  -10'sd432,  -10'sd428};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [2]     [1] = { -10'sd426,  10'd0};                          
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [2]     [0] = { -10'sd436,  -10'sd430,  -10'sd442,  -10'sd438};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [2]     [0] = { -10'sd440,  -10'sd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [1]     [3] = { -10'sd456,  -10'sd452,  -10'sd464,  -10'sd458};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [1]     [3] = { -10'sd454,  10'd1};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [1]     [2] = { -10'sd466,  -10'sd462,  -10'sd474,  -10'sd470};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [1]     [2] = { -10'sd468,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [1]     [1] = { -10'sd478,  -10'sd474,  -10'sd486,  -10'sd482};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [1]     [1] = { -10'sd480,  10'd0};
 Index__HE_2xLTF_MU_RU52 [PILOT_INTERP]  [MODE_BW_IS_80MHZ] [1]     [0] = { -10'sd490,  -10'sd484,  -10'sd496,  -10'sd492};
 IndexPilot__HE_2xLTF_RU52               [MODE_BW_IS_80MHZ] [1]     [0] = { -10'sd494,  -10'sd1};

 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [8]     [3] = { 10'd492,  10'd496,  10'd484,  10'd490};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [8]     [3] = { 10'd494,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [8]     [2] = { 10'd470,  10'd474,  10'd462,  10'd466};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [8]     [2] = { 10'd468,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [8]     [1] = { 10'd428,  10'd432,  10'd420,  10'd424};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [8]     [1] = { 10'd426,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [8]     [0] = { 10'd408,  10'd410,  10'd398,  10'd402};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [8]     [0] = { 10'd400,  -10'sd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [7]     [3] = { 10'd358,  10'd362,  10'd350,  10'd356}; 
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [7]     [3] = { 10'd360,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [7]     [2] = { 10'd336,  10'd340,  10'd328,  10'd332};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [7]     [2] = { 10'd334,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [7]     [1] = { 10'd294,  10'd298,  10'd286,  10'd290};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [7]     [1] = { 10'd292,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [7]     [0] = { 10'd274,  10'd276,  10'd264,  10'd268};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [7]     [0] = { 10'd266,  -10'sd1};                    
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [6]     [3] = { 10'd250,  10'd254,  10'd242,  10'd248};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [6]     [3] = { 10'd252,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [6]     [2] = { 10'd228,  10'd232,  10'd220,  10'd224};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [6]     [2] = { 10'd226,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [6]     [1] = { 10'd186,  10'd190,  10'd178,  10'd182};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [6]     [1] = { 10'd184,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [6]     [0] = { 10'd166,  10'd168,  10'd156,  10'd160};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [6]     [0] = { 10'd158,  -10'sd1};                   
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [5]     [3] = { 10'd116,  10'd120,  10'd108,  10'd114};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [5]     [3] = { 10'd118,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [5]     [2] = { 10'd94,  10'd98,  10'd86,  10'd90};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [5]     [2] = { 10'd92,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [5]     [1] = { 10'd52,  10'd56,  10'd44,  10'd48};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [5]     [1] = { 10'd50,  10'd0}; 
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [5]     [0] = { 10'd32,  10'd34,  10'd22,  10'd26};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [5]     [0] = { 10'd24,  -10'sd1};                 
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [4]     [3] = { -10'sd26,  -10'sd22,  -10'sd34,  -10'sd28};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [4]     [3] = { -10'sd24,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [4]     [2] = { -10'sd48,  -10'sd44,  -10'sd56,  -10'sd52};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [4]     [2] = { -10'sd50,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [4]     [1] = { -10'sd90,  -10'sd86,  -10'sd98,  -10'sd94};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [4]     [1] = { -10'sd92,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [4]     [0] = { -10'sd110,  -10'sd108,  -10'sd120,  -10'sd116};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [4]     [0] = { -10'sd118,  -10'sd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [3]     [3] = { -10'sd160,  -10'sd156,  -10'sd168,  -10'sd166};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [3]     [3] = { -10'sd158,  10'd1};                          
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [3]     [2] = { -10'sd182,  -10'sd178,  -10'sd190,  -10'sd186};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [3]     [2] = { -10'sd184,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [3]     [1] = { -10'sd224,  -10'sd220,  -10'sd232,  -10'sd228};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [3]     [1] = { -10'sd226,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [3]     [0] = { -10'sd248,  -10'sd242,  -10'sd254,  -10'sd250};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [3]     [0] = { -10'sd252,  -10'sd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [2]     [3] = { -10'sd268,  -10'sd264,  -10'sd276,  -10'sd270};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [2]     [3] = { -10'sd266,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [2]     [2] = { -10'sd290,  -10'sd286,  -10'sd298,  -10'sd294};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [2]     [2] = { -10'sd292,  10'd0};                       
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [2]     [1] = { -10'sd332,  -10'sd328,  -10'sd340,  -10'sd336};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [2]     [1] = { -10'sd334,  10'd0};                           
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [2]     [0] = { -10'sd356,  -10'sd350,  -10'sd362,  -10'sd358};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [2]     [0] = { -10'sd360,  -10'sd1};                        
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [1]     [3] = { -10'sd402,  -10'sd398,  -10'sd410,  -10'sd404};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [1]     [3] = { -10'sd400,  10'd1};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [1]     [2] = { -10'sd424,  -10'sd420,  -10'sd432,  -10'sd428};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [1]     [2] = { -10'sd426,  10'd0};                          
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [1]     [1] = { -10'sd466,  -10'sd462,  -10'sd474,  -10'sd470};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [1]     [1] = { -10'sd468,  10'd0};
 Index__HE_2xLTF_MU_RU106 [PILOT_INTERP] [MODE_BW_IS_80MHZ] [1]     [0] = { -10'sd490,  -10'sd484,  -10'sd496,  -10'sd492};
 IndexPilot__HE_2xLTF_RU106              [MODE_BW_IS_80MHZ] [1]     [0] = { -10'sd494,  -10'sd1};

end // block: Index_gen

// FOR VIRTUAL INTERPOLATION
// index virtual
//                                             Mode_BW  PilotCount and 2 indexes
//                                                |            |     |    0: VirtualIndex
//                                                |            |     |    1: CoeffVirtual
 Index__HE_1xLTF         [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [1]   [0] = { 10'd100,  10'd120,  10'd76,  10'd96};
 IndexVirt__HE_1xLTF                                          [1]       = { 10'd124,  10'd0};
 Index__HE_1xLTF         [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [0]   [0] = { -10'sd96,  -10'sd76,  -10'sd120,  -10'sd100};
 IndexVirt__HE_1xLTF                                          [0]       = { -10'sd124,  {10{1'b1}}};

 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [9]   [0] = { 10'd112,  10'd120,  10'd98,  10'd110};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [9]       = { 10'd122,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [8]   [0] = { 10'd86,  10'd94,  10'd72,  10'd84};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [8]       = { 10'd96,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [7]   [0] = { 10'd58,  10'd66,  10'd44,  10'd56};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [7]       = { 10'd42,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [6]   [0] = { 10'd32,  10'd40,  10'd18,  10'd30};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [6]       = { 10'd16,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [4]   [0] = { -10'sd30,  -10'sd18,  -10'sd40,  -10'sd32};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [4]       = { -10'sd16,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [3]   [0] = { -10'sd56,  -10'sd44,  -10'sd66,  -10'sd58};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [3]       = { -10'sd42,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd84,  -10'sd72,  -10'sd94,  -10'sd86};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [2]       = { -10'sd96,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd98,  -10'sd120,  -10'sd112};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_20MHZ] [1]       = { -10'sd122,  {10{1'b1}}};

 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [4]   [0] = { 10'd112,  10'd120,  10'd98,  10'd110};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_20MHZ] [4]       = { 10'd122,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [3]   [0] = { 10'd32,  10'd40,  10'd18,  10'd30};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_20MHZ] [3]       = { 10'd16,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd30,  -10'sd18,  -10'sd40,  -10'sd32};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_20MHZ] [2]       = { -10'sd16,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd98,  -10'sd120,  -10'sd112};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_20MHZ] [1]       = { -10'sd122,  {10{1'b1}}};

 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_20MHZ] [2]   [0] = { 10'd32,  10'd40,  10'd18,  10'd30};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_20MHZ] [2]       = { 10'd16,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd30,  -10'sd18,  -10'sd40,  -10'sd32};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_20MHZ] [1]       = { -10'sd16,  10'd0};

 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd232,  -10'sd220,  -10'sd242,  -10'sd234};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [1]       = { -10'sd244,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd204,  -10'sd194,  -10'sd216,  -10'sd208};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [2]       = { -10'sd218,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd178,  -10'sd166,  -10'sd188,  -10'sd180};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [3]       = { -10'sd190,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd152,  -10'sd140,  -10'sd162,  -10'sd154};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [4]       = { -10'sd164,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [5]   [0] = { -10'sd124,  -10'sd112,  -10'sd134,  -10'sd126};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [5]       = { -10'sd110,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [6]   [0] = { -10'sd98,  -10'sd86,  -10'sd108,  -10'sd100};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [6]       = { -10'sd110,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [7]   [0] = { -10'sd72,  -10'sd60,  -10'sd82,  -10'sd74};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [7]       = { -10'sd84,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [8]   [0] = { -10'sd44,  -10'sd32,  -10'sd54,  -10'sd46};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [8]       = { -10'sd56,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [9]   [0] = { -10'sd18,  -10'sd6,  -10'sd28,  -10'sd20};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [9]       = { -10'sd30,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [10]  [0] = { 10'd16,  10'd28,  10'd6,  10'd14};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [10]      = { 10'd30,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [11]  [0] = { 10'd42,  10'd54,  10'd32,  10'd40};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [11]      = { 10'd56,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [12]  [0] = { 10'd70,  10'd82,  10'd60,  10'd68};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [12]      = { 10'd84,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [13]  [0] = { 10'd96,  10'd108,  10'd86,  10'd94};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [13]      = { 10'd110,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [14]  [0] = { 10'd122,  10'd134,  10'd112,  10'd120};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [14]      = { 10'd110,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [15]  [0] = { 10'd150,  10'd162,  10'd140,  10'd148};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [15]      = { 10'd164,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [16]  [0] = { 10'd176,  10'd188,  10'd166,  10'd174};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [16]      = { 10'd190,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [17]  [0] = { 10'd204,  10'd216,  10'd194,  10'd202};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [17]      = { 10'd218,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [18]  [0] = { 10'd230,  10'd242,  10'd220,  10'd228};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_40MHZ] [18]      = { 10'd244,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd232,  -10'sd220,  -10'sd242,  -10'sd234};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [1]       = { -10'sd244,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd178,  -10'sd166,  -10'sd188,  -10'sd180};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [2]       = { -10'sd190,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd98,  -10'sd86,  -10'sd108,  -10'sd100};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [3]       = { -10'sd110,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd44,  -10'sd32,  -10'sd54,  -10'sd46};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [4]       = { -10'sd56,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [5]   [0] = { 10'd42,  10'd54,  10'd32,  10'd40};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [5]       = { 10'd56,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [6]   [0] = { 10'd96,  10'd108,  10'd86,  10'd94};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [6]       = { 10'd110,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [7]   [0] = { 10'd176,  10'd188,  10'd166,  10'd174};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [7]       = { 10'd190,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_40MHZ] [8]   [0] = { 10'd230,  10'd242,  10'd220,  10'd228};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_40MHZ] [8]       = { 10'd244,  10'd0};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd232,  -10'sd220,  -10'sd242,  -10'sd234};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_40MHZ] [1]       = { -10'sd244,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd98,  -10'sd86,  -10'sd108,  -10'sd100};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_40MHZ] [2]       = { -10'sd110,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_40MHZ] [3]   [0] = { 10'd96,  10'd108,  10'd86,  10'd94};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_40MHZ] [3]       = { 10'd110,  10'd0};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_40MHZ] [4]   [0] = { 10'd230,  10'd242,  10'd220,  10'd228};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_40MHZ] [4]       = { 10'd244,  10'd0};
 Index__HE_2xLTF_MU_RU242 [VIRTUAL_INTERP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd16,  -10'sd4,  -10'sd26,  -10'sd18};
 IndexVirt__HE_2xLTF_RU242                 [MODE_BW_IS_40MHZ] [1]       = { -10'sd2,  10'd0};
 Index__HE_2xLTF_MU_RU242 [VIRTUAL_INTERP] [MODE_BW_IS_40MHZ] [2]   [0] = { 10'd14,  10'd26,  10'd4,  10'd12};
 IndexVirt__HE_2xLTF_RU242                 [MODE_BW_IS_40MHZ] [2]       = { 10'd2,  {10{1'b1}}};

 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd488,  -10'sd476,  -10'sd498,  -10'sd490};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [1]       = { -10'sd500,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd462,  -10'sd450,  -10'sd472,  -10'sd464};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [2]       = { -10'sd474,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd434,  -10'sd422,  -10'sd444,  -10'sd436};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [3]       = { -10'sd446,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd408,  -10'sd396,  -10'sd418,  -10'sd410};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [4]       = { -10'sd420,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd384,  -10'sd368,  -10'sd390,  -10'sd382};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [5]       = { -10'sd366,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd354,  -10'sd342,  -10'sd364,  -10'sd356};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [6]       = { -10'sd366,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd328,  -10'sd316,  -10'sd338,  -10'sd330};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [7]       = { -10'sd340,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd300,  -10'sd288,  -10'sd310,  -10'sd302};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [8]       = { -10'sd312,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [9]   [0] = { -10'sd274,  -10'sd262,  -10'sd284,  -10'sd276};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [9]       = { -10'sd286,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [10]  [0] = { -10'sd246,  -10'sd234,  -10'sd256,  -10'sd248};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [10]      = { -10'sd258,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [11]  [0] = { -10'sd226,  -10'sd208,  -10'sd230,  -10'sd228};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [11]      = { -10'sd232,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [12]  [0] = { -10'sd194,  -10'sd180,  -10'sd202,  -10'sd196};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [12]      = { -10'sd204,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [13]  [0] = { -10'sd166,  -10'sd154,  -10'sd176,  -10'sd168};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [13]      = { -10'sd178,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [14]  [0] = { -10'sd138,  -10'sd126,  -10'sd148,  -10'sd140};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [14]      = { -10'sd124,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [15]  [0] = { -10'sd114,  -10'sd100,  -10'sd122,  -10'sd116};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [15]      = { -10'sd124,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [16]  [0] = { -10'sd86,  -10'sd74,  -10'sd96,  -10'sd88};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [16]      = { -10'sd98,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [17]  [0] = { -10'sd58,  -10'sd46,  -10'sd68,  -10'sd60};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [17]      = { -10'sd70,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [18]  [0] = { -10'sd32,  -10'sd20,  -10'sd42,  -10'sd34};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [18]      = { -10'sd44,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [19]  [0] = { -10'sd32,  -10'sd20,  -10'sd42,  -10'sd34}; // Unused
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [19]      = { -10'sd44,  {10{1'b1}}};                     // Unused
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [20]  [0] = { 10'd30,  10'd42,  10'd20,  10'd28};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [20]      = { 10'd44,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [21]  [0] = { 10'd56,  10'd68,  10'd46,  10'd54};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [21]      = { 10'd70,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [22]  [0] = { 10'd84,  10'd96,  10'd74,  10'd82};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [22]      = { 10'd98,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [23]  [0] = { 10'd110,  10'd122,  10'd100,  10'd108};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [23]      = { 10'd124,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [24]  [0] = { 10'd136,  10'd148,  10'd126,  10'd134};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [24]      = { 10'd124,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [25]  [0] = { 10'd164,  10'd176,  10'd154,  10'd162};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [25]      = { 10'd178,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [26]  [0] = { 10'd190,  10'd202,  10'd180,  10'd188};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [26]      = { 10'd204,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [27]  [0] = { 10'd218,  10'd230,  10'd208,  10'd216};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [27]      = { 10'd232,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [28]  [0] = { 10'd244,  10'd256,  10'd234,  10'd242};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [28]      = { 10'd258,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [29]  [0] = { 10'd272,  10'd284,  10'd262,  10'd270};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [29]      = { 10'd286,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [30]  [0] = { 10'd298,  10'd310,  10'd288,  10'd296};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [30]      = { 10'd312,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [31]  [0] = { 10'd326,  10'd338,  10'd316,  10'd324};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [31]      = { 10'd340,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [32]  [0] = { 10'd352,  10'd364,  10'd342,  10'd350};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [32]      = { 10'd366,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [33]  [0] = { 10'd378,  10'd390,  10'd368,  10'd376};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [33]      = { 10'd366,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [34]  [0] = { 10'd406,  10'd418,  10'd396,  10'd404};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [34]      = { 10'd420,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [35]  [0] = { 10'd432,  10'd444,  10'd422,  10'd430};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [35]      = { 10'd446,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [36]  [0] = { 10'd460,  10'd472,  10'd450,  10'd458};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [36]      = { 10'd474,  10'd0};
 Index__HE_2xLTF_MU_RU26 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [37]  [0] = { 10'd486,  10'd498,  10'd476,  10'd484};
 IndexVirt__HE_2xLTF_RU26                  [MODE_BW_IS_80MHZ] [37]      = { 10'd500,  10'd0};

 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd488,  -10'sd476,  -10'sd498,  -10'sd490};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [1]       = { -10'sd500,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd434,  -10'sd422,  -10'sd444,  -10'sd436};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [2]       = { -10'sd446,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd354,  -10'sd342,  -10'sd364,  -10'sd356};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [3]       = { -10'sd366,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd300,  -10'sd288,  -10'sd310,  -10'sd302};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [4]       = { -10'sd312,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd246,  -10'sd234,  -10'sd256,  -10'sd248};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [5]       = { -10'sd258,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd194,  -10'sd180,  -10'sd202,  -10'sd196};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [6]       = { -10'sd204,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd114,  -10'sd100,  -10'sd122,  -10'sd116};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [7]       = { -10'sd124,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd58,  -10'sd46,  -10'sd68,  -10'sd60};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [8]       = { -10'sd70,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [9]   [0] = { 10'd56,  10'd68,  10'd46,  10'd54};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [9]       = { 10'd70,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [10]  [0] = { 10'd110,  10'd122,  10'd100,  10'd108};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [10]      = { 10'd124,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [11]  [0] = { 10'd190,  10'd202,  10'd180,  10'd188};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [11]      = { 10'd204,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [12]  [0] = { 10'd244,  10'd256,  10'd234,  10'd242};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [12]      = { 10'd258,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [13]  [0] = { 10'd298,  10'd310,  10'd288,  10'd296};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [13]      = { 10'd312,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [14]  [0] = { 10'd352,  10'd364,  10'd342,  10'd350};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [14]      = { 10'd366,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [15]  [0] = { 10'd432,  10'd444,  10'd422,  10'd430};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [15]      = { 10'd446,  10'd0};
 Index__HE_2xLTF_MU_RU52 [VIRTUAL_INTERP]  [MODE_BW_IS_80MHZ] [16]  [0] = { 10'd486,  10'd498,  10'd476,  10'd484};
 IndexVirt__HE_2xLTF_RU52                  [MODE_BW_IS_80MHZ] [16]      = { 10'd500,  10'd0};

 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd488,  -10'sd476,  -10'sd498,  -10'sd490};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [1]       = { -10'sd500,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd354,  -10'sd342,  -10'sd364,  -10'sd356};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [2]       = { -10'sd366,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd246,  -10'sd234,  -10'sd256,  -10'sd248};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [3]       = { -10'sd258,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd114,  -10'sd100,  -10'sd122,  -10'sd116};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [4]       = { -10'sd124,  {10{1'b1}}};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [5]   [0] = { 10'd110,  10'd122,  10'd100,  10'd108};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [5]       = { 10'd124,  10'd0};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [6]   [0] = { 10'd244,  10'd256,  10'd234,  10'd242};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [6]       = { 10'd258,  10'd0};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [7]   [0] = { 10'd352,  10'd364,  10'd342,  10'd350};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [7]       = { 10'd366,  10'd0};
 Index__HE_2xLTF_MU_RU106 [VIRTUAL_INTERP] [MODE_BW_IS_80MHZ] [8]   [0] = { 10'd486,  10'd498,  10'd476,  10'd484};
 IndexVirt__HE_2xLTF_RU106                 [MODE_BW_IS_80MHZ] [8]       = { 10'd500,  10'd0};

// FOR SMOOTHING
//                                         Mode_BW    RUIndex PilotCount
//                                            |           |     |
 Index__L_LTF             [SMOOTHING]                              = { 10'd0,  10'd26,  -10'sd26,  -10'sd1};
 Index__HE_LSIG           [SMOOTHING]                              = { 10'd27,  10'd28,  -10'sd28,  -10'sd27};

 Index__VHT_LTF           [SMOOTHING] [MODE_BW_IS_20MHZ]           = { 10'd0,  10'd28,  -10'sd28,  -10'sd1};
 Index__HT_LTF            [SMOOTHING] [MODE_BW_IS_20MHZ]           = { 10'd0,  10'd28,  -10'sd28,  -10'sd1};

 Index__HE_4xLTF_SU       [SMOOTHING] [MODE_BW_IS_20MHZ]           = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};                    
 Index__HE_2xLTF_SU       [SMOOTHING] [MODE_BW_IS_20MHZ] [0]       = { 10'd0,  10'd122,  -10'sd122,  -10'sd2}; 
 Index__HE_1xLTF          [SMOOTHING] [MODE_BW_IS_20MHZ] [0]   [0] = { 10'd0,  10'd120,  -10'sd120,  -10'sd4};

 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [1]       = { -10'sd110,  -10'sd96,  -10'sd121,  -10'sd111};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [2]       = { -10'sd84,  -10'sd70,  -10'sd95,  -10'sd85};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [3]       = { -10'sd57,  -10'sd43,  -10'sd68,  -10'sd58};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [4]       = { -10'sd31,  -10'sd17,  -10'sd42,  -10'sd32};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [5]       = { 10'd0,  10'd16,  -10'sd16,  -10'sd1};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [6]       = { 10'd28,  10'd42,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [7]       = { 10'd54,  10'd68,  10'd43,  10'd53};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [8]       = { 10'd81,  10'd95,  10'd70,  10'd80};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [9]       = { 10'd107,  10'd121,  10'd96,  10'd106};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [1]       = { -10'sd110,  -10'sd70,  -10'sd121,  -10'sd111};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [2]       = { -10'sd57,  -10'sd17,  -10'sd68,  -10'sd58};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [3]       = { 10'd28,  10'd68,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [4]       = { 10'd81,  10'd121,  10'd70,  10'd80};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_20MHZ] [1]       = { -10'sd111,  -10'sd17,  -10'sd122,  -10'sd112};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_20MHZ] [2]       = { 10'd28,  10'd122,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU242 [SMOOTHING] [MODE_BW_IS_20MHZ] [1]       = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};

 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd96,  -10'sd120,  -10'sd111};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd84,  -10'sd70,  -10'sd94,  -10'sd85};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [3]   [0] = { -10'sd57,  -10'sd44,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [4]   [0] = { -10'sd31,  -10'sd18,  -10'sd42,  -10'sd32};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [5]   [0] = { 10'd0,  10'd16,  -10'sd16,  -10'sd2};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [6]   [0] = { 10'd28,  10'd42,  10'd18,  10'd27};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [7]   [0] = { 10'd54,  10'd68,  10'd44,  10'd53};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [8]   [0] = { 10'd81,  10'd94,  10'd70,  10'd80};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_20MHZ] [9]   [0] = { 10'd107,  10'd120,  10'd96,  10'd106};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd70,  -10'sd120,  -10'sd111};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd57,  -10'sd18,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [3]   [0] = { 10'd28,  10'd68,  10'd18,  10'd27};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_20MHZ] [4]   [0] = { 10'd81,  10'd120,  10'd70,  10'd80};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd111,  -10'sd18,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_20MHZ] [2]   [0] = { 10'd28,  10'd122,  10'd18,  10'd27};
 Index__HE_2xLTF_MU_RU242 [SMOOTHING] [MODE_BW_IS_20MHZ] [1]   [0] = { 10'd0,  10'd122,  -10'sd122,  -10'sd2};

 Index__VHT_LTF           [SMOOTHING] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd58,  -10'sd58,  -10'sd1};
 Index__HT_LTF            [SMOOTHING] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd58,  -10'sd58,  -10'sd1};

 Index__HE_4xLTF_SU       [SMOOTHING] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};
 Index__HE_2xLTF_SU       [SMOOTHING] [MODE_BW_IS_40MHZ]       [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd2};
 Index__HE_1xLTF          [SMOOTHING] [MODE_BW_IS_40MHZ] [0]   [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd4};

 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [1]       = { -10'sd232,  -10'sd218,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [2]       = { -10'sd206,  -10'sd192,  -10'sd217,  -10'sd207};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [3]       = { -10'sd178,  -10'sd164,  -10'sd189,  -10'sd179};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [4]       = { -10'sd152,  -10'sd138,  -10'sd163,  -10'sd153};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [5]       = { -10'sd125,  -10'sd111,  -10'sd136,  -10'sd126};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [6]       = { -10'sd98,  -10'sd84,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [7]       = { -10'sd72,  -10'sd58,  -10'sd83,  -10'sd73};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [8]       = { -10'sd44,  -10'sd30,  -10'sd55,  -10'sd45};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [9]       = { -10'sd18,  -10'sd4,  -10'sd29,  -10'sd19};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [10]      = { 10'd15,  10'd29,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [11]      = { 10'd41,  10'd55,  10'd30,  10'd40};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [12]      = { 10'd69,  10'd83,  10'd58,  10'd68};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [13]      = { 10'd95,  10'd109,  10'd84,  10'd94};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [14]      = { 10'd122,  10'd136,  10'd111,  10'd121};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [15]      = { 10'd149,  10'd163,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [16]      = { 10'd175,  10'd189,  10'd164,  10'd174};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [17]      = { 10'd203,  10'd217,  10'd192,  10'd202};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [18]      = { 10'd229,  10'd243,  10'd218,  10'd228};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [1]       = { -10'sd232,  -10'sd192,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [2]       = { -10'sd178,  -10'sd138,  -10'sd189,  -10'sd179};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [3]       = { -10'sd98,  -10'sd58,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [4]       = { -10'sd44,  -10'sd4,  -10'sd55,  -10'sd45};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [5]       = { 10'd15,  10'd55,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [6]       = { 10'd69,  10'd109,  10'd58,  10'd68};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [7]       = { 10'd149,  10'd189,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [8]       = { 10'd203,  10'd243,  10'd192,  10'd202};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [1]       = { -10'sd232,  -10'sd138,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [2]       = { -10'sd98,  -10'sd4,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [3]       = { 10'd15,  10'd109,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [4]       = { 10'd149,  10'd243,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU242 [SMOOTHING] [MODE_BW_IS_40MHZ] [1]       = { -10'sd233,  -10'sd3,  -10'sd244,  -10'sd234};
 Index__HE_4xLTF_MU_RU242 [SMOOTHING] [MODE_BW_IS_40MHZ] [2]       = { 10'd14,  10'd244,  10'd3,  10'd13};
 Index__HE_4xLTF_MU_RU484 [SMOOTHING] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};

 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd230,  -10'sd218,  -10'sd242,  -10'sd232};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd208,  -10'sd192,  -10'sd216,  -10'sd206};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd176,  -10'sd164,  -10'sd188,  -10'sd178};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd150,  -10'sd138,  -10'sd162,  -10'sd152};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [5]   [0] = { -10'sd124,  -10'sd112,  -10'sd136,  -10'sd126};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [6]   [0] = { -10'sd96,  -10'sd84,  -10'sd108,  -10'sd98};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [7]   [0] = { -10'sd70,  -10'sd58,  -10'sd82,  -10'sd72};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [8]   [0] = { -10'sd42,  -10'sd30,  -10'sd54,  -10'sd44};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [9]   [0] = { -10'sd16,  -10'sd4,  -10'sd28,  -10'sd18};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [10]  [0] = { 10'd16,  10'd28,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [11]  [0] = { 10'd42,  10'd54,  10'd30,  10'd40};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [12]  [0] = { 10'd70,  10'd82,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [13]  [0] = { 10'd96,  10'd108,  10'd84,  10'd94};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [14]  [0] = { 10'd124,  10'd136,  10'd112,  10'd122};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [15]  [0] = { 10'd150,  10'd162,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [16]  [0] = { 10'd176,  10'd188,  10'd164,  10'd174};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [17]  [0] = { 10'd204,  10'd216,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_40MHZ] [18]  [0] = { 10'd230,  10'd242,  10'd218,  10'd228};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd230,  -10'sd192,  -10'sd242,  -10'sd232};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd176,  -10'sd138,  -10'sd188,  -10'sd178};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd96,  -10'sd58,  -10'sd108,  -10'sd98};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd42,  -10'sd4,  -10'sd54,  -10'sd44};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [5]   [0] = { 10'd16,  10'd54,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [6]   [0] = { 10'd70,  10'd108,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [7]   [0] = { 10'd150,  10'd188,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_40MHZ] [8]   [0] = { 10'd204,  10'd242,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd230,  -10'sd138,  -10'sd242,  -10'sd232};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd96,  -10'sd4,  -10'sd108,  -10'sd98};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [3]   [0] = { 10'd16,  10'd108,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_40MHZ] [4]   [0] = { 10'd150,  10'd242,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU242 [SMOOTHING] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd232,  -10'sd4,  -10'sd244,  -10'sd234};
 Index__HE_2xLTF_MU_RU242 [SMOOTHING] [MODE_BW_IS_40MHZ] [2]   [0] = { 10'd16,  10'd244,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU484 [SMOOTHING]                          [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd2};

 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [1]       = { -10'sd487,  -10'sd474,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [2]       = { -10'sd461,  -10'sd448,  -10'sd473,  -10'sd463};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [3]       = { -10'sd433,  -10'sd420,  -10'sd445,  -10'sd435};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [4]       = { -10'sd407,  -10'sd394,  -10'sd419,  -10'sd409};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [5]       = { -10'sd380,  -10'sd367,  -10'sd392,  -10'sd382};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [6]       = { -10'sd353,  -10'sd340,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [7]       = { -10'sd327,  -10'sd314,  -10'sd339,  -10'sd329};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [8]       = { -10'sd299,  -10'sd286,  -10'sd311,  -10'sd301};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [9]       = { -10'sd273,  -10'sd260,  -10'sd285,  -10'sd275};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [10]      = { -10'sd245,  -10'sd232,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [11]      = { -10'sd219,  -10'sd206,  -10'sd231,  -10'sd221};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [12]      = { -10'sd191,  -10'sd178,  -10'sd203,  -10'sd193};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [13]      = { -10'sd165,  -10'sd152,  -10'sd177,  -10'sd167};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [14]      = { -10'sd138,  -10'sd125,  -10'sd150,  -10'sd140};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [15]      = { -10'sd111,  -10'sd98,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [16]      = { -10'sd85,  -10'sd72,  -10'sd97,  -10'sd87};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [17]      = { -10'sd57,  -10'sd44,  -10'sd69,  -10'sd59};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [18]      = { -10'sd31,  -10'sd18,  -10'sd43,  -10'sd33};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [19]      = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [20]      = { 10'd30,  10'd43,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [21]      = { 10'd56,  10'd69,  10'd44,  10'd54};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [22]      = { 10'd84,  10'd97,  10'd72,  10'd82};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [23]      = { 10'd110,  10'd123,  10'd98,  10'd108};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [24]      = { 10'd137,  10'd150,  10'd125,  10'd135};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [25]      = { 10'd164,  10'd177,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [26]      = { 10'd190,  10'd203,  10'd178,  10'd188};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [27]      = { 10'd218,  10'd231,  10'd206,  10'd216};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [28]      = { 10'd244,  10'd257,  10'd232,  10'd242};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [29]      = { 10'd272,  10'd285,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [30]      = { 10'd298,  10'd311,  10'd286,  10'd296};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [31]      = { 10'd326,  10'd339,  10'd314,  10'd324};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [32]      = { 10'd352,  10'd365,  10'd340,  10'd350};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [33]      = { 10'd379,  10'd392,  10'd367,  10'd377};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [34]      = { 10'd406,  10'd419,  10'd394,  10'd404};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [35]      = { 10'd432,  10'd445,  10'd420,  10'd430};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [36]      = { 10'd460,  10'd473,  10'd448,  10'd458};
 Index__HE_4xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [37]      = { 10'd486,  10'd499,  10'd474,  10'd484};

 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [1]       = { -10'sd487,  -10'sd448,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [2]       = { -10'sd433,  -10'sd394,  -10'sd445,  -10'sd435};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [3]       = { -10'sd353,  -10'sd314,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [4]       = { -10'sd299,  -10'sd260,  -10'sd311,  -10'sd301};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [5]       = { -10'sd245,  -10'sd206,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [6]       = { -10'sd191,  -10'sd152,  -10'sd203,  -10'sd193};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [7]       = { -10'sd111,  -10'sd72,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [8]       = { -10'sd57,  -10'sd18,  -10'sd69,  -10'sd59};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [9]       = { 10'd30,  10'd69,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [10]      = { 10'd84,  10'd123,  10'd72,  10'd82};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [11]      = { 10'd164,  10'd203,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [12]      = { 10'd218,  10'd257,  10'd206,  10'd216};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [13]      = { 10'd272,  10'd311,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [14]      = { 10'd326,  10'd365,  10'd314,  10'd324};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [15]      = { 10'd406,  10'd445,  10'd394,  10'd404};
 Index__HE_4xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [16]      = { 10'd460,  10'd499,  10'd448,  10'd458};

 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [1]       = { -10'sd487,  -10'sd394,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [2]       = { -10'sd353,  -10'sd260,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [3]       = { -10'sd245,  -10'sd152,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [4]       = { -10'sd111,  -10'sd18,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [5]       = { 10'd30,  10'd123,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [6]       = { 10'd164,  10'd257,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [7]       = { 10'd272,  10'd365,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [8]       = { 10'd406,  10'd499,  10'd394,  10'd404};

 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd474,  -10'sd498,  -10'sd488};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd460,  -10'sd448,  -10'sd472,  -10'sd462};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd433,  -10'sd420,  -10'sd444,  -10'sd435};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd406,  -10'sd394,  -10'sd418,  -10'sd408};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd380,  -10'sd368,  -10'sd392,  -10'sd382};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd352,  -10'sd340,  -10'sd364,  -10'sd354};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd326,  -10'sd314,  -10'sd338,  -10'sd328};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd298,  -10'sd286,  -10'sd310,  -10'sd300};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [9]   [0] = { -10'sd272,  -10'sd260,  -10'sd284,  -10'sd274};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [10]  [0] = { -10'sd244,  -10'sd232,  -10'sd256,  -10'sd246};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [11]  [0] = { -10'sd218,  -10'sd206,  -10'sd230,  -10'sd220};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [12]  [0] = { -10'sd190,  -10'sd178,  -10'sd202,  -10'sd192};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [13]  [0] = { -10'sd164,  -10'sd152,  -10'sd176,  -10'sd166};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [14]  [0] = { -10'sd138,  -10'sd126,  -10'sd150,  -10'sd140};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [15]  [0] = { -10'sd110,  -10'sd98,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [16]  [0] = { -10'sd84,  -10'sd72,  -10'sd96,  -10'sd86};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [17]  [0] = { -10'sd56,  -10'sd44,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [18]  [0] = { -10'sd30,  -10'sd18,  -10'sd42,  -10'sd32};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [19]  [0] = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [20]  [0] = { 10'd30,  10'd42,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [21]  [0] = { 10'd56,  10'd68,  10'd44,  10'd54};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [22]  [0] = { 10'd84,  10'd96,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [23]  [0] = { 10'd110,  10'd122,  10'd98,  10'd108};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [24]  [0] = { 10'd138,  10'd150,  10'd126,  10'd136};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [25]  [0] = { 10'd164,  10'd176,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [26]  [0] = { 10'd190,  10'd202,  10'd178,  10'd188};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [27]  [0] = { 10'd218,  10'd230,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [28]  [0] = { 10'd244,  10'd256,  10'd232,  10'd242};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [29]  [0] = { 10'd272,  10'd284,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [30]  [0] = { 10'd298,  10'd310,  10'd286,  10'd296};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [31]  [0] = { 10'd326,  10'd338,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [32]  [0] = { 10'd352,  10'd364,  10'd340,  10'd350};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [33]  [0] = { 10'd378,  10'd392,  10'd366,  10'd376};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [34]  [0] = { 10'd406,  10'd418,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [35]  [0] = { 10'd432,  10'd444,  10'd420,  10'd430};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [36]  [0] = { 10'd460,  10'd472,  10'd448,  10'd458};
 Index__HE_2xLTF_MU_RU26  [SMOOTHING] [MODE_BW_IS_80MHZ] [37]  [0] = { 10'd486,  10'd498,  10'd474,  10'd484};

 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd448,  -10'sd498,  -10'sd488};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd432,  -10'sd394,  -10'sd444,  -10'sd434};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd352,  -10'sd314,  -10'sd364,  -10'sd354};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd298,  -10'sd260,  -10'sd310,  -10'sd300};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd244,  -10'sd206,  -10'sd256,  -10'sd246};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd190,  -10'sd152,  -10'sd202,  -10'sd192};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd110,  -10'sd72,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd56,  -10'sd18,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [9]   [0] = { 10'd30,  10'd68,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [10]  [0] = { 10'd84,  10'd122,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [11]  [0] = { 10'd164,  10'd202,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [12]  [0] = { 10'd218,  10'd256,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [13]  [0] = { 10'd272,  10'd310,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [14]  [0] = { 10'd326,  10'd364,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [15]  [0] = { 10'd406,  10'd444,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU52  [SMOOTHING] [MODE_BW_IS_80MHZ] [16]  [0] = { 10'd460,  10'd498,  10'd448,  10'd458};

 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd394,  -10'sd498,  -10'sd488};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd352,  -10'sd260,  -10'sd364,  -10'sd354};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd244,  -10'sd152,  -10'sd256,  -10'sd246};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd110,  -10'sd18,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [5]   [0] = { 10'd30,  10'd122,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [6]   [0] = { 10'd164,  10'd256,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [7]   [0] = { 10'd272,  10'd364,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU106 [SMOOTHING] [MODE_BW_IS_80MHZ] [8]   [0] = { 10'd406,  10'd498,  10'd394,  10'd404};


// FOR CHAN_INTERP
//                                          Mode_BW      RUIndex PilotCount
//                                              |           |     |
 Index__HE_2xLTF_SU       [CHAN_INTERP] [MODE_BW_IS_20MHZ]       [0] = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};
 Index__HE_1xLTF          [CHAN_INTERP] [MODE_BW_IS_20MHZ] [0]   [0] = { 10'd0,  10'd124,  -10'sd124,  -10'sd4};

 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd96,  -10'sd122,  -10'sd111};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd84,  -10'sd70,  -10'sd96,  -10'sd85};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [3]   [0] = { -10'sd57,  -10'sd43,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [4]   [0] = { -10'sd31,  -10'sd17,  -10'sd42,  -10'sd32};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [5]   [0] = { 10'd0,  10'd16,  -10'sd16,  -10'sd1};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [6]   [0] = { 10'd28,  10'd42,  10'd16,  10'd27};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [7]   [0] = { 10'd54,  10'd68,  10'd42,  10'd53};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [8]   [0] = { 10'd81,  10'd95,  10'd70,  10'd80};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [9]   [0] = { 10'd107,  10'd121,  10'd96,  10'd106};

 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd70,  -10'sd122,  -10'sd111};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd57,  -10'sd17,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [3]   [0] = { 10'd28,  10'd68,  10'd16,  10'd27};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_20MHZ] [4]   [0] = { 10'd81,  10'd121,  10'd70,  10'd80};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd111,  -10'sd17,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_20MHZ] [2]   [0] = { 10'd28,  10'd122,  10'd16,  10'd27};
 Index__HE_2xLTF_MU_RU242 [CHAN_INTERP] [MODE_BW_IS_20MHZ] [1]   [0] = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};

 Index__HE_2xLTF_SU       [CHAN_INTERP] [MODE_BW_IS_40MHZ]       [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};
 Index__HE_1xLTF          [CHAN_INTERP] [MODE_BW_IS_40MHZ] [0]   [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd4};

 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd232,  -10'sd218,  -10'sd244,  -10'sd234};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd208,  -10'sd192,  -10'sd218,  -10'sd206};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd176,  -10'sd164,  -10'sd190,  -10'sd178};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd150,  -10'sd138,  -10'sd164,  -10'sd152};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [5]   [0] = { -10'sd124,  -10'sd110,  -10'sd136,  -10'sd126};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [6]   [0] = { -10'sd96,  -10'sd84,  -10'sd110,  -10'sd98};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [7]   [0] = { -10'sd70,  -10'sd58,  -10'sd84,  -10'sd72};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [8]   [0] = { -10'sd42,  -10'sd30,  -10'sd56,  -10'sd44};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [9]   [0] = { -10'sd16,  -10'sd4,  -10'sd30,  -10'sd18};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [10]  [0] = { 10'd16,  10'd30,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [11]  [0] = { 10'd42,  10'd56,  10'd30,  10'd40};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [12]  [0] = { 10'd70,  10'd84,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [13]  [0] = { 10'd96,  10'd110,  10'd84,  10'd94};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [14]  [0] = { 10'd124,  10'd136,  10'd110,  10'd122};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [15]  [0] = { 10'd150,  10'd164,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [16]  [0] = { 10'd176,  10'd190,  10'd164,  10'd174};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [17]  [0] = { 10'd204,  10'd218,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [18]  [0] = { 10'd230,  10'd244,  10'd218,  10'd228};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd230,  -10'sd192,  -10'sd244,  -10'sd232};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd176,  -10'sd138,  -10'sd190,  -10'sd178};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd96,  -10'sd58,  -10'sd110,  -10'sd98};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd42,  -10'sd4,  -10'sd56,  -10'sd44};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [5]   [0] = { 10'd16,  10'd56,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [6]   [0] = { 10'd70,  10'd110,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [7]   [0] = { 10'd150,  10'd190,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_40MHZ] [8]   [0] = { 10'd204,  10'd244,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd232,  -10'sd138,  -10'sd244,  -10'sd234};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd98,  -10'sd4,  -10'sd110,  -10'sd100};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_40MHZ] [3]   [0] = { 10'd16,  10'd110,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_40MHZ] [4]   [0] = { 10'd150,  10'd244,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU242 [CHAN_INTERP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'd232,  -10'd2,  -10'sd244,  -10'sd234};
 Index__HE_2xLTF_MU_RU242 [CHAN_INTERP] [MODE_BW_IS_40MHZ] [2]   [0] = { 10'd14,  10'd244,  10'd2,  10'd12};
 Index__HE_2xLTF_MU_RU484 [CHAN_INTERP]                          [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};

 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd474,  -10'sd500,  -10'sd488};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd460,  -10'sd448,  -10'sd474,  -10'sd462};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd432,  -10'sd420,  -10'sd446,  -10'sd434};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd406,  -10'sd394,  -10'sd420,  -10'sd408};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd380,  -10'sd366,  -10'sd392,  -10'sd382};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd352,  -10'sd340,  -10'sd366,  -10'sd354};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd326,  -10'sd314,  -10'sd340,  -10'sd328};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd298,  -10'sd286,  -10'sd312,  -10'sd300};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [9]   [0] = { -10'sd272,  -10'sd260,  -10'sd286,  -10'sd274};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [10]  [0] = { -10'sd244,  -10'sd232,  -10'sd258,  -10'sd246};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [11]  [0] = { -10'sd218,  -10'sd206,  -10'sd232,  -10'sd220};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [12]  [0] = { -10'sd190,  -10'sd178,  -10'sd204,  -10'sd192};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [13]  [0] = { -10'sd164,  -10'sd152,  -10'sd178,  -10'sd166};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [14]  [0] = { -10'sd138,  -10'sd124,  -10'sd150,  -10'sd140};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [15]  [0] = { -10'sd110,  -10'sd98,  -10'sd124,  -10'sd112};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [16]  [0] = { -10'sd84,  -10'sd72,  -10'sd98,  -10'sd86};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [17]  [0] = { -10'sd56,  -10'sd44,  -10'sd70,  -10'sd58};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [18]  [0] = { -10'sd30,  -10'sd18,  -10'sd44,  -10'sd32};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [19]  [0] = { 10'd4,  10'd16,  -10'sd16,  -10'sd4}; // Not used
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [20]  [0] = { 10'd30,  10'd44,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [21]  [0] = { 10'd56,  10'd70,  10'd44,  10'd54};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [22]  [0] = { 10'd84,  10'd98,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [23]  [0] = { 10'd110,  10'd124,  10'd98,  10'd108};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [24]  [0] = { 10'd136,  10'd150,  10'd124,  10'd134};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [25]  [0] = { 10'd164,  10'd178,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [26]  [0] = { 10'd190,  10'd204,  10'd178,  10'd188};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [27]  [0] = { 10'd218,  10'd232,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [28]  [0] = { 10'd244,  10'd258,  10'd232,  10'd242};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [29]  [0] = { 10'd272,  10'd286,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [30]  [0] = { 10'd298,  10'd312,  10'd286,  10'd296};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [31]  [0] = { 10'd326,  10'd340,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [32]  [0] = { 10'd352,  10'd366,  10'd340,  10'd350};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [33]  [0] = { 10'd378,  10'd392,  10'd366,  10'd376};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [34]  [0] = { 10'd406,  10'd420,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [35]  [0] = { 10'd432,  10'd446,  10'd420,  10'd430};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [36]  [0] = { 10'd460,  10'd474,  10'd448,  10'd458};
 Index__HE_2xLTF_MU_RU26  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [37]  [0] = { 10'd486,  10'd500,  10'd474,  10'd484};

 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd448,  -10'sd500,  -10'sd488};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd432,  -10'sd394,  -10'sd446,  -10'sd434};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd352,  -10'sd314,  -10'sd366,  -10'sd354};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd298,  -10'sd260,  -10'sd312,  -10'sd300};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd244,  -10'sd206,  -10'sd258,  -10'sd246};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd190,  -10'sd152,  -10'sd204,  -10'sd192};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd110,  -10'sd72,  -10'sd124,  -10'sd112};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd56,  -10'sd18,  -10'sd70,  -10'sd58};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [9]   [0] = { 10'd30,  10'd70,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [10]  [0] = { 10'd84,  10'd124,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [11]  [0] = { 10'd164,  10'd204,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [12]  [0] = { 10'd218,  10'd258,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [13]  [0] = { 10'd272,  10'd312,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [14]  [0] = { 10'd326,  10'd366,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [15]  [0] = { 10'd406,  10'd446,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU52  [CHAN_INTERP] [MODE_BW_IS_80MHZ] [16]  [0] = { 10'd460,  10'd500,  10'd448,  10'd458};

 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd394,  -10'sd500,  -10'sd488};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd352,  -10'sd260,  -10'sd366,  -10'sd354};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd244,  -10'sd152,  -10'sd258,  -10'sd246};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd110,  -10'sd18,  -10'sd124,  -10'sd112};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [5]   [0] = { 10'd30,  10'd124,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [6]   [0] = { 10'd164,  10'd258,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [7]   [0] = { 10'd272,  10'd366,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU106 [CHAN_INTERP] [MODE_BW_IS_80MHZ] [8]   [0] = { 10'd406,  10'd500,  10'd394,  10'd404};

// FOR POST_CORDIC
//                                              Mode_BW      RUIndex PilotCount
//                                                  |           |     |
 Index__L_LTF              [POST_CORDIC]                                  = { 10'd0,  10'd26,  -10'sd26,  -10'sd1};
 Index__HE_LSIG            [POST_CORDIC]                                  = { 10'd27,  10'd28,  -10'sd28,  -10'sd27};

 Index__VHT_LTF            [POST_CORDIC]    [MODE_BW_IS_20MHZ]            = { 10'd0,  10'd28,  -10'sd28,  -10'sd1};
 Index__HT_LTF             [POST_CORDIC]    [MODE_BW_IS_20MHZ]            = { 10'd0,  10'd28,  -10'sd28,  -10'sd1};

 Index__HE_4xLTF_SU        [POST_CORDIC]    [MODE_BW_IS_20MHZ]            = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};
 Index__HE_2xLTF_SU        [POST_CORDIC]    [MODE_BW_IS_20MHZ]       [0]  = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};
 Index__HE_1xLTF           [POST_CORDIC]    [MODE_BW_IS_20MHZ] [0]   [0]  = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};

 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]        = { -10'sd110,  -10'sd96,  -10'sd121,  -10'sd111};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [2]        = { -10'sd84,  -10'sd70,  -10'sd95,  -10'sd85};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [3]        = { -10'sd57,  -10'sd43,  -10'sd68,  -10'sd58};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [4]        = { -10'sd31,  -10'sd17,  -10'sd42,  -10'sd32};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [5]        = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [6]        = { 10'd28,  10'd42,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [7]        = { 10'd54,  10'd68,  10'd43,  10'd53};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [8]        = { 10'd81,  10'd95,  10'd70,  10'd80};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [9]        = { 10'd107,  10'd121,  10'd96,  10'd106};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]        = { -10'sd110,  -10'sd70,  -10'sd121,  -10'sd111};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [2]        = { -10'sd57,  -10'sd17,  -10'sd68,  -10'sd58};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [3]        = { 10'd28,  10'd68,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [4]        = { 10'd81,  10'd121,  10'd70,  10'd80};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]        = { -10'sd111,  -10'sd17,  -10'sd122,  -10'sd112};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_20MHZ] [2]        = { 10'd28,  10'd122,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU242  [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]        = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};

 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]   [0]  = { -10'sd110,  -10'sd96,  -10'sd121,  -10'sd111}; 
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [2]   [0]  = { -10'sd84,  -10'sd70,  -10'sd95,  -10'sd85};    
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [3]   [0]  = { -10'sd57,  -10'sd43,  -10'sd68,  -10'sd58};    
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [4]   [0]  = { -10'sd31,  -10'sd17,  -10'sd42,  -10'sd32};    
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [5]   [0]  = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};          
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [6]   [0]  = { 10'd28,  10'd42,  10'd17,  10'd27};            
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [7]   [0]  = { 10'd54,  10'd68,  10'd43,  10'd53};            
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [8]   [0]  = { 10'd81,  10'd95,  10'd70,  10'd80};            
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [9]   [0]  = { 10'd107,  10'd121,  10'd96,  10'd106};         

 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]   [0]  = { -10'sd110,  -10'sd70,  -10'sd121,  -10'sd111};    
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [2]   [0]  = { -10'sd57,  -10'sd17,  -10'sd68,  -10'sd58};       
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [3]   [0]  = { 10'd28,  10'd68,  10'd17,  10'd27};               
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_20MHZ] [4]   [0]  = { 10'd81,  10'd121,  10'd70,  10'd80};              
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]   [0]  = { -10'sd111,  -10'sd17,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_20MHZ] [2]   [0]  = { 10'd28,  10'd122,  10'd17,  10'd27};
 Index__HE_2xLTF_MU_RU242  [POST_CORDIC]    [MODE_BW_IS_20MHZ] [1]   [0]  = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};

 Index__VHT_LTF            [POST_CORDIC]    [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd58,  -10'sd58,  -10'sd1};
 Index__HT_LTF             [POST_CORDIC]    [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd58,  -10'sd58,  -10'sd1};

 Index__HE_4xLTF_SU        [POST_CORDIC]    [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};
 Index__HE_2xLTF_SU        [POST_CORDIC]    [MODE_BW_IS_40MHZ]       [0]  = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};
 Index__HE_1xLTF           [POST_CORDIC]    [MODE_BW_IS_40MHZ] [0]   [0]  = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};

 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]        = { -10'sd232,  -10'sd218,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]        = { -10'sd206,  -10'sd192,  -10'sd217,  -10'sd207};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [3]        = { -10'sd178,  -10'sd164,  -10'sd189,  -10'sd179};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [4]        = { -10'sd152,  -10'sd138,  -10'sd163,  -10'sd153};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [5]        = { -10'sd125,  -10'sd111,  -10'sd136,  -10'sd126};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [6]        = { -10'sd98,  -10'sd84,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [7]        = { -10'sd72,  -10'sd58,  -10'sd83,  -10'sd73};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [8]        = { -10'sd44,  -10'sd30,  -10'sd55,  -10'sd45};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [9]        = { -10'sd18,  -10'sd4,  -10'sd29,  -10'sd19};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [10]       = { 10'd15,  10'd29,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [11]       = { 10'd41,  10'd55,  10'd30,  10'd40};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [12]       = { 10'd69,  10'd83,  10'd58,  10'd68};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [13]       = { 10'd95,  10'd109,  10'd84,  10'd94};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [14]       = { 10'd122,  10'd136,  10'd111,  10'd121};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [15]       = { 10'd149,  10'd163,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [16]       = { 10'd175,  10'd189,  10'd164,  10'd174};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [17]       = { 10'd203,  10'd217,  10'd192,  10'd202};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [18]       = { 10'd229,  10'd243,  10'd218,  10'd228};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]        = { -10'sd232,  -10'sd192,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]        = { -10'sd178,  -10'sd138,  -10'sd189,  -10'sd179};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [3]        = { -10'sd98,  -10'sd58,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [4]        = { -10'sd44,  -10'sd4,  -10'sd55,  -10'sd45};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [5]        = { 10'd15,  10'd55,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [6]        = { 10'd69,  10'd109,  10'd58,  10'd68};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [7]        = { 10'd149,  10'd189,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [8]        = { 10'd203,  10'd243,  10'd192,  10'd202};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]        = { -10'sd232,  -10'sd138,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]        = { -10'sd98,  -10'sd4,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [3]        = { 10'd15,  10'd109,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [4]        = { 10'd149,  10'd243,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU242  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]        = { -10'sd233,  -10'sd3,  -10'sd244,  -10'sd234};
 Index__HE_4xLTF_MU_RU242  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]        = { 10'd14,  10'd244,  10'd3,  10'd13};
 Index__HE_4xLTF_MU_RU484  [POST_CORDIC]    [MODE_BW_IS_40MHZ]            = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};

 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]    [0] = { -10'sd232,  -10'sd218,  -10'sd243,  -10'sd233};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]    [0] = { -10'sd206,  -10'sd192,  -10'sd217,  -10'sd207};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [3]    [0] = { -10'sd178,  -10'sd164,  -10'sd189,  -10'sd179};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [4]    [0] = { -10'sd152,  -10'sd138,  -10'sd163,  -10'sd153};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [5]    [0] = { -10'sd125,  -10'sd111,  -10'sd136,  -10'sd126};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [6]    [0] = { -10'sd98,  -10'sd84,  -10'sd109,  -10'sd99};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [7]    [0] = { -10'sd72,  -10'sd58,  -10'sd83,  -10'sd73};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [8]    [0] = { -10'sd44,  -10'sd30,  -10'sd55,  -10'sd45};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [9]    [0] = { -10'sd18,  -10'sd4,  -10'sd29,  -10'sd19};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [10]   [0] = { 10'd15,  10'd29,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [11]   [0] = { 10'd41,  10'd55,  10'd30,  10'd40};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [12]   [0] = { 10'd69,  10'd83,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [13]   [0] = { 10'd95,  10'd109,  10'd84,  10'd94};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [14]   [0] = { 10'd122,  10'd136,  10'd111,  10'd121};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [15]   [0] = { 10'd149,  10'd163,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [16]   [0] = { 10'd175,  10'd189,  10'd164,  10'd174};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [17]   [0] = { 10'd203,  10'd217,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [18]   [0] = { 10'd229,  10'd243,  10'd218,  10'd228};

 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]    [0] = { -10'sd232,  -10'sd192,  -10'sd243,  -10'sd233};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]    [0] = { -10'sd178,  -10'sd138,  -10'sd189,  -10'sd179};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [3]    [0] = { -10'sd98,  -10'sd58,  -10'sd109,  -10'sd99};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [4]    [0] = { -10'sd44,  -10'sd4,  -10'sd55,  -10'sd45};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [5]    [0] = { 10'd15,  10'd55,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [6]    [0] = { 10'd69,  10'd109,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [7]    [0] = { 10'd149,  10'd189,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_40MHZ] [8]    [0] = { 10'd203,  10'd243,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]    [0] = { -10'sd232,  -10'sd138,  -10'sd243,  -10'sd233};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]    [0] = { -10'sd98,  -10'sd4,  -10'sd109,  -10'sd99};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [3]    [0] = { 10'd15,  10'd109,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [4]    [0] = { 10'd149,  10'd243,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU242  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [1]    [0] = { -10'sd233,  -10'sd3,  -10'sd244,  -10'sd234};
 Index__HE_2xLTF_MU_RU242  [POST_CORDIC]    [MODE_BW_IS_40MHZ] [2]    [0] = { 10'd14,  10'd244,  10'd3,  10'd13};
 Index__HE_2xLTF_MU_RU484  [POST_CORDIC]                              [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};

 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [1]        = { -10'sd488,  -10'sd474,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [2]        = { -10'sd462,  -10'sd448,  -10'sd473,  -10'sd463};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [3]        = { -10'sd434,  -10'sd420,  -10'sd445,  -10'sd435};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [4]        = { -10'sd408,  -10'sd394,  -10'sd419,  -10'sd409};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [5]        = { -10'sd381,  -10'sd367,  -10'sd392,  -10'sd382};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [6]        = { -10'sd354,  -10'sd340,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [7]        = { -10'sd328,  -10'sd314,  -10'sd339,  -10'sd329};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [8]        = { -10'sd300,  -10'sd286,  -10'sd311,  -10'sd301};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [9]        = { -10'sd274,  -10'sd260,  -10'sd285,  -10'sd275};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [10]       = { -10'sd246,  -10'sd232,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [11]       = { -10'sd220,  -10'sd206,  -10'sd231,  -10'sd221};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [12]       = { -10'sd192,  -10'sd178,  -10'sd203,  -10'sd193};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [13]       = { -10'sd166,  -10'sd152,  -10'sd177,  -10'sd167};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [14]       = { -10'sd139,  -10'sd125,  -10'sd150,  -10'sd140};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [15]       = { -10'sd112,  -10'sd98,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [16]       = { -10'sd86,  -10'sd72,  -10'sd97,  -10'sd87};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [17]       = { -10'sd58,  -10'sd44,  -10'sd69,  -10'sd59};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [18]       = { -10'sd32,  -10'sd18,  -10'sd43,  -10'sd33};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [19]       = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [20]       = { 10'd29,  10'd43,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [21]       = { 10'd55,  10'd69,  10'd44,  10'd54};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [22]       = { 10'd83,  10'd97,  10'd72,  10'd82};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [23]       = { 10'd109,  10'd123,  10'd98,  10'd108};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [24]       = { 10'd136,  10'd150,  10'd125,  10'd135};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [25]       = { 10'd163,  10'd177,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [26]       = { 10'd189,  10'd203,  10'd178,  10'd188};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [27]       = { 10'd217,  10'd231,  10'd206,  10'd216};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [28]       = { 10'd243,  10'd257,  10'd232,  10'd242};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [29]       = { 10'd271,  10'd285,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [30]       = { 10'd297,  10'd311,  10'd286,  10'd296};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [31]       = { 10'd325,  10'd339,  10'd314,  10'd324};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [32]       = { 10'd351,  10'd365,  10'd340,  10'd350};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [33]       = { 10'd378,  10'd392,  10'd367,  10'd377};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [34]       = { 10'd405,  10'd419,  10'd394,  10'd404};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [35]       = { 10'd431,  10'd445,  10'd420,  10'd430};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [36]       = { 10'd459,  10'd473,  10'd448,  10'd458};
 Index__HE_4xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [37]       = { 10'd485,  10'd499,  10'd474,  10'd484};

 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [1]        = { -10'sd488,  -10'sd448,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [2]        = { -10'sd434,  -10'sd394,  -10'sd445,  -10'sd435};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [3]        = { -10'sd354,  -10'sd314,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [4]        = { -10'sd300,  -10'sd260,  -10'sd311,  -10'sd301};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [5]        = { -10'sd246,  -10'sd206,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [6]        = { -10'sd192,  -10'sd152,  -10'sd203,  -10'sd193};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [7]        = { -10'sd112,  -10'sd72,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [8]        = { -10'sd58,  -10'sd18,  -10'sd69,  -10'sd59};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [9]        = { 10'd29,  10'd69,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [10]       = { 10'd83,  10'd123,  10'd72,  10'd82};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [11]       = { 10'd163,  10'd203,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [12]       = { 10'd217,  10'd257,  10'd206,  10'd216};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [13]       = { 10'd271,  10'd311,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [14]       = { 10'd325,  10'd365,  10'd314,  10'd324};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [15]       = { 10'd405,  10'd445,  10'd394,  10'd404};
 Index__HE_4xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [16]       = { 10'd459,  10'd499,  10'd448,  10'd458};

 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [1]        = { -10'sd488,  -10'sd394,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [2]        = { -10'sd354,  -10'sd260,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [3]        = { -10'sd246,  -10'sd152,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [4]        = { -10'sd112,  -10'sd18,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [5]        = { 10'd29,  10'd123,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [6]        = { 10'd163,  10'd257,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [7]        = { 10'd271,  10'd365,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [8]        = { 10'd405,  10'd499,  10'd394,  10'd404};

 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [1]    [0] = { -10'sd488,  -10'sd474,  -10'sd499,  -10'sd489};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [2]    [0] = { -10'sd462,  -10'sd448,  -10'sd473,  -10'sd463};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [3]    [0] = { -10'sd434,  -10'sd420,  -10'sd445,  -10'sd435};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [4]    [0] = { -10'sd408,  -10'sd394,  -10'sd419,  -10'sd409};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [5]    [0] = { -10'sd381,  -10'sd367,  -10'sd392,  -10'sd382};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [6]    [0] = { -10'sd354,  -10'sd340,  -10'sd365,  -10'sd355};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [7]    [0] = { -10'sd328,  -10'sd314,  -10'sd339,  -10'sd329};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [8]    [0] = { -10'sd300,  -10'sd286,  -10'sd311,  -10'sd301};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [9]    [0] = { -10'sd274,  -10'sd260,  -10'sd285,  -10'sd275};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [10]   [0] = { -10'sd246,  -10'sd232,  -10'sd257,  -10'sd247};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [11]   [0] = { -10'sd220,  -10'sd206,  -10'sd231,  -10'sd221};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [12]   [0] = { -10'sd192,  -10'sd178,  -10'sd203,  -10'sd193};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [13]   [0] = { -10'sd166,  -10'sd152,  -10'sd177,  -10'sd167};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [14]   [0] = { -10'sd139,  -10'sd125,  -10'sd150,  -10'sd140};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [15]   [0] = { -10'sd112,  -10'sd98,  -10'sd123,  -10'sd113};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [16]   [0] = { -10'sd86,  -10'sd72,  -10'sd97,  -10'sd87};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [17]   [0] = { -10'sd58,  -10'sd44,  -10'sd69,  -10'sd59};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [18]   [0] = { -10'sd32,  -10'sd18,  -10'sd43,  -10'sd33};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [19]   [0] = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [20]   [0] = { 10'd29,  10'd43,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [21]   [0] = { 10'd55,  10'd69,  10'd44,  10'd54};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [22]   [0] = { 10'd83,  10'd97,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [23]   [0] = { 10'd109,  10'd123,  10'd98,  10'd108};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [24]   [0] = { 10'd136,  10'd150,  10'd125,  10'd135};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [25]   [0] = { 10'd163,  10'd177,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [26]   [0] = { 10'd189,  10'd203,  10'd178,  10'd188};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [27]   [0] = { 10'd217,  10'd231,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [28]   [0] = { 10'd243,  10'd257,  10'd232,  10'd242};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [29]   [0] = { 10'd271,  10'd285,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [30]   [0] = { 10'd297,  10'd311,  10'd286,  10'd296};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [31]   [0] = { 10'd325,  10'd339,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [32]   [0] = { 10'd351,  10'd365,  10'd340,  10'd350};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [33]   [0] = { 10'd378,  10'd392,  10'd367,  10'd377};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [34]   [0] = { 10'd405,  10'd419,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [35]   [0] = { 10'd431,  10'd445,  10'd420,  10'd430};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [36]   [0] = { 10'd459,  10'd473,  10'd448,  10'd458};
 Index__HE_2xLTF_MU_RU26   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [37]   [0] = { 10'd485,  10'd499,  10'd474,  10'd484};

 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [1]    [0] = { -10'sd488,  -10'sd448,  -10'sd499,  -10'sd489};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [2]    [0] = { -10'sd434,  -10'sd394,  -10'sd445,  -10'sd435};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [3]    [0] = { -10'sd354,  -10'sd314,  -10'sd365,  -10'sd355};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [4]    [0] = { -10'sd300,  -10'sd260,  -10'sd311,  -10'sd301};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [5]    [0] = { -10'sd246,  -10'sd206,  -10'sd257,  -10'sd247};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [6]    [0] = { -10'sd192,  -10'sd152,  -10'sd203,  -10'sd193};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [7]    [0] = { -10'sd112,  -10'sd72,  -10'sd123,  -10'sd113};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [8]    [0] = { -10'sd58,  -10'sd18,  -10'sd69,  -10'sd59};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [9]    [0] = { 10'd29,  10'd69,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [10]   [0] = { 10'd83,  10'd123,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [11]   [0] = { 10'd163,  10'd203,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [12]   [0] = { 10'd217,  10'd257,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [13]   [0] = { 10'd271,  10'd311,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [14]   [0] = { 10'd325,  10'd365,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [15]   [0] = { 10'd405,  10'd445,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU52   [POST_CORDIC]    [MODE_BW_IS_80MHZ] [16]   [0] = { 10'd459,  10'd499,  10'd448,  10'd458};

 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [1]    [0] = { -10'sd488,  -10'sd394,  -10'sd499,  -10'sd489};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [2]    [0] = { -10'sd354,  -10'sd260,  -10'sd365,  -10'sd355};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [3]    [0] = { -10'sd246,  -10'sd152,  -10'sd257,  -10'sd247};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [4]    [0] = { -10'sd112,  -10'sd18,  -10'sd123,  -10'sd113};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [5]    [0] = { 10'd29,  10'd123,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [6]    [0] = { 10'd163,  10'd257,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [7]    [0] = { 10'd271,  10'd365,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU106  [POST_CORDIC]    [MODE_BW_IS_80MHZ] [8]    [0] = { 10'd405,  10'd499,  10'd394,  10'd404};

// FOR IDLE_STEP
//                                         Mode_BW    RUIndex PilotCount
//                                             |          |     |
 Index__L_LTF             [IDLE_STEP]                              = { 10'd0,  10'd26,  -10'sd26,  -10'sd1};
 Index__HE_LSIG           [IDLE_STEP]                              = { 10'd27,  10'd28,  -10'sd28,  -10'sd27};

 Index__VHT_LTF           [IDLE_STEP] [MODE_BW_IS_20MHZ]           = { 10'd0,  10'd28,  -10'sd28,  -10'sd1};
 Index__HT_LTF            [IDLE_STEP] [MODE_BW_IS_20MHZ]           = { 10'd0,  10'd28,  -10'sd28,  -10'sd1};

 Index__HE_4xLTF_SU       [IDLE_STEP] [MODE_BW_IS_20MHZ]           = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};
 Index__HE_2xLTF_SU       [IDLE_STEP] [MODE_BW_IS_20MHZ]       [0] = { 10'd0,  10'd122,  -10'sd122,  -10'sd2};
 Index__HE_1xLTF          [IDLE_STEP] [MODE_BW_IS_20MHZ] [0]   [0] = { 10'd0,  10'd120,  -10'sd120,  -10'sd4};

 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]       = { -10'sd110,  -10'sd96,  -10'sd121,  -10'sd111};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [2]       = { -10'sd84,  -10'sd70,  -10'sd95,  -10'sd85};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [3]       = { -10'sd57,  -10'sd43,  -10'sd68,  -10'sd58};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [4]       = { -10'sd31,  -10'sd17,  -10'sd42,  -10'sd32};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [5]       = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [6]       = { 10'd28,  10'd42,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [7]       = { 10'd54,  10'd68,  10'd43,  10'd53};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [8]       = { 10'd81,  10'd95,  10'd70,  10'd80};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [9]       = { 10'd107,  10'd121,  10'd96,  10'd106};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]       = { -10'sd110,  -10'sd70,  -10'sd121,  -10'sd111};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [2]       = { -10'sd57,  -10'sd17,  -10'sd68,  -10'sd58};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [3]       = { 10'd28,  10'd68,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [4]       = { 10'd81,  10'd121,  10'd70,  10'd80};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]       = { -10'sd111,  -10'sd17,  -10'sd122,  -10'sd112};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_20MHZ] [2]       = { 10'd28,  10'd122,  10'd17,  10'd27};
 Index__HE_4xLTF_MU_RU242 [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]       = { 10'd0,  10'd122,  -10'sd122,  -10'sd1};

 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd96,  -10'sd120,  -10'sd111};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd84,  -10'sd70,  -10'sd94,  -10'sd85};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [3]   [0] = { -10'sd57,  -10'sd44,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [4]   [0] = { -10'sd31,  -10'sd18,  -10'sd42,  -10'sd32};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [5]   [0] = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [6]   [0] = { 10'd28,  10'd42,  10'd18,  10'd27};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [7]   [0] = { 10'd54,  10'd68,  10'd44,  10'd53};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [8]   [0] = { 10'd81,  10'd94,  10'd70,  10'd80};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_20MHZ] [9]   [0] = { 10'd107,  10'd120,  10'd96,  10'd106};

 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd110,  -10'sd70,  -10'sd120,  -10'sd111};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [2]   [0] = { -10'sd57,  -10'sd18,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [3]   [0] = { 10'd28,  10'd68,  10'd18,  10'd27};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_20MHZ] [4]   [0] = { 10'd81,  10'd120,  10'd70,  10'd80};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]   [0] = { -10'sd111,  -10'sd18,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_20MHZ] [2]   [0] = { 10'd28,  10'd122,  10'd18,  10'd27};
 Index__HE_2xLTF_MU_RU242 [IDLE_STEP] [MODE_BW_IS_20MHZ] [1]   [0] = { 10'd0,  10'd122,  -10'sd122,  -10'sd2};

 Index__VHT_LTF           [IDLE_STEP] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd58,  -10'sd58,  -10'sd1};
 Index__HT_LTF            [IDLE_STEP] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd58,  -10'sd58,  -10'sd1};

 Index__HE_4xLTF_SU       [IDLE_STEP] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};
 Index__HE_2xLTF_SU       [IDLE_STEP] [MODE_BW_IS_40MHZ]       [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd2};
 Index__HE_1xLTF          [IDLE_STEP] [MODE_BW_IS_40MHZ] [0]   [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd4};

 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]       = { -10'sd232,  -10'sd218,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]       = { -10'sd206,  -10'sd192,  -10'sd217,  -10'sd207};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [3]       = { -10'sd178,  -10'sd164,  -10'sd189,  -10'sd179};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [4]       = { -10'sd152,  -10'sd138,  -10'sd163,  -10'sd153};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [5]       = { -10'sd125,  -10'sd111,  -10'sd136,  -10'sd126};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [6]       = { -10'sd98,  -10'sd84,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [7]       = { -10'sd72,  -10'sd58,  -10'sd83,  -10'sd73};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [8]       = { -10'sd44,  -10'sd30,  -10'sd55,  -10'sd45};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [9]       = { -10'sd18,  -10'sd4,  -10'sd29,  -10'sd19};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [10]      = { 10'd15,  10'd29,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [11]      = { 10'd41,  10'd55,  10'd30,  10'd40};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [12]      = { 10'd69,  10'd83,  10'd58,  10'd68};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [13]      = { 10'd95,  10'd109,  10'd84,  10'd94};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [14]      = { 10'd122,  10'd136,  10'd111,  10'd121};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [15]      = { 10'd149,  10'd163,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [16]      = { 10'd175,  10'd189,  10'd164,  10'd174};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [17]      = { 10'd203,  10'd217,  10'd192,  10'd202};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [18]      = { 10'd229,  10'd243,  10'd218,  10'd228};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]       = { -10'sd232,  -10'sd192,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]       = { -10'sd178,  -10'sd138,  -10'sd189,  -10'sd179};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [3]       = { -10'sd98,  -10'sd58,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [4]       = { -10'sd44,  -10'sd4,  -10'sd55,  -10'sd45};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [5]       = { 10'd15,  10'd55,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [6]       = { 10'd69,  10'd109,  10'd58,  10'd68};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [7]       = { 10'd149,  10'd189,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [8]       = { 10'd203,  10'd243,  10'd192,  10'd202};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]       = { -10'sd232,  -10'sd138,  -10'sd243,  -10'sd233};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]       = { -10'sd98,  -10'sd4,  -10'sd109,  -10'sd99};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [3]       = { 10'd15,  10'd109,  10'd4,  10'd14};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [4]       = { 10'd149,  10'd243,  10'd138,  10'd148};
 Index__HE_4xLTF_MU_RU242 [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]       = { -10'sd233,  -10'sd3,  -10'sd244,  -10'sd234};
 Index__HE_4xLTF_MU_RU242 [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]       = { 10'd14,  10'd244,  10'd3,  10'd13};
 Index__HE_4xLTF_MU_RU484 [IDLE_STEP] [MODE_BW_IS_40MHZ]           = { 10'd0,  10'd244,  -10'sd244,  -10'sd1};

 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd230,  -10'sd218,  -10'sd242,  -10'sd232};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd204,  -10'sd192,  -10'sd216,  -10'sd206};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd176,  -10'sd164,  -10'sd188,  -10'sd178};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd150,  -10'sd138,  -10'sd162,  -10'sd152};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [5]   [0] = { -10'sd124,  -10'sd112,  -10'sd136,  -10'sd126};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [6]   [0] = { -10'sd96,  -10'sd84,  -10'sd108,  -10'sd98};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [7]   [0] = { -10'sd70,  -10'sd58,  -10'sd82,  -10'sd72};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [8]   [0] = { -10'sd42,  -10'sd30,  -10'sd54,  -10'sd44};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [9]   [0] = { -10'sd16,  -10'sd4,  -10'sd28,  -10'sd18};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [10]  [0] = { 10'd16,  10'd28,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [11]  [0] = { 10'd42,  10'd54,  10'd30,  10'd40};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [12]  [0] = { 10'd70,  10'd82,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [13]  [0] = { 10'd96,  10'd108,  10'd84,  10'd94};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [14]  [0] = { 10'd124,  10'd136,  10'd112,  10'd122};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [15]  [0] = { 10'd150,  10'd162,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [16]  [0] = { 10'd176,  10'd188,  10'd164,  10'd174};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [17]  [0] = { 10'd204,  10'd216,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_40MHZ] [18]  [0] = { 10'd230,  10'd242,  10'd218,  10'd228};
 
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd230,  -10'sd192,  -10'sd242,  -10'sd232};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd176,  -10'sd138,  -10'sd188,  -10'sd178};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [3]   [0] = { -10'sd96,  -10'sd58,  -10'sd108,  -10'sd98};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [4]   [0] = { -10'sd42,  -10'sd4,  -10'sd54,  -10'sd44};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [5]   [0] = { 10'd16,  10'd54,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [6]   [0] = { 10'd70,  10'd108,  10'd58,  10'd68};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [7]   [0] = { 10'd150,  10'd188,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_40MHZ] [8]   [0] = { 10'd204,  10'd242,  10'd192,  10'd202};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd230,  -10'sd138,  -10'sd242,  -10'sd232};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]   [0] = { -10'sd96,  -10'sd4,  -10'sd108,  -10'sd98};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [3]   [0] = { 10'd16,  10'd108,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_40MHZ] [4]   [0] = { 10'd150,  10'd242,  10'd138,  10'd148};
 Index__HE_2xLTF_MU_RU242 [IDLE_STEP] [MODE_BW_IS_40MHZ] [1]   [0] = { -10'sd232,  -10'd4,  -10'sd244,  -10'sd234};
 Index__HE_2xLTF_MU_RU242 [IDLE_STEP] [MODE_BW_IS_40MHZ] [2]   [0] = { 10'd16,  10'd244,  10'd4,  10'd14};
 Index__HE_2xLTF_MU_RU484 [IDLE_STEP]                          [0] = { 10'd0,  10'd244,  -10'sd244,  -10'sd2};

 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [1]       = { -10'sd487,  -10'sd474,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [2]       = { -10'sd461,  -10'sd448,  -10'sd473,  -10'sd463};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [3]       = { -10'sd433,  -10'sd420,  -10'sd445,  -10'sd435};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [4]       = { -10'sd407,  -10'sd394,  -10'sd419,  -10'sd409};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [5]       = { -10'sd380,  -10'sd367,  -10'sd392,  -10'sd382};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [6]       = { -10'sd353,  -10'sd340,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [7]       = { -10'sd327,  -10'sd314,  -10'sd339,  -10'sd329};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [8]       = { -10'sd299,  -10'sd286,  -10'sd311,  -10'sd301};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [9]       = { -10'sd273,  -10'sd260,  -10'sd285,  -10'sd275};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [10]      = { -10'sd245,  -10'sd232,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [11]      = { -10'sd219,  -10'sd206,  -10'sd231,  -10'sd221};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [12]      = { -10'sd191,  -10'sd178,  -10'sd203,  -10'sd193};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [13]      = { -10'sd165,  -10'sd152,  -10'sd177,  -10'sd167};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [14]      = { -10'sd138,  -10'sd125,  -10'sd150,  -10'sd140};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [15]      = { -10'sd111,  -10'sd98,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [16]      = { -10'sd85,  -10'sd72,  -10'sd97,  -10'sd87};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [17]      = { -10'sd57,  -10'sd44,  -10'sd69,  -10'sd59};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [18]      = { -10'sd31,  -10'sd18,  -10'sd43,  -10'sd33};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [19]      = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [20]      = { 10'd30,  10'd43,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [21]      = { 10'd56,  10'd69,  10'd44,  10'd54};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [22]      = { 10'd84,  10'd97,  10'd72,  10'd82};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [23]      = { 10'd110,  10'd123,  10'd98,  10'd108};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [24]      = { 10'd137,  10'd150,  10'd125,  10'd135};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [25]      = { 10'd164,  10'd177,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [26]      = { 10'd190,  10'd203,  10'd178,  10'd188};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [27]      = { 10'd218,  10'd231,  10'd206,  10'd216};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [28]      = { 10'd244,  10'd257,  10'd232,  10'd242};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [29]      = { 10'd272,  10'd285,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [30]      = { 10'd298,  10'd311,  10'd286,  10'd296};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [31]      = { 10'd326,  10'd339,  10'd314,  10'd324};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [32]      = { 10'd352,  10'd365,  10'd340,  10'd350};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [33]      = { 10'd379,  10'd392,  10'd367,  10'd377};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [34]      = { 10'd406,  10'd419,  10'd394,  10'd404};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [35]      = { 10'd432,  10'd445,  10'd420,  10'd430};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [36]      = { 10'd460,  10'd473,  10'd448,  10'd458};
 Index__HE_4xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [37]      = { 10'd486,  10'd499,  10'd474,  10'd484};

 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [1]       = { -10'sd487,  -10'sd448,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [2]       = { -10'sd433,  -10'sd394,  -10'sd445,  -10'sd435};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [3]       = { -10'sd353,  -10'sd314,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [4]       = { -10'sd299,  -10'sd260,  -10'sd311,  -10'sd301};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [5]       = { -10'sd245,  -10'sd206,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [6]       = { -10'sd191,  -10'sd152,  -10'sd203,  -10'sd193};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [7]       = { -10'sd111,  -10'sd72,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [8]       = { -10'sd57,  -10'sd18,  -10'sd69,  -10'sd59};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [9]       = { 10'd30,  10'd69,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [10]      = { 10'd84,  10'd123,  10'd72,  10'd82};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [11]      = { 10'd164,  10'd203,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [12]      = { 10'd218,  10'd257,  10'd206,  10'd216};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [13]      = { 10'd272,  10'd311,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [14]      = { 10'd326,  10'd365,  10'd314,  10'd324};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [15]      = { 10'd406,  10'd445,  10'd394,  10'd404};
 Index__HE_4xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [16]      = { 10'd460,  10'd499,  10'd448,  10'd458};

 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [1]       = { -10'sd487,  -10'sd394,  -10'sd499,  -10'sd489};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [2]       = { -10'sd353,  -10'sd260,  -10'sd365,  -10'sd355};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [3]       = { -10'sd245,  -10'sd152,  -10'sd257,  -10'sd247};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [4]       = { -10'sd111,  -10'sd18,  -10'sd123,  -10'sd113};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [5]       = { 10'd30,  10'd123,  10'd18,  10'd28};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [6]       = { 10'd164,  10'd257,  10'd152,  10'd162};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [7]       = { 10'd272,  10'd365,  10'd260,  10'd270};
 Index__HE_4xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [8]       = { 10'd406,  10'd499,  10'd394,  10'd404};

 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd474,  -10'sd498,  -10'sd488};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd460,  -10'sd448,  -10'sd472,  -10'sd462};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd432,  -10'sd420,  -10'sd444,  -10'sd434};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd406,  -10'sd394,  -10'sd418,  -10'sd408};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd380,  -10'sd368,  -10'sd392,  -10'sd382};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd352,  -10'sd340,  -10'sd364,  -10'sd354};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd326,  -10'sd314,  -10'sd338,  -10'sd328};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd298,  -10'sd286,  -10'sd310,  -10'sd300};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [9]   [0] = { -10'sd272,  -10'sd260,  -10'sd284,  -10'sd274};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [10]  [0] = { -10'sd244,  -10'sd232,  -10'sd256,  -10'sd246};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [11]  [0] = { -10'sd218,  -10'sd206,  -10'sd230,  -10'sd220};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [12]  [0] = { -10'sd190,  -10'sd178,  -10'sd202,  -10'sd192};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [13]  [0] = { -10'sd164,  -10'sd152,  -10'sd176,  -10'sd166};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [14]  [0] = { -10'sd138,  -10'sd126,  -10'sd150,  -10'sd140};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [15]  [0] = { -10'sd110,  -10'sd98,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [16]  [0] = { -10'sd84,  -10'sd72,  -10'sd96,  -10'sd86};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [17]  [0] = { -10'sd56,  -10'sd44,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [18]  [0] = { -10'sd30,  -10'sd18,  -10'sd42,  -10'sd32};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [19]  [0] = { 10'd4,  10'd16,  -10'sd16,  -10'sd4};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [20]  [0] = { 10'd30,  10'd42,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [21]  [0] = { 10'd56,  10'd68,  10'd44,  10'd54};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [22]  [0] = { 10'd84,  10'd96,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [23]  [0] = { 10'd110,  10'd122,  10'd98,  10'd108};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [24]  [0] = { 10'd136,  10'd150,  10'd124,  10'd134};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [25]  [0] = { 10'd164,  10'd176,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [26]  [0] = { 10'd190,  10'd202,  10'd178,  10'd188};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [27]  [0] = { 10'd218,  10'd230,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [28]  [0] = { 10'd244,  10'd256,  10'd232,  10'd242};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [29]  [0] = { 10'd272,  10'd284,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [30]  [0] = { 10'd298,  10'd310,  10'd286,  10'd296};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [31]  [0] = { 10'd326,  10'd338,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [32]  [0] = { 10'd352,  10'd364,  10'd340,  10'd350};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [33]  [0] = { 10'd378,  10'd392,  10'd366,  10'd376};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [34]  [0] = { 10'd406,  10'd418,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [35]  [0] = { 10'd432,  10'd444,  10'd420,  10'd430};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [36]  [0] = { 10'd460,  10'd472,  10'd448,  10'd458};
 Index__HE_2xLTF_MU_RU26  [IDLE_STEP] [MODE_BW_IS_80MHZ] [37]  [0] = { 10'd486,  10'd498,  10'd474,  10'd484};
 
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd448,  -10'sd498,  -10'sd488};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd432,  -10'sd394,  -10'sd444,  -10'sd434};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd352,  -10'sd314,  -10'sd364,  -10'sd354};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd298,  -10'sd260,  -10'sd310,  -10'sd300};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [5]   [0] = { -10'sd244,  -10'sd206,  -10'sd256,  -10'sd246};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [6]   [0] = { -10'sd190,  -10'sd152,  -10'sd202,  -10'sd192};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [7]   [0] = { -10'sd110,  -10'sd72,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [8]   [0] = { -10'sd56,  -10'sd18,  -10'sd68,  -10'sd58};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [9]   [0] = { 10'd30,  10'd68,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [10]  [0] = { 10'd84,  10'd122,  10'd72,  10'd82};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [11]  [0] = { 10'd164,  10'd202,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [12]  [0] = { 10'd218,  10'd256,  10'd206,  10'd216};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [13]  [0] = { 10'd272,  10'd310,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [14]  [0] = { 10'd326,  10'd364,  10'd314,  10'd324};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [15]  [0] = { 10'd406,  10'd444,  10'd394,  10'd404};
 Index__HE_2xLTF_MU_RU52  [IDLE_STEP] [MODE_BW_IS_80MHZ] [16]  [0] = { 10'd460,  10'd498,  10'd448,  10'd458};

 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [1]   [0] = { -10'sd486,  -10'sd394,  -10'sd498,  -10'sd488};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [2]   [0] = { -10'sd352,  -10'sd260,  -10'sd364,  -10'sd354};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [3]   [0] = { -10'sd244,  -10'sd152,  -10'sd256,  -10'sd246};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [4]   [0] = { -10'sd110,  -10'sd18,  -10'sd122,  -10'sd112};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [5]   [0] = { 10'd30,  10'd122,  10'd18,  10'd28};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [6]   [0] = { 10'd164,  10'd256,  10'd152,  10'd162};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [7]   [0] = { 10'd272,  10'd364,  10'd260,  10'd270};
 Index__HE_2xLTF_MU_RU106 [IDLE_STEP] [MODE_BW_IS_80MHZ] [8]   [0] = { 10'd406,  10'd498,  10'd394,  10'd404};

end


// Symbol Counter for LTFs
// 00 = 1; 01 = 1; 10 = 3; 11 = 4
always @ (posedge PhyClk or negedge nPhyRst)
   begin: SymbolCount_Blk
      if (nPhyRst == 1'b0) begin
         SymbolCount <= 2'd0;
         LTFIndex    <= 3'd0;
      end else if (EstimEn == 1'b0) begin
         SymbolCount <= 2'd0;
         LTFIndex    <= 3'd0;
      end else if (StartSignalP == 1'b1) begin
         if (FirstLTF == 1'b1) begin
            SymbolCount <= 2'd0;
            if (FrameNDP)
              LTFIndex  <= 3'd0;
            else begin
              case (NShiftLTF) // always process only last VHT/HE-LTF
                default : LTFIndex  <= 3'd0; // 1 VHT/HE-LTF or other symbol type (HTLTF...)
                4'd2    : LTFIndex  <= 3'd1; // 2 VHT/HE-LTF - process VHT/HE-LTF 2
                4'd3    : LTFIndex  <= 3'd2; // 4 VHT/HE-LTF - process VHT/HE-LTF 3 (stbc case)
                4'd4    : LTFIndex  <= 3'd3; // 4 VHT/HE-LTF - process VHT/HE-LTF 4
                4'd5    : LTFIndex  <= 3'd4; // 6 VHT/HE-LTF - process VHT/HE-LTF 5 (stbc case)
                4'd6    : LTFIndex  <= 3'd5; // 6 VHT/HE-LTF - process VHT/HE-LTF 6
                4'd7    : LTFIndex  <= 3'd6; // 8 VHT/HE-LTF - process VHT/HE-LTF 7 (stbc case)
                4'd8    : LTFIndex  <= 3'd7; // 8 VHT/HE-LTF - process VHT/HE-LTF 8
              endcase
            end
         end
         else begin
            SymbolCount <= SymbolCount + 2'd1;
            LTFIndex    <= LTFIndex + 3'd1;
         end
      end
   end //SymbolCount_Blk

//H Data Multiplication Factor
always @ (*)
   begin: HDataMult_Blk
      case (SymbolCount)
         2'd0   : HDataMult = 2'd0;
         2'd1   : HDataMult = 2'd0;
         2'd2   : HDataMult = 2'd1;
         default: HDataMult = 2'd2;
      endcase //SymbolCount
   end //HDataMult_Blk

//Round LSB Factor
always @ (*)
   begin: RoundLSB_Blk
      case (SymbolCount)
         2'd0   : RoundLSB = 2'd0;
         2'd1   : RoundLSB = 2'd1;
         2'd2   : RoundLSB = 2'd2;
         default: RoundLSB = 2'd2;
      endcase //SymbolCount
   end //RoundLSB_Blk

// PtCoeff Computation
// Warning! Values for NShiftLTF=6 apply only to Nss=1!
always @ (*)
   begin: PtCoeff_Blk
      case (LTFIndex)
         3'd0   : PtCoeff = 4'b1000;
         3'd1   : PtCoeff = 4'b0001;
         3'd2   : PtCoeff = 4'b0010;
         3'd3   : PtCoeff = 4'b0100;
         3'd4   : PtCoeff = 4'b1000;
         3'd5   : PtCoeff = 4'b0001;
         3'd6   : PtCoeff = 4'b0010;
         default: PtCoeff = 4'b0100;
      endcase //LTFIndex
   end //PtCoeff_Blk

// Pilots Counter
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PilotCount_Blk
      if (nPhyRst == 1'b0)
         PilotCount <= 4'd0;
      else if ((SmoothStepCS != PILOT_I_ST)
`ifdef RW_NX_DERIV_CHBW20ONLY
       && (SmoothStepCS != PILOT_Q_ST)
`endif
       )
         PilotCount <= 4'b0;
      else if ((AddrGenCS == DONE) && HWriteQ)
      begin
`ifdef RW_NX_DERIV_CHBW20ONLY
         if (PilotCount < 4'd7)
`endif
           PilotCount <= PilotCount + 4'd1;
      end
   end //PilotCount_Blk

// FFT & H Memory Read Address generation
// Multiplex Indexes depending on Reception Mode and Smoothing Feature
always @ (*)
begin: Index_Sel_Blk

   case (SymbolType_1t)
     VHT_LTF: begin
        PilotCountSel                 = 4'b0;
        {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__VHT_LTF[SmoothStep][ReceptionMode_BW_1t];
        {VirtualIndex,  CoeffVirtual} = 20'b0;
        {PilotIndex,    CoeffPilot}   = 20'b0;
     end
     HT_LTF: begin
        PilotCountSel                 = 4'b0;
        {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HT_LTF[SmoothStep][ReceptionMode_BW_1t];
        {VirtualIndex,  CoeffVirtual} = 20'b0;
        {PilotIndex,    CoeffPilot}   = 20'b0;
     end
     HE_1xLTF: begin
        PilotCountSel                 = (ReceptionMode_BW_1t==MODE_BW_IS_20MHZ) ? {1'b0,PilotCount[2:0]} : PilotCount;
        {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_1xLTF[SmoothStep][ReceptionMode_BW_1t][RUIndexSel][PilotCountSel]; // If issue with pilotcount size, can be reduced to 8 or 4 depending on BW
        {VirtualIndex,  CoeffVirtual} = IndexVirt__HE_1xLTF[RUIndexSel]; // If issue with RU Index sel size, can be reduced to 1-bit
        {PilotIndex,    CoeffPilot}   = IndexPilot__HE_1xLTF[ReceptionMode_BW_1t][PilotCountSel];
     end
     HE_2xLTF: begin
        if ((ReceptionMode_1t == MODE_20MHZ_HE_MU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU))
        begin
          case (RUTypeSel)
            RU52: 
              begin
                 PilotCountSel                 = {2'b0,PilotCount[1:0]};
                 {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_2xLTF_MU_RU52[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel]; // If issue with pilotcount size, can be resized
                 {VirtualIndex,  CoeffVirtual} = IndexVirt__HE_2xLTF_RU52 [ReceptionMode_BW_1t][RUIndex_1t];
                 {PilotIndex,    CoeffPilot}   = IndexPilot__HE_2xLTF_RU52[ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel];
              end
            RU106:
              begin
                 PilotCountSel                 = {2'b0,PilotCount[1:0]};
                 {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_2xLTF_MU_RU106[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel];// If issue with pilotcount size, can be resized
                 {VirtualIndex,  CoeffVirtual} = IndexVirt__HE_2xLTF_RU106 [ReceptionMode_BW_1t][RUIndex_1t];
                 {PilotIndex,    CoeffPilot}   = IndexPilot__HE_2xLTF_RU106[ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel];
              end
            RU242:
              begin 
                 PilotCountSel                 = {1'b0,PilotCount[2:0]};
                 {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_2xLTF_MU_RU242[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel];// If issue with pilotcount size, can be resized
                 {VirtualIndex,  CoeffVirtual} = IndexVirt__HE_2xLTF_RU242 [ReceptionMode_BW_1t][RUIndex_1t];
                 {PilotIndex,    CoeffPilot}   = IndexPilot__HE_2xLTF_RU242[ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel];
              end
`ifndef RW_NX_DERIV_CHBW20ONLY
            RU484:
              begin
                 PilotCountSel                 = PilotCount;
                 {VirtualIndex,  CoeffVirtual} = 20'b0;
                 {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_2xLTF_MU_RU484[SmoothStep][PilotCount];// If issue with pilotcount size, can be resized
                 {PilotIndex,    CoeffPilot}   = IndexPilot__HE_2xLTF_RU484[PilotCount];
              end
`endif
            default: //RU26
              begin
                 PilotCountSel                 = {3'b0,PilotCount[0]};
                 {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_2xLTF_MU_RU26[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel]; // If issue with pilotcount size, can be resized
                 {VirtualIndex,  CoeffVirtual} = IndexVirt__HE_2xLTF_RU26 [ReceptionMode_BW_1t][RUIndex_1t];
                 {PilotIndex,    CoeffPilot}   = IndexPilot__HE_2xLTF_RU26[ReceptionMode_BW_1t][RUIndex_1t][PilotCountSel];
              end
          endcase
        end
        else
        begin
          PilotCountSel                 = (ReceptionMode_BW_1t==MODE_BW_IS_20MHZ) ? {1'b0,PilotCount[2:0]} : PilotCount;
          {VirtualIndex,  CoeffVirtual} = 20'b0;
          {PilotIndex,    CoeffPilot}   = IndexPilot__HE_2xLTF_SU[ReceptionMode_BW_1t][PilotCountSel];
          {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_2xLTF_SU[SmoothStep][ReceptionMode_BW_1t][PilotCountSel];
        end
     end
     HE_4xLTF: begin 
        if ((ReceptionMode_1t == MODE_20MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_SU))
          {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_4xLTF_SU[SmoothStep][ReceptionMode_BW_1t];
        else begin
          case (RUTypeSel)
            RU52: {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_4xLTF_MU_RU52[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t];
            RU106:{FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_4xLTF_MU_RU106[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t];
            RU242:{FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_4xLTF_MU_RU242[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t];
`ifndef RW_NX_DERIV_CHBW20ONLY
            RU484:{FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_4xLTF_MU_RU484[SmoothStep][ReceptionMode_BW_1t];
`endif
            default:{FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_4xLTF_MU_RU26[SmoothStep][ReceptionMode_BW_1t][RUIndex_1t]; // RU26
          endcase
        end
        {VirtualIndex,  CoeffVirtual} = 20'b0;
        {PilotIndex,    CoeffPilot}   = 20'b0;
        PilotCountSel                 = 4'b0;
     end
     HE_LSIG: begin
        {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__HE_LSIG[SmoothStep];
        {VirtualIndex,  CoeffVirtual} = 20'b0;
        {PilotIndex,    CoeffPilot}   = 20'b0;
        PilotCountSel                 = 4'b0;
     end
     default : begin // L_LTF
        {FirstPosIndex, LastPosIndex, FirstNegIndex, LastNegIndex} = Index__L_LTF[SmoothStep];
        {VirtualIndex,  CoeffVirtual} = 20'b0;
        {PilotIndex,    CoeffPilot}   = 20'b0;
        PilotCountSel                 = 4'b0;
     end
   endcase
end

assign RUTypeSel     = ((SmoothStep == VIRTUAL_INTERP) && (ReceptionMode_1t == MODE_20MHZ_HE_SU)) ? 3'b0               : RUType_1t;
assign RUIndexSel    = ((SmoothStep == VIRTUAL_INTERP) && (ReceptionMode_1t == MODE_20MHZ_HE_SU)) ? {5'b0,VirtualHigh} : RUIndex_1t;

// Address generator FSM
// Sequential Block
always @ (posedge PhyClk or negedge nPhyRst)
   begin: AddrGen_Seq
      if (nPhyRst == 1'b0)
         AddrGenCS <= IDLE;
      else if (EstimEn == 1'b0)
         AddrGenCS <= IDLE;
      else
         AddrGenCS <= AddrGenNS;
   end //AddrGen_Seq

// Combinatorial Block
always @ (*)
   begin: AddrGen_Combo
      // Default state
      AddrGenNS = IDLE;
      
      case(AddrGenCS)

         IDLE: begin
            if (AddrGenStartP)
               AddrGenNS = COUNTNEG1;
            else
               AddrGenNS = IDLE;
         end //IDLE

         COUNTNEG1: begin
            if ((SmoothStep == DC_INTERP) || (SmoothStep == VIRTUAL_INTERP) || (SmoothStep == PILOT_INTERP))
            begin
               if ((SymbolType_1t == HE_4xLTF) ||
                (ReceptionMode_1t == MODE_40MHZ_HT)    ||
               ((ReceptionMode_1t == MODE_40MHZ_HE_SU) && (SymbolType_1t == HE_2xLTF) && (SmoothStep != PILOT_INTERP)) ||
               ((ReceptionMode_1t == MODE_40MHZ_HE_MU) && (SymbolType_1t == HE_2xLTF) && (RUType_1t == RU484) && (SmoothStep == DC_INTERP)) ||
                 (SpecialRU26Idx5 && (SmoothStep != PILOT_INTERP)))
                 AddrGenNS = DC_MODE;
               else
                 AddrGenNS = COUNTNEG2;
            end
            else if ($signed(RdAddr) == $signed(FirstNegIndex_1t + RdAddrOffset)) begin
               if (SymbolType_1t == HE_LSIG)
                  AddrGenNS = COUNTPOS1;
               else if (SmoothBypass || (SmoothStep == CHAN_INTERP))
                  AddrGenNS = COUNTNEG2;
               else
                  AddrGenNS = WAITFIRST;
            end
            else
               AddrGenNS = COUNTNEG1;
         end //COUNTNEG1

         WAITFIRST: begin
            if (WaitCount == 4'b0) begin
               if ((RUType_1t == RU26) && (ReceptionMode_1t == MODE_20MHZ_HE_MU || ReceptionMode_1t == MODE_40MHZ_HE_MU || ReceptionMode_1t == MODE_80MHZ_HE_MU) && (SymbolType_1t != HE_4xLTF))
                 AddrGenNS = COUNTPOS2;
               else
                 AddrGenNS = COUNTNEG2;
            end
            else
               AddrGenNS = WAITFIRST;
         end //WAITFIRST

         COUNTNEG2: begin
            if ((RUType_1t == RU26) && (((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUIndex_1t != 6'd5)) || ReceptionMode_1t == MODE_40MHZ_HE_MU || ReceptionMode_1t == MODE_80MHZ_HE_MU) && (SymbolType_1t != HE_4xLTF) && !SmoothBypass)
               AddrGenNS = COUNTPOS2;
            else if ($signed(RdAddr) >= $signed(LastNegIndex_1t - RdAddrInc))
               AddrGenNS = COUNTPOS1;
            else
               AddrGenNS = COUNTNEG2;
         end //COUNTNEG2

         COUNTPOS1: begin
            if ((SmoothStep_1t == DC_INTERP) || (SmoothStep_1t == VIRTUAL_INTERP) || (SmoothStep_1t == PILOT_INTERP))
               AddrGenNS = COUNTPOS2;
            else if ((SmoothStep_1t == CHAN_INTERP) && (SymbolType_1t == HE_1xLTF)) begin
              if (($signed(RdAddr) == $signed(FirstPosIndex_1t)) && (ChanInterpCount == 2'b0))
                AddrGenNS = COUNTPOS2;
              else
                AddrGenNS = COUNTPOS1;
            end   
            else if ($signed(RdAddr) >= $signed(FirstPosIndex_1t + RdAddrOffset))
               AddrGenNS = COUNTPOS2;
            else
               AddrGenNS = COUNTPOS1;
         end //COUNTPOS1

         COUNTPOS2: begin
            if ((SmoothStep_1t == CHAN_INTERP) && (SymbolType_1t == HE_1xLTF)) begin
              if (($signed(RdAddr) == $signed(LastPosIndex_1t)) && (ChanInterpCount == 2'b0))
                AddrGenNS = WAITLASTWR;
              else
                AddrGenNS = COUNTPOS2;
            end
            else if ($signed(RdAddr) >= $signed(LastPosIndex_1t - RdAddrInc)) begin
               if (!PreRotBypass || !PostRotBypass)
                  AddrGenNS = EMPTYCORDIC;
               else
                  AddrGenNS = WAITLASTWR;
            end
            else
               AddrGenNS = COUNTPOS2;
         end //COUNTPOS2

         EMPTYCORDIC: begin
            if (WaitCount == 4'b0)
               AddrGenNS = WAITLASTWR;
            else
               AddrGenNS = EMPTYCORDIC;
         end //EMPTYCORDIC

         DC_MODE: begin
            if (WaitCount == 4'b0)
               AddrGenNS = WAITLASTWR;
            else
               AddrGenNS = DC_MODE;
         end //DC_MODE

         WAITLASTWR: begin
            if (LastWriteP == 1'b1)
               AddrGenNS = DONE;
            else
               AddrGenNS = WAITLASTWR;
         end //WAITLASTWR

         default: begin
            AddrGenNS = IDLE;
         end //DONE

      endcase //AddrGenCS
   end //AddrGen_Combo

// Smooth steps FSM
// Sequential Block
always @ (posedge PhyClk or negedge nPhyRst)
   begin: SmoothStep_Seq
      if (nPhyRst == 1'b0) begin
         SmoothStepCS  <= IDLE_ST;
         AddrGenStartP <= 1'b0;
         ReadFromH     <= 1'b0;
         SelIQPhase    <= 1'b0;
      end
      else if (EstimEn == 1'b0) begin
         SmoothStepCS  <= IDLE_ST;
         AddrGenStartP <= 1'b0;
         ReadFromH     <= 1'b0;
         SelIQPhase    <= 1'b0;
      end
      else begin
        // Pulse generation
        AddrGenStartP <= 1'b0;

        // Register I/Q selection for filter
        SelIQPhase    <= ~HWriteI;
        
        case(SmoothStepCS)

           IDLE_ST: begin
              ReadFromH     <= 1'b0;
              if (StartSignalP) begin
                 SmoothStepCS  <= CH_ESTIM_ST;
                 AddrGenStartP <= 1'b1;
              end
              else if (SmoothfromHP)
              begin
                 SmoothStepCS  <= PRE_CORDIC_ST;
                 AddrGenStartP <= (SymbolType_1t!=L_LTF);
              end
              else
                 SmoothStepCS <= IDLE_ST;
           end //IDLE_ST

           CH_ESTIM_ST: begin
              ReadFromH     <= 1'b0;
              if (AddrGenCS == DONE) begin
                 if (SmoothEn == 1'b1) begin
                    //Smoothing Enabled
                    if (((ReceptionMode_1t != MODE_20MHZ_HE_MU) && (ReceptionMode_1t != MODE_40MHZ_HE_MU) && (ReceptionMode_1t != MODE_80MHZ_HE_MU))    || 
                        ((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (SpecialRU26Idx5 || (RUType_1t == RU242))) || 
                        ((ReceptionMode_1t == MODE_40MHZ_HE_MU) && (RUType_1t == RU484))) begin
                      if ((RxNsts != 2'b0) && (SymbolType_1t!=HE_4xLTF) && (SymbolType_1t!=VHT_LTF) && (SymbolType_1t!=HT_LTF))
                        SmoothStepCS  <= PILOT_I_ST;
                      else
                        SmoothStepCS  <= DC_I_ST;
                    end
                    else if (SymbolType_1t!=HE_4xLTF) begin
                      if (RxNsts != 2'b0)
                        SmoothStepCS  <= PILOT_I_ST;
                      else
                        SmoothStepCS  <= VIRTUAL_I_ST;
                    end
                    else
                      SmoothStepCS  <= SMOOTH_I_ST;

                    AddrGenStartP <= 1'b1;
                 end
`ifdef RW_MUMIMO_RX_EN
                 else if (HEMuMIMOEn) begin
                    SmoothStepCS  <= PRE_CORDIC_ST;
                    AddrGenStartP <= 1'b1;
                 end
`endif // RW_MUMIMO_RX_EN
                 
                 else
                    //Smoothing Disabled
                    SmoothStepCS <= END_ST;
              end
              else
                 SmoothStepCS <= CH_ESTIM_ST;
           end //CH_ESTIM_ST

           PRE_CORDIC_ST: begin
              ReadFromH     <= 1'b1;
              if (AddrGenCS == DONE || (SymbolType_1t==L_LTF)) begin
                 if (((ReceptionMode_1t != MODE_20MHZ_HE_MU) && (ReceptionMode_1t != MODE_40MHZ_HE_MU) && (ReceptionMode_1t != MODE_80MHZ_HE_MU)) || SpecialRU26Idx5 || (RUType_1t == RU242)) begin
                    if ((RxNsts != 2'b0) && (SymbolType_1t!=HE_4xLTF))
                      SmoothStepCS  <= PILOT_I_ST;
                    else
                      SmoothStepCS  <= DC_I_ST;
                 end
                 else if (SymbolType_1t!=HE_4xLTF) begin
                    if (RxNsts != 2'b0)
                      SmoothStepCS  <= PILOT_I_ST;
                    else if (RUType_1t != RU484)
                      SmoothStepCS  <= VIRTUAL_I_ST;
                    else
                      SmoothStepCS  <= DC_I_ST;
                 end
`ifdef RW_MUMIMO_RX_EN
                 else if (HEMuMIMOEn == 1'b1)
                   SmoothStepCS <= POST_CORDIC_ST;
`endif // RW_MUMIMO_RX_EN
                 else
                   SmoothStepCS  <= SMOOTH_I_ST;
                 AddrGenStartP <= 1'b1;
              end
              else
                 SmoothStepCS <= PRE_CORDIC_ST;
           end //PRE_CORDIC_ST

           PILOT_I_ST: begin
              ReadFromH     <= 1'b1;
              if (AddrGenCS == DONE) begin
`ifdef RW_NX_DERIV_CHBW20ONLY
                 SmoothStepCS  <= PILOT_Q_ST;
`else
                 if ((((PilotCount==4'd7  && SymbolType_1t == HE_2xLTF) || (PilotCount==4'd3 && SymbolType_1t == HE_1xLTF)) && ReceptionMode == MODE_20MHZ_HE_SU) ||
                     (((PilotCount==4'd15 && SymbolType_1t == HE_2xLTF) || (PilotCount==4'd7 && SymbolType_1t == HE_1xLTF)) && ReceptionMode == MODE_40MHZ_HE_SU) ||
                       (PilotCount==4'd1  && RUType_1t == RU26 && (ReceptionMode == MODE_20MHZ_HE_MU || ReceptionMode == MODE_40MHZ_HE_MU || ReceptionMode == MODE_80MHZ_HE_MU)) ||
                       (PilotCount==4'd3  && RUType_1t == RU52)  ||
                       (PilotCount==4'd3  && RUType_1t == RU106) ||
                       (PilotCount==4'd7  && RUType_1t == RU242) ||
                       (PilotCount==4'd15 && RUType_1t == RU484))
                 begin
                   if (SmoothingDone)
                     SmoothStepCS <= CH_INTERP_I_ST;
                   else if ((ReceptionMode_1t != MODE_20MHZ_HE_MU && ReceptionMode_1t != MODE_40MHZ_HE_MU && ReceptionMode_1t != MODE_80MHZ_HE_MU) || SpecialRU26Idx5 || 
                            (RUType_1t == RU242 && ReceptionMode_1t == MODE_20MHZ_HE_MU) || 
                            (RUType_1t == RU484 && ReceptionMode_1t == MODE_40MHZ_HE_MU))
                     SmoothStepCS <= DC_I_ST;
                   else
                     SmoothStepCS <= VIRTUAL_I_ST;
                 end
                 else
                   SmoothStepCS <= PILOT_I_ST;
`endif
                 AddrGenStartP <= 1'b1;
              end
              else
                 SmoothStepCS <= PILOT_I_ST;
           end //PILOT_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
          PILOT_Q_ST: begin
              if (AddrGenCS == DONE) begin
                 if ((((PilotCount!=4'd7  && SymbolType_1t == HE_2xLTF) || (PilotCount!=4'd3 && SymbolType_1t == HE_1xLTF)) && ReceptionMode == MODE_20MHZ_HE_SU) ||
                     (((PilotCount!=4'd15 && SymbolType_1t == HE_2xLTF) || (PilotCount!=4'd7 && SymbolType_1t == HE_1xLTF)) && ReceptionMode == MODE_40MHZ_HE_SU) ||
                       (PilotCount!=4'd1  && RUType_1t == RU26 && (ReceptionMode == MODE_20MHZ_HE_MU || ReceptionMode == MODE_40MHZ_HE_MU || ReceptionMode == MODE_80MHZ_HE_MU))  ||
                       (PilotCount!=4'd3  && RUType_1t == RU52)  ||
                       (PilotCount!=4'd3  && RUType_1t == RU106) ||
                       (PilotCount!=4'd7  && RUType_1t == RU242) ||
                       (PilotCount!=4'd15 && RUType_1t == RU484))
                   SmoothStepCS <= PILOT_I_ST;
                 else if (SmoothingDone)
                   SmoothStepCS <= CH_INTERP_I_ST;
                 else if ((ReceptionMode_1t != MODE_20MHZ_HE_MU && ReceptionMode_1t != MODE_40MHZ_HE_MU && ReceptionMode_1t != MODE_80MHZ_HE_MU) || SpecialRU26Idx5 || (RUType_1t == RU242))
                   SmoothStepCS <= DC_I_ST;
                 else
                   SmoothStepCS <= VIRTUAL_I_ST;
                 AddrGenStartP <= 1'b1;
              end
              else
                 SmoothStepCS <= PILOT_Q_ST;
           end //PILOT_Q_ST
`endif

           DC_I_ST: begin
              ReadFromH     <= 1'b1;
              if (AddrGenCS == DONE) begin
`ifdef RW_NX_DERIV_CHBW20ONLY
                 SmoothStepCS  <= DC_Q_ST;
`else
                 if ((SymbolType_1t==HE_1xLTF) && (ReceptionMode_1t != MODE_40MHZ_HE_SU))
                   SmoothStepCS <= VIRTUAL_I_ST;
`ifdef RW_MUMIMO_RX_EN
                 else if (HEMuMIMOEn == 1'b1) begin
                   if (SymbolType_1t!=HE_4xLTF) begin
                     if (RUType_1t != RU484)
                       SmoothStepCS <= VIRTUAL_I_ST;
                     else
                       SmoothStepCS <= CH_INTERP_I_ST;
                   end
                   else
                     SmoothStepCS <= POST_CORDIC_ST;
                 end
`endif // RW_MUMIMO_RX_EN
                 else
                   SmoothStepCS <= SMOOTH_I_ST;
`endif
                 AddrGenStartP <= 1'b1;
              end
              else
                 SmoothStepCS <= DC_I_ST;
           end //DC_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
           DC_Q_ST: begin
              if (AddrGenCS == DONE) begin
                 if (SymbolType_1t==HE_1xLTF)
                   SmoothStepCS <= VIRTUAL_I_ST;
`ifdef RW_MUMIMO_RX_EN
                 else if (HEMuMIMOEn == 1'b1) begin
                   if (SymbolType_1t!=HE_4xLTF)
                     SmoothStepCS <= VIRTUAL_I_ST;
                   else
                     SmoothStepCS <= POST_CORDIC_ST;
                 end
`endif // RW_MUMIMO_RX_EN
                 else
                   SmoothStepCS <= SMOOTH_I_ST;
                 AddrGenStartP <= 1'b1;
              end
              else
                 SmoothStepCS <= DC_Q_ST;
           end //DC_Q_ST
`endif

           VIRTUAL_I_ST: begin
              ReadFromH     <= 1'b1;
              if (AddrGenCS == DONE) begin
`ifdef RW_NX_DERIV_CHBW20ONLY
                 SmoothStepCS  <= VIRTUAL_Q_ST;
`else
                 if (VirtualHigh || (ReceptionMode_1t == MODE_20MHZ_HE_MU)
                                 || (ReceptionMode_1t == MODE_40MHZ_HE_MU)
                                 || (ReceptionMode_1t == MODE_80MHZ_HE_MU)) begin
`ifdef RW_MUMIMO_RX_EN
                   if (HEMuMIMOEn == 1'b1)
                      SmoothStepCS <= CH_INTERP_I_ST;
                   else
`endif // RW_MUMIMO_RX_EN
                      SmoothStepCS  <= SMOOTH_I_ST;
                 end
                 else
                   SmoothStepCS  <= VIRTUAL_I_ST;
`endif
                 AddrGenStartP <= 1'b1;
              end
              else
                 SmoothStepCS <= VIRTUAL_I_ST;
           end //VIRTUAL_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
           VIRTUAL_Q_ST: begin
              if (AddrGenCS == DONE) begin
                 if (VirtualHigh || (ReceptionMode_1t == MODE_20MHZ_HE_MU)) begin
`ifdef RW_MUMIMO_RX_EN
                 if (HEMuMIMOEn == 1'b1)
                    SmoothStepCS <= CH_INTERP_I_ST;
                 else
`endif // RW_MUMIMO_RX_EN
                    SmoothStepCS <= SMOOTH_I_ST;
                 end
                 else
                   SmoothStepCS  <= VIRTUAL_I_ST;
                 AddrGenStartP <= 1'b1;
              end
              else
                 SmoothStepCS <= VIRTUAL_Q_ST;
           end //VIRTUAL_Q_ST
`endif

           SMOOTH_I_ST: begin
              ReadFromH     <= 1'b1;
              if (AddrGenCS == DONE) begin
`ifdef RW_NX_DERIV_CHBW20ONLY
                 SmoothStepCS  <= SMOOTH_Q_ST;
                 AddrGenStartP <= 1'b1;
`else
                 if (SymbolType_1t==HE_2xLTF || SymbolType_1t==HE_1xLTF) begin
                   if (RxNsts != 2'b0 && !FrameNDP)
                     SmoothStepCS <= PILOT_I_ST;
                   else
                     SmoothStepCS <= CH_INTERP_I_ST;
                   AddrGenStartP <= 1'b1;
                 end
                 else
                   SmoothStepCS  <= END_ST;
`endif
              end
              else
                 SmoothStepCS <= SMOOTH_I_ST;
           end //SMOOTH_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
           SMOOTH_Q_ST: begin
              if (AddrGenCS == DONE) begin
                 if (SymbolType_1t==HE_2xLTF || SymbolType_1t==HE_1xLTF) begin
                   if (RxNsts != 2'b0 && !FrameNDP)
                     SmoothStepCS <= PILOT_I_ST;
                   else
                     SmoothStepCS <= CH_INTERP_I_ST;
                   AddrGenStartP <= 1'b1;
                 end
                 else if (FrameNDP)
                   SmoothStepCS <= END_ST;
                 else begin
                   SmoothStepCS  <= POST_CORDIC_ST;
                   AddrGenStartP <= 1'b1;
                 end
              end
              else
                 SmoothStepCS <= SMOOTH_Q_ST;
           end //SMOOTH_Q_ST
`endif

           CH_INTERP_I_ST: begin
              if (AddrGenCS == DONE) begin
`ifdef RW_NX_DERIV_CHBW20ONLY
                 AddrGenStartP <= 1'b1;
                 SmoothStepCS  <= CH_INTERP_Q_ST;
`else
                 if (FrameNDP)
                   SmoothStepCS  <= END_ST;
                 else begin
                   AddrGenStartP <= 1'b1;
                   SmoothStepCS  <= POST_CORDIC_ST;
                 end
`endif
              end
              else
                 SmoothStepCS <= CH_INTERP_I_ST;
           end //CH_INTERP_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
           CH_INTERP_Q_ST: begin
              if (AddrGenCS == DONE) begin
                 if (FrameNDP)
                   SmoothStepCS  <= END_ST;
                 else begin
                   SmoothStepCS  <= POST_CORDIC_ST;
                   AddrGenStartP <= 1'b1;
                 end
              end
              else
                 SmoothStepCS <= CH_INTERP_Q_ST;
           end //CH_INTERP_Q_ST
`endif

           POST_CORDIC_ST: begin
              if (AddrGenCS == DONE)
                 SmoothStepCS <= END_ST;
              else
                 SmoothStepCS <= POST_CORDIC_ST;
           end //POST_CORDIC_ST

           default: begin
              SmoothStepCS <= IDLE_ST;
           end //END_ST

        endcase //SmoothStepCS
      end
   end //SmoothStep_Seq


// Control Block
// RdAddrValidSC, WaitCount, RdAddr, FilterOutValid
always @ (posedge PhyClk or negedge nPhyRst)
   begin: AddrCtrl_Blk
      if (nPhyRst == 1'b0) begin
         RdAddrValidSC     <= 1'b0;
         WaitCount         <= 4'b0;
         RdAddr            <= 10'b0;
         FFTBufRdAddrEn    <= 1'b0;
         HBufRdAddrEn      <= 1'b0;
         FilterOutValid    <= 1'b0;
      end
      else if (EstimEn == 1'b0) begin
         RdAddrValidSC     <= 1'b0;
         WaitCount         <= 4'b0;
         RdAddr            <= 10'b0;
         FFTBufRdAddrEn    <= 1'b0;
         HBufRdAddrEn      <= 1'b0;
         FilterOutValid    <= 1'b0;
      end
      else begin

         case (AddrGenCS)
            IDLE: begin
               RdAddrValidSC     <= 1'b0;
               WaitCount         <= 4'b0;
               RdAddr            <= 10'b0;
               FFTBufRdAddrEn    <= 1'b0;
               HBufRdAddrEn      <= 1'b0;
               FilterOutValid    <= 1'b0;
            end //IDLE

            COUNTNEG1: begin
               RdAddrValidSC <= 1'b1;
               WaitCount     <= 4'd4;
               if (RdAddrValidSC == 1'b0)
                  RdAddr <= FirstNegIndex;
               else 
                  RdAddr <= RdAddr + RdAddrInc;
               FFTBufRdAddrEn  <= ~ReadFromH;
               HBufRdAddrEn    <= ~FirstLTF | ReadFromH;
               FilterOutValid  <= 1'b1;
            end //COUNTNEG1

            WAITFIRST: begin
               RdAddrValidSC     <= 1'b0;
               WaitCount         <= WaitCount - 4'd1;
               FFTBufRdAddrEn    <= ~ReadFromH;
               HBufRdAddrEn      <= ~FirstLTF | ReadFromH;
               FilterOutValid    <= 1'b1;
            end //WAITFIRST

            COUNTNEG2: begin
               RdAddrValidSC     <= 1'b1;
               WaitCount         <= 4'd4;
               if ((SmoothStep == PILOT_INTERP) && ((RdAddr + RdAddrInc) == PilotIndex_1t))
                 RdAddr            <= LastNegIndex_1t;
               else
                 RdAddr            <= RdAddr + RdAddrInc;
               FFTBufRdAddrEn    <= ~ReadFromH;
               HBufRdAddrEn      <= ~FirstLTF | ReadFromH;
               FilterOutValid    <= 1'b1;
            end //COUNTNEG2

            COUNTPOS1: begin
               RdAddrValidSC <= 1'b1;
               WaitCount     <= 4'd4;
               if ((RdAddr == LastNegIndex_1t) && (ChanInterpCount == 2'b0))
                  RdAddr <= FirstPosIndex_1t;
               else if (ChanInterpCount == 2'b0)            
                  RdAddr <= RdAddr + RdAddrInc;
               FFTBufRdAddrEn  <= ~ReadFromH;
               HBufRdAddrEn    <= ~FirstLTF | ReadFromH;
               FilterOutValid  <= 1'b1;
            end //COUNTPOS1

            COUNTPOS2: begin
               WaitCount <= (CORDICLAT_SIZE -4'b1);
               if (RdAddr != LastPosIndex_1t) begin
                  if ((SmoothStep == PILOT_INTERP) && ((RdAddr + RdAddrInc) == PilotIndex_1t))
                     RdAddr         <= LastPosIndex_1t;
                  else
                     RdAddr         <= RdAddr + RdAddrInc;
                  RdAddrValidSC     <= 1'b1;
                  FFTBufRdAddrEn    <= ~ReadFromH;
                  HBufRdAddrEn      <= ~FirstLTF | ReadFromH;
               end
               else if (ChanInterpCount == 2'b0) begin
                  RdAddr            <= 10'd0;
                  RdAddrValidSC     <= 1'b0;
                  FFTBufRdAddrEn    <= 1'b0;
                  HBufRdAddrEn      <= 1'b0;
               end
               FilterOutValid  <= 1'b1;
            end //COUNTPOS2

            EMPTYCORDIC: begin
               WaitCount         <= WaitCount - 4'd1;
               RdAddrValidSC     <= 1'b0;
               RdAddr            <= 10'b0;
               FFTBufRdAddrEn    <= 1'b0;
               HBufRdAddrEn      <= 1'b0;
               FilterOutValid    <= 1'b0;
            end //EMPTYCORDIC

            DC_MODE: begin
               WaitCount         <= 4'b0;
               if ((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t != RU242))
                 RdAddr          <= 10'd4;
               else if ((ReceptionMode_1t == MODE_40MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU)) begin
                 if (SymbolType_1t == HE_4xLTF)
                   RdAddr        <= 10'd3;
                 else
                   RdAddr        <= 10'd4;
               end
               else
                 RdAddr          <= 10'd2;
               FFTBufRdAddrEn    <= 1'b0;
               if (RdAddr == FirstNegIndex_1t) begin
                 RdAddrValidSC     <= 1'b1;
                 HBufRdAddrEn      <= 1'b1;
                 FilterOutValid    <= 1'b1;
               end
               else begin
                 RdAddrValidSC     <= 1'b0;
                 HBufRdAddrEn      <= 1'b0;
                 FilterOutValid    <= 1'b0;
               end
            end //DC_MODE

            WAITLASTWR: begin
               RdAddr            <= 10'b0;
               RdAddrValidSC     <= 1'b0;
               WaitCount         <= 4'b0;
               FFTBufRdAddrEn    <= 1'b0;
               HBufRdAddrEn      <= 1'b0;
               FilterOutValid    <= 1'b0;
            end //WAITLASTWR

            default: begin
            end //DONE

         endcase //AddrGenCS
      end
   end //AddrCtrl_Blk

// Variables for address counter
assign RdAddrOffset    = ((SymbolType_1t==HE_2xLTF) && ((ReceptionMode_1t == MODE_20MHZ_HE_MU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU)) && (SmoothStep_1t != SMOOTHING) && (SmoothBypass || (RUType_1t == RU26))) ? 10'd8 :
                         ((SymbolType_1t==HE_1xLTF) && PostRotBypass)                                                                                      ? 10'd40 :
                         ((SymbolType_1t==HE_2xLTF) && PostRotBypass)                                                                                      ? 10'd20 :
                         (SymbolType_1t==HE_LSIG)                                                                                                          ? 10'd0  :
                                                                                                                                                             10'd10;
assign RdAddrInc       = ((SymbolType_1t==HE_1xLTF) && (SmoothStep == CHAN_INTERP)) ? RdAddrIncOffset :
                         ((SymbolType_1t==HE_1xLTF) && PostRotBypass) ? 10'd4 :
                         ((SymbolType_1t==HE_2xLTF) && PostRotBypass) ? 10'd2 : 10'd1;
assign RdAddrIncOffset = (ChanInterpCount == 2'd0) ? 10'd4 : 10'd0;
//
assign WrAddrInc       = ((SymbolType_1t==HE_1xLTF) && (SmoothStep == CHAN_INTERP)) ? WrAddrIncOffset :
                         ((SymbolType_1t==HE_1xLTF) && PostRotBypass) ? 10'd4 :
                         ((SymbolType_1t==HE_2xLTF) && PostRotBypass) ? 10'd2 : 10'd1;
assign WrAddrIncOffset = (ChanInterpCount == 2'd0) ? 10'd2 : 10'd1;

// Counter for interpolation address selection
always @ (posedge PhyClk or negedge nPhyRst)
begin: ChanInterpCount_Blk
  if (nPhyRst == 1'b0)
     ChanInterpCount <= 2'd0;
  else if ((SmoothStep != CHAN_INTERP) || (SymbolType_1t!=HE_1xLTF))
     ChanInterpCount <= 2'd0;
  else if (ChanInterpCount == 2'd2)
     ChanInterpCount <= 2'd0;
  else if (RdAddrValidSC || SCSmoothFiltOut)
     ChanInterpCount <= ChanInterpCount + 2'd1;
  else
     ChanInterpCount <= 2'd0;
end // ChanInterpCount_Blk

// Generate ChanInterpValidMask for Channel interpolation
always @ (posedge PhyClk or negedge nPhyRst)
begin: ChanInterpValidMask_Blk
  if (nPhyRst == 1'b0)
    ChanInterpValidMask <= 1'b1;
  else if ((SmoothStep != CHAN_INTERP) || (SymbolType_1t!=HE_1xLTF))
    ChanInterpValidMask <= 1'b1;
  else if (ChanInterpCount == 2'b0 || ~RdAddrValidSC)
    ChanInterpValidMask <= 1'b1;
  else
    ChanInterpValidMask <= 1'b0;
end

// Control Block
// SmoothStep, XXXBypass, HWriteXX
always @ (posedge PhyClk or negedge nPhyRst)
begin: StepCtrl_Blk
  if (nPhyRst == 1'b0) begin
     SmoothStep      <= IDLE_STEP;
     PreRotBypass    <= 1'b1;
     SmoothBypass    <= 1'b1;
     PostRotBypass   <= 1'b1;
     CounterSkip     <= 1'b0;
     EstimDoneP      <= 1'b0;
     SmthDoneP       <= 1'b0;
     HWriteI         <= 1'b1;
     HWriteQ         <= 1'b1;
     SmoothingDone   <= 1'b0;
     SpecialRU26Idx5 <= 1'b0;
  end
  else begin
    // Default value
    SmoothStep    <= IDLE_STEP;
    PreRotBypass  <= 1'b1;
    SmoothBypass  <= 1'b1;
    PostRotBypass <= 1'b1;
    CounterSkip   <= 1'b0;
    EstimDoneP    <= 1'b0;
    SmthDoneP     <= 1'b0;
    HWriteI       <= 1'b1;
    HWriteQ       <= 1'b1;
  
    case (SmoothStepCS)

       IDLE_ST: begin
          SmoothStep      <= IDLE_STEP;
          PreRotBypass    <= 1'b1;
          SmoothBypass    <= 1'b1;
          PostRotBypass   <= 1'b1;
          CounterSkip     <= 1'b0;
          EstimDoneP      <= 1'b0;
          SmthDoneP       <= 1'b0;
          HWriteI         <= 1'b1;
          HWriteQ         <= 1'b1;
          SmoothingDone   <= 1'b0;
          SpecialRU26Idx5 <= (RUType == RU26) & (RUIndex == 6'd5) & (ReceptionMode == MODE_20MHZ_HE_MU);
       end // IDLE_ST

       DC_I_ST: begin
          SmoothStep    <= DC_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          CounterSkip   <= (SymbolType_1t!=HE_4xLTF) & 
                           (ReceptionMode_1t!=MODE_40MHZ_HT) &
                         !((ReceptionMode_1t==MODE_40MHZ_HE_SU) && (SymbolType_1t==HE_2xLTF)) &
                         !((ReceptionMode_1t==MODE_40MHZ_HE_MU) && (SymbolType_1t==HE_2xLTF)) &
                         !((ReceptionMode_1t==MODE_80MHZ_HE_MU) && (SymbolType_1t==HE_2xLTF)) &
                            !((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t == RU26));
          HWriteI       <= 1'b1;
`ifdef RW_NX_DERIV_CHBW20ONLY
          HWriteQ       <= 1'b0;
`else
          HWriteQ       <= 1'b1;
`endif
       end // DC_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
       DC_Q_ST: begin
          SmoothStep    <= DC_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          CounterSkip   <= (SymbolType_1t!=HE_4xLTF) & !((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t == RU26));
          HWriteI       <= 1'b0;
          HWriteQ       <= 1'b1;
       end // DC_Q_ST 
`endif

       PILOT_I_ST: begin
          SmoothStep    <= PILOT_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          CounterSkip   <= 1'b1;
          HWriteI       <= 1'b1;
`ifdef RW_NX_DERIV_CHBW20ONLY
          HWriteQ       <= 1'b0;
`else
          HWriteQ       <= 1'b1;
`endif
       end //PILOT_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
       PILOT_Q_ST: begin
          SmoothStep    <= PILOT_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          CounterSkip   <= 1'b1;
          HWriteI       <= 1'b0;
          HWriteQ       <= 1'b1;
       end //PILOT_Q_ST
`endif

       VIRTUAL_I_ST: begin
          SmoothStep    <= VIRTUAL_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          CounterSkip   <= 1'b1;
          HWriteI       <= 1'b1;
`ifdef RW_NX_DERIV_CHBW20ONLY
          HWriteQ       <= 1'b0;
`else
          HWriteQ       <= 1'b1;
`endif
       end // VIRTUAL_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
       VIRTUAL_Q_ST: begin
          SmoothStep    <= VIRTUAL_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          CounterSkip   <= 1'b1;
          HWriteI       <= 1'b0;
          HWriteQ       <= 1'b1;
       end // VIRTUAL_Q_ST
`endif

       SMOOTH_I_ST: begin
          SmoothStep    <= SMOOTHING;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          HWriteI       <= 1'b1;
`ifdef RW_NX_DERIV_CHBW20ONLY
          HWriteQ       <= 1'b0;
          PostRotBypass <= 1'b1;
`else
          HWriteQ       <= 1'b1;
          PostRotBypass <= (SymbolType_1t==HE_2xLTF || SymbolType_1t==HE_1xLTF) ? 1'b1 : FrameNDP;
`endif
          SmoothingDone <= 1'b1;
       end // SMOOTH_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
       SMOOTH_Q_ST: begin
          SmoothStep    <= SMOOTHING;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          HWriteI       <= 1'b0;
          HWriteQ       <= 1'b1;
       end //SMOOTH_Q_ST
`endif

       CH_INTERP_I_ST: begin
          SmoothStep    <= CHAN_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          HWriteI       <= 1'b1;
`ifdef RW_NX_DERIV_CHBW20ONLY
          HWriteQ       <= 1'b0;
`else
          HWriteQ       <= 1'b1;
`endif
       end // CH_INTERP_I_ST

`ifdef RW_NX_DERIV_CHBW20ONLY
       CH_INTERP_Q_ST: begin
          SmoothStep    <= CHAN_INTERP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b0;
          PostRotBypass <= 1'b1;
          HWriteI       <= 1'b0;
          HWriteQ       <= 1'b1;
       end // CH_INTERP_Q_ST
`endif

       POST_CORDIC_ST: begin
          SmoothStep    <= POST_CORDIC;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b1;
          PostRotBypass <= 1'b0;
          HWriteI       <= 1'b1;
          HWriteQ       <= 1'b1;
       end // POST_CORDIC_ST

       END_ST: begin
          SmoothStep    <= IDLE_STEP;
          PreRotBypass  <= 1'b1;
          SmoothBypass  <= 1'b1;
          PostRotBypass <= 1'b1;
          SmthDoneP     <= 1'b1;
          HWriteI       <= 1'b1;
          HWriteQ       <= 1'b1;
       end // END_ST

       PRE_CORDIC_ST: begin
          SmoothStep    <= IDLE_STEP;
          PreRotBypass  <= 1'b0;
          SmoothBypass  <= 1'b1;
          PostRotBypass <= 1'b1;
          HWriteI       <= 1'b1;
          HWriteQ       <= 1'b1;
       end // PRE_CORDIC_ST

       default: begin
          SmoothStep    <= IDLE_STEP;
          if (SymbolType_1t==L_LTF)
            PreRotBypass  <= 1'b0;
          else
            PreRotBypass  <= ~SmoothEn;
          SmoothBypass  <= 1'b1;
          PostRotBypass <= 1'b1;
          HWriteI       <= 1'b1;
          HWriteQ       <= 1'b1;
          if ((AddrGenCS == DONE) && (SmoothStepCS == CH_ESTIM_ST))
            EstimDoneP  <= 1'b1;
       end // default: CH_ESTIM_ST

    endcase
  end
end //StepCtrl_Blk

// Generate VirtualHigh for Virtual interpolation I/Q loop
always @ (posedge PhyClk or negedge nPhyRst)
   begin: VirtualHigh_Blk
      if (nPhyRst == 1'b0)
        VirtualHigh <= 1'b0;
      else if (EstimEn == 1'b0)
        VirtualHigh <= 1'b0;
//       else if ((ReceptionMode_1t == MODE_20MHZ_HE_MU) && ((RUType_1t == RU106) && (RUIndex_1t == 6'd2)) ||
//                                                          ((RUType_1t == RU52)  && (RUIndex_1t  > 6'd2)) ||
//                                                          ((RUType_1t == RU26)  && (RUIndex_1t  > 6'd5)))
      else if ((ReceptionMode_1t == MODE_20MHZ_HE_MU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) ||
               (ReceptionMode_1t == MODE_80MHZ_HE_MU))
        VirtualHigh <= 1'b1;
`ifdef RW_NX_DERIV_CHBW20ONLY
      else if ((SmoothStepCS == VIRTUAL_Q_ST) && (AddrGenCS == DONE))
`else
      else if ((SmoothStepCS == VIRTUAL_I_ST) && (AddrGenCS == DONE))
`endif
        VirtualHigh <= ~VirtualHigh;
   end

// FFT memory read
assign FFTBufRdAddrEnOut = FFTBufRdAddrEn;

assign RdAddrMux = (SymbolType_1t==HE_1xLTF) ? {{2{RdAddr[9]}},RdAddr[9:2]} : // Divide addr by 4 for HE_1xLTF
                   (SymbolType_1t==HE_2xLTF) ? {RdAddr[9],RdAddr[9:1]}      : // Divide addr by 2 for HE_2xLTF
                                               RdAddr;

assign FFTBufRdAddrOut   = (FFTBufRdAddrEn) ? RdAddrMux : 10'd0;

// H memory read
assign HBufRdAddrEnOut   = HBufRdAddrEn;
assign HBufRdAddrOut     = (HBufRdAddrEn) ? RdAddr : 10'd0;

// Delayed signal
always @ (posedge PhyClk or negedge nPhyRst)
  begin: RdAddr_delay_Blk
     if (nPhyRst == 1'b0)
     begin
        RdAddr_1t         <= 10'd0;
        RdAddr_2t         <= 10'd0;
        RdAddrValidSC_1t  <= 1'b0;
        RdAddrValidSC_2t  <= 1'b0;
        FilterOutValid_1t <= 1'b0;
        FilterOutValid_2t <= 1'b0;
        FilterOutValid_3t <= 1'b0;
        SCCordicEn_1t     <= 1'b0;
     end
     else if (EstimEn == 1'b0)
     begin
        RdAddr_1t         <= 10'd0;
        RdAddr_2t         <= 10'd0;
        RdAddrValidSC_1t  <= 1'b0;
        RdAddrValidSC_2t  <= 1'b0;
        FilterOutValid_1t <= 1'b0;
        FilterOutValid_2t <= 1'b0;
        FilterOutValid_3t <= 1'b0;
        SCCordicEn_1t     <= 1'b0;
     end
     else
     begin
        RdAddr_1t         <= RdAddr;
        RdAddr_2t         <= RdAddr_1t;
        RdAddrValidSC_1t  <= RdAddrValidSC & ChanInterpValidMask;
        RdAddrValidSC_2t  <= RdAddrValidSC_1t;
        FilterOutValid_1t <= FilterOutValid;
        FilterOutValid_2t <= FilterOutValid_1t;
        FilterOutValid_3t <= FilterOutValid_2t;
        SCCordicEn_1t     <= SCCordicEn;
     end
  end //RdAddr_delay_Blk

// Generate control to CordicIndQdComp
assign LUTAddr       = (PreRotBypass) ? RdAddr2s_2t : RdAddr_2t[5:0];
assign LUTEnable     = ~PreRotBypass | ~PostRotBypass;
assign LUTEnableMask = LUTEnable
`ifdef RW_MUMIMO_RX_EN
                                 & ~HEMuMIMOEn
`endif // RW_MUMIMO_RX_EN
;
assign LUTSel        = (SymbolType_1t==HE_4xLTF) |
                       (SymbolType_1t==HE_2xLTF) |
                       (SymbolType_1t==HE_1xLTF);

assign RotDataEnIn  = SCCordicEn & LUTEnable;
assign RotDataEnOut = (RotDataEnOutInt | SCCordicEn_1t) & LUTEnable;

// 2's complement of RdAddr_2t for Post Cordic
assign RdAddr2s_2t = ~RdAddrSel[5:0] + 6'b1;

assign RdAddrSel = (!PostRotBypass && !SmoothBypass) ? SCIndSmoothFiltPreOut[5:0] :
                                                       RdAddr_2t[5:0];

// Cordic Index and Quadrant
CordicIndQdComp U_CordicIndQdC (
  // Inputs
  .nPhyRst( nPhyRst),
  .PhyClk(  PhyClk),
  .LUTAddr( LUTAddr),
  .Enable(  LUTEnableMask),
  .Sel(     LUTSel),
  .Index0(  RotIndex0),
  .Index1(  RotIndex1),
  .Index2(  RotIndex2),
  .Index3(  RotIndex3),
  .Quad0(   RotQuad0),
  .Quad1(   RotQuad1),
  .Quad2(   RotQuad2),
  .Quad3(   RotQuad3)
  );

// Preamble Generation
always @ (posedge PhyClk or negedge nPhyRst)
  begin: Preamble_Blk
    if (nPhyRst == 1'b0) begin
       PreambleCoeffInt    <= 1'b0;
       PreambleCoeffInt_1t <= 1'b0;
    end
    else if (EstimEn == 1'b0) begin
       PreambleCoeffInt    <= 1'b0;
       PreambleCoeffInt_1t <= 1'b0;
    end
    else begin
      PreambleCoeffInt_1t <= PreambleCoeffInt;

      if (ReceptionMode_1t == MODE_40MHZ_HT) begin
         //40 MHz Preamble
         //Encoding -1 = 1; 1 = 0;
         case (RdAddr)
            -10'sd57 : PreambleCoeffInt <= 1'b0; // -57
            -10'sd56 : PreambleCoeffInt <= 1'b1;
            -10'sd55 : PreambleCoeffInt <= 1'b1;
            -10'sd54 : PreambleCoeffInt <= 1'b0;
            -10'sd53 : PreambleCoeffInt <= 1'b0;
            -10'sd52 : PreambleCoeffInt <= 1'b1;
            -10'sd51 : PreambleCoeffInt <= 1'b0;
            -10'sd50 : PreambleCoeffInt <= 1'b1;
            -10'sd49 : PreambleCoeffInt <= 1'b0;
            -10'sd48 : PreambleCoeffInt <= 1'b0;
            -10'sd47 : PreambleCoeffInt <= 1'b0;
            -10'sd46 : PreambleCoeffInt <= 1'b0;
            -10'sd45 : PreambleCoeffInt <= 1'b0;
            -10'sd44 : PreambleCoeffInt <= 1'b0;
            -10'sd43 : PreambleCoeffInt <= 1'b1;
            -10'sd42 : PreambleCoeffInt <= 1'b1;
            -10'sd41 : PreambleCoeffInt <= 1'b0;
            -10'sd40 : PreambleCoeffInt <= 1'b0;
            -10'sd39 : PreambleCoeffInt <= 1'b1;
            -10'sd38 : PreambleCoeffInt <= 1'b0;
            -10'sd37 : PreambleCoeffInt <= 1'b1;
            -10'sd36 : PreambleCoeffInt <= 1'b0;
            -10'sd35 : PreambleCoeffInt <= 1'b0;
            -10'sd34 : PreambleCoeffInt <= 1'b0;
            -10'sd33 : PreambleCoeffInt <= 1'b0;
            -10'sd32 : PreambleCoeffInt <= 1'b0;
            -10'sd31 : PreambleCoeffInt <= 1'b0;
            -10'sd30 : PreambleCoeffInt <= 1'b1;
            -10'sd29 : PreambleCoeffInt <= 1'b1;
            -10'sd28 : PreambleCoeffInt <= 1'b0;
            -10'sd27 : PreambleCoeffInt <= 1'b0;
            -10'sd26 : PreambleCoeffInt <= 1'b1;
            -10'sd25 : PreambleCoeffInt <= 1'b0;
            -10'sd24 : PreambleCoeffInt <= 1'b1;
            -10'sd23 : PreambleCoeffInt <= 1'b0;
            -10'sd22 : PreambleCoeffInt <= 1'b1;
            -10'sd21 : PreambleCoeffInt <= 1'b1;
            -10'sd20 : PreambleCoeffInt <= 1'b1;
            -10'sd19 : PreambleCoeffInt <= 1'b1;
            -10'sd18 : PreambleCoeffInt <= 1'b1;
            -10'sd17 : PreambleCoeffInt <= 1'b0;
            -10'sd16 : PreambleCoeffInt <= 1'b0;
            -10'sd15 : PreambleCoeffInt <= 1'b1;
            -10'sd14 : PreambleCoeffInt <= 1'b1;
            -10'sd13 : PreambleCoeffInt <= 1'b0;
            -10'sd12 : PreambleCoeffInt <= 1'b1;
            -10'sd11 : PreambleCoeffInt <= 1'b0;
            -10'sd10 : PreambleCoeffInt <= 1'b1;
            -10'sd9  : PreambleCoeffInt <= 1'b0;
            -10'sd8  : PreambleCoeffInt <= 1'b0;
            -10'sd7  : PreambleCoeffInt <= 1'b0;
            -10'sd6  : PreambleCoeffInt <= 1'b0;
            -10'sd5  : PreambleCoeffInt <= 1'b1;
            -10'sd4  : PreambleCoeffInt <= 1'b1;
            -10'sd3  : PreambleCoeffInt <= 1'b1;
            -10'sd2  : PreambleCoeffInt <= 1'b0; // -2
            10'd2    : PreambleCoeffInt <= 1'b1; // +2
            10'd3    : PreambleCoeffInt <= 1'b0;
            10'd4    : PreambleCoeffInt <= 1'b0;
            10'd5    : PreambleCoeffInt <= 1'b1;
            10'd6    : PreambleCoeffInt <= 1'b0;
            10'd7    : PreambleCoeffInt <= 1'b0;
            10'd8    : PreambleCoeffInt <= 1'b1;
            10'd9    : PreambleCoeffInt <= 1'b1;
            10'd10   : PreambleCoeffInt <= 1'b0;
            10'd11   : PreambleCoeffInt <= 1'b0;
            10'd12   : PreambleCoeffInt <= 1'b1;
            10'd13   : PreambleCoeffInt <= 1'b0;
            10'd14   : PreambleCoeffInt <= 1'b1;
            10'd15   : PreambleCoeffInt <= 1'b0;
            10'd16   : PreambleCoeffInt <= 1'b0;
            10'd17   : PreambleCoeffInt <= 1'b0;
            10'd18   : PreambleCoeffInt <= 1'b0;
            10'd19   : PreambleCoeffInt <= 1'b0;
            10'd20   : PreambleCoeffInt <= 1'b0;
            10'd21   : PreambleCoeffInt <= 1'b1;
            10'd22   : PreambleCoeffInt <= 1'b1;
            10'd23   : PreambleCoeffInt <= 1'b0;
            10'd24   : PreambleCoeffInt <= 1'b0;
            10'd25   : PreambleCoeffInt <= 1'b1;
            10'd26   : PreambleCoeffInt <= 1'b0;
            10'd27   : PreambleCoeffInt <= 1'b1;
            10'd28   : PreambleCoeffInt <= 1'b0;
            10'd29   : PreambleCoeffInt <= 1'b0;
            10'd30   : PreambleCoeffInt <= 1'b0;
            10'd31   : PreambleCoeffInt <= 1'b0;
            10'd32   : PreambleCoeffInt <= 1'b0;
            10'd33   : PreambleCoeffInt <= 1'b0;
            10'd34   : PreambleCoeffInt <= 1'b1;
            10'd35   : PreambleCoeffInt <= 1'b1;
            10'd36   : PreambleCoeffInt <= 1'b0;
            10'd37   : PreambleCoeffInt <= 1'b0;
            10'd38   : PreambleCoeffInt <= 1'b1;
            10'd39   : PreambleCoeffInt <= 1'b0;
            10'd40   : PreambleCoeffInt <= 1'b1;
            10'd41   : PreambleCoeffInt <= 1'b0;
            10'd42   : PreambleCoeffInt <= 1'b1;
            10'd43   : PreambleCoeffInt <= 1'b1;
            10'd44   : PreambleCoeffInt <= 1'b1;
            10'd45   : PreambleCoeffInt <= 1'b1;
            10'd46   : PreambleCoeffInt <= 1'b1;
            10'd47   : PreambleCoeffInt <= 1'b0;
            10'd48   : PreambleCoeffInt <= 1'b0;
            10'd49   : PreambleCoeffInt <= 1'b1;
            10'd50   : PreambleCoeffInt <= 1'b1;
            10'd51   : PreambleCoeffInt <= 1'b0;
            10'd52   : PreambleCoeffInt <= 1'b1;
            10'd53   : PreambleCoeffInt <= 1'b0;
            10'd54   : PreambleCoeffInt <= 1'b1;
            10'd55   : PreambleCoeffInt <= 1'b0;
            10'd56   : PreambleCoeffInt <= 1'b0;
            10'd57   : PreambleCoeffInt <= 1'b0; // +57
            default : PreambleCoeffInt <= 1'b0; // Null
         endcase // RdAddr
      end // 40 MHz HT Preamble
      else if ((ReceptionMode_1t == MODE_40MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU)) begin
        //40 MHz HE 4x Preamble
        if (SymbolType_1t==HE_4xLTF)
        begin
          //Encoding -1 = 1; 1 = 0;
          case (RdAddr)
             -10'sd244 : PreambleCoeffInt <= 1'b0;  
             -10'sd243 : PreambleCoeffInt <= 1'b1;  
             -10'sd242 : PreambleCoeffInt <= 1'b1;  
             -10'sd241 : PreambleCoeffInt <= 1'b1;  
             -10'sd240 : PreambleCoeffInt <= 1'b1;  
             -10'sd239 : PreambleCoeffInt <= 1'b0;  
             -10'sd238 : PreambleCoeffInt <= 1'b1;  
             -10'sd237 : PreambleCoeffInt <= 1'b1;  
             -10'sd236 : PreambleCoeffInt <= 1'b0;  
             -10'sd235 : PreambleCoeffInt <= 1'b0;  
             -10'sd234 : PreambleCoeffInt <= 1'b1;  
             -10'sd233 : PreambleCoeffInt <= 1'b0;  
             -10'sd232 : PreambleCoeffInt <= 1'b1;  
             -10'sd231 : PreambleCoeffInt <= 1'b0;  
             -10'sd230 : PreambleCoeffInt <= 1'b1;  
             -10'sd229 : PreambleCoeffInt <= 1'b0;  
             -10'sd228 : PreambleCoeffInt <= 1'b0;  
             -10'sd227 : PreambleCoeffInt <= 1'b1;  
             -10'sd226 : PreambleCoeffInt <= 1'b0;  
             -10'sd225 : PreambleCoeffInt <= 1'b1;  
             -10'sd224 : PreambleCoeffInt <= 1'b1;  
             -10'sd223 : PreambleCoeffInt <= 1'b1;  
             -10'sd222 : PreambleCoeffInt <= 1'b0;  
             -10'sd221 : PreambleCoeffInt <= 1'b0;  
             -10'sd220 : PreambleCoeffInt <= 1'b1;  
             -10'sd219 : PreambleCoeffInt <= 1'b1;  
             -10'sd218 : PreambleCoeffInt <= 1'b1;  
             -10'sd217 : PreambleCoeffInt <= 1'b1;  
             -10'sd216 : PreambleCoeffInt <= 1'b1;  
             -10'sd215 : PreambleCoeffInt <= 1'b1;  
             -10'sd214 : PreambleCoeffInt <= 1'b1;  
             -10'sd213 : PreambleCoeffInt <= 1'b0;  
             -10'sd212 : PreambleCoeffInt <= 1'b1;  
             -10'sd211 : PreambleCoeffInt <= 1'b1;  
             -10'sd210 : PreambleCoeffInt <= 1'b0;  
             -10'sd209 : PreambleCoeffInt <= 1'b0;  
             -10'sd208 : PreambleCoeffInt <= 1'b1;  
             -10'sd207 : PreambleCoeffInt <= 1'b0;  
             -10'sd206 : PreambleCoeffInt <= 1'b1;  
             -10'sd205 : PreambleCoeffInt <= 1'b1;  
             -10'sd204 : PreambleCoeffInt <= 1'b1;  
             -10'sd203 : PreambleCoeffInt <= 1'b1;  
             -10'sd202 : PreambleCoeffInt <= 1'b1;  
             -10'sd201 : PreambleCoeffInt <= 1'b0;  
             -10'sd200 : PreambleCoeffInt <= 1'b1;  
             -10'sd199 : PreambleCoeffInt <= 1'b0;  
             -10'sd198 : PreambleCoeffInt <= 1'b0;  
             -10'sd197 : PreambleCoeffInt <= 1'b0;  
             -10'sd196 : PreambleCoeffInt <= 1'b1;  
             -10'sd195 : PreambleCoeffInt <= 1'b1;  
             -10'sd194 : PreambleCoeffInt <= 1'b0;  
             -10'sd193 : PreambleCoeffInt <= 1'b0;  
             -10'sd192 : PreambleCoeffInt <= 1'b0;  
             -10'sd191 : PreambleCoeffInt <= 1'b1;  
             -10'sd190 : PreambleCoeffInt <= 1'b1;  
             -10'sd189 : PreambleCoeffInt <= 1'b0;  
             -10'sd188 : PreambleCoeffInt <= 1'b0;  
             -10'sd187 : PreambleCoeffInt <= 1'b0;  
             -10'sd186 : PreambleCoeffInt <= 1'b0;  
             -10'sd185 : PreambleCoeffInt <= 1'b1;  
             -10'sd184 : PreambleCoeffInt <= 1'b0;  
             -10'sd183 : PreambleCoeffInt <= 1'b0;  
             -10'sd182 : PreambleCoeffInt <= 1'b1;  
             -10'sd181 : PreambleCoeffInt <= 1'b1;  
             -10'sd180 : PreambleCoeffInt <= 1'b0;  
             -10'sd179 : PreambleCoeffInt <= 1'b1;  
             -10'sd178 : PreambleCoeffInt <= 1'b0;  
             -10'sd177 : PreambleCoeffInt <= 1'b1;  
             -10'sd176 : PreambleCoeffInt <= 1'b0;  
             -10'sd175 : PreambleCoeffInt <= 1'b1;  
             -10'sd174 : PreambleCoeffInt <= 1'b1;  
             -10'sd173 : PreambleCoeffInt <= 1'b0;  
             -10'sd172 : PreambleCoeffInt <= 1'b1;  
             -10'sd171 : PreambleCoeffInt <= 1'b0;  
             -10'sd170 : PreambleCoeffInt <= 1'b0;  
             -10'sd169 : PreambleCoeffInt <= 1'b0;  
             -10'sd168 : PreambleCoeffInt <= 1'b1;  
             -10'sd167 : PreambleCoeffInt <= 1'b1;  
             -10'sd166 : PreambleCoeffInt <= 1'b0;  
             -10'sd165 : PreambleCoeffInt <= 1'b0;  
             -10'sd164 : PreambleCoeffInt <= 1'b0;  
             -10'sd163 : PreambleCoeffInt <= 1'b1;  
             -10'sd162 : PreambleCoeffInt <= 1'b1;  
             -10'sd161 : PreambleCoeffInt <= 1'b1;  
             -10'sd160 : PreambleCoeffInt <= 1'b1;  
             -10'sd159 : PreambleCoeffInt <= 1'b0;  
             -10'sd158 : PreambleCoeffInt <= 1'b1;  
             -10'sd157 : PreambleCoeffInt <= 1'b1;  
             -10'sd156 : PreambleCoeffInt <= 1'b0;  
             -10'sd155 : PreambleCoeffInt <= 1'b0;  
             -10'sd154 : PreambleCoeffInt <= 1'b1;  
             -10'sd153 : PreambleCoeffInt <= 1'b0;  
             -10'sd152 : PreambleCoeffInt <= 1'b1;  
             -10'sd151 : PreambleCoeffInt <= 1'b1;  
             -10'sd150 : PreambleCoeffInt <= 1'b1;  
             -10'sd149 : PreambleCoeffInt <= 1'b1;  
             -10'sd148 : PreambleCoeffInt <= 1'b1;  
             -10'sd147 : PreambleCoeffInt <= 1'b0;  
             -10'sd146 : PreambleCoeffInt <= 1'b1;  
             -10'sd145 : PreambleCoeffInt <= 1'b0;  
             -10'sd144 : PreambleCoeffInt <= 1'b0;  
             -10'sd143 : PreambleCoeffInt <= 1'b0;  
             -10'sd142 : PreambleCoeffInt <= 1'b1;  
             -10'sd141 : PreambleCoeffInt <= 1'b1;  
             -10'sd140 : PreambleCoeffInt <= 1'b0;  
             -10'sd139 : PreambleCoeffInt <= 1'b0;  
             -10'sd138 : PreambleCoeffInt <= 1'b0;  
             -10'sd137 : PreambleCoeffInt <= 1'b0;  
             -10'sd136 : PreambleCoeffInt <= 1'b0;  
             -10'sd135 : PreambleCoeffInt <= 1'b0;  
             -10'sd134 : PreambleCoeffInt <= 1'b0;  
             -10'sd133 : PreambleCoeffInt <= 1'b0;  
             -10'sd132 : PreambleCoeffInt <= 1'b1;  
             -10'sd131 : PreambleCoeffInt <= 1'b0;  
             -10'sd130 : PreambleCoeffInt <= 1'b0;  
             -10'sd129 : PreambleCoeffInt <= 1'b1;  
             -10'sd128 : PreambleCoeffInt <= 1'b1;  
             -10'sd127 : PreambleCoeffInt <= 1'b0;  
             -10'sd126 : PreambleCoeffInt <= 1'b1;  
             -10'sd125 : PreambleCoeffInt <= 1'b0;  
             -10'sd124 : PreambleCoeffInt <= 1'b0;  
             -10'sd123 : PreambleCoeffInt <= 1'b0;  
             -10'sd122 : PreambleCoeffInt <= 1'b0;  
             -10'sd121 : PreambleCoeffInt <= 1'b0;  
             -10'sd120 : PreambleCoeffInt <= 1'b1;  
             -10'sd119 : PreambleCoeffInt <= 1'b0;  
             -10'sd118 : PreambleCoeffInt <= 1'b1;  
             -10'sd117 : PreambleCoeffInt <= 1'b1;  
             -10'sd116 : PreambleCoeffInt <= 1'b1;  
             -10'sd115 : PreambleCoeffInt <= 1'b0;  
             -10'sd114 : PreambleCoeffInt <= 1'b0;  
             -10'sd113 : PreambleCoeffInt <= 1'b1;  
             -10'sd112 : PreambleCoeffInt <= 1'b1;  
             -10'sd111 : PreambleCoeffInt <= 1'b1;  
             -10'sd110 : PreambleCoeffInt <= 1'b1;  
             -10'sd109 : PreambleCoeffInt <= 1'b1;  
             -10'sd108 : PreambleCoeffInt <= 1'b1;  
             -10'sd107 : PreambleCoeffInt <= 1'b1;  
             -10'sd106 : PreambleCoeffInt <= 1'b1;  
             -10'sd105 : PreambleCoeffInt <= 1'b0;  
             -10'sd104 : PreambleCoeffInt <= 1'b1;  
             -10'sd103 : PreambleCoeffInt <= 1'b1;  
             -10'sd102 : PreambleCoeffInt <= 1'b0;  
             -10'sd101 : PreambleCoeffInt <= 1'b0;  
             -10'sd100 : PreambleCoeffInt <= 1'b1;  
             -10'sd99  : PreambleCoeffInt <= 1'b0;  
             -10'sd98  : PreambleCoeffInt <= 1'b1;  
             -10'sd97  : PreambleCoeffInt <= 1'b0;  
             -10'sd96  : PreambleCoeffInt <= 1'b1;  
             -10'sd95  : PreambleCoeffInt <= 1'b0;  
             -10'sd94  : PreambleCoeffInt <= 1'b0;  
             -10'sd93  : PreambleCoeffInt <= 1'b1;  
             -10'sd92  : PreambleCoeffInt <= 1'b0;  
             -10'sd91  : PreambleCoeffInt <= 1'b1;  
             -10'sd90  : PreambleCoeffInt <= 1'b1;  
             -10'sd89  : PreambleCoeffInt <= 1'b1;  
             -10'sd88  : PreambleCoeffInt <= 1'b0;  
             -10'sd87  : PreambleCoeffInt <= 1'b0;  
             -10'sd86  : PreambleCoeffInt <= 1'b1;  
             -10'sd85  : PreambleCoeffInt <= 1'b1;  
             -10'sd84  : PreambleCoeffInt <= 1'b1;  
             -10'sd83  : PreambleCoeffInt <= 1'b1;  
             -10'sd82  : PreambleCoeffInt <= 1'b1;  
             -10'sd81  : PreambleCoeffInt <= 1'b1;  
             -10'sd80  : PreambleCoeffInt <= 1'b1;  
             -10'sd79  : PreambleCoeffInt <= 1'b0;  
             -10'sd78  : PreambleCoeffInt <= 1'b1;  
             -10'sd77  : PreambleCoeffInt <= 1'b1;  
             -10'sd76  : PreambleCoeffInt <= 1'b0;  
             -10'sd75  : PreambleCoeffInt <= 1'b0;  
             -10'sd74  : PreambleCoeffInt <= 1'b1;  
             -10'sd73  : PreambleCoeffInt <= 1'b0;  
             -10'sd72  : PreambleCoeffInt <= 1'b1;  
             -10'sd71  : PreambleCoeffInt <= 1'b1;  
             -10'sd70  : PreambleCoeffInt <= 1'b1;  
             -10'sd69  : PreambleCoeffInt <= 1'b1;  
             -10'sd68  : PreambleCoeffInt <= 1'b1;  
             -10'sd67  : PreambleCoeffInt <= 1'b0;  
             -10'sd66  : PreambleCoeffInt <= 1'b1;  
             -10'sd65  : PreambleCoeffInt <= 1'b0;  
             -10'sd64  : PreambleCoeffInt <= 1'b0;  
             -10'sd63  : PreambleCoeffInt <= 1'b0;  
             -10'sd62  : PreambleCoeffInt <= 1'b1;  
             -10'sd61  : PreambleCoeffInt <= 1'b1;  
             -10'sd60  : PreambleCoeffInt <= 1'b0;  
             -10'sd59  : PreambleCoeffInt <= 1'b0;  
             -10'sd58  : PreambleCoeffInt <= 1'b0;  
             -10'sd57  : PreambleCoeffInt <= 1'b1;  
             -10'sd56  : PreambleCoeffInt <= 1'b1;  
             -10'sd55  : PreambleCoeffInt <= 1'b1;  
             -10'sd54  : PreambleCoeffInt <= 1'b1;  
             -10'sd53  : PreambleCoeffInt <= 1'b1;  
             -10'sd52  : PreambleCoeffInt <= 1'b1;  
             -10'sd51  : PreambleCoeffInt <= 1'b0;  
             -10'sd50  : PreambleCoeffInt <= 1'b1;  
             -10'sd49  : PreambleCoeffInt <= 1'b1;  
             -10'sd48  : PreambleCoeffInt <= 1'b0;  
             -10'sd47  : PreambleCoeffInt <= 1'b0;  
             -10'sd46  : PreambleCoeffInt <= 1'b1;  
             -10'sd45  : PreambleCoeffInt <= 1'b0;  
             -10'sd44  : PreambleCoeffInt <= 1'b1;  
             -10'sd43  : PreambleCoeffInt <= 1'b0;  
             -10'sd42  : PreambleCoeffInt <= 1'b1;  
             -10'sd41  : PreambleCoeffInt <= 1'b0;  
             -10'sd40  : PreambleCoeffInt <= 1'b0;  
             -10'sd39  : PreambleCoeffInt <= 1'b1;  
             -10'sd38  : PreambleCoeffInt <= 1'b0;  
             -10'sd37  : PreambleCoeffInt <= 1'b1;  
             -10'sd36  : PreambleCoeffInt <= 1'b1;  
             -10'sd35  : PreambleCoeffInt <= 1'b1;  
             -10'sd34  : PreambleCoeffInt <= 1'b0;  
             -10'sd33  : PreambleCoeffInt <= 1'b0;  
             -10'sd32  : PreambleCoeffInt <= 1'b1;  
             -10'sd31  : PreambleCoeffInt <= 1'b1;  
             -10'sd30  : PreambleCoeffInt <= 1'b1;  
             -10'sd29  : PreambleCoeffInt <= 1'b0;  
             -10'sd28  : PreambleCoeffInt <= 1'b0;  
             -10'sd27  : PreambleCoeffInt <= 1'b0;  
             -10'sd26  : PreambleCoeffInt <= 1'b0;  
             -10'sd25  : PreambleCoeffInt <= 1'b1;  
             -10'sd24  : PreambleCoeffInt <= 1'b0;  
             -10'sd23  : PreambleCoeffInt <= 1'b0;  
             -10'sd22  : PreambleCoeffInt <= 1'b1;  
             -10'sd21  : PreambleCoeffInt <= 1'b1;  
             -10'sd20  : PreambleCoeffInt <= 1'b0;  
             -10'sd19  : PreambleCoeffInt <= 1'b1;  
             -10'sd18  : PreambleCoeffInt <= 1'b0;  
             -10'sd17  : PreambleCoeffInt <= 1'b0;  
             -10'sd16  : PreambleCoeffInt <= 1'b0;  
             -10'sd15  : PreambleCoeffInt <= 1'b0;  
             -10'sd14  : PreambleCoeffInt <= 1'b0;  
             -10'sd13  : PreambleCoeffInt <= 1'b1;  
             -10'sd12  : PreambleCoeffInt <= 1'b0;  
             -10'sd11  : PreambleCoeffInt <= 1'b1;  
             -10'sd10  : PreambleCoeffInt <= 1'b1;  
             -10'sd9   : PreambleCoeffInt <= 1'b1;  
             -10'sd8   : PreambleCoeffInt <= 1'b0;  
             -10'sd7   : PreambleCoeffInt <= 1'b0;  
             -10'sd6   : PreambleCoeffInt <= 1'b1;  
             -10'sd5   : PreambleCoeffInt <= 1'b1;  
             -10'sd4   : PreambleCoeffInt <= 1'b1;  
             -10'sd3   : PreambleCoeffInt <= 1'b0;  
             -10'sd2   : PreambleCoeffInt <= 1'b0;  
             10'd2     : PreambleCoeffInt <= 1'b0;  
             10'd3     : PreambleCoeffInt <= 1'b1;  
             10'd4     : PreambleCoeffInt <= 1'b0;  
             10'd5     : PreambleCoeffInt <= 1'b0;  
             10'd6     : PreambleCoeffInt <= 1'b0;  
             10'd7     : PreambleCoeffInt <= 1'b0;  
             10'd8     : PreambleCoeffInt <= 1'b1;  
             10'd9     : PreambleCoeffInt <= 1'b0;  
             10'd10    : PreambleCoeffInt <= 1'b0;  
             10'd11    : PreambleCoeffInt <= 1'b1;  
             10'd12    : PreambleCoeffInt <= 1'b1;  
             10'd13    : PreambleCoeffInt <= 1'b0;  
             10'd14    : PreambleCoeffInt <= 1'b1;  
             10'd15    : PreambleCoeffInt <= 1'b0;  
             10'd16    : PreambleCoeffInt <= 1'b1;  
             10'd17    : PreambleCoeffInt <= 1'b0;  
             10'd18    : PreambleCoeffInt <= 1'b1;  
             10'd19    : PreambleCoeffInt <= 1'b1;  
             10'd20    : PreambleCoeffInt <= 1'b0;  
             10'd21    : PreambleCoeffInt <= 1'b1;  
             10'd22    : PreambleCoeffInt <= 1'b0;  
             10'd23    : PreambleCoeffInt <= 1'b0;  
             10'd24    : PreambleCoeffInt <= 1'b0;  
             10'd25    : PreambleCoeffInt <= 1'b1;  
             10'd26    : PreambleCoeffInt <= 1'b1;  
             10'd27    : PreambleCoeffInt <= 1'b0;  
             10'd28    : PreambleCoeffInt <= 1'b0;  
             10'd29    : PreambleCoeffInt <= 1'b0;  
             10'd30    : PreambleCoeffInt <= 1'b0;  
             10'd31    : PreambleCoeffInt <= 1'b0;  
             10'd32    : PreambleCoeffInt <= 1'b0;  
             10'd33    : PreambleCoeffInt <= 1'b0;  
             10'd34    : PreambleCoeffInt <= 1'b1;  
             10'd35    : PreambleCoeffInt <= 1'b0;  
             10'd36    : PreambleCoeffInt <= 1'b0;  
             10'd37    : PreambleCoeffInt <= 1'b1;  
             10'd38    : PreambleCoeffInt <= 1'b1;  
             10'd39    : PreambleCoeffInt <= 1'b0;  
             10'd40    : PreambleCoeffInt <= 1'b1;  
             10'd41    : PreambleCoeffInt <= 1'b0;  
             10'd42    : PreambleCoeffInt <= 1'b0;  
             10'd43    : PreambleCoeffInt <= 1'b0;  
             10'd44    : PreambleCoeffInt <= 1'b0;  
             10'd45    : PreambleCoeffInt <= 1'b0;  
             10'd46    : PreambleCoeffInt <= 1'b1;  
             10'd47    : PreambleCoeffInt <= 1'b0;  
             10'd48    : PreambleCoeffInt <= 1'b1;  
             10'd49    : PreambleCoeffInt <= 1'b1;  
             10'd50    : PreambleCoeffInt <= 1'b1;  
             10'd51    : PreambleCoeffInt <= 1'b0;  
             10'd52    : PreambleCoeffInt <= 1'b0;  
             10'd53    : PreambleCoeffInt <= 1'b1;  
             10'd54    : PreambleCoeffInt <= 1'b1;  
             10'd55    : PreambleCoeffInt <= 1'b1;  
             10'd56    : PreambleCoeffInt <= 1'b0;  
             10'd57    : PreambleCoeffInt <= 1'b0;  
             10'd58    : PreambleCoeffInt <= 1'b1;  
             10'd59    : PreambleCoeffInt <= 1'b1;  
             10'd60    : PreambleCoeffInt <= 1'b1;  
             10'd61    : PreambleCoeffInt <= 1'b1;  
             10'd62    : PreambleCoeffInt <= 1'b0;  
             10'd63    : PreambleCoeffInt <= 1'b1;  
             10'd64    : PreambleCoeffInt <= 1'b1;  
             10'd65    : PreambleCoeffInt <= 1'b0;  
             10'd66    : PreambleCoeffInt <= 1'b0;  
             10'd67    : PreambleCoeffInt <= 1'b1;  
             10'd68    : PreambleCoeffInt <= 1'b0;  
             10'd69    : PreambleCoeffInt <= 1'b1;  
             10'd70    : PreambleCoeffInt <= 1'b0;  
             10'd71    : PreambleCoeffInt <= 1'b1;  
             10'd72    : PreambleCoeffInt <= 1'b0;  
             10'd73    : PreambleCoeffInt <= 1'b0;  
             10'd74    : PreambleCoeffInt <= 1'b1;  
             10'd75    : PreambleCoeffInt <= 1'b0;  
             10'd76    : PreambleCoeffInt <= 1'b1;  
             10'd77    : PreambleCoeffInt <= 1'b1;  
             10'd78    : PreambleCoeffInt <= 1'b1;  
             10'd79    : PreambleCoeffInt <= 1'b0;  
             10'd80    : PreambleCoeffInt <= 1'b0;  
             10'd81    : PreambleCoeffInt <= 1'b1;  
             10'd82    : PreambleCoeffInt <= 1'b1;  
             10'd83    : PreambleCoeffInt <= 1'b1;  
             10'd84    : PreambleCoeffInt <= 1'b0;  
             10'd85    : PreambleCoeffInt <= 1'b0;  
             10'd86    : PreambleCoeffInt <= 1'b0;  
             10'd87    : PreambleCoeffInt <= 1'b0;  
             10'd88    : PreambleCoeffInt <= 1'b1;  
             10'd89    : PreambleCoeffInt <= 1'b0;  
             10'd90    : PreambleCoeffInt <= 1'b0;  
             10'd91    : PreambleCoeffInt <= 1'b1;  
             10'd92    : PreambleCoeffInt <= 1'b1;  
             10'd93    : PreambleCoeffInt <= 1'b0;  
             10'd94    : PreambleCoeffInt <= 1'b1;  
             10'd95    : PreambleCoeffInt <= 1'b0;  
             10'd96    : PreambleCoeffInt <= 1'b0;  
             10'd97    : PreambleCoeffInt <= 1'b0;  
             10'd98    : PreambleCoeffInt <= 1'b0;  
             10'd99    : PreambleCoeffInt <= 1'b0;  
             10'd100   : PreambleCoeffInt <= 1'b1;  
             10'd101   : PreambleCoeffInt <= 1'b0;  
             10'd102   : PreambleCoeffInt <= 1'b1;  
             10'd103   : PreambleCoeffInt <= 1'b1;  
             10'd104   : PreambleCoeffInt <= 1'b1;  
             10'd105   : PreambleCoeffInt <= 1'b0;  
             10'd106   : PreambleCoeffInt <= 1'b0;  
             10'd107   : PreambleCoeffInt <= 1'b1;  
             10'd108   : PreambleCoeffInt <= 1'b1;  
             10'd109   : PreambleCoeffInt <= 1'b1;  
             10'd110   : PreambleCoeffInt <= 1'b0;  
             10'd111   : PreambleCoeffInt <= 1'b1;  
             10'd112   : PreambleCoeffInt <= 1'b1;  
             10'd113   : PreambleCoeffInt <= 1'b1;  
             10'd114   : PreambleCoeffInt <= 1'b1;  
             10'd115   : PreambleCoeffInt <= 1'b0;  
             10'd116   : PreambleCoeffInt <= 1'b1;  
             10'd117   : PreambleCoeffInt <= 1'b1;  
             10'd118   : PreambleCoeffInt <= 1'b0;  
             10'd119   : PreambleCoeffInt <= 1'b0;  
             10'd120   : PreambleCoeffInt <= 1'b1;  
             10'd121   : PreambleCoeffInt <= 1'b0;  
             10'd122   : PreambleCoeffInt <= 1'b1;  
             10'd123   : PreambleCoeffInt <= 1'b0;  
             10'd124   : PreambleCoeffInt <= 1'b1;  
             10'd125   : PreambleCoeffInt <= 1'b0;  
             10'd126   : PreambleCoeffInt <= 1'b0;  
             10'd127   : PreambleCoeffInt <= 1'b1;  
             10'd128   : PreambleCoeffInt <= 1'b0;  
             10'd129   : PreambleCoeffInt <= 1'b1;  
             10'd130   : PreambleCoeffInt <= 1'b1;  
             10'd131   : PreambleCoeffInt <= 1'b1;  
             10'd132   : PreambleCoeffInt <= 1'b0;  
             10'd133   : PreambleCoeffInt <= 1'b0;  
             10'd134   : PreambleCoeffInt <= 1'b1;  
             10'd135   : PreambleCoeffInt <= 1'b1;  
             10'd136   : PreambleCoeffInt <= 1'b1;  
             10'd137   : PreambleCoeffInt <= 1'b0;  
             10'd138   : PreambleCoeffInt <= 1'b1;  
             10'd139   : PreambleCoeffInt <= 1'b1;  
             10'd140   : PreambleCoeffInt <= 1'b1;  
             10'd141   : PreambleCoeffInt <= 1'b1;  
             10'd142   : PreambleCoeffInt <= 1'b0;  
             10'd143   : PreambleCoeffInt <= 1'b1;  
             10'd144   : PreambleCoeffInt <= 1'b1;  
             10'd145   : PreambleCoeffInt <= 1'b0;  
             10'd146   : PreambleCoeffInt <= 1'b0;  
             10'd147   : PreambleCoeffInt <= 1'b1;  
             10'd148   : PreambleCoeffInt <= 1'b0;  
             10'd149   : PreambleCoeffInt <= 1'b1;  
             10'd150   : PreambleCoeffInt <= 1'b0;  
             10'd151   : PreambleCoeffInt <= 1'b1;  
             10'd152   : PreambleCoeffInt <= 1'b0;  
             10'd153   : PreambleCoeffInt <= 1'b0;  
             10'd154   : PreambleCoeffInt <= 1'b1;  
             10'd155   : PreambleCoeffInt <= 1'b0;  
             10'd156   : PreambleCoeffInt <= 1'b1;  
             10'd157   : PreambleCoeffInt <= 1'b1;  
             10'd158   : PreambleCoeffInt <= 1'b1;  
             10'd159   : PreambleCoeffInt <= 1'b0;  
             10'd160   : PreambleCoeffInt <= 1'b0;  
             10'd161   : PreambleCoeffInt <= 1'b1;  
             10'd162   : PreambleCoeffInt <= 1'b1;  
             10'd163   : PreambleCoeffInt <= 1'b1;  
             10'd164   : PreambleCoeffInt <= 1'b1;  
             10'd165   : PreambleCoeffInt <= 1'b1;  
             10'd166   : PreambleCoeffInt <= 1'b1;  
             10'd167   : PreambleCoeffInt <= 1'b1;  
             10'd168   : PreambleCoeffInt <= 1'b0;  
             10'd169   : PreambleCoeffInt <= 1'b1;  
             10'd170   : PreambleCoeffInt <= 1'b1;  
             10'd171   : PreambleCoeffInt <= 1'b0;  
             10'd172   : PreambleCoeffInt <= 1'b0;  
             10'd173   : PreambleCoeffInt <= 1'b1;  
             10'd174   : PreambleCoeffInt <= 1'b0;  
             10'd175   : PreambleCoeffInt <= 1'b1;  
             10'd176   : PreambleCoeffInt <= 1'b1;  
             10'd177   : PreambleCoeffInt <= 1'b1;  
             10'd178   : PreambleCoeffInt <= 1'b1;  
             10'd179   : PreambleCoeffInt <= 1'b1;  
             10'd180   : PreambleCoeffInt <= 1'b0;  
             10'd181   : PreambleCoeffInt <= 1'b1;  
             10'd182   : PreambleCoeffInt <= 1'b0;  
             10'd183   : PreambleCoeffInt <= 1'b0;  
             10'd184   : PreambleCoeffInt <= 1'b0;  
             10'd185   : PreambleCoeffInt <= 1'b1;  
             10'd186   : PreambleCoeffInt <= 1'b1;  
             10'd187   : PreambleCoeffInt <= 1'b0;  
             10'd188   : PreambleCoeffInt <= 1'b0;  
             10'd189   : PreambleCoeffInt <= 1'b0;  
             10'd190   : PreambleCoeffInt <= 1'b1;  
             10'd191   : PreambleCoeffInt <= 1'b0;  
             10'd192   : PreambleCoeffInt <= 1'b1;  
             10'd193   : PreambleCoeffInt <= 1'b1;  
             10'd194   : PreambleCoeffInt <= 1'b1;  
             10'd195   : PreambleCoeffInt <= 1'b1;  
             10'd196   : PreambleCoeffInt <= 1'b0;  
             10'd197   : PreambleCoeffInt <= 1'b1;  
             10'd198   : PreambleCoeffInt <= 1'b1;  
             10'd199   : PreambleCoeffInt <= 1'b0;  
             10'd200   : PreambleCoeffInt <= 1'b0;  
             10'd201   : PreambleCoeffInt <= 1'b1;  
             10'd202   : PreambleCoeffInt <= 1'b0;  
             10'd203   : PreambleCoeffInt <= 1'b1;  
             10'd204   : PreambleCoeffInt <= 1'b0;  
             10'd205   : PreambleCoeffInt <= 1'b1;  
             10'd206   : PreambleCoeffInt <= 1'b0;  
             10'd207   : PreambleCoeffInt <= 1'b0;  
             10'd208   : PreambleCoeffInt <= 1'b1;  
             10'd209   : PreambleCoeffInt <= 1'b0;  
             10'd210   : PreambleCoeffInt <= 1'b1;  
             10'd211   : PreambleCoeffInt <= 1'b1;  
             10'd212   : PreambleCoeffInt <= 1'b1;  
             10'd213   : PreambleCoeffInt <= 1'b0;  
             10'd214   : PreambleCoeffInt <= 1'b0;  
             10'd215   : PreambleCoeffInt <= 1'b1;  
             10'd216   : PreambleCoeffInt <= 1'b1;  
             10'd217   : PreambleCoeffInt <= 1'b1;  
             10'd218   : PreambleCoeffInt <= 1'b0;  
             10'd219   : PreambleCoeffInt <= 1'b0;  
             10'd220   : PreambleCoeffInt <= 1'b0;  
             10'd221   : PreambleCoeffInt <= 1'b0;  
             10'd222   : PreambleCoeffInt <= 1'b1;  
             10'd223   : PreambleCoeffInt <= 1'b0;  
             10'd224   : PreambleCoeffInt <= 1'b0;  
             10'd225   : PreambleCoeffInt <= 1'b1;  
             10'd226   : PreambleCoeffInt <= 1'b1;  
             10'd227   : PreambleCoeffInt <= 1'b0;  
             10'd228   : PreambleCoeffInt <= 1'b1;  
             10'd229   : PreambleCoeffInt <= 1'b0;  
             10'd230   : PreambleCoeffInt <= 1'b0;  
             10'd231   : PreambleCoeffInt <= 1'b0;  
             10'd232   : PreambleCoeffInt <= 1'b0;  
             10'd233   : PreambleCoeffInt <= 1'b0;  
             10'd234   : PreambleCoeffInt <= 1'b1;  
             10'd235   : PreambleCoeffInt <= 1'b0;  
             10'd236   : PreambleCoeffInt <= 1'b1;  
             10'd237   : PreambleCoeffInt <= 1'b1;  
             10'd238   : PreambleCoeffInt <= 1'b1;  
             10'd239   : PreambleCoeffInt <= 1'b0;  
             10'd240   : PreambleCoeffInt <= 1'b0;  
             10'd241   : PreambleCoeffInt <= 1'b1;  
             10'd242   : PreambleCoeffInt <= 1'b1;  
             10'd243   : PreambleCoeffInt <= 1'b1;  
             10'd244   : PreambleCoeffInt <= 1'b1;  
             default  : PreambleCoeffInt <= 1'b0; // Null
          endcase // RdAddr
        end //40 MHz HE 4x Preamble
        //40 MHz HE 2x Preamble
        else if (SymbolType_1t==HE_2xLTF)
        begin
          //Encoding -1 = 1; 1 = 0;
          case (RdAddr)
             -10'sd244 : PreambleCoeffInt <= 1'b0;
             -10'sd242 : PreambleCoeffInt <= 1'b1;
             -10'sd240 : PreambleCoeffInt <= 1'b1;
             -10'sd238 : PreambleCoeffInt <= 1'b1;
             -10'sd236 : PreambleCoeffInt <= 1'b1;
             -10'sd234 : PreambleCoeffInt <= 1'b1;
             -10'sd232 : PreambleCoeffInt <= 1'b0;
             -10'sd230 : PreambleCoeffInt <= 1'b0;
             -10'sd228 : PreambleCoeffInt <= 1'b1;
             -10'sd226 : PreambleCoeffInt <= 1'b1;
             -10'sd224 : PreambleCoeffInt <= 1'b0;
             -10'sd222 : PreambleCoeffInt <= 1'b1;
             -10'sd220 : PreambleCoeffInt <= 1'b0;
             -10'sd218 : PreambleCoeffInt <= 1'b1;
             -10'sd216 : PreambleCoeffInt <= 1'b1;
             -10'sd214 : PreambleCoeffInt <= 1'b1;
             -10'sd212 : PreambleCoeffInt <= 1'b1;
             -10'sd210 : PreambleCoeffInt <= 1'b1;
             -10'sd208 : PreambleCoeffInt <= 1'b1;
             -10'sd206 : PreambleCoeffInt <= 1'b0;
             -10'sd204 : PreambleCoeffInt <= 1'b0;
             -10'sd202 : PreambleCoeffInt <= 1'b0;
             -10'sd200 : PreambleCoeffInt <= 1'b0;
             -10'sd198 : PreambleCoeffInt <= 1'b1;
             -10'sd196 : PreambleCoeffInt <= 1'b0;
             -10'sd194 : PreambleCoeffInt <= 1'b1;
             -10'sd192 : PreambleCoeffInt <= 1'b0;
             -10'sd190 : PreambleCoeffInt <= 1'b0;
             -10'sd188 : PreambleCoeffInt <= 1'b1;
             -10'sd186 : PreambleCoeffInt <= 1'b1;
             -10'sd184 : PreambleCoeffInt <= 1'b1;
             -10'sd182 : PreambleCoeffInt <= 1'b1;
             -10'sd180 : PreambleCoeffInt <= 1'b1;
             -10'sd178 : PreambleCoeffInt <= 1'b0;
             -10'sd176 : PreambleCoeffInt <= 1'b0;
             -10'sd174 : PreambleCoeffInt <= 1'b1;
             -10'sd172 : PreambleCoeffInt <= 1'b1;
             -10'sd170 : PreambleCoeffInt <= 1'b0;
             -10'sd168 : PreambleCoeffInt <= 1'b1;
             -10'sd166 : PreambleCoeffInt <= 1'b0;
             -10'sd164 : PreambleCoeffInt <= 1'b1;
             -10'sd162 : PreambleCoeffInt <= 1'b0;
             -10'sd160 : PreambleCoeffInt <= 1'b0;
             -10'sd158 : PreambleCoeffInt <= 1'b0;
             -10'sd156 : PreambleCoeffInt <= 1'b0;
             -10'sd154 : PreambleCoeffInt <= 1'b0;
             -10'sd152 : PreambleCoeffInt <= 1'b1;
             -10'sd150 : PreambleCoeffInt <= 1'b1;
             -10'sd148 : PreambleCoeffInt <= 1'b1;
             -10'sd146 : PreambleCoeffInt <= 1'b1;
             -10'sd144 : PreambleCoeffInt <= 1'b0;
             -10'sd142 : PreambleCoeffInt <= 1'b1;
             -10'sd140 : PreambleCoeffInt <= 1'b0;
             -10'sd138 : PreambleCoeffInt <= 1'b1;
             -10'sd136 : PreambleCoeffInt <= 1'b0;
             -10'sd134 : PreambleCoeffInt <= 1'b0;
             -10'sd132 : PreambleCoeffInt <= 1'b0;
             -10'sd130 : PreambleCoeffInt <= 1'b0;
             -10'sd128 : PreambleCoeffInt <= 1'b0;
             -10'sd126 : PreambleCoeffInt <= 1'b1;
             -10'sd124 : PreambleCoeffInt <= 1'b0;
             -10'sd122 : PreambleCoeffInt <= 1'b1;
             -10'sd120 : PreambleCoeffInt <= 1'b1;
             -10'sd118 : PreambleCoeffInt <= 1'b0;
             -10'sd116 : PreambleCoeffInt <= 1'b1;
             -10'sd114 : PreambleCoeffInt <= 1'b0;
             -10'sd112 : PreambleCoeffInt <= 1'b1;
             -10'sd110 : PreambleCoeffInt <= 1'b1;
             -10'sd108 : PreambleCoeffInt <= 1'b0;
             -10'sd106 : PreambleCoeffInt <= 1'b0;
             -10'sd104 : PreambleCoeffInt <= 1'b0;
             -10'sd102 : PreambleCoeffInt <= 1'b0;
             -10'sd100 : PreambleCoeffInt <= 1'b0;
             -10'sd98  : PreambleCoeffInt <= 1'b1;
             -10'sd96  : PreambleCoeffInt <= 1'b1;
             -10'sd94  : PreambleCoeffInt <= 1'b0;
             -10'sd92  : PreambleCoeffInt <= 1'b0;
             -10'sd90  : PreambleCoeffInt <= 1'b1;
             -10'sd88  : PreambleCoeffInt <= 1'b0;
             -10'sd86  : PreambleCoeffInt <= 1'b1;
             -10'sd84  : PreambleCoeffInt <= 1'b0;
             -10'sd82  : PreambleCoeffInt <= 1'b0;
             -10'sd80  : PreambleCoeffInt <= 1'b0;
             -10'sd78  : PreambleCoeffInt <= 1'b0;
             -10'sd76  : PreambleCoeffInt <= 1'b0;
             -10'sd74  : PreambleCoeffInt <= 1'b0;
             -10'sd72  : PreambleCoeffInt <= 1'b1;
             -10'sd70  : PreambleCoeffInt <= 1'b1;
             -10'sd68  : PreambleCoeffInt <= 1'b1;
             -10'sd66  : PreambleCoeffInt <= 1'b1;
             -10'sd64  : PreambleCoeffInt <= 1'b0;
             -10'sd62  : PreambleCoeffInt <= 1'b1;
             -10'sd60  : PreambleCoeffInt <= 1'b0;
             -10'sd58  : PreambleCoeffInt <= 1'b1;
             -10'sd56  : PreambleCoeffInt <= 1'b1;
             -10'sd54  : PreambleCoeffInt <= 1'b1;
             -10'sd52  : PreambleCoeffInt <= 1'b1;
             -10'sd50  : PreambleCoeffInt <= 1'b1;
             -10'sd48  : PreambleCoeffInt <= 1'b1;
             -10'sd46  : PreambleCoeffInt <= 1'b1;
             -10'sd44  : PreambleCoeffInt <= 1'b0;
             -10'sd42  : PreambleCoeffInt <= 1'b0;
             -10'sd40  : PreambleCoeffInt <= 1'b1;
             -10'sd38  : PreambleCoeffInt <= 1'b1;
             -10'sd36  : PreambleCoeffInt <= 1'b0;
             -10'sd34  : PreambleCoeffInt <= 1'b1;
             -10'sd32  : PreambleCoeffInt <= 1'b0;
             -10'sd30  : PreambleCoeffInt <= 1'b1;
             -10'sd28  : PreambleCoeffInt <= 1'b0;
             -10'sd26  : PreambleCoeffInt <= 1'b0;
             -10'sd24  : PreambleCoeffInt <= 1'b0;
             -10'sd22  : PreambleCoeffInt <= 1'b0;
             -10'sd20  : PreambleCoeffInt <= 1'b0;
             -10'sd18  : PreambleCoeffInt <= 1'b1;
             -10'sd16  : PreambleCoeffInt <= 1'b1;
             -10'sd14  : PreambleCoeffInt <= 1'b1;
             -10'sd12  : PreambleCoeffInt <= 1'b1;
             -10'sd10  : PreambleCoeffInt <= 1'b0;
             -10'sd8   : PreambleCoeffInt <= 1'b1;
             -10'sd6   : PreambleCoeffInt <= 1'b0;
             -10'sd4   : PreambleCoeffInt <= 1'b1;
             10'd4     : PreambleCoeffInt <= 1'b1;
             10'd6     : PreambleCoeffInt <= 1'b1;
             10'd8     : PreambleCoeffInt <= 1'b1;
             10'd10    : PreambleCoeffInt <= 1'b1;
             10'd12    : PreambleCoeffInt <= 1'b1;
             10'd14    : PreambleCoeffInt <= 1'b0;
             10'd16    : PreambleCoeffInt <= 1'b0;
             10'd18    : PreambleCoeffInt <= 1'b1;
             10'd20    : PreambleCoeffInt <= 1'b1;
             10'd22    : PreambleCoeffInt <= 1'b0;
             10'd24    : PreambleCoeffInt <= 1'b1;
             10'd26    : PreambleCoeffInt <= 1'b0;
             10'd28    : PreambleCoeffInt <= 1'b1;
             10'd30    : PreambleCoeffInt <= 1'b1;
             10'd32    : PreambleCoeffInt <= 1'b1;
             10'd34    : PreambleCoeffInt <= 1'b1;
             10'd36    : PreambleCoeffInt <= 1'b1;
             10'd38    : PreambleCoeffInt <= 1'b1;
             10'd40    : PreambleCoeffInt <= 1'b0;
             10'd42    : PreambleCoeffInt <= 1'b1;
             10'd44    : PreambleCoeffInt <= 1'b0;
             10'd46    : PreambleCoeffInt <= 1'b0;
             10'd48    : PreambleCoeffInt <= 1'b1;
             10'd50    : PreambleCoeffInt <= 1'b0;
             10'd52    : PreambleCoeffInt <= 1'b1;
             10'd54    : PreambleCoeffInt <= 1'b0;
             10'd56    : PreambleCoeffInt <= 1'b0;
             10'd58    : PreambleCoeffInt <= 1'b0;
             10'd60    : PreambleCoeffInt <= 1'b0;
             10'd62    : PreambleCoeffInt <= 1'b0;
             10'd64    : PreambleCoeffInt <= 1'b0;
             10'd66    : PreambleCoeffInt <= 1'b0;
             10'd68    : PreambleCoeffInt <= 1'b1;
             10'd70    : PreambleCoeffInt <= 1'b1;
             10'd72    : PreambleCoeffInt <= 1'b0;
             10'd74    : PreambleCoeffInt <= 1'b0;
             10'd76    : PreambleCoeffInt <= 1'b1;
             10'd78    : PreambleCoeffInt <= 1'b0;
             10'd80    : PreambleCoeffInt <= 1'b1;
             10'd82    : PreambleCoeffInt <= 1'b0;
             10'd84    : PreambleCoeffInt <= 1'b1;
             10'd86    : PreambleCoeffInt <= 1'b1;
             10'd88    : PreambleCoeffInt <= 1'b1;
             10'd90    : PreambleCoeffInt <= 1'b1;
             10'd92    : PreambleCoeffInt <= 1'b1;
             10'd94    : PreambleCoeffInt <= 1'b0;
             10'd96    : PreambleCoeffInt <= 1'b1;
             10'd98    : PreambleCoeffInt <= 1'b0;
             10'd100   : PreambleCoeffInt <= 1'b0;
             10'd102   : PreambleCoeffInt <= 1'b1;
             10'd104   : PreambleCoeffInt <= 1'b0;
             10'd106   : PreambleCoeffInt <= 1'b1;
             10'd108   : PreambleCoeffInt <= 1'b0;
             10'd110   : PreambleCoeffInt <= 1'b0;
             10'd112   : PreambleCoeffInt <= 1'b0;
             10'd114   : PreambleCoeffInt <= 1'b0;
             10'd116   : PreambleCoeffInt <= 1'b0;
             10'd118   : PreambleCoeffInt <= 1'b0;
             10'd120   : PreambleCoeffInt <= 1'b0;
             10'd122   : PreambleCoeffInt <= 1'b1;
             10'd124   : PreambleCoeffInt <= 1'b1;
             10'd126   : PreambleCoeffInt <= 1'b0;
             10'd128   : PreambleCoeffInt <= 1'b0;
             10'd130   : PreambleCoeffInt <= 1'b1;
             10'd132   : PreambleCoeffInt <= 1'b0;
             10'd134   : PreambleCoeffInt <= 1'b1;
             10'd136   : PreambleCoeffInt <= 1'b0;
             10'd138   : PreambleCoeffInt <= 1'b1;
             10'd140   : PreambleCoeffInt <= 1'b1;
             10'd142   : PreambleCoeffInt <= 1'b1;
             10'd144   : PreambleCoeffInt <= 1'b1;
             10'd146   : PreambleCoeffInt <= 1'b1;
             10'd148   : PreambleCoeffInt <= 1'b0;
             10'd150   : PreambleCoeffInt <= 1'b0;
             10'd152   : PreambleCoeffInt <= 1'b1;
             10'd154   : PreambleCoeffInt <= 1'b1;
             10'd156   : PreambleCoeffInt <= 1'b0;
             10'd158   : PreambleCoeffInt <= 1'b1;
             10'd160   : PreambleCoeffInt <= 1'b0;
             10'd162   : PreambleCoeffInt <= 1'b1;
             10'd164   : PreambleCoeffInt <= 1'b1;
             10'd166   : PreambleCoeffInt <= 1'b1;
             10'd168   : PreambleCoeffInt <= 1'b1;
             10'd170   : PreambleCoeffInt <= 1'b1;
             10'd172   : PreambleCoeffInt <= 1'b1;
             10'd174   : PreambleCoeffInt <= 1'b0;
             10'd176   : PreambleCoeffInt <= 1'b1;
             10'd178   : PreambleCoeffInt <= 1'b0;
             10'd180   : PreambleCoeffInt <= 1'b0;
             10'd182   : PreambleCoeffInt <= 1'b1;
             10'd184   : PreambleCoeffInt <= 1'b0;
             10'd186   : PreambleCoeffInt <= 1'b1;
             10'd188   : PreambleCoeffInt <= 1'b0;
             10'd190   : PreambleCoeffInt <= 1'b0;
             10'd192   : PreambleCoeffInt <= 1'b1;
             10'd194   : PreambleCoeffInt <= 1'b1;
             10'd196   : PreambleCoeffInt <= 1'b1;
             10'd198   : PreambleCoeffInt <= 1'b1;
             10'd200   : PreambleCoeffInt <= 1'b1;
             10'd202   : PreambleCoeffInt <= 1'b0;
             10'd204   : PreambleCoeffInt <= 1'b0;
             10'd206   : PreambleCoeffInt <= 1'b1;
             10'd208   : PreambleCoeffInt <= 1'b1;
             10'd210   : PreambleCoeffInt <= 1'b0;
             10'd212   : PreambleCoeffInt <= 1'b1;
             10'd214   : PreambleCoeffInt <= 1'b0;
             10'd216   : PreambleCoeffInt <= 1'b1;
             10'd218   : PreambleCoeffInt <= 1'b0;
             10'd220   : PreambleCoeffInt <= 1'b0;
             10'd222   : PreambleCoeffInt <= 1'b0;
             10'd224   : PreambleCoeffInt <= 1'b0;
             10'd226   : PreambleCoeffInt <= 1'b0;
             10'd228   : PreambleCoeffInt <= 1'b1;
             10'd230   : PreambleCoeffInt <= 1'b0;
             10'd232   : PreambleCoeffInt <= 1'b1;
             10'd234   : PreambleCoeffInt <= 1'b1;
             10'd236   : PreambleCoeffInt <= 1'b0;
             10'd238   : PreambleCoeffInt <= 1'b1;
             10'd240   : PreambleCoeffInt <= 1'b0;
             10'd242   : PreambleCoeffInt <= 1'b1;
             10'd244   : PreambleCoeffInt <= 1'b0;
             default  : PreambleCoeffInt <= 1'b0; // Null
          endcase // RdAddr
        end //40 MHz HE 2x Preamble
        //40 MHz HE 1x Preamble
        else
        begin
          //Encoding -1 = 1; 1 = 0;
          case (RdAddr)
             -10'sd244 : PreambleCoeffInt <= 1'b0;  
             -10'sd240 : PreambleCoeffInt <= 1'b0;  
             -10'sd236 : PreambleCoeffInt <= 1'b0;  
             -10'sd232 : PreambleCoeffInt <= 1'b0;  
             -10'sd228 : PreambleCoeffInt <= 1'b1;  
             -10'sd224 : PreambleCoeffInt <= 1'b0;  
             -10'sd220 : PreambleCoeffInt <= 1'b0;  
             -10'sd216 : PreambleCoeffInt <= 1'b1;  
             -10'sd212 : PreambleCoeffInt <= 1'b1;  
             -10'sd208 : PreambleCoeffInt <= 1'b0;  
             -10'sd204 : PreambleCoeffInt <= 1'b1;  
             -10'sd200 : PreambleCoeffInt <= 1'b0;  
             -10'sd196 : PreambleCoeffInt <= 1'b1;  
             -10'sd192 : PreambleCoeffInt <= 1'b0;  
             -10'sd188 : PreambleCoeffInt <= 1'b1;  
             -10'sd184 : PreambleCoeffInt <= 1'b1;  
             -10'sd180 : PreambleCoeffInt <= 1'b0;  
             -10'sd176 : PreambleCoeffInt <= 1'b1;  
             -10'sd172 : PreambleCoeffInt <= 1'b0;  
             -10'sd168 : PreambleCoeffInt <= 1'b0;  
             -10'sd164 : PreambleCoeffInt <= 1'b0;  
             -10'sd160 : PreambleCoeffInt <= 1'b1;  
             -10'sd156 : PreambleCoeffInt <= 1'b1;  
             -10'sd152 : PreambleCoeffInt <= 1'b0;  
             -10'sd148 : PreambleCoeffInt <= 1'b0;  
             -10'sd144 : PreambleCoeffInt <= 1'b0;  
             -10'sd140 : PreambleCoeffInt <= 1'b0;  
             -10'sd136 : PreambleCoeffInt <= 1'b0;  
             -10'sd132 : PreambleCoeffInt <= 1'b0;  
             -10'sd128 : PreambleCoeffInt <= 1'b0;  
             -10'sd124 : PreambleCoeffInt <= 1'b0;  
             -10'sd120 : PreambleCoeffInt <= 1'b1;  
             -10'sd116 : PreambleCoeffInt <= 1'b0;  
             -10'sd112 : PreambleCoeffInt <= 1'b0;  
             -10'sd108 : PreambleCoeffInt <= 1'b1;  
             -10'sd104 : PreambleCoeffInt <= 1'b1;  
             -10'sd100 : PreambleCoeffInt <= 1'b0;  
             -10'sd96  : PreambleCoeffInt <= 1'b1;  
             -10'sd92  : PreambleCoeffInt <= 1'b0;  
             -10'sd88  : PreambleCoeffInt <= 1'b1;  
             -10'sd84  : PreambleCoeffInt <= 1'b0;  
             -10'sd80  : PreambleCoeffInt <= 1'b1;  
             -10'sd76  : PreambleCoeffInt <= 1'b1;  
             -10'sd72  : PreambleCoeffInt <= 1'b0;  
             -10'sd68  : PreambleCoeffInt <= 1'b1;  
             -10'sd64  : PreambleCoeffInt <= 1'b0;  
             -10'sd60  : PreambleCoeffInt <= 1'b0;  
             -10'sd56  : PreambleCoeffInt <= 1'b0;  
             -10'sd52  : PreambleCoeffInt <= 1'b1;  
             -10'sd48  : PreambleCoeffInt <= 1'b1;  
             -10'sd44  : PreambleCoeffInt <= 1'b0;  
             -10'sd40  : PreambleCoeffInt <= 1'b0;  
             -10'sd36  : PreambleCoeffInt <= 1'b0;  
             -10'sd32  : PreambleCoeffInt <= 1'b0;  
             -10'sd28  : PreambleCoeffInt <= 1'b1;  
             -10'sd24  : PreambleCoeffInt <= 1'b0;  
             -10'sd20  : PreambleCoeffInt <= 1'b1;  
             -10'sd16  : PreambleCoeffInt <= 1'b0;  
             -10'sd12  : PreambleCoeffInt <= 1'b1;  
             -10'sd8   : PreambleCoeffInt <= 1'b1;  
             -10'sd4   : PreambleCoeffInt <= 1'b1;  
             10'd4     : PreambleCoeffInt <= 1'b0;  
             10'd8     : PreambleCoeffInt <= 1'b0;  
             10'd12    : PreambleCoeffInt <= 1'b1;  
             10'd16    : PreambleCoeffInt <= 1'b1;  
             10'd20    : PreambleCoeffInt <= 1'b0;  
             10'd24    : PreambleCoeffInt <= 1'b1;  
             10'd28    : PreambleCoeffInt <= 1'b0;  
             10'd32    : PreambleCoeffInt <= 1'b0;  
             10'd36    : PreambleCoeffInt <= 1'b0;  
             10'd40    : PreambleCoeffInt <= 1'b0;  
             10'd44    : PreambleCoeffInt <= 1'b0;  
             10'd48    : PreambleCoeffInt <= 1'b0;  
             10'd52    : PreambleCoeffInt <= 1'b1;  
             10'd56    : PreambleCoeffInt <= 1'b0;  
             10'd60    : PreambleCoeffInt <= 1'b0;  
             10'd64    : PreambleCoeffInt <= 1'b1;  
             10'd68    : PreambleCoeffInt <= 1'b1;  
             10'd72    : PreambleCoeffInt <= 1'b0;  
             10'd76    : PreambleCoeffInt <= 1'b1;  
             10'd80    : PreambleCoeffInt <= 1'b0;  
             10'd84    : PreambleCoeffInt <= 1'b0;  
             10'd88    : PreambleCoeffInt <= 1'b0;  
             10'd92    : PreambleCoeffInt <= 1'b0;  
             10'd96    : PreambleCoeffInt <= 1'b0;  
             10'd100   : PreambleCoeffInt <= 1'b1;  
             10'd104   : PreambleCoeffInt <= 1'b0;  
             10'd108   : PreambleCoeffInt <= 1'b1;  
             10'd112   : PreambleCoeffInt <= 1'b1;  
             10'd116   : PreambleCoeffInt <= 1'b1;  
             10'd120   : PreambleCoeffInt <= 1'b0;  
             10'd124   : PreambleCoeffInt <= 1'b0;  
             10'd128   : PreambleCoeffInt <= 1'b1;  
             10'd132   : PreambleCoeffInt <= 1'b1;  
             10'd136   : PreambleCoeffInt <= 1'b1;  
             10'd140   : PreambleCoeffInt <= 1'b0;  
             10'd144   : PreambleCoeffInt <= 1'b1;  
             10'd148   : PreambleCoeffInt <= 1'b1;  
             10'd152   : PreambleCoeffInt <= 1'b1;  
             10'd156   : PreambleCoeffInt <= 1'b1;  
             10'd160   : PreambleCoeffInt <= 1'b0;  
             10'd164   : PreambleCoeffInt <= 1'b1;  
             10'd168   : PreambleCoeffInt <= 1'b1;  
             10'd172   : PreambleCoeffInt <= 1'b0;  
             10'd176   : PreambleCoeffInt <= 1'b0;  
             10'd180   : PreambleCoeffInt <= 1'b1;  
             10'd184   : PreambleCoeffInt <= 1'b0;  
             10'd188   : PreambleCoeffInt <= 1'b1;  
             10'd192   : PreambleCoeffInt <= 1'b1;  
             10'd196   : PreambleCoeffInt <= 1'b1;  
             10'd200   : PreambleCoeffInt <= 1'b1;  
             10'd204   : PreambleCoeffInt <= 1'b1;  
             10'd208   : PreambleCoeffInt <= 1'b0;  
             10'd212   : PreambleCoeffInt <= 1'b1;  
             10'd216   : PreambleCoeffInt <= 1'b0;  
             10'd220   : PreambleCoeffInt <= 1'b0;  
             10'd224   : PreambleCoeffInt <= 1'b0;  
             10'd228   : PreambleCoeffInt <= 1'b1;  
             10'd232   : PreambleCoeffInt <= 1'b1;  
             10'd236   : PreambleCoeffInt <= 1'b0;  
             10'd240   : PreambleCoeffInt <= 1'b0;  
             10'd244   : PreambleCoeffInt <= 1'b0;  
             default   : PreambleCoeffInt <= 1'b0; // Null
          endcase // RdAddr
        end //40 MHz HE 1x Preamble
      end
      else if (ReceptionMode_1t == MODE_80MHZ_HE_MU) begin
        //80 MHz HE 4x Preamble
        if (SymbolType_1t==HE_4xLTF)
        begin
          //Encoding -1 = 1; 1 = 0;
          case (RdAddr)
             -10'sd500 : PreambleCoeffInt <= 1'b0;
             -10'sd499 : PreambleCoeffInt <= 1'b0;
             -10'sd498 : PreambleCoeffInt <= 1'b1;
             -10'sd497 : PreambleCoeffInt <= 1'b0;
             -10'sd496 : PreambleCoeffInt <= 1'b1;
             -10'sd495 : PreambleCoeffInt <= 1'b0;
             -10'sd494 : PreambleCoeffInt <= 1'b1;
             -10'sd493 : PreambleCoeffInt <= 1'b1;
             -10'sd492 : PreambleCoeffInt <= 1'b1;
             -10'sd491 : PreambleCoeffInt <= 1'b0;
             -10'sd490 : PreambleCoeffInt <= 1'b1;
             -10'sd489 : PreambleCoeffInt <= 1'b1;
             -10'sd488 : PreambleCoeffInt <= 1'b1;
             -10'sd487 : PreambleCoeffInt <= 1'b0;
             -10'sd486 : PreambleCoeffInt <= 1'b0;
             -10'sd485 : PreambleCoeffInt <= 1'b1;
             -10'sd484 : PreambleCoeffInt <= 1'b0;
             -10'sd483 : PreambleCoeffInt <= 1'b0;
             -10'sd482 : PreambleCoeffInt <= 1'b0;
             -10'sd481 : PreambleCoeffInt <= 1'b0;
             -10'sd480 : PreambleCoeffInt <= 1'b0;
             -10'sd479 : PreambleCoeffInt <= 1'b1;
             -10'sd478 : PreambleCoeffInt <= 1'b1;
             -10'sd477 : PreambleCoeffInt <= 1'b0;
             -10'sd476 : PreambleCoeffInt <= 1'b0;
             -10'sd475 : PreambleCoeffInt <= 1'b0;
             -10'sd474 : PreambleCoeffInt <= 1'b0;
             -10'sd473 : PreambleCoeffInt <= 1'b1;
             -10'sd472 : PreambleCoeffInt <= 1'b0;
             -10'sd471 : PreambleCoeffInt <= 1'b1;
             -10'sd470 : PreambleCoeffInt <= 1'b0;
             -10'sd469 : PreambleCoeffInt <= 1'b1;
             -10'sd468 : PreambleCoeffInt <= 1'b1;
             -10'sd467 : PreambleCoeffInt <= 1'b0;
             -10'sd466 : PreambleCoeffInt <= 1'b0;
             -10'sd465 : PreambleCoeffInt <= 1'b1;
             -10'sd464 : PreambleCoeffInt <= 1'b0;
             -10'sd463 : PreambleCoeffInt <= 1'b0;
             -10'sd462 : PreambleCoeffInt <= 1'b0;
             -10'sd461 : PreambleCoeffInt <= 1'b1;
             -10'sd460 : PreambleCoeffInt <= 1'b1;
             -10'sd459 : PreambleCoeffInt <= 1'b0;
             -10'sd458 : PreambleCoeffInt <= 1'b1;
             -10'sd457 : PreambleCoeffInt <= 1'b1;
             -10'sd456 : PreambleCoeffInt <= 1'b1;
             -10'sd455 : PreambleCoeffInt <= 1'b1;
             -10'sd454 : PreambleCoeffInt <= 1'b0;
             -10'sd453 : PreambleCoeffInt <= 1'b0;
             -10'sd452 : PreambleCoeffInt <= 1'b0;
             -10'sd451 : PreambleCoeffInt <= 1'b1;
             -10'sd450 : PreambleCoeffInt <= 1'b1;
             -10'sd449 : PreambleCoeffInt <= 1'b1;
             -10'sd448 : PreambleCoeffInt <= 1'b1;
             -10'sd447 : PreambleCoeffInt <= 1'b1;
             -10'sd446 : PreambleCoeffInt <= 1'b1;
             -10'sd445 : PreambleCoeffInt <= 1'b0;
             -10'sd444 : PreambleCoeffInt <= 1'b0;
             -10'sd443 : PreambleCoeffInt <= 1'b0;
             -10'sd442 : PreambleCoeffInt <= 1'b0;
             -10'sd441 : PreambleCoeffInt <= 1'b0;
             -10'sd440 : PreambleCoeffInt <= 1'b0;
             -10'sd439 : PreambleCoeffInt <= 1'b1;
             -10'sd438 : PreambleCoeffInt <= 1'b0;
             -10'sd437 : PreambleCoeffInt <= 1'b0;
             -10'sd436 : PreambleCoeffInt <= 1'b0;
             -10'sd435 : PreambleCoeffInt <= 1'b1;
             -10'sd434 : PreambleCoeffInt <= 1'b0;
             -10'sd433 : PreambleCoeffInt <= 1'b0;
             -10'sd432 : PreambleCoeffInt <= 1'b1;
             -10'sd431 : PreambleCoeffInt <= 1'b1;
             -10'sd430 : PreambleCoeffInt <= 1'b1;
             -10'sd429 : PreambleCoeffInt <= 1'b0;
             -10'sd428 : PreambleCoeffInt <= 1'b1;
             -10'sd427 : PreambleCoeffInt <= 1'b0;
             -10'sd426 : PreambleCoeffInt <= 1'b1;
             -10'sd425 : PreambleCoeffInt <= 1'b1;
             -10'sd424 : PreambleCoeffInt <= 1'b0;
             -10'sd423 : PreambleCoeffInt <= 1'b0;
             -10'sd422 : PreambleCoeffInt <= 1'b1;
             -10'sd421 : PreambleCoeffInt <= 1'b0;
             -10'sd420 : PreambleCoeffInt <= 1'b1;
             -10'sd419 : PreambleCoeffInt <= 1'b0;
             -10'sd418 : PreambleCoeffInt <= 1'b0;
             -10'sd417 : PreambleCoeffInt <= 1'b0;
             -10'sd416 : PreambleCoeffInt <= 1'b0;
             -10'sd415 : PreambleCoeffInt <= 1'b0;
             -10'sd414 : PreambleCoeffInt <= 1'b1;
             -10'sd413 : PreambleCoeffInt <= 1'b1;
             -10'sd412 : PreambleCoeffInt <= 1'b0;
             -10'sd411 : PreambleCoeffInt <= 1'b0;
             -10'sd410 : PreambleCoeffInt <= 1'b0;
             -10'sd409 : PreambleCoeffInt <= 1'b1;
             -10'sd408 : PreambleCoeffInt <= 1'b0;
             -10'sd407 : PreambleCoeffInt <= 1'b0;
             -10'sd406 : PreambleCoeffInt <= 1'b1;
             -10'sd405 : PreambleCoeffInt <= 1'b1;
             -10'sd404 : PreambleCoeffInt <= 1'b1;
             -10'sd403 : PreambleCoeffInt <= 1'b0;
             -10'sd402 : PreambleCoeffInt <= 1'b1;
             -10'sd401 : PreambleCoeffInt <= 1'b0;
             -10'sd400 : PreambleCoeffInt <= 1'b0;
             -10'sd399 : PreambleCoeffInt <= 1'b1;
             -10'sd398 : PreambleCoeffInt <= 1'b0;
             -10'sd397 : PreambleCoeffInt <= 1'b0;
             -10'sd396 : PreambleCoeffInt <= 1'b1;
             -10'sd395 : PreambleCoeffInt <= 1'b0;
             -10'sd394 : PreambleCoeffInt <= 1'b1;
             -10'sd393 : PreambleCoeffInt <= 1'b1;
             -10'sd392 : PreambleCoeffInt <= 1'b0;
             -10'sd391 : PreambleCoeffInt <= 1'b0;
             -10'sd390 : PreambleCoeffInt <= 1'b0;
             -10'sd389 : PreambleCoeffInt <= 1'b0;
             -10'sd388 : PreambleCoeffInt <= 1'b1;
             -10'sd387 : PreambleCoeffInt <= 1'b1;
             -10'sd386 : PreambleCoeffInt <= 1'b0;
             -10'sd385 : PreambleCoeffInt <= 1'b0;
             -10'sd384 : PreambleCoeffInt <= 1'b0;
             -10'sd383 : PreambleCoeffInt <= 1'b0;
             -10'sd382 : PreambleCoeffInt <= 1'b0;
             -10'sd381 : PreambleCoeffInt <= 1'b1;
             -10'sd380 : PreambleCoeffInt <= 1'b0;
             -10'sd379 : PreambleCoeffInt <= 1'b0;
             -10'sd378 : PreambleCoeffInt <= 1'b1;
             -10'sd377 : PreambleCoeffInt <= 1'b1;
             -10'sd376 : PreambleCoeffInt <= 1'b1;
             -10'sd375 : PreambleCoeffInt <= 1'b0;
             -10'sd374 : PreambleCoeffInt <= 1'b1;
             -10'sd373 : PreambleCoeffInt <= 1'b1;
             -10'sd372 : PreambleCoeffInt <= 1'b1;
             -10'sd371 : PreambleCoeffInt <= 1'b0;
             -10'sd370 : PreambleCoeffInt <= 1'b1;
             -10'sd369 : PreambleCoeffInt <= 1'b0;
             -10'sd368 : PreambleCoeffInt <= 1'b1;
             -10'sd367 : PreambleCoeffInt <= 1'b0;
             -10'sd366 : PreambleCoeffInt <= 1'b0;
             -10'sd365 : PreambleCoeffInt <= 1'b1;
             -10'sd364 : PreambleCoeffInt <= 1'b0;
             -10'sd363 : PreambleCoeffInt <= 1'b1;
             -10'sd362 : PreambleCoeffInt <= 1'b0;
             -10'sd361 : PreambleCoeffInt <= 1'b1;
             -10'sd360 : PreambleCoeffInt <= 1'b0;
             -10'sd359 : PreambleCoeffInt <= 1'b0;
             -10'sd358 : PreambleCoeffInt <= 1'b0;
             -10'sd357 : PreambleCoeffInt <= 1'b1;
             -10'sd356 : PreambleCoeffInt <= 1'b0;
             -10'sd355 : PreambleCoeffInt <= 1'b0;
             -10'sd354 : PreambleCoeffInt <= 1'b0;
             -10'sd353 : PreambleCoeffInt <= 1'b1;
             -10'sd352 : PreambleCoeffInt <= 1'b1;
             -10'sd351 : PreambleCoeffInt <= 1'b0;
             -10'sd350 : PreambleCoeffInt <= 1'b1;
             -10'sd349 : PreambleCoeffInt <= 1'b1;
             -10'sd348 : PreambleCoeffInt <= 1'b1;
             -10'sd347 : PreambleCoeffInt <= 1'b1;
             -10'sd346 : PreambleCoeffInt <= 1'b1;
             -10'sd345 : PreambleCoeffInt <= 1'b0;
             -10'sd344 : PreambleCoeffInt <= 1'b0;
             -10'sd343 : PreambleCoeffInt <= 1'b1;
             -10'sd342 : PreambleCoeffInt <= 1'b1;
             -10'sd341 : PreambleCoeffInt <= 1'b1;
             -10'sd340 : PreambleCoeffInt <= 1'b1;
             -10'sd339 : PreambleCoeffInt <= 1'b0;
             -10'sd338 : PreambleCoeffInt <= 1'b1;
             -10'sd337 : PreambleCoeffInt <= 1'b0;
             -10'sd336 : PreambleCoeffInt <= 1'b1;
             -10'sd335 : PreambleCoeffInt <= 1'b0;
             -10'sd334 : PreambleCoeffInt <= 1'b0;
             -10'sd333 : PreambleCoeffInt <= 1'b1;
             -10'sd332 : PreambleCoeffInt <= 1'b1;
             -10'sd331 : PreambleCoeffInt <= 1'b0;
             -10'sd330 : PreambleCoeffInt <= 1'b1;
             -10'sd329 : PreambleCoeffInt <= 1'b1;
             -10'sd328 : PreambleCoeffInt <= 1'b1;
             -10'sd327 : PreambleCoeffInt <= 1'b0;
             -10'sd326 : PreambleCoeffInt <= 1'b0;
             -10'sd325 : PreambleCoeffInt <= 1'b1;
             -10'sd324 : PreambleCoeffInt <= 1'b0;
             -10'sd323 : PreambleCoeffInt <= 1'b0;
             -10'sd322 : PreambleCoeffInt <= 1'b0;
             -10'sd321 : PreambleCoeffInt <= 1'b0;
             -10'sd320 : PreambleCoeffInt <= 1'b1;
             -10'sd319 : PreambleCoeffInt <= 1'b1;
             -10'sd318 : PreambleCoeffInt <= 1'b1;
             -10'sd317 : PreambleCoeffInt <= 1'b0;
             -10'sd316 : PreambleCoeffInt <= 1'b0;
             -10'sd315 : PreambleCoeffInt <= 1'b0;
             -10'sd314 : PreambleCoeffInt <= 1'b0;
             -10'sd313 : PreambleCoeffInt <= 1'b1;
             -10'sd312 : PreambleCoeffInt <= 1'b0;
             -10'sd311 : PreambleCoeffInt <= 1'b0;
             -10'sd310 : PreambleCoeffInt <= 1'b0;
             -10'sd309 : PreambleCoeffInt <= 1'b0;
             -10'sd308 : PreambleCoeffInt <= 1'b0;
             -10'sd307 : PreambleCoeffInt <= 1'b0;
             -10'sd306 : PreambleCoeffInt <= 1'b0;
             -10'sd305 : PreambleCoeffInt <= 1'b1;
             -10'sd304 : PreambleCoeffInt <= 1'b0;
             -10'sd303 : PreambleCoeffInt <= 1'b0;
             -10'sd302 : PreambleCoeffInt <= 1'b0;
             -10'sd301 : PreambleCoeffInt <= 1'b1;
             -10'sd300 : PreambleCoeffInt <= 1'b0;
             -10'sd299 : PreambleCoeffInt <= 1'b0;
             -10'sd298 : PreambleCoeffInt <= 1'b1;
             -10'sd297 : PreambleCoeffInt <= 1'b1;
             -10'sd296 : PreambleCoeffInt <= 1'b1;
             -10'sd295 : PreambleCoeffInt <= 1'b0;
             -10'sd294 : PreambleCoeffInt <= 1'b1;
             -10'sd293 : PreambleCoeffInt <= 1'b0;
             -10'sd292 : PreambleCoeffInt <= 1'b1;
             -10'sd291 : PreambleCoeffInt <= 1'b1;
             -10'sd290 : PreambleCoeffInt <= 1'b0;
             -10'sd289 : PreambleCoeffInt <= 1'b0;
             -10'sd288 : PreambleCoeffInt <= 1'b1;
             -10'sd287 : PreambleCoeffInt <= 1'b0;
             -10'sd286 : PreambleCoeffInt <= 1'b1;
             -10'sd285 : PreambleCoeffInt <= 1'b0;
             -10'sd284 : PreambleCoeffInt <= 1'b0;
             -10'sd283 : PreambleCoeffInt <= 1'b0;
             -10'sd282 : PreambleCoeffInt <= 1'b0;
             -10'sd281 : PreambleCoeffInt <= 1'b0;
             -10'sd280 : PreambleCoeffInt <= 1'b1;
             -10'sd279 : PreambleCoeffInt <= 1'b1;
             -10'sd278 : PreambleCoeffInt <= 1'b0;
             -10'sd277 : PreambleCoeffInt <= 1'b0;
             -10'sd276 : PreambleCoeffInt <= 1'b0;
             -10'sd275 : PreambleCoeffInt <= 1'b1;
             -10'sd274 : PreambleCoeffInt <= 1'b0;
             -10'sd273 : PreambleCoeffInt <= 1'b0;
             -10'sd272 : PreambleCoeffInt <= 1'b1;
             -10'sd271 : PreambleCoeffInt <= 1'b1;
             -10'sd270 : PreambleCoeffInt <= 1'b1;
             -10'sd269 : PreambleCoeffInt <= 1'b0;
             -10'sd268 : PreambleCoeffInt <= 1'b1;
             -10'sd267 : PreambleCoeffInt <= 1'b0;
             -10'sd266 : PreambleCoeffInt <= 1'b0;
             -10'sd265 : PreambleCoeffInt <= 1'b1;
             -10'sd264 : PreambleCoeffInt <= 1'b0;
             -10'sd263 : PreambleCoeffInt <= 1'b0;
             -10'sd262 : PreambleCoeffInt <= 1'b1;
             -10'sd261 : PreambleCoeffInt <= 1'b0;
             -10'sd260 : PreambleCoeffInt <= 1'b1;
             -10'sd259 : PreambleCoeffInt <= 1'b1;
             -10'sd258 : PreambleCoeffInt <= 1'b0;
             -10'sd257 : PreambleCoeffInt <= 1'b1;
             -10'sd256 : PreambleCoeffInt <= 1'b0;
             -10'sd255 : PreambleCoeffInt <= 1'b1;
             -10'sd254 : PreambleCoeffInt <= 1'b0;
             -10'sd253 : PreambleCoeffInt <= 1'b1;
             -10'sd252 : PreambleCoeffInt <= 1'b0;
             -10'sd251 : PreambleCoeffInt <= 1'b0;
             -10'sd250 : PreambleCoeffInt <= 1'b0;
             -10'sd249 : PreambleCoeffInt <= 1'b1;
             -10'sd248 : PreambleCoeffInt <= 1'b0;
             -10'sd247 : PreambleCoeffInt <= 1'b0;
             -10'sd246 : PreambleCoeffInt <= 1'b0;
             -10'sd245 : PreambleCoeffInt <= 1'b1;
             -10'sd244 : PreambleCoeffInt <= 1'b1;
             -10'sd243 : PreambleCoeffInt <= 1'b0;
             -10'sd242 : PreambleCoeffInt <= 1'b1;
             -10'sd241 : PreambleCoeffInt <= 1'b1;
             -10'sd240 : PreambleCoeffInt <= 1'b1;
             -10'sd239 : PreambleCoeffInt <= 1'b1;
             -10'sd238 : PreambleCoeffInt <= 1'b1;
             -10'sd237 : PreambleCoeffInt <= 1'b0;
             -10'sd236 : PreambleCoeffInt <= 1'b0;
             -10'sd235 : PreambleCoeffInt <= 1'b1;
             -10'sd234 : PreambleCoeffInt <= 1'b1;
             -10'sd233 : PreambleCoeffInt <= 1'b1;
             -10'sd232 : PreambleCoeffInt <= 1'b1;
             -10'sd231 : PreambleCoeffInt <= 1'b0;
             -10'sd230 : PreambleCoeffInt <= 1'b1;
             -10'sd229 : PreambleCoeffInt <= 1'b0;
             -10'sd228 : PreambleCoeffInt <= 1'b1;
             -10'sd227 : PreambleCoeffInt <= 1'b0;
             -10'sd226 : PreambleCoeffInt <= 1'b0;
             -10'sd225 : PreambleCoeffInt <= 1'b1;
             -10'sd224 : PreambleCoeffInt <= 1'b1;
             -10'sd223 : PreambleCoeffInt <= 1'b0;
             -10'sd222 : PreambleCoeffInt <= 1'b1;
             -10'sd221 : PreambleCoeffInt <= 1'b1;
             -10'sd220 : PreambleCoeffInt <= 1'b1;
             -10'sd219 : PreambleCoeffInt <= 1'b0;
             -10'sd218 : PreambleCoeffInt <= 1'b0;
             -10'sd217 : PreambleCoeffInt <= 1'b1;
             -10'sd216 : PreambleCoeffInt <= 1'b0;
             -10'sd215 : PreambleCoeffInt <= 1'b0;
             -10'sd214 : PreambleCoeffInt <= 1'b0;
             -10'sd213 : PreambleCoeffInt <= 1'b0;
             -10'sd212 : PreambleCoeffInt <= 1'b1;
             -10'sd211 : PreambleCoeffInt <= 1'b1;
             -10'sd210 : PreambleCoeffInt <= 1'b1;
             -10'sd209 : PreambleCoeffInt <= 1'b0;
             -10'sd208 : PreambleCoeffInt <= 1'b0;
             -10'sd207 : PreambleCoeffInt <= 1'b0;
             -10'sd206 : PreambleCoeffInt <= 1'b0;
             -10'sd205 : PreambleCoeffInt <= 1'b1;
             -10'sd204 : PreambleCoeffInt <= 1'b0;
             -10'sd203 : PreambleCoeffInt <= 1'b1;
             -10'sd202 : PreambleCoeffInt <= 1'b1;
             -10'sd201 : PreambleCoeffInt <= 1'b1;
             -10'sd200 : PreambleCoeffInt <= 1'b1;
             -10'sd199 : PreambleCoeffInt <= 1'b1;
             -10'sd198 : PreambleCoeffInt <= 1'b1;
             -10'sd197 : PreambleCoeffInt <= 1'b0;
             -10'sd196 : PreambleCoeffInt <= 1'b1;
             -10'sd195 : PreambleCoeffInt <= 1'b1;
             -10'sd194 : PreambleCoeffInt <= 1'b1;
             -10'sd193 : PreambleCoeffInt <= 1'b0;
             -10'sd192 : PreambleCoeffInt <= 1'b1;
             -10'sd191 : PreambleCoeffInt <= 1'b1;
             -10'sd190 : PreambleCoeffInt <= 1'b0;
             -10'sd189 : PreambleCoeffInt <= 1'b0;
             -10'sd188 : PreambleCoeffInt <= 1'b0;
             -10'sd187 : PreambleCoeffInt <= 1'b1;
             -10'sd186 : PreambleCoeffInt <= 1'b0;
             -10'sd185 : PreambleCoeffInt <= 1'b1;
             -10'sd184 : PreambleCoeffInt <= 1'b0;
             -10'sd183 : PreambleCoeffInt <= 1'b0;
             -10'sd182 : PreambleCoeffInt <= 1'b1;
             -10'sd181 : PreambleCoeffInt <= 1'b1;
             -10'sd180 : PreambleCoeffInt <= 1'b0;
             -10'sd179 : PreambleCoeffInt <= 1'b1;
             -10'sd178 : PreambleCoeffInt <= 1'b0;
             -10'sd177 : PreambleCoeffInt <= 1'b1;
             -10'sd176 : PreambleCoeffInt <= 1'b1;
             -10'sd175 : PreambleCoeffInt <= 1'b1;
             -10'sd174 : PreambleCoeffInt <= 1'b1;
             -10'sd173 : PreambleCoeffInt <= 1'b1;
             -10'sd172 : PreambleCoeffInt <= 1'b0;
             -10'sd171 : PreambleCoeffInt <= 1'b0;
             -10'sd170 : PreambleCoeffInt <= 1'b1;
             -10'sd169 : PreambleCoeffInt <= 1'b1;
             -10'sd168 : PreambleCoeffInt <= 1'b1;
             -10'sd167 : PreambleCoeffInt <= 1'b0;
             -10'sd166 : PreambleCoeffInt <= 1'b1;
             -10'sd165 : PreambleCoeffInt <= 1'b1;
             -10'sd164 : PreambleCoeffInt <= 1'b0;
             -10'sd163 : PreambleCoeffInt <= 1'b0;
             -10'sd162 : PreambleCoeffInt <= 1'b0;
             -10'sd161 : PreambleCoeffInt <= 1'b1;
             -10'sd160 : PreambleCoeffInt <= 1'b0;
             -10'sd159 : PreambleCoeffInt <= 1'b1;
             -10'sd158 : PreambleCoeffInt <= 1'b1;
             -10'sd157 : PreambleCoeffInt <= 1'b0;
             -10'sd156 : PreambleCoeffInt <= 1'b1;
             -10'sd155 : PreambleCoeffInt <= 1'b1;
             -10'sd154 : PreambleCoeffInt <= 1'b0;
             -10'sd153 : PreambleCoeffInt <= 1'b1;
             -10'sd152 : PreambleCoeffInt <= 1'b0;
             -10'sd151 : PreambleCoeffInt <= 1'b0;
             -10'sd150 : PreambleCoeffInt <= 1'b0;
             -10'sd149 : PreambleCoeffInt <= 1'b0;
             -10'sd148 : PreambleCoeffInt <= 1'b0;
             -10'sd147 : PreambleCoeffInt <= 1'b0;
             -10'sd146 : PreambleCoeffInt <= 1'b1;
             -10'sd145 : PreambleCoeffInt <= 1'b1;
             -10'sd144 : PreambleCoeffInt <= 1'b0;
             -10'sd143 : PreambleCoeffInt <= 1'b0;
             -10'sd142 : PreambleCoeffInt <= 1'b0;
             -10'sd141 : PreambleCoeffInt <= 1'b0;
             -10'sd140 : PreambleCoeffInt <= 1'b0;
             -10'sd139 : PreambleCoeffInt <= 1'b1;
             -10'sd138 : PreambleCoeffInt <= 1'b0;
             -10'sd137 : PreambleCoeffInt <= 1'b0;
             -10'sd136 : PreambleCoeffInt <= 1'b1;
             -10'sd135 : PreambleCoeffInt <= 1'b1;
             -10'sd134 : PreambleCoeffInt <= 1'b1;
             -10'sd133 : PreambleCoeffInt <= 1'b0;
             -10'sd132 : PreambleCoeffInt <= 1'b1;
             -10'sd131 : PreambleCoeffInt <= 1'b1;
             -10'sd130 : PreambleCoeffInt <= 1'b1;
             -10'sd129 : PreambleCoeffInt <= 1'b0;
             -10'sd128 : PreambleCoeffInt <= 1'b1;
             -10'sd127 : PreambleCoeffInt <= 1'b0;
             -10'sd126 : PreambleCoeffInt <= 1'b1;
             -10'sd125 : PreambleCoeffInt <= 1'b0;
             -10'sd124 : PreambleCoeffInt <= 1'b0;
             -10'sd123 : PreambleCoeffInt <= 1'b1;
             -10'sd122 : PreambleCoeffInt <= 1'b0;
             -10'sd121 : PreambleCoeffInt <= 1'b1;
             -10'sd120 : PreambleCoeffInt <= 1'b0;
             -10'sd119 : PreambleCoeffInt <= 1'b1;
             -10'sd118 : PreambleCoeffInt <= 1'b0;
             -10'sd117 : PreambleCoeffInt <= 1'b0;
             -10'sd116 : PreambleCoeffInt <= 1'b0;
             -10'sd115 : PreambleCoeffInt <= 1'b1;
             -10'sd114 : PreambleCoeffInt <= 1'b0;
             -10'sd113 : PreambleCoeffInt <= 1'b0;
             -10'sd112 : PreambleCoeffInt <= 1'b0;
             -10'sd111 : PreambleCoeffInt <= 1'b1;
             -10'sd110 : PreambleCoeffInt <= 1'b1;
             -10'sd109 : PreambleCoeffInt <= 1'b0;
             -10'sd108 : PreambleCoeffInt <= 1'b1;
             -10'sd107 : PreambleCoeffInt <= 1'b1;
             -10'sd106 : PreambleCoeffInt <= 1'b1;
             -10'sd105 : PreambleCoeffInt <= 1'b1;
             -10'sd104 : PreambleCoeffInt <= 1'b1;
             -10'sd103 : PreambleCoeffInt <= 1'b0;
             -10'sd102 : PreambleCoeffInt <= 1'b0;
             -10'sd101 : PreambleCoeffInt <= 1'b1;
             -10'sd100 : PreambleCoeffInt <= 1'b1;
             -10'sd99  : PreambleCoeffInt <= 1'b1;
             -10'sd98  : PreambleCoeffInt <= 1'b1;
             -10'sd97  : PreambleCoeffInt <= 1'b0;
             -10'sd96  : PreambleCoeffInt <= 1'b1;
             -10'sd95  : PreambleCoeffInt <= 1'b0;
             -10'sd94  : PreambleCoeffInt <= 1'b1;
             -10'sd93  : PreambleCoeffInt <= 1'b0;
             -10'sd92  : PreambleCoeffInt <= 1'b0;
             -10'sd91  : PreambleCoeffInt <= 1'b1;
             -10'sd90  : PreambleCoeffInt <= 1'b1;
             -10'sd89  : PreambleCoeffInt <= 1'b0;
             -10'sd88  : PreambleCoeffInt <= 1'b1;
             -10'sd87  : PreambleCoeffInt <= 1'b1;
             -10'sd86  : PreambleCoeffInt <= 1'b1;
             -10'sd85  : PreambleCoeffInt <= 1'b0;
             -10'sd84  : PreambleCoeffInt <= 1'b0;
             -10'sd83  : PreambleCoeffInt <= 1'b1;
             -10'sd82  : PreambleCoeffInt <= 1'b0;
             -10'sd81  : PreambleCoeffInt <= 1'b0;
             -10'sd80  : PreambleCoeffInt <= 1'b0;
             -10'sd79  : PreambleCoeffInt <= 1'b0;
             -10'sd78  : PreambleCoeffInt <= 1'b1;
             -10'sd77  : PreambleCoeffInt <= 1'b1;
             -10'sd76  : PreambleCoeffInt <= 1'b1;
             -10'sd75  : PreambleCoeffInt <= 1'b0;
             -10'sd74  : PreambleCoeffInt <= 1'b0;
             -10'sd73  : PreambleCoeffInt <= 1'b0;
             -10'sd72  : PreambleCoeffInt <= 1'b0;
             -10'sd71  : PreambleCoeffInt <= 1'b1;
             -10'sd70  : PreambleCoeffInt <= 1'b1;
             -10'sd69  : PreambleCoeffInt <= 1'b0;
             -10'sd68  : PreambleCoeffInt <= 1'b0;
             -10'sd67  : PreambleCoeffInt <= 1'b0;
             -10'sd66  : PreambleCoeffInt <= 1'b0;
             -10'sd65  : PreambleCoeffInt <= 1'b0;
             -10'sd64  : PreambleCoeffInt <= 1'b0;
             -10'sd63  : PreambleCoeffInt <= 1'b1;
             -10'sd62  : PreambleCoeffInt <= 1'b0;
             -10'sd61  : PreambleCoeffInt <= 1'b0;
             -10'sd60  : PreambleCoeffInt <= 1'b0;
             -10'sd59  : PreambleCoeffInt <= 1'b1;
             -10'sd58  : PreambleCoeffInt <= 1'b0;
             -10'sd57  : PreambleCoeffInt <= 1'b0;
             -10'sd56  : PreambleCoeffInt <= 1'b1;
             -10'sd55  : PreambleCoeffInt <= 1'b1;
             -10'sd54  : PreambleCoeffInt <= 1'b1;
             -10'sd53  : PreambleCoeffInt <= 1'b0;
             -10'sd52  : PreambleCoeffInt <= 1'b1;
             -10'sd51  : PreambleCoeffInt <= 1'b0;
             -10'sd50  : PreambleCoeffInt <= 1'b1;
             -10'sd49  : PreambleCoeffInt <= 1'b1;
             -10'sd48  : PreambleCoeffInt <= 1'b0;
             -10'sd47  : PreambleCoeffInt <= 1'b0;
             -10'sd46  : PreambleCoeffInt <= 1'b1;
             -10'sd45  : PreambleCoeffInt <= 1'b0;
             -10'sd44  : PreambleCoeffInt <= 1'b1;
             -10'sd43  : PreambleCoeffInt <= 1'b0;
             -10'sd42  : PreambleCoeffInt <= 1'b0;
             -10'sd41  : PreambleCoeffInt <= 1'b0;
             -10'sd40  : PreambleCoeffInt <= 1'b0;
             -10'sd39  : PreambleCoeffInt <= 1'b0;
             -10'sd38  : PreambleCoeffInt <= 1'b1;
             -10'sd37  : PreambleCoeffInt <= 1'b1;
             -10'sd36  : PreambleCoeffInt <= 1'b0;
             -10'sd35  : PreambleCoeffInt <= 1'b0;
             -10'sd34  : PreambleCoeffInt <= 1'b0;
             -10'sd33  : PreambleCoeffInt <= 1'b1;
             -10'sd32  : PreambleCoeffInt <= 1'b0;
             -10'sd31  : PreambleCoeffInt <= 1'b0;
             -10'sd30  : PreambleCoeffInt <= 1'b1;
             -10'sd29  : PreambleCoeffInt <= 1'b1;
             -10'sd28  : PreambleCoeffInt <= 1'b1;
             -10'sd27  : PreambleCoeffInt <= 1'b0;
             -10'sd26  : PreambleCoeffInt <= 1'b1;
             -10'sd25  : PreambleCoeffInt <= 1'b0;
             -10'sd24  : PreambleCoeffInt <= 1'b0;
             -10'sd23  : PreambleCoeffInt <= 1'b1;
             -10'sd22  : PreambleCoeffInt <= 1'b0;
             -10'sd21  : PreambleCoeffInt <= 1'b0;
             -10'sd20  : PreambleCoeffInt <= 1'b1;
             -10'sd19  : PreambleCoeffInt <= 1'b0;
             -10'sd18  : PreambleCoeffInt <= 1'b1;
             -10'sd17  : PreambleCoeffInt <= 1'b1;
             -10'sd16  : PreambleCoeffInt <= 1'b1;
             -10'sd15  : PreambleCoeffInt <= 1'b0;
             -10'sd14  : PreambleCoeffInt <= 1'b1;
             -10'sd13  : PreambleCoeffInt <= 1'b0;
             -10'sd12  : PreambleCoeffInt <= 1'b1;
             -10'sd11  : PreambleCoeffInt <= 1'b1;
             -10'sd10  : PreambleCoeffInt <= 1'b1;
             -10'sd9   : PreambleCoeffInt <= 1'b1;
             -10'sd8   : PreambleCoeffInt <= 1'b0;
             -10'sd7   : PreambleCoeffInt <= 1'b0;
             -10'sd6   : PreambleCoeffInt <= 1'b0;
             -10'sd5   : PreambleCoeffInt <= 1'b1;
             -10'sd4   : PreambleCoeffInt <= 1'b1;
             -10'sd3   : PreambleCoeffInt <= 1'b0;
             -10'sd2   : PreambleCoeffInt <= 1'b0;
             10'd2     : PreambleCoeffInt <= 1'b0;
             10'd3     : PreambleCoeffInt <= 1'b0;
             10'd4     : PreambleCoeffInt <= 1'b1;
             10'd5     : PreambleCoeffInt <= 1'b1;
             10'd6     : PreambleCoeffInt <= 1'b1;
             10'd7     : PreambleCoeffInt <= 1'b1;
             10'd8     : PreambleCoeffInt <= 1'b1;
             10'd9     : PreambleCoeffInt <= 1'b1;
             10'd10    : PreambleCoeffInt <= 1'b0;
             10'd11    : PreambleCoeffInt <= 1'b1;
             10'd12    : PreambleCoeffInt <= 1'b0;
             10'd13    : PreambleCoeffInt <= 1'b0;
             10'd14    : PreambleCoeffInt <= 1'b1;
             10'd15    : PreambleCoeffInt <= 1'b1;
             10'd16    : PreambleCoeffInt <= 1'b0;
             10'd17    : PreambleCoeffInt <= 1'b0;
             10'd18    : PreambleCoeffInt <= 1'b1;
             10'd19    : PreambleCoeffInt <= 1'b0;
             10'd20    : PreambleCoeffInt <= 1'b1;
             10'd21    : PreambleCoeffInt <= 1'b0;
             10'd22    : PreambleCoeffInt <= 1'b0;
             10'd23    : PreambleCoeffInt <= 1'b1;
             10'd24    : PreambleCoeffInt <= 1'b1;
             10'd25    : PreambleCoeffInt <= 1'b0;
             10'd26    : PreambleCoeffInt <= 1'b1;
             10'd27    : PreambleCoeffInt <= 1'b0;
             10'd28    : PreambleCoeffInt <= 1'b1;
             10'd29    : PreambleCoeffInt <= 1'b1;
             10'd30    : PreambleCoeffInt <= 1'b1;
             10'd31    : PreambleCoeffInt <= 1'b0;
             10'd32    : PreambleCoeffInt <= 1'b0;
             10'd33    : PreambleCoeffInt <= 1'b1;
             10'd34    : PreambleCoeffInt <= 1'b0;
             10'd35    : PreambleCoeffInt <= 1'b0;
             10'd36    : PreambleCoeffInt <= 1'b0;
             10'd37    : PreambleCoeffInt <= 1'b1;
             10'd38    : PreambleCoeffInt <= 1'b0;
             10'd39    : PreambleCoeffInt <= 1'b0;
             10'd40    : PreambleCoeffInt <= 1'b0;
             10'd41    : PreambleCoeffInt <= 1'b0;
             10'd42    : PreambleCoeffInt <= 1'b0;
             10'd43    : PreambleCoeffInt <= 1'b0;
             10'd44    : PreambleCoeffInt <= 1'b0;
             10'd45    : PreambleCoeffInt <= 1'b1;
             10'd46    : PreambleCoeffInt <= 1'b0;
             10'd47    : PreambleCoeffInt <= 1'b1;
             10'd48    : PreambleCoeffInt <= 1'b1;
             10'd49    : PreambleCoeffInt <= 1'b0;
             10'd50    : PreambleCoeffInt <= 1'b1;
             10'd51    : PreambleCoeffInt <= 1'b1;
             10'd52    : PreambleCoeffInt <= 1'b0;
             10'd53    : PreambleCoeffInt <= 1'b1;
             10'd54    : PreambleCoeffInt <= 1'b0;
             10'd55    : PreambleCoeffInt <= 1'b0;
             10'd56    : PreambleCoeffInt <= 1'b0;
             10'd57    : PreambleCoeffInt <= 1'b1;
             10'd58    : PreambleCoeffInt <= 1'b1;
             10'd59    : PreambleCoeffInt <= 1'b0;
             10'd60    : PreambleCoeffInt <= 1'b1;
             10'd61    : PreambleCoeffInt <= 1'b1;
             10'd62    : PreambleCoeffInt <= 1'b1;
             10'd63    : PreambleCoeffInt <= 1'b0;
             10'd64    : PreambleCoeffInt <= 1'b0;
             10'd65    : PreambleCoeffInt <= 1'b1;
             10'd66    : PreambleCoeffInt <= 1'b1;
             10'd67    : PreambleCoeffInt <= 1'b1;
             10'd68    : PreambleCoeffInt <= 1'b1;
             10'd69    : PreambleCoeffInt <= 1'b1;
             10'd70    : PreambleCoeffInt <= 1'b0;
             10'd71    : PreambleCoeffInt <= 1'b1;
             10'd72    : PreambleCoeffInt <= 1'b1;
             10'd73    : PreambleCoeffInt <= 1'b1;
             10'd74    : PreambleCoeffInt <= 1'b1;
             10'd75    : PreambleCoeffInt <= 1'b1;
             10'd76    : PreambleCoeffInt <= 1'b0;
             10'd77    : PreambleCoeffInt <= 1'b0;
             10'd78    : PreambleCoeffInt <= 1'b1;
             10'd79    : PreambleCoeffInt <= 1'b1;
             10'd80    : PreambleCoeffInt <= 1'b1;
             10'd81    : PreambleCoeffInt <= 1'b1;
             10'd82    : PreambleCoeffInt <= 1'b1;
             10'd83    : PreambleCoeffInt <= 1'b0;
             10'd84    : PreambleCoeffInt <= 1'b1;
             10'd85    : PreambleCoeffInt <= 1'b1;
             10'd86    : PreambleCoeffInt <= 1'b0;
             10'd87    : PreambleCoeffInt <= 1'b0;
             10'd88    : PreambleCoeffInt <= 1'b0;
             10'd89    : PreambleCoeffInt <= 1'b1;
             10'd90    : PreambleCoeffInt <= 1'b0;
             10'd91    : PreambleCoeffInt <= 1'b0;
             10'd92    : PreambleCoeffInt <= 1'b0;
             10'd93    : PreambleCoeffInt <= 1'b1;
             10'd94    : PreambleCoeffInt <= 1'b0;
             10'd95    : PreambleCoeffInt <= 1'b1;
             10'd96    : PreambleCoeffInt <= 1'b0;
             10'd97    : PreambleCoeffInt <= 1'b1;
             10'd98    : PreambleCoeffInt <= 1'b1;
             10'd99    : PreambleCoeffInt <= 1'b1;
             10'd100   : PreambleCoeffInt <= 1'b1;
             10'd101   : PreambleCoeffInt <= 1'b1;
             10'd102   : PreambleCoeffInt <= 1'b0;
             10'd103   : PreambleCoeffInt <= 1'b0;
             10'd104   : PreambleCoeffInt <= 1'b0;
             10'd105   : PreambleCoeffInt <= 1'b1;
             10'd106   : PreambleCoeffInt <= 1'b1;
             10'd107   : PreambleCoeffInt <= 1'b1;
             10'd108   : PreambleCoeffInt <= 1'b1;
             10'd109   : PreambleCoeffInt <= 1'b0;
             10'd110   : PreambleCoeffInt <= 1'b1;
             10'd111   : PreambleCoeffInt <= 1'b1;
             10'd112   : PreambleCoeffInt <= 1'b0;
             10'd113   : PreambleCoeffInt <= 1'b0;
             10'd114   : PreambleCoeffInt <= 1'b0;
             10'd115   : PreambleCoeffInt <= 1'b1;
             10'd116   : PreambleCoeffInt <= 1'b0;
             10'd117   : PreambleCoeffInt <= 1'b0;
             10'd118   : PreambleCoeffInt <= 1'b1;
             10'd119   : PreambleCoeffInt <= 1'b1;
             10'd120   : PreambleCoeffInt <= 1'b0;
             10'd121   : PreambleCoeffInt <= 1'b1;
             10'd122   : PreambleCoeffInt <= 1'b0;
             10'd123   : PreambleCoeffInt <= 1'b1;
             10'd124   : PreambleCoeffInt <= 1'b1;
             10'd125   : PreambleCoeffInt <= 1'b1;
             10'd126   : PreambleCoeffInt <= 1'b1;
             10'd127   : PreambleCoeffInt <= 1'b1;
             10'd128   : PreambleCoeffInt <= 1'b1;
             10'd129   : PreambleCoeffInt <= 1'b1;
             10'd130   : PreambleCoeffInt <= 1'b0;
             10'd131   : PreambleCoeffInt <= 1'b0;
             10'd132   : PreambleCoeffInt <= 1'b1;
             10'd133   : PreambleCoeffInt <= 1'b1;
             10'd134   : PreambleCoeffInt <= 1'b1;
             10'd135   : PreambleCoeffInt <= 1'b0;
             10'd136   : PreambleCoeffInt <= 1'b1;
             10'd137   : PreambleCoeffInt <= 1'b1;
             10'd138   : PreambleCoeffInt <= 1'b0;
             10'd139   : PreambleCoeffInt <= 1'b0;
             10'd140   : PreambleCoeffInt <= 1'b0;
             10'd141   : PreambleCoeffInt <= 1'b1;
             10'd142   : PreambleCoeffInt <= 1'b0;
             10'd143   : PreambleCoeffInt <= 1'b1;
             10'd144   : PreambleCoeffInt <= 1'b1;
             10'd145   : PreambleCoeffInt <= 1'b0;
             10'd146   : PreambleCoeffInt <= 1'b1;
             10'd147   : PreambleCoeffInt <= 1'b1;
             10'd148   : PreambleCoeffInt <= 1'b0;
             10'd149   : PreambleCoeffInt <= 1'b1;
             10'd150   : PreambleCoeffInt <= 1'b0;
             10'd151   : PreambleCoeffInt <= 1'b0;
             10'd152   : PreambleCoeffInt <= 1'b0;
             10'd153   : PreambleCoeffInt <= 1'b1;
             10'd154   : PreambleCoeffInt <= 1'b0;
             10'd155   : PreambleCoeffInt <= 1'b1;
             10'd156   : PreambleCoeffInt <= 1'b1;
             10'd157   : PreambleCoeffInt <= 1'b0;
             10'd158   : PreambleCoeffInt <= 1'b0;
             10'd159   : PreambleCoeffInt <= 1'b1;
             10'd160   : PreambleCoeffInt <= 1'b0;
             10'd161   : PreambleCoeffInt <= 1'b1;
             10'd162   : PreambleCoeffInt <= 1'b0;
             10'd163   : PreambleCoeffInt <= 1'b0;
             10'd164   : PreambleCoeffInt <= 1'b0;
             10'd165   : PreambleCoeffInt <= 1'b1;
             10'd166   : PreambleCoeffInt <= 1'b1;
             10'd167   : PreambleCoeffInt <= 1'b0;
             10'd168   : PreambleCoeffInt <= 1'b1;
             10'd169   : PreambleCoeffInt <= 1'b1;
             10'd170   : PreambleCoeffInt <= 1'b1;
             10'd171   : PreambleCoeffInt <= 1'b0;
             10'd172   : PreambleCoeffInt <= 1'b1;
             10'd173   : PreambleCoeffInt <= 1'b1;
             10'd174   : PreambleCoeffInt <= 1'b1;
             10'd175   : PreambleCoeffInt <= 1'b1;
             10'd176   : PreambleCoeffInt <= 1'b1;
             10'd177   : PreambleCoeffInt <= 1'b1;
             10'd178   : PreambleCoeffInt <= 1'b1;
             10'd179   : PreambleCoeffInt <= 1'b0;
             10'd180   : PreambleCoeffInt <= 1'b1;
             10'd181   : PreambleCoeffInt <= 1'b0;
             10'd182   : PreambleCoeffInt <= 1'b0;
             10'd183   : PreambleCoeffInt <= 1'b1;
             10'd184   : PreambleCoeffInt <= 1'b0;
             10'd185   : PreambleCoeffInt <= 1'b0;
             10'd186   : PreambleCoeffInt <= 1'b1;
             10'd187   : PreambleCoeffInt <= 1'b0;
             10'd188   : PreambleCoeffInt <= 1'b1;
             10'd189   : PreambleCoeffInt <= 1'b1;
             10'd190   : PreambleCoeffInt <= 1'b1;
             10'd191   : PreambleCoeffInt <= 1'b0;
             10'd192   : PreambleCoeffInt <= 1'b0;
             10'd193   : PreambleCoeffInt <= 1'b1;
             10'd194   : PreambleCoeffInt <= 1'b0;
             10'd195   : PreambleCoeffInt <= 1'b0;
             10'd196   : PreambleCoeffInt <= 1'b0;
             10'd197   : PreambleCoeffInt <= 1'b1;
             10'd198   : PreambleCoeffInt <= 1'b1;
             10'd199   : PreambleCoeffInt <= 1'b0;
             10'd200   : PreambleCoeffInt <= 1'b0;
             10'd201   : PreambleCoeffInt <= 1'b0;
             10'd202   : PreambleCoeffInt <= 1'b0;
             10'd203   : PreambleCoeffInt <= 1'b0;
             10'd204   : PreambleCoeffInt <= 1'b1;
             10'd205   : PreambleCoeffInt <= 1'b0;
             10'd206   : PreambleCoeffInt <= 1'b1;
             10'd207   : PreambleCoeffInt <= 1'b1;
             10'd208   : PreambleCoeffInt <= 1'b1;
             10'd209   : PreambleCoeffInt <= 1'b1;
             10'd210   : PreambleCoeffInt <= 1'b0;
             10'd211   : PreambleCoeffInt <= 1'b0;
             10'd212   : PreambleCoeffInt <= 1'b1;
             10'd213   : PreambleCoeffInt <= 1'b1;
             10'd214   : PreambleCoeffInt <= 1'b1;
             10'd215   : PreambleCoeffInt <= 1'b1;
             10'd216   : PreambleCoeffInt <= 1'b1;
             10'd217   : PreambleCoeffInt <= 1'b0;
             10'd218   : PreambleCoeffInt <= 1'b1;
             10'd219   : PreambleCoeffInt <= 1'b1;
             10'd220   : PreambleCoeffInt <= 1'b0;
             10'd221   : PreambleCoeffInt <= 1'b0;
             10'd222   : PreambleCoeffInt <= 1'b0;
             10'd223   : PreambleCoeffInt <= 1'b1;
             10'd224   : PreambleCoeffInt <= 1'b0;
             10'd225   : PreambleCoeffInt <= 1'b0;
             10'd226   : PreambleCoeffInt <= 1'b0;
             10'd227   : PreambleCoeffInt <= 1'b1;
             10'd228   : PreambleCoeffInt <= 1'b0;
             10'd229   : PreambleCoeffInt <= 1'b1;
             10'd230   : PreambleCoeffInt <= 1'b0;
             10'd231   : PreambleCoeffInt <= 1'b1;
             10'd232   : PreambleCoeffInt <= 1'b1;
             10'd233   : PreambleCoeffInt <= 1'b1;
             10'd234   : PreambleCoeffInt <= 1'b1;
             10'd235   : PreambleCoeffInt <= 1'b1;
             10'd236   : PreambleCoeffInt <= 1'b0;
             10'd237   : PreambleCoeffInt <= 1'b0;
             10'd238   : PreambleCoeffInt <= 1'b0;
             10'd239   : PreambleCoeffInt <= 1'b1;
             10'd240   : PreambleCoeffInt <= 1'b1;
             10'd241   : PreambleCoeffInt <= 1'b1;
             10'd242   : PreambleCoeffInt <= 1'b1;
             10'd243   : PreambleCoeffInt <= 1'b0;
             10'd244   : PreambleCoeffInt <= 1'b1;
             10'd245   : PreambleCoeffInt <= 1'b1;
             10'd246   : PreambleCoeffInt <= 1'b0;
             10'd247   : PreambleCoeffInt <= 1'b0;
             10'd248   : PreambleCoeffInt <= 1'b0;
             10'd249   : PreambleCoeffInt <= 1'b1;
             10'd250   : PreambleCoeffInt <= 1'b0;
             10'd251   : PreambleCoeffInt <= 1'b0;
             10'd252   : PreambleCoeffInt <= 1'b1;
             10'd253   : PreambleCoeffInt <= 1'b1;
             10'd254   : PreambleCoeffInt <= 1'b0;
             10'd255   : PreambleCoeffInt <= 1'b1;
             10'd256   : PreambleCoeffInt <= 1'b0;
             10'd257   : PreambleCoeffInt <= 1'b1;
             10'd258   : PreambleCoeffInt <= 1'b0;
             10'd259   : PreambleCoeffInt <= 1'b0;
             10'd260   : PreambleCoeffInt <= 1'b0;
             10'd261   : PreambleCoeffInt <= 1'b1;
             10'd262   : PreambleCoeffInt <= 1'b0;
             10'd263   : PreambleCoeffInt <= 1'b1;
             10'd264   : PreambleCoeffInt <= 1'b1;
             10'd265   : PreambleCoeffInt <= 1'b0;
             10'd266   : PreambleCoeffInt <= 1'b0;
             10'd267   : PreambleCoeffInt <= 1'b1;
             10'd268   : PreambleCoeffInt <= 1'b0;
             10'd269   : PreambleCoeffInt <= 1'b1;
             10'd270   : PreambleCoeffInt <= 1'b0;
             10'd271   : PreambleCoeffInt <= 1'b0;
             10'd272   : PreambleCoeffInt <= 1'b0;
             10'd273   : PreambleCoeffInt <= 1'b1;
             10'd274   : PreambleCoeffInt <= 1'b1;
             10'd275   : PreambleCoeffInt <= 1'b0;
             10'd276   : PreambleCoeffInt <= 1'b1;
             10'd277   : PreambleCoeffInt <= 1'b1;
             10'd278   : PreambleCoeffInt <= 1'b1;
             10'd279   : PreambleCoeffInt <= 1'b0;
             10'd280   : PreambleCoeffInt <= 1'b1;
             10'd281   : PreambleCoeffInt <= 1'b1;
             10'd282   : PreambleCoeffInt <= 1'b1;
             10'd283   : PreambleCoeffInt <= 1'b1;
             10'd284   : PreambleCoeffInt <= 1'b1;
             10'd285   : PreambleCoeffInt <= 1'b1;
             10'd286   : PreambleCoeffInt <= 1'b1;
             10'd287   : PreambleCoeffInt <= 1'b0;
             10'd288   : PreambleCoeffInt <= 1'b1;
             10'd289   : PreambleCoeffInt <= 1'b0;
             10'd290   : PreambleCoeffInt <= 1'b0;
             10'd291   : PreambleCoeffInt <= 1'b1;
             10'd292   : PreambleCoeffInt <= 1'b0;
             10'd293   : PreambleCoeffInt <= 1'b0;
             10'd294   : PreambleCoeffInt <= 1'b1;
             10'd295   : PreambleCoeffInt <= 1'b0;
             10'd296   : PreambleCoeffInt <= 1'b1;
             10'd297   : PreambleCoeffInt <= 1'b1;
             10'd298   : PreambleCoeffInt <= 1'b1;
             10'd299   : PreambleCoeffInt <= 1'b0;
             10'd300   : PreambleCoeffInt <= 1'b0;
             10'd301   : PreambleCoeffInt <= 1'b1;
             10'd302   : PreambleCoeffInt <= 1'b0;
             10'd303   : PreambleCoeffInt <= 1'b0;
             10'd304   : PreambleCoeffInt <= 1'b0;
             10'd305   : PreambleCoeffInt <= 1'b1;
             10'd306   : PreambleCoeffInt <= 1'b1;
             10'd307   : PreambleCoeffInt <= 1'b0;
             10'd308   : PreambleCoeffInt <= 1'b0;
             10'd309   : PreambleCoeffInt <= 1'b0;
             10'd310   : PreambleCoeffInt <= 1'b0;
             10'd311   : PreambleCoeffInt <= 1'b0;
             10'd312   : PreambleCoeffInt <= 1'b1;
             10'd313   : PreambleCoeffInt <= 1'b0;
             10'd314   : PreambleCoeffInt <= 1'b0;
             10'd315   : PreambleCoeffInt <= 1'b0;
             10'd316   : PreambleCoeffInt <= 1'b0;
             10'd317   : PreambleCoeffInt <= 1'b0;
             10'd318   : PreambleCoeffInt <= 1'b1;
             10'd319   : PreambleCoeffInt <= 1'b1;
             10'd320   : PreambleCoeffInt <= 1'b0;
             10'd321   : PreambleCoeffInt <= 1'b0;
             10'd322   : PreambleCoeffInt <= 1'b0;
             10'd323   : PreambleCoeffInt <= 1'b0;
             10'd324   : PreambleCoeffInt <= 1'b0;
             10'd325   : PreambleCoeffInt <= 1'b1;
             10'd326   : PreambleCoeffInt <= 1'b0;
             10'd327   : PreambleCoeffInt <= 1'b0;
             10'd328   : PreambleCoeffInt <= 1'b1;
             10'd329   : PreambleCoeffInt <= 1'b1;
             10'd330   : PreambleCoeffInt <= 1'b1;
             10'd331   : PreambleCoeffInt <= 1'b0;
             10'd332   : PreambleCoeffInt <= 1'b1;
             10'd333   : PreambleCoeffInt <= 1'b1;
             10'd334   : PreambleCoeffInt <= 1'b1;
             10'd335   : PreambleCoeffInt <= 1'b0;
             10'd336   : PreambleCoeffInt <= 1'b1;
             10'd337   : PreambleCoeffInt <= 1'b0;
             10'd338   : PreambleCoeffInt <= 1'b1;
             10'd339   : PreambleCoeffInt <= 1'b0;
             10'd340   : PreambleCoeffInt <= 1'b0;
             10'd341   : PreambleCoeffInt <= 1'b0;
             10'd342   : PreambleCoeffInt <= 1'b0;
             10'd343   : PreambleCoeffInt <= 1'b0;
             10'd344   : PreambleCoeffInt <= 1'b1;
             10'd345   : PreambleCoeffInt <= 1'b1;
             10'd346   : PreambleCoeffInt <= 1'b1;
             10'd347   : PreambleCoeffInt <= 1'b0;
             10'd348   : PreambleCoeffInt <= 1'b0;
             10'd349   : PreambleCoeffInt <= 1'b0;
             10'd350   : PreambleCoeffInt <= 1'b0;
             10'd351   : PreambleCoeffInt <= 1'b1;
             10'd352   : PreambleCoeffInt <= 1'b0;
             10'd353   : PreambleCoeffInt <= 1'b0;
             10'd354   : PreambleCoeffInt <= 1'b1;
             10'd355   : PreambleCoeffInt <= 1'b1;
             10'd356   : PreambleCoeffInt <= 1'b1;
             10'd357   : PreambleCoeffInt <= 1'b0;
             10'd358   : PreambleCoeffInt <= 1'b1;
             10'd359   : PreambleCoeffInt <= 1'b1;
             10'd360   : PreambleCoeffInt <= 1'b0;
             10'd361   : PreambleCoeffInt <= 1'b0;
             10'd362   : PreambleCoeffInt <= 1'b1;
             10'd363   : PreambleCoeffInt <= 1'b0;
             10'd364   : PreambleCoeffInt <= 1'b1;
             10'd365   : PreambleCoeffInt <= 1'b0;
             10'd366   : PreambleCoeffInt <= 1'b1;
             10'd367   : PreambleCoeffInt <= 1'b1;
             10'd368   : PreambleCoeffInt <= 1'b1;
             10'd369   : PreambleCoeffInt <= 1'b1;
             10'd370   : PreambleCoeffInt <= 1'b1;
             10'd371   : PreambleCoeffInt <= 1'b1;
             10'd372   : PreambleCoeffInt <= 1'b0;
             10'd373   : PreambleCoeffInt <= 1'b0;
             10'd374   : PreambleCoeffInt <= 1'b1;
             10'd375   : PreambleCoeffInt <= 1'b1;
             10'd376   : PreambleCoeffInt <= 1'b1;
             10'd377   : PreambleCoeffInt <= 1'b0;
             10'd378   : PreambleCoeffInt <= 1'b1;
             10'd379   : PreambleCoeffInt <= 1'b1;
             10'd380   : PreambleCoeffInt <= 1'b0;
             10'd381   : PreambleCoeffInt <= 1'b0;
             10'd382   : PreambleCoeffInt <= 1'b0;
             10'd383   : PreambleCoeffInt <= 1'b1;
             10'd384   : PreambleCoeffInt <= 1'b0;
             10'd385   : PreambleCoeffInt <= 1'b1;
             10'd386   : PreambleCoeffInt <= 1'b1;
             10'd387   : PreambleCoeffInt <= 1'b0;
             10'd388   : PreambleCoeffInt <= 1'b1;
             10'd389   : PreambleCoeffInt <= 1'b1;
             10'd390   : PreambleCoeffInt <= 1'b0;
             10'd391   : PreambleCoeffInt <= 1'b1;
             10'd392   : PreambleCoeffInt <= 1'b0;
             10'd393   : PreambleCoeffInt <= 1'b1;
             10'd394   : PreambleCoeffInt <= 1'b0;
             10'd395   : PreambleCoeffInt <= 1'b1;
             10'd396   : PreambleCoeffInt <= 1'b0;
             10'd397   : PreambleCoeffInt <= 1'b1;
             10'd398   : PreambleCoeffInt <= 1'b1;
             10'd399   : PreambleCoeffInt <= 1'b0;
             10'd400   : PreambleCoeffInt <= 1'b0;
             10'd401   : PreambleCoeffInt <= 1'b1;
             10'd402   : PreambleCoeffInt <= 1'b0;
             10'd403   : PreambleCoeffInt <= 1'b1;
             10'd404   : PreambleCoeffInt <= 1'b0;
             10'd405   : PreambleCoeffInt <= 1'b0;
             10'd406   : PreambleCoeffInt <= 1'b0;
             10'd407   : PreambleCoeffInt <= 1'b1;
             10'd408   : PreambleCoeffInt <= 1'b1;
             10'd409   : PreambleCoeffInt <= 1'b0;
             10'd410   : PreambleCoeffInt <= 1'b1;
             10'd411   : PreambleCoeffInt <= 1'b1;
             10'd412   : PreambleCoeffInt <= 1'b1;
             10'd413   : PreambleCoeffInt <= 1'b0;
             10'd414   : PreambleCoeffInt <= 1'b1;
             10'd415   : PreambleCoeffInt <= 1'b1;
             10'd416   : PreambleCoeffInt <= 1'b1;
             10'd417   : PreambleCoeffInt <= 1'b1;
             10'd418   : PreambleCoeffInt <= 1'b1;
             10'd419   : PreambleCoeffInt <= 1'b1;
             10'd420   : PreambleCoeffInt <= 1'b1;
             10'd421   : PreambleCoeffInt <= 1'b0;
             10'd422   : PreambleCoeffInt <= 1'b1;
             10'd423   : PreambleCoeffInt <= 1'b0;
             10'd424   : PreambleCoeffInt <= 1'b0;
             10'd425   : PreambleCoeffInt <= 1'b1;
             10'd426   : PreambleCoeffInt <= 1'b0;
             10'd427   : PreambleCoeffInt <= 1'b0;
             10'd428   : PreambleCoeffInt <= 1'b1;
             10'd429   : PreambleCoeffInt <= 1'b0;
             10'd430   : PreambleCoeffInt <= 1'b1;
             10'd431   : PreambleCoeffInt <= 1'b1;
             10'd432   : PreambleCoeffInt <= 1'b1;
             10'd433   : PreambleCoeffInt <= 1'b0;
             10'd434   : PreambleCoeffInt <= 1'b0;
             10'd435   : PreambleCoeffInt <= 1'b1;
             10'd436   : PreambleCoeffInt <= 1'b0;
             10'd437   : PreambleCoeffInt <= 1'b0;
             10'd438   : PreambleCoeffInt <= 1'b0;
             10'd439   : PreambleCoeffInt <= 1'b1;
             10'd440   : PreambleCoeffInt <= 1'b1;
             10'd441   : PreambleCoeffInt <= 1'b0;
             10'd442   : PreambleCoeffInt <= 1'b0;
             10'd443   : PreambleCoeffInt <= 1'b0;
             10'd444   : PreambleCoeffInt <= 1'b0;
             10'd445   : PreambleCoeffInt <= 1'b0;
             10'd446   : PreambleCoeffInt <= 1'b1;
             10'd447   : PreambleCoeffInt <= 1'b0;
             10'd448   : PreambleCoeffInt <= 1'b1;
             10'd449   : PreambleCoeffInt <= 1'b1;
             10'd450   : PreambleCoeffInt <= 1'b1;
             10'd451   : PreambleCoeffInt <= 1'b1;
             10'd452   : PreambleCoeffInt <= 1'b0;
             10'd453   : PreambleCoeffInt <= 1'b0;
             10'd454   : PreambleCoeffInt <= 1'b1;
             10'd455   : PreambleCoeffInt <= 1'b1;
             10'd456   : PreambleCoeffInt <= 1'b1;
             10'd457   : PreambleCoeffInt <= 1'b1;
             10'd458   : PreambleCoeffInt <= 1'b1;
             10'd459   : PreambleCoeffInt <= 1'b0;
             10'd460   : PreambleCoeffInt <= 1'b1;
             10'd461   : PreambleCoeffInt <= 1'b1;
             10'd462   : PreambleCoeffInt <= 1'b0;
             10'd463   : PreambleCoeffInt <= 1'b0;
             10'd464   : PreambleCoeffInt <= 1'b0;
             10'd465   : PreambleCoeffInt <= 1'b1;
             10'd466   : PreambleCoeffInt <= 1'b0;
             10'd467   : PreambleCoeffInt <= 1'b0;
             10'd468   : PreambleCoeffInt <= 1'b0;
             10'd469   : PreambleCoeffInt <= 1'b1;
             10'd470   : PreambleCoeffInt <= 1'b0;
             10'd471   : PreambleCoeffInt <= 1'b1;
             10'd472   : PreambleCoeffInt <= 1'b0;
             10'd473   : PreambleCoeffInt <= 1'b1;
             10'd474   : PreambleCoeffInt <= 1'b1;
             10'd475   : PreambleCoeffInt <= 1'b1;
             10'd476   : PreambleCoeffInt <= 1'b1;
             10'd477   : PreambleCoeffInt <= 1'b1;
             10'd478   : PreambleCoeffInt <= 1'b0;
             10'd479   : PreambleCoeffInt <= 1'b0;
             10'd480   : PreambleCoeffInt <= 1'b0;
             10'd481   : PreambleCoeffInt <= 1'b1;
             10'd482   : PreambleCoeffInt <= 1'b1;
             10'd483   : PreambleCoeffInt <= 1'b1;
             10'd484   : PreambleCoeffInt <= 1'b1;
             10'd485   : PreambleCoeffInt <= 1'b0;
             10'd486   : PreambleCoeffInt <= 1'b1;
             10'd487   : PreambleCoeffInt <= 1'b1;
             10'd488   : PreambleCoeffInt <= 1'b0;
             10'd489   : PreambleCoeffInt <= 1'b0;
             10'd490   : PreambleCoeffInt <= 1'b0;
             10'd491   : PreambleCoeffInt <= 1'b1;
             10'd492   : PreambleCoeffInt <= 1'b0;
             10'd493   : PreambleCoeffInt <= 1'b0;
             10'd494   : PreambleCoeffInt <= 1'b1;
             10'd495   : PreambleCoeffInt <= 1'b1;
             10'd496   : PreambleCoeffInt <= 1'b0;
             10'd497   : PreambleCoeffInt <= 1'b1;
             10'd498   : PreambleCoeffInt <= 1'b0;
             10'd499   : PreambleCoeffInt <= 1'b1;
             10'd500   : PreambleCoeffInt <= 1'b0;
             default   : PreambleCoeffInt <= 1'b0; // Null
          endcase // RdAddr
        end //80 MHz HE 4x Preamble
        //80 MHz HE 2x Preamble
        else
        begin
          //Encoding -1 = 1; 1 = 0;
          case (RdAddr)
             -10'sd500 : PreambleCoeffInt <= 1'b0;
             -10'sd498 : PreambleCoeffInt <= 1'b0;
             -10'sd496 : PreambleCoeffInt <= 1'b1;
             -10'sd494 : PreambleCoeffInt <= 1'b0;
             -10'sd492 : PreambleCoeffInt <= 1'b0;
             -10'sd490 : PreambleCoeffInt <= 1'b0;
             -10'sd488 : PreambleCoeffInt <= 1'b1;
             -10'sd486 : PreambleCoeffInt <= 1'b0;
             -10'sd484 : PreambleCoeffInt <= 1'b0;
             -10'sd482 : PreambleCoeffInt <= 1'b0;
             -10'sd480 : PreambleCoeffInt <= 1'b0;
             -10'sd478 : PreambleCoeffInt <= 1'b1;
             -10'sd476 : PreambleCoeffInt <= 1'b1;
             -10'sd474 : PreambleCoeffInt <= 1'b1;
             -10'sd472 : PreambleCoeffInt <= 1'b0;
             -10'sd470 : PreambleCoeffInt <= 1'b1;
             -10'sd468 : PreambleCoeffInt <= 1'b1;
             -10'sd466 : PreambleCoeffInt <= 1'b0;
             -10'sd464 : PreambleCoeffInt <= 1'b0;
             -10'sd462 : PreambleCoeffInt <= 1'b1;
             -10'sd460 : PreambleCoeffInt <= 1'b0;
             -10'sd458 : PreambleCoeffInt <= 1'b0;
             -10'sd456 : PreambleCoeffInt <= 1'b0;
             -10'sd454 : PreambleCoeffInt <= 1'b1;
             -10'sd452 : PreambleCoeffInt <= 1'b1;
             -10'sd450 : PreambleCoeffInt <= 1'b1;
             -10'sd448 : PreambleCoeffInt <= 1'b1;
             -10'sd446 : PreambleCoeffInt <= 1'b0;
             -10'sd444 : PreambleCoeffInt <= 1'b0;
             -10'sd442 : PreambleCoeffInt <= 1'b0;
             -10'sd440 : PreambleCoeffInt <= 1'b0;
             -10'sd438 : PreambleCoeffInt <= 1'b1;
             -10'sd436 : PreambleCoeffInt <= 1'b0;
             -10'sd434 : PreambleCoeffInt <= 1'b0;
             -10'sd432 : PreambleCoeffInt <= 1'b0;
             -10'sd430 : PreambleCoeffInt <= 1'b1;
             -10'sd428 : PreambleCoeffInt <= 1'b0;
             -10'sd426 : PreambleCoeffInt <= 1'b1;
             -10'sd424 : PreambleCoeffInt <= 1'b1;
             -10'sd422 : PreambleCoeffInt <= 1'b0;
             -10'sd420 : PreambleCoeffInt <= 1'b1;
             -10'sd418 : PreambleCoeffInt <= 1'b1;
             -10'sd416 : PreambleCoeffInt <= 1'b1;
             -10'sd414 : PreambleCoeffInt <= 1'b0;
             -10'sd412 : PreambleCoeffInt <= 1'b0;
             -10'sd410 : PreambleCoeffInt <= 1'b1;
             -10'sd408 : PreambleCoeffInt <= 1'b1;
             -10'sd406 : PreambleCoeffInt <= 1'b1;
             -10'sd404 : PreambleCoeffInt <= 1'b0;
             -10'sd402 : PreambleCoeffInt <= 1'b1;
             -10'sd400 : PreambleCoeffInt <= 1'b1;
             -10'sd398 : PreambleCoeffInt <= 1'b0;
             -10'sd396 : PreambleCoeffInt <= 1'b1;
             -10'sd394 : PreambleCoeffInt <= 1'b0;
             -10'sd392 : PreambleCoeffInt <= 1'b0;
             -10'sd390 : PreambleCoeffInt <= 1'b1;
             -10'sd388 : PreambleCoeffInt <= 1'b0;
             -10'sd386 : PreambleCoeffInt <= 1'b0;
             -10'sd384 : PreambleCoeffInt <= 1'b1;
             -10'sd382 : PreambleCoeffInt <= 1'b0;
             -10'sd380 : PreambleCoeffInt <= 1'b1;
             -10'sd378 : PreambleCoeffInt <= 1'b1;
             -10'sd376 : PreambleCoeffInt <= 1'b1;
             -10'sd374 : PreambleCoeffInt <= 1'b0;
             -10'sd372 : PreambleCoeffInt <= 1'b1;
             -10'sd370 : PreambleCoeffInt <= 1'b1;
             -10'sd368 : PreambleCoeffInt <= 1'b1;
             -10'sd366 : PreambleCoeffInt <= 1'b1;
             -10'sd364 : PreambleCoeffInt <= 1'b0;
             -10'sd362 : PreambleCoeffInt <= 1'b1;
             -10'sd360 : PreambleCoeffInt <= 1'b1;
             -10'sd358 : PreambleCoeffInt <= 1'b0;
             -10'sd356 : PreambleCoeffInt <= 1'b0;
             -10'sd354 : PreambleCoeffInt <= 1'b1;
             -10'sd352 : PreambleCoeffInt <= 1'b0;
             -10'sd350 : PreambleCoeffInt <= 1'b0;
             -10'sd348 : PreambleCoeffInt <= 1'b0;
             -10'sd346 : PreambleCoeffInt <= 1'b1;
             -10'sd344 : PreambleCoeffInt <= 1'b1;
             -10'sd342 : PreambleCoeffInt <= 1'b1;
             -10'sd340 : PreambleCoeffInt <= 1'b1;
             -10'sd338 : PreambleCoeffInt <= 1'b0;
             -10'sd336 : PreambleCoeffInt <= 1'b1;
             -10'sd334 : PreambleCoeffInt <= 1'b0;
             -10'sd332 : PreambleCoeffInt <= 1'b0;
             -10'sd330 : PreambleCoeffInt <= 1'b0;
             -10'sd328 : PreambleCoeffInt <= 1'b1;
             -10'sd326 : PreambleCoeffInt <= 1'b0;
             -10'sd324 : PreambleCoeffInt <= 1'b0;
             -10'sd322 : PreambleCoeffInt <= 1'b0;
             -10'sd320 : PreambleCoeffInt <= 1'b0;
             -10'sd318 : PreambleCoeffInt <= 1'b1;
             -10'sd316 : PreambleCoeffInt <= 1'b1;
             -10'sd314 : PreambleCoeffInt <= 1'b1;
             -10'sd312 : PreambleCoeffInt <= 1'b0;
             -10'sd310 : PreambleCoeffInt <= 1'b0;
             -10'sd308 : PreambleCoeffInt <= 1'b0;
             -10'sd306 : PreambleCoeffInt <= 1'b1;
             -10'sd304 : PreambleCoeffInt <= 1'b1;
             -10'sd302 : PreambleCoeffInt <= 1'b0;
             -10'sd300 : PreambleCoeffInt <= 1'b0;
             -10'sd298 : PreambleCoeffInt <= 1'b0;
             -10'sd296 : PreambleCoeffInt <= 1'b1;
             -10'sd294 : PreambleCoeffInt <= 1'b0;
             -10'sd292 : PreambleCoeffInt <= 1'b0;
             -10'sd290 : PreambleCoeffInt <= 1'b1;
             -10'sd288 : PreambleCoeffInt <= 1'b0;
             -10'sd286 : PreambleCoeffInt <= 1'b1;
             -10'sd284 : PreambleCoeffInt <= 1'b1;
             -10'sd282 : PreambleCoeffInt <= 1'b1;
             -10'sd280 : PreambleCoeffInt <= 1'b1;
             -10'sd278 : PreambleCoeffInt <= 1'b0;
             -10'sd276 : PreambleCoeffInt <= 1'b1;
             -10'sd274 : PreambleCoeffInt <= 1'b1;
             -10'sd272 : PreambleCoeffInt <= 1'b1;
             -10'sd270 : PreambleCoeffInt <= 1'b0;
             -10'sd268 : PreambleCoeffInt <= 1'b1;
             -10'sd266 : PreambleCoeffInt <= 1'b0;
             -10'sd264 : PreambleCoeffInt <= 1'b0;
             -10'sd262 : PreambleCoeffInt <= 1'b1;
             -10'sd260 : PreambleCoeffInt <= 1'b0;
             -10'sd258 : PreambleCoeffInt <= 1'b0;
             -10'sd256 : PreambleCoeffInt <= 1'b1;
             -10'sd254 : PreambleCoeffInt <= 1'b1;
             -10'sd252 : PreambleCoeffInt <= 1'b1;
             -10'sd250 : PreambleCoeffInt <= 1'b0;
             -10'sd248 : PreambleCoeffInt <= 1'b1;
             -10'sd246 : PreambleCoeffInt <= 1'b1;
             -10'sd244 : PreambleCoeffInt <= 1'b1;
             -10'sd242 : PreambleCoeffInt <= 1'b0;
             -10'sd240 : PreambleCoeffInt <= 1'b1;
             -10'sd238 : PreambleCoeffInt <= 1'b0;
             -10'sd236 : PreambleCoeffInt <= 1'b0;
             -10'sd234 : PreambleCoeffInt <= 1'b1;
             -10'sd232 : PreambleCoeffInt <= 1'b0;
             -10'sd230 : PreambleCoeffInt <= 1'b0;
             -10'sd228 : PreambleCoeffInt <= 1'b0;
             -10'sd226 : PreambleCoeffInt <= 1'b1;
             -10'sd224 : PreambleCoeffInt <= 1'b1;
             -10'sd222 : PreambleCoeffInt <= 1'b0;
             -10'sd220 : PreambleCoeffInt <= 1'b0;
             -10'sd218 : PreambleCoeffInt <= 1'b0;
             -10'sd216 : PreambleCoeffInt <= 1'b1;
             -10'sd214 : PreambleCoeffInt <= 1'b0;
             -10'sd212 : PreambleCoeffInt <= 1'b0;
             -10'sd210 : PreambleCoeffInt <= 1'b1;
             -10'sd208 : PreambleCoeffInt <= 1'b0;
             -10'sd206 : PreambleCoeffInt <= 1'b1;
             -10'sd204 : PreambleCoeffInt <= 1'b1;
             -10'sd202 : PreambleCoeffInt <= 1'b0;
             -10'sd200 : PreambleCoeffInt <= 1'b1;
             -10'sd198 : PreambleCoeffInt <= 1'b0;
             -10'sd196 : PreambleCoeffInt <= 1'b0;
             -10'sd194 : PreambleCoeffInt <= 1'b0;
             -10'sd192 : PreambleCoeffInt <= 1'b1;
             -10'sd190 : PreambleCoeffInt <= 1'b0;
             -10'sd188 : PreambleCoeffInt <= 1'b0;
             -10'sd186 : PreambleCoeffInt <= 1'b0;
             -10'sd184 : PreambleCoeffInt <= 1'b0;
             -10'sd182 : PreambleCoeffInt <= 1'b1;
             -10'sd180 : PreambleCoeffInt <= 1'b1;
             -10'sd178 : PreambleCoeffInt <= 1'b1;
             -10'sd176 : PreambleCoeffInt <= 1'b0;
             -10'sd174 : PreambleCoeffInt <= 1'b1;
             -10'sd172 : PreambleCoeffInt <= 1'b1;
             -10'sd170 : PreambleCoeffInt <= 1'b0;
             -10'sd168 : PreambleCoeffInt <= 1'b0;
             -10'sd166 : PreambleCoeffInt <= 1'b1;
             -10'sd164 : PreambleCoeffInt <= 1'b0;
             -10'sd162 : PreambleCoeffInt <= 1'b0;
             -10'sd160 : PreambleCoeffInt <= 1'b0;
             -10'sd158 : PreambleCoeffInt <= 1'b1;
             -10'sd156 : PreambleCoeffInt <= 1'b1;
             -10'sd154 : PreambleCoeffInt <= 1'b1;
             -10'sd152 : PreambleCoeffInt <= 1'b1;
             -10'sd150 : PreambleCoeffInt <= 1'b0;
             -10'sd148 : PreambleCoeffInt <= 1'b0;
             -10'sd146 : PreambleCoeffInt <= 1'b0;
             -10'sd144 : PreambleCoeffInt <= 1'b1;
             -10'sd142 : PreambleCoeffInt <= 1'b1;
             -10'sd140 : PreambleCoeffInt <= 1'b1;
             -10'sd138 : PreambleCoeffInt <= 1'b1;
             -10'sd136 : PreambleCoeffInt <= 1'b0;
             -10'sd134 : PreambleCoeffInt <= 1'b1;
             -10'sd132 : PreambleCoeffInt <= 1'b1;
             -10'sd130 : PreambleCoeffInt <= 1'b1;
             -10'sd128 : PreambleCoeffInt <= 1'b0;
             -10'sd126 : PreambleCoeffInt <= 1'b1;
             -10'sd124 : PreambleCoeffInt <= 1'b0;
             -10'sd122 : PreambleCoeffInt <= 1'b0;
             -10'sd120 : PreambleCoeffInt <= 1'b0;
             -10'sd118 : PreambleCoeffInt <= 1'b1;
             -10'sd116 : PreambleCoeffInt <= 1'b1;
             -10'sd114 : PreambleCoeffInt <= 1'b0;
             -10'sd112 : PreambleCoeffInt <= 1'b0;
             -10'sd110 : PreambleCoeffInt <= 1'b0;
             -10'sd108 : PreambleCoeffInt <= 1'b1;
             -10'sd106 : PreambleCoeffInt <= 1'b0;
             -10'sd104 : PreambleCoeffInt <= 1'b0;
             -10'sd102 : PreambleCoeffInt <= 1'b1;
             -10'sd100 : PreambleCoeffInt <= 1'b0;
             -10'sd98  : PreambleCoeffInt <= 1'b1;
             -10'sd96  : PreambleCoeffInt <= 1'b1;
             -10'sd94  : PreambleCoeffInt <= 1'b1;
             -10'sd92  : PreambleCoeffInt <= 1'b1;
             -10'sd90  : PreambleCoeffInt <= 1'b0;
             -10'sd88  : PreambleCoeffInt <= 1'b1;
             -10'sd86  : PreambleCoeffInt <= 1'b1;
             -10'sd84  : PreambleCoeffInt <= 1'b1;
             -10'sd82  : PreambleCoeffInt <= 1'b0;
             -10'sd80  : PreambleCoeffInt <= 1'b1;
             -10'sd78  : PreambleCoeffInt <= 1'b0;
             -10'sd76  : PreambleCoeffInt <= 1'b0;
             -10'sd74  : PreambleCoeffInt <= 1'b1;
             -10'sd72  : PreambleCoeffInt <= 1'b0;
             -10'sd70  : PreambleCoeffInt <= 1'b0;
             -10'sd68  : PreambleCoeffInt <= 1'b1;
             -10'sd66  : PreambleCoeffInt <= 1'b0;
             -10'sd64  : PreambleCoeffInt <= 1'b0;
             -10'sd62  : PreambleCoeffInt <= 1'b1;
             -10'sd60  : PreambleCoeffInt <= 1'b1;
             -10'sd58  : PreambleCoeffInt <= 1'b0;
             -10'sd56  : PreambleCoeffInt <= 1'b1;
             -10'sd54  : PreambleCoeffInt <= 1'b1;
             -10'sd52  : PreambleCoeffInt <= 1'b1;
             -10'sd50  : PreambleCoeffInt <= 1'b0;
             -10'sd48  : PreambleCoeffInt <= 1'b0;
             -10'sd46  : PreambleCoeffInt <= 1'b0;
             -10'sd44  : PreambleCoeffInt <= 1'b0;
             -10'sd42  : PreambleCoeffInt <= 1'b1;
             -10'sd40  : PreambleCoeffInt <= 1'b0;
             -10'sd38  : PreambleCoeffInt <= 1'b1;
             -10'sd36  : PreambleCoeffInt <= 1'b1;
             -10'sd34  : PreambleCoeffInt <= 1'b1;
             -10'sd32  : PreambleCoeffInt <= 1'b0;
             -10'sd30  : PreambleCoeffInt <= 1'b1;
             -10'sd28  : PreambleCoeffInt <= 1'b1;
             -10'sd26  : PreambleCoeffInt <= 1'b1;
             -10'sd24  : PreambleCoeffInt <= 1'b1;
             -10'sd22  : PreambleCoeffInt <= 1'b0;
             -10'sd20  : PreambleCoeffInt <= 1'b0;
             -10'sd18  : PreambleCoeffInt <= 1'b0;
             -10'sd16  : PreambleCoeffInt <= 1'b0;
             -10'sd14  : PreambleCoeffInt <= 1'b1;
             -10'sd12  : PreambleCoeffInt <= 1'b1;
             -10'sd10  : PreambleCoeffInt <= 1'b1;
             -10'sd8   : PreambleCoeffInt <= 1'b0;
             -10'sd6   : PreambleCoeffInt <= 1'b0;
             -10'sd4   : PreambleCoeffInt <= 1'b0;
             -10'sd2   : PreambleCoeffInt <= 1'b0;
             10'd2     : PreambleCoeffInt <= 1'b0;
             10'd4     : PreambleCoeffInt <= 1'b0;
             10'd6     : PreambleCoeffInt <= 1'b1;
             10'd8     : PreambleCoeffInt <= 1'b1;
             10'd10    : PreambleCoeffInt <= 1'b0;
             10'd12    : PreambleCoeffInt <= 1'b0;
             10'd14    : PreambleCoeffInt <= 1'b1;
             10'd16    : PreambleCoeffInt <= 1'b0;
             10'd18    : PreambleCoeffInt <= 1'b1;
             10'd20    : PreambleCoeffInt <= 1'b1;
             10'd22    : PreambleCoeffInt <= 1'b1;
             10'd24    : PreambleCoeffInt <= 1'b0;
             10'd26    : PreambleCoeffInt <= 1'b0;
             10'd28    : PreambleCoeffInt <= 1'b0;
             10'd30    : PreambleCoeffInt <= 1'b0;
             10'd32    : PreambleCoeffInt <= 1'b1;
             10'd34    : PreambleCoeffInt <= 1'b0;
             10'd36    : PreambleCoeffInt <= 1'b0;
             10'd38    : PreambleCoeffInt <= 1'b0;
             10'd40    : PreambleCoeffInt <= 1'b1;
             10'd42    : PreambleCoeffInt <= 1'b0;
             10'd44    : PreambleCoeffInt <= 1'b1;
             10'd46    : PreambleCoeffInt <= 1'b1;
             10'd48    : PreambleCoeffInt <= 1'b1;
             10'd50    : PreambleCoeffInt <= 1'b1;
             10'd52    : PreambleCoeffInt <= 1'b0;
             10'd54    : PreambleCoeffInt <= 1'b0;
             10'd56    : PreambleCoeffInt <= 1'b0;
             10'd58    : PreambleCoeffInt <= 1'b1;
             10'd60    : PreambleCoeffInt <= 1'b0;
             10'd62    : PreambleCoeffInt <= 1'b0;
             10'd64    : PreambleCoeffInt <= 1'b1;
             10'd66    : PreambleCoeffInt <= 1'b1;
             10'd68    : PreambleCoeffInt <= 1'b0;
             10'd70    : PreambleCoeffInt <= 1'b1;
             10'd72    : PreambleCoeffInt <= 1'b1;
             10'd74    : PreambleCoeffInt <= 1'b0;
             10'd76    : PreambleCoeffInt <= 1'b1;
             10'd78    : PreambleCoeffInt <= 1'b1;
             10'd80    : PreambleCoeffInt <= 1'b0;
             10'd82    : PreambleCoeffInt <= 1'b1;
             10'd84    : PreambleCoeffInt <= 1'b0;
             10'd86    : PreambleCoeffInt <= 1'b0;
             10'd88    : PreambleCoeffInt <= 1'b0;
             10'd90    : PreambleCoeffInt <= 1'b1;
             10'd92    : PreambleCoeffInt <= 1'b0;
             10'd94    : PreambleCoeffInt <= 1'b0;
             10'd96    : PreambleCoeffInt <= 1'b0;
             10'd98    : PreambleCoeffInt <= 1'b0;
             10'd100   : PreambleCoeffInt <= 1'b1;
             10'd102   : PreambleCoeffInt <= 1'b0;
             10'd104   : PreambleCoeffInt <= 1'b1;
             10'd106   : PreambleCoeffInt <= 1'b1;
             10'd108   : PreambleCoeffInt <= 1'b0;
             10'd110   : PreambleCoeffInt <= 1'b1;
             10'd112   : PreambleCoeffInt <= 1'b1;
             10'd114   : PreambleCoeffInt <= 1'b1;
             10'd116   : PreambleCoeffInt <= 1'b0;
             10'd118   : PreambleCoeffInt <= 1'b0;
             10'd120   : PreambleCoeffInt <= 1'b1;
             10'd122   : PreambleCoeffInt <= 1'b1;
             10'd124   : PreambleCoeffInt <= 1'b1;
             10'd126   : PreambleCoeffInt <= 1'b0;
             10'd128   : PreambleCoeffInt <= 1'b1;
             10'd130   : PreambleCoeffInt <= 1'b0;
             10'd132   : PreambleCoeffInt <= 1'b0;
             10'd134   : PreambleCoeffInt <= 1'b0;
             10'd136   : PreambleCoeffInt <= 1'b1;
             10'd138   : PreambleCoeffInt <= 1'b0;
             10'd140   : PreambleCoeffInt <= 1'b0;
             10'd142   : PreambleCoeffInt <= 1'b0;
             10'd144   : PreambleCoeffInt <= 1'b0;
             10'd146   : PreambleCoeffInt <= 1'b1;
             10'd148   : PreambleCoeffInt <= 1'b1;
             10'd150   : PreambleCoeffInt <= 1'b1;
             10'd152   : PreambleCoeffInt <= 1'b0;
             10'd154   : PreambleCoeffInt <= 1'b0;
             10'd156   : PreambleCoeffInt <= 1'b0;
             10'd158   : PreambleCoeffInt <= 1'b0;
             10'd160   : PreambleCoeffInt <= 1'b1;
             10'd162   : PreambleCoeffInt <= 1'b1;
             10'd164   : PreambleCoeffInt <= 1'b1;
             10'd166   : PreambleCoeffInt <= 1'b0;
             10'd168   : PreambleCoeffInt <= 1'b1;
             10'd170   : PreambleCoeffInt <= 1'b1;
             10'd172   : PreambleCoeffInt <= 1'b0;
             10'd174   : PreambleCoeffInt <= 1'b0;
             10'd176   : PreambleCoeffInt <= 1'b1;
             10'd178   : PreambleCoeffInt <= 1'b0;
             10'd180   : PreambleCoeffInt <= 1'b0;
             10'd182   : PreambleCoeffInt <= 1'b0;
             10'd184   : PreambleCoeffInt <= 1'b1;
             10'd186   : PreambleCoeffInt <= 1'b1;
             10'd188   : PreambleCoeffInt <= 1'b1;
             10'd190   : PreambleCoeffInt <= 1'b1;
             10'd192   : PreambleCoeffInt <= 1'b0;
             10'd194   : PreambleCoeffInt <= 1'b1;
             10'd196   : PreambleCoeffInt <= 1'b1;
             10'd198   : PreambleCoeffInt <= 1'b1;
             10'd200   : PreambleCoeffInt <= 1'b0;
             10'd202   : PreambleCoeffInt <= 1'b1;
             10'd204   : PreambleCoeffInt <= 1'b0;
             10'd206   : PreambleCoeffInt <= 1'b0;
             10'd208   : PreambleCoeffInt <= 1'b1;
             10'd210   : PreambleCoeffInt <= 1'b0;
             10'd212   : PreambleCoeffInt <= 1'b1;
             10'd214   : PreambleCoeffInt <= 1'b1;
             10'd216   : PreambleCoeffInt <= 1'b0;
             10'd218   : PreambleCoeffInt <= 1'b1;
             10'd220   : PreambleCoeffInt <= 1'b1;
             10'd222   : PreambleCoeffInt <= 1'b1;
             10'd224   : PreambleCoeffInt <= 1'b0;
             10'd226   : PreambleCoeffInt <= 1'b0;
             10'd228   : PreambleCoeffInt <= 1'b1;
             10'd230   : PreambleCoeffInt <= 1'b1;
             10'd232   : PreambleCoeffInt <= 1'b1;
             10'd234   : PreambleCoeffInt <= 1'b0;
             10'd236   : PreambleCoeffInt <= 1'b1;
             10'd238   : PreambleCoeffInt <= 1'b1;
             10'd240   : PreambleCoeffInt <= 1'b0;
             10'd242   : PreambleCoeffInt <= 1'b1;
             10'd244   : PreambleCoeffInt <= 1'b0;
             10'd246   : PreambleCoeffInt <= 1'b0;
             10'd248   : PreambleCoeffInt <= 1'b0;
             10'd250   : PreambleCoeffInt <= 1'b1;
             10'd252   : PreambleCoeffInt <= 1'b0;
             10'd254   : PreambleCoeffInt <= 1'b0;
             10'd256   : PreambleCoeffInt <= 1'b0;
             10'd258   : PreambleCoeffInt <= 1'b1;
             10'd260   : PreambleCoeffInt <= 1'b0;
             10'd262   : PreambleCoeffInt <= 1'b1;
             10'd264   : PreambleCoeffInt <= 1'b0;
             10'd266   : PreambleCoeffInt <= 1'b0;
             10'd268   : PreambleCoeffInt <= 1'b1;
             10'd270   : PreambleCoeffInt <= 1'b0;
             10'd272   : PreambleCoeffInt <= 1'b1;
             10'd274   : PreambleCoeffInt <= 1'b1;
             10'd276   : PreambleCoeffInt <= 1'b1;
             10'd278   : PreambleCoeffInt <= 1'b0;
             10'd280   : PreambleCoeffInt <= 1'b1;
             10'd282   : PreambleCoeffInt <= 1'b1;
             10'd284   : PreambleCoeffInt <= 1'b1;
             10'd286   : PreambleCoeffInt <= 1'b1;
             10'd288   : PreambleCoeffInt <= 1'b0;
             10'd290   : PreambleCoeffInt <= 1'b1;
             10'd292   : PreambleCoeffInt <= 1'b0;
             10'd294   : PreambleCoeffInt <= 1'b0;
             10'd296   : PreambleCoeffInt <= 1'b1;
             10'd298   : PreambleCoeffInt <= 1'b0;
             10'd300   : PreambleCoeffInt <= 1'b0;
             10'd302   : PreambleCoeffInt <= 1'b0;
             10'd304   : PreambleCoeffInt <= 1'b1;
             10'd306   : PreambleCoeffInt <= 1'b1;
             10'd308   : PreambleCoeffInt <= 1'b0;
             10'd310   : PreambleCoeffInt <= 1'b0;
             10'd312   : PreambleCoeffInt <= 1'b0;
             10'd314   : PreambleCoeffInt <= 1'b1;
             10'd316   : PreambleCoeffInt <= 1'b1;
             10'd318   : PreambleCoeffInt <= 1'b1;
             10'd320   : PreambleCoeffInt <= 1'b0;
             10'd322   : PreambleCoeffInt <= 1'b0;
             10'd324   : PreambleCoeffInt <= 1'b0;
             10'd326   : PreambleCoeffInt <= 1'b0;
             10'd328   : PreambleCoeffInt <= 1'b1;
             10'd330   : PreambleCoeffInt <= 1'b0;
             10'd332   : PreambleCoeffInt <= 1'b0;
             10'd334   : PreambleCoeffInt <= 1'b0;
             10'd336   : PreambleCoeffInt <= 1'b1;
             10'd338   : PreambleCoeffInt <= 1'b0;
             10'd340   : PreambleCoeffInt <= 1'b1;
             10'd342   : PreambleCoeffInt <= 1'b1;
             10'd344   : PreambleCoeffInt <= 1'b1;
             10'd346   : PreambleCoeffInt <= 1'b1;
             10'd348   : PreambleCoeffInt <= 1'b0;
             10'd350   : PreambleCoeffInt <= 1'b0;
             10'd352   : PreambleCoeffInt <= 1'b0;
             10'd354   : PreambleCoeffInt <= 1'b1;
             10'd356   : PreambleCoeffInt <= 1'b0;
             10'd358   : PreambleCoeffInt <= 1'b0;
             10'd360   : PreambleCoeffInt <= 1'b1;
             10'd362   : PreambleCoeffInt <= 1'b1;
             10'd364   : PreambleCoeffInt <= 1'b0;
             10'd366   : PreambleCoeffInt <= 1'b1;
             10'd368   : PreambleCoeffInt <= 1'b1;
             10'd370   : PreambleCoeffInt <= 1'b1;
             10'd372   : PreambleCoeffInt <= 1'b1;
             10'd374   : PreambleCoeffInt <= 1'b0;
             10'd376   : PreambleCoeffInt <= 1'b1;
             10'd378   : PreambleCoeffInt <= 1'b1;
             10'd380   : PreambleCoeffInt <= 1'b1;
             10'd382   : PreambleCoeffInt <= 1'b0;
             10'd384   : PreambleCoeffInt <= 1'b1;
             10'd386   : PreambleCoeffInt <= 1'b0;
             10'd388   : PreambleCoeffInt <= 1'b0;
             10'd390   : PreambleCoeffInt <= 1'b1;
             10'd392   : PreambleCoeffInt <= 1'b0;
             10'd394   : PreambleCoeffInt <= 1'b0;
             10'd396   : PreambleCoeffInt <= 1'b1;
             10'd398   : PreambleCoeffInt <= 1'b0;
             10'd400   : PreambleCoeffInt <= 1'b1;
             10'd402   : PreambleCoeffInt <= 1'b1;
             10'd404   : PreambleCoeffInt <= 1'b0;
             10'd406   : PreambleCoeffInt <= 1'b1;
             10'd408   : PreambleCoeffInt <= 1'b1;
             10'd410   : PreambleCoeffInt <= 1'b1;
             10'd412   : PreambleCoeffInt <= 1'b0;
             10'd414   : PreambleCoeffInt <= 1'b0;
             10'd416   : PreambleCoeffInt <= 1'b1;
             10'd418   : PreambleCoeffInt <= 1'b1;
             10'd420   : PreambleCoeffInt <= 1'b1;
             10'd422   : PreambleCoeffInt <= 1'b0;
             10'd424   : PreambleCoeffInt <= 1'b1;
             10'd426   : PreambleCoeffInt <= 1'b1;
             10'd428   : PreambleCoeffInt <= 1'b0;
             10'd430   : PreambleCoeffInt <= 1'b1;
             10'd432   : PreambleCoeffInt <= 1'b0;
             10'd434   : PreambleCoeffInt <= 1'b0;
             10'd436   : PreambleCoeffInt <= 1'b0;
             10'd438   : PreambleCoeffInt <= 1'b1;
             10'd440   : PreambleCoeffInt <= 1'b0;
             10'd442   : PreambleCoeffInt <= 1'b0;
             10'd444   : PreambleCoeffInt <= 1'b0;
             10'd446   : PreambleCoeffInt <= 1'b0;
             10'd448   : PreambleCoeffInt <= 1'b1;
             10'd450   : PreambleCoeffInt <= 1'b1;
             10'd452   : PreambleCoeffInt <= 1'b1;
             10'd454   : PreambleCoeffInt <= 1'b1;
             10'd456   : PreambleCoeffInt <= 1'b0;
             10'd458   : PreambleCoeffInt <= 1'b0;
             10'd460   : PreambleCoeffInt <= 1'b0;
             10'd462   : PreambleCoeffInt <= 1'b1;
             10'd464   : PreambleCoeffInt <= 1'b0;
             10'd466   : PreambleCoeffInt <= 1'b0;
             10'd468   : PreambleCoeffInt <= 1'b1;
             10'd470   : PreambleCoeffInt <= 1'b1;
             10'd472   : PreambleCoeffInt <= 1'b0;
             10'd474   : PreambleCoeffInt <= 1'b1;
             10'd476   : PreambleCoeffInt <= 1'b1;
             10'd478   : PreambleCoeffInt <= 1'b1;
             10'd480   : PreambleCoeffInt <= 1'b0;
             10'd482   : PreambleCoeffInt <= 1'b0;
             10'd484   : PreambleCoeffInt <= 1'b0;
             10'd486   : PreambleCoeffInt <= 1'b0;
             10'd488   : PreambleCoeffInt <= 1'b1;
             10'd490   : PreambleCoeffInt <= 1'b0;
             10'd492   : PreambleCoeffInt <= 1'b0;
             10'd494   : PreambleCoeffInt <= 1'b0;
             10'd496   : PreambleCoeffInt <= 1'b1;
             10'd498   : PreambleCoeffInt <= 1'b0;
             10'd500   : PreambleCoeffInt <= 1'b0;
             default   : PreambleCoeffInt <= 1'b0; // Null
          endcase // RdAddr
        end //80 MHz HE 2x Preamble
      end
      else  // 20 MHz Preamble
      begin
        // 20 MHz L-SIG HE
        if (SymbolType_1t==HE_LSIG)
        begin
          // Encoding -1 = 1; 1 = 0;
          case (RdAddr)
             -10'sd28 : PreambleCoeffInt <= 1'b1; // -28
             -10'sd27 : PreambleCoeffInt <= 1'b1; // -27
             10'd27   : PreambleCoeffInt <= 1'b1; // +27
             default  : PreambleCoeffInt <= 1'b0; // +28
          endcase //RdAddr
        end
        // 20 MHz L/HT/VHT Preamble
        else if (SymbolType_1t==L_LTF || SymbolType_1t==VHT_LTF || SymbolType_1t==HT_LTF)
        begin
          // Encoding -1 = 1; 1 = 0;
          case (RdAddr)
             -10'sd28 : PreambleCoeffInt <= 1'b0; // -28
             -10'sd27 : PreambleCoeffInt <= 1'b0;
             -10'sd26 : PreambleCoeffInt <= 1'b0;
             -10'sd25 : PreambleCoeffInt <= 1'b0;
             -10'sd24 : PreambleCoeffInt <= 1'b1;
             -10'sd23 : PreambleCoeffInt <= 1'b1;
             -10'sd22 : PreambleCoeffInt <= 1'b0;
             -10'sd21 : PreambleCoeffInt <= 1'b0;
             -10'sd20 : PreambleCoeffInt <= 1'b1;
             -10'sd19 : PreambleCoeffInt <= 1'b0;
             -10'sd18 : PreambleCoeffInt <= 1'b1;
             -10'sd17 : PreambleCoeffInt <= 1'b0;
             -10'sd16 : PreambleCoeffInt <= 1'b0;
             -10'sd15 : PreambleCoeffInt <= 1'b0;
             -10'sd14 : PreambleCoeffInt <= 1'b0;
             -10'sd13 : PreambleCoeffInt <= 1'b0;
             -10'sd12 : PreambleCoeffInt <= 1'b0;
             -10'sd11 : PreambleCoeffInt <= 1'b1;
             -10'sd10 : PreambleCoeffInt <= 1'b1;
             -10'sd9  : PreambleCoeffInt <= 1'b0;
             -10'sd8  : PreambleCoeffInt <= 1'b0;
             -10'sd7  : PreambleCoeffInt <= 1'b1;
             -10'sd6  : PreambleCoeffInt <= 1'b0;
             -10'sd5  : PreambleCoeffInt <= 1'b1;
             -10'sd4  : PreambleCoeffInt <= 1'b0;
             -10'sd3  : PreambleCoeffInt <= 1'b0;
             -10'sd2  : PreambleCoeffInt <= 1'b0;
             -10'sd1  : PreambleCoeffInt <= 1'b0; // -1
             10'd1    : PreambleCoeffInt <= 1'b0; // +1
             10'd2    : PreambleCoeffInt <= 1'b1;
             10'd3    : PreambleCoeffInt <= 1'b1;
             10'd4    : PreambleCoeffInt <= 1'b0;
             10'd5    : PreambleCoeffInt <= 1'b0;
             10'd6    : PreambleCoeffInt <= 1'b1;
             10'd7    : PreambleCoeffInt <= 1'b0;
             10'd8    : PreambleCoeffInt <= 1'b1;
             10'd9    : PreambleCoeffInt <= 1'b0;
             10'd10   : PreambleCoeffInt <= 1'b1;
             10'd11   : PreambleCoeffInt <= 1'b1;
             10'd12   : PreambleCoeffInt <= 1'b1;
             10'd13   : PreambleCoeffInt <= 1'b1;
             10'd14   : PreambleCoeffInt <= 1'b1;
             10'd15   : PreambleCoeffInt <= 1'b0;
             10'd16   : PreambleCoeffInt <= 1'b0;
             10'd17   : PreambleCoeffInt <= 1'b1;
             10'd18   : PreambleCoeffInt <= 1'b1;
             10'd19   : PreambleCoeffInt <= 1'b0;
             10'd20   : PreambleCoeffInt <= 1'b1;
             10'd21   : PreambleCoeffInt <= 1'b0;
             10'd22   : PreambleCoeffInt <= 1'b1;
             10'd23   : PreambleCoeffInt <= 1'b0;
             10'd24   : PreambleCoeffInt <= 1'b0;
             10'd25   : PreambleCoeffInt <= 1'b0;
             10'd26   : PreambleCoeffInt <= 1'b0;
             10'd27   : PreambleCoeffInt <= 1'b1;
             10'd28   : PreambleCoeffInt <= 1'b1; // +28
             default  : PreambleCoeffInt <= 1'b0; // Null
          endcase //RdAddr
        end
        // 20 MHz HE 4x Preamble
        else if (SymbolType_1t==HE_4xLTF)
        begin
          // Encoding -1 = 1; 1 = 0;
          case (RdAddr)
            -10'sd122 : PreambleCoeffInt <= 1'b1; // -122
            -10'sd121 : PreambleCoeffInt <= 1'b1;
            -10'sd120 : PreambleCoeffInt <= 1'b0;
            -10'sd119 : PreambleCoeffInt <= 1'b1;
            -10'sd118 : PreambleCoeffInt <= 1'b0;
            -10'sd117 : PreambleCoeffInt <= 1'b1;
            -10'sd116 : PreambleCoeffInt <= 1'b0;
            -10'sd115 : PreambleCoeffInt <= 1'b0;
            -10'sd114 : PreambleCoeffInt <= 1'b0;
            -10'sd113 : PreambleCoeffInt <= 1'b1;
            -10'sd112 : PreambleCoeffInt <= 1'b0;
            -10'sd111 : PreambleCoeffInt <= 1'b0;
            -10'sd110 : PreambleCoeffInt <= 1'b0;
            -10'sd109 : PreambleCoeffInt <= 1'b1;
            -10'sd108 : PreambleCoeffInt <= 1'b1;
            -10'sd107 : PreambleCoeffInt <= 1'b0;
            -10'sd106 : PreambleCoeffInt <= 1'b1;
            -10'sd105 : PreambleCoeffInt <= 1'b1;
            -10'sd104 : PreambleCoeffInt <= 1'b1;
            -10'sd103 : PreambleCoeffInt <= 1'b1;
            -10'sd102 : PreambleCoeffInt <= 1'b1;
            -10'sd101 : PreambleCoeffInt <= 1'b0;
            -10'sd100 : PreambleCoeffInt <= 1'b0;
            -10'sd99  : PreambleCoeffInt <= 1'b1;
            -10'sd98  : PreambleCoeffInt <= 1'b1;
            -10'sd97  : PreambleCoeffInt <= 1'b1;
            -10'sd96  : PreambleCoeffInt <= 1'b1;
            -10'sd95  : PreambleCoeffInt <= 1'b0;
            -10'sd94  : PreambleCoeffInt <= 1'b0;
            -10'sd93  : PreambleCoeffInt <= 1'b1;
            -10'sd92  : PreambleCoeffInt <= 1'b0;
            -10'sd91  : PreambleCoeffInt <= 1'b1;
            -10'sd90  : PreambleCoeffInt <= 1'b0;
            -10'sd89  : PreambleCoeffInt <= 1'b0;
            -10'sd88  : PreambleCoeffInt <= 1'b0;
            -10'sd87  : PreambleCoeffInt <= 1'b0;
            -10'sd86  : PreambleCoeffInt <= 1'b1;
            -10'sd85  : PreambleCoeffInt <= 1'b0;
            -10'sd84  : PreambleCoeffInt <= 1'b1;
            -10'sd83  : PreambleCoeffInt <= 1'b1;
            -10'sd82  : PreambleCoeffInt <= 1'b0;
            -10'sd81  : PreambleCoeffInt <= 1'b0;
            -10'sd80  : PreambleCoeffInt <= 1'b1;
            -10'sd79  : PreambleCoeffInt <= 1'b0;
            -10'sd78  : PreambleCoeffInt <= 1'b0;
            -10'sd77  : PreambleCoeffInt <= 1'b0;
            -10'sd76  : PreambleCoeffInt <= 1'b0;
            -10'sd75  : PreambleCoeffInt <= 1'b1;
            -10'sd74  : PreambleCoeffInt <= 1'b1;
            -10'sd73  : PreambleCoeffInt <= 1'b0;
            -10'sd72  : PreambleCoeffInt <= 1'b1;
            -10'sd71  : PreambleCoeffInt <= 1'b1;
            -10'sd70  : PreambleCoeffInt <= 1'b1;
            -10'sd69  : PreambleCoeffInt <= 1'b0;
            -10'sd68  : PreambleCoeffInt <= 1'b0;
            -10'sd67  : PreambleCoeffInt <= 1'b0;
            -10'sd66  : PreambleCoeffInt <= 1'b0;
            -10'sd65  : PreambleCoeffInt <= 1'b1;
            -10'sd64  : PreambleCoeffInt <= 1'b0;
            -10'sd63  : PreambleCoeffInt <= 1'b0;
            -10'sd62  : PreambleCoeffInt <= 1'b1;
            -10'sd61  : PreambleCoeffInt <= 1'b1;
            -10'sd60  : PreambleCoeffInt <= 1'b1;
            -10'sd59  : PreambleCoeffInt <= 1'b1;
            -10'sd58  : PreambleCoeffInt <= 1'b0;
            -10'sd57  : PreambleCoeffInt <= 1'b1;
            -10'sd56  : PreambleCoeffInt <= 1'b1;
            -10'sd55  : PreambleCoeffInt <= 1'b0;
            -10'sd54  : PreambleCoeffInt <= 1'b0;
            -10'sd53  : PreambleCoeffInt <= 1'b1;
            -10'sd52  : PreambleCoeffInt <= 1'b0;
            -10'sd51  : PreambleCoeffInt <= 1'b1;
            -10'sd50  : PreambleCoeffInt <= 1'b1;
            -10'sd49  : PreambleCoeffInt <= 1'b1;
            -10'sd48  : PreambleCoeffInt <= 1'b1;
            -10'sd47  : PreambleCoeffInt <= 1'b0;
            -10'sd46  : PreambleCoeffInt <= 1'b1;
            -10'sd45  : PreambleCoeffInt <= 1'b0;
            -10'sd44  : PreambleCoeffInt <= 1'b1;
            -10'sd43  : PreambleCoeffInt <= 1'b1;
            -10'sd42  : PreambleCoeffInt <= 1'b1;
            -10'sd41  : PreambleCoeffInt <= 1'b1;
            -10'sd40  : PreambleCoeffInt <= 1'b1;
            -10'sd39  : PreambleCoeffInt <= 1'b1;
            -10'sd38  : PreambleCoeffInt <= 1'b0;
            -10'sd37  : PreambleCoeffInt <= 1'b0;
            -10'sd36  : PreambleCoeffInt <= 1'b1;
            -10'sd35  : PreambleCoeffInt <= 1'b1;
            -10'sd34  : PreambleCoeffInt <= 1'b1;
            -10'sd33  : PreambleCoeffInt <= 1'b1;
            -10'sd32  : PreambleCoeffInt <= 1'b1;
            -10'sd31  : PreambleCoeffInt <= 1'b0;
            -10'sd30  : PreambleCoeffInt <= 1'b1;
            -10'sd29  : PreambleCoeffInt <= 1'b1;
            -10'sd28  : PreambleCoeffInt <= 1'b0;
            -10'sd27  : PreambleCoeffInt <= 1'b0;
            -10'sd26  : PreambleCoeffInt <= 1'b0;
            -10'sd25  : PreambleCoeffInt <= 1'b1;
            -10'sd24  : PreambleCoeffInt <= 1'b0;
            -10'sd23  : PreambleCoeffInt <= 1'b0;
            -10'sd22  : PreambleCoeffInt <= 1'b0;
            -10'sd21  : PreambleCoeffInt <= 1'b1;
            -10'sd20  : PreambleCoeffInt <= 1'b0;
            -10'sd19  : PreambleCoeffInt <= 1'b1;
            -10'sd18  : PreambleCoeffInt <= 1'b0;
            -10'sd17  : PreambleCoeffInt <= 1'b1;
            -10'sd16  : PreambleCoeffInt <= 1'b1;
            -10'sd15  : PreambleCoeffInt <= 1'b1;
            -10'sd14  : PreambleCoeffInt <= 1'b1;
            -10'sd13  : PreambleCoeffInt <= 1'b1;
            -10'sd12  : PreambleCoeffInt <= 1'b0;
            -10'sd11  : PreambleCoeffInt <= 1'b0;
            -10'sd10  : PreambleCoeffInt <= 1'b0;
            -10'sd9   : PreambleCoeffInt <= 1'b1;
            -10'sd8   : PreambleCoeffInt <= 1'b1;
            -10'sd7   : PreambleCoeffInt <= 1'b1;
            -10'sd6   : PreambleCoeffInt <= 1'b0;
            -10'sd5   : PreambleCoeffInt <= 1'b1;
            -10'sd4   : PreambleCoeffInt <= 1'b0;
            -10'sd3   : PreambleCoeffInt <= 1'b0;
            -10'sd2   : PreambleCoeffInt <= 1'b0; // -2
            10'd2     : PreambleCoeffInt <= 1'b1; // +2
            10'd3     : PreambleCoeffInt <= 1'b0;
            10'd4     : PreambleCoeffInt <= 1'b1;
            10'd5     : PreambleCoeffInt <= 1'b0;
            10'd6     : PreambleCoeffInt <= 1'b1;
            10'd7     : PreambleCoeffInt <= 1'b0;
            10'd8     : PreambleCoeffInt <= 1'b0;
            10'd9     : PreambleCoeffInt <= 1'b1;
            10'd10    : PreambleCoeffInt <= 1'b0;
            10'd11    : PreambleCoeffInt <= 1'b0;
            10'd12    : PreambleCoeffInt <= 1'b0;
            10'd13    : PreambleCoeffInt <= 1'b1;
            10'd14    : PreambleCoeffInt <= 1'b1;
            10'd15    : PreambleCoeffInt <= 1'b0;
            10'd16    : PreambleCoeffInt <= 1'b1;
            10'd17    : PreambleCoeffInt <= 1'b1;
            10'd18    : PreambleCoeffInt <= 1'b0;
            10'd19    : PreambleCoeffInt <= 1'b1;
            10'd20    : PreambleCoeffInt <= 1'b0;
            10'd21    : PreambleCoeffInt <= 1'b1;
            10'd22    : PreambleCoeffInt <= 1'b0;
            10'd23    : PreambleCoeffInt <= 1'b0;
            10'd24    : PreambleCoeffInt <= 1'b0;
            10'd25    : PreambleCoeffInt <= 1'b1;
            10'd26    : PreambleCoeffInt <= 1'b0;
            10'd27    : PreambleCoeffInt <= 1'b0;
            10'd28    : PreambleCoeffInt <= 1'b0;
            10'd29    : PreambleCoeffInt <= 1'b1;
            10'd30    : PreambleCoeffInt <= 1'b1;
            10'd31    : PreambleCoeffInt <= 1'b0;
            10'd32    : PreambleCoeffInt <= 1'b1;
            10'd33    : PreambleCoeffInt <= 1'b1;
            10'd34    : PreambleCoeffInt <= 1'b1;
            10'd35    : PreambleCoeffInt <= 1'b1;
            10'd36    : PreambleCoeffInt <= 1'b1;
            10'd37    : PreambleCoeffInt <= 1'b0;
            10'd38    : PreambleCoeffInt <= 1'b0;
            10'd39    : PreambleCoeffInt <= 1'b1;
            10'd40    : PreambleCoeffInt <= 1'b1;
            10'd41    : PreambleCoeffInt <= 1'b1;
            10'd42    : PreambleCoeffInt <= 1'b1;
            10'd43    : PreambleCoeffInt <= 1'b1;
            10'd44    : PreambleCoeffInt <= 1'b1;
            10'd45    : PreambleCoeffInt <= 1'b0;
            10'd46    : PreambleCoeffInt <= 1'b1;
            10'd47    : PreambleCoeffInt <= 1'b0;
            10'd48    : PreambleCoeffInt <= 1'b1;
            10'd49    : PreambleCoeffInt <= 1'b1;
            10'd50    : PreambleCoeffInt <= 1'b1;
            10'd51    : PreambleCoeffInt <= 1'b1;
            10'd52    : PreambleCoeffInt <= 1'b0;
            10'd53    : PreambleCoeffInt <= 1'b1;
            10'd54    : PreambleCoeffInt <= 1'b0;
            10'd55    : PreambleCoeffInt <= 1'b0;
            10'd56    : PreambleCoeffInt <= 1'b1;
            10'd57    : PreambleCoeffInt <= 1'b1;
            10'd58    : PreambleCoeffInt <= 1'b0;
            10'd59    : PreambleCoeffInt <= 1'b1;
            10'd60    : PreambleCoeffInt <= 1'b1;
            10'd61    : PreambleCoeffInt <= 1'b1;
            10'd62    : PreambleCoeffInt <= 1'b1;
            10'd63    : PreambleCoeffInt <= 1'b0;
            10'd64    : PreambleCoeffInt <= 1'b0;
            10'd65    : PreambleCoeffInt <= 1'b1;
            10'd66    : PreambleCoeffInt <= 1'b0;
            10'd67    : PreambleCoeffInt <= 1'b0;
            10'd68    : PreambleCoeffInt <= 1'b0;
            10'd69    : PreambleCoeffInt <= 1'b0;
            10'd70    : PreambleCoeffInt <= 1'b0;
            10'd71    : PreambleCoeffInt <= 1'b0;
            10'd72    : PreambleCoeffInt <= 1'b0;
            10'd73    : PreambleCoeffInt <= 1'b1;
            10'd74    : PreambleCoeffInt <= 1'b0;
            10'd75    : PreambleCoeffInt <= 1'b0;
            10'd76    : PreambleCoeffInt <= 1'b1;
            10'd77    : PreambleCoeffInt <= 1'b1;
            10'd78    : PreambleCoeffInt <= 1'b1;
            10'd79    : PreambleCoeffInt <= 1'b1;
            10'd80    : PreambleCoeffInt <= 1'b0;
            10'd81    : PreambleCoeffInt <= 1'b1;
            10'd82    : PreambleCoeffInt <= 1'b1;
            10'd83    : PreambleCoeffInt <= 1'b0;
            10'd84    : PreambleCoeffInt <= 1'b0;
            10'd85    : PreambleCoeffInt <= 1'b1;
            10'd86    : PreambleCoeffInt <= 1'b0;
            10'd87    : PreambleCoeffInt <= 1'b1;
            10'd88    : PreambleCoeffInt <= 1'b1;
            10'd89    : PreambleCoeffInt <= 1'b1;
            10'd90    : PreambleCoeffInt <= 1'b1;
            10'd91    : PreambleCoeffInt <= 1'b0;
            10'd92    : PreambleCoeffInt <= 1'b1;
            10'd93    : PreambleCoeffInt <= 1'b0;
            10'd94    : PreambleCoeffInt <= 1'b1;
            10'd95    : PreambleCoeffInt <= 1'b1;
            10'd96    : PreambleCoeffInt <= 1'b0;
            10'd97    : PreambleCoeffInt <= 1'b0;
            10'd98    : PreambleCoeffInt <= 1'b0;
            10'd99    : PreambleCoeffInt <= 1'b0;
            10'd100   : PreambleCoeffInt <= 1'b1;
            10'd101   : PreambleCoeffInt <= 1'b1;
            10'd102   : PreambleCoeffInt <= 1'b0;
            10'd103   : PreambleCoeffInt <= 1'b0;
            10'd104   : PreambleCoeffInt <= 1'b0;
            10'd105   : PreambleCoeffInt <= 1'b0;
            10'd106   : PreambleCoeffInt <= 1'b0;
            10'd107   : PreambleCoeffInt <= 1'b1;
            10'd108   : PreambleCoeffInt <= 1'b0;
            10'd109   : PreambleCoeffInt <= 1'b0;
            10'd110   : PreambleCoeffInt <= 1'b1;
            10'd111   : PreambleCoeffInt <= 1'b1;
            10'd112   : PreambleCoeffInt <= 1'b1;
            10'd113   : PreambleCoeffInt <= 1'b0;
            10'd114   : PreambleCoeffInt <= 1'b1;
            10'd115   : PreambleCoeffInt <= 1'b1;
            10'd116   : PreambleCoeffInt <= 1'b1;
            10'd117   : PreambleCoeffInt <= 1'b0;
            10'd118   : PreambleCoeffInt <= 1'b1;
            10'd119   : PreambleCoeffInt <= 1'b0;
            10'd120   : PreambleCoeffInt <= 1'b1;
            10'd121   : PreambleCoeffInt <= 1'b0;
            10'd122   : PreambleCoeffInt <= 1'b0; // +122
            default   : PreambleCoeffInt <= 1'b0; // Null
          endcase //RdAddr
        end
        // 20 MHz HE 2x Preamble
        else if (SymbolType_1t==HE_2xLTF)
        begin
          // Encoding -1 = 1; 1 = 0;
          case (RdAddr)
            -10'sd122 : PreambleCoeffInt <= 1'b1; // -122
            -10'sd120 : PreambleCoeffInt <= 1'b1;
            -10'sd118 : PreambleCoeffInt <= 1'b1;
            -10'sd116 : PreambleCoeffInt <= 1'b0;
            -10'sd114 : PreambleCoeffInt <= 1'b0;
            -10'sd112 : PreambleCoeffInt <= 1'b1;
            -10'sd110 : PreambleCoeffInt <= 1'b0;
            -10'sd108 : PreambleCoeffInt <= 1'b1;
            -10'sd106 : PreambleCoeffInt <= 1'b1;
            -10'sd104 : PreambleCoeffInt <= 1'b1;
            -10'sd102 : PreambleCoeffInt <= 1'b1;
            -10'sd100 : PreambleCoeffInt <= 1'b0;
            -10'sd98  : PreambleCoeffInt <= 1'b1;
            -10'sd96  : PreambleCoeffInt <= 1'b0;
            -10'sd94  : PreambleCoeffInt <= 1'b1;
            -10'sd92  : PreambleCoeffInt <= 1'b1;
            -10'sd90  : PreambleCoeffInt <= 1'b0;
            -10'sd88  : PreambleCoeffInt <= 1'b0;
            -10'sd86  : PreambleCoeffInt <= 1'b1;
            -10'sd84  : PreambleCoeffInt <= 1'b0;
            -10'sd82  : PreambleCoeffInt <= 1'b0;
            -10'sd80  : PreambleCoeffInt <= 1'b0;
            -10'sd78  : PreambleCoeffInt <= 1'b0;
            -10'sd76  : PreambleCoeffInt <= 1'b0;
            -10'sd74  : PreambleCoeffInt <= 1'b1;
            -10'sd72  : PreambleCoeffInt <= 1'b0;
            -10'sd70  : PreambleCoeffInt <= 1'b1;
            -10'sd68  : PreambleCoeffInt <= 1'b0;
            -10'sd66  : PreambleCoeffInt <= 1'b1;
            -10'sd64  : PreambleCoeffInt <= 1'b1;
            -10'sd62  : PreambleCoeffInt <= 1'b0;
            -10'sd60  : PreambleCoeffInt <= 1'b0;
            -10'sd58  : PreambleCoeffInt <= 1'b1;
            -10'sd56  : PreambleCoeffInt <= 1'b0;
            -10'sd54  : PreambleCoeffInt <= 1'b1;
            -10'sd52  : PreambleCoeffInt <= 1'b1;
            -10'sd50  : PreambleCoeffInt <= 1'b1;
            -10'sd48  : PreambleCoeffInt <= 1'b1;
            -10'sd46  : PreambleCoeffInt <= 1'b0;
            -10'sd44  : PreambleCoeffInt <= 1'b1;
            -10'sd42  : PreambleCoeffInt <= 1'b0;
            -10'sd40  : PreambleCoeffInt <= 1'b0;
            -10'sd38  : PreambleCoeffInt <= 1'b0;
            -10'sd36  : PreambleCoeffInt <= 1'b1;
            -10'sd34  : PreambleCoeffInt <= 1'b1;
            -10'sd32  : PreambleCoeffInt <= 1'b0;
            -10'sd30  : PreambleCoeffInt <= 1'b1;
            -10'sd28  : PreambleCoeffInt <= 1'b1;
            -10'sd26  : PreambleCoeffInt <= 1'b1;
            -10'sd24  : PreambleCoeffInt <= 1'b1;
            -10'sd22  : PreambleCoeffInt <= 1'b1;
            -10'sd20  : PreambleCoeffInt <= 1'b0;
            -10'sd18  : PreambleCoeffInt <= 1'b1;
            -10'sd16  : PreambleCoeffInt <= 1'b1;
            -10'sd14  : PreambleCoeffInt <= 1'b1;
            -10'sd12  : PreambleCoeffInt <= 1'b0;
            -10'sd10  : PreambleCoeffInt <= 1'b0;
            -10'sd8   : PreambleCoeffInt <= 1'b0;
            -10'sd6   : PreambleCoeffInt <= 1'b1;
            -10'sd4   : PreambleCoeffInt <= 1'b1;
            -10'sd2   : PreambleCoeffInt <= 1'b0; // -2
            10'd2     : PreambleCoeffInt <= 1'b0; // +2
            10'd4     : PreambleCoeffInt <= 1'b1;
            10'd6     : PreambleCoeffInt <= 1'b0;
            10'd8     : PreambleCoeffInt <= 1'b0;
            10'd10    : PreambleCoeffInt <= 1'b1;
            10'd12    : PreambleCoeffInt <= 1'b0;
            10'd14    : PreambleCoeffInt <= 1'b0;
            10'd16    : PreambleCoeffInt <= 1'b1;
            10'd18    : PreambleCoeffInt <= 1'b0;
            10'd20    : PreambleCoeffInt <= 1'b0;
            10'd22    : PreambleCoeffInt <= 1'b1;
            10'd24    : PreambleCoeffInt <= 1'b1;
            10'd26    : PreambleCoeffInt <= 1'b0;
            10'd28    : PreambleCoeffInt <= 1'b1;
            10'd30    : PreambleCoeffInt <= 1'b0;
            10'd32    : PreambleCoeffInt <= 1'b0;
            10'd34    : PreambleCoeffInt <= 1'b0;
            10'd36    : PreambleCoeffInt <= 1'b0;
            10'd38    : PreambleCoeffInt <= 1'b1;
            10'd40    : PreambleCoeffInt <= 1'b0;
            10'd42    : PreambleCoeffInt <= 1'b1;
            10'd44    : PreambleCoeffInt <= 1'b0;
            10'd46    : PreambleCoeffInt <= 1'b0;
            10'd48    : PreambleCoeffInt <= 1'b1;
            10'd50    : PreambleCoeffInt <= 1'b1;
            10'd52    : PreambleCoeffInt <= 1'b0;
            10'd54    : PreambleCoeffInt <= 1'b1;
            10'd56    : PreambleCoeffInt <= 1'b1;
            10'd58    : PreambleCoeffInt <= 1'b1;
            10'd60    : PreambleCoeffInt <= 1'b1;
            10'd62    : PreambleCoeffInt <= 1'b1;
            10'd64    : PreambleCoeffInt <= 1'b0;
            10'd66    : PreambleCoeffInt <= 1'b1;
            10'd68    : PreambleCoeffInt <= 1'b0;
            10'd70    : PreambleCoeffInt <= 1'b0;
            10'd72    : PreambleCoeffInt <= 1'b1;
            10'd74    : PreambleCoeffInt <= 1'b1;
            10'd76    : PreambleCoeffInt <= 1'b0;
            10'd78    : PreambleCoeffInt <= 1'b0;
            10'd80    : PreambleCoeffInt <= 1'b1;
            10'd82    : PreambleCoeffInt <= 1'b0;
            10'd84    : PreambleCoeffInt <= 1'b1;
            10'd86    : PreambleCoeffInt <= 1'b1;
            10'd88    : PreambleCoeffInt <= 1'b1;
            10'd90    : PreambleCoeffInt <= 1'b1;
            10'd92    : PreambleCoeffInt <= 1'b0;
            10'd94    : PreambleCoeffInt <= 1'b1;
            10'd96    : PreambleCoeffInt <= 1'b0;
            10'd98    : PreambleCoeffInt <= 1'b0;
            10'd100   : PreambleCoeffInt <= 1'b0;
            10'd102   : PreambleCoeffInt <= 1'b1;
            10'd104   : PreambleCoeffInt <= 1'b1;
            10'd106   : PreambleCoeffInt <= 1'b0;
            10'd108   : PreambleCoeffInt <= 1'b1;
            10'd110   : PreambleCoeffInt <= 1'b1;
            10'd112   : PreambleCoeffInt <= 1'b1;
            10'd114   : PreambleCoeffInt <= 1'b1;
            10'd116   : PreambleCoeffInt <= 1'b1;
            10'd118   : PreambleCoeffInt <= 1'b0;
            10'd120   : PreambleCoeffInt <= 1'b1;
            10'd122   : PreambleCoeffInt <= 1'b0; // +122
            default  : PreambleCoeffInt <= 1'b0; // Null
          endcase //RdAddr
        end
        // 20 MHz HE 1x Preamble
        else 
        begin
          // Encoding -1 = 1; 1 = 0;
          case (RdAddr)
            -10'sd120 : PreambleCoeffInt <= 1'b1; // -120
            -10'sd116 : PreambleCoeffInt <= 1'b0;
            -10'sd112 : PreambleCoeffInt <= 1'b0;
            -10'sd108 : PreambleCoeffInt <= 1'b1;
            -10'sd104 : PreambleCoeffInt <= 1'b0;
            -10'sd100 : PreambleCoeffInt <= 1'b1;
            -10'sd96  : PreambleCoeffInt <= 1'b0;
            -10'sd92  : PreambleCoeffInt <= 1'b0;
            -10'sd88  : PreambleCoeffInt <= 1'b0;
            -10'sd84  : PreambleCoeffInt <= 1'b0;
            -10'sd80  : PreambleCoeffInt <= 1'b1;
            -10'sd76  : PreambleCoeffInt <= 1'b1;
            -10'sd72  : PreambleCoeffInt <= 1'b0;
            -10'sd68  : PreambleCoeffInt <= 1'b0;
            -10'sd64  : PreambleCoeffInt <= 1'b0;
            -10'sd60  : PreambleCoeffInt <= 1'b1;
            -10'sd56  : PreambleCoeffInt <= 1'b1;
            -10'sd52  : PreambleCoeffInt <= 1'b1;
            -10'sd48  : PreambleCoeffInt <= 1'b0;
            -10'sd44  : PreambleCoeffInt <= 1'b1;
            -10'sd40  : PreambleCoeffInt <= 1'b1;
            -10'sd36  : PreambleCoeffInt <= 1'b0;
            -10'sd32  : PreambleCoeffInt <= 1'b0;
            -10'sd28  : PreambleCoeffInt <= 1'b1;
            -10'sd24  : PreambleCoeffInt <= 1'b1;
            -10'sd20  : PreambleCoeffInt <= 1'b0;
            -10'sd16  : PreambleCoeffInt <= 1'b1;
            -10'sd12  : PreambleCoeffInt <= 1'b1;
            -10'sd8   : PreambleCoeffInt <= 1'b0;
            -10'sd4   : PreambleCoeffInt <= 1'b1; // -4
            10'd4     : PreambleCoeffInt <= 1'b1; // +4
            10'd8     : PreambleCoeffInt <= 1'b0;
            10'd12    : PreambleCoeffInt <= 1'b0;
            10'd16    : PreambleCoeffInt <= 1'b0;
            10'd20    : PreambleCoeffInt <= 1'b0;
            10'd24    : PreambleCoeffInt <= 1'b0;
            10'd28    : PreambleCoeffInt <= 1'b0;
            10'd32    : PreambleCoeffInt <= 1'b1;
            10'd36    : PreambleCoeffInt <= 1'b1;
            10'd40    : PreambleCoeffInt <= 1'b1;
            10'd44    : PreambleCoeffInt <= 1'b1;
            10'd48    : PreambleCoeffInt <= 1'b1;
            10'd52    : PreambleCoeffInt <= 1'b0;
            10'd56    : PreambleCoeffInt <= 1'b1;
            10'd60    : PreambleCoeffInt <= 1'b1;
            10'd64    : PreambleCoeffInt <= 1'b1;
            10'd68    : PreambleCoeffInt <= 1'b0;
            10'd72    : PreambleCoeffInt <= 1'b1;
            10'd76    : PreambleCoeffInt <= 1'b1;
            10'd80    : PreambleCoeffInt <= 1'b0;
            10'd84    : PreambleCoeffInt <= 1'b1;
            10'd88    : PreambleCoeffInt <= 1'b1;
            10'd92    : PreambleCoeffInt <= 1'b0;
            10'd96    : PreambleCoeffInt <= 1'b1;
            10'd100   : PreambleCoeffInt <= 1'b0;
            10'd104   : PreambleCoeffInt <= 1'b1;
            10'd108   : PreambleCoeffInt <= 1'b1;
            10'd112   : PreambleCoeffInt <= 1'b1;
            10'd116   : PreambleCoeffInt <= 1'b1;
            10'd120   : PreambleCoeffInt <= 1'b1; // +120
            default   : PreambleCoeffInt <= 1'b0; // Null
          endcase //RdAddr
        end
      end // 20 MHz Preamble
    end // nPhyRst == 1'b1
  end // Preamble_Blk

assign PreambleCoeff = PreambleCoeffInt_1t;

// Subcarrier index number for reference
// at the input of Channel Estimation Block
// wire    signed     [8:0]               SCIndChEstIn;
// assign SCIndChEstIn = RdAddr_1t;

// At the input of 1st Rotation Cordic (output of ChEstBlock)
// Delay will be 1 clock.
ChEstDelayLine # (
             .DATAWIDTH(10),
             .LATENCY(1)
             )
              U_SCIndChEstOut_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(RdAddr_2t),
                            .DataOut(SCIndChEstOut)
                            );

assign No_Bypass = 1'b0;

// At the output of Rotation Cordic
// Delay will be bypassed when Cordic Bypass is high
assign {SCCordicEn,SCIndCordic} =
               (!PostRotBypass && !SmoothBypass) ? {SCSmoothFiltOut,SCIndSmoothFiltOut} :
                                                   {FilterOutValid_3t,SCIndChEstOut};

ChEstDelayLine # (
             .DATAWIDTH(10),
             .LATENCY(CORDICLAT)
             )
              U_CordicSCIndex_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(~LUTEnable),
                            .DataIn(SCIndCordic),
                            .DataOut(SCIndCordicOut)
                            );

ChEstDelayLine # (
             .DATAWIDTH(1),
             .LATENCY(CORDICLAT)
             )
          U_CordicEn_DL (
             .PhyClk(PhyClk),
             .nPhyRst(nPhyRst),
             .Bypass(~LUTEnable),
             .DataIn(SCCordicEn),
             .DataOut(RotDataEnOutInt)
             );


// Data Channel Enable Out
always @ (posedge PhyClk or negedge nPhyRst)
   begin: DataChEnOut_Blk
      if (nPhyRst == 1'b0)
         DataChannelEnOut <= 1'b0;
      else if (!EstimEn || SmthDoneP)
         DataChannelEnOut <= 1'b0;
      else if (!PostRotBypass || ((SmoothStepCS == CH_ESTIM_ST) && !SmoothEn
`ifdef RW_MUMIMO_RX_EN
                                                                && !HEMuMIMOEn
`endif // RW_MUMIMO_RX_EN
      ))
         DataChannelEnOut <= 1'b1;
   end //DataChEnOut_Blk

///////////////////////////////////////////////
// Smooth filter control
///////////////////////////////////////////////

// Generate ShiftEn signal for shifting data into smoothing filter
// This should be generated whenever Data from FFT/H is read
// with the delays of FFT/H Memory, Ch Est Block and Pilot averaging
// accounted for.
ChEstDelayLine # (
             .DATAWIDTH(1), // Width of delay line
             .LATENCY(1)    // 1 clocks data delayed
             )
              U_ShiftInInt_1t_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(RdAddrValidSC_2t),
                            .DataOut(ShiftInInt_1t)
                            );

ChEstDelayLine # (
             .DATAWIDTH(1), // Width of delay line
             .LATENCY(3)    // 3 clocks data delayed
             )
              U_ShiftInInt_3t_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(ShiftInInt_1t),
                            .DataOut(ShiftInInt_3t)
                            );

assign ShiftIn = ShiftInInt_3t & (SmoothEn 
`ifdef RW_MUMIMO_RX_EN
                                           | HEMuMIMOEn
`endif // RW_MUMIMO_RX_EN
                                  );

// Generate Pilot Average signals for shifting data into pilot averaging
assign PilotAvDataInValid = ShiftInInt_1t;
assign PilotAvSCIndex     = SCIndChEstOut;
assign PilotAvEn          = (((SymbolType_1t == VHT_LTF) || (SymbolType_1t == HE_4xLTF)) &&
                             (RxNsts != 2'b0) && SmoothEn && (SmoothStep == SMOOTHING)) ? 1'b1 : 1'b0;

// At the output of pilot averaging
ChEstDelayLine # (
             .DATAWIDTH(1),
             .LATENCY(3)
             )
              U_PilotAvDataEnOut_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(PilotAvDataEnIn),
                            .DataOut(PilotAvDataEnOut)
                            );

assign PilotAvDataEnIn = FilterOutValid_3t & ~SmoothBypass;

// Generation of DivByJ signal
// This should be generated for the upper subcarriers in case of 40 MHz HT
assign DivByJ = ((ReceptionMode_1t == MODE_40MHZ_HT) && (SCIndChEstOut >= 10'sd2) && !PreRotBypass) ? 1'b1 : 1'b0;

// Enable Smooth filter
assign SmoothFilterEn = ~SmoothBypass;

// Generate a counter which corresponds to the sub carrier at the filter input
// This is different from SCIndCordicOut.
// It points to the carrier which will be processed 12 clocks later and
// that would be available after a further 5 clocks
// However when Smoothing is disabled, it is convenient to use SCIndCordicOut.

// Start Counting when PilotAvDataEnOut goes high.
// This counter only serves as a reference for calculating
// the subcarrier index that is being processed by the filter
// and corresponding values down the pipeline.
// The SCIndSmoothFiltProc indicates the index that is
// actually being processed by the filter.

always @ (*)
   begin: SCIndSmoothFiltProcInt_Blk
      if (PilotAvDataEnOut && !PilotAvDataEnOut_1t)
      begin
         if (SmoothStep == CHAN_INTERP)
            SCIndSmoothFiltProcInt = FirstNegIndex + 10'd1;
         else if ((SmoothStep != DC_INTERP) || ((SymbolType_1t!=HE_4xLTF) && 
         (ReceptionMode_1t!=MODE_40MHZ_HT) &&
         !((ReceptionMode_1t==MODE_40MHZ_HE_SU) && (SymbolType_1t==HE_2xLTF)) &&
         !((ReceptionMode_1t==MODE_40MHZ_HE_MU) && (SymbolType_1t==HE_2xLTF)) &&
         !((ReceptionMode_1t==MODE_80MHZ_HE_MU) && (SymbolType_1t==HE_2xLTF)) &&
          !((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t == RU26))))
            SCIndSmoothFiltProcInt = FirstNegIndex;
         else
            SCIndSmoothFiltProcInt = LastNegIndex;
      end
      else if (($signed(SCIndSmoothFiltProcInt_1t) >= $signed(LastPosIndex)) || CountDone)
         SCIndSmoothFiltProcInt = 10'sd0;
      else if (SCIndSmoothFiltProcEn && !SmoothBypass)
         SCIndSmoothFiltProcInt = SCIndSmoothFiltProcInt_1t + WrAddrInc;
      else
         SCIndSmoothFiltProcInt = SCIndSmoothFiltProcInt_1t;
   end //SCIndSmoothFiltProcInt_Blk

assign SCIndSmoothFiltProcEn = ((SmoothStep == DC_INTERP) && (ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t != RU242)) ? 
                               ((SymbolType_1t == HE_4xLTF) ? PilotAvDataEnOut_1t | PilotAvDataEnOut_2t | PilotAvDataEnOut_3t | PilotAvDataEnOut_4t | PilotAvDataEnOut_5t : 
                                                           PilotAvDataEnOut_1t | PilotAvDataEnOut_2t) :
                               ((SmoothStep == DC_INTERP) && ((ReceptionMode_1t == MODE_40MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU)) && (SymbolType_1t == HE_4xLTF)) ? PilotAvDataEnOut_1t | PilotAvDataEnOut_3t :                    
                               (SmoothStep == VIRTUAL_INTERP) ? (PilotAvDataEnOut_1t & ~PilotAvDataEnOut_2t) :
                               PilotAvDataEnOut_1t;

assign ReceptionMode_BW = (ReceptionMode == MODE_40MHZ_HT    ||
                           ReceptionMode == MODE_40MHZ_HE_SU ||
                           ReceptionMode == MODE_40MHZ_HE_MU) ? MODE_BW_IS_40MHZ :
                          (ReceptionMode == MODE_80MHZ_HE_MU) ? MODE_BW_IS_80MHZ : MODE_BW_IS_20MHZ;

// Count Done for SCIndSmoothFiltProcInt
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CountDone_Blk
      if (nPhyRst == 1'b0)
        CountDone <= 1'b0;
      else if ((($signed(SCIndSmoothFiltProcInt_1t) >= $signed(LastPosIndex)) && (SCIndSmoothFiltProcInt_1t != 10'b0)) || CounterSkip)
        CountDone <= 1'b1;
      else if (!PilotAvDataEnOut)
        CountDone <= 1'b0;
  end //CountDone_Blk

// Delayed signal version
always @ (posedge PhyClk or negedge nPhyRst)
   begin: DelayReg_Blk
      if (nPhyRst == 1'b0) begin
         ReceptionMode_1t          <= MODE_20MHZ_L;
         ReceptionMode_BW_1t       <= MODE_BW_IS_20MHZ;
         SmoothStep_1t             <= IDLE_STEP;
         SymbolType_1t             <= L_LTF;
         RUIndex_1t                <= 6'b0;
         RUType_1t                 <= RU26;
         SCIndSmoothFiltProcInt_1t <= 10'b0;
         PilotAvDataEnOut_1t       <= 1'b0;
         PilotAvDataEnOut_2t       <= 1'b0;
         PilotAvDataEnOut_3t       <= 1'b0;
         PilotAvDataEnOut_4t       <= 1'b0;
         PilotAvDataEnOut_5t       <= 1'b0;
         FirstNegIndex_1t          <= 10'b0;
         LastNegIndex_1t           <= 10'b0;
         FirstPosIndex_1t          <= 10'b0;
         LastPosIndex_1t           <= 10'b0;
         PilotIndex_1t             <= 10'b0;
         RdAddrInc_1t              <= 10'b0;
      end
      else begin
         ReceptionMode_1t          <= ReceptionMode;
         ReceptionMode_BW_1t       <= ReceptionMode_BW;
         SmoothStep_1t             <= SmoothStep;
         SymbolType_1t             <= SymbolType;
         RUIndex_1t                <= RUIndex;
         RUType_1t                 <= RUType;
         SCIndSmoothFiltProcInt_1t <= SCIndSmoothFiltProcInt;
         PilotAvDataEnOut_1t       <= PilotAvDataEnOut;
         PilotAvDataEnOut_2t       <= PilotAvDataEnOut_1t;
         PilotAvDataEnOut_3t       <= PilotAvDataEnOut_2t;
         PilotAvDataEnOut_4t       <= PilotAvDataEnOut_3t;
         PilotAvDataEnOut_5t       <= PilotAvDataEnOut_4t;
         FirstNegIndex_1t          <= FirstNegIndex;
         LastNegIndex_1t           <= LastNegIndex;
         FirstPosIndex_1t          <= FirstPosIndex;
         LastPosIndex_1t           <= LastPosIndex;
         PilotIndex_1t             <= PilotIndex;
         RdAddrInc_1t              <= RdAddrInc;
      end
  end //DelayReg_Blk

// Subcarrier Index for choosing Smoothing Filter Coeffecients
// This is a 12 clock delayed version of SCIndSmoothFiltProcInt
// It is because the Smoothing Filter has 12 taps.
ChEstDelayLine # (
             .DATAWIDTH(10), // Width of Subcarrier Index
             .LATENCY(2)     // 2 taps delay line
            )
              U_DLDataInTap2_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(SCIndSmoothFiltProcInt),
                            .DataOut(DLDataInTap2)
                            );

ChEstDelayLine # (
             .DATAWIDTH(10), // Width of Subcarrier Index
             .LATENCY(2)     // 2+2 = 4 taps delay line
            )
              U_DLDataInTap4_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(DLDataInTap2),
                            .DataOut(DLDataInTap4)
                            );

ChEstDelayLine # (
             .DATAWIDTH(10), // Width of Subcarrier Index
             .LATENCY(8)     // 4+8 = 12 taps delay line
            )
              U_SCIndSmoothFiltProc_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(DLDataInTap4Mask),
                            .DataOut(SCIndSmoothFiltProc)
                            );

// Mask DLDataInTap4 in case of Channel interpolation avoiding extra toggle
assign DLDataInTap4Mask = DLDataInTap4 & {10{(SmoothStep != CHAN_INTERP)}};

// By pass 12 taps latency for DC or Channel interpolation
assign SCIndSmoothFiltProcMux = ((SmoothStep == DC_INTERP) && !CounterSkip) ? DLDataInTap4 :
                                 (SmoothStep == CHAN_INTERP)                ? DLDataInTap2 :
                                 (SmoothStep == PILOT_INTERP)               ? 10'b0        :  SCIndSmoothFiltProc;

// Subcarrier Index should be a 6 clock delayed version of SCIndSmoothFiltProc for LUTAddr control of Post Cordic in case of 40MHz
ChEstDelayLine # (
             .DATAWIDTH(10), // Width of Subcarrier Index
             .LATENCY(6)     // 6 clocks delay 
            )
              U_SCIndSmoothFiltPreOut_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(SCIndSmoothFiltProcMux),
                            .DataOut(SCIndSmoothFiltPreOut)
                            );

// By pass 6 taps latency for Pilot interpolation
assign SCIndSmoothFiltPreOutMux = (SmoothStep == PILOT_INTERP) ? SCIndSmoothFiltProc : SCIndSmoothFiltPreOut;

// Subcarrier Index should be a 7 clock delayed version of SCIndSmoothFiltProc
ChEstDelayLine # (
             .DATAWIDTH(10), // Width of Subcarrier Index
             .LATENCY(1)     // 1 clock delay for a total of 12+6+1 = 19 latency
            )
              U_SCIndSmoothFiltOut_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(SCIndSmoothFiltPreOutMux),
                            .DataOut(SCIndSmoothFiltOut)
                            );

// Generate SC index to Filter coefficients Selection
assign SCIndFilterCoeff = (SmoothStep == VIRTUAL_INTERP) ? CoeffVirtual :
                          (SmoothStep == PILOT_INTERP)   ? CoeffPilot   : SCIndSmoothFiltProcMux;

// Channel interpolation counter for choosing Smoothing Filter Coeffecients
ChEstDelayLine # (
             .DATAWIDTH(2), // Width of Subcarrier Index
             .LATENCY(8)    // 8 taps delay line
            )
              U_ChanInterpCountCoeff_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(ChanInterpCount),
                            .DataOut(ChanInterpCountCoeff)
                            );

// Filter coefficients Selection
FilterCoeffSelect U_FilterCoeffSelect(
                             //Inputs
                             .PhyClk(             PhyClk),
                             .nPhyRst(            nPhyRst),
                             .CfgSmoothFilterSet( CfgSmoothFilterSet),
                             .RxNsts(             RxNsts),
                             .SymbolType(         SymbolType_1t),
                             .ReceptionMode(      ReceptionMode_1t),
                             .RUType(             RUType_1t),
                             .SCIndex(            SCIndFilterCoeff),
                             .SmoothStep(         SmoothStep_1t),
                             .ChanInterpCount(    ChanInterpCountCoeff),
                             .FirstNegIndex(      FirstNegIndex_1t),
                             .LastPosIndex(       LastPosIndex_1t),
                             .IndexOffset(        RdAddrInc_1t),
                             //
                             .SmoothCoeff0(       SmoothCoeff0),
                             .SmoothCoeff1(       SmoothCoeff1),
                             .SmoothCoeff2(       SmoothCoeff2),
                             .SmoothCoeff3(       SmoothCoeff3),
                             .SmoothCoeff4(       SmoothCoeff4),
                             .SmoothCoeff5(       SmoothCoeff5),
                             .SmoothCoeff6(       SmoothCoeff6),
                             .SmoothCoeff7(       SmoothCoeff7),
                             .SmoothCoeff8(       SmoothCoeff8),
                             .SmoothCoeff9(       SmoothCoeff9),
                             .SmoothCoeff10(      SmoothCoeff10),
                             .SmoothCoeff11(      SmoothCoeff11),
                             .SmoothCoeffRS(      SmoothCoeffRS)
                            );

// Mux for special case of DC interpolation
assign PilotAvDataEnOutMux  = (SmoothStep == DC_INTERP) ? 
                              ((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t != RU242)) ? ((SymbolType_1t==HE_4xLTF) ? (PilotAvDataEnOut | PilotAvDataEnOut_1t | PilotAvDataEnOut_3t | PilotAvDataEnOut_5t) : (PilotAvDataEnOut | PilotAvDataEnOut_1t)) : 
                              (((ReceptionMode_1t == MODE_40MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU)) && (SymbolType_1t==HE_4xLTF)) ? (PilotAvDataEnOut | PilotAvDataEnOut_1t | PilotAvDataEnOut_3t) :
                              (PilotAvDataEnOut | PilotAvDataEnOut_1t) : PilotAvDataEnOut;
assign PilotAvDataEnOutMask = PilotAvDataEnOutMux & ~CounterSkip;

// Filter data input valid after pilot averaging
ChEstDelayLine # (
             .DATAWIDTH(1),
             .LATENCY(9) // 9 clocks Smoothing Filter Latency
            )
              U_PilotAvDataEnOutTap8_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(PilotAvDataEnOutMask),
                            .DataOut(PilotAvDataEnOutTap8)
                            );

ChEstDelayLine # (
             .DATAWIDTH(1),
             .LATENCY(2) // 9+2=11 clocks Smoothing Filter Latency
            )
              U_PilotAvDataEnOutTap10_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(PilotAvDataEnOutTap8),
                            .DataOut(PilotAvDataEnOutTap10)
                            );

// Masking data valid in case of DC or channel interpolation
assign PilotAvDataEnOutTap10Mask = ((SmoothStep == DC_INTERP) || (SmoothStep == CHAN_INTERP)) ? 10'b0 : PilotAvDataEnOutTap10;

ChEstDelayLine # (
             .DATAWIDTH(1),
             .LATENCY(8) // 11+8= 19 clocks Smoothing Filter Latency
            )
              U_PilotAvDataEnOutTap18_DL (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(No_Bypass),
                            .DataIn(PilotAvDataEnOutTap10Mask),
                            .DataOut(PilotAvDataEnOutTap18)
                            );

// Mux for special case of DC interpolation
assign SCSmoothFiltOut = (SmoothStep == DC_INTERP)   ? PilotAvDataEnOutTap10 :
                         (SmoothStep == CHAN_INTERP) ? PilotAvDataEnOutTap8  : PilotAvDataEnOutTap18;

// Generation of MultByJ signal
// This should be generated for the upper subcarriers in case of 40 MHz HT
assign MultByJ = (ReceptionMode_1t == MODE_40MHZ_HT && !PostRotBypass &&
                  (SCIndCordicOut >= 10'sd2) && (SCIndCordicOut <= 10'sd58)) ? 1'b1 : 1'b0;

///////////////////////////////////////////////
// Pilot replacement control
///////////////////////////////////////////////
// Read pilots from buffer
assign PilotRead    = CurrentSCIsPilot & (RxNsts != 2'b0) & ~FrameNDP & (SmoothStep_1t == SMOOTHING || SmoothStep_1t == POST_CORDIC) & (SymbolType_1t == VHT_LTF || ReceptionMode_1t >= MODE_20MHZ_HE_SU);
assign CurrentSCInd = (SmoothStep_1t == POST_CORDIC) ? SCIndCordic : SCIndSmoothFiltOut;

always @*
begin
  case (ReceptionMode_1t)
      MODE_20MHZ_HT : 
      begin
        // Pilot subcarriers (21, 7)
        if ((CurrentSCInd == -10'sd7)  || (CurrentSCInd == 10'sd7) ||
            (CurrentSCInd == -10'sd21) || (CurrentSCInd == 10'sd21))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_40MHZ_HT : 
      begin
        // Pilot subcarriers (53, 25, 11)
        if ((CurrentSCInd == -10'sd11) || (CurrentSCInd == 10'sd11) || 
            (CurrentSCInd == -10'sd25) || (CurrentSCInd == 10'sd25) || 
            (CurrentSCInd == -10'sd53) || (CurrentSCInd == 10'sd53))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_40MHZ_HE_SU : 
      begin
        // Pilot subcarriers (238, 212, 170, 144, 104, 78, 36, 10)
        // Pilot subcarriers for 1x (212, 144, 104, 36)
        if ((CurrentSCInd == -10'sd10 && SymbolType_1t != HE_1xLTF)  || (CurrentSCInd == 10'sd10 && SymbolType_1t != HE_1xLTF)  ||
            (CurrentSCInd == -10'sd36)  || (CurrentSCInd == 10'sd36)  ||
            (CurrentSCInd == -10'sd78 && SymbolType_1t != HE_1xLTF)  || (CurrentSCInd == 10'sd78 && SymbolType_1t != HE_1xLTF)  ||
            (CurrentSCInd == -10'sd104) || (CurrentSCInd == 10'sd104) ||
            (CurrentSCInd == -10'sd144) || (CurrentSCInd == 10'sd144) ||
            (CurrentSCInd == -10'sd170 && SymbolType_1t != HE_1xLTF) || (CurrentSCInd == 10'sd170 && SymbolType_1t != HE_1xLTF) ||
            (CurrentSCInd == -10'sd212) || (CurrentSCInd == 10'sd212) ||
            (CurrentSCInd == -10'sd238 && SymbolType_1t != HE_1xLTF) || (CurrentSCInd == 10'sd238 && SymbolType_1t != HE_1xLTF))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_40MHZ_HE_MU : 
      begin
        // Pilot subcarriers (238, 224, 212, 198, 184, 170, 158, 144,
        // 130, 116, 104, 90, 78, 64, 50, 36, 24, 10)
        if ((CurrentSCInd == -10'sd10)  || (CurrentSCInd == 10'sd10)  ||
            (CurrentSCInd == -10'sd24 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd24 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd36)  || (CurrentSCInd == 10'sd36)  ||
            (CurrentSCInd == -10'sd50 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd50 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd64 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd64 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd78)  || (CurrentSCInd == 10'sd78)  ||
            (CurrentSCInd == -10'sd90 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd90 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd104) || (CurrentSCInd == 10'sd104) ||
            (CurrentSCInd == -10'sd116 && RUType_1t < RU106) || (CurrentSCInd == 10'sd116 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd130 && RUType_1t < RU106) || (CurrentSCInd == 10'sd130 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd144) || (CurrentSCInd == 10'sd144) ||
            (CurrentSCInd == -10'sd158 && RUType_1t < RU106) || (CurrentSCInd == 10'sd158 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd170) || (CurrentSCInd == 10'sd170) ||
            (CurrentSCInd == -10'sd184 && RUType_1t < RU106) || (CurrentSCInd == 10'sd184 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd198 && RUType_1t < RU106) || (CurrentSCInd == 10'sd198 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd212) || (CurrentSCInd == 10'sd212) ||
            (CurrentSCInd == -10'sd224 && RUType_1t < RU106) || (CurrentSCInd == 10'sd224 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd238) || (CurrentSCInd == 10'sd238))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_80MHZ_HE_MU : 
      begin
        // Pilot subcarriers (494, 480, 468, 454, 440, 426,
        // 414, 400, 386, 372, 360, 346, 334, 320,
        // 306, 292, 280, 266, 252,
        // 238, 226, 212, 198, 184, 172, 158, 144,
        // 130, 118, 104, 92, 78, 64, 50, 38, 24, 10)
        if ((CurrentSCInd == -10'sd10 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd10 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd24)  || (CurrentSCInd == 10'sd24)  ||
            (CurrentSCInd == -10'sd38 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd38 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd50)  || (CurrentSCInd == 10'sd50)  ||
            (CurrentSCInd == -10'sd64 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd64 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd78 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd78 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd92)  || (CurrentSCInd == 10'sd92)  ||
            (CurrentSCInd == -10'sd104 && RUType_1t < RU106) || (CurrentSCInd == 10'sd104 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd118) || (CurrentSCInd == 10'sd118) ||
            (CurrentSCInd == -10'sd130 && RUType_1t < RU106) || (CurrentSCInd == 10'sd130 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd144 && RUType_1t < RU106) || (CurrentSCInd == 10'sd144 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd158 && RUType_1t < RU106) || (CurrentSCInd == 10'sd158 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd172 && RUType_1t < RU106) || (CurrentSCInd == 10'sd172 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd184) || (CurrentSCInd == 10'sd184) ||
            (CurrentSCInd == -10'sd198 && RUType_1t < RU106) || (CurrentSCInd == 10'sd198 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd212 && RUType_1t < RU106) || (CurrentSCInd == 10'sd212 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd226) || (CurrentSCInd == 10'sd226) ||
            (CurrentSCInd == -10'sd238 && RUType_1t < RU106) || (CurrentSCInd == 10'sd238 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd252) || (CurrentSCInd == 10'sd252) ||
            (CurrentSCInd == -10'sd266) || (CurrentSCInd == 10'sd266) ||
            (CurrentSCInd == -10'sd280 && RUType_1t < RU106) || (CurrentSCInd == 10'sd280 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd292) || (CurrentSCInd == 10'sd292) ||
            (CurrentSCInd == -10'sd306 && RUType_1t < RU106) || (CurrentSCInd == 10'sd306 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd320 && RUType_1t < RU106) || (CurrentSCInd == 10'sd320 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd334) || (CurrentSCInd == 10'sd334) ||
            (CurrentSCInd == -10'sd346 && RUType_1t < RU106) || (CurrentSCInd == 10'sd346 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd360) || (CurrentSCInd == 10'sd360) ||
            (CurrentSCInd == -10'sd372 && RUType_1t < RU106) || (CurrentSCInd == 10'sd372 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd386 && RUType_1t < RU106) || (CurrentSCInd == 10'sd386 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd400) || (CurrentSCInd == 10'sd400) ||
            (CurrentSCInd == -10'sd414 && RUType_1t < RU106) || (CurrentSCInd == 10'sd414 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd426) || (CurrentSCInd == 10'sd426) ||
            (CurrentSCInd == -10'sd440 && RUType_1t < RU106) || (CurrentSCInd == 10'sd440 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd454 && RUType_1t < RU106) || (CurrentSCInd == 10'sd454 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd468) || (CurrentSCInd == 10'sd468) ||
            (CurrentSCInd == -10'sd480 && RUType_1t < RU106) || (CurrentSCInd == 10'sd480 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd494) || (CurrentSCInd == 10'sd494))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_20MHZ_HE_SU : 
      begin
        // Pilot subcarriers (116, 90, 48, 22)
        if ((CurrentSCInd == -10'sd22 && SymbolType_1t!=HE_1xLTF)  || (CurrentSCInd == 10'sd22 && SymbolType_1t!=HE_1xLTF)  ||
            (CurrentSCInd == -10'sd48)  || (CurrentSCInd == 10'sd48) ||
            (CurrentSCInd == -10'sd90 && SymbolType_1t!=HE_1xLTF)  || (CurrentSCInd == 10'sd90 && SymbolType_1t!=HE_1xLTF)  ||
            (CurrentSCInd == -10'sd116) || (CurrentSCInd == 10'sd116))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_20MHZ_HE_MU : 
      begin
        // Pilot subcarriers (116, 102, 90, 76, 62, 48, 36, 22, 10)
        if ((CurrentSCInd == -10'sd10 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd10 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd22)  || (CurrentSCInd == 10'sd22)  ||
            (CurrentSCInd == -10'sd36 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd36 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd48)  || (CurrentSCInd == 10'sd48)  ||
            (CurrentSCInd == -10'sd62 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd62 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd76 && RUType_1t < RU106)  || (CurrentSCInd == 10'sd76 && RUType_1t < RU106)  ||
            (CurrentSCInd == -10'sd90)  || (CurrentSCInd == 10'sd90)  ||
            (CurrentSCInd == -10'sd102 && RUType_1t < RU106) || (CurrentSCInd == 10'sd102 && RUType_1t < RU106) ||
            (CurrentSCInd == -10'sd116) || (CurrentSCInd == 10'sd116))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      default :
          CurrentSCIsPilot = 1'b0;
  endcase
end      


///////////////////////////////////////////////
// HWrite control
///////////////////////////////////////////////

// Clear DC value for equalyzer
assign DCClear = ((HWriteSCInd == 10'sd0) || 
                 ((((ReceptionMode_1t==MODE_20MHZ_HE_SU) || (ReceptionMode_1t == MODE_20MHZ_HE_MU)
                  || (ReceptionMode_1t == MODE_40MHZ_HT)
                  || (ReceptionMode_1t == MODE_40MHZ_HE_SU)
                  || (ReceptionMode_1t == MODE_40MHZ_HE_MU)
                  || (ReceptionMode_1t == MODE_80MHZ_HE_MU)) &&
                  (HWriteSCInd == 10'sd1 || HWriteSCInd == -10'sd1))
                  || (((ReceptionMode_1t == MODE_40MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU)) && 
                  (HWriteSCInd == 10'sd2 || HWriteSCInd == -10'sd2))
                  || (SpecialRU26Idx5 && (HWriteSCInd == 10'sd2 || HWriteSCInd == -10'sd2 || HWriteSCInd == 10'sd3 || HWriteSCInd == -10'sd3))
                  )) & (~PostRotBypass | ~SmoothEn) & RotDataEnOutInt;

// Muxing control to HWriteCtrl
assign HWriteSCInd     = (SmoothBypass || (!PostRotBypass && !SmoothBypass)) ? SCIndCordicOut  : 
                         (CounterSkip)  ? ((SmoothStep == DC_INTERP) ? FirstNegIndex_1t : (SmoothStep == VIRTUAL_INTERP) ? VirtualIndex : PilotIndex_1t) :
                                          SCIndSmoothFiltOut;

assign HWriteDataEnOut = (SmoothBypass || (!PostRotBypass && !SmoothBypass)) ? RotDataEnOutInt : 
                         (CounterSkip)  ? ((SCIndSmoothFiltOut == FirstNegIndex_1t) && (AddrGenCS != IDLE)) :
                                          SCSmoothFiltOut;

//pragma coverage off
// Assigning Strings to States so that it would be easy to debug
`ifdef RW_SIMU_ON
always@(*)
begin
  case (SymbolType)
    L_LTF    : SymbolTypeStr = {"L_LTF"};
    VHT_LTF  : SymbolTypeStr = {"VHT_LTF"};
    HT_LTF   : SymbolTypeStr = {"HT_LTF"};
    HE_1xLTF : SymbolTypeStr = {"HE_1xLTF"};
    HE_2xLTF : SymbolTypeStr = {"HE_2xLTF"};
    HE_4xLTF : SymbolTypeStr = {"HE_4xLTF"};
    HE_LSIG  : SymbolTypeStr = {"HE_LSIG"};
    default  : SymbolTypeStr = {"UNKNOWN"};
  endcase
end

always@(*)
begin
  case (ReceptionMode)
    MODE_20MHZ_L     : ReceptionModeStr = {"MODE_20MHZ_L"};
    MODE_20MHZ_HT    : ReceptionModeStr = {"MODE_20MHZ_HT"};
    MODE_40MHZ_HT    : ReceptionModeStr = {"MODE_40MHZ_HT"};
    MODE_20MHZ_HE_SU : ReceptionModeStr = {"MODE_20MHZ_HE_SU"};
    MODE_20MHZ_HE_MU : ReceptionModeStr = {"MODE_20MHZ_HE_MU"};
    MODE_40MHZ_HE_SU : ReceptionModeStr = {"MODE_40MHZ_HE_SU"};
    MODE_40MHZ_HE_MU : ReceptionModeStr = {"MODE_40MHZ_HE_MU"};
    MODE_80MHZ_HE_SU : ReceptionModeStr = {"MODE_80MHZ_HE_SU"};
    MODE_80MHZ_HE_MU : ReceptionModeStr = {"MODE_80MHZ_HE_MU"};
    default          : ReceptionModeStr = {"UNKNOWN"};
  endcase
end

always@(*)
begin
  case (AddrGenCS)
    IDLE        : AddrGenCSStr = {"IDLE"};
    COUNTNEG1   : AddrGenCSStr = {"COUNTNEG1"};
    WAITFIRST   : AddrGenCSStr = {"WAITFIRST"};
    COUNTNEG2   : AddrGenCSStr = {"COUNTNEG2"};
    COUNTPOS1   : AddrGenCSStr = {"COUNTPOS1"};
    COUNTPOS2   : AddrGenCSStr = {"COUNTPOS2"};
    EMPTYCORDIC : AddrGenCSStr = {"EMPTYCORDIC"};
    DC_MODE     : AddrGenCSStr = {"DC_MODE"};
    WAITLASTWR  : AddrGenCSStr = {"WAITLASTWR"};
    DONE        : AddrGenCSStr = {"DONE"};
    default     : AddrGenCSStr = {"UNKNOWN"};
  endcase
end

always@(*)
begin
  case (AddrGenNS)
    IDLE        : AddrGenNSStr = {"IDLE"};
    COUNTNEG1   : AddrGenNSStr = {"COUNTNEG1"};
    WAITFIRST   : AddrGenNSStr = {"WAITFIRST"};
    COUNTNEG2   : AddrGenNSStr = {"COUNTNEG2"};
    COUNTPOS1   : AddrGenNSStr = {"COUNTPOS1"};
    COUNTPOS2   : AddrGenNSStr = {"COUNTPOS2"};
    EMPTYCORDIC : AddrGenNSStr = {"EMPTYCORDIC"};
    DC_MODE     : AddrGenNSStr = {"DC_MODE"};
    WAITLASTWR  : AddrGenNSStr = {"WAITLASTWR"};
    DONE        : AddrGenNSStr = {"DONE"};
    default     : AddrGenNSStr = {"UNKNOWN"};
  endcase
end

always@(*)
begin
  case (SmoothStepCS)
    IDLE_ST        : SmoothStepCSStr = {"IDLE_ST"};
    CH_ESTIM_ST    : SmoothStepCSStr = {"CH_ESTIM_ST"};
    PRE_CORDIC_ST  : SmoothStepCSStr = {"PRE_CORDIC_ST"};
    PILOT_I_ST     : SmoothStepCSStr = {"PILOT_I_ST"};
    DC_I_ST        : SmoothStepCSStr = {"DC_I_ST"};
`ifdef RW_NX_DERIV_CHBW20ONLY
    PILOT_Q_ST     : SmoothStepCSStr = {"PILOT_Q_ST"};
    DC_Q_ST        : SmoothStepCSStr = {"DC_Q_ST"};
    VIRTUAL_Q_ST   : SmoothStepCSStr = {"VIRTUAL_Q_ST"};
    SMOOTH_Q_ST    : SmoothStepCSStr = {"SMOOTH_Q_ST"};
    CH_INTERP_Q_ST : SmoothStepCSStr = {"CH_INTERP_Q_ST"};
`endif
    VIRTUAL_I_ST   : SmoothStepCSStr = {"VIRTUAL_I_ST"};
    SMOOTH_I_ST    : SmoothStepCSStr = {"SMOOTH_I_ST"};
    CH_INTERP_I_ST : SmoothStepCSStr = {"CH_INTERP_I_ST"};
    POST_CORDIC_ST : SmoothStepCSStr = {"POST_CORDIC_ST"};
    END_ST         : SmoothStepCSStr = {"END_ST"};
    default        : SmoothStepCSStr = {"UNKNOWN"};
  endcase
end

always@(*)
begin
  case (SmoothStep)
    IDLE_STEP      : SmoothStepStr = {"IDLE_STEP"};
    SMOOTHING      : SmoothStepStr = {"SMOOTHING"};
    DC_INTERP      : SmoothStepStr = {"DC_INTERP"};
    VIRTUAL_INTERP : SmoothStepStr = {"VIRTUAL_INTERP"};
    PILOT_INTERP   : SmoothStepStr = {"PILOT_INTERP"};
    CHAN_INTERP    : SmoothStepStr = {"CHAN_INTERP"};
    POST_CORDIC    : SmoothStepStr = {"POST_CORDIC"};
    default        : SmoothStepStr = {"UNKNOWN"};
  endcase
end

`endif
//pragma coverage on
endmodule // EstimRotCtrl

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
