//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39777 $
// $Date: 2019-09-12 18:24:11 +0200 (Thu, 12 Sep 2019) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Filter Coeffecient Selection Module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/FilterCoeffSelect.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module FilterCoeffSelect (

            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input   wire                                  nPhyRst,  // Active Low Reset
            input   wire                                  PhyClk,   // Phy Clock

            ///////////////////////////////////////////////
            // Config register
            ///////////////////////////////////////////////
            input   wire              [1:0]               CfgSmoothFilterSet, // Smooth Filter Set
            
            ///////////////////////////////////////////////
            // Controls from Rx FD FSM
            ///////////////////////////////////////////////
            input   wire              [1:0]               RxNsts,             // No. of Space Time Steams
            input   wire              [2:0]               SymbolType,         // Type of Symbol
            input   wire              [3:0]               ReceptionMode,      // Mode of Current Pkt
            input   wire              [2:0]               RUType,             // Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)

            ///////////////////////////////////////////////
            // Controls from EstimRotCtrl
            ///////////////////////////////////////////////
            input   wire   signed     [9:0]               SCIndex,            // Sub Carrier Index
            input   wire              [2:0]               SmoothStep,         // Smoothing steps
            input   wire              [1:0]               ChanInterpCount,    // Channel interpolation counter
            input   wire   signed     [9:0]               FirstNegIndex,      // First Sub Carrier Index
            input   wire   signed     [9:0]               LastPosIndex,       // Last Sub Carrier Index
            input   wire              [9:0]               IndexOffset,        // Index offset

            ///////////////////////////////////////////////
            // Filter Coeffecients
            ///////////////////////////////////////////////
            output  reg signed     [9:0]                  SmoothCoeff0,  // Smoothing Coeffecient 0
            output  reg signed     [9:0]                  SmoothCoeff1,  // Smoothing Coeffecient 1
            output  reg signed     [9:0]                  SmoothCoeff2,  // Smoothing Coeffecient 2
            output  reg signed     [9:0]                  SmoothCoeff3,  // Smoothing Coeffecient 3
            output  reg signed     [9:0]                  SmoothCoeff4,  // Smoothing Coeffecient 4
            output  reg signed     [9:0]                  SmoothCoeff5,  // Smoothing Coeffecient 5
            output  reg signed     [9:0]                  SmoothCoeff6,  // Smoothing Coeffecient 6
            output  reg signed     [9:0]                  SmoothCoeff7,  // Smoothing Coeffecient 7
            output  reg signed     [9:0]                  SmoothCoeff8,  // Smoothing Coeffecient 8
            output  reg signed     [9:0]                  SmoothCoeff9,  // Smoothing Coeffecient 9
            output  reg signed     [9:0]                  SmoothCoeff10, // Smoothing Coeffecient 10
            output  reg signed     [9:0]                  SmoothCoeff11, // Smoothing Coeffecient 11
            //
            output  reg            [1:0]                  SmoothCoeffRS  // Smoothing Right Shift
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
// Symbol type
localparam [2:0] L_LTF    = 3'd0,
                 VHT_LTF  = 3'd1,
                 HT_LTF   = 3'd2,
                 HE_1xLTF = 3'd4,
                 HE_2xLTF = 3'd5,
                 HE_4xLTF = 3'd6;

// Smoothing steps
localparam [2:0] IDLE_STEP      = 3'd0,
                 SMOOTHING      = 3'd1,
                 DC_INTERP      = 3'd2,
                 VIRTUAL_INTERP = 3'd3,
                 CHAN_INTERP    = 3'd4,
                 POST_CORDIC    = 3'd5,
                 PILOT_INTERP   = 3'd6;

// ReceptionMode decoding
localparam [3:0] MODE_20MHZ_L     = 4'd0,
                 MODE_20MHZ_HT    = 4'd2,
                 MODE_40MHZ_HT    = 4'd3,
                 MODE_20MHZ_HE_SU = 4'd6,
                 MODE_20MHZ_HE_MU = 4'd7,
                 MODE_40MHZ_HE_SU = 4'd8,
                 MODE_40MHZ_HE_MU = 4'd9,
                 MODE_80MHZ_HE_SU = 4'd10,
                 MODE_80MHZ_HE_MU = 4'd11;

// RUType decoding
localparam [2:0] RU26  = 3'd0,
                 RU52  = 3'd1,
                 RU106 = 3'd2,
                 RU242 = 3'd3,
                 RU484 = 3'd4;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires & Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [2:0]               FilterRowSel;
reg                                    FlipSel;
reg     signed     [9:0]               SmoothCoeffInt0;
reg     signed     [9:0]               SmoothCoeffInt1;
reg     signed     [9:0]               SmoothCoeffInt2;
reg     signed     [9:0]               SmoothCoeffInt3;
reg     signed     [9:0]               SmoothCoeffInt4;
reg     signed     [9:0]               SmoothCoeffInt5;
reg     signed     [9:0]               SmoothCoeffInt6;
reg     signed     [9:0]               SmoothCoeffInt7;
reg     signed     [9:0]               SmoothCoeffInt8;
reg     signed     [9:0]               SmoothCoeffInt9;
reg     signed     [9:0]               SmoothCoeffInt10;
reg     signed     [9:0]               SmoothCoeffInt11;

reg                [1:0]               SmoothCoeffRShiftInt; // 00:*1 01:*2 11:/2 10:/4

reg    signed      [9:0]               LastPosIndexFlip;

wire   signed      [9:0]               FirstNegIndexPlus1;
wire   signed      [9:0]               FirstNegIndexPlus2;
wire   signed      [9:0]               FirstNegIndexPlus3;
wire   signed      [9:0]               FirstNegIndexPlus4;
wire   signed      [9:0]               LastPosIndexMin1;
wire   signed      [9:0]               LastPosIndexMin2;
wire   signed      [9:0]               LastPosIndexMin3;
wire   signed      [9:0]               LastPosIndexMin4;
wire               [9:0]               IndexOffsetx2;
wire               [9:0]               IndexOffsetx3;
wire               [9:0]               IndexOffsetx4;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign IndexOffsetx2 = {IndexOffset[8:0],1'b0};
assign IndexOffsetx3 = IndexOffset + {IndexOffset[8:0],1'b0};
assign IndexOffsetx4 = {IndexOffset[7:0],2'b0};

assign FirstNegIndexPlus1 = $signed(FirstNegIndex) + $signed(IndexOffset);
assign FirstNegIndexPlus2 = $signed(FirstNegIndex) + $signed(IndexOffsetx2);
assign FirstNegIndexPlus3 = $signed(FirstNegIndex) + $signed(IndexOffsetx3);
assign FirstNegIndexPlus4 = $signed(FirstNegIndex) + $signed(IndexOffsetx4);

assign LastPosIndexMin1   = $signed(LastPosIndex) - $signed(IndexOffset);
assign LastPosIndexMin2   = $signed(LastPosIndex) - $signed(IndexOffsetx2);
assign LastPosIndexMin3   = $signed(LastPosIndex) - $signed(IndexOffsetx3);
assign LastPosIndexMin4   = $signed(LastPosIndex) - $signed(IndexOffsetx4);

always @ (posedge PhyClk or negedge nPhyRst)
   begin: LastPosIndexFlip_Blk
      if (nPhyRst == 1'b0)
         LastPosIndexFlip <= 10'b0;
      else
         LastPosIndexFlip <= $signed(LastPosIndexMin4) - $signed(IndexOffsetx2);
   end //LastPosIndexFlip_Blk

// Filter Row Selection
always @ (*)
  begin: Filter_Row_Select_Blk

    if (SmoothStep == SMOOTHING)
    begin

      if ((SCIndex == FirstNegIndex) || (SCIndex == LastPosIndex))
        FilterRowSel = 3'd1;
      else if ((SCIndex == FirstNegIndexPlus1) || (SCIndex == LastPosIndexMin1))
        FilterRowSel = 3'd2;
      else if ((SCIndex == FirstNegIndexPlus2) || (SCIndex == LastPosIndexMin2))
        FilterRowSel = 3'd3;
      else if ((SCIndex == FirstNegIndexPlus3) || (SCIndex == LastPosIndexMin3))
        FilterRowSel = 3'd4;
      else if ((SCIndex == FirstNegIndexPlus4) || (SCIndex == LastPosIndexMin4))
        FilterRowSel = 3'd5;
      else
        FilterRowSel = 3'd6;

    end
    
    else if (SmoothStep == PILOT_INTERP)
    begin
      if (SymbolType==HE_2xLTF)
      begin
        case (SCIndex)
           -10'sd1 : FilterRowSel = 3'd1;
            10'sd0 : FilterRowSel = 3'd2;
          default  : FilterRowSel = 3'd3;
        endcase
      end
      else
      begin
        case (SCIndex)
           -10'sd1 : FilterRowSel = 3'd4;
            10'sd0 : FilterRowSel = 3'd5;
          default  : FilterRowSel = 3'd6;
        endcase
      end
    end
    
    else if (SmoothStep == DC_INTERP)
    begin
      if((SymbolType != HE_4xLTF) && (ReceptionMode != MODE_40MHZ_HT)  && 
            !((ReceptionMode == MODE_20MHZ_HE_MU) && (RUType == RU26)) &&
            !(((ReceptionMode == MODE_40MHZ_HE_SU) || (ReceptionMode == MODE_40MHZ_HE_MU)) && (SymbolType == HE_2xLTF))) begin
        if (RxNsts!=2'b0 && (SymbolType != VHT_LTF) && (SymbolType != HT_LTF) && (SymbolType != HE_2xLTF))
          FilterRowSel = 3'd2;
        else
          FilterRowSel = 3'd1;
      end
      else begin
        if (LastPosIndex == 10'd1 || ((SymbolType == HE_2xLTF) && ((ReceptionMode == MODE_40MHZ_HE_SU) || (ReceptionMode == MODE_40MHZ_HE_MU)))) begin // For SU-20MHz, SU-40MHz HE_2xLTF and MU-40MHz HE_2xLTF
          if (SCIndex[9])
            FilterRowSel = 3'd2; // -1 or -2
          else if (SCIndex[0] || SCIndex[1])
            FilterRowSel = 3'd6; // 1 or 2
          else
            FilterRowSel = 3'd4; // 0
        end
        else if ((LastPosIndex == 10'd2) && ((ReceptionMode == MODE_40MHZ_HE_SU) || (ReceptionMode == MODE_40MHZ_HE_MU))) begin // For SU-40MHz and MU-40MHz HE_4xLTF
          case (SCIndex)
            -10'sd2 : FilterRowSel = 3'd1;
            -10'sd1 : FilterRowSel = 3'd2;
             10'sd0 : FilterRowSel = 3'd3;
             10'sd1 : FilterRowSel = 3'd4;
            default : FilterRowSel = 3'd5;
          endcase
        end
        else begin
          case (SCIndex)
            -10'sd3 : FilterRowSel = 3'd1;
            -10'sd2 : FilterRowSel = 3'd2;
            -10'sd1 : FilterRowSel = 3'd3;
             10'sd0 : FilterRowSel = 3'd4;
             10'sd1 : FilterRowSel = 3'd5;
             10'sd2 : FilterRowSel = 3'd6;
            default : FilterRowSel = 3'd7;
          endcase
        end
      end
    end
    
    else if (SmoothStep == VIRTUAL_INTERP)
    begin
    
      if (SymbolType==HE_2xLTF || SymbolType==HE_4xLTF || RxNsts==2'b0)
      begin
        if (SCIndex[9])
          FilterRowSel = 3'd2; // Left border interpolation
        else
          FilterRowSel = 3'd1; // Right border interpolation
      end
      else
      begin
        if (SCIndex[9])
          FilterRowSel = 3'd4; // Left border interpolation
        else
          FilterRowSel = 3'd3; // Right border interpolation
      end
    end

    else // CHAN_INTERP
    begin
    
      if (SymbolType==HE_1xLTF)
      begin
        case (ChanInterpCount)
          2'd0    : FilterRowSel = 3'd2;
          2'd1    : FilterRowSel = 3'd4;
          default : FilterRowSel = 3'd6;
        endcase
      end
      else
        FilterRowSel = 3'd4;
    end

  end //Filter_Row_Select_Blk

// Coefficients Selection
always @ (*)
begin: Coeff_Sel_Blk

  if (SmoothStep == SMOOTHING)
  begin

    if (SymbolType==L_LTF || SymbolType==HT_LTF || SymbolType==VHT_LTF || SymbolType==HE_1xLTF)
    begin
      case (FilterRowSel)

         3'd1: begin
            SmoothCoeffInt0  = 10'sd447;
            SmoothCoeffInt1  = 10'sd122;
            SmoothCoeffInt2  = -10'sd31;
            SmoothCoeffInt3  = -10'sd48;
            SmoothCoeffInt4  =  -10'sd7;
            SmoothCoeffInt5  =  10'sd25;
            SmoothCoeffInt6  =  10'sd20;
            SmoothCoeffInt7  =  -10'sd4;
            SmoothCoeffInt8  = -10'sd18;
            SmoothCoeffInt9  =  -10'sd7;
            SmoothCoeffInt10 =  10'sd12;
            SmoothCoeffInt11 =   10'sd0;
            // Right shift
            SmoothCoeffRShiftInt = 2'b00;
         end //4'd1

         3'd2: begin
            SmoothCoeffInt0  = 10'sd240;
            SmoothCoeffInt1  = 10'sd462;
            SmoothCoeffInt2  = 10'sd345;
            SmoothCoeffInt3  = 10'sd103;
            SmoothCoeffInt4  = -10'sd71;
            SmoothCoeffInt5  = -10'sd98;
            SmoothCoeffInt6  = -10'sd21;
            SmoothCoeffInt7  =  10'sd55;
            SmoothCoeffInt8  =  10'sd54;
            SmoothCoeffInt9  = -10'sd13;
            SmoothCoeffInt10 = -10'sd56;
            SmoothCoeffInt11 =  10'sd25;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd2

         3'd3: begin
            SmoothCoeffInt0  = -10'sd61;
            SmoothCoeffInt1  = 10'sd347;
            SmoothCoeffInt2  = 10'sd442;
            SmoothCoeffInt3  = 10'sd303;
            SmoothCoeffInt4  =  10'sd92;
            SmoothCoeffInt5  = -10'sd51;
            SmoothCoeffInt6  = -10'sd74;
            SmoothCoeffInt7  = -10'sd16;
            SmoothCoeffInt8  =  10'sd37;
            SmoothCoeffInt9  =  10'sd31;
            SmoothCoeffInt10 = -10'sd13;
            SmoothCoeffInt11 = -10'sd13;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd3

         3'd4: begin
            SmoothCoeffInt0  = -10'sd95;
            SmoothCoeffInt1  = 10'sd105;
            SmoothCoeffInt2  = 10'sd305;
            SmoothCoeffInt3  = 10'sd380;
            SmoothCoeffInt4  = 10'sd299;
            SmoothCoeffInt5  = 10'sd127;
            SmoothCoeffInt6  = -10'sd28;
            SmoothCoeffInt7  = -10'sd84;
            SmoothCoeffInt8  = -10'sd40;
            SmoothCoeffInt9  =  10'sd37;
            SmoothCoeffInt10 =  10'sd54;
            SmoothCoeffInt11 = -10'sd36;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd4

         3'd5: begin
            SmoothCoeffInt0  = -10'sd13;
            SmoothCoeffInt1  = -10'sd72;
            SmoothCoeffInt2  =  10'sd92;
            SmoothCoeffInt3  = 10'sd299;
            SmoothCoeffInt4  = 10'sd388;
            SmoothCoeffInt5  = 10'sd306;
            SmoothCoeffInt6  = 10'sd120;
            SmoothCoeffInt7  = -10'sd41;
            SmoothCoeffInt8  = -10'sd84;
            SmoothCoeffInt9  = -10'sd16;
            SmoothCoeffInt10 =  10'sd55;
            SmoothCoeffInt11 =  -10'sd8;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd5

         default: begin
            SmoothCoeffInt0  =  10'sd49;
            SmoothCoeffInt1  = -10'sd99;
            SmoothCoeffInt2  = -10'sd51;
            SmoothCoeffInt3  = 10'sd126;
            SmoothCoeffInt4  = 10'sd305;
            SmoothCoeffInt5  = 10'sd369;
            SmoothCoeffInt6  = 10'sd288;
            SmoothCoeffInt7  = 10'sd120;
            SmoothCoeffInt8  = -10'sd28;
            SmoothCoeffInt9  = -10'sd74;
            SmoothCoeffInt10 = -10'sd22;
            SmoothCoeffInt11 =  10'sd40;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd6

      endcase //FilterRowSel
    end
    else if (SymbolType==HE_2xLTF)
    begin
      
      case (FilterRowSel) 

         3'd1: begin
            SmoothCoeffInt0  = 10'sd362;
            SmoothCoeffInt1  = 10'sd182;
            SmoothCoeffInt2  =  10'sd54;
            SmoothCoeffInt3  = -10'sd20;
            SmoothCoeffInt4  = -10'sd49;
            SmoothCoeffInt5  = -10'sd44;
            SmoothCoeffInt6  = -10'sd20;
            SmoothCoeffInt7  =   10'sd9;
            SmoothCoeffInt8  =  10'sd28;
            SmoothCoeffInt9  =  10'sd30;
            SmoothCoeffInt10 =  10'sd10;
            SmoothCoeffInt11 = -10'sd30;
            // Right shift
            SmoothCoeffRShiftInt = 2'b00;
         end //4'd1

         3'd2: begin
            SmoothCoeffInt0  = 10'sd358;
            SmoothCoeffInt1  = 10'sd307;
            SmoothCoeffInt2  = 10'sd229;
            SmoothCoeffInt3  = 10'sd142;
            SmoothCoeffInt4  =  10'sd62;
            SmoothCoeffInt5  =   10'sd1;
            SmoothCoeffInt6  = -10'sd34;
            SmoothCoeffInt7  = -10'sd42;
            SmoothCoeffInt8  = -10'sd29;
            SmoothCoeffInt9  =  -10'sd6;
            SmoothCoeffInt10 =  10'sd15;
            SmoothCoeffInt11 =  10'sd20;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd2

         3'd3: begin
            SmoothCoeffInt0  = 10'sd106;
            SmoothCoeffInt1  = 10'sd229;                 
            SmoothCoeffInt2  = 10'sd270;                 
            SmoothCoeffInt3  = 10'sd246;                 
            SmoothCoeffInt4  = 10'sd179;                 
            SmoothCoeffInt5  =  10'sd93;                 
            SmoothCoeffInt6  =  10'sd12;                 
            SmoothCoeffInt7  = -10'sd45;                 
            SmoothCoeffInt8  = -10'sd68;                 
            SmoothCoeffInt9  = -10'sd52;                 
            SmoothCoeffInt10 =  -10'sd6;
            SmoothCoeffInt11 =  10'sd59;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd3

         3'd4: begin
            SmoothCoeffInt0  = -10'sd40;
            SmoothCoeffInt1  = 10'sd142;           
            SmoothCoeffInt2  = 10'sd247;           
            SmoothCoeffInt3  = 10'sd277;           
            SmoothCoeffInt4  = 10'sd245;           
            SmoothCoeffInt5  = 10'sd172;           
            SmoothCoeffInt6  =  10'sd81;           
            SmoothCoeffInt7  =  -10'sd2;           
            SmoothCoeffInt8  = -10'sd57;           
            SmoothCoeffInt9  = -10'sd68;
            SmoothCoeffInt10 = -10'sd29;
            SmoothCoeffInt11 =  10'sd55;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd4

         3'd5: begin
            SmoothCoeffInt0  = -10'sd97;
            SmoothCoeffInt1  =  10'sd63;           
            SmoothCoeffInt2  = 10'sd181;           
            SmoothCoeffInt3  = 10'sd247;           
            SmoothCoeffInt4  = 10'sd258;           
            SmoothCoeffInt5  = 10'sd222;           
            SmoothCoeffInt6  = 10'sd153;           
            SmoothCoeffInt7  =  10'sd71;           
            SmoothCoeffInt8  =  -10'sd2;
            SmoothCoeffInt9  = -10'sd45;
            SmoothCoeffInt10 = -10'sd42;
            SmoothCoeffInt11 =  10'sd17;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd5

         default: begin
            SmoothCoeffInt0  = -10'sd175;
            SmoothCoeffInt1  =    10'sd3;           
            SmoothCoeffInt2  =  10'sd189;           
            SmoothCoeffInt3  =  10'sd347;           
            SmoothCoeffInt4  =  10'sd446;           
            SmoothCoeffInt5  =  10'sd470;           
            SmoothCoeffInt6  =  10'sd419;           
            SmoothCoeffInt7  =  10'sd307;
            SmoothCoeffInt8  =  10'sd164;
            SmoothCoeffInt9  =   10'sd25;
            SmoothCoeffInt10 =  -10'sd68;
            SmoothCoeffInt11 =  -10'sd79;
            // Right shift
            SmoothCoeffRShiftInt = 2'b10;
         end //4'd6

      endcase //FilterRowSel
    end // HE_2xLTF
    else // HE_4xLTF
    begin
      
      case (FilterRowSel)

         3'd1: begin
            SmoothCoeffInt0  = 10'sd274;
            SmoothCoeffInt1  = 10'sd184;                            
            SmoothCoeffInt2  = 10'sd108;                            
            SmoothCoeffInt3  =  10'sd47;                            
            SmoothCoeffInt4  =   10'sd3;                            
            SmoothCoeffInt5  = -10'sd27;                            
            SmoothCoeffInt6  = -10'sd42;                            
            SmoothCoeffInt7  = -10'sd44;                            
            SmoothCoeffInt8  = -10'sd34;                            
            SmoothCoeffInt9  = -10'sd14;                            
            SmoothCoeffInt10 =  10'sd13;                            
            SmoothCoeffInt11 =  10'sd45;                            
            // Right shift
            SmoothCoeffRShiftInt = 2'b00;
         end //4'd1

         3'd2: begin
            SmoothCoeffInt0  = 10'sd366;
            SmoothCoeffInt1  = 10'sd284;                       
            SmoothCoeffInt2  = 10'sd207;                       
            SmoothCoeffInt3  = 10'sd138;                       
            SmoothCoeffInt4  =  10'sd78;                       
            SmoothCoeffInt5  =  10'sd30;                       
            SmoothCoeffInt6  =  -10'sd6;                        
            SmoothCoeffInt7  = -10'sd27;                       
            SmoothCoeffInt8  = -10'sd35;                       
            SmoothCoeffInt9  = -10'sd28;                       
            SmoothCoeffInt10 =  -10'sd8;                        
            SmoothCoeffInt11 =  10'sd25;
            // Right shift
            SmoothCoeffRShiftInt = 2'b11;
         end //4'd2

         3'd3: begin
            SmoothCoeffInt0  = 10'sd429;
            SmoothCoeffInt1  = 10'sd413;                     
            SmoothCoeffInt2  = 10'sd378;                     
            SmoothCoeffInt3  = 10'sd326;                     
            SmoothCoeffInt4  = 10'sd264;                     
            SmoothCoeffInt5  = 10'sd195;                     
            SmoothCoeffInt6  = 10'sd126;                     
            SmoothCoeffInt7  =  10'sd61;                      
            SmoothCoeffInt8  =   10'sd6;                       
            SmoothCoeffInt9  = -10'sd35;                     
            SmoothCoeffInt10 = -10'sd57;
            SmoothCoeffInt11 = -10'sd57;
            // Right shift
            SmoothCoeffRShiftInt = 2'b10;
         end //4'd3

         3'd4: begin
            SmoothCoeffInt0  =  10'sd188;
            SmoothCoeffInt1  =  10'sd274;                
            SmoothCoeffInt2  =  10'sd325;                
            SmoothCoeffInt3  =  10'sd343;                
            SmoothCoeffInt4  =  10'sd331;                
            SmoothCoeffInt5  =  10'sd294;                
            SmoothCoeffInt6  =  10'sd237;                
            SmoothCoeffInt7  =  10'sd166;                
            SmoothCoeffInt8  =   10'sd87;                
            SmoothCoeffInt9  =    10'sd6;
            SmoothCoeffInt10 =  -10'sd70;
            SmoothCoeffInt11 = -10'sd134;
            // Right shift
            SmoothCoeffRShiftInt = 2'b10;
         end //4'd4

         3'd5: begin
            SmoothCoeffInt0  =   10'sd10;
            SmoothCoeffInt1  =  10'sd155;                
            SmoothCoeffInt2  =  10'sd263;                
            SmoothCoeffInt3  =  10'sd331;                
            SmoothCoeffInt4  =  10'sd362;                
            SmoothCoeffInt5  =  10'sd357;                
            SmoothCoeffInt6  =  10'sd319;                
            SmoothCoeffInt7  =  10'sd254;                
            SmoothCoeffInt8  =  10'sd166;
            SmoothCoeffInt9  =   10'sd61;
            SmoothCoeffInt10 =  -10'sd55;
            SmoothCoeffInt11 = -10'sd173;
            // Right shift
            SmoothCoeffRShiftInt = 2'b10;
         end //4'd5

         default: begin
            SmoothCoeffInt0  = -10'sd107;
            SmoothCoeffInt1  =   10'sd59;              
            SmoothCoeffInt2  =  10'sd194;              
            SmoothCoeffInt3  =  10'sd294;              
            SmoothCoeffInt4  =  10'sd356;              
            SmoothCoeffInt5  =  10'sd381;              
            SmoothCoeffInt6  =  10'sd368;              
            SmoothCoeffInt7  =  10'sd319;
            SmoothCoeffInt8  =  10'sd237;
            SmoothCoeffInt9  =  10'sd125;
            SmoothCoeffInt10 =  -10'sd11;
            SmoothCoeffInt11 = -10'sd167;
            // Right shift
            SmoothCoeffRShiftInt = 2'b10;
         end //4'd6

      endcase //FilterRowSel
    end // HE_4xLTF
  end // SMOOTHING
  else if (SmoothStep == PILOT_INTERP)
  begin
    case (FilterRowSel)
      3'd1: begin
         SmoothCoeffInt0  =    10'sd0;
         SmoothCoeffInt1  =    10'sd0;
         SmoothCoeffInt2  =    10'sd0;
         SmoothCoeffInt3  =    10'sd0;
         SmoothCoeffInt4  =    10'sd0;
         SmoothCoeffInt5  =    10'sd0;
         SmoothCoeffInt6  =  10'sd211;     
         SmoothCoeffInt7  =  10'sd305;   
         SmoothCoeffInt8  =  10'sd111;   
         SmoothCoeffInt9  =  -10'sd65;
         SmoothCoeffInt10 = -10'sd106;
         SmoothCoeffInt11 =   10'sd57; 
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd2: begin
         SmoothCoeffInt0  =    10'sd0;
         SmoothCoeffInt1  =    10'sd0;
         SmoothCoeffInt2  =    10'sd0;
         SmoothCoeffInt3  =    10'sd0;
         SmoothCoeffInt4  =    10'sd0;
         SmoothCoeffInt5  =    10'sd0;
         SmoothCoeffInt6  = -10'sd154;
         SmoothCoeffInt7  =  10'sd198;
         SmoothCoeffInt8  =  10'sd468;
         SmoothCoeffInt9  =  10'sd468;
         SmoothCoeffInt10 =  10'sd198;
         SmoothCoeffInt11 = -10'sd154;
         // Right shift
         SmoothCoeffRShiftInt = 2'b11;
      end
     
      3'd3: begin
         SmoothCoeffInt0  =    10'sd0;
         SmoothCoeffInt1  =    10'sd0;
         SmoothCoeffInt2  =    10'sd0;
         SmoothCoeffInt3  =    10'sd0;
         SmoothCoeffInt4  =    10'sd0;
         SmoothCoeffInt5  =    10'sd0;
         SmoothCoeffInt6  =   10'sd57;
         SmoothCoeffInt7  = -10'sd106; 
         SmoothCoeffInt8  =  -10'sd65;
         SmoothCoeffInt9  =  10'sd111;
         SmoothCoeffInt10 =  10'sd305;
         SmoothCoeffInt11 =  10'sd211;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd4: begin
         SmoothCoeffInt0  =    10'sd0;
         SmoothCoeffInt1  =    10'sd0;
         SmoothCoeffInt2  =    10'sd0;
         SmoothCoeffInt3  =    10'sd0;
         SmoothCoeffInt4  =    10'sd0;
         SmoothCoeffInt5  =    10'sd0;
         SmoothCoeffInt6  =  10'sd220;
         SmoothCoeffInt7  =  10'sd279;
         SmoothCoeffInt8  =  10'sd112;
         SmoothCoeffInt9  =  -10'sd44;
         SmoothCoeffInt10 =  -10'sd93;
         SmoothCoeffInt11 =   10'sd38;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd5: begin
         SmoothCoeffInt0  =    10'sd0;
         SmoothCoeffInt1  =    10'sd0;
         SmoothCoeffInt2  =    10'sd0;
         SmoothCoeffInt3  =    10'sd0;
         SmoothCoeffInt4  =    10'sd0;
         SmoothCoeffInt5  =    10'sd0;
         SmoothCoeffInt6  = -10'sd153;   
         SmoothCoeffInt7  =  10'sd206;   
         SmoothCoeffInt8  =  10'sd458;   
         SmoothCoeffInt9  =  10'sd458;   
         SmoothCoeffInt10 =  10'sd206;   
         SmoothCoeffInt11 = -10'sd153;   
         // Right shift
         SmoothCoeffRShiftInt = 2'b11;
      end
     
      default: begin
         SmoothCoeffInt0  =    10'sd0;
         SmoothCoeffInt1  =    10'sd0;
         SmoothCoeffInt2  =    10'sd0;
         SmoothCoeffInt3  =    10'sd0;
         SmoothCoeffInt4  =    10'sd0;
         SmoothCoeffInt5  =    10'sd0;
         SmoothCoeffInt6  =   10'sd38;
         SmoothCoeffInt7  =  -10'sd93;
         SmoothCoeffInt8  =  -10'sd44;
         SmoothCoeffInt9  =  10'sd112;
         SmoothCoeffInt10 =  10'sd279;
         SmoothCoeffInt11 =  10'sd220;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
    endcase // FilterRowSel
  end // PILOT_INTERP
  else if ((SmoothStep == DC_INTERP) && (SymbolType != HE_4xLTF) && (ReceptionMode != MODE_40MHZ_HT) &&
           !(((ReceptionMode == MODE_40MHZ_HE_SU) || (ReceptionMode == MODE_40MHZ_HE_MU)) && (SymbolType == HE_2xLTF)) && !((ReceptionMode == MODE_20MHZ_HE_MU) && (RUType == RU26)))
  begin
    case (FilterRowSel)
      3'd1: begin
         SmoothCoeffInt0  =   10'sd62;
         SmoothCoeffInt1  = -10'sd102;
         SmoothCoeffInt2  =  -10'sd89;
         SmoothCoeffInt3  =   10'sd46;
         SmoothCoeffInt4  =  10'sd225;
         SmoothCoeffInt5  =  10'sd370;
         SmoothCoeffInt6  =  10'sd370;
         SmoothCoeffInt7  =  10'sd225;
         SmoothCoeffInt8  =   10'sd46;
         SmoothCoeffInt9  =  -10'sd89;
         SmoothCoeffInt10 = -10'sd102;
         SmoothCoeffInt11 =   10'sd62;
         // Right shift
         SmoothCoeffRShiftInt = 2'b11;
      end
     
      default: begin
         SmoothCoeffInt0  =  10'sd45;
         SmoothCoeffInt1  = -10'sd97;
         SmoothCoeffInt2  = -10'sd72;
         SmoothCoeffInt3  =  10'sd59;
         SmoothCoeffInt4  = 10'sd223;
         SmoothCoeffInt5  = 10'sd354;
         SmoothCoeffInt6  = 10'sd354;
         SmoothCoeffInt7  = 10'sd223;
         SmoothCoeffInt8  =  10'sd59;
         SmoothCoeffInt9  = -10'sd72;
         SmoothCoeffInt10 = -10'sd97;
         SmoothCoeffInt11 =  10'sd45;
         // Right shift
         SmoothCoeffRShiftInt = 2'b11;
      end
    endcase // FilterRowSel
  end // DC_INTERP
  else if (SmoothStep == VIRTUAL_INTERP)
  begin
    case (FilterRowSel)
      3'd1: begin
         SmoothCoeffInt0  =   10'sd56;
         SmoothCoeffInt1  =  -10'sd44;
         SmoothCoeffInt2  =  -10'sd66;
         SmoothCoeffInt3  =  -10'sd24;
         SmoothCoeffInt4  =   10'sd42;
         SmoothCoeffInt5  =   10'sd84;
         SmoothCoeffInt6  =   10'sd65;
         SmoothCoeffInt7  =  -10'sd16;
         SmoothCoeffInt8  = -10'sd118;
         SmoothCoeffInt9  = -10'sd154;
         SmoothCoeffInt10 =   -10'sd9;
         SmoothCoeffInt11 =  10'sd439;
         // Right shift
         SmoothCoeffRShiftInt = 2'b01;
      end
     
      3'd2: begin
         SmoothCoeffInt0  =  10'sd439;
         SmoothCoeffInt1  =   -10'sd9; 
         SmoothCoeffInt2  = -10'sd154; 
         SmoothCoeffInt3  = -10'sd118; 
         SmoothCoeffInt4  =  -10'sd16; 
         SmoothCoeffInt5  =   10'sd65;
         SmoothCoeffInt6  =   10'sd84;
         SmoothCoeffInt7  =   10'sd42;
         SmoothCoeffInt8  =  -10'sd24;
         SmoothCoeffInt9  =  -10'sd66;
         SmoothCoeffInt10 =  -10'sd44;
         SmoothCoeffInt11 =   10'sd56;
         // Right shift
         SmoothCoeffRShiftInt = 2'b01;
      end
     
      3'd3: begin
         SmoothCoeffInt0  =   10'sd48;
         SmoothCoeffInt1  =  -10'sd56;
         SmoothCoeffInt2  =  -10'sd57;
         SmoothCoeffInt3  =   -10'sd2;
         SmoothCoeffInt4  =   10'sd58;
         SmoothCoeffInt5  =   10'sd80;
         SmoothCoeffInt6  =   10'sd44;
         SmoothCoeffInt7  =  -10'sd39;
         SmoothCoeffInt8  = -10'sd122;
         SmoothCoeffInt9  = -10'sd132;
         SmoothCoeffInt10 =   10'sd19;
         SmoothCoeffInt11 =  10'sd417;
         // Right shift
         SmoothCoeffRShiftInt = 2'b01;
      end
     
      default: begin
         SmoothCoeffInt0  =  10'sd417;
         SmoothCoeffInt1  =   10'sd19;
         SmoothCoeffInt2  = -10'sd132;
         SmoothCoeffInt3  = -10'sd122;
         SmoothCoeffInt4  =  -10'sd39;
         SmoothCoeffInt5  =   10'sd44;
         SmoothCoeffInt6  =   10'sd80;
         SmoothCoeffInt7  =   10'sd58;
         SmoothCoeffInt8  =   -10'sd2;
         SmoothCoeffInt9  =  -10'sd57;
         SmoothCoeffInt10 =  -10'sd56;
         SmoothCoeffInt11 =   10'sd48;
         // Right shift
         SmoothCoeffRShiftInt = 2'b01;
      end
    endcase // FilterRowSel
  end // VIRTUAL_INTERP
  else if ((SmoothStep == DC_INTERP) && (SymbolType == HE_4xLTF) && ((ReceptionMode == MODE_40MHZ_HE_SU) || (ReceptionMode == MODE_40MHZ_HE_MU)))
  begin
    case (FilterRowSel)
      3'd1: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd427;
         SmoothCoeffInt11 = 10'sd85;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd2: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd341;
         SmoothCoeffInt11 = 10'sd171;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd3: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd511;
         SmoothCoeffInt11 = 10'sd511;
         // Right shift
         SmoothCoeffRShiftInt = 2'b11;
      end
     
      3'd4: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd171;
         SmoothCoeffInt11 = 10'sd341;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      default: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd85;
         SmoothCoeffInt11 = 10'sd427;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
    endcase // FilterRowSel
  end // DC_INTERP with HE_4xLTF 40MHz
  else // CHAN_INTERP or DC_INTERP with HE_4xLTF / MODE_40MHZ_HT
  begin
    case (FilterRowSel)
      3'd1: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd448;
         SmoothCoeffInt11 = 10'sd64;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd2: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd384;
         SmoothCoeffInt11 = 10'sd128;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd3: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd320;
         SmoothCoeffInt11 = 10'sd192;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
     
      3'd4: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd511;
         SmoothCoeffInt11 = 10'sd511;
         // Right shift
         SmoothCoeffRShiftInt = 2'b11;
      end
     
      3'd5: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd192;
         SmoothCoeffInt11 = 10'sd320;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end

      3'd6: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd128;
         SmoothCoeffInt11 = 10'sd384;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end

      default: begin
         SmoothCoeffInt0  = 10'sd0;
         SmoothCoeffInt1  = 10'sd0;
         SmoothCoeffInt2  = 10'sd0;
         SmoothCoeffInt3  = 10'sd0;
         SmoothCoeffInt4  = 10'sd0;
         SmoothCoeffInt5  = 10'sd0;
         SmoothCoeffInt6  = 10'sd0;
         SmoothCoeffInt7  = 10'sd0;
         SmoothCoeffInt8  = 10'sd0;
         SmoothCoeffInt9  = 10'sd0;
         SmoothCoeffInt10 = 10'sd64;
         SmoothCoeffInt11 = 10'sd448;
         // Right shift
         SmoothCoeffRShiftInt = 2'b00;
      end
    endcase // FilterRowSel
  end // CHAN_INTERP or DC_INTERP with HE_4xLTF

end // Coeff_Sel_Blk

// Final Selection: To flip or not to flip!
// Depends on Symbol Type and Order of storage
// In all modes, upper subcarriers need the flipped version
always @ (*)
begin: FlipSel_Blk
   if (SmoothStep == SMOOTHING) begin
     case (SymbolType)
     L_LTF :
     // L_LTF : 20 MHz Legacy, 20 MHz HT
        if (SCIndex > 10'sd20)
           FlipSel = 1'b1;
        else
           FlipSel = 1'b0;
     HT_LTF, VHT_LTF:
     // HT_LTF, VHT_LTF : HT/VHT 20/40 MHz
        if ((ReceptionMode == MODE_40MHZ_HT && (SCIndex > 10'sd52)) || (ReceptionMode == MODE_20MHZ_HT && (SCIndex > 10'sd22)))
           FlipSel = 1'b1;
        else
           FlipSel = 1'b0;
     default:
     // HE_1xLTF, HE_2xLTF, HE_4xLTF : HE 20 MHz
        if (ReceptionMode == MODE_20MHZ_HE_SU || ReceptionMode == MODE_20MHZ_HE_MU) begin
          if (($signed(SCIndex) > $signed(LastPosIndexFlip)) || (SCIndex > 10'sd110 && SymbolType==HE_2xLTF) || (SCIndex > 10'sd96 && SymbolType==HE_1xLTF))
             FlipSel = 1'b1;
          else
             FlipSel = 1'b0;
        end
     // HE_1xLTF, HE_2xLTF, HE_4xLTF : HE 40 MHz
        else if (ReceptionMode == MODE_40MHZ_HE_SU || ReceptionMode == MODE_40MHZ_HE_MU) begin
          if (($signed(SCIndex) > $signed(LastPosIndexFlip)) || (SCIndex > 10'sd232 && SymbolType==HE_2xLTF) || (SCIndex > 10'sd220 && SymbolType==HE_1xLTF))
             FlipSel = 1'b1;
          else
             FlipSel = 1'b0;
        end
     // HE_1xLTF, HE_2xLTF, HE_4xLTF : HE 80 MHz
        else begin
          if (($signed(SCIndex) > $signed(LastPosIndexFlip)) || (SCIndex > 10'sd486 && SymbolType==HE_2xLTF) || (SCIndex > 10'sd452 && SymbolType==HE_1xLTF))
             FlipSel = 1'b1;
          else
             FlipSel = 1'b0;
        end
     endcase // case
   end
   else
     FlipSel = 1'b0;
end // FlipSel_Blk

always @ (posedge PhyClk or negedge nPhyRst)
begin: SmoothCoeff_Blk
   if (nPhyRst == 1'b0) begin
      SmoothCoeff0  <= 10'sd0;
      SmoothCoeff1  <= 10'sd0;
      SmoothCoeff2  <= 10'sd0;
      SmoothCoeff3  <= 10'sd0;
      SmoothCoeff4  <= 10'sd0;
      SmoothCoeff5  <= 10'sd0;
      SmoothCoeff6  <= 10'sd0;
      SmoothCoeff7  <= 10'sd0;
      SmoothCoeff8  <= 10'sd0;
      SmoothCoeff9  <= 10'sd0;
      SmoothCoeff10 <= 10'sd0;
      SmoothCoeff11 <= 10'sd0;
      SmoothCoeffRS <= 2'd0;
   end
   else
   begin
     SmoothCoeffRS <= SmoothCoeffRShiftInt;
     
     if (FlipSel == 1'b0) begin
        // Lower Subcarriers
        // Flipping is required because MATLAB and RTL store
        // the Coeffecients in opposite orders
        SmoothCoeff0  <= SmoothCoeffInt11;
        SmoothCoeff1  <= SmoothCoeffInt10;
        SmoothCoeff2  <= SmoothCoeffInt9;
        SmoothCoeff3  <= SmoothCoeffInt8;
        SmoothCoeff4  <= SmoothCoeffInt7;
        SmoothCoeff5  <= SmoothCoeffInt6;
        SmoothCoeff6  <= SmoothCoeffInt5;
        SmoothCoeff7  <= SmoothCoeffInt4;
        SmoothCoeff8  <= SmoothCoeffInt3;
        SmoothCoeff9  <= SmoothCoeffInt2;
        SmoothCoeff10 <= SmoothCoeffInt1;
        SmoothCoeff11 <= SmoothCoeffInt0;
     end
     else begin
        //Upper Subcarriers
        //This would require flipping twice - or no flip.
        SmoothCoeff0  <= SmoothCoeffInt0;
        SmoothCoeff1  <= SmoothCoeffInt1;
        SmoothCoeff2  <= SmoothCoeffInt2;
        SmoothCoeff3  <= SmoothCoeffInt3;
        SmoothCoeff4  <= SmoothCoeffInt4;
        SmoothCoeff5  <= SmoothCoeffInt5;
        SmoothCoeff6  <= SmoothCoeffInt6;
        SmoothCoeff7  <= SmoothCoeffInt7;
        SmoothCoeff8  <= SmoothCoeffInt8;
        SmoothCoeff9  <= SmoothCoeffInt9;
        SmoothCoeff10 <= SmoothCoeffInt10;
        SmoothCoeff11 <= SmoothCoeffInt11;
     end
   end
end //SmoothCoeff_Blk

endmodule //FilterCoeffSelect

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
