//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39777 $
// $Date: 2019-09-12 18:24:11 +0200 (Thu, 12 Sep 2019) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Controller for H Memory write process
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/HWriteCtrl.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module HWriteCtrl #(parameter NSTS_PARAM    = 2   // Number of Nsts supported
                   )(

            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input   wire                       nPhyRst,  // Active Low Reset
            input   wire                       PhyClk,   // Phy Clock

            ///////////////////////////////////////////////
            // Controls
            ///////////////////////////////////////////////
            input   wire                       EstimEn,          // Enable Ch Est
            input   wire                       SmoothEn,         // Enable smoothing
            input   wire        [3:0]          ReceptionMode,    // Reception Mode
            input   wire        [2:0]          SymbolType,       // Type of Symbol
            input   wire        [2:0]          RUType,           // Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)
            input   wire        [1:0]          RxNsts,           // No. of Space Time Steams
            input   wire                       FirstLTF,         // Indicates First LTF
            input   wire                       LastLTF,          // Indicates Last LTF
            input   wire                       FrameNDP,         // Indicates frame is NDP
            //
            input   wire        [2:0]          SmoothStep,       // Smoothing steps
            input   wire signed [9:0]          SCInd,            // Subcarrier Index
            input   wire signed [9:0]          LastPosIndex,     // Subcarrier last Index
            input   wire                       HWriteDataEnOut,  // Enable H writing
            input   wire                       HWriteI,          // Write only I phase
            input   wire                       HWriteQ,          // Write only Q phase
            //
            output  reg                        LastWriteP,       // To Pre Estim controller
            output  wire                       PilotStore,       // Indicates that the current SC is a pilot to store

            ///////////////////////////////////////////////
            // H Memory interface
            ///////////////////////////////////////////////
            output  reg [9:0]                  HBufWrAddrOut,  // Address
            output  reg [1:0]                  HBufWrEn0,      // Write Enable Buffer 0
            output  reg [1:0]                  HBufWrEn1,      // Write Enable Buffer 1
            output  reg [1:0]                  HBufWrEn2,      // Write Enable Buffer 2
            output  reg [1:0]                  HBufWrEn3,      // Write Enable Buffer 3
            output  reg                        HBufWrAddrEnOut // Addr Enable
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////

// Mode
localparam [3:0] MODE_20MHZ_L     = 4'd0,
                 MODE_20MHZ_HT    = 4'd2,
                 MODE_40MHZ_HT    = 4'd3,
                 MODE_20MHZ_HE_SU = 4'd6,
                 MODE_20MHZ_HE_MU = 4'd7,
                 MODE_40MHZ_HE_SU = 4'd8,
                 MODE_40MHZ_HE_MU = 4'd9,
                 MODE_80MHZ_HE_SU = 4'd10,
                 MODE_80MHZ_HE_MU = 4'd11;

// Symbol Type
localparam [2:0] L_LTF    = 3'd0,
                 VHT_LTF  = 3'd1,
                 HT_LTF   = 3'd2,
                 HE_1xLTF = 3'd4,
                 HE_2xLTF = 3'd5,
                 HE_4xLTF = 3'd6,
                 HE_LSIG  = 3'd7;

// RU type
localparam [2:0] RU26  = 3'd0,
                 RU52  = 3'd1,
                 RU106 = 3'd2,
                 RU242 = 3'd3,
                 RU484 = 3'd4;

// SmoothStep definition
localparam [2:0] IDLE_STEP      = 3'd0,
                 SMOOTHING      = 3'd1,
                 DC_INTERP      = 3'd2,
                 VIRTUAL_INTERP = 3'd3,
                 CHAN_INTERP    = 3'd4,
                 POST_CORDIC    = 3'd5,
                 PILOT_INTERP   = 3'd6;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires & Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg         [1:0]                     HBufWrEnInt0;
reg         [1:0]                     HBufWrEnInt1;
reg         [1:0]                     HBufWrEnInt2;
reg         [1:0]                     HBufWrEnInt3;
reg                                   CurrentSCIsPilot;
reg                                   SCIndDone;
reg         [2:0]                     SymbolType_1t;
reg         [2:0]                     RUType_1t;
reg         [2:0]                     SmoothStep_1t;
reg signed  [9:0]                     LastPosIndex_1t;
reg         [3:0]                     ReceptionMode_1t;

wire                                  CurrentSCIsPilotMux;

// Integer for loop generation
genvar       i;

// The following 2 strings are useful during simulations
// to view States using strings instead of numbers
`ifdef RW_SIMU_ON
  reg [25*8:0] SymbolTypeStr;
  reg [25*8:0] SmoothStepStr;
`endif

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Delayed version
always @ (posedge PhyClk or negedge nPhyRst)
begin: DelayReg_Blk
  if (nPhyRst == 1'b0) begin
     ReceptionMode_1t <= MODE_20MHZ_L;
     SymbolType_1t    <= L_LTF;
     RUType_1t        <= RU26;
     SmoothStep_1t    <= IDLE_STEP;
     LastPosIndex_1t  <= 10'd0;
  end
  else begin
     ReceptionMode_1t <= ReceptionMode;
     SymbolType_1t    <= SymbolType;
     RUType_1t        <= RUType;
     SmoothStep_1t    <= SmoothStep;
     LastPosIndex_1t  <= LastPosIndex;
  end
end //DelayReg_Blk

always @*
begin
  case (ReceptionMode_1t)
      MODE_20MHZ_HT : 
      begin
        // Pilot subcarriers (21, 7)
        if ((SCInd == -10'sd7)  || (SCInd == 10'sd7) ||
            (SCInd == -10'sd21) || (SCInd == 10'sd21))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_40MHZ_HT : 
      begin
        // Pilot subcarriers (53, 25, 11)
        if ((SCInd == -10'sd11) || (SCInd == 10'sd11) || 
            (SCInd == -10'sd25) || (SCInd == 10'sd25) || 
            (SCInd == -10'sd53) || (SCInd == 10'sd53))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_40MHZ_HE_SU : 
      begin
        // Pilot subcarriers (238, 212, 170, 144, 104, 78, 36, 10)
        if ((
             (SCInd == -10'sd10)  || (SCInd == 10'sd10)  ||
             (SCInd == -10'sd36)  || (SCInd == 10'sd36)  ||
             (SCInd == -10'sd78)  || (SCInd == 10'sd78)  ||
             (SCInd == -10'sd104) || (SCInd == 10'sd104) ||
             (SCInd == -10'sd144) || (SCInd == 10'sd144) ||
             (SCInd == -10'sd170) || (SCInd == 10'sd170) ||
             (SCInd == -10'sd212) || (SCInd == 10'sd212) ||
             (SCInd == -10'sd238) || (SCInd == 10'sd238)
             ) && !FrameNDP)
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_40MHZ_HE_MU : 
      begin
        // Pilot subcarriers (238, 224, 212, 198, 184, 170, 158, 144,
        // 130, 116, 104, 90, 78, 64, 50, 36, 24, 10)
        if (((SCInd == -10'sd10)  || (SCInd == 10'sd10)  ||
             (SCInd == -10'sd24 && RUType_1t < RU106)  || (SCInd == 10'sd24 && RUType_1t < RU106)  ||
             (SCInd == -10'sd36)  || (SCInd == 10'sd36)  ||
             (SCInd == -10'sd50 && RUType_1t < RU106)  || (SCInd == 10'sd50 && RUType_1t < RU106)  ||
             (SCInd == -10'sd64 && RUType_1t < RU106)  || (SCInd == 10'sd64 && RUType_1t < RU106)  ||
             (SCInd == -10'sd78)  || (SCInd == 10'sd78)  ||
             (SCInd == -10'sd90 && RUType_1t < RU106)  || (SCInd == 10'sd90 && RUType_1t < RU106)  ||
             (SCInd == -10'sd104) || (SCInd == 10'sd104) ||
             (SCInd == -10'sd116 && RUType_1t < RU106) || (SCInd == 10'sd116 && RUType_1t < RU106) ||
             (SCInd == -10'sd130 && RUType_1t < RU106) || (SCInd == 10'sd130 && RUType_1t < RU106) ||
             (SCInd == -10'sd144) || (SCInd == 10'sd144) ||
             (SCInd == -10'sd158 && RUType_1t < RU106) || (SCInd == 10'sd158 && RUType_1t < RU106) ||
             (SCInd == -10'sd170) || (SCInd == 10'sd170) ||
             (SCInd == -10'sd184 && RUType_1t < RU106) || (SCInd == 10'sd184 && RUType_1t < RU106) ||
             (SCInd == -10'sd198 && RUType_1t < RU106) || (SCInd == 10'sd198 && RUType_1t < RU106) ||
             (SCInd == -10'sd212) || (SCInd == 10'sd212) ||
             (SCInd == -10'sd224 && RUType_1t < RU106) || (SCInd == 10'sd224 && RUType_1t < RU106) ||
             (SCInd == -10'sd238) || (SCInd == 10'sd238)) && !FrameNDP)
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_80MHZ_HE_MU : 
      begin
        // Pilot subcarriers (494, 480, 468, 454, 440, 426,
        // 414, 400, 386, 372, 360, 346, 334, 320,
        // 306, 292, 280, 266, 252,
        // 238, 226, 212, 198, 184, 172, 158, 144,
        // 130, 118, 104, 92, 78, 64, 50, 38, 24, 10)
        if (((SCInd == -10'sd10 && RUType_1t < RU106)  || (SCInd == 10'sd10 && RUType_1t < RU106)  ||
             (SCInd == -10'sd24)  || (SCInd == 10'sd24)  ||
             (SCInd == -10'sd38 && RUType_1t < RU106)  || (SCInd == 10'sd38 && RUType_1t < RU106)  ||
             (SCInd == -10'sd50)  || (SCInd == 10'sd50)  ||
             (SCInd == -10'sd64 && RUType_1t < RU106)  || (SCInd == 10'sd64 && RUType_1t < RU106)  ||
             (SCInd == -10'sd78 && RUType_1t < RU106)  || (SCInd == 10'sd78 && RUType_1t < RU106)  ||
             (SCInd == -10'sd92)  || (SCInd == 10'sd92)  ||
             (SCInd == -10'sd104 && RUType_1t < RU106) || (SCInd == 10'sd104 && RUType_1t < RU106) ||
             (SCInd == -10'sd118) || (SCInd == 10'sd118) ||
             (SCInd == -10'sd130 && RUType_1t < RU106) || (SCInd == 10'sd130 && RUType_1t < RU106) ||
             (SCInd == -10'sd144 && RUType_1t < RU106) || (SCInd == 10'sd144 && RUType_1t < RU106) ||
             (SCInd == -10'sd158 && RUType_1t < RU106) || (SCInd == 10'sd158 && RUType_1t < RU106) ||
             (SCInd == -10'sd172 && RUType_1t < RU106) || (SCInd == 10'sd172 && RUType_1t < RU106) ||
             (SCInd == -10'sd184) || (SCInd == 10'sd184) ||
             (SCInd == -10'sd198 && RUType_1t < RU106) || (SCInd == 10'sd198 && RUType_1t < RU106) ||
             (SCInd == -10'sd212 && RUType_1t < RU106) || (SCInd == 10'sd212 && RUType_1t < RU106) ||
             (SCInd == -10'sd226) || (SCInd == 10'sd226) ||
             (SCInd == -10'sd238 && RUType_1t < RU106) || (SCInd == 10'sd238 && RUType_1t < RU106) ||
             (SCInd == -10'sd252) || (SCInd == 10'sd252) ||
             (SCInd == -10'sd266) || (SCInd == 10'sd266) ||
             (SCInd == -10'sd280 && RUType_1t < RU106) || (SCInd == 10'sd280 && RUType_1t < RU106) ||
             (SCInd == -10'sd292) || (SCInd == 10'sd292) ||
             (SCInd == -10'sd306 && RUType_1t < RU106) || (SCInd == 10'sd306 && RUType_1t < RU106) ||
             (SCInd == -10'sd320 && RUType_1t < RU106) || (SCInd == 10'sd320 && RUType_1t < RU106) ||
             (SCInd == -10'sd334) || (SCInd == 10'sd334) ||
             (SCInd == -10'sd346 && RUType_1t < RU106) || (SCInd == 10'sd346 && RUType_1t < RU106) ||
             (SCInd == -10'sd360) || (SCInd == 10'sd360) ||
             (SCInd == -10'sd372 && RUType_1t < RU106) || (SCInd == 10'sd372 && RUType_1t < RU106) ||
             (SCInd == -10'sd386 && RUType_1t < RU106) || (SCInd == 10'sd386 && RUType_1t < RU106) ||
             (SCInd == -10'sd400) || (SCInd == 10'sd400) ||
             (SCInd == -10'sd414 && RUType_1t < RU106) || (SCInd == 10'sd414 && RUType_1t < RU106) ||
             (SCInd == -10'sd426) || (SCInd == 10'sd426) ||
             (SCInd == -10'sd440 && RUType_1t < RU106) || (SCInd == 10'sd440 && RUType_1t < RU106) ||
             (SCInd == -10'sd454 && RUType_1t < RU106) || (SCInd == 10'sd454 && RUType_1t < RU106) ||
             (SCInd == -10'sd468) || (SCInd == 10'sd468) ||
             (SCInd == -10'sd480 && RUType_1t < RU106) || (SCInd == 10'sd480 && RUType_1t < RU106) ||
             (SCInd == -10'sd494) || (SCInd == 10'sd494)) && !FrameNDP)
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_20MHZ_HE_SU : 
      begin
        // Pilot subcarriers (116, 90, 48, 22)
        if (((SCInd == -10'sd22)  || (SCInd == 10'sd22)  ||
             (SCInd == -10'sd48)  || (SCInd == 10'sd48)  ||
             (SCInd == -10'sd90)  || (SCInd == 10'sd90)  ||
             (SCInd == -10'sd116) || (SCInd == 10'sd116)) && !FrameNDP)
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_20MHZ_HE_MU : 
      begin
        // Pilot subcarriers (116, 102, 90, 76, 62, 48, 36, 22, 10)
        if (((SCInd == -10'sd10 && RUType_1t < RU106)  || (SCInd == 10'sd10 && RUType_1t < RU106)  ||
             (SCInd == -10'sd22)  || (SCInd == 10'sd22)  ||
             (SCInd == -10'sd36 && RUType_1t < RU106)  || (SCInd == 10'sd36 && RUType_1t < RU106)  ||
             (SCInd == -10'sd48)  || (SCInd == 10'sd48)  ||
             (SCInd == -10'sd62 && RUType_1t < RU106)  || (SCInd == 10'sd62 && RUType_1t < RU106)  ||
             (SCInd == -10'sd76 && RUType_1t < RU106)  || (SCInd == 10'sd76 && RUType_1t < RU106)  ||
             (SCInd == -10'sd90)  || (SCInd == 10'sd90)  ||
             (SCInd == -10'sd102 && RUType_1t < RU106) || (SCInd == 10'sd102 && RUType_1t < RU106) ||
             (SCInd == -10'sd116) || (SCInd == 10'sd116)) && !FrameNDP)
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      default :
          CurrentSCIsPilot = 1'b0;
  endcase
end      

// Store pilots on buffer
assign PilotStore = CurrentSCIsPilot & LastLTF & (SmoothStep == IDLE_STEP) & (SymbolType_1t == VHT_LTF || ReceptionMode_1t >= MODE_20MHZ_HE_SU);

// Mux to skip CurrentSCIsPilot according to SymbolType and SmoothStep
assign CurrentSCIsPilotMux = ((!SmoothEn && (SymbolType_1t != VHT_LTF)) || 
                              (SmoothStep_1t == IDLE_STEP)              ||
                              (SmoothStep_1t == CHAN_INTERP)            ||
                              (SmoothStep_1t == PILOT_INTERP)           ||
   ((RxNsts != 2'b0) && (SmoothStep_1t == SMOOTHING || SmoothStep_1t == POST_CORDIC) && ((SymbolType_1t == VHT_LTF && !FrameNDP) || SymbolType_1t == HE_1xLTF ||  SymbolType_1t == HE_2xLTF || SymbolType_1t == HE_4xLTF))) ? 1'b0 : CurrentSCIsPilot;

// H Memory Data Write Enable
// Note that when RxNsts = 0 means there is one spatial stream
// RxNsts =1 means 2 and so on

always @ (*)
   begin: HBufWrEnInt_Blk

      HBufWrEnInt0 = 2'b0; //Default Value
      HBufWrEnInt1 = 2'b0;
      HBufWrEnInt2 = 2'b0;
      HBufWrEnInt3 = 2'b0;

      case (SymbolType_1t)

         L_LTF, HE_LSIG: begin
            //Enable First Column only
            HBufWrEnInt0 = {HWriteQ,HWriteI};
         end //L_LTF, HE_LSIG

         HT_LTF : begin
            if (FirstLTF == 1'b1) begin
               //When First LTF enable NSTS_PARAM number of columns
               HBufWrEnInt0 = {HWriteQ,HWriteI};
               if (NSTS_PARAM>32'd1)
                  HBufWrEnInt1 = {HWriteQ,HWriteI};
               if (NSTS_PARAM>32'd2)
                  HBufWrEnInt2 = {HWriteQ,HWriteI};
               if (NSTS_PARAM>32'd3)
                  HBufWrEnInt3 = {HWriteQ,HWriteI};
            end
            else begin
               
               //Enable first RxNsts number of columns only
               case (RxNsts)

                  2'b00: begin //Nsts = 1
                     HBufWrEnInt0 = {HWriteQ,HWriteI};
                  end
                  
                  2'b01: begin //Nsts = 2
                     if (NSTS_PARAM>32'd1)
                        {HBufWrEnInt0,HBufWrEnInt1} = {2{HWriteQ,HWriteI}};
                  end

                  2'b10: begin //Nsts = 3
                     if (NSTS_PARAM>32'd2)
                        {HBufWrEnInt0,HBufWrEnInt1,HBufWrEnInt2} = {3{HWriteQ,HWriteI}};
                  end

                  default: begin //Nsts = 4
                     if (NSTS_PARAM>32'd3)
                        {HBufWrEnInt0,HBufWrEnInt1,HBufWrEnInt2,HBufWrEnInt3} = {4{HWriteQ,HWriteI}};
                  end

               endcase //RxNsts
            end //FirstLTF == 1'b0
         end //HT_LTF

         default: begin // VHT_LTF, HE_1xLTF, HE_2xLTF, HE_4xLTF
            if (FirstLTF == 1'b1) begin
               //When First LTF enable NSTS_PARAM number of columns
               HBufWrEnInt0 = {HWriteQ,HWriteI};
               if (NSTS_PARAM>32'd1)
                  HBufWrEnInt1 = {HWriteQ,HWriteI};
               if (NSTS_PARAM>32'd2)
                  HBufWrEnInt2 = {HWriteQ,HWriteI};
               if (NSTS_PARAM>32'd3)
                  HBufWrEnInt3 = {HWriteQ,HWriteI};
            end
            else begin
               //Enable first RxNsts number of columns only
               case (RxNsts)

                  2'b00: begin //Nsts = 1
                     HBufWrEnInt0 = {HWriteQ,HWriteI};
                  end

                  2'b01: begin //Nsts = 2
                     HBufWrEnInt0 = {HWriteQ & ~CurrentSCIsPilotMux,HWriteI & ~CurrentSCIsPilotMux};
                     if (NSTS_PARAM>32'd1)
                        HBufWrEnInt1 = {HWriteQ & ~CurrentSCIsPilotMux, HWriteI & ~CurrentSCIsPilotMux};
                  end

                  2'b10: begin //Nsts = 3
                     HBufWrEnInt0 = {HWriteQ & ~CurrentSCIsPilotMux,HWriteI & ~CurrentSCIsPilotMux};
                     if (NSTS_PARAM>32'd1)
                        HBufWrEnInt1 = {HWriteQ & ~CurrentSCIsPilotMux, HWriteI & ~CurrentSCIsPilotMux};
                     if (NSTS_PARAM>32'd2)
                        HBufWrEnInt2 = {HWriteQ & ~CurrentSCIsPilotMux, HWriteI & ~CurrentSCIsPilotMux};
                  end

                  default: begin //Nsts = 4
                     HBufWrEnInt0 = {HWriteQ & ~CurrentSCIsPilotMux,HWriteI & ~CurrentSCIsPilotMux};
                     if (NSTS_PARAM>32'd1)
                        HBufWrEnInt1 = {HWriteQ & ~CurrentSCIsPilotMux, HWriteI & ~CurrentSCIsPilotMux};
                     if (NSTS_PARAM>32'd2)
                        HBufWrEnInt2 = {HWriteQ & ~CurrentSCIsPilotMux, HWriteI & ~CurrentSCIsPilotMux};
                     if (NSTS_PARAM>32'd3)
                        HBufWrEnInt3 = {HWriteQ & ~CurrentSCIsPilotMux, HWriteI & ~CurrentSCIsPilotMux};
                  end

               endcase //RxNsts
            end //FirstLTF == 1'b0
         end // VHT_LTF, HE_1xLTF, HE_2xLTF, HE_4xLTF

      endcase //SymbolType
   end //HBufWrEnInt_Blk


always @ (posedge PhyClk or negedge nPhyRst)
   begin: HBuf_Addr_Blk
      if (nPhyRst == 1'b0) begin
         HBufWrAddrOut   <= 10'b0;
         HBufWrAddrEnOut <= 1'b0;
      end
      else if (!EstimEn) begin
         HBufWrAddrOut   <= 10'b0;
         HBufWrAddrEnOut <= 1'b0;
      end
      else if (HWriteDataEnOut == 1'b0) begin
         HBufWrAddrEnOut <= 1'b0;
      end
      else if ((SmoothStep_1t == DC_INTERP) && (SymbolType_1t != HE_4xLTF) && !((SymbolType_1t == HE_2xLTF) && ((ReceptionMode_1t == MODE_40MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU))) && (ReceptionMode_1t != MODE_40MHZ_HT) && !((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t == RU26))) begin
         HBufWrAddrOut   <= 10'b0;
         HBufWrAddrEnOut <= HWriteDataEnOut;
      end
      else if (SCIndDone) begin
         HBufWrAddrOut   <= 10'b0;
         HBufWrAddrEnOut <= 1'b0;
      end
      else begin
         HBufWrAddrOut   <= SCInd;
         HBufWrAddrEnOut <= HWriteDataEnOut;
      end
   end // HBuf_Addr_Blk

// Generate SCIndDone
always @ (posedge PhyClk or negedge nPhyRst)
   begin: SCIndDone_Blk
      if (nPhyRst == 1'b0)
         SCIndDone <= 1'b0;
      else if (!HWriteDataEnOut)
         SCIndDone <= 1'b0;
      else if ($signed(SCInd) >= $signed(LastPosIndex_1t))
         SCIndDone <= 1'b1;
      else if ((((SmoothStep_1t == DC_INTERP) && (ReceptionMode_1t != MODE_40MHZ_HT) && 
                 (!((SymbolType_1t == HE_2xLTF) && ((ReceptionMode_1t == MODE_40MHZ_HE_SU) || (ReceptionMode_1t == MODE_40MHZ_HE_MU) || (ReceptionMode_1t == MODE_80MHZ_HE_MU)))) && 
                 (!((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (RUType_1t == RU26)))) || (SmoothStep_1t == VIRTUAL_INTERP) || (SmoothStep_1t == PILOT_INTERP)) &&
                 (SymbolType_1t != HE_4xLTF) && HWriteDataEnOut)
         SCIndDone <= 1'b1;
   end // SCIndDone_Blk


// Register HBufWrEn before sending out
always @ (posedge PhyClk or negedge nPhyRst)
   begin: HBufWrEn_Blk
      if (nPhyRst == 1'b0) begin
         HBufWrEn0 <= 2'b0;
         HBufWrEn1 <= 2'b0;
         HBufWrEn2 <= 2'b0;
         HBufWrEn3 <= 2'b0;
      end
      else if (!EstimEn || SCIndDone) begin
         HBufWrEn0 <= 2'b0;
         HBufWrEn1 <= 2'b0;
         HBufWrEn2 <= 2'b0;
         HBufWrEn3 <= 2'b0;
      end
      else if (HWriteDataEnOut) begin
         HBufWrEn0 <= HBufWrEnInt0;
         HBufWrEn1 <= HBufWrEnInt1;
         HBufWrEn2 <= HBufWrEnInt2;
         HBufWrEn3 <= HBufWrEnInt3;
      end
   end // HBufWrEn_Blk

// Last Write pulse
always @ (posedge PhyClk or negedge nPhyRst)
   begin: LastWriteP_Blk
      if (nPhyRst == 1'b0)
        LastWriteP <= 1'b0;
      else if (EstimEn == 1'b0)
        LastWriteP <= 1'b0;
      else begin
        LastWriteP <= 1'b0; // Default value
        if (((SmoothStep_1t == DC_INTERP) && !((SymbolType_1t == HE_2xLTF) && (ReceptionMode_1t == MODE_40MHZ_HE_SU)) && (ReceptionMode_1t != MODE_40MHZ_HE_MU) && (ReceptionMode_1t != MODE_80MHZ_HE_MU)) ||
             (SmoothStep_1t == VIRTUAL_INTERP) || (SmoothStep_1t == PILOT_INTERP)) begin
          if (SymbolType_1t != HE_4xLTF)
            LastWriteP <= HWriteDataEnOut;
          else begin
            if ((ReceptionMode_1t == MODE_20MHZ_HE_MU) && (SmoothStep_1t == DC_INTERP) && (RUType_1t != RU242))
              LastWriteP <= (SCInd == 10'sd3);
            else
              LastWriteP <= (SCInd == 10'sd1);
          end
        end
        else if ((SmoothStep_1t == CHAN_INTERP) && (SymbolType_1t == HE_1xLTF) && ($signed(SCInd) >= $signed(LastPosIndex_1t - 10'sd1)))
          LastWriteP <= 1'b1;
        else if (($signed(SCInd) >= $signed(LastPosIndex_1t)) && (SCInd != 10'b0))
          LastWriteP <= 1'b1;
      end
   end // LastWriteP_Blk

//pragma coverage off
// Assigning Strings to States so that it would be easy to debug
`ifdef RW_SIMU_ON
always@(*)
begin
  case (SymbolType)
    L_LTF    : SymbolTypeStr = {"L_LTF"};
    VHT_LTF  : SymbolTypeStr = {"VHT_LTF"};
    HT_LTF   : SymbolTypeStr = {"HT_LTF"};
    HE_1xLTF : SymbolTypeStr = {"HE_1xLTF"};
    HE_2xLTF : SymbolTypeStr = {"HE_2xLTF"};
    HE_4xLTF : SymbolTypeStr = {"HE_4xLTF"};
    HE_LSIG  : SymbolTypeStr = {"HE_LSIG"};
    default  : SymbolTypeStr = {"UNKNOWN"};
  endcase
end

always@(*)
begin
  case (SmoothStep)
    IDLE_STEP      : SmoothStepStr = {"IDLE_STEP"};
    SMOOTHING      : SmoothStepStr = {"SMOOTHING"};
    DC_INTERP      : SmoothStepStr = {"DC_INTERP"};
    VIRTUAL_INTERP : SmoothStepStr = {"VIRTUAL_INTERP"};
    PILOT_INTERP   : SmoothStepStr = {"PILOT_INTERP"};
    CHAN_INTERP    : SmoothStepStr = {"CHAN_INTERP"};
    POST_CORDIC    : SmoothStepStr = {"POST_CORDIC"};
    default        : SmoothStepStr = {"UNKNOWN"};
  endcase
end
`endif
//pragma coverage on

endmodule // HWriteCtrl

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
