//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission 
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39740 $
// $Date: 2019-09-09 18:42:06 +0200 (Mon, 09 Sep 2019) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Pilot Averaging for VHT Nsts=2
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/PilotAveraging.v $
//
//////////////////////////////////////////////////////////////////////////////

module PilotAveraging #(parameter DATAWIDTH  = 13
                 )(
    ///////////////////////////////////////////////
    //$port_g  Clock and Reset
    ///////////////////////////////////////////////
    input wire                       PhyClk,        // Phy clock
    input wire                       nPhyRst,       // Active Low Reset

    ///////////////////////////////////////////////
    //$port_g Register Interface
    ///////////////////////////////////////////////
    input  wire                      PilotAvEn,     // Indicate if the pilots shall be removed or not

    ///////////////////////////////////////////////
    //$port_g Control
    ///////////////////////////////////////////////
    input  wire signed [9:0]         SCIndex,       // Indicate the current SC index
    input  wire        [3:0]         ReceptionMode, // Reception Mode
    input  wire        [2:0]         RUType,        // Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)

    ///////////////////////////////////////////////
    //$port_g Data Interface
    ///////////////////////////////////////////////
    input  wire                      DataInValid,   // Flag indicating if the DataIn are valid
`ifdef RW_NX_DERIV_CHBW20ONLY
    input  wire [DATAWIDTH-1:0]      DataIn,        // Data In
    output reg  [DATAWIDTH-1:0]      DataOut        // Data Out
`else
    input  wire [DATAWIDTH-1:0]      DataInRe,      // Data In for Antenna 0 (real part)
    input  wire [DATAWIDTH-1:0]      DataInIm,      // Data In for Antenna 0 (imaginary part)

    output reg  [DATAWIDTH-1:0]      DataOutRe,     // Data Out for Antenna 0 (real part)
    output reg  [DATAWIDTH-1:0]      DataOutIm      // Data Out for Antenna 0 (imaginary part)
`endif
    );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////

// Reception Mode
localparam [3:0] MODE_20MHZ_L     = 4'd0,
                 MODE_20MHZ_HT    = 4'd2,
                 MODE_40MHZ_HT    = 4'd3,
                 MODE_20MHZ_HE_SU = 4'd6,
                 MODE_20MHZ_HE_MU = 4'd7,
                 MODE_40MHZ_HE_SU = 4'd8,
                 MODE_40MHZ_HE_MU = 4'd9,
                 MODE_80MHZ_HE_SU = 4'd10,
                 MODE_80MHZ_HE_MU = 4'd11;

// RUType definition
localparam [2:0] RU26  = 3'd0,
                 RU52  = 3'd1,
                 RU106 = 3'd2,
                 RU242 = 3'd3,
                 RU484 = 3'd4;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

`ifdef RW_NX_DERIV_CHBW20ONLY
reg  [DATAWIDTH-1:0] DataInDly1;
reg  [DATAWIDTH-1:0] DataOutDly1;

wire [DATAWIDTH:0]   DataAvr;
wire [DATAWIDTH-1:0] DataAvrRnd;
`else
reg  [DATAWIDTH-1:0] DataInReDly1;
reg  [DATAWIDTH-1:0] DataInImDly1;
reg  [DATAWIDTH-1:0] DataOutReDly1;
reg  [DATAWIDTH-1:0] DataOutImDly1;


wire [DATAWIDTH:0]   AvrRe;
wire [DATAWIDTH:0]   AvrIm;
wire [DATAWIDTH-1:0] AvrReRnd;
wire [DATAWIDTH-1:0] AvrImRnd;
`endif

reg                  CurrentSCIsPilot; // Indicates that the current SC is a pilot
reg                  CurrentSCIsPilotDly1;

reg signed [9:0]     CurrentSCIndex;

reg                  DataInValidDly1;
reg                  DataInValidDly2;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

`ifdef RW_NX_DERIV_CHBW20ONLY
// Compute the average value based on the previous and next SC value
assign DataAvr = {DataInDly1[DATAWIDTH-1],DataInDly1} + {DataOut[DATAWIDTH-1],DataOut};

// Round DataAvr by 1 bit
Round #(.INPUT_WIDTH(DATAWIDTH+1),
        .OUTPUT_WIDTH(DATAWIDTH)
       ) U_ROUND(
  .InputData(DataAvr),
  .RoundData(DataAvrRnd)
  );

// Delay line
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    DataInDly1 <= {(DATAWIDTH){1'b0}};
  else if (DataInValid)
    DataInDly1 <= DataIn;
end

// Delay line 
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    DataOutDly1 <= {(DATAWIDTH){1'b0}};
  else if (DataInValidDly1)
    DataOutDly1 <= DataInDly1;
end

// Output
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    DataOut <= {(DATAWIDTH){1'b0}};
  else if (DataInValidDly2)
  begin
    // If the SC is a pilot and the Averaging is enabled, 
    // the value is replaced by the Average between the previous and next SC
    if (PilotAvEn && CurrentSCIsPilotDly1)
      DataOut <= DataAvrRnd;
    else 
      DataOut <= DataOutDly1;
  end
end

`else

// Compute the average value based on the previous and next SC value
assign AvrRe = {DataInReDly1[DATAWIDTH-1],DataInReDly1} + {DataOutRe[DATAWIDTH-1],DataOutRe};
assign AvrIm = {DataInImDly1[DATAWIDTH-1],DataInImDly1} + {DataOutIm[DATAWIDTH-1],DataOutIm};

// Round DataAvr by 1 bit
Round #(.INPUT_WIDTH(DATAWIDTH+1),
        .OUTPUT_WIDTH(DATAWIDTH)
       ) U_ROUND_I(
  .InputData(AvrRe),
  .RoundData(AvrReRnd)
  );

Round #(.INPUT_WIDTH(DATAWIDTH+1),
        .OUTPUT_WIDTH(DATAWIDTH)
       ) U_ROUND_Q(
  .InputData(AvrIm),
  .RoundData(AvrImRnd)
  );

// Delay line 
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
  begin
    DataInReDly1 <= {(DATAWIDTH){1'b0}};
    DataInImDly1 <= {(DATAWIDTH){1'b0}};
  end
  else if (DataInValid)
  begin
    DataInReDly1 <= DataInRe;
    DataInImDly1 <= DataInIm;
  end
end

// Delay line 
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
  begin
    DataOutReDly1 <= {(DATAWIDTH){1'b0}};
    DataOutImDly1 <= {(DATAWIDTH){1'b0}};
  end
  else if (DataInValidDly1)
  begin
    DataOutReDly1 <= DataInReDly1;
    DataOutImDly1 <= DataInImDly1;
  end
end

// Output
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
  begin
    DataOutRe <= {(DATAWIDTH){1'b0}};
    DataOutIm <= {(DATAWIDTH){1'b0}};
  end
  else if (DataInValidDly2)
  begin
    // If the SC is a pilot and the Averaging is enabled, 
    // the value is replaced by the Average between the previous and next SC
    if (PilotAvEn && CurrentSCIsPilotDly1)
    begin
      DataOutRe <= AvrReRnd;
      DataOutIm <= AvrImRnd;
    end
    else 
    begin
      DataOutRe <= DataOutReDly1;
      DataOutIm <= DataOutImDly1;
    end  
  end
end

`endif

// Delay line of SCIndex
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    CurrentSCIndex <= 10'sd0;
  else if (DataInValid)
    CurrentSCIndex <= SCIndex;
end

// Delay of DataInValid
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
  begin
    DataInValidDly1 <= 1'd0;
    DataInValidDly2 <= 1'd0;
  end
  else
  begin
    DataInValidDly1 <= DataInValid;
    DataInValidDly2 <= DataInValidDly1;
  end  
end

always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    CurrentSCIsPilotDly1 <= 1'd0;
  else
    CurrentSCIsPilotDly1 <= CurrentSCIsPilot;
end

always @*
begin
  case (ReceptionMode)
      MODE_20MHZ_HT :
      begin
        // Pilot subcarriers (21, 7)
        if ((CurrentSCIndex == 10'sd7)  || (CurrentSCIndex == -10'sd7) ||
            (CurrentSCIndex == 10'sd21) || (CurrentSCIndex == -10'sd21))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_20MHZ_HE_SU : 
      begin
        // Pilot subcarriers (116, 90, 48, 22)
        if ((CurrentSCIndex == -10'sd22)  || (CurrentSCIndex == 10'sd22)  ||
            (CurrentSCIndex == -10'sd48)  || (CurrentSCIndex == 10'sd48)  ||
            (CurrentSCIndex == -10'sd90)  || (CurrentSCIndex == 10'sd90)  ||
            (CurrentSCIndex == -10'sd116) || (CurrentSCIndex == 10'sd116))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_20MHZ_HE_MU : 
      begin
        // Pilot subcarriers (116, 102, 90, 76, 62, 48, 36, 22, 10)
        if ((CurrentSCIndex == -10'sd10 && RUType < RU106)  || (CurrentSCIndex == 10'sd10 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd22)  || (CurrentSCIndex == 10'sd22)  ||
            (CurrentSCIndex == -10'sd36 && RUType < RU106)  || (CurrentSCIndex == 10'sd36 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd48)  || (CurrentSCIndex == 10'sd48)  ||
            (CurrentSCIndex == -10'sd62 && RUType < RU106)  || (CurrentSCIndex == 10'sd62 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd76 && RUType < RU106)  || (CurrentSCIndex == 10'sd76 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd90)  || (CurrentSCIndex == 10'sd90)  ||
            (CurrentSCIndex == -10'sd102 && RUType < RU106) || (CurrentSCIndex == 10'sd102 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd116) || (CurrentSCIndex == 10'sd116))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end
      
      MODE_40MHZ_HT : 
      begin
        // Pilot subcarriers (53, 25, 11)
        if ((CurrentSCIndex == 10'sd11) || (CurrentSCIndex == -10'sd11) || 
            (CurrentSCIndex == 10'sd25) || (CurrentSCIndex == -10'sd25) || 
            (CurrentSCIndex == 10'sd53) || (CurrentSCIndex == -10'sd53))
          CurrentSCIsPilot = 1'b1;
        else
          CurrentSCIsPilot = 1'b0;
      end

      MODE_40MHZ_HE_SU : 
      begin
        // Pilot subcarriers (238, 212, 170, 144, 104, 78, 36,10)
        if (
            (CurrentSCIndex == -10'sd10)  || (CurrentSCIndex == 10'sd10)  ||
            (CurrentSCIndex == -10'sd36)  || (CurrentSCIndex == 10'sd36)  ||
            (CurrentSCIndex == -10'sd78)  || (CurrentSCIndex == 10'sd78)  ||
            (CurrentSCIndex == -10'sd104) || (CurrentSCIndex == 10'sd104) ||
            (CurrentSCIndex == -10'sd144) || (CurrentSCIndex == 10'sd144) ||
            (CurrentSCIndex == -10'sd170) || (CurrentSCIndex == 10'sd170) ||
            (CurrentSCIndex == -10'sd212) || (CurrentSCIndex == 10'sd212) ||
            (CurrentSCIndex == -10'sd238) || (CurrentSCIndex == 10'sd238))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_40MHZ_HE_MU : 
      begin
        // Pilot subcarriers (238, 224, 212, 198, 184, 170, 158, 144,
        // 130, 116, 104, 90, 78, 64, 50, 36, 24, 10)
        if ((CurrentSCIndex == -10'sd10)  || (CurrentSCIndex == 10'sd10)  ||
            (CurrentSCIndex == -10'sd24 && RUType < RU106)  || (CurrentSCIndex == 10'sd24 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd36)  || (CurrentSCIndex == 10'sd36)  ||
            (CurrentSCIndex == -10'sd50 && RUType < RU106)  || (CurrentSCIndex == 10'sd50 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd64 && RUType < RU106)  || (CurrentSCIndex == 10'sd64 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd78)  || (CurrentSCIndex == 10'sd78)  ||
            (CurrentSCIndex == -10'sd90 && RUType < RU106)  || (CurrentSCIndex == 10'sd90 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd104) || (CurrentSCIndex == 10'sd104) ||
            (CurrentSCIndex == -10'sd116 && RUType < RU106) || (CurrentSCIndex == 10'sd116 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd130 && RUType < RU106) || (CurrentSCIndex == 10'sd130 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd144) || (CurrentSCIndex == 10'sd144) ||
            (CurrentSCIndex == -10'sd158 && RUType < RU106) || (CurrentSCIndex == 10'sd158 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd170) || (CurrentSCIndex == 10'sd170) ||
            (CurrentSCIndex == -10'sd184 && RUType < RU106) || (CurrentSCIndex == 10'sd184 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd198 && RUType < RU106) || (CurrentSCIndex == 10'sd198 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd212) || (CurrentSCIndex == 10'sd212) ||
            (CurrentSCIndex == -10'sd224 && RUType < RU106) || (CurrentSCIndex == 10'sd224 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd238) || (CurrentSCIndex == 10'sd238))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      MODE_80MHZ_HE_MU : 
      begin
        // Pilot subcarriers (494, 480, 468, 454, 440, 426,
        // 414, 400, 386, 372, 360, 346, 334, 320,
        // 306, 292, 280, 266, 252,
        // 238, 226, 212, 198, 184, 172, 158, 144,
        // 130, 118, 104, 92, 78, 64, 50, 38, 24, 10)
        if ((CurrentSCIndex == -10'sd10 && RUType < RU106)  || (CurrentSCIndex == 10'sd10 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd24)  || (CurrentSCIndex == 10'sd24)  ||
            (CurrentSCIndex == -10'sd38 && RUType < RU106)  || (CurrentSCIndex == 10'sd38 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd50)  || (CurrentSCIndex == 10'sd50)  ||
            (CurrentSCIndex == -10'sd64 && RUType < RU106)  || (CurrentSCIndex == 10'sd64 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd78 && RUType < RU106)  || (CurrentSCIndex == 10'sd78 && RUType < RU106)  ||
            (CurrentSCIndex == -10'sd92)  || (CurrentSCIndex == 10'sd92)  ||
            (CurrentSCIndex == -10'sd104 && RUType < RU106) || (CurrentSCIndex == 10'sd104 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd118) || (CurrentSCIndex == 10'sd118) ||
            (CurrentSCIndex == -10'sd130 && RUType < RU106) || (CurrentSCIndex == 10'sd130 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd144 && RUType < RU106) || (CurrentSCIndex == 10'sd144 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd158 && RUType < RU106) || (CurrentSCIndex == 10'sd158 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd172 && RUType < RU106) || (CurrentSCIndex == 10'sd172 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd184) || (CurrentSCIndex == 10'sd184) ||
            (CurrentSCIndex == -10'sd198 && RUType < RU106) || (CurrentSCIndex == 10'sd198 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd212 && RUType < RU106) || (CurrentSCIndex == 10'sd212 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd226) || (CurrentSCIndex == 10'sd226) ||
            (CurrentSCIndex == -10'sd238 && RUType < RU106) || (CurrentSCIndex == 10'sd238 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd252) || (CurrentSCIndex == 10'sd252) ||
            (CurrentSCIndex == -10'sd266) || (CurrentSCIndex == 10'sd266) ||
            (CurrentSCIndex == -10'sd280 && RUType < RU106) || (CurrentSCIndex == 10'sd280 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd292) || (CurrentSCIndex == 10'sd292) ||
            (CurrentSCIndex == -10'sd306 && RUType < RU106) || (CurrentSCIndex == 10'sd306 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd320 && RUType < RU106) || (CurrentSCIndex == 10'sd320 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd334) || (CurrentSCIndex == 10'sd334) ||
            (CurrentSCIndex == -10'sd346 && RUType < RU106) || (CurrentSCIndex == 10'sd346 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd360) || (CurrentSCIndex == 10'sd360) ||
            (CurrentSCIndex == -10'sd372 && RUType < RU106) || (CurrentSCIndex == 10'sd372 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd386 && RUType < RU106) || (CurrentSCIndex == 10'sd386 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd400) || (CurrentSCIndex == 10'sd400) ||
            (CurrentSCIndex == -10'sd414 && RUType < RU106) || (CurrentSCIndex == 10'sd414 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd426) || (CurrentSCIndex == 10'sd426) ||
            (CurrentSCIndex == -10'sd440 && RUType < RU106) || (CurrentSCIndex == 10'sd440 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd454 && RUType < RU106) || (CurrentSCIndex == 10'sd454 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd468) || (CurrentSCIndex == 10'sd468) ||
            (CurrentSCIndex == -10'sd480 && RUType < RU106) || (CurrentSCIndex == 10'sd480 && RUType < RU106) ||
            (CurrentSCIndex == -10'sd494) || (CurrentSCIndex == 10'sd494))
          CurrentSCIsPilot = 1'b1;
        else 
          CurrentSCIsPilot = 1'b0;
      end

      default :
          CurrentSCIsPilot = 1'b0;
  endcase
end

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
