//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39740 $
// $Date: 2019-09-09 18:42:06 +0200 (Mon, 09 Sep 2019) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Smoothing Filter Top Level Module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/SmoothTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module SmoothTop #(parameter DATAWIDTH    = 13
                  )(

            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input   wire                                   nPhyRst,  // Active Low Reset
            input   wire                                   PhyClk,   // Phy Clock

            ///////////////////////////////////////////////
            // Controls
            ///////////////////////////////////////////////
            input   wire                                   BlockEn,            // Enable Block
            input   wire [3:0]                             ReceptionMode,      // Mode of Current Pkt
            input   wire [2:0]                             SymbolType,         // LTF of Current Pkt
            input   wire [2:0]                             RUType,             // Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)
            //
            input   wire                                   PilotAvEn,          // Enable Pilot Averaging
            input   wire [9:0]                             PilotAvSCIndex,     // SC Index at Pilot Averaging input
            input   wire                                   PilotAvDataInValid, // Enable Pilot Averaging input
            //
            input   wire                                   RotCordicEnIn,      // Enable Rotation
            input   wire [1:0]                             RotQuad,            // Rotation Quadrant
            input   wire [3:0]                             RotIndex,           // Rotation Index
            input   wire                                   DivByJ,             // Indicates upper 20 MHz
            //
            input   wire                                   MultByJ,            // Indicates upper 20 MHz
            input   wire                                   SelIQPhase,         // Select I or Q phase for filter
            //
            input   wire                                   PilotStore,         // Store pilots in buffer
            input   wire                                   PilotRead,          // Read pilots from buffer

            ///////////////////////////////////////////////
            // Filter interface
            ///////////////////////////////////////////////
            input   wire                                   SmoothEnIn,         // Enable smoothing filter
            input   wire                                   ShiftIn,            // Enable input to filter
            //
            input   wire signed     [9:0]                  SmoothCoeff0,  // Smoothing Coeffecient 0
            input   wire signed     [9:0]                  SmoothCoeff1,  // Smoothing Coeffecient 1
            input   wire signed     [9:0]                  SmoothCoeff2,  // Smoothing Coeffecient 2
            input   wire signed     [9:0]                  SmoothCoeff3,  // Smoothing Coeffecient 3
            input   wire signed     [9:0]                  SmoothCoeff4,  // Smoothing Coeffecient 4
            input   wire signed     [9:0]                  SmoothCoeff5,  // Smoothing Coeffecient 5
            input   wire signed     [9:0]                  SmoothCoeff6,  // Smoothing Coeffecient 6
            input   wire signed     [9:0]                  SmoothCoeff7,  // Smoothing Coeffecient 7
            input   wire signed     [9:0]                  SmoothCoeff8,  // Smoothing Coeffecient 8
            input   wire signed     [9:0]                  SmoothCoeff9,  // Smoothing Coeffecient 9
            input   wire signed     [9:0]                  SmoothCoeff10, // Smoothing Coeffecient 10
            input   wire signed     [9:0]                  SmoothCoeff11, // Smoothing Coeffecient 11
            input   wire            [1:0]                  SmoothCoeffRS, // Smoothing Right Shift

            ///////////////////////////////////////////////
            // Data
            ///////////////////////////////////////////////
            input   wire signed     [DATAWIDTH-1:0]        SmoothDataInRe, // Real Component
            input   wire signed     [DATAWIDTH-1:0]        SmoothDataInIm, // Imag Component
            //
            output  reg signed      [DATAWIDTH-1:0]        SmoothDataOutRe, // Real Component
            output  reg signed      [DATAWIDTH-1:0]        SmoothDataOutIm  // Imag Component
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////

// Cordic iteration
localparam ITERATION = 13;

// ReceptionMode definition
localparam [3:0] MODE_20MHZ_L     = 4'd0,
                 MODE_20MHZ_HT    = 4'd2,
                 MODE_40MHZ_HT    = 4'd3,
                 MODE_20MHZ_HE_SU = 4'd6,
                 MODE_20MHZ_HE_MU = 4'd7,
                 MODE_40MHZ_HE_SU = 4'd8,
                 MODE_40MHZ_HE_MU = 4'd9,
                 MODE_80MHZ_HE_SU = 4'd10,
                 MODE_80MHZ_HE_MU = 4'd11;

// SymbolType definition
localparam [2:0] L_LTF    = 3'd0,
                 VHT_LTF  = 3'd1,
                 HT_LTF   = 3'd2,
                 HE_1xLTF = 3'd4,
                 HE_2xLTF = 3'd5,
                 HE_4xLTF = 3'd6,
                 HE_LSIG  = 3'd7;

// RUType definition
localparam [2:0] RU26  = 3'd0,
                 RU52  = 3'd1,
                 RU106 = 3'd2,
                 RU242 = 3'd3,
                 RU484 = 3'd4;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire                                   SmoothEn;
wire                                   RotEnable;
wire    signed     [DATAWIDTH-1:0]     CordicInRe;
wire    signed     [DATAWIDTH-1:0]     CordicInIm;
wire    signed     [DATAWIDTH-1:0]     CordicOutRe;
wire    signed     [DATAWIDTH-1:0]     CordicOutIm;
`ifdef RW_NX_DERIV_CHBW20ONLY
wire    signed     [DATAWIDTH-1:0]     PilotAveragingOut;
wire    signed     [DATAWIDTH-1:0]     SmoothDataIn;
wire    signed     [DATAWIDTH-1:0]     SmoothFilterOut;
`else
wire    signed     [DATAWIDTH-1:0]     CordicInReMux;
wire    signed     [DATAWIDTH-1:0]     CordicInImMux;
wire    signed     [DATAWIDTH-1:0]     CordicInReMux2;
wire    signed     [DATAWIDTH-1:0]     CordicInImMux2;
wire    signed     [DATAWIDTH-1:0]     PilotAveragingOutRe;
wire    signed     [DATAWIDTH-1:0]     PilotAveragingOutIm;
wire    signed     [DATAWIDTH-1:0]     SmoothFilterOutRe;
wire    signed     [DATAWIDTH-1:0]     SmoothFilterOutIm;
`endif

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [DATAWIDTH-1:0]     PilotRe[0:15];
reg                [DATAWIDTH-1:0]     PilotIm[0:15];
reg                [4:0]               PilotCount;

integer i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Enables for logic generated using BlockEn
assign RotEnable = BlockEn & RotCordicEnIn;
assign SmoothEn  = BlockEn & SmoothEnIn;


// DivByJ and MultByJ signals are used to indicate that the upper 20 MHz
// subcarriers are being processed. When DivByJ is high, the inputs to the
// pre rotation Cordic are divided by j. When MultByJ is high the output
// of the post rotation Cordic is multiplied by J and given out.
// This is required because the smoothing process shouldnt be influenced
// by the j factor of the upper subcarriers.

// Rotation Cordic
// Input to Cordic
`ifdef RW_NX_DERIV_CHBW20ONLY
assign CordicInRe = (PilotRead) ? PilotRe[PilotCount[3:0]] : SmoothDataInRe;
assign CordicInIm = (PilotRead) ? PilotIm[PilotCount[3:0]] : SmoothDataInIm;
`else
assign CordicInReMux  = (SmoothEn)  ?  SmoothFilterOutRe   : SmoothDataInRe;
assign CordicInImMux  = (SmoothEn)  ?  SmoothFilterOutIm   : SmoothDataInIm;
assign CordicInReMux2 = (PilotRead) ?  PilotRe[PilotCount[3:0]] : CordicInReMux;
assign CordicInImMux2 = (PilotRead) ?  PilotIm[PilotCount[3:0]] : CordicInImMux;
assign CordicInRe     = (DivByJ)    ?  CordicInImMux2      : CordicInReMux2;
assign CordicInIm     = (DivByJ)    ? -CordicInReMux2      : CordicInImMux2;
`endif

// Pre-rotation Cordic
// Instantiate Cordic
CordicTop # (
             .ITERATION(ITERATION),
             .DATAWIDTH(DATAWIDTH)
            ) 
             U_CordicTop (
                             //Inputs
                             .PhyClk(PhyClk),
                             .nPhyRst(nPhyRst),
                             .CordicEn(RotEnable),
                             .Index(RotIndex),
                             .QuadRot(RotQuad),
                             .ReDataIn(CordicInRe),
                             .ImDataIn(CordicInIm),
                             // Outputs
                             .ReDataOut(CordicOutRe),
                             .ImDataOut(CordicOutIm)
                            );

// Mux filter input
`ifdef RW_NX_DERIV_CHBW20ONLY
assign SmoothDataIn = (SelIQPhase) ? SmoothDataInIm : SmoothDataInRe;
`endif

// Instantiate Pilot Averaging
PilotAveraging # (
             .DATAWIDTH(DATAWIDTH)
            ) 
             U_PilotAveraging (
                             .PhyClk(PhyClk),
                             .nPhyRst(nPhyRst),
                             .PilotAvEn(PilotAvEn),
                             .SCIndex(PilotAvSCIndex),
                             .ReceptionMode(ReceptionMode),
                             .RUType(RUType),
                             .DataInValid(PilotAvDataInValid),
`ifdef RW_NX_DERIV_CHBW20ONLY
                             .DataIn(SmoothDataIn),
                             .DataOut(PilotAveragingOut)
`else
                             .DataInRe(SmoothDataInRe),
                             .DataInIm(SmoothDataInIm),
                             .DataOutRe(PilotAveragingOutRe),
                             .DataOutIm(PilotAveragingOutIm)
`endif
                            );

// Mux from PilotAveraging

// Instantiate Smoothing Filter
`ifdef RW_NX_DERIV_CHBW20ONLY
Smoother # (
            .DATAWIDTH(DATAWIDTH)
           )
            U_Smoother (
                          .nPhyRst(nPhyRst),
                          .PhyClk(PhyClk),
                          .ShiftIn(ShiftIn),
                          .SmoothEn(SmoothEn),
                          .SmoothCoeff0(SmoothCoeff0),
                          .SmoothCoeff1(SmoothCoeff1),
                          .SmoothCoeff2(SmoothCoeff2),
                          .SmoothCoeff3(SmoothCoeff3),
                          .SmoothCoeff4(SmoothCoeff4),
                          .SmoothCoeff5(SmoothCoeff5),
                          .SmoothCoeff6(SmoothCoeff6),
                          .SmoothCoeff7(SmoothCoeff7),
                          .SmoothCoeff8(SmoothCoeff8),
                          .SmoothCoeff9(SmoothCoeff9),
                          .SmoothCoeff10(SmoothCoeff10),
                          .SmoothCoeff11(SmoothCoeff11),
                          .SmoothCoeffRS(SmoothCoeffRS),

                          .SmoothDataIn(PilotAveragingOut),
                          .SmoothDataOut(SmoothFilterOut)
                          );

`else

Smoother # (
            .DATAWIDTH(DATAWIDTH)
           )
            U_SmootherRe (
                          .nPhyRst(nPhyRst),
                          .PhyClk(PhyClk),
                          .ShiftIn(ShiftIn),
                          .SmoothEn(SmoothEn),
                          .SmoothCoeff0(SmoothCoeff0),
                          .SmoothCoeff1(SmoothCoeff1),
                          .SmoothCoeff2(SmoothCoeff2),
                          .SmoothCoeff3(SmoothCoeff3),
                          .SmoothCoeff4(SmoothCoeff4),
                          .SmoothCoeff5(SmoothCoeff5),
                          .SmoothCoeff6(SmoothCoeff6),
                          .SmoothCoeff7(SmoothCoeff7),
                          .SmoothCoeff8(SmoothCoeff8),
                          .SmoothCoeff9(SmoothCoeff9),
                          .SmoothCoeff10(SmoothCoeff10),
                          .SmoothCoeff11(SmoothCoeff11),
                          .SmoothCoeffRS(SmoothCoeffRS),

                          .SmoothDataIn(PilotAveragingOutRe),
                          .SmoothDataOut(SmoothFilterOutRe)
                          );

Smoother # (
            .DATAWIDTH(DATAWIDTH)
           )
            U_SmootherIm (
                          .nPhyRst(nPhyRst),
                          .PhyClk(PhyClk),
                          .ShiftIn(ShiftIn),
                          .SmoothEn(SmoothEn),
                          .SmoothCoeff0(SmoothCoeff0),
                          .SmoothCoeff1(SmoothCoeff1),
                          .SmoothCoeff2(SmoothCoeff2),
                          .SmoothCoeff3(SmoothCoeff3),
                          .SmoothCoeff4(SmoothCoeff4),
                          .SmoothCoeff5(SmoothCoeff5),
                          .SmoothCoeff6(SmoothCoeff6),
                          .SmoothCoeff7(SmoothCoeff7),
                          .SmoothCoeff8(SmoothCoeff8),
                          .SmoothCoeff9(SmoothCoeff9),
                          .SmoothCoeff10(SmoothCoeff10),
                          .SmoothCoeff11(SmoothCoeff11),
                          .SmoothCoeffRS(SmoothCoeffRS),

                          .SmoothDataIn(PilotAveragingOutIm),
                          .SmoothDataOut(SmoothFilterOutIm)
                          );

`endif

// Store pilots
always @ (posedge PhyClk or negedge nPhyRst)
   begin: PilotBuffer_Blk
      if (nPhyRst == 1'b0) begin
         for (i=0; i<16;i=i+1) begin
            PilotRe[i] <= {DATAWIDTH{1'b0}};
            PilotIm[i] <= {DATAWIDTH{1'b0}};
         end
         PilotCount <= 5'b0;
      end
      else if (!RotEnable && !SmoothEn) begin
         PilotCount <= 5'b0;
      end
      else begin
         if ((PilotCount==5'd16)                                                                                                        ||
             (PilotCount==5'd8  && ((ReceptionMode == MODE_20MHZ_HE_SU)                                || 
                                   ((SymbolType == HE_1xLTF) && (ReceptionMode == MODE_40MHZ_HE_SU))   ||
                                   ((RUType == RU242) && (ReceptionMode == MODE_20MHZ_HE_MU || ReceptionMode == MODE_40MHZ_HE_MU))))    ||
             (PilotCount==5'd6  && (ReceptionMode == MODE_40MHZ_HT))                                                                    ||
             (PilotCount==5'd4  && (ReceptionMode == MODE_20MHZ_HT || (SymbolType == HE_1xLTF && ReceptionMode == MODE_20MHZ_HE_SU) ||
                                   ((RUType == RU106 || RUType == RU52) && (ReceptionMode == MODE_20MHZ_HE_MU || ReceptionMode == MODE_40MHZ_HE_MU || ReceptionMode == MODE_80MHZ_HE_MU)))) ||
             (PilotCount==5'd2  && ((RUType == RU26) && (ReceptionMode == MODE_20MHZ_HE_MU || ReceptionMode == MODE_40MHZ_HE_MU || ReceptionMode == MODE_80MHZ_HE_MU))))
           PilotCount <= 5'b0;
         else if (PilotStore || PilotRead)
           PilotCount <= PilotCount + 5'd1;
         
         if (PilotStore) begin
           PilotRe[PilotCount[3:0]] <= SmoothDataOutRe;
           PilotIm[PilotCount[3:0]] <= SmoothDataOutIm;
         end
      end
   end //PilotBuffer_Blk

// Generate Outputs
always @ (*)
   begin: Output_Blk
      if (SmoothEn == 1'b0 && RotEnable == 1'b0) begin
         // Bypass
         SmoothDataOutRe = SmoothDataInRe;
         SmoothDataOutIm = SmoothDataInIm;
      end
      else if (SmoothEn == 1'b1 && RotEnable == 1'b0) begin
         // Smoothing filter
         if (PilotRead) begin
           SmoothDataOutRe = PilotRe[PilotCount[3:0]];
           SmoothDataOutIm = PilotIm[PilotCount[3:0]];
         end
         else begin
`ifdef RW_NX_DERIV_CHBW20ONLY
           SmoothDataOutRe = SmoothFilterOut;
           SmoothDataOutIm = SmoothFilterOut;
`else
           SmoothDataOutRe = SmoothFilterOutRe;
           SmoothDataOutIm = SmoothFilterOutIm;
`endif
         end
      end
`ifndef RW_NX_DERIV_CHBW20ONLY
      else if (MultByJ == 1'b1) begin
         // Multiply outputs by J
         // Rotation
         SmoothDataOutRe = -CordicOutIm;
         SmoothDataOutIm =  CordicOutRe;
      end
`endif
      else begin
         // Rotation
         SmoothDataOutRe = CordicOutRe;
         SmoothDataOutIm = CordicOutIm;
      end
   end //Output_Blk

endmodule // SmoothTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
