//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 35679 $
// $Date: 2018-10-12 18:06:26 +0200 (Fri, 12 Oct 2018) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Smoothing Filter
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/ChEstSmth/verilog/rtl/Smoother.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module Smoother #(parameter DATAWIDTH  = 13
                 )(

            ///////////////////////////////////////////////
            //Clock and Reset
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                                   nPhyRst,  // Active Low Reset
            input   wire                                   PhyClk,   // Phy Clock

            ///////////////////////////////////////////////
            // Control
            ///////////////////////////////////////////////
            input   wire                                   ShiftIn,       // Enable input to filter
            input   wire                                   SmoothEn,      // Enable Smoothing

            // Filter Coeffecients
            input   wire   signed [9:0]                    SmoothCoeff0,  // Smoothing Coeffecient 0.
            input   wire   signed [9:0]                    SmoothCoeff1,  // Smoothing Coeff 1.
            input   wire   signed [9:0]                    SmoothCoeff2,  // Smoothing Coeff 2.
            input   wire   signed [9:0]                    SmoothCoeff3,  // Smoothing Coeff 3.
            input   wire   signed [9:0]                    SmoothCoeff4,  // Smoothing Coeff 4.
            input   wire   signed [9:0]                    SmoothCoeff5,  // Smoothing Coeff 5.
            input   wire   signed [9:0]                    SmoothCoeff6,  // Smoothing Coeff 6.
            input   wire   signed [9:0]                    SmoothCoeff7,  // Smoothing Coeff 7.
            input   wire   signed [9:0]                    SmoothCoeff8,  // Smoothing Coeff 8.
            input   wire   signed [9:0]                    SmoothCoeff9,  // Smoothing Coeff 9.
            input   wire   signed [9:0]                    SmoothCoeff10, // Smoothing Coeff 10.
            input   wire   signed [9:0]                    SmoothCoeff11, // Smoothing Coeff 11.

            // Filter Right shift
            input   wire          [1:0]                    SmoothCoeffRS,

            ///////////////////////////////////////////////
            // Data
            ///////////////////////////////////////////////
            input   wire signed [DATAWIDTH-1:0]            SmoothDataIn,  // Data Input
            output  reg  signed [DATAWIDTH-1:0]            SmoothDataOut  // Data Output
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam  DATAWIDTH_P9  = DATAWIDTH+9;
localparam  DATAWIDTH_P10 = DATAWIDTH+10;
localparam  DATAWIDTH_P11 = DATAWIDTH+11;
localparam  DATAWIDTH_P12 = DATAWIDTH+12;
localparam  DATAWIDTH_P13 = DATAWIDTH+13;
localparam  DATAWIDTH_P14 = DATAWIDTH+14;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires & Registers Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed     [DATAWIDTH_P13:0]   ShiftSmoothData;
wire    signed     [DATAWIDTH+4:0]     RndSmoothData;
wire    signed     [DATAWIDTH-1:0]     SatSmoothData;

reg     signed     [DATAWIDTH_P10:0]   IntSum0;
reg     signed     [DATAWIDTH_P10:0]   IntSum1;
reg     signed     [DATAWIDTH_P10:0]   IntSum2;
reg     signed     [DATAWIDTH_P10:0]   IntSum3;
reg     signed     [DATAWIDTH_P10:0]   IntSum4;
reg     signed     [DATAWIDTH_P10:0]   IntSum5;
reg     signed     [DATAWIDTH-1:0]     FilterTap0;
reg     signed     [DATAWIDTH-1:0]     FilterTap1;
reg     signed     [DATAWIDTH-1:0]     FilterTap2;
reg     signed     [DATAWIDTH-1:0]     FilterTap3;
reg     signed     [DATAWIDTH-1:0]     FilterTap4;
reg     signed     [DATAWIDTH-1:0]     FilterTap5;
reg     signed     [DATAWIDTH-1:0]     FilterTap6;
reg     signed     [DATAWIDTH-1:0]     FilterTap7;
reg     signed     [DATAWIDTH-1:0]     FilterTap8;
reg     signed     [DATAWIDTH-1:0]     FilterTap9;
reg     signed     [DATAWIDTH-1:0]     FilterTap10;
reg     signed     [DATAWIDTH-1:0]     FilterTap11;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd0;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd1;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd2;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd3;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd4;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd5;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd6;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd7;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd8;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd9;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd10;
reg     signed     [DATAWIDTH_P9:0]    FilterAdd11;
reg     signed     [DATAWIDTH_P11:0]   SmoothDataInt0;
reg     signed     [DATAWIDTH_P11:0]   SmoothDataInt1;
reg     signed     [DATAWIDTH_P11:0]   SmoothDataInt2;
reg     signed     [DATAWIDTH_P12:0]   SmoothDataInt;
reg     signed     [DATAWIDTH-1:0]     SmDataInD;
reg                                    ShiftInD;
reg                [1:0]               SmoothCoeffRS_1t;
reg                [1:0]               SmoothCoeffRS_2t;
reg                [1:0]               SmoothCoeffRS_3t;
reg                [1:0]               SmoothCoeffRS_4t;
reg     signed     [DATAWIDTH_P13:0]   ShiftSmoothData_1t;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module performs smoothing of the input data.

// Step 1 - Read data into the filter
// An extra delay is added at the input to compensate for
// one clock delay required in selecting the appropriate
// filter coeffecients for a particular subcarrier
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Delay_Blk
      if (nPhyRst == 1'b0) begin
         SmDataInD <= $signed({{DATAWIDTH}{1'b0}});
         ShiftInD  <= 1'b0;
      end
      else begin
         SmDataInD <= SmoothDataIn;
         ShiftInD  <= ShiftIn;
      end
   end //Delay_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: Filter_In_Blk
      if (nPhyRst == 1'b0) begin
         FilterTap0  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap1  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap2  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap3  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap4  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap5  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap6  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap7  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap8  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap9  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap10 <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap11 <= $signed({{DATAWIDTH}{1'b0}});
      end
      else if (SmoothEn == 1'b0) begin
         //Retain Data
         FilterTap0  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap1  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap2  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap3  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap4  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap5  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap6  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap7  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap8  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap9  <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap10 <= $signed({{DATAWIDTH}{1'b0}});
         FilterTap11 <= $signed({{DATAWIDTH}{1'b0}});
      end
      else if (ShiftInD == 1'b1) begin
         //Shift Data In
         FilterTap0  <= SmDataInD;
         FilterTap1  <= FilterTap0;
         FilterTap2  <= FilterTap1;
         FilterTap3  <= FilterTap2;
         FilterTap4  <= FilterTap3;
         FilterTap5  <= FilterTap4;
         FilterTap6  <= FilterTap5;
         FilterTap7  <= FilterTap6;
         FilterTap8  <= FilterTap7;
         FilterTap9  <= FilterTap8;
         FilterTap10 <= FilterTap9;
         FilterTap11 <= FilterTap10;
      end
   end //Filter_In_Blk

// Multiply the filter inputs with the corresponding
// coeffecients and register the product
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Adder_Blk
      if (nPhyRst == 1'b0) begin
         FilterAdd0  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd1  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd2  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd3  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd4  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd5  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd6  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd7  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd8  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd9  <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd10 <= $signed({{DATAWIDTH_P10}{1'b0}});
         FilterAdd11 <= $signed({{DATAWIDTH_P10}{1'b0}});
      end
      else if (SmoothEn == 1'b1) begin
         //Register Products
         FilterAdd0  <= $signed(FilterTap0)  * $signed(SmoothCoeff0);
         FilterAdd1  <= $signed(FilterTap1)  * $signed(SmoothCoeff1);
         FilterAdd2  <= $signed(FilterTap2)  * $signed(SmoothCoeff2);
         FilterAdd3  <= $signed(FilterTap3)  * $signed(SmoothCoeff3);
         FilterAdd4  <= $signed(FilterTap4)  * $signed(SmoothCoeff4);
         FilterAdd5  <= $signed(FilterTap5)  * $signed(SmoothCoeff5);
         FilterAdd6  <= $signed(FilterTap6)  * $signed(SmoothCoeff6);
         FilterAdd7  <= $signed(FilterTap7)  * $signed(SmoothCoeff7);
         FilterAdd8  <= $signed(FilterTap8)  * $signed(SmoothCoeff8);
         FilterAdd9  <= $signed(FilterTap9)  * $signed(SmoothCoeff9);
         FilterAdd10 <= $signed(FilterTap10) * $signed(SmoothCoeff10);
         FilterAdd11 <= $signed(FilterTap11) * $signed(SmoothCoeff11);
      end
   end //Adder_Blk

// Step 3 - Add the intermediate values
// Register after the first stage, then the second stage 
// and register the final sum
// This is required to meet timing on FPGA
always @ (posedge PhyClk or negedge nPhyRst)
   begin: IntSum_Blk
      if (nPhyRst == 1'b0) begin
         IntSum0 <= $signed({{DATAWIDTH_P11}{1'b0}});
         IntSum1 <= $signed({{DATAWIDTH_P11}{1'b0}});
         IntSum2 <= $signed({{DATAWIDTH_P11}{1'b0}});
         IntSum3 <= $signed({{DATAWIDTH_P11}{1'b0}});
         IntSum4 <= $signed({{DATAWIDTH_P11}{1'b0}});
         IntSum5 <= $signed({{DATAWIDTH_P11}{1'b0}});
      end
      else if (SmoothEn == 1'b1) begin
         //Compute First Stage Sums and Register
         IntSum0 <= FilterAdd0 + FilterAdd1;
         IntSum1 <= FilterAdd2 + FilterAdd3;
         IntSum2 <= FilterAdd4 + FilterAdd5;
         IntSum3 <= FilterAdd6 + FilterAdd7;
         IntSum4 <= FilterAdd8 + FilterAdd9;
         IntSum5 <= FilterAdd10 + FilterAdd11;
      end
   end //IntSum_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: SmoothDataIntn_Blk
      if (nPhyRst == 1'b0) begin
         SmoothDataInt0 <= $signed({{DATAWIDTH_P12}{1'b0}});
         SmoothDataInt1 <= $signed({{DATAWIDTH_P12}{1'b0}});
         SmoothDataInt2 <= $signed({{DATAWIDTH_P12}{1'b0}});
      end
      else if (SmoothEn == 1'b1) begin
         //Compute Second Stage Sums and Register
         SmoothDataInt0 <= IntSum0 + IntSum1;
         SmoothDataInt1 <= IntSum2 + IntSum3;
         SmoothDataInt2 <= IntSum4 + IntSum5;
      end
   end //SmoothDataIntn_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: SmoothDataInt_Blk
      if (nPhyRst == 1'b0)
         SmoothDataInt <= $signed({{DATAWIDTH_P13}{1'b0}});
      else if (SmoothEn == 1'b1)
         //Compute Final Sum and Register
         SmoothDataInt <= SmoothDataInt0 + SmoothDataInt1 + SmoothDataInt2;
   end //SmoothDataInt_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: DelayRS_Blk
      if (nPhyRst == 1'b0) begin
         SmoothCoeffRS_1t <= 2'b0;
         SmoothCoeffRS_2t <= 2'b0;
         SmoothCoeffRS_3t <= 2'b0;
         SmoothCoeffRS_4t <= 2'b0;
      end
      else begin
         SmoothCoeffRS_1t <= SmoothCoeffRS;
         SmoothCoeffRS_2t <= SmoothCoeffRS_1t;
         SmoothCoeffRS_3t <= SmoothCoeffRS_2t;
         SmoothCoeffRS_4t <= SmoothCoeffRS_3t;
      end
   end //DelayRS_Blk

// Step 4 - Shift according to SmoothCoeffRS and register
assign ShiftSmoothData = (SmoothCoeffRS_4t == 2'b00) ? $signed({SmoothDataInt[DATAWIDTH_P12],SmoothDataInt}) :                   // 2^0
                         (SmoothCoeffRS_4t == 2'b01) ? $signed({SmoothDataInt,1'b0}) :                                           // 2^1
                         (SmoothCoeffRS_4t == 2'b11) ? $signed({SmoothDataInt[DATAWIDTH_P12],SmoothDataInt[DATAWIDTH_P12:1]}) + 
                                                                            $signed({{DATAWIDTH_P13{1'b0}},SmoothDataInt[0]}) :  // 2^-1 + rounding
                                                  $signed({{2{SmoothDataInt[DATAWIDTH_P12]}},SmoothDataInt[DATAWIDTH_P12:2]}) + 
                                                                            $signed({{DATAWIDTH_P13{1'b0}},SmoothDataInt[1]});   // 2^-2 + rounding

always @ (posedge PhyClk or negedge nPhyRst)
   begin: DelayShift_Blk
      if (nPhyRst == 1'b0)
         ShiftSmoothData_1t <= {DATAWIDTH_P14{1'b0}};
      else
         ShiftSmoothData_1t <= ShiftSmoothData;
   end //DelayShift_Blk

// Step 5 - Rounding by 9 LSBs
Round #(
        .INPUT_WIDTH(DATAWIDTH+14),
        .OUTPUT_WIDTH(DATAWIDTH+5)
       )
        U_ROUNDFILT(
                    .InputData(ShiftSmoothData_1t),
                    .RoundData(RndSmoothData)
                   );

// Step 6 - Saturation to DATAWIDTH
SatSymSigned #(
               .INPUT_WIDTH(DATAWIDTH+5),
               .OUTPUT_WIDTH(DATAWIDTH)
              )
               U_SATFILT(
                        .InputData(RndSmoothData),
                        .SatSymData(SatSmoothData)
                        );

// Step 7 - Register Output Data
always @ (posedge PhyClk or negedge nPhyRst)
   begin: SmoothDataOut_Blk
      if (nPhyRst == 1'b0)
         SmoothDataOut <= $signed({{DATAWIDTH}{1'b0}});
      else if (SmoothEn == 1'b1)
         //Send out the saturated value
         SmoothDataOut <= SatSmoothData;
   end //SmoothDataOut_Blk

endmodule // Smoother

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
