///////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 40465 $
// $Date: 2019-11-20 14:37:14 +0100 (Wed, 20 Nov 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :Calculation of Unwrap theta based on ref and equalised 
//                   pilot.                                                      
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/CoarseStoCpe.v $
//
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

module  CoarseStoCpe (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock

            // Clear control
            input    wire                                  FDReset,

            //Index from Equaliser.
            input    wire       signed     [9:0]           SubCarrierIndex,

            // Reference Pilot.
            input    wire                                  RefPilot,

            //Equalised Pilot & Valid
            input    wire                                  EqPilotValid,
            input    wire                 [19:0]           EqPilotRe,
            input    wire                 [19:0]           EqPilotIm,

            //Previous Symbol STO & CPE Fine.
            input    wire                 [23:0]           STOfine,
            input    wire                 [21:0]           CPEfine,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //UnwrapTheta and Done signal
            output   reg signed           [16:0]           UnwrapTheta,
            output   reg                                   UnwrapThetaDone
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
//Reference theta Memory
reg     signed        [14:0]    RefTheta[5:0];
//Counter for Reference Theta
reg                   [2:0]     CountRef;
//Counter for Equalised Theta
reg                   [2:0]     CountEq;
//Intermediate Theta
reg     signed        [15:0]    Thetatmp;
reg     signed        [14:0]    ThetaReg;

//Delay of valid signal
reg                             DEqThetaDone;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
//Equalised Theta valid
wire                            EqThetaDone;

//Reference Theta valid
wire                            RefThetaDone;
wire    signed        [11:0]    EqTheta;

//Output of Round component
wire    signed        [3:0]     ThetaRound;

//Shifted theta
wire    signed        [15:0]    ShiftTheta;

//Output of Reference Phase calculation
wire                  [14:0]    ReferenceTheta;

integer                         i;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
//----------------------------------------------------------------------------

//Left shift by 12 bits
assign ShiftTheta = ThetaRound <<< 4'd12;

//----------------------------------------------------------------------------
//Hold Reference theta
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    for (i=0;i<6;i=i+1)
      RefTheta[i] <= $signed({15{1'b0}});
  end
  else if (RefThetaDone==1'b1) // valid signal so that RefTheta[0] is not reased when CountRef is reset
    RefTheta[CountRef] <= $signed(ReferenceTheta);
end

//----------------------------------------------------------------------------
//Counter for indexing Reference Theta array
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    CountRef <= 3'b0;
  else if (RefThetaDone)
    CountRef <=CountRef + 3'b1;
  else
    CountRef <= 3'b0;
end

//----------------------------------------------------------------------------
//Calculation of Unwrap factor
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    CountEq      <= 3'b0;
    Thetatmp     <= $signed({16{1'b0}});
    ThetaReg     <= $signed({15{1'b0}});
    DEqThetaDone <= 1'b0;
  end
  else if(EqThetaDone) begin
    CountEq      <= CountEq + 3'b1;
    //Equalised theta subtracted from Reference theta of particular subcarrier.
    Thetatmp     <= $signed(RefTheta[CountEq]) - $signed({{3{EqTheta[11]}},EqTheta});
    ThetaReg     <= {{3{EqTheta[11]}},EqTheta};
    DEqThetaDone <= EqThetaDone;
  end
  else begin
    CountEq      <= 3'b0;
    Thetatmp     <= 16'sd0;
    ThetaReg     <= $signed({15{1'b0}});
    DEqThetaDone <= 1'b0;
  end
end

//----------------------------------------------------------------------------
//Addition of Unwrap factor with equalised theta
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    UnwrapTheta     <= 17'b0;
    UnwrapThetaDone <= 1'b0;
  end
  else if (FDReset)begin
    UnwrapThetaDone <= 1'b0;
  end
  else begin
    UnwrapTheta     <= ShiftTheta + $signed({ThetaReg[14],ThetaReg});
    UnwrapThetaDone <= DEqThetaDone;
  end
end

//Instantiation of equalised angle calculation component.
EqPilotPhase U_EqPilotPhase (
                 //Inputs
                 .PhyClk(PhyClk),
                 .nPhyRst(nPhyRst),
                 .FDReset(FDReset),
                 .RefPilot(RefPilot),
                 .EqPilotValid(EqPilotValid),
                 .EqPilotRe(EqPilotRe),
                 .EqPilotIm(EqPilotIm),
                 //Outputs
                 .Theta(EqTheta),
                 .ThetaDone(EqThetaDone)
                 );

//Instantiation of reference phase calculation component.
RefPhase U_RefPhase (
                 //Inputs
                 .PhyClk(PhyClk),
                 .nPhyRst(nPhyRst),
                 .STOfine($signed(STOfine)),
                 .CPEfine($signed(CPEfine)),
                 .EqPilotValid(EqPilotValid),
                 .SubCarrierIndex(SubCarrierIndex),
                  //Outputs
                 .RefTheta(ReferenceTheta),
                 .RefThetaDone(RefThetaDone)
                 );

//Instantiation of Round component
Round #(
        .INPUT_WIDTH(16),
        .OUTPUT_WIDTH(4)
       )
        U_ROUND(
                 .InputData(Thetatmp),
                 .RoundData(ThetaRound)
                 );

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
