//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 40465 $
// $Date: 2019-11-20 14:37:14 +0100 (Wed, 20 Nov 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :Top file for  Coarse FD Offset Estimation                                                     
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/CoarseStoCpeTop.v $
//
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
module  CoarseStoCpeTop #(parameter NSTS_PARAM    = 2,
`ifdef RW_NX_DERIV_NESS_EN
                          parameter NESS_PARAM    = 0,
`endif
                          parameter HMEMDATAWIDTH = 13, // Data width of one channel coefficient
                          parameter HMEMWIDTH     = 416 // Width per SC
                         )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock

            // Data and Control
            //Rx Control
            input    wire                                  FDReset,
            // No. of Rx Ant for Cur Pkt
            input    wire                 [1:0]            CfgNrx,           

            //Next clock after WLSStart HMatrix should come.
            //HMatrix from Equaliser.
            input    wire          [HMEMWIDTH-1:0]         HMatrix,

            //Index from Equaliser.
            input    wire       signed     [9:0]           SubCarrierIndex,

            //Number of Spatial Stream:0-->1 SS,1-->2 SS
            input    wire                                  RxNsts,
            //ReceptionMode
            input    wire                  [3:0]           ReceptionMode,
            //Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)
            input   wire                  [2:0]            RUType,
            //Start calculating WLS Matrix
            //Start signal from FSM.
            input    wire                                  WLSStartP,
            //Bandwidth of Packet.
            input    wire                  [1:0]           Bandwidth,
            //Input to GenPilot for HT packet.
            input    wire                                  InitPilotShRegP,
            //Input to GenPilot for VHT packet.
            input    wire                                  LFSRShiftP,
            
            //Real & Imag part of Equalised pilot.
            input    wire       signed     [19:0]          EqPilotRe,
            input    wire       signed     [19:0]          EqPilotIm,
            input    wire                                  EqPilotValid,
            
            //STO & CPE fine for coarse calculation.
            //STO & CPE Value of last symbol
            input    wire       signed     [23:0]          STOFine,
            input    wire       signed     [21:0]          CPEFine,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //STO & CPE COarse output with valid signal.
            output wire signed     [17:0]                  STOCoarse,
            output wire signed     [17:0]                  CPECoarse,
            output wire                                    CoarseOutValid
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////

// Reception mode
localparam [3:0] MODE_20MHZ_L     = 4'd0,
                 MODE_40MHZ_LDUP  = 4'd1,
                 MODE_20MHZ_HT    = 4'd2,
                 MODE_40MHZ_HT    = 4'd3,
                 MODE_80MHZ_HT_8PILOTS = 4'd4, // not used
                 MODE_80MHZ_HT    = 4'd5,
                 MODE_20MHZ_HE_SU = 4'd6,
                 MODE_20MHZ_HE_MU = 4'd7,
                 MODE_40MHZ_HE_SU = 4'd8,
                 MODE_40MHZ_HE_MU = 4'd9;

// RU type
localparam [2:0] RU26  = 3'd0,
                 RU52  = 3'd1,
                 RU106 = 3'd2,
                 RU242 = 3'd3,
                 RU484 = 3'd4;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
//Determinant Output
reg       signed       [29:0]   DetMSRTReg;
reg                             FlagPilotGen;
reg                             FlagEn;

//Counters
reg                    [2:0]    count;
reg                    [2:0]    Encount;
reg                    [2:0]    countP;
reg                    [2:0]    countmul;
reg                    [2:0]    countadd;
reg                             FlagEqPilot;
reg                    [2:0]    Nsp;

//Input to GenPilot for shifting out pilots.
reg                             InitPilotShRegP_1t;
reg                             InitPilotShRegP_2t;
// Pilot Mode
reg                    [2:0]    GenPilotMode;

//M X N Output(P Matrix) registered.
reg       signed       [21:0]   P11reg[5:0];
reg       signed       [21:0]   P21reg[5:0];

//P Matrix multiplied by phi(resultant of equalised and reference pilots).
reg       signed       [40:0]   P1MulPhi;
reg       signed       [40:0]   P2MulPhi;
reg       signed       [40:0]   P1MulPhiSum;
reg       signed       [40:0]   P2MulPhiSum;
reg                             PMulPhiValid;
reg                             PMulPhiValid_1t;
reg                             PMulPhiValid_2t;
reg                             UnwrapThetaDone_1t;

//Registering Equalized Pilots & Valid signal.
reg       signed       [19:0]   EqPilotReReg;
reg       signed       [19:0]   EqPilotImReg;
reg                             EqPilotValidReg;
reg                             PilotEnReg;

//Input of SRT Divider.
reg       signed     [29:0]     DetMSRT;
reg       signed     [29:0]     DividendSto;
reg       signed     [29:0]     DividendCpe;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////

//P Matrix multiplied by phi(resultant of equalised and reference pilots).
wire      signed       [40:0]   P1MulPhiComb;
wire      signed       [40:0]   P2MulPhiComb;

//Reference Pilot
wire                            RefPilotSTS1;
wire                            RefPilotSTS2;

//Pilots from GenPilot Module
wire                   [5:0]    PilotSTS1;
wire                   [5:0]    PilotSTS2;
wire                   [5:0]    PilotRevSTS1;
wire                   [5:0]    PilotRevSTS2;

//Valid signal for pilot output for NSTS1 & 2.
wire                            PilotDataValid1;
wire                            PilotDataValid2;

//P Matrix with enable signal.
wire      signed       [21:0]   P11;
wire      signed       [21:0]   P21;
wire                            EnableOutPMatrix;

//Determinant M with valid signal.
wire      signed       [21:0]   DetM;
wire                            EnableOutMDet;

//Unwrap Theta and Valid
wire      signed       [16:0]   UnwrapTheta;
wire                            UnwrapThetaDone;

//Output of Round instantiation.
wire      signed       [29:0]   RndP1MulPhi;
wire      signed       [29:0]   RndP2MulPhi;
wire      signed       [28:0]   RndP1MulPhi_6pilots;
wire      signed       [28:0]   RndP2MulPhi_6pilots;

//Enable output of SRT Dividers.
wire                            DataEnOutSrt1;
wire                            DataEnOutSrt2;

//Input to GenPilot.
wire                            InitPilotScramP;

//Previous STO & CPE coarse value.
wire      signed       [17:0]   STOCoarsePrevSym;
wire      signed       [17:0]   CPECoarsePrevSym;
wire                            PilotEn;

wire      signed        [9:0]   SubCarrierIndexOut;

integer                         i;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//If BW is 40/80 then 6 pilot subcarriers else 4.
always @(posedge PhyClk or negedge nPhyRst)
begin:RegNsp
  if (nPhyRst == 1'b0)
    Nsp <= 3'd4;
  // For non-HE frames
  else if (ReceptionMode < MODE_20MHZ_HE_SU)
  begin
    if (Bandwidth == 2'd0)
      Nsp <= 3'd4;
    else
      Nsp <= 3'd6;
  end
  // For HE frames
  else if (RUType == RU26)
    Nsp <= 3'd2;
  else if ((RUType == RU52) || (RUType == RU106))
    Nsp <= 3'd4;
  else
    Nsp <= 3'd6;
end

assign STOCoarse = STOCoarsePrevSym;
assign CPECoarse = CPECoarsePrevSym;

//----------------------------------------------------------------------------
// Divisor input to SRT divider.
//If EnableOutMDet=1 take new value otherwise hold previous value
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:DivisorDetM
  if (nPhyRst == 1'b0)
    DetMSRT <= $signed({30{1'b0}});
  else if (FDReset)
    DetMSRT <= $signed({30{1'b0}});
  else if (EnableOutMDet)
    DetMSRT <= {{8{DetM[21]}},DetM};
  else
    DetMSRT <= DetMSRTReg;
end

//----------------------------------------------------------------------------
// Dividend input to SRT divider.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:RegDividend
  if (nPhyRst == 1'b0)begin
    DividendSto <= $signed({30{1'b0}});
    DividendCpe <= $signed({30{1'b0}});
  end
  else if (FDReset)begin
    DividendSto <= $signed({30{1'b0}});
    DividendCpe <= $signed({30{1'b0}});
  end
  else if (Nsp <= 3'd4)  begin
    DividendSto <= RndP1MulPhi;
    DividendCpe <= RndP2MulPhi;
  end
  else begin
    DividendSto <= {{1{RndP1MulPhi_6pilots[28]}},RndP1MulPhi_6pilots};
    DividendCpe <= {{1{RndP2MulPhi_6pilots[28]}},RndP2MulPhi_6pilots};
  end
end

//----------------------------------------------------------------------------
//Registering Determinant Output
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:RegDetM
  if (nPhyRst == 1'b0)
    DetMSRTReg <= $signed({30{1'b0}});
  else if (EnableOutMDet)
    DetMSRTReg <= DetMSRT;
end

//----------------------------------------------------------------------------
//Registering Pilot Enable signal for input to WLS Matrix.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:RegPilotEn
  if (nPhyRst == 1'b0)
    PilotEnReg <= 1'b0;
  else
    PilotEnReg <= PilotEn;
end


//InitPilotScramP signal to Pilot Generator only once.
assign InitPilotScramP = (FlagPilotGen)? 1'b0 : WLSStartP;

//Input to Pilot Generator if 40/80 Mode gen 6 pilots else 4 pilots.
always @(posedge PhyClk or negedge nPhyRst)
begin:RegPilotMode
  if (nPhyRst == 1'b0)
    GenPilotMode <= 3'd0;
  else if ( (ReceptionMode==MODE_20MHZ_HE_SU) || (RUType == RU242))
    GenPilotMode <= 3'd2;
  else if (RUType == RU484)
    GenPilotMode <= 3'd4;
  else if ((ReceptionMode==MODE_20MHZ_HE_MU || ReceptionMode == MODE_40MHZ_HE_MU) && (RUType == RU26))
    GenPilotMode <= 3'd3;
  else if ((Bandwidth != 2'd0) && (ReceptionMode != MODE_40MHZ_HE_MU) && (ReceptionMode != MODE_20MHZ_HE_MU)) begin
    if (Bandwidth == 2'd2)
      GenPilotMode <= 3'd2;
    else
      GenPilotMode <= 3'd1;
  end
  else
    GenPilotMode <= 3'd0;
end

//Extracting reference pilot from 4 bit Pilot Shift Register.
assign PilotRevSTS1   = {PilotSTS1[0],PilotSTS1[1],PilotSTS1[2],PilotSTS1[3],PilotSTS1[4],PilotSTS1[5]};
assign PilotRevSTS2   = {PilotSTS2[0],PilotSTS2[1],PilotSTS2[2],PilotSTS2[3],PilotSTS2[4],PilotSTS2[5]};
assign RefPilotSTS1   = PilotRevSTS1[count];
assign RefPilotSTS2   = PilotRevSTS2[count];
assign PilotEn        = PilotDataValid1 | PilotDataValid2;

//Output valid signal from this module.
assign CoarseOutValid = DataEnOutSrt1 | DataEnOutSrt2;

//----------------------------------------------------------------------------
//InitPilotShRegP should be 1 only if pilots for HT packets are generated.
//otherwise it should be deasserted.
//Mapped to pilot generator.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:EnablePilotShRegP
  if (nPhyRst == 1'b0) begin
    InitPilotShRegP_1t <= 1'b0;
    InitPilotShRegP_2t <= 1'b0;
  end
  else begin
    InitPilotShRegP_1t <= WLSStartP;
    InitPilotShRegP_2t <= InitPilotShRegP_1t;
  end
end

//----------------------------------------------------------------------------
//FlagPilotGen for controlling InitPilotScram to pilot Generator.
//Only given once at the start of packet.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:FlagPilotGenCtrl
  if (nPhyRst == 1'b0)
    FlagPilotGen <= 1'b0;
  else if (FDReset)
    FlagPilotGen <= 1'b0;
  else if (WLSStartP)
    FlagPilotGen <= 1'b1;
 end

//----------------------------------------------------------------------------
//Counting no.of pilot subcarriers to generate 6 pilots out of 4 pilots for
//Legacy 40 MHz packet.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:EnablePilotValid
  if (nPhyRst == 1'b0)begin
    FlagEn  <= 1'b0;
    Encount <= 3'd0;
  end
  else if(FDReset)begin
    FlagEn  <= 1'b0;
    Encount <= 3'd0;
  end
  else if ((PilotDataValid1 || PilotDataValid2) && FlagEn  ==1'b0 ) begin
    FlagEn  <= 1'b1;
    Encount <= Encount + 3'd1;
  end 
  else if (Encount == (Nsp-3'd1))  begin
    FlagEn  <= 1'b0;
    Encount <= 3'd0;
  end
  else if (FlagEn == 1'b1) begin
    FlagEn  <= 1'b1;
    Encount <= Encount + 3'd1;
  end
end

//----------------------------------------------------------------------------
//Selecting ref.pilot from 4-bit output of Pilot Generator.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:count4pilotsubcarrier
  if (nPhyRst == 1'b0)
    count <= 3'd0;
  else if (FDReset)
    count <= 3'd0;
  //Added to start counter again when Eq Pilot Valid comes.
  //Reference pilot is generated during WLS calculation and
  //pilot extraction should for reference pilot should be done
  //in same way as for WLS Matrix.
  else if (PilotEn || EqPilotValid || EqPilotValidReg) begin    
    if (!FlagEqPilot && EqPilotValid)
      count <= 3'd0;                                    
    else if (count == (Nsp-3'd1))
      count <= 3'd0;
    else
      count <= count+3'd1;
  end
end

//----------------------------------------------------------------------------
//FlagEqPilot for reseting count when EqPilotValid becomes 1.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:EqPilotFlag
  if (nPhyRst == 1'b0)
    FlagEqPilot <= 1'b0;
  else if (EqPilotValid || PilotEn)
    FlagEqPilot <= 1'b1;
  else
    FlagEqPilot <= 1'b0;
end

//----------------------------------------------------------------------------
//Registering Equalised Pilots to match latency with
//Reference pilots.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:EqPilotReg
  if (nPhyRst == 1'b0)begin
    EqPilotReReg    <= $signed({20{1'b0}});
    EqPilotImReg    <= $signed({20{1'b0}});
    EqPilotValidReg <= 1'b0;
  end
  else begin
    EqPilotReReg    <= EqPilotRe;
    EqPilotImReg    <= EqPilotIm;
    EqPilotValidReg <= EqPilotValid;
  end
end

//----------------------------------------------------------------------------
//Registering P Matrix to sync with theta output.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:RegPMatrix
  if (nPhyRst == 1'b0)begin
    for (i=0;i<6;i=i+1) begin
      P11reg[i] <= $signed({22{1'b0}});
      P21reg[i] <= $signed({22{1'b0}});
    end
  end
  else if (EnableOutPMatrix) begin
    P11reg[countP] <= P11;
    P21reg[countP] <= P21;
  end
end//RegPMatrix

//----------------------------------------------------------------------------
//Multiplication of P matrix with Unwrap Theta.
//Unwrap Theta comes sequentially and Pmatrix is read from an array. 
//----------------------------------------------------------------------------
assign P1MulPhiComb = P11reg[countmul] * $signed({{6{UnwrapTheta[16]}},UnwrapTheta});
assign P2MulPhiComb = P21reg[countmul] * $signed({{6{UnwrapTheta[16]}},UnwrapTheta});

// Cut combinatorial path between multiplier and adder
always @(posedge PhyClk or negedge nPhyRst)
begin:PMulPhi
  if (nPhyRst == 1'b0)begin
    P1MulPhi <= $signed({41{1'b0}});
    P2MulPhi <= $signed({41{1'b0}});
  end
  else begin
    P1MulPhi <= P1MulPhiComb;
    P2MulPhi <= P2MulPhiComb;
  end
end

always @(posedge PhyClk or negedge nPhyRst)
begin:PMulPhiSum
  if (nPhyRst == 1'b0)begin
    P1MulPhiSum <= $signed({41{1'b0}});
    P2MulPhiSum <= $signed({41{1'b0}});
  end
  else if (UnwrapThetaDone_1t)  begin
    if (countadd == 3'b0) begin
      P1MulPhiSum <= P1MulPhi;
      P2MulPhiSum <= P2MulPhi;
    end
    else begin
      P1MulPhiSum <= P1MulPhiSum + P1MulPhi;
      P2MulPhiSum <= P2MulPhiSum + P2MulPhi;
    end
  end
  else begin
    P1MulPhiSum <= $signed({41{1'b0}});
    P2MulPhiSum <= $signed({41{1'b0}});
  end
end

//----------------------------------------------------------------------------
//Count for Multiplied output.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:countmulp
  if (nPhyRst == 1'b0)begin
    countmul           <= 3'd0;
    countadd           <= 3'd0;
    PMulPhiValid       <= 1'b0;
    PMulPhiValid_1t    <= 1'b0;
    PMulPhiValid_2t    <= 1'b0;
    UnwrapThetaDone_1t <= 1'b0;
  end
  else begin
    PMulPhiValid_1t    <= PMulPhiValid;
    PMulPhiValid_2t    <= PMulPhiValid_1t;
    UnwrapThetaDone_1t <= UnwrapThetaDone;
    countadd           <= countmul;
    
    if (UnwrapThetaDone)  begin
      if (countmul == (Nsp-3'd1))begin
        countmul     <= 3'd0;
        PMulPhiValid <= 1'b1;
      end
      else begin
        countmul     <= countmul + 3'd1;
        PMulPhiValid <= 1'b0;
      end
    end
    else begin
      countmul     <= 3'd0;
      PMulPhiValid <= 1'b0;
    end
  end
end

//----------------------------------------------------------------------------
//Count for Registring P Matrix.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:countPMatrix
  if (nPhyRst == 1'b0)
    countP <= 3'd0;
  else if (EnableOutPMatrix) begin
    if (countP == (Nsp - 3'd1))
      countP <= 3'd0;
    else
      countP <= countP + 3'd1;
  end
  else
    countP <= 3'd0;
end

//Instantiation of Pilot Generator
GenPilot U_GenPilot (
                  // Clock & reset
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  // Inputs
                  .Nsts(RxNsts),
                  .Nsp(Nsp),
                  .GenPilotMode(GenPilotMode),
                  .PilotShiftOutP(InitPilotShRegP_2t),
                  .InitPilotScramP(InitPilotScramP),
                  .InitPilotShRegP(InitPilotShRegP),
                  .LFSRShiftP(LFSRShiftP),
                  // Outputs
                  .Pilot1(PilotSTS1),
                  .Pilot2(PilotSTS2),
                  .PilotDataValid1(PilotDataValid1),
                  .PilotDataValid2(PilotDataValid2)
                  );

//Instantiation of WLS Matrix Computation.
WLSCoefCompTop # (.NSTS_PARAM(NSTS_PARAM),
`ifdef RW_NX_DERIV_NESS_EN
                  .NESS_PARAM(NESS_PARAM),
`endif
                  .HMEMDATAWIDTH(HMEMDATAWIDTH),
                  .HMEMWIDTH(HMEMWIDTH))
U_WLSCoefCompTop ( 
                 //Inputs
                 .PhyClk(PhyClk),
                 .nPhyRst(nPhyRst),
                 // Data and Control
                 // Reference Pilots for STS = 1 & STS =2.
                 .RefPilotSTS1(RefPilotSTS1),
                 .RefPilotSTS2(RefPilotSTS2),
                 //Valid Signals for Pilot Bits
                 .PilotEn(PilotEn),
                 //HMatrix from Equaliser.
                 .HMatrix(HMatrix),
                 //Index from Equaliser.
                 .SubCarrierIndex(SubCarrierIndex),
                 //No. of Rx Ant for Cur Pkt
                 .CfgNrx(CfgNrx),
                 //Number of Spatial Stream.
                 .RxNsts(RxNsts),
                 .Nsp(Nsp),
                 //Outputs
                 .SubCarrierIndexOut(SubCarrierIndexOut),
                 .P11(P11),
                 .P21(P21),
                 .EnableOutPMatrix(EnableOutPMatrix),
                 .DetM(DetM),
                 .EnableOutMDet(EnableOutMDet)
                 );

//Instantiation of Unwrapped angle calculation.
CoarseStoCpe U_CoarseStoCpe(
                 //Inputs
                 .PhyClk(PhyClk),
                 .nPhyRst(nPhyRst),
                 // Data and Control
                 .FDReset(FDReset),
                 // Reference Pilot.
                 .RefPilot(RefPilotSTS1),
                 .EqPilotValid(EqPilotValidReg),
                 .EqPilotRe(EqPilotReReg[19:0]),
                 .EqPilotIm(EqPilotImReg[19:0]),
                 .STOfine(STOFine),
                 .CPEfine(CPEFine),
                 // SubCarrier Index
                 .SubCarrierIndex(SubCarrierIndexOut),
                 //Outputs
                 .UnwrapTheta(UnwrapTheta[16:0]),
                 .UnwrapThetaDone(UnwrapThetaDone)
                 );

//Instantiation of Round component
// Round when up to 4 pilots processed
Round #(
            .INPUT_WIDTH(32),
            .OUTPUT_WIDTH(30)
           )
            UP1Mul_ROUND(
                        .InputData($signed(P1MulPhiSum[40:9])),
                        .RoundData(RndP1MulPhi)
                       );

// Round when more than 4 pilots processed
Round #(
            .INPUT_WIDTH(32),
            .OUTPUT_WIDTH(29)
           )
            UP2Mul_ROUND(
                        .InputData($signed(P1MulPhiSum[40:9])),
                        .RoundData(RndP1MulPhi_6pilots)
                       );

//Instantiation of Round component
// Round when up to 4 pilots processed
Round #(
            .INPUT_WIDTH(32),
            .OUTPUT_WIDTH(30)
           )
            UP3Mul_ROUND(
                        .InputData($signed(P2MulPhiSum[40:9])),
                        .RoundData(RndP2MulPhi)
                       );

// Round when more than 4 pilots processed
Round #(
            .INPUT_WIDTH(32),
            .OUTPUT_WIDTH(29)
           )
            UP4Mul_ROUND(
                        .InputData($signed(P2MulPhiSum[40:9])),
                        .RoundData(RndP2MulPhi_6pilots)
                       );


//Instantiation of SRT Divider for P11.
SrtDivider #(.NBITINPUT(30),
             .NBITQUOTIENT(18),
             .NINTBITQUOTIENT(2),
             .NBITQUOTIENTSRT(20),
             .NBITPREMSRT(20),
             .SIZEBITSHIFTSRTQUOTIENT(10)) //TBD
SrtDivider_P1(
             .PhyClk(PhyClk),
             .nPhyRst(nPhyRst),
             .Clear(FDReset),
             .DataEnIn(PMulPhiValid_2t),
             .DataEnOut(DataEnOutSrt1),
             .Dividend(DividendSto),
             .Divisor(DetMSRT),
             .Quotient(STOCoarsePrevSym)
             );

//Instantiation of SRT Divider for P21.
SrtDivider #(.NBITINPUT(30),
             .NBITQUOTIENT(18),
             .NINTBITQUOTIENT(2),
             .NBITQUOTIENTSRT(20),
             .NBITPREMSRT(20),
             .SIZEBITSHIFTSRTQUOTIENT(10)) //TBD
SrtDivider_P2(
             .PhyClk(PhyClk),
             .nPhyRst(nPhyRst),
             .Clear(FDReset),
             .DataEnIn(PMulPhiValid_2t),
             .DataEnOut(DataEnOutSrt2),
             .Dividend(DividendCpe),
             .Divisor(DetMSRT),
             .Quotient(CPECoarsePrevSym)
             );

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
