//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 34754 $
// $Date: 2018-08-03 13:41:25 +0200 (Fri, 03 Aug 2018) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : Cordic Rotator Module
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       : cordic_rot module from RW_WLAN RefIP
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/CordicRotFD.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module CordicRotFD #(parameter NPIPE       =  7, //Number of Pipeline stages
                     parameter NCOMBSTG    =  2, //Number of Combinatorial stages
                     parameter SCALING     =  1, //Scaling to [-4,4] range
                     parameter DATAWIDTH   = 13, //Width of Input Data
                     parameter ANGLEWIDTH  = 21  //Widht of Input Angle
                    )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, // Active LOW Reset
            input    wire                                  PhyClk,  // PHY Clock

            //Control Signals
            input    wire                                  Enable,  //Block Enable
            input    wire              [ANGLEWIDTH-1:0]    AngleIn, //Angle for Rotation

            //Data
            input    wire              [DATAWIDTH-1:0]     ReDataIn0, //Real Comp of Data 0
            input    wire              [DATAWIDTH-1:0]     ImDataIn0, //Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            input    wire              [DATAWIDTH-1:0]     ReDataIn1, //Real Comp of Data 1
            input    wire              [DATAWIDTH-1:0]     ImDataIn1, //Imag Comp of Data 1
`endif

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Data
            output   wire      [DATAWIDTH+1:0]             ReDataOut0, //Real Comp of Data 0
            output   wire      [DATAWIDTH+1:0]             ImDataOut0, //Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            output   wire      [DATAWIDTH+1:0]             ReDataOut1, //Real Comp of Data 1
            output   wire      [DATAWIDTH+1:0]             ImDataOut1, //Imag Comp of Data 1
`endif

            //Control
            output   wire                                  DataOutValid //Qualifies Output
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam [31:0] PI                = 32'b01100100100001111110110101010001;
localparam [30:0] PI_SCALED         = {31{1'b0}};
localparam        NROT              = (NCOMBSTG * NPIPE) + 1;


//localparam    [NPIPE+1:0]      NPIPE_P2_PARAM_0   =  'b0; 


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
`ifdef RW_NX_DERIV_PATH1
wire             [DATAWIDTH+1:0]     ImDataOut1Inter; //Imag Comp of Data 1
wire             [DATAWIDTH+1:0]     ReDataOut1Inter; //Real Comp of Data 1
wire             [DATAWIDTH+1:0]     X1Int[NPIPE:0];
wire             [DATAWIDTH+1:0]     Y1Int[NPIPE:0];
`endif
wire             [DATAWIDTH+1:0]     ImDataOut0Inter; //Imag Comp of Data 0
wire             [DATAWIDTH+1:0]     ReDataOut0Inter; //Real Comp of Data 0
wire             [DATAWIDTH+1:0]     X0Int[NPIPE:0];
wire             [DATAWIDTH+1:0]     Y0Int[NPIPE:0];
wire             [ANGLEWIDTH+1:0]    AngleInExt;
wire             [ANGLEWIDTH+1:0]    ZIntExt;
wire             [ANGLEWIDTH-1:0]    ZInt[NROT-1:0];
wire             [ANGLEWIDTH-1:0]    AngleInNeg;
wire             [NROT-2:0]          ArcTanArrMem[31:0];
wire             [31:0]              ArcTanArrRef[31:0];
wire             [31:0]              PiVal;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                [ANGLEWIDTH-1:0]  AngleInt;
reg                [NPIPE+1:0]       AngleFlag;
reg                [DATAWIDTH-1:0]   ReDataInt0;
reg                [DATAWIDTH-1:0]   ImDataInt0;
reg                [DATAWIDTH+1:0]   SReDataOut0;
reg                [DATAWIDTH+1:0]   SImDataOut0;
`ifdef RW_NX_DERIV_PATH1
reg                [DATAWIDTH-1:0]   ReDataInt1;
reg                [DATAWIDTH-1:0]   ImDataInt1;
reg                [DATAWIDTH+1:0]   SReDataOut1;
reg                [DATAWIDTH+1:0]   SImDataOut1;
`endif

//Genvars
genvar n,j,k;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Assign value of PI depending on the parameter
assign PiVal = (SCALING == 0) ? PI : {1'b1, PI_SCALED};

assign ReDataOut0Inter = SReDataOut0;
assign ImDataOut0Inter = SImDataOut0;
`ifdef RW_NX_DERIV_PATH1
assign ReDataOut1Inter = SReDataOut1;
assign ImDataOut1Inter = SImDataOut1;
`endif


//Register Inputs
always @ (posedge PhyClk or negedge nPhyRst)
   begin: RegIp_Blk
      if (nPhyRst == 1'b0) begin
         ReDataInt0 <= {{DATAWIDTH}{1'b0}};
         ImDataInt0 <= {{DATAWIDTH}{1'b0}};
`ifdef RW_NX_DERIV_PATH1
         ReDataInt1 <= {{DATAWIDTH}{1'b0}};
         ImDataInt1 <= {{DATAWIDTH}{1'b0}};
`endif
      end
      else if (Enable == 1'b1) begin
         ReDataInt0 <= ReDataIn0;
         ImDataInt0 <= ImDataIn0;
`ifdef RW_NX_DERIV_PATH1
         ReDataInt1 <= ReDataIn1;
         ImDataInt1 <= ImDataIn1;
`endif
      end
   end //RegIp_Blk

//Sign extend inputs
   assign X0Int[0] = {{(2){ReDataInt0[DATAWIDTH-1]}},
                           ReDataInt0[DATAWIDTH-1:0]};
   assign Y0Int[0] = {{(2){ImDataInt0[DATAWIDTH-1]}},
                           ImDataInt0[DATAWIDTH-1:0]};

`ifdef RW_NX_DERIV_PATH1
      assign X1Int[0] = {{(2){ReDataInt1[DATAWIDTH-1]}},
                              ReDataInt1[DATAWIDTH-1:0]};
      assign Y1Int[0] = {{(2){ImDataInt1[DATAWIDTH-1]}},
                              ImDataInt1[DATAWIDTH-1:0]};
`endif


//AngleIn must be within [-PI/2,+PI/2] interval. If AngleIn is in
//[PI/2;3PI/2] internal, then AngleIn - PI will be used, and the
//result will be inverted.
assign AngleInNeg = ~(AngleIn) + {{(ANGLEWIDTH-1){1'b0}},1'b1};

always @ (*)
   begin: AngleFlag_Blk
      if ((AngleIn > ({1'b0, PiVal[31:31 - ANGLEWIDTH + 2]})
             && !(AngleIn[ANGLEWIDTH - 1])) ||
          (AngleInNeg > ({1'b0, PiVal[31:31 - ANGLEWIDTH + 2]})
             && (AngleIn[ANGLEWIDTH - 1])))
         AngleFlag[0] = 1'b0;
      else
         AngleFlag[0] = 1'b1;
   end //AngleFlag_Blk

always @ (posedge PhyClk or negedge nPhyRst)
   begin: AngleInt_Blk
      if (nPhyRst == 1'b0)
         AngleInt <= {(ANGLEWIDTH){1'b0}};
      else if (Enable == 1'b1) begin
         if (AngleFlag[0] == 1'b1)
            AngleInt <= AngleIn;
         else if (AngleIn[ANGLEWIDTH - 1] == 1'b0)
            AngleInt <= (AngleIn) - PiVal[31:(31 - ANGLEWIDTH + 1)];
         else
            AngleInt <= (AngleIn) + PiVal[31:(31 - ANGLEWIDTH + 1)];
      end
   end //AngleInt_Blk

assign ZInt[0]    = AngleInt;
assign AngleInExt = {{(2){AngleIn[ANGLEWIDTH-1]}}, AngleIn};
assign ZIntExt    = AngleInExt + {2'b00, PiVal[31:(31-ANGLEWIDTH+1)]};

//Rearranging input angles for Cordic combinatorial stages
generate

   for ( j = 0; j < 32; j = j+1) begin:ArcArrayRefrow
      for(k = 0; k < NROT-1; k = k+1) begin:ArcArrayRefCol
         assign  ArcTanArrMem[j][k] = ArcTanArrRef[k][j];
      end
   end

endgenerate//ArcArrayMem1

//Generating Cordic combinatorial stages
generate 
   for(n = 0; n < NPIPE; n = n+1) begin:CordicComb
   CordicCombStageFD #(
                     //Parameters
                     .CCDATAWIDTH(DATAWIDTH+2),
                     .ANGLEWIDTH(ANGLEWIDTH),
                     .STARTSTAGE(n*NCOMBSTG),
                     .NCOMBSTG(NCOMBSTG)
                    ) U_CordicCombStageFD (
                              //Inputs
                             .PhyClk(PhyClk),
                             .nPhyRst(nPhyRst),
                             .Enable(Enable),
                             .AngleIn(ZInt[n]),
                             .ReDataIn0(X0Int[n]),
                             .ImDataIn0(Y0Int[n]),
`ifdef RW_NX_DERIV_PATH1
                             .ReDataIn1(X1Int[n]),
                             .ImDataIn1(Y1Int[n]),
`endif
                             .ArcTanArrRefBit31(ArcTanArrMem[31][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit30(ArcTanArrMem[30][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit29(ArcTanArrMem[29][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit28(ArcTanArrMem[28][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit27(ArcTanArrMem[27][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit26(ArcTanArrMem[26][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit25(ArcTanArrMem[25][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit24(ArcTanArrMem[24][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit23(ArcTanArrMem[23][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit22(ArcTanArrMem[22][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit21(ArcTanArrMem[21][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit20(ArcTanArrMem[20][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit19(ArcTanArrMem[19][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit18(ArcTanArrMem[18][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit17(ArcTanArrMem[17][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit16(ArcTanArrMem[16][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit15(ArcTanArrMem[15][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit14(ArcTanArrMem[14][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit13(ArcTanArrMem[13][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit12(ArcTanArrMem[12][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit11(ArcTanArrMem[11][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit10(ArcTanArrMem[10][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit9 (ArcTanArrMem[09][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]), 
                             .ArcTanArrRefBit8 (ArcTanArrMem[08][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]), 
                             .ArcTanArrRefBit7 (ArcTanArrMem[07][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit6 (ArcTanArrMem[06][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit5 (ArcTanArrMem[05][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit4 (ArcTanArrMem[04][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit3 (ArcTanArrMem[03][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit2 (ArcTanArrMem[02][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit1 (ArcTanArrMem[01][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),
                             .ArcTanArrRefBit0 (ArcTanArrMem[00][((n + 1)
                                                * NCOMBSTG)-1:n * NCOMBSTG]),

                             //Outputs
                             .ReDataOut0(X0Int[n+1]),
                             .ImDataOut0(Y0Int[n+1]),
`ifdef RW_NX_DERIV_PATH1
                             .ReDataOut1(X1Int[n+1]),
                             .ImDataOut1(Y1Int[n+1]),
`endif
                             .AngleOut(ZInt[n+1])
                            );
   end //for
endgenerate

always @(posedge PhyClk or negedge nPhyRst)
   begin: AngleFlag1_Blk
      if (nPhyRst == 1'b0)
         AngleFlag[NPIPE+1:1] <= {{(NPIPE+1)}{1'b0}};
      else if (Enable == 1'b1)
         AngleFlag[NPIPE+1:1] <= AngleFlag[NPIPE:0];
   end //AngleFlag1_Blk

//Outputs
always @(posedge PhyClk or negedge nPhyRst)
   begin: Output_Blk
      if (nPhyRst == 1'b0) begin
         SReDataOut0 <= {{(DATAWIDTH+2)}{1'b0}};
         SImDataOut0 <= {{(DATAWIDTH+2)}{1'b0}};
`ifdef RW_NX_DERIV_PATH1
         SReDataOut1 <= {{(DATAWIDTH+2)}{1'b0}};
         SImDataOut1 <= {{(DATAWIDTH+2)}{1'b0}};
`endif
      end
      else if (Enable == 1'b1) begin
         if (AngleFlag[NPIPE+1] == 1'b1) begin
            SReDataOut0 <= X0Int[NPIPE];
            SImDataOut0 <= Y0Int[NPIPE];
`ifdef RW_NX_DERIV_PATH1
            SReDataOut1 <= X1Int[NPIPE];
            SImDataOut1 <= Y1Int[NPIPE];
`endif
         end //AngleFlag[NPIPE + 1] == 1'b1
         else begin
            SReDataOut0 <= ~(X0Int[NPIPE]) + {{{(DATAWIDTH+1)}{1'b0}},1'b1};
            SImDataOut0 <= ~(Y0Int[NPIPE]) + {{{(DATAWIDTH+1)}{1'b0}},1'b1};
`ifdef RW_NX_DERIV_PATH1
            SReDataOut1 <= ~(X1Int[NPIPE]) + {{{(DATAWIDTH+1)}{1'b0}},1'b1};
            SImDataOut1 <= ~(Y1Int[NPIPE]) + {{{(DATAWIDTH+1)}{1'b0}},1'b1};
`endif
         end //AngleFlag[NPIPE + 1] == 1'b0
      end //Enable == 1'b1
   end //Output_Blk


generate
   if(SCALING ==0) begin
      //No Scaling
      //Reference values for ArcTan.
      assign ArcTanArrRef[0]  = 32'b11001001000011111101101010100010;
      assign ArcTanArrRef[1]  = 32'b01110110101100011001110000010110;
      assign ArcTanArrRef[2]  = 32'b00111110101101101110101111110010;
      assign ArcTanArrRef[3]  = 32'b00011111110101011011101010011011;
      assign ArcTanArrRef[4]  = 32'b00001111111110101010110111011100;
      assign ArcTanArrRef[5]  = 32'b00000111111111110101010101101111;
      assign ArcTanArrRef[6]  = 32'b00000011111111111110101010101011;
      assign ArcTanArrRef[7]  = 32'b00000001111111111111110101010101;
      assign ArcTanArrRef[8]  = 32'b00000000111111111111111110101011;
      assign ArcTanArrRef[9]  = 32'b00000000011111111111111111110101;
      assign ArcTanArrRef[10] = 32'b00000000001111111111111111111111;
      assign ArcTanArrRef[11] = 32'b00000000001000000000000000000000;
      assign ArcTanArrRef[12] = 32'b00000000000100000000000000000000;
      assign ArcTanArrRef[13] = 32'b00000000000010000000000000000000;
      assign ArcTanArrRef[14] = 32'b00000000000001000000000000000000;
      assign ArcTanArrRef[15] = 32'b00000000000000100000000000000000;
      assign ArcTanArrRef[16] = 32'b00000000000000010000000000000000;
      assign ArcTanArrRef[17] = 32'b00000000000000001000000000000000;
      assign ArcTanArrRef[18] = 32'b00000000000000000100000000000000;
      assign ArcTanArrRef[19] = 32'b00000000000000000010000000000000;
      assign ArcTanArrRef[20] = 32'b00000000000000000001000000000000;
      assign ArcTanArrRef[21] = 32'b00000000000000000000100000000000;
      assign ArcTanArrRef[22] = 32'b00000000000000000000010000000000;
      assign ArcTanArrRef[23] = 32'b00000000000000000000001000000000;
      assign ArcTanArrRef[24] = 32'b00000000000000000000000100000000;
      assign ArcTanArrRef[25] = 32'b00000000000000000000000010000000;
      assign ArcTanArrRef[26] = 32'b00000000000000000000000001000000;
      assign ArcTanArrRef[27] = 32'b00000000000000000000000000100000;
      assign ArcTanArrRef[28] = 32'b00000000000000000000000000010000;
      assign ArcTanArrRef[29] = 32'b00000000000000000000000000001000;
      assign ArcTanArrRef[30] = 32'b00000000000000000000000000000100;
      assign ArcTanArrRef[31] = 32'b00000000000000000000000000000010;
   end
   else begin
      //Scaling = Pi/4 => 111111111111111
      //Reference values for ArcTan.
      assign ArcTanArrRef[0]  = 32'b11111111111111111111111111111111;
      assign ArcTanArrRef[1]  = 32'b10010111001000000010100011101101;
      assign ArcTanArrRef[2]  = 32'b01001111110110011100001011011011;
      assign ArcTanArrRef[3]  = 32'b00101000100010001000111010100001;
      assign ArcTanArrRef[4]  = 32'b00010100010110000110101000011000;
      assign ArcTanArrRef[5]  = 32'b00001010001011101011111100001011;
      assign ArcTanArrRef[6]  = 32'b00000101000101111011000011110011;
      assign ArcTanArrRef[7]  = 32'b00000010100010111110001010101001;
      assign ArcTanArrRef[8]  = 32'b00000001010001011111001010011010;
      assign ArcTanArrRef[9]  = 32'b00000000101000101111100101110110;
      assign ArcTanArrRef[10] = 32'b00000000010100010111110011000000;
      assign ArcTanArrRef[11] = 32'b00000000001010001011111001100001;
      assign ArcTanArrRef[12] = 32'b00000000000101000101111100110000;
      assign ArcTanArrRef[13] = 32'b00000000000010100010111110011000;
      assign ArcTanArrRef[14] = 32'b00000000000001010001011111001100;
      assign ArcTanArrRef[15] = 32'b00000000000000101000101111100110;
      assign ArcTanArrRef[16] = 32'b00000000000000010100010111110011;
      assign ArcTanArrRef[17] = 32'b00000000000000001010001011111010;
      assign ArcTanArrRef[18] = 32'b00000000000000000101000101111101;
      assign ArcTanArrRef[19] = 32'b00000000000000000010100010111110;
      assign ArcTanArrRef[20] = 32'b00000000000000000001010001011111;
      assign ArcTanArrRef[21] = 32'b00000000000000000000101000110000;
      assign ArcTanArrRef[22] = 32'b00000000000000000000010100011000;
      assign ArcTanArrRef[23] = 32'b00000000000000000000001010001100;
      assign ArcTanArrRef[24] = 32'b00000000000000000000000101000110;
      assign ArcTanArrRef[25] = 32'b00000000000000000000000010100011;
      assign ArcTanArrRef[26] = 32'b00000000000000000000000001010001;
      assign ArcTanArrRef[27] = 32'b00000000000000000000000000101001;
      assign ArcTanArrRef[28] = 32'b00000000000000000000000000010100;
      assign ArcTanArrRef[29] = 32'b00000000000000000000000000001010;
      assign ArcTanArrRef[30] = 32'b00000000000000000000000000000101;
      assign ArcTanArrRef[31] = 32'b00000000000000000000000000000011;
   end

endgenerate

//Generate DataOutValid
//The total latency of this module is NPIPE + 2
DelayLine # (
             .DATAWIDTH(1),     //Width of Enable
             .LATENCY(NPIPE+2)  //Latency
             )
              U_DelayLineCR (
                            .PhyClk(PhyClk),
                            .nPhyRst(nPhyRst),
                            .Bypass(~Enable),
                            .DataIn(Enable),
                            .DataOut(DataOutValid)
                            );

assign ReDataOut0 = ReDataOut0Inter;
assign ImDataOut0 = ImDataOut0Inter;
`ifdef RW_NX_DERIV_PATH1
assign ReDataOut1 = ReDataOut1Inter;
assign ImDataOut1 = ImDataOut1Inter;
`endif

endmodule //CordicRotFD

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
