//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 34754 $
// $Date: 2018-08-03 13:41:25 +0200 (Fri, 03 Aug 2018) $
// ---------------------------------------------------------------------------
// Dependencies     : 
// Description      : Top Level File for Cordic Rotator
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       : 
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/CordicRotFDTop.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module CordicRotFDTop #(parameter NPIPE      =  2, //Number of Pipeline stages
                        parameter NCOMBSTG   =  7, //Number of Combinatorial stages
                        parameter SCALING    =  1, //Scaling to [-4,4] range
                        parameter DATAWIDTH  = 13, //Width of Input Data
                        parameter ANGLEWIDTH = 25  //Widht of Input Angle
                       )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock

            //Control Signals
            input    wire                                  EnableIn,  //Block Enable
            input    wire              [ANGLEWIDTH-1:0]    AngleIn, //Angle for Rotation

            //Data
            input    wire              [DATAWIDTH-1:0]     ReDataIn0, //Real Comp of Data 0
            input    wire              [DATAWIDTH-1:0]     ImDataIn0, //Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            input    wire              [DATAWIDTH-1:0]     ReDataIn1, //Real Comp of Data 1
            input    wire              [DATAWIDTH-1:0]     ImDataIn1, //Imag Comp of Data 1
`endif

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Data
            output   wire      [DATAWIDTH-1:0]             ReDataOut0, //Real Comp of Data 0
            output   wire      [DATAWIDTH-1:0]             ImDataOut0, //Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
            output   wire      [DATAWIDTH-1:0]             ReDataOut1, //Real Comp of Data 1
            output   wire      [DATAWIDTH-1:0]             ImDataOut1, //Imag Comp of Data 1
`endif
            //Control
            output   wire                                  DataOutValid //Qualifies Output
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire             [DATAWIDTH-1:0]     ImDataOut0Inter; //Imag Comp of Data 0
wire             [DATAWIDTH-1:0]     ReDataOut0Inter; //Real Comp of Data 0

wire    signed   [DATAWIDTH+1:0]     ReDataOut0Int;
wire    signed   [DATAWIDTH+1:0]     ImDataOut0Int;
wire    signed   [DATAWIDTH+1:0]     ReDataOut0Norm;
wire    signed   [DATAWIDTH+1:0]     ImDataOut0Norm;
wire    signed   [DATAWIDTH-1:0]     ReDataOut0Sat;
wire    signed   [DATAWIDTH-1:0]     ImDataOut0Sat;
wire                                 DataOutValidInt;
wire                                 Enable;

reg              [10:0]              EnableReg;
reg              [DATAWIDTH-1:0]     SReDataOut0;
reg              [DATAWIDTH-1:0]     SImDataOut0;

`ifdef RW_NX_DERIV_PATH1
wire             [DATAWIDTH-1:0]     ImDataOut1Inter; //Imag Comp of Data 1
wire             [DATAWIDTH-1:0]     ReDataOut1Inter; //Real Comp of Data 1
wire    signed   [DATAWIDTH+1:0]     ReDataOut1Int;
wire    signed   [DATAWIDTH+1:0]     ImDataOut1Int;
wire    signed   [DATAWIDTH+1:0]     ReDataOut1Norm;
wire    signed   [DATAWIDTH+1:0]     ImDataOut1Norm;
wire    signed   [DATAWIDTH-1:0]     ReDataOut1Sat;
wire    signed   [DATAWIDTH-1:0]     ImDataOut1Sat;

reg              [DATAWIDTH-1:0]     SReDataOut1;
reg              [DATAWIDTH-1:0]     SImDataOut1;
`endif

integer i;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

always @ (posedge PhyClk or negedge nPhyRst)
   begin
      if (nPhyRst == 1'b0) begin
        for (i=0;i<=10;i=i+1)begin
          EnableReg <=11'b0;
        end
      end
      else begin
        if(EnableIn == 1'b1) begin
          for (i=0;i<=10;i=i+1)begin
            EnableReg[i] <=1'b1;
          end
        end
        else begin
          EnableReg[10] <=1'b0;
          EnableReg[9:0] <=EnableReg[10:1];
        end
      end
  end

assign  Enable = EnableReg[0] | EnableIn;


//This block instantiates the CordicRot module. It also performs
//normalization on the outputs. It then saturates the output to DATAWIDTH. 
//This operation adds one clock latency to the Cordic operation. 
//The DataOutValid is hence delayed by a clock.

//Instantiate the CordicRot module
CordicRotFD #(
            //Parameters
           .DATAWIDTH(DATAWIDTH),
           .ANGLEWIDTH(ANGLEWIDTH),
           .NCOMBSTG(NCOMBSTG),
           .NPIPE(NPIPE),
           .SCALING(SCALING)
           )U_CR (
                  //Inputs
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .Enable(Enable),
                  .AngleIn(AngleIn),
                  .ReDataIn0(ReDataIn0),
                  .ImDataIn0(ImDataIn0),
`ifdef RW_NX_DERIV_PATH1
                  .ReDataIn1(ReDataIn1),
                  .ImDataIn1(ImDataIn1),
`endif

                  //Outputs
                  .ReDataOut0(ReDataOut0Int[DATAWIDTH+1:0]),
                  .ImDataOut0(ImDataOut0Int[DATAWIDTH+1:0]),
`ifdef RW_NX_DERIV_PATH1
                  .ReDataOut1(ReDataOut1Int[DATAWIDTH+1:0]),
                  .ImDataOut1(ImDataOut1Int[DATAWIDTH+1:0]),
`endif
                  .DataOutValid(DataOutValidInt)
                 );

//Normalize the outputs
assign ReDataOut0Norm = $signed(Normalize(ReDataOut0Int));
assign ImDataOut0Norm = $signed(Normalize(ImDataOut0Int));
`ifdef RW_NX_DERIV_PATH1
assign ReDataOut1Norm = $signed(Normalize(ReDataOut1Int));
assign ImDataOut1Norm = $signed(Normalize(ImDataOut1Int));
`endif

//Saturate the Normalized Outputs
   SatSymSigned #(
                  .INPUT_WIDTH(DATAWIDTH+2),
                  .OUTPUT_WIDTH(DATAWIDTH)
                 )
                  U_SATCRT0(
                           .InputData(ReDataOut0Norm),
                           .SatSymData(ReDataOut0Sat)
                           );

   SatSymSigned #(
                  .INPUT_WIDTH(DATAWIDTH+2),
                  .OUTPUT_WIDTH(DATAWIDTH)
                 )
                  U_SATCRT1(
                           .InputData(ImDataOut0Norm),
                           .SatSymData(ImDataOut0Sat)
                           );

`ifdef RW_NX_DERIV_PATH1
      SatSymSigned #(
                     .INPUT_WIDTH(DATAWIDTH+2),
                     .OUTPUT_WIDTH(DATAWIDTH)
                    )
                     U_SATCRT2(
                              .InputData(ReDataOut1Norm),
                              .SatSymData(ReDataOut1Sat)
                              );

      SatSymSigned #(
                     .INPUT_WIDTH(DATAWIDTH+2),
                     .OUTPUT_WIDTH(DATAWIDTH)
                    )
                     U_SATCRT3(
                              .InputData(ImDataOut1Norm),
                              .SatSymData(ImDataOut1Sat)
                              );
`endif

//Register the final outputs
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Op0Reg_Blk
      if (nPhyRst == 1'b0) begin
         SReDataOut0 <= {{DATAWIDTH}{1'b0}};
         SImDataOut0 <= {{DATAWIDTH}{1'b0}};
      end
      else if (Enable == 1'b1) begin
         SReDataOut0 <= ReDataOut0Sat[DATAWIDTH-1:0];
         SImDataOut0 <= ImDataOut0Sat[DATAWIDTH-1:0];
      end
   end //Op0Reg_Blk

assign ReDataOut0Inter = SReDataOut0;
assign ImDataOut0Inter = SImDataOut0;

`ifdef RW_NX_DERIV_PATH1
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Op1Reg_Blk
      if (nPhyRst == 1'b0) begin
         SReDataOut1 <= {{DATAWIDTH}{1'b0}};
         SImDataOut1 <= {{DATAWIDTH}{1'b0}};
      end
      else if (Enable == 1'b1) begin
         SReDataOut1 <= ReDataOut1Sat[DATAWIDTH-1:0];
         SImDataOut1 <= ImDataOut1Sat[DATAWIDTH-1:0];
      end
   end //Op1Reg_Blk
 
assign ReDataOut1Inter = SReDataOut1;
assign ImDataOut1Inter = SImDataOut1;
`endif

//Function to generate the Normalized Value of the output
//Multiply by normalization factor 2487/4096

function           [DATAWIDTH+1:0]                Normalize;

//Inputs
input       signed    [DATAWIDTH+1:0]                DataIn;

//Internal Registers
reg         signed    [DATAWIDTH+13:0]               Normal;

   begin
      //Multiply by 2487
      //2487 = 2048 + 256 + 128 + 32 + 16 + 4 + 2 + 1
      Normal =   (DataIn <<< 'sd11) + (DataIn <<< 'sd8) + (DataIn <<< 'sd7)
               + (DataIn <<< 'sd5)  + (DataIn <<< 'sd4) + (DataIn <<< 'sd2)
               + (DataIn <<< 'sd1)  + (DataIn);

      // Divide by 4096
      Normalize = Normal[DATAWIDTH+13:12];

   end

endfunction  //Normalize;

//Delay DataValidOut by a clock
DelayLine # (
             .DATAWIDTH(1),     //Width of Enable
             .LATENCY(1)        //Latency
             )
              U_DelayLineCRT (
                             .PhyClk(PhyClk),
                             .nPhyRst(nPhyRst),
                             .Bypass(~Enable),
                             .DataIn(DataOutValidInt),
                             .DataOut(DataOutValid)
                             );

assign ReDataOut0 = ReDataOut0Inter;
assign ImDataOut0 = ImDataOut0Inter;
`ifdef RW_NX_DERIV_PATH1
assign ReDataOut1 = ReDataOut1Inter;
assign ImDataOut1 = ImDataOut1Inter;
`endif

endmodule //CordicRotFDTop

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
