//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39740 $
// $Date: 2019-09-09 18:42:06 +0200 (Mon, 09 Sep 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :Top file for FD Offset Compensation                                                     
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/FDOComp.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

module  FDOComp (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock

            // Data
            //STO & CPE Est Values.
            input    wire       signed     [21:0]          Sto,
            input    wire       signed     [21:0]          Cpe,
            //Rx Input Data & valid
            input    wire       signed     [12:0]          RxDataInRe1,
            input    wire       signed     [12:0]          RxDataInIm1,
`ifdef RW_NX_DERIV_PATH1
            input    wire       signed     [12:0]          RxDataInRe2,
            input    wire       signed     [12:0]          RxDataInIm2,
`endif
            input    wire                                  RxDataInValid,
            input    wire       signed     [9:0]           SubCarrierIndex,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output wire signed     [12:0]                  RxDataOutRe1,
            output wire signed     [12:0]                  RxDataOutIm1,
`ifdef RW_NX_DERIV_PATH1
            output wire signed     [12:0]                  RxDataOutRe2,
            output wire signed     [12:0]                  RxDataOutIm2,
`endif
            output wire                                    RxDataOutValid
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam    signed   [12:0] SIG_13_PARAM_0 = {13{1'b0}};
localparam    signed   [29:0] SIG_30_PARAM_0 = {30{1'b0}};

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers  & Wires Declarations
//////////////////////////////////////////////////////////////////////////////
//Compensating Angle
reg         signed     [29:0]   CompAngle;
//Delay of Valid and data signals
reg                             RxDataInValidD;
reg         signed     [12:0]   RxDataInRe1D;
reg         signed     [12:0]   RxDataInIm1D;
`ifdef RW_NX_DERIV_PATH1
reg         signed     [12:0]   RxDataInRe2D;
reg         signed     [12:0]   RxDataInIm2D;
`endif
reg         signed     [21:0]   Sto1D;
reg         signed     [21:0]   Cpe1D;

//Negate of Compensating angle.
wire        signed     [16:0]   AngleIn;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Compensate all subcarriers by negative of calculated angle.
assign AngleIn = - CompAngle[16:0];

//----------------------------------------------------------------------------
//Registering input valid signal & data.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:DataDelay
  if (nPhyRst == 1'b0)begin
    RxDataInValidD <= 1'b0;
    RxDataInRe1D   <= SIG_13_PARAM_0;
    RxDataInIm1D   <= SIG_13_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    RxDataInRe2D   <= SIG_13_PARAM_0;
    RxDataInIm2D   <= SIG_13_PARAM_0;
`endif
  end
  else begin
    RxDataInValidD <= RxDataInValid;
    RxDataInRe1D   <= RxDataInRe1;
    RxDataInIm1D   <= RxDataInIm1;
`ifdef RW_NX_DERIV_PATH1
    RxDataInRe2D   <= RxDataInRe2;
    RxDataInIm2D   <= RxDataInIm2;
`endif
  end
end

//----------------------------------------------------------------------------
//Registering sto & cpe.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:EstDelay
  if (nPhyRst == 1'b0)begin
    Sto1D <= 22'd0;
    Cpe1D <= 22'd0;
  end
  else begin
    Sto1D <= Sto;
    Cpe1D <= Cpe;
  end
end

//----------------------------------------------------------------------------
//Calculation of Compensation angle depending nulls & pilot locations.
//Pilot locations are read from PilotNb array.
//Null location and no of nulls are calculated from CarrierMidSkipAddr
//and CarrierMidSkipVal.Skipping null and pilot location data subcarriers are
//compensated.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:StoCpeCalc
  if (nPhyRst == 1'b0)begin
    CompAngle <= SIG_30_PARAM_0;
  end
  else begin
    CompAngle <= ($signed({{12{SubCarrierIndex[9]}},SubCarrierIndex}) * Sto1D ) + Cpe1D;
  end
end


//Instantiation of Cordic Rotator for rotating inputs.
 CordicRotFDTop #  (
                 .NPIPE(7),
                 .NCOMBSTG(2),
                 .SCALING(1),
                 .DATAWIDTH(13),
                 .ANGLEWIDTH(17))

 U_CordicRotFDTop(
                     //Inputs
                     .PhyClk(PhyClk),
                     .nPhyRst(nPhyRst),
                     .EnableIn(RxDataInValidD),
                     .AngleIn(AngleIn[16:0]),
                     .ReDataIn0(RxDataInRe1D[12:0]),
                     .ImDataIn0(RxDataInIm1D[12:0]),
`ifdef RW_NX_DERIV_PATH1
                     .ReDataIn1(RxDataInRe2D[12:0]),
                     .ImDataIn1(RxDataInIm2D[12:0]),
`endif
                     //Outputs
                     .ReDataOut0(RxDataOutRe1[12:0]),
                     .ImDataOut0(RxDataOutIm1[12:0]),
`ifdef RW_NX_DERIV_PATH1
                     .ReDataOut1(RxDataOutRe2[12:0]),
                     .ImDataOut1(RxDataOutIm2[12:0]),
`endif
                     .DataOutValid(RxDataOutValid)
                     );

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
