//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 16731 $
// $Date: 2014-11-07 11:48:26 +0100 (Fri, 07 Nov 2014) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :Top file for FD Offset Estimation & Compensation.
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://oringot@svn.frso.rivierawaves.com/svn/rw_wlan_nx/trunk/IPs/HW/Modem/Src/MODEMCORE/OFDMCORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/FDOffsetTop.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
module  FDOffsetTop #(parameter NSTS_PARAM    = 2,
`ifdef RW_NX_DERIV_NESS_EN
                      parameter NESS_PARAM    = 0,
`endif
                      parameter HMEMDATAWIDTH = 13, // Data width of one channel coefficient
                      parameter HMEMWIDTH     = 52  // Width per SC
                     )(

            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock

            ///////////////////////////////////////////////
            // Control
            ///////////////////////////////////////////////
            // Rx Control
            input    wire                                  FDReset,
            // ReceptionMode
              // 20MHz LEG         = 0
              // 40MHz DUP         = 1
              // 20MHz HT          = 2
              // 40MHz HT          = 3
              // 40MHz DUP mode HT = 4
              // 80MHz VHT         = 5
              // 20MHz HE SU       = 6
              // 20MHz HE MU       = 7
              // 40MHz HE SU       = 8
              // 40MHz HE MU       = 9
            input    wire                 [3:0]            ReceptionMode,
            // 0-LM;1-VHT,2-Mixed Mode;3-GF Mode
            input    wire                 [1:0]            Mode,
            // Bandwidth
            input    wire                 [1:0]            Bandwidth,
            // Start of every symbol.
            input    wire                                  WLSStartP,
            //
            input    wire                                  STOFineMuxSel,
            // HT Data start for Pilot Generation.
            input    wire                                  InitPilotShRegP,
            // Clear Kalman filter.
            input    wire                                  InitKalmanRegP,
            // Clear Kalman filter slope (HE only).
            input    wire                                  InitKalmanSlopeRegP,
            // Subcarriers Counter
            input    wire      signed     [9:0]            SubCarrierIndex,
            // Symbol Counter (Resets when signal fiels starts)
            input    wire                 [16:0]           SymCount,
            // Number of Spatial Stream:0-->1 SS,1-->2 SS
            input    wire                                  RxNsts,
            input    wire                 [2:0]            Dltf,
`ifdef RW_NX_DERIV_NESS_EN
            input    wire                 [2:0]            Eltf,
`endif
            input   wire                  [2:0]            RUType,          // Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)
            input   wire                  [3:0]            MidMulSlope,
            input   wire                  [1:0]            MidRShiftSlope,

            // End of pilot processing
            output wire                                    FDOCompDoneP,

            ///////////////////////////////////////////////
            // Equalised Pilot
            ///////////////////////////////////////////////
            input    wire      signed     [19:0]           EqPilotRe,
            input    wire      signed     [19:0]           EqPilotIm,
            input    wire                                  EqPilotValid,
            
            ///////////////////////////////////////////////
            // H matrix
            ///////////////////////////////////////////////
            //Next clock after WLSStart HMatrix should come.
            //HMatrix from Equaliser.
            input    wire         [HMEMWIDTH-1:0]          HMatrix,

            ///////////////////////////////////////////////
            // Rx Data
            ///////////////////////////////////////////////
            input    wire      signed     [12:0]           RxDataInRe1,
            input    wire      signed     [12:0]           RxDataInIm1,
`ifdef RW_NX_DERIV_PATH1
            input    wire      signed     [12:0]           RxDataInRe2,
            input    wire      signed     [12:0]           RxDataInIm2,
`endif
            input    wire                                  RxDataInValid,
            //
            output wire signed     [12:0]                  RxDataOutRe1,
            output wire signed     [12:0]                  RxDataOutIm1,
`ifdef RW_NX_DERIV_PATH1
            output wire signed     [12:0]                  RxDataOutRe2,
            output wire signed     [12:0]                  RxDataOutIm2,
`endif
            output wire                                    RxDataOutValid,
            
            ///////////////////////////////////////////////
            // Configuration Register
            ///////////////////////////////////////////////
            // No. of Rx Ant for Cur Pkt
            input    wire                 [1:0]            CfgNrx,           
            //Enable TD Compensation.When 0 sync skip sig is set to 0.
            input    wire                                  CfgStoTdCompEn,
            //Max +ive value of sync signal
            input    wire      signed     [5:0]            CfgMaxPtdSyncOff,
            //Max -ive value of sync signal
            input    wire      signed     [5:0]            CfgMaxNtdSyncOff,
            //Enable FD Compensation.When 0 STO O/P is set to 0.
            input    wire                                  CfgStoFdCompEn,
            //Fine or Coarse value is sent to FDO Compensation
            input    wire                                  CfgStoMode,
            //Enable FD compensation,when 0 CPE output is 0.
            input    wire                                  CfgCpeFdCompEn,
            //Coarse or Fine CPE value is sent.1-Fine/0-Coarse
            input    wire                                  CfgCpeMode,
            //Enable  STO slope of Kalman Filter.
            //0-Filled by CfgStoSlopeForced register value.
            input    wire                                  CfgStoSlopeEstEn,
            //STO slope to be forced.
            input    wire                [21:0]            CfgStoSlopeForced,
            //When 1 STO slope register is reinitialized when switching in HT mode.
            input    wire                                  CfgStoSlopeLgHt,

            ////Enable  CPE slope of Kalman Filter.
            //0-Filled by CfgCpeSlopeForced register value.
            input    wire                                  CfgCpeSlopeEstEn,
            //CPE slope to be forced.
            input    wire                [21:0]            CfgCpeSlopeForced,
            //1 --> CPE fine used for Coarse Ref Phase else CPE Coarse
            input    wire                                  CfgCpeRef,
            //1 --> STO fine used for Coarse Ref Phase else STO Coarse
            input    wire                                  CfgStoRef,
            //Cpe Wrap depending on Coarse or Fine value.
            input    wire                                  CfgCpeWalk,
            //TD compensation use coarse or fine value.
            input    wire                                  CfgSTO4TDComp,
            //When 1 CPE slope register is reinitialized when switching in HT mode.
            input    wire                                  CfgCpeSlopeLgHt,

            ///////////////////////////////////////////////
            // Status Register 
            ///////////////////////////////////////////////
            // value of synch skip signal at end of the packet.
            output wire signed    [5:0]                    CfgTdSynchOffStat,
            //value of STO slope at end of the packet.
            output wire signed    [21:0]                   CfgStoSlopeStat,
            //value of CPE slope at end of the packet.
            output wire signed    [21:0]                   CfgCpeSlopeStat,
            
            ///////////////////////////////////////////////
            // Samples to skip in TD
            ///////////////////////////////////////////////
            output wire signed    [5:0]                    SynchSkip
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////

// Mode
//       LM    = 0
//       VHT   = 1
//       HT-MM = 2
//       HT-GF = 3
localparam [1:0] MODE_LM   = 2'd0,
                 MODE_VHT  = 2'd1,
                 MODE_HTMM = 2'd2,
                 MODE_HTGF = 2'd3;

// Reception mode
localparam [3:0] MODE_20MHZ_L     = 4'd0,
                 MODE_40MHZ_LDUP  = 4'd1,
                 MODE_20MHZ_HT    = 4'd2,
                 MODE_40MHZ_HT    = 4'd3,
                 MODE_80MHZ_HT_8PILOTS = 4'd4, // not used
                 MODE_80MHZ_HT    = 4'd5,
                 MODE_20MHZ_HE_SU = 4'd6,
                 MODE_20MHZ_HE_MU = 4'd7,
                 MODE_40MHZ_HE_SU = 4'd8,
                 MODE_40MHZ_HE_MU = 4'd9;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers & Wires Declarations
//////////////////////////////////////////////////////////////////////////////
reg                   [1:0]     Bandwidth_1t;
reg                   [1:0]     NFFT_1t;
reg        signed     [21:0]    STOFine_1t;
reg        signed     [21:0]    CPEFine_1t;

wire       signed     [17:0]    STOCoarse;
wire       signed     [17:0]    CPECoarse;
wire                            CoarseOutValid;
wire       signed     [21:0]    STOFine;
wire       signed     [21:0]    CPEFine;
wire       signed     [21:0]    STOFineMux;
wire       signed     [21:0]    CPERefOut;
wire       signed     [23:0]    STORefOut;
wire                            LFSRShiftP;
wire                            InitPilotShRegPMux;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Relax timing constraints
always @(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    NFFT_1t      <= 2'b0;
    Bandwidth_1t <= 2'b0;
    STOFine_1t   <= 22'b0;
    CPEFine_1t   <= 22'b0;
  end
  else begin
    case (ReceptionMode)
      MODE_20MHZ_HE_SU,MODE_20MHZ_HE_MU : NFFT_1t <= 2'd2; // FFT 256
      MODE_40MHZ_HE_SU,MODE_40MHZ_HE_MU : NFFT_1t <= 2'd3; // FFT 512
      default                           : NFFT_1t <= Bandwidth;
    endcase
    Bandwidth_1t <= Bandwidth;
    STOFine_1t   <= STOFineMux;
    CPEFine_1t   <= CPEFine;
  end
end

//----------------------------------------------------------------------------
//Signal Field/Fields (LM/MM) & 1st two data symbols are not compensated with
//STO Fine value.
assign STOFineMux = STOFineMuxSel ? STOFine : 22'b0;

//----------------------------------------------------------------------------
//Reference previous symbol fine value for coarse frequency offset calculation.
//----------------------------------------------------------------------------

//LFSR shift in case of MM or VHT in order to by-pass STO/CPE estimation during HT-SIG-A2 or VHT-SIG-A2
// assign LFSRShiftP = (SymCount == 17'd1 && (Mode==MODE_HTMM || Mode==MODE_LM)) ? InitPilotShRegP : 1'b0;
assign LFSRShiftP = 1'b0;

//By-pass InitPilotShRegP in legacy case
assign InitPilotShRegPMux = (Mode != MODE_LM) ? InitPilotShRegP : 1'b0;

//Instantiation of Coarse Estimation.
 CoarseStoCpeTop # (.NSTS_PARAM(NSTS_PARAM),
`ifdef RW_NX_DERIV_NESS_EN
                    .NESS_PARAM(NESS_PARAM),
`endif
                    .HMEMDATAWIDTH(HMEMDATAWIDTH),
                    .HMEMWIDTH(HMEMWIDTH))

// Instantiation of coarse estimation.
 U_CoarseStoCpeTop(
                  //Inputs
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .FDReset(FDReset),
                  .CfgNrx(CfgNrx),
                  // Data and Control
                  //HMatrix from Equaliser.
                  .HMatrix(HMatrix),
                  .SubCarrierIndex(SubCarrierIndex),
                  .Bandwidth(Bandwidth_1t),
                  //Start calculating WLS Matrix
                  .WLSStartP(WLSStartP),
                  .ReceptionMode(ReceptionMode),
                  .RxNsts(RxNsts),
                  .RUType(RUType),
                  .InitPilotShRegP(InitPilotShRegPMux),
                  .LFSRShiftP(LFSRShiftP),
                  //Equalized Pilots
                  .EqPilotRe(EqPilotRe),
                  .EqPilotIm(EqPilotIm),
                  .EqPilotValid(EqPilotValid),
                  //
                  .STOFine(STORefOut),
                  .CPEFine(CPERefOut),
                  //Outputs
                  .STOCoarse(STOCoarse),
                  .CPECoarse(CPECoarse),
                  .CoarseOutValid(CoarseOutValid)
                  );

// Instantiation of fine estimation.
 StoCpeFilter U_StoCpeFilter(
           //Inputs
           .PhyClk(PhyClk),
           .nPhyRst(nPhyRst),
           .FDReset(FDReset),
           .EnableInCoarse(CoarseOutValid),
           .CpeCoarse(CPECoarse),
           .StoCoarse(STOCoarse),
           //Symbol Count
           .SymCount(SymCount),
           .NFFT(NFFT_1t),
           .Mode(Mode),
           .Dltf(Dltf),
`ifdef RW_NX_DERIV_NESS_EN
           .Eltf(Eltf),
`endif
           .InitKalmanRegP(InitKalmanRegP),
           .InitKalmanSlopeRegP(InitKalmanSlopeRegP),
           .MidMulSlope(MidMulSlope),
           .MidRShiftSlope(MidRShiftSlope),
           .MidSlopeUpdateP(WLSStartP),
           .CfgStoTdCompEn(CfgStoTdCompEn),
           .CfgMaxPtdSyncOff(CfgMaxPtdSyncOff),
           .CfgMaxNtdSyncOff(CfgMaxNtdSyncOff),
           .CfgStoFdCompEn(CfgStoFdCompEn),
           .CfgStoMode(CfgStoMode),
           .CfgCpeFdCompEn(CfgCpeFdCompEn),
           .CfgCpeMode(CfgCpeMode),
           .CfgStoSlopeEstEn(CfgStoSlopeEstEn),
           .CfgStoSlopeForced(CfgStoSlopeForced),
           .CfgCpeSlopeEstEn(CfgCpeSlopeEstEn),
           .CfgCpeSlopeForced(CfgCpeSlopeForced),
           .CfgStoSlopeLgHt(CfgStoSlopeLgHt),
           .CfgCpeSlopeLgHt(CfgCpeSlopeLgHt),
           .CfgTdSynchOffStat(CfgTdSynchOffStat),
           .CfgStoSlopeStat(CfgStoSlopeStat),
           .CfgCpeSlopeStat(CfgCpeSlopeStat),
           .CfgCpeRef(CfgCpeRef),
           .CfgStoRef(CfgStoRef),
           .CfgCpeWalk(CfgCpeWalk),
           .CfgSTO4TDComp(CfgSTO4TDComp),
           .STOFine(STOFine),
           .CPEFine(CPEFine),
           .FDOCompDoneP(FDOCompDoneP),
           .SynchSkip(SynchSkip),
           .CPERefOut(CPERefOut),
           .STORefOut(STORefOut)
           );

// Instantiation of Compensation.
FDOComp U_FDOComp (
         //Inputs
         .PhyClk(PhyClk),
         .nPhyRst(nPhyRst),
         // Data and Control
         //STO & CPE Est Values.
         .Sto(STOFine_1t),
         .Cpe(CPEFine_1t),
         //Rx Input Data & valid
         .RxDataInRe1(RxDataInRe1),
         .RxDataInIm1(RxDataInIm1),
`ifdef RW_NX_DERIV_PATH1
         .RxDataInRe2(RxDataInRe2),
         .RxDataInIm2(RxDataInIm2),
`endif
         .RxDataInValid(RxDataInValid),
         .SubCarrierIndex(SubCarrierIndex),
         //Outputs
         .RxDataOutRe1(RxDataOutRe1),
         .RxDataOutIm1(RxDataOutIm1),
`ifdef RW_NX_DERIV_PATH1
         .RxDataOutRe2(RxDataOutRe2),
         .RxDataOutIm2(RxDataOutIm2),
`endif
         .RxDataOutValid(RxDataOutValid)
         );

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
