////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39231 $
// $Date: 2019-06-25 17:34:30 +0200 (Tue, 25 Jun 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                       
// Description      :This is the Pilot Generator module which generates
//                   Pilot Sub-Carriers and uses a Scrambler to Scramble 
//                   the Bits. The Scrambler uses a Linear Feedback
//                   Shift Register with a Generator Polynomial
//                   of X^7+X^4+1.
// *                 For 20 MHz mode, it generates 4 Pilot Subcarriers
//                    per symbol.
// *                 For 40 and 80 MHz mode, it generates 6 Pilot Subcarriers
//                    per symbol.
// *                 The Scrambler/Pseudo-Random Number Generator 
//                   remains in the same state for all bits of a 
//                   particular OFDM symbol.
// *                 This module also contains a Look up table
//                   to assign initial values to the pilots for the first 
//                   symbol of a Packet.
// *                 The pilots generated by the Pilot Generator
//                   are cyclically rotated for the next symbol.
// *                 The Pilots are given out in the increasing order
//                   i.e from -103,-75,-11,11,75,103 for 80 MHz mode.
//                   i.e from -53,-25,-11,11,25,53 for 40 MHz mode.
//                   and from -21,-7,7,21 for 20 MHz mode. 
//                   
// Simulation Notes :                                       
// Synthesis Notes  :                                        
// Application Note :                                        
// Simulator        : NC-VERILOG                                      
// Parameters       :             
// Terms & concepts : 
// Bugs             :
// Open issues and future enhancements :         
// References       :
// Revision History : 
// -------------------------------------------------------------------------
//                                     
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/GenPilot.v $
// 
////////////////////////////////////////////////////////////////////////////
module GenPilot (
            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            // Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock

            // Data & Control
            input    wire                                  Nsts, // Number of space time streams
            input    wire  [2:0]                           Nsp,
            // Control for 20/40/80 MHz mode pilots:
            // This signal is relevant only when InitPilotShRegP has been received.
            //      0: generate 4 pilots by symbol (20 MHz) 
            //      1: generate 6 pilots by symbol in 11n mode (40 MHz).
            //      2: generate 6 pilots by symbol in 11ac mode (80 MHz).
            //      3: generate 2 pilots by symbol in 11ax mode (RU26).
            //      4: generate 6 pilots by symbol in 11ax mode (RU484).
            input    wire  [2:0]                           GenPilotMode,
            // Signal to indicate the Pilot generator to start the shifting of pilots
            input    wire                                  PilotShiftOutP,
            // Signal to initialize the scrambler of the Pilot Generator
            input    wire                                  InitPilotScramP,
            // Signal to initialize the Pilot shift registers
            input    wire                                  InitPilotShRegP,
            // Signal to indicate the Pilot generator to start the shifting of LFSR
            input    wire                                  LFSRShiftP,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            // Pilots corresponding to STS 1 & 2.
            output wire  [5:0]                             Pilot1,
            output wire  [5:0]                             Pilot2,
            // Valid pilots for the STS 1 & 2.
            output reg                                     PilotDataValid1,
            output reg                                     PilotDataValid2
            );

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
// Valid pilot bits
reg         [1:0]          PilotDataValid;

// LUT for first symbol Pilot sequences
reg         [15:0]         PilotSeqInit[1:0];

// 7-bit LFSR
reg         [6:0]          LFSRReg;

// Pilot shift registers
reg         [15:0]         PilotShiftReg[1:0];
reg         [5:0]          Pilot1Reg;
reg         [5:0]          Pilot2Reg;

//delayed version of PilotShiftOutP
reg                        PilotShiftOutP_1t;

// Pilot state: 0=Legacy, 1=HT
reg                        PilotState;

// Pilot counter
reg         [2:0]          PilotCounter;


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires & Vars Declarations
//////////////////////////////////////////////////////////////////////////////
// Feedback of 7-bit LFSR
wire                       LFSRFeedback;
// Number of active streams
wire                       NumStreams;
// Pilot selection
wire        [5:0]          PilotShiftRegSel [1:0];
// Pilot output
wire        [5:0]          Pilot1Out;
wire        [5:0]          Pilot2Out;

integer                    j;
integer                    k;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//----------------------------------------------------------------------------
// *Lookup table for first symbol Pilot sequences, each location holds
// sequences for 1 Space Time Stream.
// The signal GenPilotMode chooses whether 4 or 6 pilot bits are generated
// ----------------------------------------------------------------------------
 always @(*)
 begin: pPilotInitSeqComb
   case(GenPilotMode)
     3'd0: begin // 4 pilots, 20 MHz
       case(Nsts)
         1'b0: begin
           PilotSeqInit[0] = 16'b0000000011100000;
           PilotSeqInit[1] = 16'b0000000011100000;
         end
         1'b1: begin
           PilotSeqInit[0] = 16'b0000000011000000;
           PilotSeqInit[1] = 16'b0000000010010000;
         end
       endcase
     end
     3'd1: begin // 6 pilots, 40 MHz
       if(Nsts == 1'b0)   begin
         PilotSeqInit[0] = 16'b0000000011100100;
         PilotSeqInit[1] = 16'b0000000011100000;
       end
       else begin
         PilotSeqInit[0] = 16'b0000000011000000;
         PilotSeqInit[1] = 16'b0000000011101100;
       end
     end
     3'd2: begin // 6 pilots, 80 MHz ac or 20 MHz HE
         PilotSeqInit[0] = 16'b0000000011100111;
         PilotSeqInit[1] = 16'b0000000011100111;
     end
     3'd3: begin // RU 26, Nsp=2
         PilotSeqInit[0] = 16'b0000000010000000;
         PilotSeqInit[1] = 16'b0000000000000000; // TODO
     end
     default: begin // RU484, Nsp=16
         PilotSeqInit[0] = 16'b1110011111100111;
         PilotSeqInit[1] = 16'b0000000000000000; // TODO
     end
   endcase
 end


//----------------------------------------------------------------------------
// Pilot counter:
// * For each OFDM symbol, it is set to either 4 or 6 depending on the
// mode(20 MHz or 40/80 MHz)
// * The counter is implemented as a down counter.
// * It counts down from 4 to 1 for 20 MHz mode.
// and from 6 to 1 for 40/80 MHz mode.
// * After the counter reaches 1, it will be reset to 0
//----------------------------------------------------------------------------
 always @(posedge PhyClk or negedge nPhyRst)
 begin: pCounterSeq
   if(nPhyRst == 1'b0) begin
     PilotCounter <= 3'd0;
   end
   else begin
     if(PilotShiftOutP == 1'b1) begin
       PilotCounter <= Nsp;
     end
       else
     begin
       if(PilotCounter > 3'd1)
         PilotCounter <= PilotCounter-3'd1;
       else begin
         PilotCounter <= 3'd0;
       end
     end
   end
 end

//----------------------------------------------------------------------------
// 7-bit LFSR:
// The LFSR shifts changes state after all Pilot bits corresponding
// to a symbol are shifted out
//----------------------------------------------------------------------------
 assign LFSRFeedback  = LFSRReg[6] ^ LFSRReg[3];

 always @(posedge PhyClk or negedge nPhyRst)
 begin: pLFSRSeq
   if(nPhyRst == 1'b0)
     LFSRReg <= 7'b1111111;
   else if(InitPilotScramP == 1'b1)
     LFSRReg <= 7'b1111111;
   else if(PilotShiftOutP_1t==1'b1 || LFSRShiftP==1'b1)
     LFSRReg <= {LFSRReg[5:0],LFSRFeedback};
 end

//----------------------------------------------------------------------------
// Pilot state:
// *For Legacy mode, only InitPilotScramP signal is asserted
//for one clock.InitPilotShRegP is not asserted for legacy mode.
// *For HT mode, InitPilotScramP is asserted for one clock,
//followed by the assertion of InitPilotShRegP
// *So ,for HT mode, PilotState will be 1,
//whereas for Legacy mode, Pilot State will be 0
//----------------------------------------------------------------------------
 always @(posedge PhyClk or negedge nPhyRst)
 begin: pPilotStateSeq
   if(nPhyRst == 1'b0)
     PilotState <= 1'b0;
   else if(InitPilotScramP == 1'b1)
     PilotState <= 1'b0;
   else if(InitPilotShRegP == 1'b1)
     PilotState <= 1'b1;
 end


//----------------------------------------------------------------------------
// For Legacy mode, take the Number of Space Time streams as 0
//                                      (one stream only).
// For HT mode, take the Number of Space time streams as Nsts.
//----------------------------------------------------------------------------
 assign NumStreams = (PilotState == 1'b1) ? Nsts : 1'b0;

 always @(posedge PhyClk or negedge nPhyRst)
 begin: pValidComb
   if(nPhyRst==1'b0) begin
     PilotDataValid[0] <= 1'b0;
     PilotDataValid[1] <= 1'b0;
   end
   else if (NumStreams == 1'b0)
     PilotDataValid[0] <= PilotShiftOutP;
   else begin
     for(j=0; j<=1; j=j+1)
       PilotDataValid[j] <= PilotShiftOutP;
   end
 end

 always @(posedge PhyClk or negedge nPhyRst)
 begin
   if(nPhyRst==1'b0)
     PilotShiftOutP_1t <= 1'b0;
   else
     PilotShiftOutP_1t <= PilotShiftOutP;
 end

  always @(*) begin
    if (PilotCounter == Nsp) begin
      PilotDataValid1 = PilotDataValid[0];
      PilotDataValid2 = PilotDataValid[1];
    end
    else if (PilotCounter == 3'd0) begin
      PilotDataValid1 = 1'b0;
      PilotDataValid2 = 1'b0;
    end
    else if (PilotCounter < 3'd6 && NumStreams==1'b1) begin
      PilotDataValid1 = 1'b1;
      PilotDataValid2 = 1'b1;
    end
    else if (PilotCounter < 3'd6 && NumStreams==1'b0) begin
      PilotDataValid1 = 1'b1;
      PilotDataValid2 = 1'b0;
    end
    else begin
      PilotDataValid1 = 1'b0;
      PilotDataValid2 = 1'b0;
    end
  end

//----------------------------------------------------------------------------
// Pilot shift registers:
//  These Shift Registers apply  cyclic left shifts
//    after all the Pilots of a symbol are shifted out.
//  For 80MHz mode, a left shift happens and
// the MSB is shifted to the LSB
//  For 40MHz mode, a left shift happens and
// the MSB is shifted to bit position 2
//  (Since the positions 1 and 0 are not used)
//  For 20MHz mode, a left shift happens and
// the MSB is shifted to bit position 4
//  (Since the positions 3,2,1 and 0 are not used)
//----------------------------------------------------------------------------
 always @(posedge PhyClk or negedge nPhyRst)
 begin
   if(nPhyRst == 1'b0) begin
     for(k=0; k<=1; k=k+1)
       PilotShiftReg[k] <= 16'b0000000011100000;
   end
   else if(InitPilotScramP == 1'b1) begin
     for(k=0; k<=1; k=k+1)
       PilotShiftReg[k] <= 16'b0000000011100000;
   end
   else if(InitPilotShRegP == 1'b1) begin
     for(k=0; k<=Nsts; k=k+1)
       PilotShiftReg[k] <= PilotSeqInit[k];
   end
   else if((PilotState == 1'b1) && (PilotShiftOutP_1t== 1'b1)) begin
     for(k=0; k<=NumStreams; k=k+1) begin
       case (GenPilotMode)
         3'd0   : PilotShiftReg[k] <= {8'b0,PilotShiftReg[k][6:4], PilotShiftReg[k][7], PilotShiftReg[k][3:0]}; // 20 MHz non he
         3'd1   : PilotShiftReg[k] <= {8'b0,PilotShiftReg[k][6:2], PilotShiftReg[k][7], PilotShiftReg[k][1:0]}; // 40 MHz
         3'd2   : PilotShiftReg[k] <= {8'b0,PilotShiftReg[k][6:0], PilotShiftReg[k][7]};                        // 80 MHz ac, 20 MHz he
         3'd3   : PilotShiftReg[k] <= {8'b0,PilotShiftReg[k][6],   PilotShiftReg[k][7], PilotShiftReg[k][5:0]}; // RU26 He
         default: PilotShiftReg[k] <= {PilotShiftReg[k][14:0], PilotShiftReg[k][15]};                           // RU484 He
       endcase
     end
   end
 end

 //----------------------------------------------------------------------------
 //Registering Pilots
 //----------------------------------------------------------------------------
 always@(posedge PhyClk or negedge nPhyRst)
 begin
   if(nPhyRst == 1'b0) begin
     Pilot1Reg <= 6'b0;
     Pilot2Reg <= 6'b0;
   end
   else if (PilotShiftOutP) begin
     Pilot1Reg <= Pilot1Out;
     Pilot2Reg <= Pilot2Out;
   end
 end

 //----------------------------------------------------------------------------
 // Pilot selection : 6 pilots selection on the 8 or 16 pilots
 //----------------------------------------------------------------------------
 assign PilotShiftRegSel[0] = (GenPilotMode == 3'd4) ? {PilotShiftReg[0][14],PilotShiftReg[0][12],PilotShiftReg[0][9],
                                                        PilotShiftReg[0][6],PilotShiftReg[0][3],PilotShiftReg[0][1]} :
                              (GenPilotMode != 3'd2) ? PilotShiftReg[0][7:2] : 
          {PilotShiftReg[0][7:6],PilotShiftReg[0][4:3],PilotShiftReg[0][1:0]};
 assign PilotShiftRegSel[1] = (GenPilotMode == 3'd4) ? {PilotShiftReg[1][14],PilotShiftReg[1][12],PilotShiftReg[1][9],
                                                        PilotShiftReg[1][6],PilotShiftReg[1][3],PilotShiftReg[1][1]} :
                              (GenPilotMode != 3'd2) ? PilotShiftReg[1][7:2] :
          {PilotShiftReg[1][7:6],PilotShiftReg[1][4:3],PilotShiftReg[1][1:0]};

 //----------------------------------------------------------------------------
 // Output Pilot Bits:
 // Scramble the Pilot bit to be sent out
 //----------------------------------------------------------------------------
 assign Pilot1Out = (PilotShiftOutP) ? (PilotShiftRegSel[0] ^ {6{LFSRFeedback}}) : Pilot1Reg;
 assign Pilot2Out = (PilotShiftOutP) ? (PilotShiftRegSel[1] ^ {6{LFSRFeedback}}) : Pilot2Reg;

 assign Pilot1 = Pilot1Out;
 assign Pilot2 = Pilot2Out;

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
