//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 40472 $
// $Date: 2019-11-21 10:36:04 +0100 (Thu, 21 Nov 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :Kalman Filter Implementation                                              
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/KalmanFilter.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

module KalmanFilter (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock
            //Configuration registers
            input    wire                                  CfgSlopeEstEn,
            input    wire    signed      [21:0]            CfgSlopeForced,
            //Init
            input    wire                                  FDReset,
            input    wire                                  SlopeReset,
            input    wire                                  InitPsi,
            //Theta Coarse is valid.
            input    wire                                  EnableInCoarse,
            input    wire     signed     [22:0]            ThetaCoarse,
            //Previous fine value.
            input    wire     signed     [23:0]            Psi,
            //Symbol Counter
            input    wire                [16:0]            SymCount,
            //Midamble skip slope adjustment
            input   wire                  [3:0]            MidMulSlope,
            input   wire                  [1:0]            MidRShiftSlope,
            input   wire                                   MidSlopeUpdateP,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Output slope for coarse update
            output  reg      signed      [20:0]            MidSlopeSkipCoarse,
            //Output Theta with enable
            output  reg                                    EnableOutFine,
            output  reg      signed      [21:0]            ThetaFine,
            //Output delta psi.
            output  reg      signed      [21:0]            Slope
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
//Delta Psi of last symbol.
reg             signed          [21:0]               DeltaPsiPriori;

//Psi of last symbol.
reg             signed          [23:0]               PsiPriori;

//Innovation value (Psi Predicted - 2*Theta Coarse)
reg             signed          [25:0]               Innovation;

//Delay version of enable.
reg                                                  Enable_1t;
reg                                                  Enable_2t;

//Delay version of MidSlopeUpdateP.
reg                                                  MidSlopeUpdateP_1t;

//Delay version of SymCount.
reg                             [5:0]                KalmanSymCount;

reg             signed          [21:0]               DeltaThetaFine;

//Register rounding outputs.
reg             signed          [22:0]               RegRndWeightedInKdelta;
reg             signed          [22:0]               RegRndWeightedInK;

//Register slope adjustment in case of midamble skip.
reg             signed          [21:0]               MidSlopeSkip;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
//Multiplication of Psi & Delta psi with kalman gain.
wire            signed          [32:0]               WeightedInK;
wire            signed          [32:0]               WeightedInKdelta;

//Kalman gain
wire                            [8:0]                KTab[49:0];
wire                            [8:0]                KdeltaTab[49:0];
wire                            [8:0]                KGain;
wire                            [8:0]                KdeltaGain;

//PsiPredicted = Psi +Delta Psi.
wire            signed          [24:0]               PsiPredicted;

//Psi of current symbol.
wire            signed          [24:0]               PsiPosterior;

//Delta Psi of current symbol.
wire            signed          [23:0]               DeltaPsiPosteriori;

//Saturated Innovation
wire            signed          [21:0]               SatInnovation;
wire            signed          [24:0]               PsiNextSymbol;
wire            signed          [21:0]               SatThetaFine;
wire            signed          [21:0]               SatDeltaPsiPosteriori;
wire            signed          [22:0]               RndWeightedInKdelta;
wire            signed          [22:0]               RndWeightedInK;
wire            signed          [21:0]               DeltaPsi;

//Slope adjusted
wire            signed          [21:0]               SlopeMul;
wire            signed          [20:0]               SlopeRShift;
wire            signed          [20:0]               Rnd2Slope;
wire            signed          [19:0]               Rnd4Slope;
wire            signed          [20:0]               MidSlopeSkipRnd;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//----------------------------------------------------------------------------
//Registering EnableInCoarse
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:Del_en
  if (nPhyRst == 1'b0)
  begin
    Enable_1t          <= 1'b0;
    Enable_2t          <= 1'b0;
    EnableOutFine      <= 1'b0;
    MidSlopeUpdateP_1t <= 1'b0;
  end
  else if (FDReset)
  begin
    Enable_1t          <= 1'b0;
    Enable_2t          <= 1'b0;
    EnableOutFine      <= 1'b0;
    MidSlopeUpdateP_1t <= 1'b0;
  end
  else begin
    Enable_1t          <= EnableInCoarse;
    Enable_2t          <= Enable_1t;
    EnableOutFine      <= Enable_2t;
    MidSlopeUpdateP_1t <= MidSlopeUpdateP;
  end
end//Del_en

//----------------------------------------------------------------------------
//Kalman Gain Coefficients.
//Coeficients depends on Symbol Count.For L-Sig,X-Sig,X-Sym,HT-Sig
//Symbol Count is '0'.Then it increments for every symbol.
//For Symbol Count more than 50 always 50th value is considered.
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:Del_SymCount
  if (nPhyRst == 1'b0)
    KalmanSymCount <= 6'b0;
  else if (SymCount < 17'd49)
    KalmanSymCount <= SymCount[5:0];
  else
    KalmanSymCount <= 6'd49;
end//Del_SymCount

assign KTab[0] = 9'd341;
assign KTab[1] = 9'd341;
assign KTab[2] = 9'd384;
assign KTab[3] = 9'd391;
assign KTab[4] = 9'd378;
assign KTab[5] = 9'd358;
assign KTab[6] = 9'd336;
assign KTab[7] = 9'd315;
assign KTab[8] = 9'd296;
assign KTab[9] = 9'd278;
assign KTab[10] = 9'd261;
assign KTab[11] = 9'd247;
assign KTab[12] = 9'd233;
assign KTab[13] = 9'd221;
assign KTab[14] = 9'd210;
assign KTab[15] = 9'd200;
assign KTab[16] = 9'd191;
assign KTab[17] = 9'd183;
assign KTab[18] = 9'd175;
assign KTab[19] = 9'd168;
assign KTab[20] = 9'd162;
assign KTab[21] = 9'd156;
assign KTab[22] = 9'd150;
assign KTab[23] = 9'd145;
assign KTab[24] = 9'd140;
assign KTab[25] = 9'd135;
assign KTab[26] = 9'd131;
assign KTab[27] = 9'd127;
assign KTab[28] = 9'd123;
assign KTab[29] = 9'd120;
assign KTab[30] = 9'd116;
assign KTab[31] = 9'd113;
assign KTab[32] = 9'd110;
assign KTab[33] = 9'd107;
assign KTab[34] = 9'd105;
assign KTab[35] = 9'd102;
assign KTab[36] = 9'd99;
assign KTab[37] = 9'd97;
assign KTab[38] = 9'd95;
assign KTab[39] = 9'd93;
assign KTab[40] = 9'd91;
assign KTab[41] = 9'd89;
assign KTab[42] = 9'd87;
assign KTab[43] = 9'd85;
assign KTab[44] = 9'd83;
assign KTab[45] = 9'd82;
assign KTab[46] = 9'd80;
assign KTab[47] = 9'd79;
assign KTab[48] = 9'd77;
assign KTab[49] = 9'd76;

assign KdeltaTab[0] = 9'd341;
assign KdeltaTab[1] = 9'd341;
assign KdeltaTab[2] = 9'd256;
assign KdeltaTab[3] = 9'd186;
assign KdeltaTab[4] = 9'd138;
assign KdeltaTab[5] = 9'd106;
assign KdeltaTab[6] = 9'd83;
assign KdeltaTab[7] = 9'd67;
assign KdeltaTab[8] = 9'd55;
assign KdeltaTab[9] = 9'd46;
assign KdeltaTab[10] = 9'd39;
assign KdeltaTab[11] = 9'd34;
assign KdeltaTab[12] = 9'd29;
assign KdeltaTab[13] = 9'd26;
assign KdeltaTab[14] = 9'd23;
assign KdeltaTab[15] = 9'd20;
assign KdeltaTab[16] = 9'd18;
assign KdeltaTab[17] = 9'd16;
assign KdeltaTab[18] = 9'd15;
assign KdeltaTab[19] = 9'd13;
assign KdeltaTab[20] = 9'd12;
assign KdeltaTab[21] = 9'd11;
assign KdeltaTab[22] = 9'd10;
assign KdeltaTab[23] = 9'd9;
assign KdeltaTab[24] = 9'd9;
assign KdeltaTab[25] = 9'd8;
assign KdeltaTab[26] = 9'd8;
assign KdeltaTab[27] = 9'd7;
assign KdeltaTab[28] = 9'd7;
assign KdeltaTab[29] = 9'd6;
assign KdeltaTab[30] = 9'd6;
assign KdeltaTab[31] = 9'd5;
assign KdeltaTab[32] = 9'd5;
assign KdeltaTab[33] = 9'd5;
assign KdeltaTab[34] = 9'd5;
assign KdeltaTab[35] = 9'd4;
assign KdeltaTab[36] = 9'd4;
assign KdeltaTab[37] = 9'd4;
assign KdeltaTab[38] = 9'd4;
assign KdeltaTab[39] = 9'd4;
assign KdeltaTab[40] = 9'd3;
assign KdeltaTab[41] = 9'd3;
assign KdeltaTab[42] = 9'd3;
assign KdeltaTab[43] = 9'd3;
assign KdeltaTab[44] = 9'd3;
assign KdeltaTab[45] = 9'd3;
assign KdeltaTab[46] = 9'd3;
assign KdeltaTab[47] = 9'd3;
assign KdeltaTab[48] = 9'd2;
assign KdeltaTab[49] = 9'd2;

//Assign forced slope if  CfgSlopeEstEn is asserted otherwise previous
//symbol slope.
assign DeltaPsi = (CfgSlopeEstEn) ? DeltaThetaFine : CfgSlopeForced;

//----------------------------------------------------------------------------
//STEP:1 Predicted Measure
//----------------------------------------------------------------------------
//Current Symbol slope= Previous Symbol fine value +Previous Slope.
assign PsiPredicted = Psi + $signed({{2{DeltaPsi[21]}},DeltaPsi});

//----------------------------------------------------------------------------
//STEP:2 Calculation of Innovation.
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:CalcInnovation
  if (nPhyRst == 1'b0)
    Innovation <= 26'b0;
  else if(FDReset || InitPsi)
    Innovation <= 26'b0;
  else if(EnableInCoarse)
    Innovation <= $signed({ThetaCoarse,1'b0}) - PsiPredicted;
end//CalcInnovation

//----------------------------------------------------------------------------
//STEP: 3 Innovation updated with weighted kalman gain factor
//----------------------------------------------------------------------------
assign KGain            = KTab[KalmanSymCount];
assign KdeltaGain       = KdeltaTab[KalmanSymCount];
assign WeightedInK      = $signed({1'b0,KGain})      * SatInnovation;
assign WeightedInKdelta = $signed({1'b0,KdeltaGain}) * SatInnovation;

//----------------------------------------------------------------------------
//STEP: 4 Calculation of Posterior Estimate from Priori Estimate
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:PosteriorEstimate
  if (nPhyRst == 1'b0)begin
    RegRndWeightedInK      <= 23'b0;
    RegRndWeightedInKdelta <= 23'b0;
  end
  else if (FDReset || InitPsi) begin
    RegRndWeightedInK      <= 23'b0;
    RegRndWeightedInKdelta <= 23'b0;
  end
  else begin
    RegRndWeightedInK      <= RndWeightedInK;
    RegRndWeightedInKdelta <= RndWeightedInKdelta;
  end
end//PosteriorEstimate

assign PsiPosterior       = PsiPriori                                          +
                            $signed({RegRndWeightedInK[22],RegRndWeightedInK});
assign DeltaPsiPosteriori = $signed({DeltaPsiPriori[21],DeltaPsiPriori})       +
                            RegRndWeightedInKdelta;

//----------------------------------------------------------------------------
//STEP: 5 Calculation of Psi for next symbol
//----------------------------------------------------------------------------
assign PsiNextSymbol = PsiPosterior + $signed({DeltaPsiPosteriori[23],DeltaPsiPosteriori});

//----------------------------------------------------------------------------
//STEP: 6 Registering Theta Fine
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:ThetaFineReg
  if (nPhyRst == 1'b0)begin
    ThetaFine      <= 22'b0;
    DeltaThetaFine <= 22'b0;
  end
  else if (FDReset || SlopeReset) begin
    ThetaFine      <= 22'b0;
    DeltaThetaFine <= 22'b0;
  end
  else if (InitPsi)
    ThetaFine      <= 22'b0;
  else if (MidSlopeUpdateP_1t)
    ThetaFine      <= ThetaFine + MidSlopeSkip;
  else if (Enable_2t) begin
    ThetaFine      <= SatThetaFine;
    DeltaThetaFine <= SatDeltaPsiPosteriori;
  end
end//ThetaFineReg

//----------------------------------------------------------------------------
//Registering Psi
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:DelayPsi
  if (nPhyRst == 1'b0)begin
    PsiPriori      <= 24'b0;
    DeltaPsiPriori <= 22'b0;
  end
  else if (FDReset || InitPsi) begin
    PsiPriori      <= 24'b0;
    DeltaPsiPriori <= 22'b0;
  end
  else begin
    PsiPriori      <= Psi;
    DeltaPsiPriori <= DeltaPsi;
  end
end//DelayPsi

//----------------------------------------------------------------------------
//Registering Slope
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:SlopeReg
  if (nPhyRst == 1'b0)
    Slope <= 22'b0;
  else
    Slope <= DeltaThetaFine;
end//SlopeReg

//Saturation of innovation to 22 bits
SatSigned #(
               .INPUT_WIDTH(26),
               .OUTPUT_WIDTH(22)
              )
               U1_SAT(
                        .InputData(Innovation),
                        .SatData(SatInnovation)
                        );


//Instantiation of Round component
Round #(
        .INPUT_WIDTH(33),
        .OUTPUT_WIDTH(23)
       )
        U1_ROUND(
                    .InputData(WeightedInK),
                    .RoundData(RndWeightedInK)
                   );


//Instantiation of Round component
Round #(
        .INPUT_WIDTH(33),
        .OUTPUT_WIDTH(23)
       )
        U2_ROUND(
                    .InputData(WeightedInKdelta),
                    .RoundData(RndWeightedInKdelta)
                   );

//Saturation of DeltaPsiPosteriori to 22 bits
SatSigned #(
               .INPUT_WIDTH(24),
               .OUTPUT_WIDTH(22)
              )
               U2_SAT(
                        .InputData(DeltaPsiPosteriori),
                        .SatData(SatDeltaPsiPosteriori)
                        );

//Saturation of PsiNextSymbol to 22 bits
SatSigned #(
               .INPUT_WIDTH(25),
               .OUTPUT_WIDTH(22)
              )
               U3_SAT(
                        .InputData(PsiNextSymbol),
                        .SatData(SatThetaFine)
                        );

//----------------------------------------------------------------------------
//Compute Slope adjustment for midamble skip
//----------------------------------------------------------------------------
assign SlopeMul    = Slope * $signed({1'b0,MidMulSlope});
assign SlopeRShift = (MidRShiftSlope == 2'd1) ? Rnd2Slope :
                     (MidRShiftSlope == 2'd2) ? {Rnd4Slope[19],Rnd4Slope} : 21'b0;

Round #(
        .INPUT_WIDTH(22),
        .OUTPUT_WIDTH(21)
       )
        U3_ROUND(
                    .InputData(Slope),
                    .RoundData(Rnd2Slope)
                   );

Round #(
        .INPUT_WIDTH(22),
        .OUTPUT_WIDTH(20)
       )
        U4_ROUND(
                    .InputData(Slope),
                    .RoundData(Rnd4Slope)
                   );

always@(posedge PhyClk or negedge nPhyRst)
begin:MidSlopeSkipReg
  if (nPhyRst == 1'b0)
    MidSlopeSkip <= 22'b0;
  else
    MidSlopeSkip <= SlopeMul + $signed({SlopeRShift[20],SlopeRShift});
end//MidSlopeSkipReg

Round #(
        .INPUT_WIDTH(22),
        .OUTPUT_WIDTH(21)
       )
        U5_ROUND(
                    .InputData(MidSlopeSkip),
                    .RoundData(MidSlopeSkipRnd)
                   );

always@(posedge PhyClk or negedge nPhyRst)
begin:MidSlopeSkipCoarseReg
  if (nPhyRst == 1'b0)
    MidSlopeSkipCoarse <= 21'b0;
  else
    MidSlopeSkipCoarse <= MidSlopeSkipRnd;
end//MidSlopeSkipCoarseReg


endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
