///////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39740 $
// $Date: 2019-09-09 18:42:06 +0200 (Mon, 09 Sep 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :Calculation of M & N Matrix.                                                     
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/MNMatrix.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

module  MNMatrix (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                                   nPhyRst, //Active LOW Reset
            input   wire                                   PhyClk,  //PHY Clock

            // Data and Control
            // Input Gamma & Enable from EquCoefModTop.
            input   wire                                   EnableIn,
            input   wire        [5:0]                      GammaIn,
            input   wire        [2:0]                      Nsp,

            //Index from Equaliser.
            input   wire signed [9:0]                      SubCarrierIndex,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Enable and M & N Outputs
            output   wire signed     [9:0]                 SubCarrierIndexOut,
            output   reg                                   EnableNMatrix,
            output   reg                                   EnableOut,
            output   wire signed    [14:0]                 NMatrixRow1,
            output   wire signed    [13:0]                 NMatrixRow2,
            output   reg  signed    [20:0]                 M22,
            output   reg  signed    [21:0]                 M21,
            output   wire           [24:0]                 M11
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                        [2:0]        Count;
reg                        [2:0]        Count_1t;
reg                        [2:0]        Count_2t;

//Registering M11,M12,M22.
reg              signed    [20:0]       M22Reg;
reg              signed    [20:0]       M22Reg_1t;
reg              signed    [21:0]       M21Reg;
reg                        [24:0]       M11Reg;
reg              signed    [5:0]        GammaReg;
reg                                     EnableIn_1t;
reg                                     EnableIn_2t;
reg                        [6:0]        EnableDelay;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
//K*K*Gamma
wire                       [22:0]       KSquGamma;
//K*Gamma
wire             signed    [14:0]       KGamma;
wire             signed    [6:0]        GammaMul;

wire             signed    [9:0]        SubCarrierIndex_8t;
wire             signed    [9:0]        SubCarrierIndex_10t;
wire             signed    [9:0]        SubCarrierIndex_12t;
wire             signed    [9:0]        SubCarrierIndexMux;
wire             signed    [9:0]        SubCarrierIndexK2;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

assign M11         = M11Reg;
assign NMatrixRow2 = {8'b0,GammaReg};
assign NMatrixRow1 = KGamma;
assign GammaMul    = {1'b0,GammaIn};

//----------------------------------------------------------------------------
//Keeps track of No. of Pilot subcarrier depending on Nsp.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:Counter
  if (nPhyRst == 1'b0)begin
     Count    <= 3'd0;
     Count_1t <= 3'd0;
     Count_2t <= 3'd0;
  end
  else begin
    Count_1t <= Count;
    Count_2t <= Count_1t;
    if (EnableIn) begin
      if (Count < (Nsp-3'd1))
        Count <= Count + 3'd1;
      else
        Count <= 3'd0;
    end
  end
end //Counter

//----------------------------------------------------------------------------
//Generating EnableIn_1t to assign initial value gamma,kgamma or k2gamma.
//Used along with zero to avoid assignmets for reset zero.
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:EnableIn_1tBlk
  if (nPhyRst == 1'b0)begin
     EnableIn_1t <= 1'b0;
     EnableIn_2t <= 1'b0;
  end
  else begin
     EnableIn_1t <= EnableIn;
     EnableIn_2t <= EnableIn_1t;
  end
end //EnableIn_1tBlk

//----------------------------------------------------------------------------
//Calculation of M22 element of M Matrix
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:ADDGAMMA
  if (nPhyRst == 1'b0)begin
     M22Reg <= $signed({21{1'b0}});
  end
  else if (Count== 3'b0 && EnableIn == 1'b1) begin
     M22Reg  <= {15'b0,GammaIn};
  end
  else if (Count >3'b0 && Count<Nsp) begin
     M22Reg  <= M22Reg + {15'b0,GammaIn};
  end
end

//----------------------------------------------------------------------------
//Calculation of M21 element of M Matrix
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:ADDKGAMMA
  if (nPhyRst == 1'b0)begin
     M21Reg <= $signed({22{1'b0}});
  end
  else if (Count_1t== 3'b0 && EnableIn_1t == 1'b1) begin
     M21Reg  <= $signed({{7{KGamma[14]}},KGamma});
  end
  else if (Count_1t >3'b0 && Count_1t<Nsp) begin
     M21Reg  <= M21Reg + $signed({{7{KGamma[14]}},KGamma});
  end
end

//----------------------------------------------------------------------------
//Calculation of M11 element of M Matrix
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:ADDK2GAMMA
  if (nPhyRst == 1'b0)begin
     M11Reg <= {25{1'b0}};
  end
  else if (Count_2t== 3'b0 && EnableIn_2t == 1'b1) begin
     M11Reg  <= {2'b0,KSquGamma};
  end
  else if (Count_2t >3'b0 && Count_2t<Nsp) begin
     M11Reg  <= M11Reg + {2'b0,KSquGamma};
  end
end

//----------------------------------------------------------------------------
//Delay of M22Reg & M21Reg to synchronise with M11Reg
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:Delay
  if (nPhyRst == 1'b0)begin
    M22Reg_1t        <= $signed({21{1'b0}});
    M22              <= $signed({21{1'b0}});
    M21              <= $signed({22{1'b0}});
    GammaReg         <= $signed({6{1'b0}});
    EnableNMatrix    <= 1'b0;
    EnableOut        <= 1'b0;
    EnableDelay      <= 7'b0;
  end
  else begin
    EnableNMatrix    <= EnableIn;
    EnableOut        <= EnableDelay[Nsp];
    EnableDelay[0]   <= EnableIn;
    EnableDelay[6:1] <= EnableDelay[5:0];
    GammaReg         <= GammaIn;
    M22Reg_1t        <= M22Reg;
    M22              <= M22Reg_1t;
    M21              <= M21Reg;
  end
end

//Delay SubCarrierIndex to be aligned with Gamma
DelayLine # (
             .DATAWIDTH(10),    //Width of Enable
             .LATENCY(8)        //Latency
             )
              U_DelayLineIndex_8t (
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .Bypass(1'b0),
                  .DataIn(SubCarrierIndex),
                  .DataOut(SubCarrierIndex_8t)
                  );

DelayLine # (
             .DATAWIDTH(10),    //Width of Enable
             .LATENCY(2)        //Latency
             )
              U_DelayLineIndex_10t (
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .Bypass(1'b0),
                  .DataIn(SubCarrierIndex_8t),
                  .DataOut(SubCarrierIndex_10t)
                  );

DelayLine # (
             .DATAWIDTH(10),    //Width of Enable
             .LATENCY(2)       //Latency
             )
              U_DelayLineIndex_12t (
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .Bypass(1'b0),
                  .DataIn(SubCarrierIndex_10t),
                  .DataOut(SubCarrierIndex_12t)
                  );

// Mux subcarrier index latency according to Nsp
assign SubCarrierIndexMux = (Nsp == 3'd2) ? SubCarrierIndex_8t :
                            (Nsp == 3'd4) ? SubCarrierIndex_10t :
                                            SubCarrierIndex_12t;

// Subcarrier index needed by STOCPECoarse, same delay for all Nsp
assign SubCarrierIndexOut = SubCarrierIndex_12t;

 KMulGamma # (    .INPUTWIDTH(7))
 U_KMulGamma (
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                   //Input Gamma from MNMatrix.
                  .Gamma(GammaMul),
                  .SubCarrierIndex(SubCarrierIndexMux),
                   //Multiplied Output.
                  .KGamma(KGamma)
                  );


//Delay SubCarrierIndex to be aligned with KGamma
DelayLine # (
             .DATAWIDTH(10),    //Width of Enable
             .LATENCY(1)        //Latency
             )
              U_DelayLineIndexOut (
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                  .Bypass(1'b0),
                  .DataIn(SubCarrierIndexMux),
                  .DataOut(SubCarrierIndexK2)
                  );

 KMulGamma # (    .INPUTWIDTH(15))
 U_K2MulGamma (
                  .PhyClk(PhyClk),
                  .nPhyRst(nPhyRst),
                   // Input Gamma & Count from MNMatrix.
                  .Gamma(KGamma),
                  .SubCarrierIndex(SubCarrierIndexK2),
                   //Multiplied Output.
                  .KGamma(KSquGamma)
                  );


endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
