/////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 40550 $
// $Date: 2019-11-28 18:00:40 +0100 (Thu, 28 Nov 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      :STO and CPE Filtering top level module.                
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXFD/FDOffset/verilog/rtl/StoCpeFilter.v $
//
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
module StoCpeFilter (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input    wire                                  nPhyRst, //Active LOW Reset
            input    wire                                  PhyClk,  //PHY Clock

            //Init
            input    wire                                  FDReset,

            //Coarse Angle Valid.
            input    wire                                  EnableInCoarse,

            //Coarse Angle
            input    wire   signed   [17:0]                CpeCoarse,
            input    wire   signed   [17:0]                StoCoarse,

            //Symbol Count
            input    wire            [16:0]                SymCount,
            input    wire            [1:0]                 Mode,
            input    wire            [1:0]                 NFFT,
            input    wire            [2:0]                 Dltf,
`ifdef RW_NX_DERIV_NESS_EN
            input    wire            [2:0]                 Eltf,
`endif
            input    wire                                  InitKalmanRegP,
            input    wire                                  InitKalmanSlopeRegP,

            //Midamble skip slope adjustment
            input   wire                  [3:0]            MidMulSlope,
            input   wire                  [1:0]            MidRShiftSlope,
            input   wire                                   MidSlopeUpdateP,

            //Enable TD Compensation.When 0 sync skip sig is set to 0.
            input    wire                                  CfgStoTdCompEn,

            //Max +ive value of sync signal
            input    wire   signed   [5:0]                 CfgMaxPtdSyncOff,


            //Max -ive value of sync signal
            input    wire   signed   [5:0]                 CfgMaxNtdSyncOff,


            //Enable FD Compensation.When 0 STO O/P is set to 0.
            input    wire                                  CfgStoFdCompEn,

            //Fine or Coarse value is sent to FDO Compensation
            input    wire                                  CfgStoMode,

            //Enable FD compensation,when 0 CPE output is 0.
            input    wire                                  CfgCpeFdCompEn,


            //Coarse or Fine CPE value is sent.1-Fine/0-Coarse
            input    wire                                  CfgCpeMode,

            //Enable  STO slope of Kalman Filter.
            //0-Filled by CfgStoSlopeForced register value.
            input    wire                                  CfgStoSlopeEstEn,


            //STO slope to be forced.
            input    wire   signed   [21:0]                CfgStoSlopeForced,

            //When 1 STO slope register is reinitialized when switching in HT mode.
            input    wire                                  CfgStoSlopeLgHt,

            //Enable  CPE slope of Kalman Filter.
            //0-Filled by CfgCpeSlopeForced register value.
            input    wire                                  CfgCpeSlopeEstEn,

            //CPE slope to be forced.
            input    wire   signed   [21:0]                CfgCpeSlopeForced,

            //When 1 CPE slope register is reinitialized when switching in HT mode.
            input    wire                                  CfgCpeSlopeLgHt,
            
            //1 --> CPE fine used for Coarse Ref Phase else CPE Coarse
            input    wire                                  CfgCpeRef,

            //1 --> STO fine used for Coarse Ref Phase else STO Coarse
            input    wire                                  CfgStoRef,

            //1 -->CPE Wrap depending on coarse or FIne value.
            input    wire                                  CfgCpeWalk,

            //TD Compensation use coarse or fine value.
            input    wire                                  CfgSTO4TDComp,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            // value of synch skip signal at end of the packet.
            output wire signed       [5:0]                 CfgTdSynchOffStat,

            //value of STO slope at end of the packet.
            output wire signed       [21:0]                CfgStoSlopeStat,

            //value of CPE slope at end of the packet.
            output wire signed       [21:0]                CfgCpeSlopeStat,

            //FDO estimation done pulse to fd fsm
            output wire                                    FDOCompDoneP,

            //TD Compensation use coarse or fine value.
            output wire signed       [5:0]                 SynchSkip,

            //STO/CPE to FDO compensation
            output wire signed       [21:0]                STOFine,
            output wire signed       [21:0]                CPEFine,

            //STO/CPE reference to coarse estimation
            output reg  signed       [23:0]                STORefOut,
            output reg  signed       [21:0]                CPERefOut
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
//Max positive val for CPE Wrap.
localparam signed [21:0] CPEWrapPosVal =  22'd65536;
//Min negitive val for CPE Wrap.
localparam signed [21:0] CPEWrapNegVal = -22'd65536;

// Mode
//       LM    = 0
//       VHT   = 1
//       HT-MM = 2
//       HT-GF = 3
localparam [1:0] MODE_LM   = 2'd0,
                 MODE_VHT  = 2'd1,
                 MODE_HTMM = 2'd2,
                 MODE_HTGF = 2'd3;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
//Delayed STO & CPE Fine.
reg            signed              [21:0]           CPEKalmanFilt_1t;
reg            signed              [21:0]           STOKalmanFilt_1t;
//CPE Wrap output
reg            signed              [21:0]           CpeWrap;
//CPE & STO Slope.
reg            signed              [23:0]           CpePsi;
reg            signed              [23:0]           StoPsi;
//Used for generating synch skip.
reg            signed              [5:0]            w;
reg            signed              [1:0]            v;
//Delayed Enable signals.
reg                                                 EnSTOFine_1t;
//Timer used for generating synch skip and STO Wrap.
reg                                [12:0]           Timer;
reg            signed              [6:0]            vwreg;
//Register for STO & CPE wrap.
reg            signed              [22:0]           CpeCoarseWrapReg;
reg            signed              [12:0]           ShftStoWrap_1t;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire           signed              [21:0]           CpeWalk;
wire           signed              [22:0]           ShftCpeWrap;
//Enable output
wire                                                EnOutCPEFine;
wire                                                EnOutSTOFine;
wire           signed              [22:0]           CpeCoarseWrap;
wire           signed              [18:0]           StoCoarseWrap;
wire           signed              [10:0]           RnSTOKalmanFilt_1t_fft64;
wire           signed              [11:0]           RnSTOKalmanFilt_1t_fft128;
wire           signed              [12:0]           RnSTOKalmanFilt_1t_fft256;
wire           signed              [13:0]           RnSTOKalmanFilt_1t_fft512;
wire           signed              [13:0]           Rnd2SatStoKalmanFilt;
wire           signed              [1:0]            SatStoKalmanFilt;
wire           signed              [12:0]           ShftStoWrap;
wire           signed              [21:0]           StoSync;
wire           signed              [6:0]            vw;
wire           signed              [5:0]            Satw;
wire           signed              [22:0]           StoCoarseinKF;
wire           signed              [21:0]           STOSlope;
wire           signed              [21:0]           CPESlope;
wire                               [3:0]            Ltf;
wire                                                InitPsi;
wire           signed              [20:0]           CPEMidSlopeSkipCoarse;
wire           signed              [20:0]           STOMidSlopeSkipCoarse;
wire                                                MidSlopeCoarseUpdateP;

//STO & CPE fine values.
wire           signed              [21:0]           STOKalmanFilt;
wire           signed              [21:0]           CPEKalmanFilt;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//Selection of Coarse or Fine Reference value.
always@(posedge PhyClk or negedge nPhyRst)
begin:CalcCPERefOut
  if (nPhyRst == 1'b0)
    CPERefOut <= 22'b0;
  else if (FDReset || (InitPsi && CfgCpeSlopeLgHt))
    CPERefOut <= 22'b0;
  else if (CfgCpeRef) // Fine estimation as reference
    CPERefOut <= CPEKalmanFilt_1t;
  else                // 2x Coarse estimation as reference
    CPERefOut <= $signed({CpeCoarseWrap[20:0],1'b0}) + $signed({CPEMidSlopeSkipCoarse,1'b0});
end //CalcCPERefOut

always@(posedge PhyClk or negedge nPhyRst)
begin:CalcSTORefOut
  if (nPhyRst == 1'b0)
    STORefOut <= 24'b0;
  else if (FDReset)
    STORefOut <= 24'b0;
  else if (CfgStoRef) // Fine estimation as reference
    STORefOut <= StoPsi;
  else                // 2x Coarse estimation as reference
    STORefOut <= $signed({{5{StoCoarse[17]}},StoCoarse,1'b0}) + $signed({{2{STOMidSlopeSkipCoarse[20]}},STOMidSlopeSkipCoarse,1'b0});
end //CalcSTORefOut

//Calculate sync from coarse if fine is disabled.
assign StoSync   = CfgSTO4TDComp  ? STOKalmanFilt : (StoCoarse <<< 18'sd1);
assign SynchSkip = CfgStoTdCompEn ? w             : $signed(6'b0);

//Assignment of Synch Skip value to Status Register.
assign CfgTdSynchOffStat = SynchSkip;

//Assignment of STO & CPE Slope value to Status Registers.
assign CfgStoSlopeStat = STOSlope;
assign CfgCpeSlopeStat = CPESlope;

assign StoCoarseinKF   = $signed({{5{StoCoarse[17]}},StoCoarse});

//Assign STO & CPE Fine to FDO compensation
assign STOFine  = (CfgStoFdCompEn) ? ((CfgStoMode)?STOKalmanFilt:
                                     (StoCoarse<<<18'sd1)) : 22'b0;
assign CPEFine  = (CfgCpeFdCompEn) ? ((CfgCpeMode)?CPEKalmanFilt:
                                      (CpeCoarseWrap<<<23'b1)) : 22'b0;

//FDO estimation done pulse to fd fsm
assign FDOCompDoneP = (CfgCpeMode) ? EnOutCPEFine:EnableInCoarse;


assign Ltf          = {1'b0,Dltf}
`ifdef RW_NX_DERIV_NESS_EN
                    + {1'b0,Eltf}
`endif
                      ;
 
//----------------------------------------------------------------------------
//CPE FINE ESTIMATION:
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
//Register CPE Signals.
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:RegisterCPEKalmanFilt
  if (nPhyRst == 1'b0)
    CPEKalmanFilt_1t <= 22'b0;
  else if (FDReset)
    CPEKalmanFilt_1t <= 22'b0;
  else
    CPEKalmanFilt_1t <= CPEKalmanFilt;
end //RegisterCPEKalmanFilt

//----------------------------------------------------------------------------
//STEP: 1 CPE Wrap Calculation
//----------------------------------------------------------------------------
// Select fine or coarse estimation for CPE wrapping
assign CpeWalk = (CfgCpeWalk) ? CPEKalmanFilt : $signed({CpeCoarseWrap[20:0],1'b0});

always@(posedge PhyClk or negedge nPhyRst)
begin:CPEWrapping
  if (nPhyRst == 1'b0)
    CpeWrap <= 22'b0;
  else if (FDReset || (InitPsi && CfgCpeSlopeLgHt))
    CpeWrap <= 22'b0;
  else if (EnOutCPEFine || MidSlopeCoarseUpdateP) begin
    if(CpeWalk >= CPEWrapPosVal)
      CpeWrap <= CPEWrapNegVal;
    else if(CpeWalk < CPEWrapNegVal)
      CpeWrap <= CPEWrapPosVal;
    else
      CpeWrap <= 22'b0;
  end
end//CPEWrapping

//----------------------------------------------------------------------------
//Input to Kalman Filter.
//----------------------------------------------------------------------------
assign CpeCoarseWrap = EnableInCoarse ? ($signed({{4{CpeCoarse[17]}},CpeCoarse}) + CpeWrap) : CpeCoarseWrapReg; 

//----------------------------------------------------------------------------
// CPE Wrap Registering
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:RegCpeCoarseWrap
  if (nPhyRst == 1'b0)
    CpeCoarseWrapReg <= 23'b0;
  else if (FDReset)
    CpeCoarseWrapReg <= 23'b0;
  else if (EnableInCoarse)
    CpeCoarseWrapReg <= CpeCoarseWrap;
end//RegCpeCoarseWrap

//----------------------------------------------------------------------------
//STEP: 2 CPE Wrap Shift by 1 bit.
//----------------------------------------------------------------------------
assign ShftCpeWrap = $signed({CpeWrap,1'b0});

//----------------------------------------------------------------------------
//STEP: 3 CPE PSI calculation as an input to Kalman Filter.
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:CalcCPEPsi
  if (nPhyRst == 1'b0)
    CpePsi <= 24'b0;
  else if (FDReset || (InitPsi && CfgCpeSlopeLgHt))
    CpePsi <= 24'b0;
  else
    CpePsi <= $signed({{2{CPEKalmanFilt_1t[21]}},CPEKalmanFilt_1t}) + ShftCpeWrap;
end //CalcCPEPsi

assign InitPsi = InitKalmanRegP & ((Ltf > 4'd1 & Mode == MODE_HTGF) | (Mode != MODE_HTGF));

KalmanFilter  CPE_KalmanFilter(
                 //Inputs
                 .PhyClk(PhyClk),
                 .nPhyRst(nPhyRst),
                 //Init
                 .FDReset(FDReset),
                 .SlopeReset(InitKalmanSlopeRegP),
                 .InitPsi(InitPsi),
                 //Coarse Angle Valid.
                 .EnableInCoarse(EnableInCoarse),
                 //Coarse Angle
                 .ThetaCoarse(CpeCoarseWrap),
                 //Predicted measure
                 .Psi(CpePsi),
                 //Symbol Count
                 .SymCount(SymCount),
                 //Midamble skip slope adjustment
                 .MidMulSlope(MidMulSlope),
                 .MidRShiftSlope(MidRShiftSlope),
                 .MidSlopeUpdateP(MidSlopeUpdateP),
                 //Enable est.of STO slope by Kalman
                 .CfgSlopeEstEn(CfgCpeSlopeEstEn),
                 //STO slope value is forced.
                 .CfgSlopeForced(CfgCpeSlopeForced),
                 //Output slope for coarse update
                 .MidSlopeSkipCoarse(CPEMidSlopeSkipCoarse),
                 //Output Theta with enable
                 .EnableOutFine(EnOutCPEFine),
                 .Slope(CPESlope),
                 .ThetaFine(CPEKalmanFilt)
                 );

//----------------------------------------------------------------------------
//STO FINE ESTIMATION:
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
//Register STO Signals.
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:RegisterSTOCoarse
  if (nPhyRst == 1'b0)begin
    EnSTOFine_1t     <= 1'b0;
    STOKalmanFilt_1t <= 22'b0;
  end
  else if (FDReset) begin
    EnSTOFine_1t     <= 1'b0;
    STOKalmanFilt_1t <= 22'b0;
  end
  else begin
    EnSTOFine_1t     <= EnOutSTOFine;
    STOKalmanFilt_1t <= STOKalmanFilt;
  end
end //RegisterSTOCoarse


//Instantiation of Round component
// In Matlab, division by D=2^11*64/RxParameter.NFFT 
// i.e. D = 2048, 1024, 512, 256 for FFT 64, 128, 256, 512

// Cut 11 bits for FFT64 
Round #(
        .INPUT_WIDTH(22),
        .OUTPUT_WIDTH(11)
       )
        U1_ROUND(
                    .InputData(StoSync),
                    .RoundData(RnSTOKalmanFilt_1t_fft64)
                   );

// Cut 10 bits for FFT128
Round #(
        .INPUT_WIDTH(22),
        .OUTPUT_WIDTH(12)
       )
        U2_ROUND(
                    .InputData(StoSync),
                    .RoundData(RnSTOKalmanFilt_1t_fft128)
                   );

// Cut 9 bits for FFT256
Round #(
       .INPUT_WIDTH(22),
       .OUTPUT_WIDTH(13)
       )
        U3_ROUND(
                    .InputData(StoSync),
                    .RoundData(RnSTOKalmanFilt_1t_fft256)
                   );

// Cut 8 bits for FFT512
Round #(
       .INPUT_WIDTH(22),
       .OUTPUT_WIDTH(14)
       )
        U4_ROUND(
                    .InputData(StoSync),
                    .RoundData(RnSTOKalmanFilt_1t_fft512)
                   );


  assign Rnd2SatStoKalmanFilt = (NFFT == 2'd0) ? {{3{RnSTOKalmanFilt_1t_fft64[10]}},RnSTOKalmanFilt_1t_fft64} :
                                (NFFT == 2'd1) ? {{2{RnSTOKalmanFilt_1t_fft128[11]}},RnSTOKalmanFilt_1t_fft128} :
                                (NFFT == 2'd2) ? {RnSTOKalmanFilt_1t_fft256[12],RnSTOKalmanFilt_1t_fft256} :
                                                  RnSTOKalmanFilt_1t_fft512;

//Instantiation of Saturation component
SatSymSigned #(.INPUT_WIDTH(14),
               .OUTPUT_WIDTH(2)
              )
        U1_SAT(
               .InputData(Rnd2SatStoKalmanFilt),
               .SatSymData(SatStoKalmanFilt) // StoWrap in Matlab
               );

//----------------------------------------------------------------------------
//TD Synchronisation skip computation
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
//STEP: 1 STO Wrap Calculation
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:CalcSTOWrap
  if (nPhyRst == 1'b0)
    v <= 2'b0;
  else if (FDReset)
    v <= 2'b0;
  else if (EnOutSTOFine) begin
    if ((w > -CfgMaxNtdSyncOff) && (w < CfgMaxPtdSyncOff) && (Timer > 13'd9))
      v <= SatStoKalmanFilt;
    else
      v <= 2'b0;
  end
end //CalcSTOWrap

//----------------------------------------------------------------------------
//STEP:2 W calculation
//----------------------------------------------------------------------------
assign vw = (EnSTOFine_1t) ? ($signed({{4{v[1]}},v}) + w) : vwreg;

//----------------------------------------------------------------------------  
//Registering vw Output
//----------------------------------------------------------------------------
always @(posedge PhyClk or negedge nPhyRst)
begin:RegVW
  if (nPhyRst == 1'b0)
    vwreg <= 7'b0;
  else if (EnSTOFine_1t)
    vwreg <= vw;
end

//Instantiation of Saturation component
SatSigned #(.INPUT_WIDTH(7),
            .OUTPUT_WIDTH(6)
            )
      W_SAT(
            .InputData(vw),
            .SatData(Satw)
            );

//----------------------------------------------------------------------------
//Registering W for next symbol.
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:CalcW
  if (nPhyRst == 1'b0)
    w <= 6'b0;
  else if (FDReset)
    w <= 6'b0;
  else if (EnSTOFine_1t) begin
    if (SymCount == 17'd0)
      w <= 6'b0;
    else
      w <= Satw;
  end
end //CalcW

//----------------------------------------------------------------------------
//STEP:3 STO Wrap Shift by n bit. 
//----------------------------------------------------------------------------
assign ShftStoWrap = (NFFT == 2'd0) ? $signed({v,11'b0}) :
                     (NFFT == 2'd1) ? $signed({v[1],v,10'b0}) :
                     (NFFT == 2'd2) ? $signed({v[1],v[1],v,9'b0})  :
                                      $signed({v[1],v[1],v[1],v,8'b0});
assign StoCoarseWrap = $signed({StoCoarse[17],StoCoarse}) - $signed(ShftStoWrap_1t);

always@(posedge PhyClk or negedge nPhyRst)
begin:STOWrapDelay
  if (nPhyRst == 1'b0)
    ShftStoWrap_1t <= 13'b0;
  else if (FDReset)
    ShftStoWrap_1t <= 13'b0;
  else if (EnOutSTOFine)
    ShftStoWrap_1t <= ShftStoWrap;
end

//----------------------------------------------------------------------------
//STEP: 4 STO PSI calculation as an input to Kalman Filter.
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:CalcSTOPsi
  if (nPhyRst == 1'b0)
    StoPsi <= 24'b0;
  else if (FDReset || (InitPsi && CfgStoSlopeLgHt))
    StoPsi <= 24'b0;
  else
    StoPsi <= $signed({{2{STOKalmanFilt_1t[21]}},STOKalmanFilt_1t}) - $signed(ShftStoWrap_1t);
end //CalcSTOPsi


KalmanFilterSTO STO_KalmanFilter(
                 //Inputs
                 .PhyClk(PhyClk),
                 .nPhyRst(nPhyRst),
                 //Init
                 .FDReset(FDReset),
                 .SlopeReset(InitKalmanSlopeRegP),
                 .InitPsi(InitPsi),
                 //Coarse Angle Valid.
                 .EnableInCoarse(EnableInCoarse),
                 //Coarse Angle
                 .ThetaCoarse(StoCoarseinKF),
                 //Predicted measure
                 .Psi(StoPsi),
                 //Symbol Count
                 .SymCount(SymCount),
                 //Midamble skip slope adjustment
                 .MidMulSlope(MidMulSlope),
                 .MidRShiftSlope(MidRShiftSlope),
                 .MidSlopeUpdateP(MidSlopeUpdateP),
                 //Enable est.of STO slope by Kalman
                 .CfgSlopeEstEn(CfgStoSlopeEstEn),
                 //STO slope value is forced.
                 .CfgSlopeForced(CfgStoSlopeForced),
                 //slope for STO coarse update during midamble skip
                 .MidSlopeSkipCoarse(STOMidSlopeSkipCoarse),
                 .MidSlopeCoarseUpdateP(MidSlopeCoarseUpdateP),
                 //Output Theta with enable
                 .EnableOutFine(EnOutSTOFine),
                 .Slope(STOSlope),
                 .ThetaFine(STOKalmanFilt)
                 );

//----------------------------------------------------------------------------
//Timer
//Resets at SymCount 10
//----------------------------------------------------------------------------
always@(posedge PhyClk or negedge nPhyRst)
begin:TimerControl
  if (nPhyRst == 1'b0)begin
    Timer <= 13'd0;
  end
  else if (FDReset || InitPsi) begin
    Timer <= 13'd0;
  end
  else if (EnSTOFine_1t) begin
    if (SymCount == 17'd0)
      Timer <= 13'd1;
    else  if (Satw != w && Timer > 13'd9)
      Timer <= 13'd0;
    else
      Timer <= Timer + 13'd1;
  end
end //TimerControl

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
