//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission 
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 38380 $
// $Date: 2019-04-10 11:13:39 +0200 (Wed, 10 Apr 2019) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Noise Variance conversion to linear for Equaliser
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module RxFDNoiseVarLin (
            ///////////////////////////////////////////////
            // Clock and Reset
            ///////////////////////////////////////////////
            input  wire               nPhyRst, // Active LOW Reset
            input  wire               PhyClk,

            ///////////////////////////////////////////////
            // Noise variance adjustement from Registers
            ///////////////////////////////////////////////
            input  wire        [7:0]  CfgRegNoiseVarAdjustdB,  

            ///////////////////////////////////////////////
            // Noise variance in dB
            ///////////////////////////////////////////////
            input  wire signed [7:0]  NoiseVardB,

            ///////////////////////////////////////////////
            // Noise variance in linear
            ///////////////////////////////////////////////
            output reg         [24:0] SigmaSquared
            );

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire               [7:0]               SigmadBPos;
wire    signed     [6:0]               nR_lin_shift;
wire               [24:0]              SigmaSquaredRnd;

//////////////////////////////////////////////////////////////////////////////
//  Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed     [7:0]               NoiseVardB_1t;
reg     signed     [8:0]               SigmadB;
reg     signed     [5:0]               n2_bit;
reg                [8:0]               nR_LUT_sel;
reg                [11:0]              nR_lin;
reg                [25:0]              SigmaSquaredPreRnd;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// This module computes the Noise Variance conversion in dB to linear

//Capture noise variance from AGC
always @ (posedge PhyClk or negedge nPhyRst)
   begin: NoiseVardB_Blk
      if (nPhyRst == 1'b0)
         NoiseVardB_1t <= 8'sd0;
      else
         NoiseVardB_1t <= NoiseVardB;
   end //NoiseVardB_Blk

//Noise variance adjustement
always @ (posedge PhyClk or negedge nPhyRst)
   begin: SigmadB_Blk
      if (nPhyRst == 1'b0)
         SigmadB <= 9'sd0;
      else
         SigmadB <= $signed({NoiseVardB_1t[7],NoiseVardB_1t}) + 
                    $signed({CfgRegNoiseVarAdjustdB[7],CfgRegNoiseVarAdjustdB});
   end //SigmadB_Blk

assign SigmadBPos = (SigmadB[8]) ? 8'b0 : SigmadB[7:0];

//Use the above value as the index to a LUT
always @ *
   begin: n2_bit_Blk
      case (SigmadBPos) // n2_bit
           8'd00,8'd01,8'd02    : n2_bit = 6'd00;
           8'd03,8'd04,8'd05    : n2_bit = 6'd01;
           8'd06,8'd07,8'd08    : n2_bit = 6'd02;
           8'd09,8'd10,8'd11    : n2_bit = 6'd03;
           8'd12,8'd13,8'd14    : n2_bit = 6'd04;
           8'd15,8'd16,8'd17    : n2_bit = 6'd05;
           8'd18,8'd19,8'd20    : n2_bit = 6'd06;
           8'd21,8'd22,8'd23    : n2_bit = 6'd07;
           8'd24,8'd25,8'd26    : n2_bit = 6'd08;
           8'd27,8'd28,8'd29    : n2_bit = 6'd09;
           8'd30,8'd31,8'd32    : n2_bit = 6'd10;
           8'd33,8'd34,8'd35    : n2_bit = 6'd11;
           8'd36,8'd37,8'd38    : n2_bit = 6'd12;
           8'd39,8'd40,8'd41    : n2_bit = 6'd13;
           8'd42,8'd43,8'd44    : n2_bit = 6'd14;
           8'd45,8'd46,8'd47    : n2_bit = 6'd15;
           8'd48,8'd49,8'd50    : n2_bit = 6'd16;
           8'd51,8'd52,8'd53    : n2_bit = 6'd17;
           8'd54,8'd55,8'd56    : n2_bit = 6'd18;
           8'd57,8'd58,8'd59    : n2_bit = 6'd19;
           8'd60,8'd61,8'd62    : n2_bit = 6'd20;
           8'd63,8'd64,8'd65    : n2_bit = 6'd21;
           8'd66,8'd67,8'd68    : n2_bit = 6'd22;
           8'd69,8'd70,8'd71    : n2_bit = 6'd23;
           8'd72,8'd73,8'd74    : n2_bit = 6'd24;
           8'd75,8'd76,8'd77    : n2_bit = 6'd25;
           8'd78,8'd79,8'd80    : n2_bit = 6'd26;
           8'd81,8'd82,8'd83    : n2_bit = 6'd27;
           8'd84,8'd85,8'd86    : n2_bit = 6'd28;
           8'd87,8'd88,8'd89    : n2_bit = 6'd29;
           8'd90,8'd91,8'd92    : n2_bit = 6'd30;
           8'd93,8'd94,8'd95    : n2_bit = 6'd31;
           8'd96,8'd97,8'd98    : n2_bit = 6'd32;
           8'd99,8'd100,8'd101  : n2_bit = 6'd33;
           8'd102,8'd103,8'd104 : n2_bit = 6'd34;
           8'd105,8'd106,8'd107 : n2_bit = 6'd35;
           8'd108,8'd109,8'd110 : n2_bit = 6'd36;
           8'd111,8'd112,8'd113 : n2_bit = 6'd37;
           8'd114,8'd115,8'd116 : n2_bit = 6'd38;
           8'd117,8'd118,8'd119 : n2_bit = 6'd39;
           8'd120,8'd121,8'd122 : n2_bit = 6'd40;
           8'd123,8'd124,8'd125 : n2_bit = 6'd41;
         default                : n2_bit = 6'd42;
      endcase //n2_LUT_sel
   end //n2_bit_Blk

//Multiply n2_bit by 3 and subtract it from SigmadB
always @ (posedge PhyClk or negedge nPhyRst)
   begin: nR_LUT_sel_Blk
      if (nPhyRst == 1'b0)
         nR_LUT_sel <= 9'sd0;
      else
         nR_LUT_sel <= $signed({1'b0,SigmadBPos}) - $signed({1'b0,n2_bit,1'b0}) - $signed({2'b0,n2_bit});
   end //nR_LUT_sel_Blk

//Use this as the index of an LUT
always @ (*)
   begin: nR_lin_Blk
      case (nR_LUT_sel)
            9'd0 : nR_lin = 12'd2048;
            9'd1 : nR_lin = 12'd2578;
            9'd2 : nR_lin = 12'd3246;
         default : nR_lin = 12'd4086;
      endcase //nR_LUT_sel
   end //nR_lin_Blk

//Compute shift indice of nR_lin
assign nR_lin_shift = $signed({1'b0,n2_bit}) - 7'sd11;

//Compute Sigma shift
always @ (posedge PhyClk or negedge nPhyRst)
   begin: SigmaShift_Blk
      if (nPhyRst == 1'b0)
        SigmaSquaredPreRnd <= 26'b0;
      else if (nR_lin_shift >= 6'sd0)
      begin
        //Left shift nR_lin by nR_lin_shift bits
        case (nR_lin_shift)
             7'sd00 : SigmaSquaredPreRnd <= {13'b0,nR_lin,1'b0};
             7'sd01 : SigmaSquaredPreRnd <= {12'b0,nR_lin,2'b0};
             7'sd02 : SigmaSquaredPreRnd <= {11'b0,nR_lin,3'b0};
             7'sd03 : SigmaSquaredPreRnd <= {10'b0,nR_lin,4'b0};
             7'sd04 : SigmaSquaredPreRnd <= {9'b0,nR_lin,5'b0};
             7'sd05 : SigmaSquaredPreRnd <= {8'b0,nR_lin,6'b0};
             7'sd06 : SigmaSquaredPreRnd <= {7'b0,nR_lin,7'b0};
             7'sd07 : SigmaSquaredPreRnd <= {6'b0,nR_lin,8'b0};
             7'sd08 : SigmaSquaredPreRnd <= {5'b0,nR_lin,9'b0};
             7'sd09 : SigmaSquaredPreRnd <= {4'b0,nR_lin,10'b0};
             7'sd10 : SigmaSquaredPreRnd <= {3'b0,nR_lin,11'b0};
             7'sd11 : SigmaSquaredPreRnd <= {2'b0,nR_lin,12'b0};
             7'sd12 : SigmaSquaredPreRnd <= {1'b0,nR_lin,13'b0};
            default : SigmaSquaredPreRnd <= {nR_lin,14'b0};
        endcase //nR_lin_shift
      end
      else
      begin
        //Right shift nR_lin by nR_lin_shift bits
        case (nR_lin_shift)
            -7'sd01 : SigmaSquaredPreRnd <= {14'b0,nR_lin[11:0]};
            -7'sd02 : SigmaSquaredPreRnd <= {15'b0,nR_lin[11:1]};
            -7'sd03 : SigmaSquaredPreRnd <= {16'b0,nR_lin[11:2]};
            -7'sd04 : SigmaSquaredPreRnd <= {17'b0,nR_lin[11:3]};
            -7'sd05 : SigmaSquaredPreRnd <= {18'b0,nR_lin[11:4]};
            -7'sd06 : SigmaSquaredPreRnd <= {19'b0,nR_lin[11:5]};
            -7'sd07 : SigmaSquaredPreRnd <= {20'b0,nR_lin[11:6]};
            -7'sd08 : SigmaSquaredPreRnd <= {21'b0,nR_lin[11:7]};
            -7'sd09 : SigmaSquaredPreRnd <= {22'b0,nR_lin[11:8]};
            -7'sd10 : SigmaSquaredPreRnd <= {23'b0,nR_lin[11:9]};
            -7'sd11 : SigmaSquaredPreRnd <= {24'b0,nR_lin[11:10]};
            -7'sd12 : SigmaSquaredPreRnd <= {25'b0,nR_lin[11]};
           default  : SigmaSquaredPreRnd <= 26'b0;
        endcase //nR_lin_shift
      end
   end //SigmaShift_Blk

// Sigma rounding
USgnRound #(.INPUT_WIDTH(26),
        .OUTPUT_WIDTH(25))
U_USgnRound_SigmaSquared (
        .InputData(SigmaSquaredPreRnd),
        .RoundData (SigmaSquaredRnd)
        );

//Find the final Sigma output after rounding
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Sigma_Blk
      if (nPhyRst == 1'b0)
         SigmaSquared <= 25'b0;
      else if (SigmadBPos < 8'd128)
         SigmaSquared <= SigmaSquaredRnd;
      else
         SigmaSquared <= 25'h1000000; // 2^24
   end //Sigma_Blk

endmodule //NoiseVarEst

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
