/*******************************************************************************
* copyright (C) by RivieraWaves Ltd.This module is a confidential and proprietary
* property of RivieraWaves Ltd. and a possession or use of this module requires  
*  written  permission from RivieraWaves Ltd.                                     
********************************************************************************
* $Author: cvandebu $
* Company          : RivieraWaves
********************************************************************************
* $Revision: 17391 $
* $Date: 2014-12-12 16:46:27 +0100 (Fri, 12 Dec 2014) $
********************************************************************************
* Dependencies     :       
* Description      : This module is the RX Freq Domain FSM, which controls
*                    Equaliser, Channel estimation and Freq Domain Off
*                    modules
*                    
* Simulation Notes :       
* Synthesis Notes  :       
* Application Note :       
* Simulator        :       
* Parameters       :  
* Terms & concepts :       
* Bugs             :       
* Open issues and future enhancements :                                    
* References       :       
* Revision History :       
*******************************************************************************/
`default_nettype none
module RxFreqDomainStMc
(
  /*****************************************************************************
  * system 
  *****************************************************************************/ 
  input  wire           PhyClk,       
  input  wire           nPhyRst,      
  
  /*****************************************************************************
  * gated clock enables
  *****************************************************************************/ 
  output wire           ChEstGClkEn,        
  output wire           FDOGClkEn,          
  output wire           EquGClkEn,          
       
  /*****************************************************************************
  * Global Control           
  *****************************************************************************/ 
  input  wire           FDEnable,
  output reg            FDDone,                          

  /*****************************************************************************
  * configuration registers        
  *****************************************************************************/ 
  input  wire [ 1:0]    CfgRegNrx,      
  input  wire           CfgRegMUSmoothForce,
  input  wire           CfgRegBFSmoothForce,
  input  wire [ 4:0]    CfgRegSTOCompTime,
  input  wire           CfgRegForce20,
  input  wire [ 2:0]    CfgRegPrimary,
  input  wire           CfgRegSkipMidamble1x,    // Skip midamble HE-LTF 1x
  input  wire           CfgRegSkipMidamble2x,    // Skip midamble HE-LTF 2x
  input  wire           CfgRegSkipMidamble4x,    // Skip midamble HE-LTF 4x
  
  /*****************************************************************************
  * frame parameters
  *****************************************************************************/
  input  wire           FrameParam1Update,
  input  wire [ 1:0]    FrameParam1NSD,
  input  wire [ 2:0]    FrameParam1NBPSC0,
  input  wire [ 2:0]    FrameParam1NBPSC1,
  input  wire [ 2:0]    FrameParam1NSS,
  input  wire [ 2:0]    FrameParam1NSTS,
`ifdef RW_NX_DERIV_NESS_EN
  input  wire [ 1:0]    FrameParam1NESS,
`endif
  input  wire           FrameParam1STBC,
  input  wire           FrameParam1FEC,
  input  wire           FrameParam1Smoothing,
  input  wire           FrameParam1HTNDP,
`ifdef RW_MUMIMO_RX_EN
  input  wire           FrameParam1MuMIMO,
  input  wire [ 2:0]    FrameParam1MuNSTS,
  input  wire [ 2:0]    FrameParam1MuPreUserNSTS,
  input  wire           FrameParam1MuFEC,
  input  wire [ 1:0]    FrameParam1MuPosID,
`endif // RW_MUMIMO_RX_EN
  input  wire [ 2:0]    FrameParam1NHeltf,
  input  wire [ 2:0]    FrameParam1NBPSCHeSigB,
  input  wire [ 1:0]    FrameParam1CBW,
  input  wire [ 1:0]    FrameParam1HeltfType, // 0-3.2us / 1-6.4us / 2-12.8us
  input  wire           FrameParam1Doppler,
  input  wire           FrameParam1Midamble,
  input  wire           FrameParam1DCM,
  input  wire [ 2:0]    FrameParam1HESIGBMCS,
  input  wire           FrameParam1HESIGBDCM,
  
  input  wire           FrameParam2Update,
  input  wire [ 2:0]    FrameParam2MuNBPSC,
  input  wire [ 2:0]    FrameParam2RUType,    // Type of RU (0:26, 1:52, 2:106, 3:242, 4:484)
  input  wire [ 5:0]    FrameParam2RUIndex,   // RU allocation index (up to 37)
  input  wire           FrameParam2HEMUFEC,
  input  wire           FrameParam2HEMUDCM,
`ifdef RW_MUMIMO_RX_EN
  input  wire [ 2:0]    FrameParam2HEMUNSTS,
  input  wire [ 2:0]    FrameParam2HEMUNSTSTot,
  input  wire [ 2:0]    FrameParam2HEMUPreUserNSTS,
  input  wire [ 3:0]    FrameParam2HEMURUNUser,
  input  wire [ 2:0]    FrameParam2HEMURUUser,
`endif // RW_MUMIMO_RX_EN
 
  input  wire           FrameParam3RLSIG,
  input  wire           FrameParam3Update,

  input  wire           FrameParam4Update,
  input  wire           FrameParam4LSIG6M,

  input  wire           FrameParam6HeSu,      // If frame is HE, frame is HE-SU or HE-TB
  input  wire           FrameParam6Update,

  input  wire           FrameParam7Update,
  input  wire [ 7:0]    FrameParam7NHeSigB,
  
  input  wire           FrameParam8Update,
  input  wire [15:0]    FrameParam8NSym,
  input  wire [ 8:0]    FrameParam8NMA,

  /*****************************************************************************
  * Buffer management      
  *****************************************************************************/
  /* TD-FD */
  input  wire [1:0]     FFTBufPtr,
  output wire [1:0]     FDFFTBufRdPtr,
  
  /* FD-BD data pointer */
  input  wire           bdfd_rdptr,
  output reg            bdfd_wrptr,
  output reg            bdfd_qbpsk,
  
  /* FD-BD HESIGB pointer */
  input  wire [2:0]     bdfd_ch0_rdptr,
  output reg  [2:0]     bdfd_ch0_wrptr,
  input  wire [2:0]     bdfd_ch1_rdptr,
  output reg  [2:0]     bdfd_ch1_wrptr,
  
  /*****************************************************************************
  * BD
  *****************************************************************************/
  input  wire           BDDone,

  /*****************************************************************************
  * FD
  *****************************************************************************/
  output wire           FDReset,
`ifdef RW_NX_DERIV_NESS_EN
  output reg    [2:0]   FDNELTF,
`endif
  output reg    [3:0]   FDNDLTF,                                               
  output reg            FD2DEn,   // High when processing secondary 20 MHz band
  output reg    [9:0]   FDHEMUOffset,
`ifdef RW_MUMIMO_RX_EN
  output reg    [1:0]   HMuMIMOSel,
`endif // RW_MUMIMO_RX_EN

  /*****************************************************************************
  * TD
  *****************************************************************************/
  output reg    [3:0]   TDFormatMod,
  
  /*****************************************************************************
  * Channel Estimate       
  *****************************************************************************/
  input  wire           ChEstDoneP,
  input  wire           ChEstSmthDoneP,
  //
  output wire           ChEstStartP,  
  output reg    [3:0]   ChEstReceptionMode,                                           
  output reg    [2:0]   ChEstRxNsts,  
  output reg    [3:0]   ChEstNShiftLTF,
`ifdef RW_NX_DERIV_NESS_EN
  output reg    [1:0]   ChEstRxNess,  
`endif
  output wire           ChEstSmoothEn,
  output wire           ChEstSmoothFromH,                                             
  output reg            ChEstFirstLTF,
  output reg            ChEstLastLTF, 
  output reg    [2:0]   ChEstSymbolType,
  output reg    [2:0]   ChEstRUType,
  output reg    [5:0]   ChEstRUIndex,
  output reg            ChEstFrameNDP,
`ifdef RW_MUMIMO_RX_EN
  output reg            ChEstHEMuMIMOEn,
`endif // RW_MUMIMO_RX_EN

  /*****************************************************************************
  * Equalizer              
  *****************************************************************************/
  input  wire           EqQbpskDet,   
  input  wire           EqQbpskDetDoneP,   
  input  wire           EqPipeFlushDoneP,                                             
  input  wire           EqDoneP,                                             
  input  wire           EqPilotDoneP, 
  input  wire           EqCompDoneP,
  //
  output wire           EqStartP,     
  output wire           EqCompStartP, 
  output reg    [3:0]   EqReceptionMode,                                              
  output reg            EqStride,
  output reg            EqHeSigBPAPREn,
  output reg            EqMHUpdate, // Update MH when eqMod is modified for SIGB
`ifdef RW_MUMIMO_RX_EN
  output reg            EqMuMIMOEn,
  output reg            EqVHTSIGBNss2,
  output reg            EqVHTSIGBSub,
  output reg            EqVHTSIGBInv,
  output reg            EqMuMIMOICEn,
`endif // RW_MUMIMO_RX_EN
  output reg            EqSTBCEn,     
  output reg            EqSTBCSymOdd,     
  output reg            EqSTBCFlush,     
  input  wire           EqSTBCFlushDone,     
  output reg            EqLDPCEn,     
  output reg            EqDCMEn,     
  output reg            EqPilotData,  
  output reg    [2:0]   EqRxNsts,     
  output reg    [2:0]   EqRxNss,      
  output reg            EqDiscEn,     
  output reg    [2:0]   EqModTypeSS1, 
  output wire   [2:0]   EqRUType,
  output wire   [5:0]   EqRUIndex,
// STBC or 2x2  output reg    [2:0]   EqModTypeSS2, 
       
  /*****************************************************************************
  * FDO
  *****************************************************************************/
  input  wire           FDOCompDoneP,
  //
  output wire           FDOEstStartP,
  output reg   [ 1:0]   FDOMode,    
  output wire  [ 3:0]   FDOReceptionMode,                                           
  output reg   [ 2:0]   FDORUType,    
  output wire           FDOInitPilotShP,                                         
  output wire           FDOInitKalmanP,
  output wire           FDOInitKalmanSlopeP,
  output reg   [ 1:0]   FDOBW,                                          
  output reg   [16:0]   FDOSymCount,
  output reg            FDOSTOFineMuxSel,
  output reg            FDORxNsts,                                         
  output reg            FDOStorePreamb,
  output reg            FDOStoreData,
  output reg   [ 3:0]   FDOMidMulSlope,
  output reg   [ 1:0]   FDOMidRShiftSlope,

  /*****************************************************************************
  * Deinterleaver
  *****************************************************************************/
  output wire           deintlv_start,
  input  wire           deintlv_busy,
  
  output reg            deintlv_row_fifo_en,
  output reg            deintlv_chsel,
  
  output reg    [1:0]   deintlv_iss,
  output reg    [2:0]   deintlv_nbpsc,
  output reg            deintlv_fec,
  output reg            deintlv_dcm,
  output reg    [1:0]   deintlv_nsd,
  output reg            deintlv_ruen,
  output reg    [2:0]   deintlv_rulen,
  output reg            deintlv_sigen,
  output reg    [1:0]   deintlv_sigsel,
 
  /*****************************************************************************
  * Diag
  *****************************************************************************/
  output wire  [31:0]   Diag
);   

  // Encoding of ModTypeSS# signals for Equalizer
  localparam [2:0] BPSK    = 3'd0;
  localparam [2:0] QBPSK   = 3'd1;
  localparam [2:0] QPSK    = 3'd2;
  localparam [2:0] QAM16   = 3'd3;
  localparam [2:0] QAM64   = 3'd4;
  localparam [2:0] QAM256  = 3'd5;
  localparam [2:0] QAM1024 = 3'd6;

  localparam  NSD_48=2'd0,NSD_52=2'd1,NSD_108=2'd2,NSD_234=2'd3;
  localparam  NBPSC_1=3'd0,NBPSC_2=3'd1,NBPSC_4=3'd2,NBPSC_6=3'd3,NBPSC_8=3'd4,NBPSC_10=3'd5;
  localparam  NSS_1=3'd0,NSS_2=3'd1;

  /* symbol state */
  localparam FD_RESET=5'd0,
             LLTF=5'd1,
             LSIG_HELSIG=5'd2,
             LDATA0_HTMMSIG1_VHTSIGA1_HERLSIG=5'd3,
             LDATA=5'd4,   
             HTSIG2=5'd5,
             HTDLTF=5'd6,
`ifdef RW_NX_DERIV_NESS_EN
             HTELTF=5'd7,
`endif
             HTDATA=5'd8,
             LDATA1_VHTSIGA2=5'd9,
             VHTLTF=5'd10,
             VHTSIGB=5'd11,
             VHTDATA=5'd12,
             FD_DONE=5'd13,
             HESIGA=5'd14, 
             HELTF=5'd15, 
             HESIGB=5'd16, 
             HEDATA=5'd17;

  /* step state */
  localparam WAIT_FFT=4'd0,
             ESTIM_H1=4'd1,
             EQU_PILOT=4'd2,
             EQU_DATA=4'd3,
             WAIT_BD=4'd4,
             SMOOTH_H1=4'd5,
             ESTIM_H2=4'd6,
             SMOOTH_H2=4'd7,
             WAIT_STBC_FLUSH=4'd8;
             
  // Reception Mode 
  localparam [3:0] MODE_20MHZ_L     = 4'd0,
                   MODE_40MHZ_LDUP  = 4'd1,
                   MODE_20MHZ_HT    = 4'd2,
                   MODE_40MHZ_HT    = 4'd3,
                   MODE_80MHZ_HT_8PILOTS = 4'd4, // not used
                   MODE_80MHZ_HT    = 4'd5,
                   MODE_20MHZ_HE_SU = 4'd6,
                   MODE_20MHZ_HE_MU = 4'd7,
                   MODE_40MHZ_HE_SU = 4'd8,
                   MODE_40MHZ_HE_MU = 4'd9,
                   MODE_80MHZ_HE_SU = 4'd10,
                   MODE_80MHZ_HE_MU = 4'd11;
                   
  // FDO Mode 
  localparam [1:0] FDO_MODE_LM   = 2'd0, // bypass pilot init
                   FDO_MODE_VHT  = 2'd1,
                   FDO_MODE_HTMM = 2'd2,
                   FDO_MODE_HTGF = 2'd3; // not used - bypass psi init on first LTF

  // Symbol type for ChEst
  localparam [2:0] L_LTF    = 3'd0,
                   D_VHTLTF = 3'd1,
                   D_HTLTF  = 3'd2,
                   E_HTLTF  = 3'd3,
                   HE_1xLTF = 3'd4,
                   HE_2xLTF = 3'd5,
                   HE_4xLTF = 3'd6,
                   HE_LSIG  = 3'd7;

  // FormatMod (to Time Domain FSM)
  localparam FORMATMOD_LEG   = 4'd0,
             FORMATMOD_DUP   = 4'd1,
             FORMATMOD_MM    = 4'd2,
             FORMATMOD_GF    = 4'd3, // not used
             FORMATMOD_VHT   = 4'd4,
             FORMATMOD_HE_SU = 4'd5, // must be set with HE-SIGA
             FORMATMOD_HE_MU = 4'd6,
             FORMATMOD_HE_ER = 4'd7,
             FORMATMOD_HE_TB = 4'd8;

  localparam RU_26=3'd0,
             RU_52=3'd1,
             RU_106=3'd2,
             RU_242=3'd3,
             RU_484=3'd4,
             RU_996=3'd5;

  //////////////////////////////////////////////////////////////////////////////
  // Internal Registers Declarations
  //////////////////////////////////////////////////////////////////////////////
  reg [ 4:0]  ChEstClkEnSR;
  reg [ 4:0]  EqClkEnSR;
  reg [ 4:0]  FDOClkEnSR;
  reg [ 4:0]  ChEstSR;
  reg [ 4:0]  EqCompSR;
  reg [ 4:0]  EqSR;
  reg [ 4:0]  FDOEstSR;
  reg [ 4:0]  DeIntlvSR;
  reg [ 4:0]  FDOInitPilotSR; 
  reg [ 4:0]  FDOInitKalmanSR;    
  reg [ 4:0]  ChEstSmoothEnSR;    
  reg [ 4:0]  ChEstSmoothFromHSR;    
  reg [ 4:0]  FDResetSR;    
  reg [ 3:0]  FDFormatMod;

  reg [16:0]  FDSymCount;     // Counter for data symbol
  reg [ 4:0]  FDSymState;    
  reg [ 3:0]  FDStepState;
  wire        LastSymbol;    

  wire        FFTBufEmpty;   
  wire [16:0] n_FDSymCount;  
  wire [16:0] n_FDOSymCount; 
  
  wire        bdfd_fifo_full;
  wire        bdfd_ch0_fifo_full;
  wire        bdfd_ch1_fifo_full;
 
  wire [ 2:0] n_bdfd_ch0_wrptr;
  wire [ 2:0] n_bdfd_ch1_wrptr;

  reg         HESigACount;
  reg  [ 7:0] HESigBSymCount;
  wire [ 7:0] n_HESigBSymCount;
  wire        LastHeSigBSymbol;

  reg  [ 3:0] LTFCount;
  wire [ 3:0] n_LTFCount;

  wire        FFTFifoFull;
  
  reg  [ 1:0] FDFFTBufRdPtrInt;

  wire [ 2:0] FrameParam1NSTSMux;
  wire        FrameParam1VHTFEC;
  wire [ 2:0] FrameParam1VHTNSS;
  wire        FrameParam1VHTSmoothing;

  reg  [4:0]  MidambleSymbolCount;
  reg         MidambleSymbol;
  reg         MidambleSymbol_1t;
  reg         MidambleSymbolFDO;
  reg  [9:0]  NMACount;
  reg         MidambleSkipFFT;
  wire [4:0]  MidambleSymbolCountLimit;
  wire        MidambleSymbolMask;
  
  reg         equ_flushing_state;
  
  //////////////////////////////////////////////////////////////////////////////
  // Begining of Logic part
  //////////////////////////////////////////////////////////////////////////////
  /* BDFD */
  assign bdfd_fifo_full     = bdfd_rdptr!=bdfd_wrptr;
  assign bdfd_ch0_fifo_full = bdfd_ch0_rdptr == {~bdfd_ch0_wrptr[2],bdfd_ch0_wrptr[1:0]};
  assign bdfd_ch1_fifo_full = bdfd_ch1_rdptr == {~bdfd_ch1_wrptr[2],bdfd_ch1_wrptr[1:0]};
 
  /* FFT */
  assign FFTFifoFull  = FDFFTBufRdPtrInt=={~FFTBufPtr[1],FFTBufPtr[0]};

  /* TODO: unify equalizer parameters with standard NBPSC */
  wire   [2:0] ModTypeLUT[0:7];
  assign ModTypeLUT[0] = BPSK;    // NBPSC_1
  assign ModTypeLUT[1] = QPSK;    // NBPSC_2
  assign ModTypeLUT[2] = QAM16;   // NBPSC_4
  assign ModTypeLUT[3] = QAM64;   // NBPSC_6
  assign ModTypeLUT[4] = QAM256;  // NBPSC_8
  assign ModTypeLUT[5] = QAM1024; // NBPSC_10
  assign ModTypeLUT[6] = 3'b0;
  assign ModTypeLUT[7] = 3'b0;

  assign LastHeSigBSymbol = (FrameParam7Update & (FrameParam7NHeSigB==n_HESigBSymCount));
  assign LastSymbol       = (FrameParam8Update & ({1'b0,FrameParam8NSym}==n_FDSymCount));

`ifdef RW_NX_DERIV_NESS_EN
  reg  [ 2:0] n_FDNELTF;
  always @(*)
    case(FrameParam1NESS)
         2'd0: n_FDNELTF = 3'd0;
         2'd1: n_FDNELTF = 3'd1;
         2'd2: n_FDNELTF = 3'd2;
      default: n_FDNELTF = 3'd4;
    endcase
`endif
 
  reg  [ 3:0] n_FDNDLTF;
  always @(*)
  begin
    case(FrameParam1NSTSMux)
      3'd0:      n_FDNDLTF = 4'd1;
      3'd1:      n_FDNDLTF = 4'd2;
      3'd2,3'd3: n_FDNDLTF = 4'd4;
      3'd4,3'd5: n_FDNDLTF = 4'd6;
      default:   n_FDNDLTF = 4'd8;
    endcase
  end

  reg [3:0] FDNHELTF;
  always @(posedge PhyClk,negedge nPhyRst)
  begin
    if(!nPhyRst)
      FDNHELTF <= 4'd0;
    else if(!FDEnable)
      FDNHELTF <= 4'd0;
    else if (FrameParam1Update)
    begin
      case(FrameParam1NHeltf)
        3'd0:    FDNHELTF <= 4'd1;
        3'd1:    FDNHELTF <= 4'd2;
        3'd3:    FDNHELTF <= 4'd4;
        3'd5:    FDNHELTF <= 4'd6;
        default: FDNHELTF <= 4'd8;
      endcase
    end
  end
  
  always @(posedge PhyClk,negedge nPhyRst)
  begin
    if(!nPhyRst)
      ChEstNShiftLTF <= 4'd0;
    else if(!FDEnable)
      ChEstNShiftLTF <= 4'd0;
    else if (FrameParam1Update)
    begin
      if (FrameParam1STBC)
      begin
        if (FDFormatMod == FORMATMOD_HE_MU)
        begin
          case(FrameParam1NHeltf)
            3'd3:    ChEstNShiftLTF <= 4'd3; // Estimates on HELTF 3 and 4
            3'd5:    ChEstNShiftLTF <= 4'd5; // Estimates on HELTF 5 and 6
            3'd7:    ChEstNShiftLTF <= 4'd7; // Estimates on HELTF 7 and 8
            default: ChEstNShiftLTF <= 4'd1; // Estimates on HELTF 1 and 2
          endcase
        end
        else
          ChEstNShiftLTF <= 4'd1;
      end
      else
      begin
        if (FDFormatMod == FORMATMOD_HE_SU || FDFormatMod == FORMATMOD_HE_MU)
          ChEstNShiftLTF <= FDNHELTF;
        else
          ChEstNShiftLTF <= FDNDLTF;
      end
    end
  end

  // Sample FEC value for HE depending on SU/MU
  reg FrameParamHEFEC;
  always @(posedge PhyClk,negedge nPhyRst)
  begin
    if(!nPhyRst)
      FrameParamHEFEC <= 1'd0;
    else if(!FDEnable)
      FrameParamHEFEC <= 1'd0;
    else if (FrameParam6Update && FrameParam6HeSu && FrameParam1Update)
      FrameParamHEFEC <= FrameParam1FEC;
    else if (FrameParam6Update && !FrameParam6HeSu && FrameParam2Update)
      FrameParamHEFEC <= FrameParam2HEMUFEC;
  end

  /******************************************************************************
  * FDO specific Control
  ******************************************************************************/
  always @(posedge PhyClk,negedge nPhyRst)
  begin
    if(!nPhyRst) 
      FDOSTOFineMuxSel <= 1'b0;
    else if(!FDEnable)
      FDOSTOFineMuxSel <= 1'b0;
    else if ((FDStepState == WAIT_FFT) && !FFTBufEmpty && (FDOSymCount>={12'b0,CfgRegSTOCompTime}))
      FDOSTOFineMuxSel <= 1'b1;
  end

  always @(posedge PhyClk,negedge nPhyRst)
  begin
    if(!nPhyRst)
    begin
      FDOMidMulSlope    <= 4'b0;
      FDOMidRShiftSlope <= 2'b0;
    end
    else if(!FDEnable)
    begin
      FDOMidMulSlope    <= 4'b0;
      FDOMidRShiftSlope <= 2'b0;
    end
    else if (MidambleSymbolFDO)
    begin
      if (FrameParam1HeltfType == 2'd0)      // HE-LTF-1x
      begin
        FDOMidMulSlope    <= {2'b0,FDNHELTF[3:2]};
        FDOMidRShiftSlope <= {(FDNHELTF==4'd1),(FDNHELTF==4'd2)|(FDNHELTF==4'd6)};
      end
      else if (FrameParam1HeltfType == 2'd1) // HE-LTF-2x
      begin
        FDOMidMulSlope    <= {1'b0,FDNHELTF[3:1]};
        FDOMidRShiftSlope <= {1'b0,(FDNHELTF==4'd1)};
      end
      else                                   // HE-LTF-4x
      begin
        FDOMidMulSlope    <= FDNHELTF;
        FDOMidRShiftSlope <= 2'b0;
      end
    end
    else
    begin
      FDOMidMulSlope    <= 4'b0;
      FDOMidRShiftSlope <= 2'b0;
    end
  end
    
  /******************************************************************************
  * Midamble control
  ******************************************************************************/

  // MidambleSymbolCount keeps track of the number of data symbols for MidambleSymbol periodicity
  always @ (posedge PhyClk or negedge nPhyRst)
  begin
    if (nPhyRst == 1'b0)
    begin
      MidambleSymbolCount <= 5'b0;
      MidambleSymbol      <= 1'b0;
      MidambleSymbolFDO   <= 1'b0;
      NMACount            <= 10'b0;
      MidambleSkipFFT     <= 1'b0;
    end
    else if(!FDEnable)
    begin
      MidambleSymbolCount <= 5'b0;
      MidambleSymbol      <= 1'b0;
      MidambleSymbolFDO   <= 1'b0;
      NMACount            <= 10'b0;
      MidambleSkipFFT     <= 1'b0;
    end
    else
    begin
      // Counter for MidambleSymbol periodicity
      if((FDSymState == HEDATA) && EqDoneP && FrameParam1Doppler && FrameParam1Update)
      begin
        if (MidambleSymbolCount < MidambleSymbolCountLimit)
          MidambleSymbolCount <= MidambleSymbolCount + 5'd1;
        else
        begin
          MidambleSymbolCount <= 5'b0;
          if (NMACount < {1'b0,FrameParam8NMA})
          begin
            MidambleSymbol    <= 1'b1;
            MidambleSymbolFDO <= MidambleSkipFFT;
          end
        end
      end
      else if ((FDSymState == HEDATA) && (FDStepState == WAIT_FFT) && !FFTBufEmpty)
        MidambleSymbol <= 1'b0;
      // Clear MidambleSymbolFDO on every FDOCompDoneP
      if (FDOCompDoneP)
        MidambleSymbolFDO <= 1'b0;
      
      // Counter for NMA limit on falling edge of MidambleSymbol
      if (!MidambleSymbol && MidambleSymbol_1t)
        NMACount <= NMACount + 10'd1;
      // Skip midamble FFT according to config register
      if (FrameParam1Update)
      begin
        if (FrameParam1HeltfType == 2'd0)
          MidambleSkipFFT <= CfgRegSkipMidamble1x;
        else if (FrameParam1HeltfType == 2'd1)
          MidambleSkipFFT <= CfgRegSkipMidamble2x;
        else
          MidambleSkipFFT <= CfgRegSkipMidamble4x;
      end
      else
        MidambleSkipFFT <= 1'b0;
    end
  end

  assign MidambleSymbolCountLimit = (FrameParam1Midamble) ? 5'd19 : 5'd9;

  always @ (posedge PhyClk or negedge nPhyRst)
  begin
    if (nPhyRst == 1'b0)
      MidambleSymbol_1t <= 1'b0;
    else
      MidambleSymbol_1t <= MidambleSymbol;
  end
  
  assign MidambleSymbolMask = MidambleSymbol & ~MidambleSkipFFT;

`ifdef RW_MUMIMO_RX_EN
  /* Mu-MIMO Control re-registering */
  always @(negedge nPhyRst,posedge PhyClk)
  begin:g_h_selection
    if(!nPhyRst)
      HMuMIMOSel <= 2'b0;
    else if (FrameParam1Update && FrameParam1MuMIMO)               // VHT Mu-MIMO
    begin
      if (FDStepState == WAIT_FFT)
        HMuMIMOSel <= FrameParam1MuPosID;
    end
    else if (FrameParam2Update && (FrameParam2HEMURUNUser > 4'd1)) // HE Mu-MIMO
    begin
      if (FDStepState == WAIT_FFT)
        HMuMIMOSel <= FrameParam2HEMURUUser[1:0]; // Overflow as we support only the 4 1st position due to limitation on HETLF up to 4
    end
    else
      HMuMIMOSel <= 2'b0;
  end
`endif // RW_MUMIMO_RX_EN

  /******************************************************************************
  * FD FSM
  ******************************************************************************/
  reg [1:0] n_FDFFTBufRdPtr;
  always @(*) 
    case(FDFFTBufRdPtrInt)
      2'd0:    n_FDFFTBufRdPtr = 2'd1;
      2'd1:    n_FDFFTBufRdPtr = 2'd2;
      2'd2:    n_FDFFTBufRdPtr = 2'd3;
      default: n_FDFFTBufRdPtr = 2'd0; 
    endcase

  assign n_LTFCount       = LTFCount + 4'd1;
  
  assign FFTBufEmpty      = FFTBufPtr==FDFFTBufRdPtrInt;
  assign FDFFTBufRdPtr    = FDFFTBufRdPtrInt;
  
  assign n_FDSymCount     = FDSymCount + 17'd1;
  assign n_HESigBSymCount = HESigBSymCount + 8'd1;
 
  assign n_bdfd_ch0_wrptr = bdfd_ch0_wrptr + 3'd1;
  assign n_bdfd_ch1_wrptr = bdfd_ch1_wrptr + 3'd1;
  
  wire   deintlv_busy_int;
  assign deintlv_busy_int   = deintlv_busy | (DeIntlvSR!=5'b0);
  
  
  assign n_FDOSymCount    = FDOSymCount + 17'd1;
  
  
  // Mux VHT frame parameters in case of Mu-MIMO
  assign FrameParam1NSTSMux = 
`ifdef RW_MUMIMO_RX_EN
                              (FrameParam1MuMIMO) ? FrameParam1MuNSTS :
`endif // RW_MUMIMO_RX_EN
                              FrameParam1NSTS;
  assign FrameParam1VHTNSS  = 
`ifdef RW_MUMIMO_RX_EN
                              (FrameParam1MuMIMO) ? FrameParam1MuNSTS :
`endif // RW_MUMIMO_RX_EN
                              FrameParam1NSS;
  assign FrameParam1VHTFEC  = 
`ifdef RW_MUMIMO_RX_EN
                              (FrameParam1MuMIMO) ? FrameParam1MuFEC :
`endif // RW_MUMIMO_RX_EN
                              FrameParam1FEC;
  assign FrameParam1VHTSmoothing =
`ifdef RW_MUMIMO_RX_EN
                              (!FrameParam1Smoothing && FrameParam1MuMIMO) ? CfgRegMUSmoothForce :
`endif // RW_MUMIMO_RX_EN
                              (!FrameParam1Smoothing)                      ? CfgRegBFSmoothForce :
                              1'b1;

  always @(posedge PhyClk,negedge nPhyRst)
  begin
    if(!nPhyRst) 
    begin
      ChEstClkEnSR       <= 5'b0;  
      EqClkEnSR          <= 5'b0;  
      FDOClkEnSR         <= 5'b0;  
      ChEstSR            <= 5'b0; 
      EqCompSR           <= 5'b0;
      EqSR               <= 5'b0;   
      FDOEstSR           <= 5'b0;  
      DeIntlvSR          <= 5'b0; 
      FDOInitPilotSR     <= 5'b0;   
      FDOInitKalmanSR    <= 5'b0;   
      ChEstSmoothEnSR    <= 5'b0;   
      ChEstSmoothFromHSR <= 5'b0;
      FDResetSR          <= 5'b0; 

      ChEstReceptionMode <= 4'b0;  
      ChEstRxNsts        <= 3'b0;  
`ifdef RW_NX_DERIV_NESS_EN
      ChEstRxNess        <= 2'b0;  
`endif
      ChEstFirstLTF      <= 1'b0;  
      ChEstLastLTF       <= 1'b0;  
      ChEstSymbolType    <= 3'b0;  
      ChEstRUType        <= 3'b0;  
      ChEstRUIndex       <= 6'b0;  
      ChEstFrameNDP      <= 1'b0;
`ifdef RW_MUMIMO_RX_EN
      ChEstHEMuMIMOEn    <= 1'b0;
`endif
      
      FD2DEn             <= 1'b0;
      FDHEMUOffset       <= 9'b0;
   
      EqStride           <= 1'b0;
      EqHeSigBPAPREn     <= 1'b0;
`ifdef RW_MUMIMO_RX_EN
      EqMuMIMOEn         <= 1'b0;
      EqVHTSIGBNss2      <= 1'b0;
      EqVHTSIGBSub       <= 1'b0;
      EqVHTSIGBInv       <= 1'b0;
      EqMuMIMOICEn       <= 1'b0;
`endif // RW_MUMIMO_RX_EN
      EqReceptionMode    <= 4'b0;  
      EqRxNsts           <= 3'b0;  
      EqRxNss            <= 3'b0;  
      EqSTBCEn           <= 1'b0;  
      EqSTBCSymOdd       <= 1'b0;
      EqSTBCFlush        <= 1'b0;
      EqLDPCEn           <= 1'b0;  
      EqDCMEn            <= 1'b0;  
      EqPilotData        <= 1'b0;  
      EqDiscEn           <= 1'b0;  
      EqModTypeSS1       <= 3'b0;  
      EqMHUpdate         <= 1'b0;  
// STBC or 2x2      EqModTypeSS2       <= 3'b0;  
     
      FDOBW              <= 2'b0;  
      FDOSymCount        <= 17'b0;
      FDOMode            <= 2'b0; 
      FDORUType          <= 3'b0; 
      FDORxNsts          <= 1'b0;
      
      bdfd_wrptr         <= 1'b0;
      bdfd_qbpsk         <= 1'b0;
      bdfd_ch0_wrptr     <= 3'd0;
      bdfd_ch1_wrptr     <= 3'd0;

      deintlv_chsel      <= 1'b0;
      deintlv_fec        <= 1'd0;  
      deintlv_nsd        <= 2'd0;  
      deintlv_iss        <= 2'd0;  
      deintlv_nbpsc      <= 3'd0;  
      deintlv_ruen       <= 1'd0;  
      deintlv_rulen      <= 3'd0;
      deintlv_sigen      <= 1'b0;
      deintlv_sigsel     <= 2'd0; 
      deintlv_dcm        <= 1'b0;
      deintlv_row_fifo_en<= 1'b0; 
      
      LTFCount           <= 4'd0;
      
      FDFFTBufRdPtrInt   <= 2'b0;
`ifdef RW_NX_DERIV_NESS_EN
      FDNELTF            <= 3'b0;
`endif
      FDNDLTF            <= 4'd1;           
      FDDone             <= 1'b0;
      FDFormatMod        <= 4'b0;
      FDSymCount         <= 17'b0;
      FDOStorePreamb     <= 1'b0;
      FDOStoreData       <= 1'b0;
      FDSymState         <= FD_RESET;
      FDStepState        <= WAIT_FFT;

      HESigACount        <= 1'b0;
      HESigBSymCount     <= 8'd0;
      
      TDFormatMod        <= 4'b0;
    
      equ_flushing_state <= 1'b0;
    end
    else if(!FDEnable)
    begin
      ChEstClkEnSR       <= 5'b0;  
      EqClkEnSR          <= 5'b0;  
      FDOClkEnSR         <= 5'b0;  
      ChEstSR            <= 5'b0; 
      EqCompSR           <= 5'b0;
      EqSR               <= 5'b0;   
      FDOEstSR           <= 5'b0;  
      DeIntlvSR          <= 5'b0; 
      FDOInitPilotSR     <= 5'b0;   
      FDOInitKalmanSR    <= 5'b0;   
      ChEstSmoothEnSR    <= 5'b0;   
      ChEstSmoothFromHSR <= 5'b0; 
      FDResetSR          <= 5'b11111; 

      ChEstReceptionMode <= 4'b0;  
      ChEstRxNsts        <= 3'b0;  
`ifdef RW_NX_DERIV_NESS_EN
      ChEstRxNess        <= 2'b0;  
`endif
      ChEstFirstLTF      <= 1'b0;  
      ChEstLastLTF       <= 1'b0;  
      ChEstSymbolType    <= 3'b0;
      ChEstRUType        <= 3'b0;  
      ChEstRUIndex       <= 6'b0;  
      ChEstFrameNDP      <= 1'b0;
`ifdef RW_MUMIMO_RX_EN
      ChEstHEMuMIMOEn    <= 1'b0;
`endif
      
      FD2DEn             <= 1'b0;
      FDHEMUOffset       <= 9'b0;
   
      EqStride           <= 1'b0;
      EqHeSigBPAPREn     <= 1'b0;
`ifdef RW_MUMIMO_RX_EN
      EqMuMIMOEn         <= 1'b0;
      EqVHTSIGBNss2      <= 1'b0;
      EqVHTSIGBSub       <= 1'b0;
      EqVHTSIGBInv       <= 1'b0;
      EqMuMIMOICEn       <= 1'b0;
`endif // RW_MUMIMO_RX_EN
      EqReceptionMode    <= 4'b0;  
      EqRxNsts           <= 3'b0;  
      EqRxNss            <= 3'b0;  
      EqSTBCEn           <= 1'b0;  
      EqSTBCSymOdd       <= 1'b0;
      EqSTBCFlush        <= 1'b0;
      EqLDPCEn           <= 1'b0;  
      EqDCMEn            <= 1'b0;  
      EqPilotData        <= 1'b0;  
      EqDiscEn           <= 1'b0;  
      EqModTypeSS1       <= 3'b0;  
      EqMHUpdate         <= 1'b0;  
// STBC or 2x2      EqModTypeSS2       <= 3'b0;  
     
      FDOBW              <= 2'b0;  
      FDOSymCount        <= 17'b0;
      FDOMode            <= 2'b0; 
      FDORUType          <= 3'b0; 
      FDORxNsts          <= 1'b0;
      FDOStorePreamb     <= 1'b0;
      FDOStoreData       <= 1'b0;
      
      bdfd_wrptr         <= 1'b0;
      bdfd_qbpsk         <= 1'b0;
      bdfd_ch0_wrptr     <= 3'd0;
      bdfd_ch1_wrptr     <= 3'd0;

      deintlv_chsel      <= 1'b0;
      deintlv_fec        <= 1'd0;  
      deintlv_nsd        <= 2'd0;  
      deintlv_iss        <= 2'd0;  
      deintlv_nbpsc      <= 3'd0;  
      deintlv_ruen       <= 1'd0;  
      deintlv_rulen      <= 3'd0;
      deintlv_sigen      <= 1'b0;
      deintlv_sigsel     <= 2'd0;  
      deintlv_dcm        <= 1'b0;
      deintlv_row_fifo_en<= 1'b0; 
    
      LTFCount           <= 4'd0;
      
      FDFFTBufRdPtrInt   <= 2'b0;
`ifdef RW_NX_DERIV_NESS_EN
      FDNELTF            <= 3'b0;
`endif
      FDNDLTF            <= 4'd1;           
      FDDone             <= 1'b0;
      FDFormatMod        <= 4'b0;
      FDSymCount         <= 17'b0;
      FDSymState         <= FD_RESET;
      FDStepState        <= WAIT_FFT;

      HESigACount        <= 1'b0;
      HESigBSymCount     <= 8'd0;

      TDFormatMod        <= 4'b0;
      
      equ_flushing_state <= 1'b0;
      
      
    end
    else
    begin
      
      /* selectable delayed level */
      ChEstClkEnSR       <= {      ChEstClkEnSR[4],       ChEstClkEnSR[4:1]};  
      EqClkEnSR          <= {         EqClkEnSR[4],          EqClkEnSR[4:1]};  
      FDOClkEnSR         <= {        FDOClkEnSR[4],         FDOClkEnSR[4:1]}; 
      ChEstSmoothEnSR    <= {   ChEstSmoothEnSR[4],    ChEstSmoothEnSR[4:1]};    
     
      /* selectable delayed pulse */
      ChEstSR            <= {                 1'b0,            ChEstSR[4:1]};  
      ChEstSmoothFromHSR <= {                 1'b0, ChEstSmoothFromHSR[4:1]};   
      EqCompSR           <= {                 1'b0,           EqCompSR[4:1]};  
      EqSR               <= {                 1'b0,               EqSR[4:1]};  
      FDOEstSR           <= {                 1'b0,           FDOEstSR[4:1]};  
      DeIntlvSR          <= {                 1'b0,          DeIntlvSR[4:1]};  
      FDOInitPilotSR     <= {                 1'b0,     FDOInitPilotSR[4:1]};  
      FDOInitKalmanSR    <= {                 1'b0,    FDOInitKalmanSR[4:1]};  
      FDResetSR          <= {                 1'b0,          FDResetSR[4:1]};  

      /* TD format mode */
      TDFormatMod        <= FDFormatMod;

      /* Pulses  */ 
      FDOStorePreamb     <= 1'b0;
      FDOStoreData       <= 1'b0;
      EqSTBCFlush        <= 1'b0;
      
      /* tracking equalizer flushing state, for equlizer clock gating */
      if(!equ_flushing_state)
      begin
        if(EqSTBCFlush)
          equ_flushing_state <= 1'b1;
      end
      else
      begin
        if(EqSTBCFlushDone)
          equ_flushing_state <= 1'b0;
      end
      
      /* FSM */
      case (FDSymState)
        /***********************************************************************
        * FD_RESET
        ***********************************************************************/
        FD_RESET:
        begin
          // Enable all clocks when FDEnable goes high
          EqClkEnSR    <= 5'b11111;
          ChEstClkEnSR <= 5'b11111;
          FDOClkEnSR   <= 5'b11111;
          EqSTBCSymOdd <= 1'b0;
          // FDReset pulse sent: start all clocks for some cycles, start frame processing
          if(FDResetSR==5'b00000)
          begin
            EqClkEnSR    <= 5'b01111;
            ChEstClkEnSR <= 5'b01111;
            FDOClkEnSR   <= 5'b01111;
            FDSymState   <= LLTF;
            FDStepState  <= WAIT_FFT;
          end
        end
        
        /***********************************************************************
        * LLTF
        ***********************************************************************/
        LLTF:
        begin
          case(FDStepState)
            WAIT_FFT: 
            begin
              if(!FFTBufEmpty)  
              begin             
                ChEstRxNsts     <= {1'b0,CfgRegNrx};              
`ifdef RW_NX_DERIV_NESS_EN
                ChEstRxNess     <= 2'd0;                       
`endif
                ChEstFirstLTF   <= 1'b1;                       
                ChEstLastLTF    <= 1'b1;                       
                ChEstSymbolType <= L_LTF;
                FD2DEn          <= 1'b0;
                FDHEMUOffset    <= 10'b0;
                EqRxNsts        <= 3'd0;                       
                EqRxNss         <= 3'd0;                       
                EqSTBCEn        <= 1'b0;                       
                EqLDPCEn        <= 1'b0;                       

                FDOBW              <= 2'd0;                  
                FDORxNsts          <= 1'b0;
                ChEstReceptionMode <= MODE_20MHZ_L;
                EqReceptionMode    <= MODE_20MHZ_L;
         
                // Enable ChEst clock
                ChEstClkEnSR     <= 5'b11111;
                // Start ChEst, Smoothing and Eq some cycles later
                ChEstSR          <= 5'b10000;
                if (CfgRegForce20==1'b0) 
                begin
                  FDStepState      <= ESTIM_H1; // 2 estim needed, pri and sec
                end 
                else 
                begin
                  // Enable Eq clock
                  EqClkEnSR        <= 5'b11111;
                  ChEstSmoothEnSR  <= 5'b10000;
                  EqCompSR         <= 5'b10000;
                  FDStepState      <= SMOOTH_H2; // only 1 smoothing needed, go directly to 2
                end
              end                            
            end
            
`ifndef RW_NX_DERIV_CHBW20ONLY                
            ESTIM_H1: 
            begin
              if(ChEstDoneP) // Start again on secondary
              begin
                FD2DEn           <= 1'b1;
                ChEstSR          <= 5'b00010; // Quick start as clock is already enabled and time is limited
                FDStepState      <= ESTIM_H2;
              end
            end

            ESTIM_H2: // ESTIM_H2 in 40 MHz config only!
            begin 
              if (ChEstDoneP)
              begin
                FDFFTBufRdPtrInt   <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
                ChEstFirstLTF      <= 1'b0; 
                ChEstLastLTF       <= 1'b0;
                // Enable Equalizer clock
                EqClkEnSR          <= 5'b11111;
                // Start smoothing on Primary
                FD2DEn             <= 1'b0;
                ChEstSmoothFromHSR <= 5'b10000;
                ChEstSmoothEnSR    <= 5'b10000;
                EqCompSR           <= 5'b10000;
                FDStepState        <= SMOOTH_H1;
              end
            end

            SMOOTH_H1:
            begin
              if (EqCompDoneP)
              begin
                // Start smoothing on Secondary - Quick start as clocks are already running and time is limited
                FD2DEn             <= 1'b1;
                ChEstSmoothFromHSR <= 5'b00010;
                EqCompSR           <= 5'b00010;
                FDStepState        <= SMOOTH_H2;
              end
            end
`endif
            
            default: /* SMOOTH_H2 */
            begin
              if (ChEstDoneP && CfgRegForce20)
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD for 20 MHz configuration
              else if (EqCompDoneP)
              begin
                FD2DEn             <= 1'b0;
                // Stop ChEst and Eq clocks after some cycles
                ChEstClkEnSR       <= 5'b01111;
                EqClkEnSR          <= 5'b01111;
                // Disable Smoothing
                ChEstSmoothEnSR    <= 5'b00000;
                ChEstSmoothFromHSR <= 5'b00000;
                LTFCount           <= n_LTFCount;
                FDStepState        <= WAIT_FFT;
                FDSymState         <= LSIG_HELSIG;
              end
            end
            
          endcase
        end
        
        /***********************************************************************
        * LSIG_HELSIG
        ***********************************************************************/
        LSIG_HELSIG:
        begin
          case(FDStepState)
            WAIT_FFT: 
            begin
              if(!FFTBufEmpty)
              begin            
                EqPilotData     <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqReceptionMode <= MODE_20MHZ_L;
               
                // FDO and Eq: start clocks, then enable blocks after some cycles
                FDOClkEnSR  <= 5'b11111;
                EqClkEnSR   <= 5'b11111;
                EqSR        <= 5'b10000;
                FDOEstSR    <= 5'b10000;
                FDStepState <= EQU_PILOT; 
              end
            end
           
            EQU_PILOT:
            begin
              if(EqPilotDoneP)
                EqClkEnSR <= 5'b00011;

              if(FDOCompDoneP) 
              begin
                // Settings for LSIG data equalization: BPSK, 48 sc, Nbpsc=1, interleaved
                EqPilotData    <= 1'b0;
                EqStride       <= 1'b1; // Enable interleaving
                EqModTypeSS1   <= BPSK;  
                
                deintlv_sigen  <= 1'b1;
                deintlv_sigsel <= 2'd0;
                deintlv_nsd    <= NSD_48;
                deintlv_iss    <= 2'b00;
                deintlv_nbpsc  <= NBPSC_1;
                
                // Start Eq and DeIntl - keep FDO clock for data equalization
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                DeIntlvSR      <= 5'b10000;
                FDStepState    <= EQU_DATA;
              end
            end
            
            EQU_DATA:
            begin
              if(!deintlv_busy_int)
              begin
                FDOSymCount      <= n_FDOSymCount;
                
                bdfd_wrptr       <= ~bdfd_wrptr;
                bdfd_qbpsk       <= EqQbpskDet;

                // Stop Eq and FDO clocks
                FDOClkEnSR       <= 5'b00000;
                EqClkEnSR        <= 5'b00000;

                // Possibly HE-LSIG. Start again ChEst and equalization on edge sub-carriers,
                // before updating FDFFTBufRdPtrInt. Do that before waiting for FrameParam as Rx BD
                // latency is too big on 120 MHz FPGA.
                ChEstFirstLTF      <= 1'b1;
                ChEstLastLTF       <= 1'b1;
                ChEstSymbolType    <= HE_LSIG; // Process only edge sub-carriers
                ChEstReceptionMode <= MODE_20MHZ_L;
                ChEstClkEnSR       <= 5'b11111;
                ChEstSR            <= 5'b10000;
                ChEstSmoothEnSR    <= 5'b00000; // No smoothing
                if (CfgRegForce20==1'b0)
                  FDStepState      <= ESTIM_H1; // 2 estim needed, pri and sec
                else
                  FDStepState      <= ESTIM_H2; // only 1 estim needed, go directly to 2
              end
            end
            
            WAIT_BD:
            begin
              if(FrameParam4Update) // LSIG decoded in bit domain
              begin
                if(FrameParam4LSIG6M) 
                begin
                  // Possibly HE-LSIG.
                  FDSymState          <= LDATA0_HTMMSIG1_VHTSIGA1_HERLSIG;
                  FDStepState         <= WAIT_FFT;
                end 
                else if (FrameParam1Update) 
                begin 
                  // LDATA
                  FDSymState          <= LDATA;
                  FDFormatMod         <= FORMATMOD_LEG;
                  EqModTypeSS1        <= ModTypeLUT[FrameParam1NBPSC0];
                  EqMHUpdate          <= 1'b1;            // Update mh with new modulation
                  
                  deintlv_nsd         <= NSD_48;
                  deintlv_iss         <= 2'b00;
                  deintlv_nbpsc       <= FrameParam1NBPSC0;
                  
                  FDStepState         <= WAIT_FFT;
                end
              end
            end

`ifndef RW_NX_DERIV_CHBW20ONLY                
            ESTIM_H1: 
            begin
              if(ChEstDoneP) // Start again on secondary
              begin
                ChEstSR          <= 5'b10000;
                FD2DEn           <= 1'b1;
                FDStepState      <= ESTIM_H2;
              end
            end
`endif
            default: // ESTIM_H2:
            begin
              if(ChEstDoneP)
              begin
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
                ChEstFirstLTF    <= 1'b0; 
                ChEstLastLTF     <= 1'b0;
                FD2DEn           <= 1'b0;
                ChEstClkEnSR     <= 5'b01111;
                FDStepState      <= WAIT_BD;
              end
            end
            
          endcase
        end

        /***********************************************************************
        * LDATA
        ***********************************************************************/
        LDATA:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqReceptionMode <= MODE_20MHZ_L;
                
                // FDO Est and Eq: start clocks, then enable blocks after some cycles
                EqClkEnSR   <= 5'b11111;
                FDOClkEnSR  <= 5'b11111;
                EqSR        <= 5'b10000;
                FDOEstSR    <= 5'b10000;
                FDStepState <= EQU_PILOT; 
              end
            end
            
            EQU_PILOT:
            begin
              if(EqPilotDoneP) begin
                EqClkEnSR  <= 5'b00011;
                EqMHUpdate <= 1'b0;
              end
                
             
              if(FDOCompDoneP) 
              begin
                // FDO Comp and Eq: start clocks, then enable blocks after some cycles
                EqPilotData    <= 1'b0;    
                EqClkEnSR      <= 5'b11111;
                FDOClkEnSR     <= 5'b11111;
                EqSR           <= 5'b10000;
                DeIntlvSR      <= 5'b10000;
                FDStepState    <= EQU_DATA;
              end
            end
            
            default: /* EQU_DATA */
            begin
              if(EqDoneP)
                FDOClkEnSR    <= 5'b00011;
                
              if(!deintlv_busy_int)
              begin
                FDOClkEnSR       <= 5'b00000;
                EqClkEnSR        <= 5'b00000;
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;
                
                bdfd_wrptr       <= ~bdfd_wrptr;

                FDSymCount       <= n_FDSymCount;                
                FDOSymCount      <= n_FDOSymCount;

                if(LastSymbol)
                begin
                  FDOStoreData   <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over data
                  FDSymState     <= FD_DONE;
                end
                else
                begin
                  FDStepState    <= WAIT_FFT;
//                   if(FDSymCount==17'd1)       // to be aligned with matlab, but not used for legacy frames.
//                     FDOStorePreamb   <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over preamble
                end
              end
            end
          endcase
        end

        /***********************************************************************
        * LDATA0_HTMMSIG1_VHTSIGA1_HERLSIG
        ***********************************************************************/
        LDATA0_HTMMSIG1_VHTSIGA1_HERLSIG:
        begin
          case(FDStepState)
            WAIT_FFT:
              if(!FFTBufEmpty)
              begin
                EqPilotData <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqReceptionMode <= MODE_20MHZ_L;
               
                // FDO Est and Eq: start clocks, then enable blocks after some cycles
                EqClkEnSR   <= 5'b11111;
                FDOClkEnSR  <= 5'b11111;
                EqSR        <= 5'b10000;
                FDOEstSR    <= 5'b10000;
                FDStepState <= EQU_PILOT; 
              end
              
            EQU_PILOT:
            begin
              if(EqPilotDoneP)
                EqClkEnSR <= 5'b00011;
              
              if(FDOCompDoneP) 
              begin
                // HT-MM discrimination: start equalizer and Deinterleaver in BPSK mode
                EqPilotData    <= 1'b0;     
                EqModTypeSS1   <= BPSK;  
                EqStride       <= 1'b1;
                EqDiscEn       <= 1'b1;   
                
                deintlv_nsd    <= NSD_48;
                deintlv_iss    <= 2'b00;
                deintlv_nbpsc  <= NBPSC_1;
                
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                DeIntlvSR      <= 5'b10000;
                FDStepState    <= EQU_DATA;
              end
            end
            
            default: /* EQU_DATA */
            begin
              if(!deintlv_busy_int)
              begin
                LTFCount         <= 4'd0;
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;
                
                bdfd_wrptr       <= ~bdfd_wrptr;
                bdfd_qbpsk       <= EqQbpskDet;
                
                if(EqQbpskDet) // QBPSK indicates the symbol was HT-SIG
                begin                  
                  FDFormatMod    <= FORMATMOD_MM;
                  FDSymState     <= HTSIG2;
                  FDOSymCount    <= n_FDOSymCount;
                  FDStepState    <= WAIT_FFT;
                end
                else
                begin // wait BD for HE differenciation
                  FDStepState    <= WAIT_BD;
                end
                EqClkEnSR        <= 5'b00011;
                EqDiscEn         <= 1'b0;
              end
            end

            WAIT_BD:
            begin
              if(FrameParam3Update) 
              begin
                FDStepState      <= WAIT_FFT;
                FDOSymCount      <= n_FDOSymCount;
                if(FrameParam3RLSIG) 
                begin
                  FDSymState     <= HESIGA;
                  HESigACount    <= 1'b0;
                  if (FrameParam6Update && !FrameParam6HeSu)
                    FDFormatMod    <= FORMATMOD_HE_MU;
                  else
                    FDFormatMod    <= FORMATMOD_HE_SU;
                end
                else
                begin
                  FDOInitPilotSR <= 5'b10000;  
                  FDSymState     <= LDATA1_VHTSIGA2;
                end
              end
            end

          endcase
        end

        /***********************************************************************
        * LDATA1_VHTSIGA2
        ***********************************************************************/
        LDATA1_VHTSIGA2:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqReceptionMode <= MODE_20MHZ_L;
               
                EqClkEnSR   <= 5'b11111;
                FDOClkEnSR  <= 5'b11111;
                EqSR        <= 5'b10000;
                FDOEstSR    <= 5'b10000;
                FDStepState <= EQU_PILOT; 
              end
            end

            EQU_PILOT:
            begin
              if(EqPilotDoneP)
                EqClkEnSR <= 5'b00011;

              if(FDOCompDoneP) 
              begin
                // VHT discrimination: start equalizer and Deinterleaver in BPSK mode
                EqPilotData    <= 1'b0;     
                EqModTypeSS1   <= BPSK;  
                EqDiscEn       <= 1'b1;   
                
                deintlv_nsd    <= NSD_48;
                deintlv_iss    <= 2'b00;
                deintlv_nbpsc  <= NBPSC_1;
                
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                DeIntlvSR      <= 5'b10000;
                FDStepState    <= EQU_DATA;
              end
            end
            
            EQU_DATA:
            begin
              if(EqQbpskDet && EqQbpskDetDoneP)
                /* VHTSIGA2 early signalling for TD */
                FDFormatMod <= FORMATMOD_VHT;
              
              if(!deintlv_busy_int)
              begin
                
                EqClkEnSR     <= 5'b00000;
                FDOClkEnSR    <= 5'b00000;
                
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;
                
                bdfd_wrptr    <= ~bdfd_wrptr;
                bdfd_qbpsk    <= EqQbpskDet;
                
                if(EqQbpskDet)
                begin
                  /* VHTSIGA2 */
                  FDStepState      <= WAIT_BD;
                  FDOStorePreamb   <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over preamble
                end
                else
                begin
                  /* LDATA 6Mb */
                  ChEstRxNsts      <= 3'd0;
`ifdef RW_NX_DERIV_NESS_EN
                  ChEstRxNess      <= 2'd0;                
`endif
                  FDOSymCount      <= n_FDOSymCount;
                  FDSymCount       <= 17'd2;
                  
                  FDSymState       <= LDATA;
                  FDStepState      <= WAIT_FFT;
                end
                
                // Clear discriminant
                EqDiscEn         <= 1'b0;
              end
            end
            
            default: /* WAIT_BD*/
            begin
              if(FrameParam1Update)
              begin
                case(FrameParam1CBW)
                  2'd0   : {ChEstReceptionMode,EqReceptionMode,FDOBW} <= {MODE_20MHZ_HT,MODE_20MHZ_HT,2'd0};
                  2'd1   : {ChEstReceptionMode,EqReceptionMode,FDOBW} <= {MODE_40MHZ_HT,MODE_40MHZ_HT,2'd1};
                  default: {ChEstReceptionMode,EqReceptionMode,FDOBW} <= {MODE_80MHZ_HT,MODE_80MHZ_HT,2'd2}; // VHT80 with 6 pilots
                endcase  
                FDOMode         <= FDO_MODE_VHT;
                FDNDLTF         <= n_FDNDLTF;      
                
                FDOClkEnSR      <= 5'b11111;       
                FDOInitPilotSR  <= 5'b10000;       
                FDOInitKalmanSR <= 5'b10000;       
               
                FDStepState     <= WAIT_FFT;       
                FDSymState      <= VHTLTF;         
              end
            end
          endcase
        end
        
        /***********************************************************************
        * HESIGA
        ***********************************************************************/
        HESIGA:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData     <= 1'b1;           // 1 : Pilots equalization. 0: Equalization.
                EqReceptionMode <= MODE_20MHZ_HT;  // To process 52 data sc
                EqMHUpdate      <= ~HESigACount;   // Update mh with HESIGA modulation

                // FDO Est and Eq: start clocks, then enable blocks after some cycles
                EqClkEnSR       <= 5'b11111;
                FDOClkEnSR      <= 5'b11111;
                EqSR            <= 5'b10000;
                FDOEstSR        <= 5'b10000;
                FDStepState     <= EQU_PILOT; 
              end
            end

            EQU_PILOT:
            begin
              if(EqPilotDoneP) 
              begin
                EqMHUpdate     <= 1'b0;
                EqClkEnSR      <= 5'b00011;
              end
              
              if(FDOCompDoneP) 
              begin
                // Settings for HE-SIGA data equalization: BPSK, 52 sc, Nbpsc=1, interleaved
                EqPilotData    <= 1'b0;     
                EqModTypeSS1   <= BPSK;  
                EqStride       <= 1'b1; // Enable interleaving
                
                deintlv_nsd    <= NSD_52;
                deintlv_iss    <= 2'b00;
                deintlv_nbpsc  <= NBPSC_1;
                
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                DeIntlvSR      <= 5'b10000;
                FDStepState    <= EQU_DATA;
              end
            end
            
            EQU_DATA:
            begin
              if(!deintlv_busy_int)
              begin
                EqClkEnSR        <= 5'b00000;
                FDOClkEnSR       <= 5'b00000;
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;
                
                bdfd_wrptr       <= ~bdfd_wrptr;
                bdfd_qbpsk       <= EqQbpskDet;
               
                FDOSymCount      <= n_FDOSymCount;
                
                if(HESigACount==1'b0) 
                begin
                  FDStepState    <= WAIT_FFT;
                  HESigACount    <= 1'b1;
                end 
                else 
                begin
                  FDStepState    <= WAIT_BD;
                end
              end
            end
                        
            default: /* WAIT_BD*/
            begin
              if(FrameParam6Update) 
              begin
                if(FrameParam6HeSu)
                begin
                  // HE-SU
                  FDStepState      <= WAIT_FFT;
                  FDSymState       <= HELTF;         
                  FDOStorePreamb   <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over preamble
                end
                else
                begin
                  // HE-MU
                  if (FrameParam1Update)
                  begin
                    EqModTypeSS1   <= ModTypeLUT[FrameParam1NBPSCHeSigB];
                    deintlv_nbpsc  <= FrameParam1NBPSCHeSigB;
                    deintlv_dcm    <= FrameParam1HESIGBDCM;

                    HESigBSymCount <= 8'd0;
                    FDStepState    <= WAIT_FFT;
                    FDSymState     <= HESIGB;
                    FD2DEn         <= 1'b0;              // Always start with FDO on primary
                    
                    if ((FrameParam1CBW != 2'b0) && CfgRegForce20 && (CfgRegPrimary == 3'd1 || CfgRegPrimary == 3'd3))
                      deintlv_chsel  <= 1'b1;
                    else
                      deintlv_chsel  <= 1'b0;
                  end
                end
              end
            end
         endcase
        end
        
        /***********************************************************************
        * HESIGB
        ***********************************************************************/
        HESIGB:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData     <= 1'b1;            // 1 : Pilots equalization. 0: Equalization.
                EqReceptionMode <= MODE_20MHZ_HT;   // to process 52 data sc
                EqSTBCEn        <= 1'b0;
                EqDCMEn         <= 1'b0;
                EqRxNsts        <= 3'd0;
                EqRxNss         <= 3'd0;
                EqStride        <= 1'b0;
                EqMHUpdate      <= (HESigBSymCount==8'b0); // Update mh with HESIGB modulation

                // FDO Est and Eq: start clocks, then enable blocks after some cycles
                EqClkEnSR       <= 5'b11111;
                FDOClkEnSR      <= 5'b11111;
                EqSR            <= 5'b10000;
                FDOEstSR        <= {!FD2DEn,4'b0000}; // Start FDO only on Primary
                FDStepState     <= EQU_PILOT;
              end
            end
            
            EQU_PILOT:
            begin
              if(EqPilotDoneP) 
              begin
                EqClkEnSR      <= 5'b00011;
                EqMHUpdate     <= 1'b0;
              end
                
              if(FDOCompDoneP) 
              begin
                FDStepState    <= WAIT_BD;
              end
            end
            
            WAIT_BD:
            begin
              /* ensure BDBF buffers are available, LDPC might borrow cycles from FD */
              if(!deintlv_chsel && !bdfd_ch0_fifo_full || deintlv_chsel  && !bdfd_ch1_fifo_full) 
              begin
                EqSTBCEn       <= 1'b0;
                EqLDPCEn       <= 1'b0;
                EqDCMEn        <= FrameParam1HESIGBDCM;
                EqPilotData    <= 1'b0;     
                EqStride       <= 1'b1;
                EqHeSigBPAPREn <= ((FrameParam1HESIGBMCS == 3'b0) && FrameParam1HESIGBDCM) ? 1'b0 : 1'b1;
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                DeIntlvSR      <= 5'b10000;
                FDStepState    <= EQU_DATA;
                // Set FD2DEn for data
                if(deintlv_chsel==1'b0) 
                begin
                  deintlv_sigsel <= bdfd_ch0_wrptr[1:0];
                  if(FrameParam1CBW==2'd1 && !CfgRegForce20)
                    FD2DEn <= CfgRegPrimary[0];
                end 
                else 
                begin
                  deintlv_sigsel <= bdfd_ch1_wrptr[1:0];
                  FD2DEn         <= ~CfgRegPrimary[0];
                end
              end
            end
            
            default:  /* EQU_DATA */
            begin
              if(EqDoneP) 
              begin
                if((deintlv_chsel==1'b1) || (FrameParam1CBW==2'd0) || CfgRegForce20)
                  FDOClkEnSR <= 5'b00011;
              end
            
              if(!deintlv_busy_int)
              begin
                if((deintlv_chsel==1'b1) || (FrameParam1CBW==2'd0) || CfgRegForce20)
                begin
                  FDStepState      <= WAIT_FFT;
                  FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr; // todo move to if(EqDoneP) ?
                  FDOSymCount      <= n_FDOSymCount;
                  EqClkEnSR        <= 5'b00000;
                  if(deintlv_chsel==1'b1)
                  begin
                    bdfd_ch1_wrptr   <= n_bdfd_ch1_wrptr;
                  end
                  else
                  begin
                    bdfd_ch0_wrptr   <= n_bdfd_ch0_wrptr;
                  end
                  if (!CfgRegForce20)
                    deintlv_chsel  <= 1'b0; // toggles deintlv_chsel only in CHBW40
                  FD2DEn           <= 1'b0;
                  HESigBSymCount   <= n_HESigBSymCount;              
                  if(LastHeSigBSymbol) 
                  begin
                    FDOStorePreamb <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over preamble
                    FDSymState     <= HELTF;
                    if ((FrameParam1CBW == 2'd1) && CfgRegForce20)
                    begin
                      if (CfgRegPrimary==3'b0)
                        FDHEMUOffset <= 10'd128;
                      else
                        FDHEMUOffset <= -10'sd128;
                    end
                    else if (FrameParam1CBW == 2'd2)
                    begin
                      if (CfgRegForce20) // 20MHz in 80MHZ BW
                      begin
                        if (CfgRegPrimary==3'd0)
                          FDHEMUOffset <= 10'd384;
                        else if (CfgRegPrimary==3'd1)
                          FDHEMUOffset <= 10'd128;
                        else if (CfgRegPrimary==3'd2)
                          FDHEMUOffset <= -10'sd128;
                        else
                          FDHEMUOffset <= -10'sd384;
                      end
                      else               // 40MHz in 80MHZ BW
                      begin
                        if (CfgRegPrimary[1])
                          FDHEMUOffset <= -10'd256;
                        else
                          FDHEMUOffset <=  10'sd256;
                      end
                    end
                  end
                end 
                else 
                begin
                  deintlv_chsel    <= 1'b1;
                  FD2DEn           <= ~FD2DEn;
                  FDStepState      <= WAIT_BD;
                  bdfd_ch0_wrptr   <= n_bdfd_ch0_wrptr;
                end
              end
            end
          endcase
        end
        
        /***********************************************************************
        * HELTF
        ***********************************************************************/
        HELTF:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty && FrameParam8Update) // Check completion of RXBD calculator
              begin
                if (FrameParam8NSym==16'd0) begin
                  // Special case with NDP, all HELTF are estimated up to 4
                  ChEstRxNsts        <= FrameParam1NSTS & {{3'd3}{LTFCount!=4'd0}};
                  ChEstFirstLTF      <= LTFCount==4'd0;
                  ChEstFrameNDP      <= 1'b1;
                  ChEstClkEnSR       <= 5'b11111;
                  ChEstSR            <= 5'b10000;
                  ChEstSmoothEnSR    <= {n_LTFCount==FDNHELTF,4'b0000};
                end
                else if (FrameParam1STBC) begin
                  // Special case with STBC, 2x HELTF are estimated.
                  ChEstRxNsts        <= FrameParam1NSTS;
                  ChEstFrameNDP      <= 1'b0;
                  ChEstSmoothEnSR    <= {n_LTFCount==FDNHELTF,4'b0000};
                  if (FDFormatMod == FORMATMOD_HE_MU)
                  begin
                    ChEstFirstLTF      <= n_LTFCount==ChEstNShiftLTF;
                    ChEstClkEnSR       <= {5{n_LTFCount>=ChEstNShiftLTF}};
                    ChEstSR            <= {n_LTFCount>=ChEstNShiftLTF,4'b0000};
                  end
                  else
                  begin
                    ChEstFirstLTF      <= LTFCount==4'd0;
                    ChEstClkEnSR       <= 5'b11111;
                    ChEstSR            <= 5'b10000;
                  end
                end
                else begin
                  // Normal case, only last HELTF is estimated
                  ChEstRxNsts        <= 3'd0;
                  ChEstFirstLTF      <= n_LTFCount==FDNHELTF;
                  ChEstFrameNDP      <= 1'b0;
                  ChEstClkEnSR       <= {5{n_LTFCount>=FDNHELTF}};
                  ChEstSR            <= {n_LTFCount>=FDNHELTF,4'b0000};
`ifdef RW_MUMIMO_RX_EN
                  if (FrameParam2HEMURUNUser > 4'd1) begin
                    ChEstSmoothEnSR    <= {n_LTFCount==FDNHELTF &  CfgRegMUSmoothForce,4'b0000};
                    ChEstHEMuMIMOEn    <= (n_LTFCount==FDNHELTF) & ~CfgRegMUSmoothForce;
                  end
                  else begin
`endif // RW_MUMIMO_RX_EN
                    ChEstSmoothEnSR    <= {n_LTFCount==FDNHELTF,4'b0000};
`ifdef RW_MUMIMO_RX_EN
                  end
`endif // RW_MUMIMO_RX_EN
                end
                ChEstLastLTF       <= n_LTFCount==FDNHELTF;
                if (FrameParam6HeSu) begin
                  EqModTypeSS1       <= ModTypeLUT[FrameParam1NBPSC0];
                  if(FrameParam1CBW==2'd0) begin
                    ChEstReceptionMode <= MODE_20MHZ_HE_SU;
                    EqReceptionMode    <= MODE_20MHZ_HE_SU;
                  end
                  else begin
                    ChEstReceptionMode <= MODE_40MHZ_HE_SU;
                    EqReceptionMode    <= MODE_40MHZ_HE_SU;
                  end
                end
                else begin
                  EqModTypeSS1       <= ModTypeLUT[FrameParam2MuNBPSC];  
                  ChEstRUType        <= FrameParam2RUType;
                  ChEstRUIndex       <= FrameParam2RUIndex;
                  if(FrameParam1CBW==2'd0) begin
                    ChEstReceptionMode <= MODE_20MHZ_HE_MU;
                    EqReceptionMode    <= MODE_20MHZ_HE_MU;
                  end
                  else if(FrameParam1CBW==2'd1) begin
                    ChEstReceptionMode <= MODE_40MHZ_HE_MU;
                    EqReceptionMode    <= MODE_40MHZ_HE_MU;
                  end
                  else begin
                    ChEstReceptionMode <= MODE_80MHZ_HE_MU;
                    EqReceptionMode    <= MODE_80MHZ_HE_MU;
                  end
                end
                EqHeSigBPAPREn     <= 1'b0;
                case (FrameParam1HeltfType)
                  2'd0:    ChEstSymbolType <= HE_1xLTF; 
                  2'd1:    ChEstSymbolType <= HE_2xLTF; 
                  default: ChEstSymbolType <= HE_4xLTF; 
                endcase
                
`ifdef RW_MUMIMO_RX_EN
                if (FrameParam2HEMURUNUser > 4'd1) begin
                  EqRxNsts           <= FrameParam2HEMUNSTSTot;
                  EqMuMIMOEn         <= 1'b1;
                end
                else
`endif // RW_MUMIMO_RX_EN
                  EqRxNsts           <= FrameParam1NSTS;
                EqRxNss            <= 3'd0;  // TODO for 2ss
                EqLDPCEn           <= FrameParamHEFEC;
                EqSTBCEn           <= FrameParam1STBC;
                
                EqClkEnSR          <= {5{n_LTFCount>=FDNHELTF}};
                EqCompSR           <= {n_LTFCount>=FDNHELTF,4'b0000};
               
                if ((FrameParam1STBC && ((FDFormatMod == FORMATMOD_HE_SU) || (n_LTFCount>=ChEstNShiftLTF))) ||
                 (n_LTFCount>=FDNHELTF) || (FrameParam8NSym==16'd0)) begin
                  FDStepState      <= ESTIM_H1;
                end else begin
                  FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
                  LTFCount         <= n_LTFCount;
                end
              end
            end
            
            default: /* ESTIM_H1 */
            begin
              if (ChEstDoneP)
              begin
                FDFFTBufRdPtrInt   <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
              end
              else if(EqCompDoneP || (ChEstSmthDoneP && n_LTFCount!=FDNHELTF))
              begin
                LTFCount           <= n_LTFCount;
                FDStepState        <= WAIT_FFT;
                
                ChEstClkEnSR       <= 5'b00000;
                ChEstFirstLTF      <= 1'b0;
                ChEstLastLTF       <= 1'b0;
                ChEstFrameNDP      <= 1'b0;
                ChEstSmoothFromHSR <= 5'b00000;
                ChEstSmoothEnSR    <= 5'b00000;
`ifdef RW_MUMIMO_RX_EN
                ChEstHEMuMIMOEn    <= 1'b0;
`endif // RW_MUMIMO_RX_EN
                
                if(n_LTFCount==FDNHELTF)
                begin
                  if (!MidambleSymbolMask)
                  begin
                    FDOSymCount      <= 17'd0;
                  end
                  FDOClkEnSR         <= 5'b11111;
                  FDOInitPilotSR     <= 5'b10000 & {5{~MidambleSymbolMask}};
                  FDOInitKalmanSR    <= 5'b10000;
                  
                  
                  deintlv_ruen   <= 1'b1;   // RuEn and RuLen used instead of NSD
                  if(FrameParam6HeSu) 
                  begin
                    if (FrameParam1CBW==2'd0) 
                    begin
                      FDORUType      <= RU_242;  // SU value is RU 242
                      deintlv_nsd    <= NSD_234; // Applicable only to 20 MHz, for BCC
                      deintlv_rulen  <= RU_242;
                    end 
                    else 
                    begin
                      FDORUType      <= RU_484; // SU value is RU 484
                      deintlv_rulen  <= RU_484;
                    end
                    FDOBW            <= FrameParam1CBW; 
                    EqModTypeSS1     <= ModTypeLUT[FrameParam1NBPSC0];  
                    deintlv_nbpsc    <= FrameParam1NBPSC0;
                  end 
                  else 
                  begin
                    FDORUType        <= FrameParam2RUType;
                    FDOBW            <= FrameParam1CBW;
                    EqModTypeSS1     <= ModTypeLUT[FrameParam2MuNBPSC];  
                    deintlv_rulen    <= FrameParam2RUType;
                    deintlv_nbpsc    <= FrameParam2MuNBPSC;
                  end
                  if(FDFormatMod==FORMATMOD_HE_MU)
                    deintlv_dcm        <= FrameParam2HEMUDCM;
                  else
                    deintlv_dcm        <= FrameParam1DCM;
                  
                  deintlv_iss        <= 2'b00;
                  deintlv_sigen      <= 1'b0;
                  deintlv_sigsel     <= 2'b0;
                  if(!FrameParamHEFEC)
                  begin
                    /* BCC */ 
                    deintlv_fec         <= 1'b0;
                    deintlv_row_fifo_en <= 1'b1;
                  end
                  else
                  begin
                    /* LDPC */
                    deintlv_fec         <= 1'b1;
                    deintlv_row_fifo_en <= 1'b0;
                  end
                 
                  
                  // FDO mode: should not be FDO_MODE_LM else pilot init is bypassed (FDOffsetTop)
                  //           should not be FDO_MODE_HTGF else psi init is bypassed (StoCpeFilter)
                  FDOMode            <= FDO_MODE_VHT; // HE SU pilots same as 80 MHz VHT

                  if(FrameParam8NSym==16'd0)
                    FDSymState <= FD_DONE;
                  else
                    FDSymState <= HEDATA;
                end
                else 
                begin
                  FDStepState <= WAIT_FFT;
                end
              end
            end
          endcase
        end

        /***********************************************************************
        * HEDATA
        ***********************************************************************/
        HEDATA:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData     <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqDCMEn         <= 1'b0;
                EqRxNsts        <= 3'd0;                       
                EqRxNss         <= 3'd0;                       
                EqStride        <= 1'b0;
                EqClkEnSR       <= 5'b11111;
                FDOClkEnSR      <= 5'b11111;
                EqSR            <= 5'b10000;
                FDOEstSR        <= 5'b10000;
                FDStepState     <= EQU_PILOT; 
              end
              else
              begin
                if(EqSTBCFlushDone)
                begin
                  /* gate equalizer clock if stbc flushing done */
                  EqClkEnSR     <= 5'b01111;
                end
              end
            end
            
            EQU_PILOT:
            begin
              if(EqPilotDoneP)
              begin
                if(!equ_flushing_state)
                begin
                  EqClkEnSR <= 5'b00011;
                end
              end
                
              if(FDOCompDoneP) 
              begin
                if(FDFormatMod==FORMATMOD_HE_MU)
                  EqDCMEn      <= FrameParam2HEMUDCM;
                else
                  EqDCMEn      <= FrameParam1DCM;
                
                EqPilotData    <= 1'b0;     
                EqRxNsts       <= FrameParam1NSTS;  
                EqRxNss        <= FrameParam1NSS;   
                // Disable tone mapping for RU26 LDPC frames. !FrameParam6HeSu is a qualifier for FrameParam2RUType.
                if(FrameParamHEFEC && !FrameParam6HeSu && ((FrameParam2RUType==RU_26) || ((FrameParam2RUType==RU_52) && FrameParam1DCM)))
                  EqStride       <= 1'b0;
                else
                  EqStride       <= 1'b1;
                if(!EqSTBCEn || EqSTBCSymOdd)
                begin
                  DeIntlvSR      <= 5'b10000;
                end
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                FDStepState    <= EQU_DATA;
              end
            end
            
            EQU_DATA:  /* EQU_DATA */
            begin
              if(EqDoneP)
                FDOClkEnSR <= 5'b00011;
            
              // Clear LTF counter for midamble support
              LTFCount <= 4'd0;
             
              if(BDDone)
              begin
                FDResetSR     <= 5'b11111;
                FDOStoreData  <= 1'b1; 
                FDSymState    <= FD_DONE;
              end
              else if(!EqSTBCEn)
              begin
                /* non-stbc */
                if((deintlv_fec && EqPipeFlushDoneP) || (!deintlv_fec && !deintlv_busy_int))
                begin
                  FDOClkEnSR <= 5'b00000;
                  
                  FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;
                  
                  FDSymCount    <= n_FDSymCount;                
                  FDOSymCount   <= n_FDOSymCount;
                 
                  if(LastSymbol)
                  begin
                    FDSymState    <= FD_DONE;
                    FDOStoreData  <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over data
                  end
                  else
                  begin
                    EqClkEnSR     <= 5'b00000;
                    FDStepState   <= WAIT_FFT;
                    if (MidambleSymbolMask)
                    begin
                      FDSymState  <= HELTF;
                    end
                  end
                end
              end
              else
              begin
                if(!EqSTBCSymOdd)
                begin
                  /* stbc partial even symbol equaliztion */
                  if(EqPipeFlushDoneP)
                  begin
                    /* stbc, even symbol partial equalization done */
                    EqSTBCSymOdd     <= 1'b1;       
                    FDOClkEnSR       <= 5'b00000;              
                    FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;       
                    FDSymCount       <= n_FDSymCount;          
                    FDOSymCount      <= n_FDOSymCount;
                    EqClkEnSR        <= 5'b01111;                                                                            
                    FDStepState      <= WAIT_FFT;                                                         
                  end
                end
                else
                begin
                  /* stbc even+odd full equalization*/
                  if((!deintlv_fec && !deintlv_busy_int) || (deintlv_fec && EqPipeFlushDoneP))
                  begin
                    EqSTBCSymOdd     <= 1'b0;       
                    DeIntlvSR        <= 5'b00001;           
                    EqSTBCFlush      <= 1'b1;               
                    FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;       
                    FDSymCount       <= n_FDSymCount;          
                    FDOSymCount      <= n_FDOSymCount;
                    if(LastSymbol)
                      FDStepState <= WAIT_STBC_FLUSH;
                    else
                    begin
                      FDStepState <= WAIT_FFT;
                      if (MidambleSymbolMask)
                      begin
                        FDSymState  <= HELTF;
                      end
                    end
                  end
                end
              end
            end
            
            default: /* WAIT_STBC_FLUSH */
            begin
              if(BDDone)
              begin
                FDResetSR     <= 5'b11111;
                FDOStoreData  <= 1'b1; 
                FDSymState    <= FD_DONE;
              end
              else if(EqSTBCFlushDone)
              if(EqSTBCFlushDone)
              begin
                FDOStoreData <= 1'b1; // flag 
                FDSymState   <= FD_DONE;           
              end
            end
            
          endcase
        end
        
        /***********************************************************************
        * HTSIG2
        ***********************************************************************/
        HTSIG2:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqReceptionMode <= MODE_20MHZ_L;
                
                EqClkEnSR   <= 5'b11111;
                EqSR        <= 5'b10000;
                FDOEstSR    <= 5'b10000;
                FDStepState <= EQU_PILOT; 
              end
            end
             
            EQU_PILOT:
            begin
              if(EqPilotDoneP)
                EqClkEnSR <= 5'b00011;
                
              if(FDOCompDoneP) 
              begin
                EqPilotData    <= 1'b0;    
                EqModTypeSS1   <= QBPSK;  
                EqDiscEn       <= 1'b1;   
                
                deintlv_nsd    <= NSD_48;
                deintlv_iss    <= 2'b00;
                deintlv_nbpsc  <= NBPSC_1;
              
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                DeIntlvSR      <= 5'b10000;
               
                FDStepState    <= EQU_DATA;
              end
            end
            
            EQU_DATA:
            begin
              if(!deintlv_busy_int)
              begin
                EqClkEnSR        <= 5'b00000;
                FDOClkEnSR       <= 5'b00000;
                FDOStorePreamb   <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over preamble
                
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;
              
                bdfd_wrptr       <= ~bdfd_wrptr;
                bdfd_qbpsk       <= EqQbpskDet;
               
                FDStepState      <= WAIT_BD;
              end
            end
           
            default: /* WAIT_BD */
            begin
              if(FrameParam1Update)
              begin
                FDOClkEnSR     <= 5'b11111;
                FDOInitPilotSR <= 5'b10000;
                  
                if((FDFormatMod==FORMATMOD_MM) || (FrameParam1NSTS!=3'd0)
`ifdef RW_NX_DERIV_NESS_EN
           || (FrameParam1NESS!=2'd0)
`endif
                 )
                  FDOInitKalmanSR <= 5'b10000;      
               
                FDOBW <= FrameParam1CBW;
                if(FrameParam1CBW==2'b0)
                begin
                  EqReceptionMode <= MODE_20MHZ_HT;
                  FDOMode         <= FDO_MODE_HTMM;
                  FDORxNsts       <= FrameParam1NSTS[0];
                end
                else
                begin
                  if(FrameParam1NSD==2'd0)
                  begin
                    EqReceptionMode <= MODE_20MHZ_L; /* mcs32 */               
                    FDOMode         <= FDO_MODE_LM;                
                    FDOBW           <= 2'd0;
                  end
                  else
                  begin
                    EqReceptionMode <= MODE_40MHZ_HT;                
                    FDOMode         <= FDO_MODE_HTMM;
                    FDORxNsts       <= FrameParam1NSTS[0];
                  end
                end
                
                EqDiscEn       <= 1'b0; 
                EqModTypeSS1   <= ModTypeLUT[FrameParam1NBPSC0];
                EqSTBCEn       <= FrameParam1STBC;
                
                if(!FrameParam1FEC)
                begin
                  /* BCC */ 
                  deintlv_fec         <= 1'b0;
                  deintlv_row_fifo_en <= 1'b1;
                  EqStride            <= 1'b1;
                end
                else
                begin
                  /* LDPC */
                  deintlv_fec         <= 1'b1;
                  deintlv_row_fifo_en <= 1'b0;
                  EqStride            <= 1'b0;
                end
                
                deintlv_sigen       <= 1'b0;
                deintlv_sigsel      <= 2'd0;
                deintlv_nsd         <= FrameParam1NSD;
                deintlv_iss         <= 2'b00;
                deintlv_nbpsc       <= FrameParam1NBPSC0;
               
                FDNDLTF             <= n_FDNDLTF;
`ifdef RW_NX_DERIV_NESS_EN
                FDNELTF             <= n_FDNELTF;
`endif
                if(LTFCount==n_FDNDLTF)
                begin
                  /* all DLTF processed */
`ifdef RW_NX_DERIV_NESS_EN
                  if(FrameParam1NESS!=2'd0)
                  begin
                    /* there are ELTF to process */
                    LTFCount    <= 4'd0;
                    FDSymState  <= HTELTF;
                    FDStepState <= WAIT_FFT;
                  end
                  else
                  begin
`endif
                    /* no ELTF to process, go to HTDATA */
                    if(FrameParam1HTNDP)
                      FDSymState <= FD_DONE;
                    else
                      FDSymState <= HTDATA;
                    
                    FDStepState <= WAIT_FFT;
`ifdef RW_NX_DERIV_NESS_EN
                  end
`endif
                end
                else
                begin
                  FDSymState  <= HTDLTF;
                  FDStepState <= WAIT_FFT;
                end
              end
            end
          endcase
        end
  
        /***********************************************************************
        * HTDLTF
        ***********************************************************************/
        HTDLTF:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                ChEstRxNsts        <= FrameParam1NSTS;
`ifdef RW_NX_DERIV_NESS_EN
                ChEstRxNess        <= FrameParam1NESS;
`endif
                ChEstFirstLTF      <= LTFCount==4'd0; 
                ChEstLastLTF       <= n_LTFCount==FDNDLTF;
                ChEstSymbolType    <= D_HTLTF;
                ChEstFrameNDP      <= FrameParam1HTNDP & (n_LTFCount==FDNDLTF);
                if(FrameParam1CBW==2'd0 || FrameParam1NSD==2'd0) // 20MHz or mcs32
                  ChEstReceptionMode <= MODE_20MHZ_HT;
                else
                  ChEstReceptionMode <= MODE_40MHZ_HT;
                EqRxNsts           <= FrameParam1NSTS;  
                EqRxNss            <= FrameParam1NSS;   
                EqLDPCEn           <= FrameParam1FEC;
                
                ChEstClkEnSR       <= 5'b11111;
                EqClkEnSR          <= 5'b11111;
                ChEstSmoothEnSR    <= {FrameParam1Smoothing && n_LTFCount==FDNDLTF,4'b0000};
                ChEstSR            <= 5'b10000;
                EqCompSR           <= {n_LTFCount==FDNDLTF,4'b0000};
               
                FDStepState        <= ESTIM_H1;
              end
            end
            
            default: /* ESTIM_H1 */
            begin
              if (ChEstDoneP)
                FDFFTBufRdPtrInt   <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
              else if(EqCompDoneP || ChEstSmthDoneP && n_LTFCount!=FDNDLTF)
              begin
                ChEstFirstLTF      <= 1'b0;
                ChEstLastLTF       <= 1'b0;
                ChEstClkEnSR       <= 5'b01111;
                ChEstSmoothEnSR    <= 5'b00000; 
                ChEstSmoothFromHSR <= 5'b00000; 
                LTFCount           <= n_LTFCount;
                if(n_LTFCount==FDNDLTF)
                begin                
                  EqClkEnSR        <= 5'b00011;

                  if(FDFormatMod==FORMATMOD_MM)
                    FDOSymCount <= n_FDOSymCount;

`ifdef RW_NX_DERIV_NESS_EN
                  if(FDNELTF!=3'd0)
                  begin
                    LTFCount    <= 4'd0;
                    FDSymState  <= HTELTF;
                    FDStepState <= WAIT_FFT;
                  end
                  else
                  begin
`endif
                    FDStepState <= WAIT_FFT;
                    if(FrameParam1HTNDP)
                      FDSymState <= FD_DONE;
                    else
                      FDSymState <= HTDATA;
`ifdef RW_NX_DERIV_NESS_EN
                  end
`endif
                end
                else
                begin
                  FDStepState <= WAIT_FFT;
                end
              end
            end
          endcase
        end
     
`ifdef RW_NX_DERIV_NESS_EN
        /***********************************************************************
        * HTELTF
        ***********************************************************************/
        HTELTF:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                ChEstRxNess     <= FrameParam1NESS;
                ChEstFirstLTF   <= LTFCount==4'd0; 
                ChEstLastLTF    <= n_LTFCount=={1'b0,FDNELTF};
                ChEstSymbolType <= E_HTLTF;
                ChEstClkEnSR    <= 5'b11111;
                ChEstSmoothEnSR <= 5'b00000;
                ChEstSR         <= 5'b10000;
                FDStepState     <= ESTIM_H1;
              end
            end
            
            default: /* ESTIM_H1 */
            begin
              if (ChEstDoneP)
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
              else if(ChEstSmthDoneP)
              begin
                ChEstFirstLTF    <= 1'b0;
                ChEstLastLTF     <= 1'b0;
                LTFCount         <= n_LTFCount;
                FDStepState      <= WAIT_FFT;
                if(n_LTFCount=={1'b0,FDNELTF})
                begin
                  if(FrameParam1HTNDP)
                    FDSymState <= FD_DONE;
                  else
                    FDSymState <= HTDATA;
                end
              end
            end
          endcase
        end
`endif

        /***********************************************************************
        * HTDATA
        ***********************************************************************/
        HTDATA:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqClkEnSR   <= 5'b11111;
                FDOClkEnSR  <= 5'b11111;
                EqSR        <= 5'b10000;
                FDOEstSR    <= 5'b10000;
                FDStepState <= EQU_PILOT; 
              end
              else
              begin
                if(EqSTBCFlushDone)
                begin
                  /* gate equalizer clock if stbc flushing done */
                  EqClkEnSR     <= 5'b01111;
                end
              end
            end
            
            EQU_PILOT:
            begin
              if(EqPilotDoneP)
              begin
                if(!equ_flushing_state)
                begin
                  EqClkEnSR <= 5'b00011;
                end
              end
                
              if(FDOCompDoneP) 
              begin
                EqPilotData    <= 1'b0;
                if(!EqSTBCEn || EqSTBCSymOdd)
                begin
                  DeIntlvSR      <= 5'b10000;
                end
                EqClkEnSR      <= 5'b11111;
                EqSR           <= 5'b10000;
                FDStepState    <= EQU_DATA;
              end
            end
            
            EQU_DATA:  /* EQU_DATA */
            begin
              if(EqDoneP)
                FDOClkEnSR <= 5'b00011;

              if(BDDone)
              begin
                FDResetSR     <= 5'b11111;
                FDOStoreData  <= 1'b1; 
                FDSymState    <= FD_DONE;
              end
              else if(!EqSTBCEn)
              begin
                /* non-stbc */
                if(!deintlv_fec  && !deintlv_busy_int || deintlv_fec  && EqPipeFlushDoneP)
                begin
                  FDOClkEnSR       <= 5'b00000;                                                                              
                  FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;                                                                       
                  FDSymCount       <= n_FDSymCount;                                                                          
                  FDOSymCount      <= n_FDOSymCount;   
                
                  if(LastSymbol)                                                                                             
                  begin                                                                                                      
                    FDOStoreData  <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over data    
                    FDSymState    <= FD_DONE;                                                                                
                  end                                                                                                        
                  else                                                                                                       
                  begin 
                    EqClkEnSR   <= 5'b00000;                                                                            
                    FDStepState   <= WAIT_FFT;                                                                               
                  end                                                                                                              
                end           
              end
              else
              begin
                if(!EqSTBCSymOdd)
                begin
                  /* stbc partial even symbol equaliztion */
                  if(EqPipeFlushDoneP)
                  begin
                    /* stbc, even symbol partial equalization done */
                    EqSTBCSymOdd     <= 1'b1;       
                    FDOClkEnSR       <= 5'b00000;              
                    FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;       
                    FDSymCount       <= n_FDSymCount;          
                    FDOSymCount      <= n_FDOSymCount;
                    EqClkEnSR        <= 5'b01111;                                                                            
                    FDStepState      <= WAIT_FFT;                                                                             
                  end
                end
                else
                begin
                  /* stbc even+odd full equalization*/
                  if((!deintlv_fec  && !deintlv_busy_int || deintlv_fec  && EqPipeFlushDoneP))
                  begin
                    EqSTBCSymOdd     <= 1'b0;       
                    DeIntlvSR        <= 5'b00001;           
                    EqSTBCFlush      <= 1'b1;               
                    FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;       
                    FDSymCount       <= n_FDSymCount;          
                    FDOSymCount      <= n_FDOSymCount;
                    if(LastSymbol)
                      FDStepState <= WAIT_STBC_FLUSH;
                    else
                      FDStepState <= WAIT_FFT;                                                                             
                  end
                end
              end
            end
            
            WAIT_STBC_FLUSH:
            begin
              if(BDDone)
              begin
                FDResetSR    <= 5'b11111;
                FDOStoreData <= 1'b1; 
                FDSymState   <= FD_DONE;
              end
              else if(EqSTBCFlushDone)
              begin
                FDOStoreData <= 1'b1; // flag 
                FDSymState   <= FD_DONE;           
              end
            end
          endcase
        end
        
        /***********************************************************************
        * VHTLTF
        ***********************************************************************/
        VHTLTF:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
`ifdef RW_MUMIMO_RX_EN
                if(FrameParam1MuMIMO) 
                begin
                  // Special case with Mu-MIMO, only last VHTLTF is estimated
                  ChEstRxNsts        <= 3'd0;
                  ChEstFirstLTF      <= n_LTFCount==FDNDLTF; 
                  ChEstFrameNDP      <= 1'b0;
                  ChEstClkEnSR       <= {5{n_LTFCount>=FDNDLTF}};
                  ChEstSR            <= {n_LTFCount>=FDNDLTF,4'b0000};
                  ChEstSmoothEnSR    <= {n_LTFCount==FDNDLTF &  CfgRegMUSmoothForce,4'b0000};
                end
                else 
                begin
`endif // RW_MUMIMO_RX_EN
                  // Normal case, all VHTLTFs are estimated (STBC/NDP)
                  ChEstRxNsts        <= FrameParam1NSTSMux;
                  ChEstFirstLTF      <= LTFCount==4'd0;
                  ChEstFrameNDP      <= FrameParam8Update & (FrameParam8NSym==16'd0);
                  ChEstClkEnSR       <= 5'b11111;
                  ChEstSR            <= 5'b10000;
                  ChEstSmoothEnSR    <= {FrameParam1VHTSmoothing && n_LTFCount==FDNDLTF,4'b0000};
`ifdef RW_MUMIMO_RX_EN
                end
`endif // RW_MUMIMO_RX_EN
`ifdef RW_NX_DERIV_NESS_EN
                ChEstRxNess        <= 2'd0;
`endif
                ChEstLastLTF       <= n_LTFCount==FDNDLTF;
                ChEstSymbolType    <= D_VHTLTF;

                EqRxNsts           <= FrameParam1NSTSMux;
                EqRxNss            <= FrameParam1VHTNSS;
`ifdef RW_MUMIMO_RX_EN
                EqMuMIMOEn         <= FrameParam1MuMIMO;
                EqMuMIMOICEn       <= FrameParam1MuMIMO & (n_LTFCount==FDNDLTF);
`endif // RW_MUMIMO_RX_EN
               
                ChEstSmoothFromHSR <= 5'b00000;
                
                EqClkEnSR          <= 5'b11111 & {{5'd5}{n_LTFCount==FDNDLTF}};
                EqCompSR           <= 5'b10000 & {{5'd5}{n_LTFCount==FDNDLTF}};
                
                if(FrameParam1STBC || (n_LTFCount>=FDNDLTF) || (FrameParam8Update && (FrameParam8NSym==16'd0))) 
                begin
                  FDStepState      <= ESTIM_H1;
                end 
                else 
                begin
                  FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
                  LTFCount         <= n_LTFCount;
                end
              end
            end
            
            default: /* ESTIM_H1 */
            begin
              if (ChEstDoneP)
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr; // Free FFT buffer for RxTD
              else if(EqCompDoneP || (ChEstSmthDoneP && n_LTFCount!=FDNDLTF))
              begin
                LTFCount           <= n_LTFCount;
                FDStepState        <= WAIT_FFT;

                ChEstClkEnSR       <= 5'b00000;
                ChEstFirstLTF      <= 1'b0;                         
                ChEstLastLTF       <= 1'b0;                         
                ChEstFrameNDP      <= 1'b0;
                ChEstSmoothFromHSR <= 5'b00000;
                ChEstSmoothEnSR    <= 5'b00000;
                  
                if(n_LTFCount==FDNDLTF)                
                begin
                  FDOSymCount        <= n_FDOSymCount;
                  case(FrameParam1CBW)
                    2'd0   : {ChEstReceptionMode,EqReceptionMode,FDOBW} <= {MODE_20MHZ_HT,MODE_20MHZ_HT,2'd0};
                    2'd1   : {ChEstReceptionMode,EqReceptionMode,FDOBW} <= {MODE_40MHZ_HT,MODE_40MHZ_HT,2'd1};
                    default: {ChEstReceptionMode,EqReceptionMode,FDOBW} <= {MODE_80MHZ_HT,MODE_80MHZ_HT,2'd2}; // VHT80 with 6 pilots
                  endcase  
                  EqModTypeSS1       <= ModTypeLUT[NBPSC_1];
// STBC or 2x2                  EqModTypeSS2       <= ModTypeLUT[NBPSC_1];
                  EqDiscEn           <= 1'b0;
                  
                  deintlv_sigen      <= 1'b0;
                  deintlv_sigsel     <= 2'd0;
                  deintlv_row_fifo_en<= 1'b1;
                  
                  deintlv_nsd        <= FrameParam1NSD;
                  deintlv_iss        <= 2'b00;
                  deintlv_nbpsc      <= NBPSC_1;
                  
                  
                  if(FrameParam8Update && (FrameParam8NSym==16'd0)) // For NDP, VHTSIGB is not processed
                    FDSymState  <= FD_DONE;
                  else
                    FDSymState  <= VHTSIGB; 
                end
                else 
                begin
                  FDStepState <= WAIT_FFT;
                end
              end
            end
          endcase
        end
     
        /***********************************************************************
        * VHTSIGB
        ***********************************************************************/
        VHTSIGB:
        begin
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqPilotData <= 1'b1;     
                EqSTBCEn    <= 1'b0;
                EqLDPCEn    <= 1'b0;
                EqRxNsts    <= 3'd0;                       
                EqRxNss     <= 3'd0;                       
`ifdef RW_MUMIMO_RX_EN
                EqMuMIMOICEn <= 1'b0;
`endif // RW_MUMIMO_RX_EN
                
                EqClkEnSR   <= 5'b11111;
                FDOClkEnSR  <= 5'b11111;
                EqSR        <= 5'b10000;
                FDOEstSR    <= 5'b10000;
                
                FDStepState <= EQU_PILOT; 
              end
              else
                FDStepState <= WAIT_FFT;
            end
            
            EQU_PILOT:
            begin
              if(FDOCompDoneP) 
              begin
                EqPilotData <= 1'b0;
`ifdef RW_MUMIMO_RX_EN
                // Specific case for VHT-SIG-B on H operation with Nss=2
                EqVHTSIGBNss2 <= | FrameParam1NSTS;
                EqVHTSIGBSub  <= (|FrameParam1NSTS) & (FrameParam1MuPreUserNSTS==3'd2) & FrameParam1MuMIMO;
                // Specific case for VHT-SIG-B on user position 3, softbit are inversed
                EqVHTSIGBInv  <= (FrameParam1MuPreUserNSTS==3'd3) & FrameParam1MuMIMO;
`endif // RW_MUMIMO_RX_EN
                EqSR          <= 5'b00001;
                DeIntlvSR     <= 5'b00001;
                FDStepState   <= EQU_DATA;
              end
            end
            
            EQU_DATA:
            begin
              if(!deintlv_busy_int)
              begin
                FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;
                FDOSymCount      <= n_FDOSymCount;
                FDOInitPilotSR   <= 5'b10000;  
                FDOInitKalmanSR  <= 5'b10000;  
                EqModTypeSS1     <= ModTypeLUT[FrameParam1NBPSC0];  
                EqLDPCEn         <= FrameParam1FEC;
                EqSTBCEn         <= FrameParam1STBC;
                EqMHUpdate       <= 1'b1;            // Update mh with new modulation and FEC
// STBC or 2x2                EqModTypeSS2     <= ModTypeLUT[FrameParam1NBPSC0];   
`ifdef RW_MUMIMO_RX_EN
                EqVHTSIGBNss2    <= 1'b0;
                EqVHTSIGBSub     <= 1'b0;
                EqVHTSIGBInv     <= 1'b0;
`endif // RW_MUMIMO_RX_EN
                deintlv_iss      <= 2'b00;
                deintlv_nbpsc    <= FrameParam1NBPSC0;
                if(!FrameParam1VHTFEC)
                begin
                  /* BCC */ 
                  deintlv_fec         <= 1'b0;
                  deintlv_row_fifo_en <= 1'b1;
                end
                else
                begin
                  /* LDPC */
                  deintlv_fec         <= 1'b1;
                  deintlv_row_fifo_en <= 1'b0;
                end
                                
`ifdef RW_MUMIMO_RX_EN
                if (FrameParam1MuMIMO)
                  FDStepState <= WAIT_BD;
                else
                begin
`endif // RW_MUMIMO_RX_EN
                  FDStepState <= WAIT_FFT;
                  FDSymState  <= VHTDATA;
`ifdef RW_MUMIMO_RX_EN
                end
`endif // RW_MUMIMO_RX_EN
              end
            end

`ifdef RW_MUMIMO_RX_EN
            default: /* WAIT_BD*/
            begin
              if(FrameParam2Update)
              begin
                EqModTypeSS1    <= ModTypeLUT[FrameParam2MuNBPSC];  
// STBC or 2x2                EqModTypeSS2    <= ModTypeLUT[FrameParam2MuNBPSC];   
                deintlv_nbpsc   <= FrameParam2MuNBPSC;
//                 FDOStorePreamb  <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over preamble // TODO: align with matlab
               
                FDStepState     <= WAIT_FFT;       
                FDSymState      <= VHTDATA;         
              end
            end
`endif // RW_MUMIMO_RX_EN

          endcase
        end
 
        /***********************************************************************
        * VHTDATA
        ***********************************************************************/
        VHTDATA:
          case(FDStepState)
            WAIT_FFT:
            begin
              if(!FFTBufEmpty)
              begin
                EqMHUpdate    <= 1'b0;
                EqPilotData   <= 1'b1;       // 1 : Pilots equalization. 0: Equalization.
                EqRxNsts      <= 3'd0;                       
                EqRxNss       <= 3'd0;                       
               
                EqClkEnSR     <= 5'b11111;
                FDOClkEnSR    <= 5'b11111;
                EqSR          <= 5'b10000;
                FDOEstSR      <= 5'b10000;
               
                FDStepState   <= EQU_PILOT; 
              end
              else
              begin
                if(EqSTBCFlushDone)
                begin
                  /* gate equalizer clock if stbc flushing done */
                  EqClkEnSR     <= 5'b01111;
                end
              end
            end
            
            EQU_PILOT:
            begin
              if(EqPilotDoneP)
              begin
                if(!equ_flushing_state)
                begin
                  EqClkEnSR <= 5'b00011;
                end
              end
                
              if(FDOCompDoneP) 
              begin
                EqPilotData <= 1'b0;     
                EqRxNsts    <= FrameParam1NSTS;  
                EqRxNss     <= FrameParam1NSS;   
                if(!EqSTBCEn || EqSTBCSymOdd)
                begin
                  DeIntlvSR      <= 5'b10000;
                end
                EqClkEnSR   <= 5'b11111;
                EqSR        <= 5'b10000;
                FDStepState <= EQU_DATA;
              end
            end
            
            EQU_DATA: /* EQU_DATA */ 
            begin
              if(EqDoneP)
                FDOClkEnSR <= 5'b00011;

              if(BDDone)
              begin
                FDResetSR     <= 5'b11111;
                FDOStoreData  <= 1'b1; 
                FDSymState    <= FD_DONE;
              end
              else if(!EqSTBCEn)
              begin
                /* non-stbc */
                if((!deintlv_fec && !deintlv_busy_int) || (deintlv_fec && EqPipeFlushDoneP))
                begin
                  FDOClkEnSR       <= 5'b00000;                                                                              
                  FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;                                                                       
                  FDSymCount       <= n_FDSymCount;                                                                          
                  FDOSymCount      <= n_FDOSymCount;   
                
                  if(LastSymbol)                                                                                             
                  begin                                                                                                      
                    FDOStoreData  <= 1'b1; // flag indicating to FDO it must store the CPESlope value estimated over data    
                    FDSymState    <= FD_DONE;                                                                                
                  end                                                                                                        
                  else                                                                                                       
                  begin 
                    EqClkEnSR     <= 5'b00000;                                                                            
                    FDStepState   <= WAIT_FFT;                                                                               
                  end                                                                                                              
                end           
              end
              else
              begin
                if(!EqSTBCSymOdd)
                begin
                  /* stbc partial even symbol equalization */
                  if(EqPipeFlushDoneP)
                  begin
                    /* stbc, even symbol partial equalization done */
                    EqSTBCSymOdd     <= 1'b1;       
                    FDOClkEnSR       <= 5'b00000;              
                    FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;       
                    FDSymCount       <= n_FDSymCount;          
                    FDOSymCount      <= n_FDOSymCount;                    
                    EqClkEnSR        <= 5'b01111;                                                                            
                    FDStepState      <= WAIT_FFT;                                                                             
                  end
                end
                else
                begin
                  /* stbc even+odd full equalization*/
                  if((!deintlv_fec && !deintlv_busy_int) || (deintlv_fec && EqPipeFlushDoneP))
                  begin
                    EqSTBCSymOdd     <= 1'b0;       
                    DeIntlvSR        <= 5'b00001;           
                    EqSTBCFlush      <= 1'b1;               
                    FDFFTBufRdPtrInt <= n_FDFFTBufRdPtr;       
                    FDSymCount       <= n_FDSymCount;          
                    FDOSymCount      <= n_FDOSymCount;
                    if(LastSymbol)
                      FDStepState <= WAIT_STBC_FLUSH;
                    else
                      FDStepState <= WAIT_FFT;                                                                             
                  end
                end
              end
            end
            
            default: /* WAIT_STBC_FLUSH */
            begin
              if(BDDone)
              begin
                FDResetSR     <= 5'b11111;
                FDOStoreData  <= 1'b1; 
                FDSymState    <= FD_DONE;
              end
              else if(EqSTBCFlushDone)
              begin
                FDOStoreData <= 1'b1; // flag 
                FDSymState   <= FD_DONE;           
              end
            end
            
          endcase

        /***********************************************************************
        * FD_DONE
        ***********************************************************************/
        default:
        begin
          ChEstClkEnSR <= 5'b00000;
          EqClkEnSR    <= 5'b00000;
          FDOClkEnSR   <= 5'b00000;
          FDDone       <= 1'b1;
        end
      endcase
    end
  end
  
  // Specific case of HE MU 40/80 with CHBW20
  assign FDOReceptionMode    = (CfgRegForce20 && (EqReceptionMode == MODE_40MHZ_HE_MU || EqReceptionMode == MODE_80MHZ_HE_MU)) ? MODE_20MHZ_HE_MU :
                                                                                        (EqReceptionMode == MODE_80MHZ_HE_MU)  ? MODE_40MHZ_HE_MU : // TBD : only support 40MHz BW max
                                                                                                                                 EqReceptionMode;
  
  assign ChEstStartP         = ChEstSR[0];
  assign EqStartP            = EqSR[0];         
  assign EqCompStartP        = EqCompSR[0];      
  assign FDOEstStartP        = FDOEstSR[0];
  assign FDOInitPilotShP     = FDOInitPilotSR[0];
  assign FDOInitKalmanP      = FDOInitKalmanSR[0];
  assign FDOInitKalmanSlopeP = FDOInitKalmanSR[0] & (FDOReceptionMode>=4'd6) & ~MidambleSymbolMask;
  assign deintlv_start       = DeIntlvSR[0];
  assign EqRUType            = ChEstRUType;
  assign EqRUIndex           = ChEstRUIndex;
  assign ChEstSmoothEn       = ChEstSmoothEnSR[0];
  assign ChEstSmoothFromH    = ChEstSmoothFromHSR[0]; 
  assign FDReset             = FDResetSR[0]; 

  assign ChEstGClkEn      = ChEstClkEnSR[0];    
  assign FDOGClkEn        = FDOClkEnSR[0];             
  assign EquGClkEn        = EqClkEnSR[0];             
  

  /*****************************************************************************
  * Diag
  *****************************************************************************/
  assign Diag[31:30] = FDSymCount[1:0];
  assign Diag[29:26] = FDFormatMod;
  assign Diag[25:24] = FrameParam1CBW;
  assign Diag[   23] = ChEstDoneP | ChEstSmthDoneP;
  assign Diag[   22] = EqPipeFlushDoneP;
  assign Diag[   21] = EqSR[0];
  assign Diag[   20] = EqCompSR[0];
  assign Diag[   19] = ChEstSR[0];
  assign Diag[   18] = FDOEstSR[0];
  assign Diag[   17] = LastSymbol;
  assign Diag[   16] = FrameParam4LSIG6M;
  assign Diag[   15] = EqQbpskDet;
  assign Diag[   14] = deintlv_busy_int;
  assign Diag[   13] = FDOCompDoneP;
  assign Diag[   12] = EqCompDoneP;
  assign Diag[   11] = bdfd_fifo_full;
  assign Diag[   10] = FFTBufEmpty;
  assign Diag[    9] = FFTFifoFull; 
  assign Diag[ 8: 4] = FDSymState[4:0];
  assign Diag[ 3: 0] = FDStepState; // 1 bit shift TODO doc
  
//pragma coverage off
`ifdef RW_SIMU_ON
  /*****************************************************************************
  * Simu only
  *****************************************************************************/
  reg [50*8:0] FDSymStateStr;
  always @(*)
    case(FDSymState)          
      FD_RESET:                         FDSymStateStr = {"FD_RESET"};      
      LLTF:                             FDSymStateStr = {"LLTF"};      
      LSIG_HELSIG:                      FDSymStateStr = {"LSIG_HELSIG"};
      LDATA0_HTMMSIG1_VHTSIGA1_HERLSIG: FDSymStateStr = {"LDATA0_HTMMSIG1_VHTSIGA1_HERLSIG"};      
      LDATA:                            FDSymStateStr = {"LDATA"}; 
      HTSIG2:                           FDSymStateStr = {"HTSIG2"};
      HTDLTF:                           FDSymStateStr = {"HTDLTF"};
`ifdef RW_NX_DERIV_NESS_EN
      HTELTF:                   FDSymStateStr = {"HTELTF"}; 
`endif
      HTDATA:                   FDSymStateStr = {"HTDATA"}; 
      LDATA1_VHTSIGA2:          FDSymStateStr = {"LDATA1_VHTSIGA2"};      
      VHTLTF:                   FDSymStateStr = {"VHTLTF"};    
      VHTSIGB:                  FDSymStateStr = {"VHTSIGB"};   
      VHTDATA:                  FDSymStateStr = {"VHTDATA"};   
      FD_DONE:                  FDSymStateStr = {"FD_DONE"};   
      HESIGA:                   FDSymStateStr = {"HESIGA"};   
      HELTF:                    FDSymStateStr = {"HELTF"};   
      HEDATA:                   FDSymStateStr = {"HEDATA"};   
      HESIGB:                   FDSymStateStr = {"HESIGB"};   
      default:                  FDSymStateStr = {"UNKNOWN"};   
    endcase                   

  reg [50*8:0] FDStepStateStr;
  always @(*)
    case(FDStepState)  
      WAIT_FFT:        FDStepStateStr = {"WAIT_FFT"};    
      ESTIM_H1:        FDStepStateStr = {"ESTIM_H1"};   
      ESTIM_H2:        FDStepStateStr = {"ESTIM_H2"};   
      EQU_PILOT:       FDStepStateStr = {"EQU_PILOT"}; 
      EQU_DATA:        FDStepStateStr = {"EQU_DATA"};  
      WAIT_BD:         FDStepStateStr = {"WAIT_BD"};   
      SMOOTH_H1:       FDStepStateStr = {"SMOOTH_H1"};  
      SMOOTH_H2:       FDStepStateStr = {"SMOOTH_H2"};  
      WAIT_STBC_FLUSH: FDStepStateStr = {"WAIT_STBC_FLUSH"};  
      default:         FDStepStateStr = {"UNKNOWN"};           
    endcase                   
`endif
//pragma coverage on

endmodule
`default_nettype wire
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
