/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      :
* Simulation Notes :
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module equalizer_genindex
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire         clk,
  input  wire         rst_n,

  /*****************************************************************************
  * control
  *****************************************************************************/
  input  wire         enable,
  input  wire         stride,
  input  wire         hesigb_PAPR_en,
  input  wire  [3:0]  reception_mode,
  input  wire  [2:0]  ru_type,
  input  wire  [5:0]  ru_index,
  input  wire         pilot,
  input  wire         stbc,
  input  wire         stbc_sym_odd,
  input  wire         dcm_en,

  input  wire         ldpc_fifo_en,
  input  wire         ldpc_fifo_ready,
  input  wire         row_fifo_en,
  input  wire  [2:0]  row_fifo_rdptr,

  input  wire         start,
  input  wire         stbc_flush,

  /*****************************************************************************
  * fft address generation
  *****************************************************************************/
  output wire  [9:0]  f_index,
  output wire         f_last,
  output wire         f_valid,

  /*****************************************************************************
  * h address generation
  *****************************************************************************/
  output wire  [9:0]  h_index,
  output wire         h_odd,
  output wire         h_papr,
  output wire  [2:0]  h_pilot_count,
  output wire         h_last,
  output wire         h_valid,

  /*****************************************************************************
  * stbc odd symbol
  *****************************************************************************/
  output wire         stbc_sb_ren,
  input  wire         stbc_sb_peq_dist
);
  /*****************************************************************************
  * declaration
  *****************************************************************************/
`ifdef RW_EQU_MAX_PILOT_6
  localparam        NPSC2  = 5'd2,
                    NPSC4  = 5'd4,
                    NPSC6  = 5'd6,
                    NPSC8  = 5'd6,
                    NPSC16 = 5'd6;
`else
  localparam        NPSC2  = 5'd2,
                    NPSC4  = 5'd4,
                    NPSC6  = 5'd6,
                    NPSC8  = 5'd8,
                    NPSC16 = 5'd16;
`endif

  /* constants */
  localparam        MHZ20_LEG  = 4'd0,
                    MHZ40_DUP  = 4'd1,
                    MHZ20_HT   = 4'd2,
                    MHZ40_HT   = 4'd3,
                    MHZ40_DUP2 = 4'd4,
                    MHZ80_VHT  = 4'd5,
                    MHZ20_HESU = 4'd6,
                    MHZ20_HEMU = 4'd7,
                    MHZ40_HESU = 4'd8,
                    MHZ40_HEMU = 4'd9,
                    MHZ80_HESU = 4'd10,
                    MHZ80_HEMU = 4'd11;

  // RU type
  localparam  [2:0] RU26  = 3'd0,
                    RU52  = 3'd1,
                    RU106 = 3'd2,
                    RU242 = 3'd3,
                    RU484 = 3'd4,
                    RU996 = 3'd5;

  // FSM States
  localparam        S_IDLE       = 2'd0,
                    S_WAIT_FIFO  = 2'd1,
                    S_PUSH_ROW   = 2'd2,
                    S_PUSH_PILOT = 2'd3;

  // HESIG-B PAPR index lut
  wire              papr_mhz20_hesigb[0:1][0:63];
  wire              papr_mhz20_hesigb_dcm[0:1][0:63];

  /* parameters */
  reg         [4:0] nrow;
  reg         [5:0] ncol;
  reg         [9:0] ndsc;
  reg         [4:0] npsc;
  wire        [9:0] dcm_offset;
  wire              dcm_active;

  /* tables */
  reg       [198:0] entry;
  wire      [198:0] nsd48;
  wire      [198:0] nsd52;
  wire      [198:0] nsd108;
  wire      [198:0] nsd234;

  wire      [198:0] ru26_20[0:15];
  wire      [198:0] ru52_20[0: 7];
  wire      [198:0] ru106_20[0: 3];
  wire      [198:0] ru242_20[0: 1];

  wire      [198:0] ru26_40[0:31];
  wire      [198:0] ru52_40[0:15];
  wire      [198:0] ru106_40[0: 7];
  wire      [198:0] ru242_40[0: 3];
  wire      [198:0] ru484_40[0: 1];

  wire      [198:0] ru26_80[0:63];
  wire      [198:0] ru52_80[0:31];
  wire      [198:0] ru106_80[0:15];
  wire      [198:0] ru242_80[0: 7];
  wire      [198:0] ru484_80[0: 3];
  wire      [198:0] ru996_80[0: 1];

`ifdef RW_EQU_MAX_PILOT_6
  wire      [4:0]   p8to6[0:7];
  wire      [4:0]   p16to6[0:7];
`endif

  /* Start index, Pilot and DC subcarriers comparators signals */
  wire signed [9:0] si;

  wire       [15:0] p_ch;
  wire        [9:0] p_cp[0:15];
  wire       [15:0] p_cg;

  wire              dc_ch;
  wire        [9:0] dc_cp;
  wire        [2:0] dc_cg;
  
  reg         [4:0] p_offset;
  
  /* Row and columns counters */
  reg        [9:0]  pos;
  reg        [4:0]  row;
  reg        [5:0]  col;
  wire       [4:0]  n_row;
  wire       [5:0]  n_col;
  wire              n_row_last;
  wire              n_col_last;

  /* FSM */
  reg        [ 5:0] row_fifo_max;
  reg        [ 1:0] sc_state;
  reg        [ 5:0] row_fifo_count;
  wire       [ 5:0] n_row_fifo_count;
  reg        [ 2:0] fifo_row_rqptr;
  wire       [ 2:0] n_fifo_row_rqptr;
  wire       [ 9:0] sc_max;

  reg        [ 1:0] sc_state2;
  reg               sc_last2;
  reg               sc_valid2;
  reg        [ 9:0] sc_count2;
  reg        [ 9:0] sc_max2;
  wire       [ 9:0] n_sc_count2;
  reg        [ 5:0] row_fifo_count2;
  wire       [ 5:0] n_row_fifo_count2;

  /* Flow control */
  wire              row_fifo_full;

  /* haddr */
  wire              n_sc_odd;
  wire signed [9:0] n_sc_index;
  wire        [9:0] n_sc_count;
  reg         [9:0] sc_count;
  wire              n_sc_papr;

  reg signed  [9:0] sc_index,sc_index_1t,sc_index_2t,sc_index_3t,sc_index_4t,sc_index_5t,sc_index_6t,sc_index_7t,sc_index_8t,sc_index_9t,sc_index_10t,sc_index_11t;
  reg               sc_odd,sc_odd_1t,sc_odd_2t,sc_odd_3t,sc_odd_4t,sc_odd_5t,sc_odd_6t,sc_odd_7t,sc_odd_8t,sc_odd_9t,sc_odd_10t,sc_odd_11t;

  reg               sc_papr,sc_papr_1t,sc_papr_2t,sc_papr_3t,sc_papr_4t,sc_papr_5t,sc_papr_6t,sc_papr_7t,sc_papr_8t,sc_papr_9t,sc_papr_10t,sc_papr_11t;
  //TODO sc_pilot count should be on 4 bits when max number pilot supported will be 16
  reg         [2:0] sc_pilot_count,sc_pilot_count_1t,sc_pilot_count_2t,sc_pilot_count_3t,sc_pilot_count_4t,sc_pilot_count_5t,sc_pilot_count_6t,sc_pilot_count_7t,sc_pilot_count_8t,sc_pilot_count_9t,sc_pilot_count_10t,sc_pilot_count_11t;
  reg               sc_last,sc_last_1t,sc_last_2t,sc_last_3t,sc_last_4t,sc_last_5t,sc_last_6t,sc_last_7t,sc_last_8t,sc_last_9t,sc_last_10t,sc_last_11t;
  reg               sc_valid,sc_valid_1t,sc_valid_2t,sc_valid_3t,sc_valid_4t,sc_valid_5t,sc_valid_6t,sc_valid_7t,sc_valid_8t,sc_valid_9t,sc_valid_10t,sc_valid_11t;

  /*****************************************************************************
  * Subcarrier parameter generation
  *****************************************************************************/
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
      {ndsc, npsc, nrow, ncol} <= 26'b0;
    else if(!enable)
      {ndsc, npsc, nrow, ncol} <= 26'b0;
    else if(start)
     case(reception_mode)
       MHZ20_LEG:     {ndsc, npsc, nrow, ncol} <= {10'd48 , NPSC4,              5'd3,  6'd16};
       MHZ20_HT:      {ndsc, npsc, nrow, ncol} <= {10'd52 , NPSC4,  (!dcm_en)?{ 5'd4,  6'd13}: {5'd2,  6'd13}};
       MHZ40_HT:      {ndsc, npsc, nrow, ncol} <= {10'd108, NPSC6,              5'd6,  6'd18};
       MHZ80_VHT:     {ndsc, npsc, nrow, ncol} <= {10'd234, NPSC8,              5'd9,  6'd26};
       MHZ20_HESU:    {ndsc, npsc, nrow, ncol} <= {10'd234, NPSC8,  (!dcm_en)?{ 5'd9,  6'd26}:{ 5'd9,  6'd13}};
       MHZ40_HESU:    {ndsc, npsc, nrow, ncol} <= {10'd468, NPSC16, (!dcm_en)?{ 5'd12, 6'd39}:{ 5'd9,  6'd26}};
       MHZ80_HESU:    {ndsc, npsc, nrow, ncol} <= {10'd980, NPSC16, (!dcm_en)?{ 5'd20, 6'd49}:{ 5'd14, 6'd35}};
       MHZ20_HEMU,MHZ40_HEMU,MHZ80_HEMU:
         case (ru_type)
             RU26:    {ndsc, npsc, nrow, ncol} <= {10'd24 , NPSC2,  (!dcm_en)?{ 5'd3,  6'd8} :{ 5'd3,  6'd4 }};
             RU52:    {ndsc, npsc, nrow, ncol} <= {10'd48 , NPSC4,  (!dcm_en)?{ 5'd3,  6'd16}:{ 5'd3,  6'd8 }};
             RU106:   {ndsc, npsc, nrow, ncol} <= {10'd102, NPSC4,  (!dcm_en)?{ 5'd6,  6'd17}:{ 5'd3,  6'd17}};
             RU242:   {ndsc, npsc, nrow, ncol} <= {10'd234, NPSC8,  (!dcm_en)?{ 5'd9,  6'd26}:{ 5'd9,  6'd13}};
             RU484:   {ndsc, npsc, nrow, ncol} <= {10'd468, NPSC16, (!dcm_en)?{ 5'd12, 6'd39}:{ 5'd9,  6'd26}};
             default: {ndsc, npsc, nrow, ncol} <= {10'd980, NPSC16, (!dcm_en)?{ 5'd20, 6'd49}:{ 5'd14, 6'd35}};
         endcase
     default:{ndsc, npsc, nrow, ncol} <=  26'b0;
     endcase
  end

    /* number of sub-carrier per row */
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
      row_fifo_max <= 6'd0;
    else if(!enable)
      row_fifo_max <= 6'd0;
    else if(start)
      case(reception_mode)
        MHZ20_LEG:  row_fifo_max <= 6'd16;
        MHZ20_HT:   row_fifo_max <= 6'd13;
        MHZ40_HT:   row_fifo_max <= 6'd18;
        MHZ80_VHT:  row_fifo_max <= 6'd26;
        MHZ20_HESU: row_fifo_max <= 6'd26;
        MHZ40_HESU: row_fifo_max <= 6'd26; /* ldpc, push per group of 26sc */
        MHZ80_HESU: row_fifo_max <= 6'd26; /* ldpc, push per group of 26sc */ //TODO : TO BE CONFIRMED
        default: /* HEMU */
        begin
          case(ru_type)
            RU26:    row_fifo_max <= 6'd8;
            RU52:    row_fifo_max <= 6'd16;
            RU106:   row_fifo_max <= (!dcm_en)?6'd17:6'd34;
            default: row_fifo_max <= 6'd26;
          endcase
        end
      endcase
  end

  // DCM offset = NSD/2
  assign dcm_offset = {1'b0,ndsc[9:1]};
  assign dcm_active = sc_count[0] & ~pilot & dcm_en;
  assign sc_max     = (pilot) ? npsc : ndsc;

  /*****************************************************************************
  * Pilot and DC inputs for comparators generation
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
      entry <= 199'b0;
    else if(!enable)
      entry <= 199'b0;
    else if(start || pilot)
      case(reception_mode)
        MHZ20_LEG:     entry <= nsd48;
        MHZ20_HT:      entry <= nsd52;
        MHZ40_HT:      entry <= nsd108;
        MHZ80_VHT:     entry <= nsd234;
        MHZ20_HESU:    entry <= ru242_20[1]; /* RU_242 */
        MHZ20_HEMU:
          case (ru_type)
            RU26:      entry <=  ru26_20[ru_index[3:0]];
            RU52:      entry <=  ru52_20[ru_index[2:0]];
            RU106:     entry <= ru106_20[ru_index[1:0]];
            default:   entry <= ru242_20[ru_index[  0]]; /* RU_242 */
          endcase
        MHZ40_HESU:    entry <= ru484_40[1]; /* RU_484 */
        MHZ40_HEMU:
          case (ru_type)
            RU26:      entry <=  ru26_40[ru_index[4:0]];
            RU52:      entry <=  ru52_40[ru_index[3:0]];
            RU106:     entry <= ru106_40[ru_index[2:0]];
            RU242:     entry <= ru242_40[ru_index[1:0]];
            default:   entry <= ru484_40[ru_index[  0]]; /* RU_484 */
          endcase
        MHZ80_HESU:    entry <= ru996_80[1]; /* RU_996 */
        MHZ80_HEMU:
          case (ru_type)
            RU26:      entry <=  ru26_80[ru_index[5:0]];
            RU52:      entry <=  ru52_80[ru_index[4:0]];
            RU106:     entry <= ru106_80[ru_index[3:0]];
            RU242:     entry <= ru242_80[ru_index[2:0]];
            RU484:     entry <= ru484_80[ru_index[1:0]];
            default:   entry <= ru996_80[1]; /* RU_996 */
          endcase
        default:       entry <= 199'b0;
      endcase
  end

  assign {si,dc_cp    ,dc_cg ,
              p_cp[0] , p_cg[0] ,
              p_cp[1] , p_cg[1] ,
              p_cp[2] , p_cg[2] ,
              p_cp[3] , p_cg[3] ,
              p_cp[4] , p_cg[4] ,
              p_cp[5] , p_cg[5] ,
              p_cp[6] , p_cg[6] ,
              p_cp[7] , p_cg[7] ,
              p_cp[8] , p_cg[8] ,
              p_cp[9] , p_cg[9] ,
              p_cp[10], p_cg[10],
              p_cp[11], p_cg[11],
              p_cp[12], p_cg[12],
              p_cp[13], p_cg[13],
              p_cp[14], p_cg[14],
              p_cp[15], p_cg[15]} = entry;

  /* segment position comparators */
  assign  p_ch[0]  =  pos >=  p_cp[0];
  assign  p_ch[1]  =  pos >=  p_cp[1];
  assign  p_ch[2]  =  pos >=  p_cp[2];
  assign  p_ch[3]  =  pos >=  p_cp[3];
  assign  p_ch[4]  =  pos >=  p_cp[4];
  assign  p_ch[5]  =  pos >=  p_cp[5];
  assign  p_ch[6]  =  pos >=  p_cp[6];
  assign  p_ch[7]  =  pos >=  p_cp[7];
  assign  p_ch[8]  =  pos >=  p_cp[8];
  assign  p_ch[9]  =  pos >=  p_cp[9];
  assign  p_ch[10] =  pos >=  p_cp[10];
  assign  p_ch[11] =  pos >=  p_cp[11];
  assign  p_ch[12] =  pos >=  p_cp[12];
  assign  p_ch[13] =  pos >=  p_cp[13];
  assign  p_ch[14] =  pos >=  p_cp[14];
  assign  p_ch[15] =  pos >=  p_cp[15];
  assign dc_ch     =  pos >= dc_cp;

`ifdef RW_EQU_MAX_PILOT_6
  assign p8to6[0]  = 5'd0;
  assign p8to6[1]  = 5'd1;
  assign p8to6[2]  = 5'd3;
  assign p8to6[3]  = 5'd4;
  assign p8to6[4]  = 5'd6;
  assign p8to6[5]  = 5'd7;
  assign p8to6[6]  = 5'd0;
  assign p8to6[7]  = 5'd0;

  assign p16to6[0] = 5'd1;
  assign p16to6[1] = 5'd3;
  assign p16to6[2] = 5'd6;
  assign p16to6[3] = 5'd9;
  assign p16to6[4] = 5'd12;
  assign p16to6[5] = 5'd14;
  assign p16to6[6] = 5'd0;
  assign p16to6[7] = 5'd0;
`endif

  always @(*)
  begin:b_pos
    reg [9:0] v;
    if(pilot)
    `ifdef RW_EQU_MAX_PILOT_6
      case(reception_mode)
        MHZ20_LEG,MHZ20_HT,MHZ40_HT: v = p_cp[sc_count[3:0]];
        MHZ80_VHT,MHZ20_HESU:        v = p_cp[p8to6[sc_count[2:0]]];
        MHZ40_HESU,MHZ80_HESU:       v = p_cp[p16to6[sc_count[2:0]]];
        MHZ20_HEMU,MHZ40_HEMU,MHZ80_HEMU:
          case (ru_type)
              RU26,RU52,RU106:       v = p_cp[sc_count[3:0]];
              RU242:                 v = p_cp[p8to6[sc_count[2:0]]];
              default:               v = p_cp[p16to6[sc_count[2:0]]];
          endcase
      default:                       v = p_cp[sc_count[3:0]];
      endcase

    `else
      v = p_cp[sc_count[3:0]];
    `endif
    else if(stride)
      v = {4'b0,col&{6{nrow[0]}}}      +
          {3'b0,col&{6{nrow[1]}},1'b0} +
          {2'b0,col&{6{nrow[2]}},2'b0} +
          {1'b0,col&{6{nrow[3]}},3'b0} +
          {     col&{6{nrow[4]}},4'b0} +
          {5'b0,row};
    else if(dcm_en)
      v = {1'b0,sc_count[9:1]};
    else
      v = sc_count;

    pos = v + (dcm_offset & {10{dcm_active}});
  end
  
  always @*
  begin
    case(p_cg & p_ch)
      16'h0001 : p_offset = pilot ? 5'd0  : 5'd1;
      16'h0003 : p_offset = pilot ? 5'd1  : 5'd2;
      16'h0007 : p_offset = pilot ? 5'd2  : 5'd3;
      16'h000F : p_offset = pilot ? 5'd3  : 5'd4;
      16'h001F : p_offset = pilot ? 5'd4  : 5'd5;
      16'h003F : p_offset = pilot ? 5'd5  : 5'd6;
      16'h007F : p_offset = pilot ? 5'd6  : 5'd7;
      16'h00FF : p_offset = pilot ? 5'd7  : 5'd8;
      16'h01FF : p_offset = pilot ? 5'd8  : 5'd9;
      16'h03FF : p_offset = pilot ? 5'd9  : 5'd10;
      16'h07FF : p_offset = pilot ? 5'd10 : 5'd11;
      16'h0FFF : p_offset = pilot ? 5'd11 : 5'd12;
      16'h1FFF : p_offset = pilot ? 5'd12 : 5'd13;
      16'h3FFF : p_offset = pilot ? 5'd13 : 5'd14;
      16'h7FFF : p_offset = pilot ? 5'd14 : 5'd15;
      16'hFFFF : p_offset = pilot ? 5'd15 : 5'd16;
      default  : p_offset = 5'd0;
    endcase
  end
  
  
  /* index compuation */
  assign n_sc_index = pos                       + /* position offset  */
                      si                        + /* first index      */
                      {7'b0,dc_cg & {3{dc_ch}}} + /* comparator DC    */
                      {5'b0,p_offset};            /* comparator pilot */

  assign n_sc_papr = hesigb_PAPR_en & ~pilot & (pos[5:0] >= 6'd26) & pos[0];
  
  assign n_sc_odd = dcm_en & pos[0];

  /*****************************************************************************
  * Subcarrier index generation
  *****************************************************************************/

  assign n_row             = row + 5'd1;
  assign n_col             = col + 6'd1;

  assign n_row_last        = n_row==nrow;
  assign n_col_last        = n_col==ncol;

  assign row_fifo_full     = row_fifo_rdptr=={~fifo_row_rqptr[2],fifo_row_rqptr[1:0]};

  assign n_sc_count        = sc_count        + 10'd1;
  assign n_row_fifo_count  = row_fifo_count  + 6'd1;
  assign n_fifo_row_rqptr  = fifo_row_rqptr  + 3'd1;
  assign n_row_fifo_count2 = row_fifo_count2 + 6'd1;
  assign n_sc_count2       = sc_count2       + 10'd1;

  assign stbc_sb_ren       = sc_valid2;

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      row              <= 5'd0;
      col              <= 6'd0;

      sc_index         <= 10'd0;
      sc_odd           <= 1'b0;
      sc_papr          <= 1'd0;
      sc_pilot_count   <= 3'b0;
      sc_last          <= 1'b0;
      sc_valid         <= 1'b0;
      sc_count         <= 10'b0;

      fifo_row_rqptr   <= 3'd0;
      row_fifo_count   <= 6'd0;

      sc_state         <= S_IDLE;

      sc_max2          <= 10'b0;
      sc_state2        <= 2'd0;
      sc_last2         <= 1'b0;
      sc_valid2        <= 1'b0;
      sc_count2        <= 9'd0;
      row_fifo_count2  <= 6'd0;
    end
    else if(!enable)
    begin
      row              <= 5'd0;
      col              <= 6'd0;

      sc_index         <= 10'd0;
      sc_odd           <= 1'b0;
      sc_papr          <= 1'd0;
      sc_pilot_count   <= 3'b0;
      sc_last          <= 1'b0;
      sc_valid         <= 1'b0;
      sc_count         <= 10'b0;

      fifo_row_rqptr   <= 3'd0;
      row_fifo_count   <= 6'd0;

      sc_state         <= S_IDLE;

      sc_max2          <= 10'b0;
      sc_state2        <= 2'd0;
      sc_last2         <= 1'b0;
      sc_valid2        <= 1'b0;
      sc_count2        <= 9'd0;
      row_fifo_count2  <= 6'd0;
    end
    else
    begin
    /* fsm non-stbc or stbc with partial equalization of even symbol */
      case(sc_state)
        /***********************************************************************
        * IDLE
        ***********************************************************************/
        S_IDLE:
        begin
          if(start)
          begin
            if(!pilot)
            begin
              row            <= 5'd0;
              col            <= 6'd0;
              sc_count       <= 10'd0;

              sc_state       <= S_WAIT_FIFO;
              sc_max2        <= sc_max; /* keep only the number of data subcarrier max for sc_state2 */
            end
            else
            begin
              sc_index       <= n_sc_index;
              sc_odd         <= n_sc_odd;
              sc_papr        <= n_sc_papr;
              sc_pilot_count <= sc_count[2:0]; // rank of the pilot
              sc_last        <= n_sc_count==sc_max;
              sc_valid       <= 1'b1;
              sc_count       <= n_sc_count;
              sc_state       <= S_PUSH_PILOT;
            end
          end
          else
          begin
            sc_index <= 10'sd0;
            sc_last  <= 1'b0;
            sc_valid <= 1'b0;
          end
        end

        /***********************************************************************
        * WAIT_FIFO
        ***********************************************************************/
        S_WAIT_FIFO:
        begin
          if((!row_fifo_en && !ldpc_fifo_en)   || // no flow control
             (row_fifo_en  && !row_fifo_full)  || // flow controlled by row   (bcc)
             (ldpc_fifo_en && ldpc_fifo_ready) || // flow controlled by ready (ldpc)
             (stbc         && !stbc_sym_odd))
          begin
            if(dcm_active || !dcm_en)
            begin
              if(n_col_last)
              begin
                col <= 6'd0;
                if(n_row_last)
                begin
                  row      <= 5'd0;
                  sc_last  <= 1'b1;
                end
                else
                begin
                  row      <= n_row;
                  sc_last  <= 1'b0;
                end
              end
              else
              begin
                col        <= n_col;
              end
            end
            sc_count       <= n_sc_count;
            sc_index       <= n_sc_index;
            sc_odd         <= n_sc_odd;
            sc_papr        <= n_sc_papr;
            sc_valid       <= 1'b1;
            row_fifo_count <= 6'd0;
            sc_state       <= S_PUSH_ROW;
          end
        end

        /***********************************************************************
        * PUSH_ROW
        ***********************************************************************/
        S_PUSH_ROW:
        begin
          if((!stbc || stbc_sym_odd) && row_fifo_en && sc_valid && n_row_fifo_count==row_fifo_max)
          begin
            fifo_row_rqptr   <= n_fifo_row_rqptr;
          end

          if(sc_last)
          begin
            sc_index       <= 10'd0;
            sc_odd         <= 1'b0;
            sc_papr        <= 1'b0;
            sc_pilot_count <= 3'd0;
            sc_last        <= 1'b0;
            sc_valid       <= 1'b0;

            sc_count       <= 10'd0;
            sc_state       <= S_IDLE;

          end
          else if(((!stbc || stbc_sym_odd) &&  (row_fifo_en && n_row_fifo_count==row_fifo_max)) ||
                  ((!stbc || stbc_sym_odd) &&  (ldpc_fifo_en && !ldpc_fifo_ready))    )
          begin
            sc_valid       <= 1'b0;
            sc_state       <= S_WAIT_FIFO;
          end
          else
          begin
            if(stbc_sb_peq_dist)
            begin
              sc_valid       <= 1'b0;
            end
            else
            begin
              if(dcm_active || !dcm_en)
              begin
                if(n_col_last)
                begin
                  col <= 6'd0;
                  if(n_row_last)
                  begin
                    row      <= 5'd0;
                    sc_last  <= 1'b1;
                  end
                  else
                  begin
                    row      <= n_row;
                    sc_last  <= 1'b0;
                  end
                end
                else
                begin
                  col        <= n_col;
                  sc_count   <= n_sc_count;
                end
              end
              sc_count       <= n_sc_count;
              sc_index       <= n_sc_index;
              sc_valid       <= 1'b1;
              sc_odd         <= n_sc_odd;
              sc_papr        <= n_sc_papr;
              row_fifo_count <= n_row_fifo_count;
            end
          end
        end

        /***********************************************************************
        * PUSH_PILOT
        ***********************************************************************/
        default:
        begin
          if(sc_last)
          begin
            sc_index       <= 10'd0;
            sc_odd         <= 1'b0;
            sc_papr        <= 1'b0;
            sc_pilot_count <= 3'd0;
            sc_last        <= 1'b0;
            sc_valid       <= 1'b0;
            sc_count       <= 10'd0;
            sc_state       <= S_IDLE;
          end
          else
          begin
            sc_index       <= n_sc_index;
            sc_odd         <= 1'b0;
            sc_papr        <= n_sc_papr;
            sc_pilot_count <= sc_count[2:0]; /* rank of the pilot */
            sc_last        <= n_sc_count==sc_max;
            sc_valid       <= 1'b1;
            sc_count       <= n_sc_count;
          end
        end
      endcase


      /* stbc, flush odd symbol sb stored in stbc memory */
      case(sc_state2)
        /***********************************************************************
        * IDLE
        ***********************************************************************/
        S_IDLE:
        begin
          if(stbc_flush)
          begin
            sc_state2 <= S_WAIT_FIFO;
          end
        end
        /***********************************************************************
        * WAIT_FIFO
        ***********************************************************************/
        S_WAIT_FIFO:
        begin
          if((row_fifo_en  && !row_fifo_full)  || // flow controlled by row   (bcc)
             (ldpc_fifo_en && ldpc_fifo_ready))   // flow controlled by ready (ldpc)
          begin
             sc_last2      <= n_sc_count2==sc_max2;
             sc_valid2     <= 1'b1;
             sc_count2     <= n_sc_count2;
             row_fifo_count2 <= 6'd0;
             sc_state2     <= S_PUSH_ROW;
          end
        end

        /***********************************************************************
        * PUSH_ROW
        ***********************************************************************/
        S_PUSH_ROW:
        begin
          if(sc_valid2 && (n_row_fifo_count2==row_fifo_max))
          begin
            fifo_row_rqptr   <= n_fifo_row_rqptr;
          end

          if(sc_last2)
          begin
            sc_last2   <= 1'b0;
            sc_valid2  <= 1'b0;
            sc_count2  <= 10'd0;
            sc_state2  <= S_IDLE;
          end
          else if(  (row_fifo_en && n_row_fifo_count2==row_fifo_max) ||
                   (ldpc_fifo_en && !ldpc_fifo_ready)    )
          begin
            sc_valid2     <= 1'b0;
            sc_state2     <= S_WAIT_FIFO;
          end
          else
          begin
            sc_last2      <= n_sc_count2==sc_max2;
            sc_valid2     <= 1'b1;
            sc_count2     <= n_sc_count2;
            row_fifo_count2 <= n_row_fifo_count2;
          end
        end
      endcase
    end
  end

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      { sc_index_1t, sc_odd_1t, sc_papr_1t, sc_pilot_count_1t, sc_last_1t, sc_valid_1t} <= 17'd0;
      { sc_index_2t, sc_odd_2t, sc_papr_2t, sc_pilot_count_2t, sc_last_2t, sc_valid_2t} <= 17'd0;
      { sc_index_3t, sc_odd_3t, sc_papr_3t, sc_pilot_count_3t, sc_last_3t, sc_valid_3t} <= 17'd0;
      { sc_index_4t, sc_odd_4t, sc_papr_4t, sc_pilot_count_4t, sc_last_4t, sc_valid_4t} <= 17'd0;
      { sc_index_5t, sc_odd_5t, sc_papr_5t, sc_pilot_count_5t, sc_last_5t, sc_valid_5t} <= 17'd0;
      { sc_index_6t, sc_odd_6t, sc_papr_6t, sc_pilot_count_6t, sc_last_6t, sc_valid_6t} <= 17'd0;
      { sc_index_7t, sc_odd_7t, sc_papr_7t, sc_pilot_count_7t, sc_last_7t, sc_valid_7t} <= 17'd0;
      { sc_index_8t, sc_odd_8t, sc_papr_8t, sc_pilot_count_8t, sc_last_8t, sc_valid_8t} <= 17'd0;
      { sc_index_9t, sc_odd_9t, sc_papr_9t, sc_pilot_count_9t, sc_last_9t, sc_valid_9t} <= 17'd0;
      {sc_index_10t,sc_odd_10t,sc_papr_10t,sc_pilot_count_10t,sc_last_10t,sc_valid_10t} <= 17'd0;
      {sc_index_11t,sc_odd_11t,sc_papr_11t,sc_pilot_count_11t,sc_last_11t,sc_valid_11t} <= 17'd0;
    end
    else if(!enable)
    begin
      { sc_index_1t, sc_odd_1t, sc_papr_1t, sc_pilot_count_1t, sc_last_1t, sc_valid_1t} <= 17'd0;
      { sc_index_2t, sc_odd_2t, sc_papr_2t, sc_pilot_count_2t, sc_last_2t, sc_valid_2t} <= 17'd0;
      { sc_index_3t, sc_odd_3t, sc_papr_3t, sc_pilot_count_3t, sc_last_3t, sc_valid_3t} <= 17'd0;
      { sc_index_4t, sc_odd_4t, sc_papr_4t, sc_pilot_count_4t, sc_last_4t, sc_valid_4t} <= 17'd0;
      { sc_index_5t, sc_odd_5t, sc_papr_5t, sc_pilot_count_5t, sc_last_5t, sc_valid_5t} <= 17'd0;
      { sc_index_6t, sc_odd_6t, sc_papr_6t, sc_pilot_count_6t, sc_last_6t, sc_valid_6t} <= 17'd0;
      { sc_index_7t, sc_odd_7t, sc_papr_7t, sc_pilot_count_7t, sc_last_7t, sc_valid_7t} <= 17'd0;
      { sc_index_8t, sc_odd_8t, sc_papr_8t, sc_pilot_count_8t, sc_last_8t, sc_valid_8t} <= 17'd0;
      { sc_index_9t, sc_odd_9t, sc_papr_9t, sc_pilot_count_9t, sc_last_9t, sc_valid_9t} <= 17'd0;
      {sc_index_10t,sc_odd_10t,sc_papr_10t,sc_pilot_count_10t,sc_last_10t,sc_valid_10t} <= 17'd0;
      {sc_index_11t,sc_odd_11t,sc_papr_11t,sc_pilot_count_11t,sc_last_11t,sc_valid_11t} <= 17'd0;
    end
    else
    begin
      /* delay line */
      if(pilot)
      begin
      { sc_index_1t, sc_odd_1t, sc_papr_1t, sc_pilot_count_1t, sc_last_1t, sc_valid_1t} <= 17'd0;
      { sc_index_2t, sc_odd_2t, sc_papr_2t, sc_pilot_count_2t, sc_last_2t, sc_valid_2t} <= 17'd0;
      { sc_index_3t, sc_odd_3t, sc_papr_3t, sc_pilot_count_3t, sc_last_3t, sc_valid_3t} <= 17'd0;
      { sc_index_4t, sc_odd_4t, sc_papr_4t, sc_pilot_count_4t, sc_last_4t, sc_valid_4t} <= 17'd0;
      { sc_index_5t, sc_odd_5t, sc_papr_5t, sc_pilot_count_5t, sc_last_5t, sc_valid_5t} <= 17'd0;
      { sc_index_6t, sc_odd_6t, sc_papr_6t, sc_pilot_count_6t, sc_last_6t, sc_valid_6t} <= 17'd0;
      { sc_index_7t, sc_odd_7t, sc_papr_7t, sc_pilot_count_7t, sc_last_7t, sc_valid_7t} <= 17'd0;
      { sc_index_8t, sc_odd_8t, sc_papr_8t, sc_pilot_count_8t, sc_last_8t, sc_valid_8t} <= 17'd0;
      { sc_index_9t, sc_odd_9t, sc_papr_9t, sc_pilot_count_9t, sc_last_9t, sc_valid_9t} <= 17'd0;
      {sc_index_10t,sc_odd_10t,sc_papr_10t,sc_pilot_count_10t,sc_last_10t,sc_valid_10t} <= 17'd0;
      {sc_index_11t,sc_odd_11t,sc_papr_11t,sc_pilot_count_11t,sc_last_11t,sc_valid_11t} <= 17'd0;
      end
      else
      begin
        { sc_index_1t, sc_odd_1t, sc_papr_1t, sc_pilot_count_1t, sc_last_1t, sc_valid_1t} <= { sc_index   , sc_odd,    sc_papr,    sc_pilot_count,    sc_last,    sc_valid};
        { sc_index_2t, sc_odd_2t, sc_papr_2t, sc_pilot_count_2t, sc_last_2t, sc_valid_2t} <= { sc_index_1t, sc_odd_1t, sc_papr_1t, sc_pilot_count_1t, sc_last_1t, sc_valid_1t};
        { sc_index_3t, sc_odd_3t, sc_papr_3t, sc_pilot_count_3t, sc_last_3t, sc_valid_3t} <= { sc_index_2t, sc_odd_2t, sc_papr_2t, sc_pilot_count_2t, sc_last_2t, sc_valid_2t};
        { sc_index_4t, sc_odd_4t, sc_papr_4t, sc_pilot_count_4t, sc_last_4t, sc_valid_4t} <= { sc_index_3t, sc_odd_3t, sc_papr_3t, sc_pilot_count_3t, sc_last_3t, sc_valid_3t};
        { sc_index_5t, sc_odd_5t, sc_papr_5t, sc_pilot_count_5t, sc_last_5t, sc_valid_5t} <= { sc_index_4t, sc_odd_4t, sc_papr_4t, sc_pilot_count_4t, sc_last_4t, sc_valid_4t};
        { sc_index_6t, sc_odd_6t, sc_papr_6t, sc_pilot_count_6t, sc_last_6t, sc_valid_6t} <= { sc_index_5t, sc_odd_5t, sc_papr_5t, sc_pilot_count_5t, sc_last_5t, sc_valid_5t};
        { sc_index_7t, sc_odd_7t, sc_papr_7t, sc_pilot_count_7t, sc_last_7t, sc_valid_7t} <= { sc_index_6t, sc_odd_6t, sc_papr_6t, sc_pilot_count_6t, sc_last_6t, sc_valid_6t};
        { sc_index_8t, sc_odd_8t, sc_papr_8t, sc_pilot_count_8t, sc_last_8t, sc_valid_8t} <= { sc_index_7t, sc_odd_7t, sc_papr_7t, sc_pilot_count_7t, sc_last_7t, sc_valid_7t};
        { sc_index_9t, sc_odd_9t, sc_papr_9t, sc_pilot_count_9t, sc_last_9t, sc_valid_9t} <= { sc_index_8t, sc_odd_8t, sc_papr_8t, sc_pilot_count_8t, sc_last_8t, sc_valid_8t};
        {sc_index_10t,sc_odd_10t,sc_papr_10t,sc_pilot_count_10t,sc_last_10t,sc_valid_10t} <= { sc_index_9t, sc_odd_9t, sc_papr_9t, sc_pilot_count_9t, sc_last_9t, sc_valid_9t};
        {sc_index_11t,sc_odd_11t,sc_papr_11t,sc_pilot_count_11t,sc_last_11t,sc_valid_11t} <= {sc_index_10t,sc_odd_10t,sc_papr_10t,sc_pilot_count_10t,sc_last_10t,sc_valid_10t};
      end
    end
  end


  assign f_index       = sc_index;
  assign f_last        = sc_last;
  assign f_valid       = sc_valid;

  assign h_index       = pilot?       sc_index:       sc_index_11t;
  assign h_odd         = pilot?         sc_odd:         sc_odd_11t;
  assign h_papr        = pilot?        sc_papr:        sc_papr_11t;
  assign h_pilot_count = pilot? sc_pilot_count: sc_pilot_count_11t;
  assign h_last        = pilot?        sc_last:        sc_last_11t;
  assign h_valid       = pilot?       sc_valid:       sc_valid_11t;


  /*****************************************************************************
  * lut
  *****************************************************************************/
  assign        nsd48 = {  -10'd26, 10'd24, 3'd1, 10'd5, 1'b1, 10'd18, 1'b1, 10'd30, 1'b1,  10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign        nsd52 = {  -10'd28, 10'd26, 3'd1, 10'd7, 1'b1, 10'd20, 1'b1, 10'd32, 1'b1,  10'd45, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign       nsd108 = {  -10'd58, 10'd54, 3'd3, 10'd5, 1'b1, 10'd32, 1'b1, 10'd45, 1'b1,  10'd63, 1'b1, 10'd76, 1'b1,10'd103, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign       nsd234 = { -10'd122,10'd117, 3'd3,10'd19, 1'b1, 10'd46, 1'b1, 10'd81, 1'b1, 10'd108, 1'b1,10'd126, 1'b1,10'd153, 1'b1,10'd188, 1'b1,10'd215, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[0] = 199'b0;
  assign   ru26_20[1] = { -10'd121,  10'd0, 3'd0, 10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[2] = {  -10'd95,  10'd0, 3'd0, 10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[3] = {  -10'd68,  10'd0, 3'd0, 10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[4] = {  -10'd42,  10'd0, 3'd0, 10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[5] = {  -10'd16, 10'd12, 3'd7, 10'd6, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[6] = {   10'd17,  10'd0, 3'd0, 10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[7] = {   10'd43,  10'd0, 3'd0, 10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[8] = {   10'd70,  10'd0, 3'd0, 10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_20[9] = {   10'd96,  10'd0, 3'd0, 10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,   10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_20[10] = 199'b0;
  assign  ru26_20[11] = 199'b0;
  assign  ru26_20[12] = 199'b0;
  assign  ru26_20[13] = 199'b0;
  assign  ru26_20[14] = 199'b0;
  assign  ru26_20[15] = 199'b0;
  assign   ru52_20[0] = 199'b0;
  assign   ru52_20[1] = { -10'd121,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_20[2] = {  -10'd68,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_20[3] = {   10'd17,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_20[4] = {   10'd70,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_20[5] = 199'b0;
  assign   ru52_20[6] = 199'b0;
  assign   ru52_20[7] = 199'b0;
  assign  ru106_20[0] = 199'b0;
  assign  ru106_20[1] = { -10'd122,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_20[2] = {   10'd17,  10'd0, 3'd0,  10'd5, 1'b1, 10'd30, 1'b1, 10'd71, 1'b1, 10'd96, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_20[3] = 199'b0;
  assign  ru242_20[0] = 199'b0;
  assign  ru242_20[1] = { -10'd122,10'd117, 3'd3,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,10'd137, 1'b1,10'd162, 1'b1,10'd203, 1'b1,10'd228, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[0] = 199'b0;
  assign   ru26_40[1] = { -10'd243,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[2] = { -10'd217,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[3] = { -10'd189,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[4] = { -10'd163,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[5] = { -10'd136,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[6] = { -10'd109,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[7] = {  -10'd83,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[8] = {  -10'd55,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_40[9] = {  -10'd29,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[10] = {    10'd4,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[11] = {   10'd30,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[12] = {   10'd58,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[13] = {   10'd84,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[14] = {  10'd111,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[15] = {  10'd138,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[16] = {  10'd164,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[17] = {  10'd192,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[18] = {  10'd218,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_40[19] = 199'b0;
  assign  ru26_40[20] = 199'b0;
  assign  ru26_40[21] = 199'b0;
  assign  ru26_40[22] = 199'b0;
  assign  ru26_40[23] = 199'b0;
  assign  ru26_40[24] = 199'b0;
  assign  ru26_40[25] = 199'b0;
  assign  ru26_40[26] = 199'b0;
  assign  ru26_40[27] = 199'b0;
  assign  ru26_40[28] = 199'b0;
  assign  ru26_40[29] = 199'b0;
  assign  ru26_40[30] = 199'b0;
  assign  ru26_40[31] = 199'b0;
  assign   ru52_40[0] = 199'b0;
  assign   ru52_40[1] = { -10'd243,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[2] = { -10'd189,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[3] = { -10'd109,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[4] = {  -10'd55,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[5] = {    10'd4,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[6] = {   10'd58,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[7] = {  10'd138,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[8] = {  10'd192,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_40[9] = 199'b0;
  assign  ru52_40[10] = 199'b0;
  assign  ru52_40[11] = 199'b0;
  assign  ru52_40[12] = 199'b0;
  assign  ru52_40[13] = 199'b0;
  assign  ru52_40[14] = 199'b0;
  assign  ru52_40[15] = 199'b0;
  assign  ru106_40[0] = 199'b0;
  assign  ru106_40[1] = { -10'd243,  10'd0, 3'd0,  10'd5, 1'b1, 10'd30, 1'b1, 10'd71, 1'b1, 10'd96, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_40[2] = { -10'd109,  10'd0, 3'd0,  10'd5, 1'b1, 10'd30, 1'b1, 10'd71, 1'b1, 10'd96, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_40[3] = {    10'd4,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_40[4] = {  10'd138,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_40[5] = 199'b0;
  assign  ru106_40[6] = 199'b0;
  assign  ru106_40[7] = 199'b0;
  assign  ru242_40[0] = 199'b0;
  assign  ru242_40[1] = { -10'd244,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,10'd136, 1'b1,10'd161, 1'b1,10'd202, 1'b1,10'd227, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru242_40[2] = {    10'd3,  10'd0, 3'd0,  10'd7, 1'b1, 10'd32, 1'b1, 10'd73, 1'b1, 10'd98, 1'b1,10'd137, 1'b1,10'd162, 1'b1,10'd203, 1'b1,10'd228, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru242_40[3] = 199'b0;
  assign  ru484_40[0] = 199'b0;
  assign  ru484_40[1] = { -10'd244,10'd234, 3'd5,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,10'd136, 1'b1,10'd161, 1'b1,10'd202, 1'b1,10'd227, 1'b1,10'd241, 1'b1,10'd266, 1'b1,10'd307, 1'b1,10'd332, 1'b1,10'd371, 1'b1,10'd396, 1'b1,10'd437, 1'b1,10'd462, 1'b1};
  assign   ru26_80[0] = 199'b0;
  assign   ru26_80[1] = { -10'd499,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[2] = { -10'd473,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[3] = { -10'd445,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[4] = { -10'd419,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[5] = { -10'd392,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[6] = { -10'd365,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[7] = { -10'd339,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[8] = { -10'd311,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru26_80[9] = { -10'd285,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[10] = { -10'd257,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[11] = { -10'd231,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[12] = { -10'd203,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[13] = { -10'd177,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[14] = { -10'd150,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[15] = { -10'd123,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[16] = {  -10'd97,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[17] = {  -10'd69,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[18] = {  -10'd43,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[19] = {  -10'd16, 10'd12, 3'd7,  10'd6, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[20] = {   10'd18,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[21] = {   10'd44,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[22] = {   10'd72,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[23] = {   10'd98,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[24] = {  10'd125,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[25] = {  10'd152,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[26] = {  10'd178,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[27] = {  10'd206,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[28] = {  10'd232,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[29] = {  10'd260,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[30] = {  10'd286,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[31] = {  10'd314,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[32] = {  10'd340,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[33] = {  10'd367,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[34] = {  10'd394,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[35] = {  10'd420,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[36] = {  10'd448,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[37] = {  10'd474,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru26_80[38] = 199'b0;
  assign  ru26_80[39] = 199'b0;
  assign  ru26_80[40] = 199'b0;
  assign  ru26_80[41] = 199'b0;
  assign  ru26_80[42] = 199'b0;
  assign  ru26_80[43] = 199'b0;
  assign  ru26_80[44] = 199'b0;
  assign  ru26_80[45] = 199'b0;
  assign  ru26_80[46] = 199'b0;
  assign  ru26_80[47] = 199'b0;
  assign  ru26_80[48] = 199'b0;
  assign  ru26_80[49] = 199'b0;
  assign  ru26_80[50] = 199'b0;
  assign  ru26_80[51] = 199'b0;
  assign  ru26_80[52] = 199'b0;
  assign  ru26_80[53] = 199'b0;
  assign  ru26_80[54] = 199'b0;
  assign  ru26_80[55] = 199'b0;
  assign  ru26_80[56] = 199'b0;
  assign  ru26_80[57] = 199'b0;
  assign  ru26_80[58] = 199'b0;
  assign  ru26_80[59] = 199'b0;
  assign  ru26_80[60] = 199'b0;
  assign  ru26_80[61] = 199'b0;
  assign  ru26_80[62] = 199'b0;
  assign  ru26_80[63] = 199'b0;
  assign   ru52_80[0] = 199'b0;
  assign   ru52_80[1] = { -10'd499,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[2] = { -10'd445,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[3] = { -10'd365,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[4] = { -10'd311,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[5] = { -10'd257,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[6] = { -10'd203,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[7] = { -10'd123,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[8] = {  -10'd69,  10'd0, 3'd0,  10'd5, 1'b1, 10'd18, 1'b1, 10'd29, 1'b1, 10'd42, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign   ru52_80[9] = {   10'd18,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[10] = {   10'd72,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[11] = {  10'd152,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[12] = {  10'd206,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[13] = {  10'd260,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[14] = {  10'd314,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[15] = {  10'd394,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[16] = {  10'd448,  10'd0, 3'd0,  10'd6, 1'b1, 10'd19, 1'b1, 10'd30, 1'b1, 10'd43, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru52_80[17] = 199'b0;
  assign  ru52_80[18] = 199'b0;
  assign  ru52_80[19] = 199'b0;
  assign  ru52_80[20] = 199'b0;
  assign  ru52_80[21] = 199'b0;
  assign  ru52_80[22] = 199'b0;
  assign  ru52_80[23] = 199'b0;
  assign  ru52_80[24] = 199'b0;
  assign  ru52_80[25] = 199'b0;
  assign  ru52_80[26] = 199'b0;
  assign  ru52_80[27] = 199'b0;
  assign  ru52_80[28] = 199'b0;
  assign  ru52_80[29] = 199'b0;
  assign  ru52_80[30] = 199'b0;
  assign  ru52_80[31] = 199'b0;
  assign  ru106_80[0] = 199'b0;
  assign  ru106_80[1] = { -10'd499,  10'd0, 3'd0,  10'd5, 1'b1, 10'd30, 1'b1, 10'd71, 1'b1, 10'd96, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[2] = { -10'd365,  10'd0, 3'd0,  10'd5, 1'b1, 10'd30, 1'b1, 10'd71, 1'b1, 10'd96, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[3] = { -10'd257,  10'd0, 3'd0,  10'd5, 1'b1, 10'd30, 1'b1, 10'd71, 1'b1, 10'd96, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[4] = { -10'd123,  10'd0, 3'd0,  10'd5, 1'b1, 10'd30, 1'b1, 10'd71, 1'b1, 10'd96, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[5] = {   10'd18,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[6] = {  10'd152,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[7] = {  10'd260,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[8] = {  10'd394,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru106_80[9] = 199'b0;
  assign ru106_80[10] = 199'b0;
  assign ru106_80[11] = 199'b0;
  assign ru106_80[12] = 199'b0;
  assign ru106_80[13] = 199'b0;
  assign ru106_80[14] = 199'b0;
  assign ru106_80[15] = 199'b0;
  assign  ru242_80[0] = 199'b0;
  assign  ru242_80[1] = { -10'd500,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,10'd136, 1'b1,10'd161, 1'b1,10'd202, 1'b1,10'd227, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru242_80[2] = { -10'd258,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,10'd136, 1'b1,10'd161, 1'b1,10'd202, 1'b1,10'd227, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru242_80[3] = {   10'd17,  10'd0, 3'd0,  10'd7, 1'b1, 10'd32, 1'b1, 10'd73, 1'b1, 10'd98, 1'b1,10'd137, 1'b1,10'd162, 1'b1,10'd203, 1'b1,10'd228, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru242_80[4] = {  10'd259,  10'd0, 3'd0,  10'd7, 1'b1, 10'd32, 1'b1, 10'd73, 1'b1, 10'd98, 1'b1,10'd137, 1'b1,10'd162, 1'b1,10'd203, 1'b1,10'd228, 1'b1,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0,  10'd0, 1'b0};
  assign  ru242_80[5] = 199'b0;
  assign  ru242_80[6] = 199'b0;
  assign  ru242_80[7] = 199'b0;
  assign  ru484_80[0] = 199'b0;
  assign  ru484_80[1] = { -10'd500,  10'd0, 3'd0,  10'd6, 1'b1, 10'd31, 1'b1, 10'd72, 1'b1, 10'd97, 1'b1,10'd136, 1'b1,10'd161, 1'b1,10'd202, 1'b1,10'd227, 1'b1,10'd240, 1'b1,10'd265, 1'b1,10'd306, 1'b1,10'd331, 1'b1,10'd370, 1'b1,10'd395, 1'b1,10'd436, 1'b1,10'd461, 1'b1};
  assign  ru484_80[2] = {   10'd17,  10'd0, 3'd0,  10'd7, 1'b1, 10'd32, 1'b1, 10'd73, 1'b1, 10'd98, 1'b1,10'd137, 1'b1,10'd162, 1'b1,10'd203, 1'b1,10'd228, 1'b1,10'd241, 1'b1,10'd266, 1'b1,10'd307, 1'b1,10'd332, 1'b1,10'd371, 1'b1,10'd396, 1'b1,10'd437, 1'b1,10'd462, 1'b1};
  assign  ru484_80[3] = 199'b0;
  assign  ru996_80[0] = 199'b0;
  assign  ru996_80[1] = { -10'd500,10'd490, 3'd5, 10'd32, 1'b1, 10'd99, 1'b1,10'd164, 1'b1,10'd231, 1'b1,10'd270, 1'b1,10'd337, 1'b1,10'd402, 1'b1,10'd469, 1'b1,10'd511, 1'b1,10'd578, 1'b1,10'd643, 1'b1,10'd710, 1'b1,10'd749, 1'b1,10'd816, 1'b1,10'd881, 1'b1,10'd948, 1'b1};


`ifdef RW_SIMU_ON
  reg [32*8-1:0] s_sc_state;
  /*****************************************************************************
  * simulation
  *****************************************************************************/
  // pragma coverage block = off, expr = off, toggle = off
  always @(*)
  begin
    case(sc_state)
      S_IDLE:       s_sc_state = "S_IDLE";
      S_WAIT_FIFO:  s_sc_state = "S_WAIT_FIFO";
      S_PUSH_ROW:   s_sc_state = "S_PUSH_ROW";
      S_PUSH_PILOT: s_sc_state = "S_PUSH_PILOT";
      default:      s_sc_state = "XXXXXXXXX";
    endcase
  end
  reg [32*8-1:0] s_sc_state2;
  always @(*)
  begin
    case(sc_state2)
      S_IDLE:       s_sc_state2 = "S_IDLE";
      S_WAIT_FIFO:  s_sc_state2 = "S_WAIT_FIFO";
      S_PUSH_ROW:   s_sc_state2 = "S_PUSH_ROW";
      S_PUSH_PILOT: s_sc_state2 = "S_PUSH_PILOT";
      default:      s_sc_state2 = "XXXXXXXXX";
    endcase
  end
  // pragma coverage block = on, expr = on, toggle = on
`endif

`ifdef RW_SIMU_ON
// pragma coverage block = off, expr = off, toggle = off
`ifndef RW_NX_DERIV_EQU_VHT
  /*****************************************************************************
  * ensure that RW_NX_DERIV_EQU_VHT is defined for VHT80 operation
  *****************************************************************************/
  always @(*)
    if(reception_mode==MHZ80_VHT)
    begin
      $display("error:equalizer_genindex: RW_NX_DERIV_EQU_VHT not defined but trying to do VHT80 !");
      $display("Simulation FAILURE");
      $stop();
    end

`endif
// pragma coverage block = on, expr = on, toggle = on
`endif
endmodule
`default_nettype wire
