/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module equalizer_mh
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire         clk,
  input  wire         rst_n,
 
  /*****************************************************************************
  * controls
  *****************************************************************************/
  input  wire         mh_clr,     // Clear mh computation, do not clear previous mh results
  input  wire         eq2d_en,    // Indicates if current computation is for primary (0) or secondary (1)
  input  wire         mh3_update, // Update only mh3 starting from previously computed mh2

  /*****************************************************************************
  * parameters 
  *****************************************************************************/
  // mh_* and mh2_* parameters are not needed for mh3_update
  input  wire [ 8:0]  mh_nsd,
  input  wire [ 2:0]  mh_shift_val,
  input  wire [ 7:0]  mh_shift_thr,
  input  wire [24:0]  mh2_sigma_squared,
  // mh3_* parameters are needed for mh computation and for mh3_update
  input  wire [ 2:0]  mh3_satsb,
  input  wire         mh3_ldpc,
  input  wire [ 2:0]  mh3_mod_type,
  
  /*****************************************************************************
  * d data needed for mh computation
  *****************************************************************************/
  input  wire         t0_valid,
  input  wire [18:0]  t0_d,
  input  wire         t0_last,
  
  /*****************************************************************************
  * status registers
  *****************************************************************************/
  output reg [25:0]   status_pri_mh,
  output reg [18:0]   status_pri_dmax,
  
  /*****************************************************************************
  * Computed value to soft-bit compression
  *****************************************************************************/
  output wire [27:0]  mh_x_satsb,
  output wire [15:0]  mh_x_satsb_rnd12,
  output wire         mh_done
);
 
  /*****************************************************************************
  * few remarks about vector dimension
  *
  * for normal operations:
  *
  *    d_max = (4096^2+4096^2+4096^2+4096^2)/256 = 262144
  *   mh_max = d_max*234*satsb_max               = 245366784   (28 bits)
  *    2**28                                     = 268435456   (28 bits)
  *   mh_x_satsb_rnd12_max = 245366784/4096      = 59904       (16 bits)
  *                               
  *****************************************************************************/
  /* declaration */
  /* Compute mh, part of Equalizer computation (cf EqualizerComp_NRx1_FXPT) */
  reg   [ 8:0]  static_nsd;
  reg   [25:0]  mh;
  reg   [25:0]  mh_shift;
  reg   [18:0]  d_max;
  wire  [26:0]  n_d_max_x_nsd;
  reg   [18:0]  d_max_x_nsd;
  wire  [25:0]  n_mh_x_thr;
  reg   [21:0]  mh_x_thr;
  wire  [25:0]  n_mh;
  wire  [27:0]  n_pri_mh_satsb_shift,n_pri_mh_satsb_shift_rnd12;
  wire  [34:0]  n_nsd_x_sigma;
  reg   [34:0]  nsd_x_sigma;
  wire  [26:0]  n_nsd_x_sigma_rnd;
  wire  [25:0]  n_nsd_x_sigma_sat;
  wire  [25:0]  mh2;
  reg   [25:0]  nsd_x_sigma_sat;
  /* Compute mh2 and mh3, part of soft bit compression (cf SoftBitCompression_FXPT) */
  reg           mh3_update_d1;
  reg           mh3_update_d2;
  wire          update_mh;
  reg   [12:0]  static_gain;
  wire  [38:0]  n_pri_mhnsr_x_gain;
  reg   [38:0]  pri_mhnsr_x_gain;
  wire  [36:0]  n_pri_mhnsr_x_gain_bcc;
  wire  [36:0]  n_pri_mhnsr_x_gain_ldpc;
  wire  [36:0]  n_pri_mhnsr_x_gain_rnd;
  reg   [36:0]  pri_mhnsr_x_gain_rnd;
  wire  [25:0]  pri_mhnsr_x_gain_sat;
  wire  [25:0]  pri_mh3;
  /* Store results for primary */
  reg   [ 3:0]  last_del;
  reg   [27:0]  pri_mh_x_satsb;
  reg   [15:0]  pri_mh_x_satsb_rnd12;
  reg   [25:0]  pri_mh2;



  assign mh_done = last_del[3];


  /*****************************************************************************
  * d_max x nsd
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
    if(!rst_n)
      static_nsd <= 9'b0; 
    else
      static_nsd <= mh_nsd;
  
  assign n_d_max_x_nsd = {10'b0,static_nsd} * d_max;
 
  /*****************************************************************************
  * (mh x 2^-8) x mh_shift_thr
  *****************************************************************************/
  assign n_mh_x_thr = mh[25:8] * {10'b0,mh_shift_thr};
 
  /*****************************************************************************
  * MH : mh accumulation
  *****************************************************************************/
  assign n_mh = mh + {7'b0,t0_d};
  
  /*****************************************************************************
  * mh2 = nsd x nss x mh2_sigma_squared / 512; 
  * mh2 = max(Round(mh2),1);
  *****************************************************************************/
  assign n_nsd_x_sigma = {26'b0, static_nsd} * {10'b0,mh2_sigma_squared};

  assign n_nsd_x_sigma_rnd = { 1'b0, nsd_x_sigma[34:9]} + // shift by 9-bit (division by 512)
                             {26'd0, nsd_x_sigma[8]};     // rounding

  // Saturation
  SatUnsigned #(.INPUT_WIDTH(27),.OUTPUT_WIDTH(26))
    u_mh2sat (.InputData(n_nsd_x_sigma_rnd),.SatData(n_nsd_x_sigma_sat));
    
  assign mh2 = (nsd_x_sigma_sat == 26'd0) ? 26'd1 : nsd_x_sigma_sat;
  
  /*****************************************************************************
  * mh3 = mh2 x gain Table (depending on modulation and fec)
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
    if(!rst_n) begin
      static_gain   <= 14'b0; 
      mh3_update_d1 <= 1'd0;
      mh3_update_d2 <= 1'd0;
    end else begin
      mh3_update_d1 <= mh3_update; // to detect clock
      mh3_update_d2 <= mh3_update_d1; // to detect clock
      case(mh3_mod_type)
        3'd0:    static_gain <= 13'd11;     /* BPSK   */
        3'd1:    static_gain <= 13'd11;     /* QBPSK  */ 
        3'd2:    static_gain <= 13'd23;     /* QPSK   */ 
        3'd3:    static_gain <= 13'd113;    /* QAM16  */
        3'd4:    static_gain <= 13'd475;    /* QAM64  */
        3'd5:    static_gain <= 13'd1923;   /* QAM256 */
        default: static_gain <= 13'd7716;   /* QAM1024 */
      endcase
    end

  assign n_pri_mhnsr_x_gain      = {13'd0,pri_mh2} * {26'b0,static_gain};
  assign n_pri_mhnsr_x_gain_bcc  = pri_mhnsr_x_gain[38:2] + {36'd0,pri_mhnsr_x_gain[1]};
  assign n_pri_mhnsr_x_gain_ldpc = {1'b0,pri_mhnsr_x_gain[38:3]} + {36'd0,pri_mhnsr_x_gain[2]};
  assign n_pri_mhnsr_x_gain_rnd  = mh3_ldpc ? n_pri_mhnsr_x_gain_ldpc : n_pri_mhnsr_x_gain_bcc;
  
  // Saturation
  SatUnsigned #(.INPUT_WIDTH(37),.OUTPUT_WIDTH(26))
    u_pri_mh3sat (.InputData( pri_mhnsr_x_gain_rnd),
                  .SatData(   pri_mhnsr_x_gain_sat)
                  );

  assign pri_mh3 = (pri_mhnsr_x_gain_sat == 26'd0) ? 26'd1 : pri_mhnsr_x_gain_sat;
    
  /*****************************************************************************
  * compute satsb right shift
  *****************************************************************************/
  assign n_pri_mh_satsb_shift       = {pri_mh3,1'b0} >> mh3_satsb;
  assign n_pri_mh_satsb_shift_rnd12 = n_pri_mh_satsb_shift + 28'd2048;
  
  /*****************************************************************************
  * Registers
  *****************************************************************************/
  assign update_mh = t0_last || (mh3_update_d1 && !mh3_update_d2);
  
  always @(posedge clk, negedge rst_n)
    if(!rst_n)
    begin
      nsd_x_sigma          <= 35'b0;
      nsd_x_sigma_sat      <= 26'b0; // mh2 in matlb
      mh                   <= 26'b0;
      mh_shift             <= 26'b0;
      d_max                <= 19'b0;
      d_max_x_nsd          <= 19'b0;
      mh_x_thr             <= 22'b0;
      last_del             <= 4'd0;
      pri_mhnsr_x_gain     <= 39'b0;
      pri_mhnsr_x_gain_rnd <= 37'b0; // mh3 in Matlab
      pri_mh2              <= 26'd0;
      pri_mh_x_satsb       <= 28'd0;
      pri_mh_x_satsb_rnd12 <= 16'd0;
    end
    else if(mh_clr)
    begin
      nsd_x_sigma          <= 35'b0;
      nsd_x_sigma_sat      <= 26'b0;
      mh                   <= 26'b0;
      mh_shift             <= 26'b0;
      d_max                <= 19'b0;
      d_max_x_nsd          <= 19'b0;
      mh_x_thr             <= 22'b0;
      last_del             <= 4'd0;
      if (eq2d_en==1'b0) begin
        pri_mhnsr_x_gain     <= 39'b0;
        pri_mhnsr_x_gain_rnd <= 37'b0;
        pri_mh2              <= 26'd0;
        pri_mh_x_satsb       <= 28'd0;
        pri_mh_x_satsb_rnd12 <= 16'd0;
      end
    end
    else
    begin
      last_del             <= {last_del[2:0],update_mh};
      
      // mh computation
      mh_x_thr             <= n_mh_x_thr[25:4];

      d_max_x_nsd          <= n_d_max_x_nsd[26:8];
      
      if((t0_d>d_max) && t0_valid)
        d_max <= t0_d;
      if(t0_valid)
        mh <= n_mh;
      
      // Compare d_max_x_nsd and mh_x_thr for Mh shift
      if ({3'b0,d_max_x_nsd} > mh_x_thr)
      begin
        case (mh_shift_val)
          3'd0   : mh_shift <= mh;
          3'd1   : mh_shift <= {1'b0,mh[25:1]};
          3'd2   : mh_shift <= {2'b0,mh[25:2]};
          default: mh_shift <= {3'b0,mh[25:3]};
        endcase
      end
      else
        mh_shift <= mh;

      // mh2 computation
      nsd_x_sigma       <= n_nsd_x_sigma;
      nsd_x_sigma_sat   <= n_nsd_x_sigma_sat;
      
      if (!eq2d_en && (mh_done || mh3_update_d2)) begin // latch on mh update and on mh3 update
        pri_mh2         <= mh2;
      end

      pri_mhnsr_x_gain     <= n_pri_mhnsr_x_gain;
      pri_mhnsr_x_gain_rnd <= n_pri_mhnsr_x_gain_rnd;
      pri_mh_x_satsb       <= n_pri_mh_satsb_shift;
      pri_mh_x_satsb_rnd12 <= n_pri_mh_satsb_shift_rnd12[27:12];
    end

  /* Values to use (not registered, must be available when block is not clocked)*/ 
  assign mh_x_satsb = pri_mh_x_satsb;

  assign mh_x_satsb_rnd12 = pri_mh_x_satsb_rnd12;
                      

  /* Status registers */ 
  always @ (posedge clk or negedge rst_n)
    if(!rst_n)
    begin
      status_pri_mh   <= 26'b0;
      status_pri_dmax <= 19'b0;
    end else if ((eq2d_en==1'b0) && (mh_done==1'b1) && (mh3_update==1'b0)) begin
      status_pri_mh   <= mh;
      status_pri_dmax <= d_max;
    end
  
endmodule
`default_nettype wire
