/*******************************************************************************
* Copyright (C) by RivieraWaves.
* This module is a confidential and proprietary property of RivieraWaves
* and a possession or use of this module requires written permission
* from RivieraWaves.
********************************************************************************
* Company: RivieraWaves
* $Author: $
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     : None
* Description      : 
* Simulation Notes : 
* Synthesis Notes  :
* Application Note :
* Simulator        :
* Parameters       :
* Terms & concepts :
* Bugs             :
* Open issues and future enhancements :
* References       :
* Revision History :
********************************************************************************
* $HeadURL: $
*******************************************************************************/
`default_nettype none
module equalizer_pequ
#(
  /*****************************************************************************
  *  g_arch
  *  'd0     booth multiplier 
  *  'd1     behavioural model (synthesizable)
  *****************************************************************************/
  parameter g_arch=0
)
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire         clk,
  input  wire         rst_n,
 
  /*****************************************************************************
  * parameters
  * 
  * pilot_sel : 00   sub-carrier is data
  *             01   sub-carrier is pilot for nsts=1, displaying x0
  *             10   sub-carrier is pilot for nsts=2, mixing with x0+x1
  *             11   sub-carrier is pilot for nsts=2, mixing with x0-x1
  *
  *****************************************************************************/
  input  wire         enable,
  input  wire         stbc,
  input  wire         odd,
  input  wire         qbpsk_detect_en,
  input  wire         qbpsk_detect_clr,
  
  /*****************************************************************************
  * input operands
  *****************************************************************************/
  input  wire         t0_valid,
  input  wire [ 1:0]  t0_pilot_sel,
  input  wire [12:0]  t0_r_i,
  input  wire [12:0]  t0_r_q,
  input  wire [12:0]  t0_h1_i,
  input  wire [12:0]  t0_h1_q,
  input  wire [12:0]  t0_h2_i,
  input  wire [12:0]  t0_h2_q,
  input  wire         t0_last,
  
  /*****************************************************************************
  * x data
  *****************************************************************************/
  output reg          t6_valid,
  output reg  [19:0]  t6_x0_i,
  output reg  [19:0]  t6_x0_q,
  output reg  [19:0]  t6_x1_i,
  output reg  [19:0]  t6_x1_q,
 
  /*****************************************************************************
  * x pilot
  *****************************************************************************/
  output reg          t3_pilot_valid,
  output reg  [19:0]  t3_pilot_i,
  output reg  [19:0]  t3_pilot_q,

  /*****************************************************************************
  * qbpsk detection
  *****************************************************************************/
  output reg  [25:0]  qbpsk_accu_i,
  output reg  [25:0]  qbpsk_accu_q,

 /*****************************************************************************
  * stbc
  *****************************************************************************/
  output wire         stbc_wen,
  output wire         stbc_ren,
  output wire [ 8:0]  stbc_addr,
  output wire [75:0]  stbc_wdata,
  input  wire [75:0]  stbc_rdata
);
 
  /*****************************************************************************
  * static
  *****************************************************************************/
  reg  t0_neg_ri_h2i,t0_neg_rq_h2q,t0_neg_rq_h2i,t0_neg_ri_h2q;

  always @(*)
  begin
    if(!stbc || t0_pilot_sel!=2'd0)
    begin
      /* no stbc or pilot*/ 
      {t0_neg_ri_h2i, t0_neg_rq_h2q, t0_neg_rq_h2i, t0_neg_ri_h2q} = 4'b0001; 
    end
    else
    begin
      if(!odd)
      begin
        /* stbc, input symbol is even */ 
        {t0_neg_ri_h2i, t0_neg_rq_h2q, t0_neg_rq_h2i, t0_neg_ri_h2q} = 4'b1101; 
      end
      else
      begin
        /* stbc, input symbol is odd  */  
        {t0_neg_ri_h2i, t0_neg_rq_h2q, t0_neg_rq_h2i, t0_neg_ri_h2q} = 4'b0010;
      end
    end    
  end
  
  /*****************************************************************************
  * S1 : partial equalization, 3 possible data path configurations:
  *
  * 1) Non-STBC or pilot
  *
  *   o R*conj(H1)    = ( ri*h1i +rq*h1q) + j*( rq*h1i -ri*h1q)
  *   o H2 is zeroed by S0 stage for data
  *   o R*conj(H2) for pilot in stbc
  *
  * 2) STBC, input even symbol (r0)
  *   
  *   o R0*conj( H1)    = ( r0i*h1i +r0q*h1q) + j*( r0q*h1i -r0i*h1q)
  *   o conj(R0)*(-H2)  = (-r0i*h2i -r0q*h2q) + j*(+r0q*h2i -r0i*h2q)
  *
  * 3) STBC, input odd symbol (r1)
  *   
  *   o conj(R1)*H2     =  ( r1i*h2i +r1q*h2q) + j*(-r1q*h2i +r1i*h2q)
  *   o R1*conj(H1)     =  ( r1i*h1i +r1q*h1q) + j*(+r1q*h1i -r1i*h1q)
  *
  * note: at first glance, a complex multiplier based on 3 real multipliers 
  *       looks like smaller but then serialization of adders cancels this gain.
  *       we remain on classical 4 real multipliers.
  *****************************************************************************/
  reg           t1_valid;
  reg  [ 1:0]   t1_pilot_sel;
  reg           t1_last;
  wire [25:0]   t1_ri_h1i;
  wire [25:0]   t1_rq_h1q;
  wire [25:0]   t1_rq_h1i;
  wire [25:0]   t1_ri_h1q;
  wire [25:0]   t1_ri_h2i;
  wire [25:0]   t1_rq_h2q;
  wire [25:0]   t1_rq_h2i;
  wire [25:0]   t1_ri_h2q;
  
  /*  r_i*h1_i */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_ri_x_h1i
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               1'b0),
    .a(                 t0_r_i),
    .b(                 t0_h1_i),
    .p(                 t1_ri_h1i)
  );
 
  /*  r_q*h1_q */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_r_q_x_h1_q
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               1'b0),
    .a(                 t0_r_q),
    .b(                 t0_h1_q),
    .p(                 t1_rq_h1q)
  );
 
  /*  r_q*h1_i */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_r_q_x_h1_i
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               1'b0),
    .a(                 t0_r_q),
    .b(                 t0_h1_i),
    .p(                 t1_rq_h1i)
  );
 
  /* r_i*h1_q */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_r_i_x_h1_q
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               1'b1),
    .a(                 t0_r_i),
    .b(                 t0_h1_q),
    .p(                 t1_ri_h1q)
  );

  /*  r_i*h2_i */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_r_i_x_h2_i
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               t0_neg_ri_h2i),
    .a(                 t0_r_i),
    .b(                 t0_h2_i),
    .p(                 t1_ri_h2i)
  );
 
  /*  r_q*h2_q */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_r_q_x_h2_q
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               t0_neg_rq_h2q),
    .a(                 t0_r_q),
    .b(                 t0_h2_q),
    .p(                 t1_rq_h2q)
  );
 
  /*  r_q*h2_i */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_r_q_x_h2_i
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               t0_neg_rq_h2i),
    .a(                 t0_r_q),
    .b(                 t0_h2_i),
    .p(                 t1_rq_h2i)
  );
 
  /*  -1 * r_i*h2_q */
  mult_13x13
  #(
    .g_arch(            g_arch)
  )
  u_r_i_x_h2_q
  (
    .rst_n(             rst_n),
    .clk(               clk),
    .neg(               t0_neg_ri_h2q),
    .a(                 t0_r_i),
    .b(                 t0_h2_q),
    .p(                 t1_ri_h2q)
  );
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      t1_valid     <= 1'b0;
      t1_pilot_sel <= 2'b0;   
      t1_last      <= 1'b0;   
    end
    else if(!enable)
    begin
      t1_valid     <= 1'b0;
      t1_pilot_sel <= 2'b0;   
      t1_last      <= 1'b0;   
    end
    else
    begin
      t1_valid     <= t0_valid;   
      t1_pilot_sel <= t0_pilot_sel;   
      t1_last      <= t0_last;   
    end
  end

  /*****************************************************************************
  * S2: real/imaginary parts build / STBC address display
  *****************************************************************************/
  reg           t2_valid;
  reg  [ 1:0]   t2_pilot_sel;
  reg           t2_last;
  reg           t2_wen;
  reg           t2_ren;
  reg  [ 8:0]   t2_addr;
  wire [ 8:0]   n_t2_addr;
  reg  [18:0]   t2_r_h1_i;
  reg  [18:0]   t2_r_h2_i;
  reg  [18:0]   t2_r_h1_q;
  reg  [18:0]   t2_r_h2_q;
  wire [26:0]   n_t2_r_h1_i,n_t2_r_h1_i_prernd;
  wire [26:0]   n_t2_r_h2_i,n_t2_r_h2_i_prernd;
  wire [26:0]   n_t2_r_h1_q,n_t2_r_h1_q_prernd;
  wire [26:0]   n_t2_r_h2_q,n_t2_r_h2_q_prernd;
  
  reg  [26:0]   t2_pilot_h1_i;
  reg  [26:0]   t2_pilot_h1_q;
  reg  [26:0]   t2_pilot_h2_i;
  reg  [26:0]   t2_pilot_h2_q;
  
  /* data sub-carrier sum partial real parts */
  assign n_t2_r_h1_i        = {t1_ri_h1i[25],t1_ri_h1i} + {t1_rq_h1q[25],t1_rq_h1q};
  assign n_t2_r_h2_i        = {t1_ri_h2i[25],t1_ri_h2i} + {t1_rq_h2q[25],t1_rq_h2q};
  assign n_t2_r_h1_i_prernd = n_t2_r_h1_i + 27'd128;
  assign n_t2_r_h2_i_prernd = n_t2_r_h2_i + 27'd128;

  /* data sub-carrier sum partial imag parts */
  assign n_t2_r_h1_q        = {t1_rq_h1i[25],t1_rq_h1i} + {t1_ri_h1q[25],t1_ri_h1q};
  assign n_t2_r_h2_q        = {t1_rq_h2i[25],t1_rq_h2i} + {t1_ri_h2q[25],t1_ri_h2q};
  assign n_t2_r_h1_q_prernd = n_t2_r_h1_q + 27'd128;
  assign n_t2_r_h2_q_prernd = n_t2_r_h2_q + 27'd128;

  /* stbc address */
  assign n_t2_addr          = t2_addr + 9'd1;

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      t2_valid      <= 1'b0;
      t2_pilot_sel  <= 2'b0;
      t2_last       <= 1'b0;
      t2_r_h1_i     <= 19'b0;
      t2_r_h2_i     <= 19'b0;
      t2_r_h1_q     <= 19'b0;
      t2_r_h2_q     <= 19'b0;
      t2_pilot_h1_i <= 27'b0;
      t2_pilot_h1_q <= 27'b0;
      t2_pilot_h2_i <= 27'b0;
      t2_pilot_h2_q <= 27'b0;
      t2_wen        <= 1'b0;
      t2_ren        <= 1'b0;
      t2_addr       <= 9'b0;
    end
    else if(!enable)
    begin
      t2_valid      <= 1'b0;
      t2_pilot_sel  <= 2'b0;
      t2_last       <= 1'b0;
      t2_r_h1_i     <= 19'b0;
      t2_r_h2_i     <= 19'b0;
      t2_r_h1_q     <= 19'b0;
      t2_r_h2_q     <= 19'b0;
      t2_pilot_h1_i <= 27'b0;
      t2_pilot_h1_q <= 27'b0;
      t2_pilot_h2_i <= 27'b0;
      t2_pilot_h2_q <= 27'b0;
      t2_wen        <= 1'b0;
      t2_ren        <= 1'b0;
      t2_addr       <= 9'b0;
    end
    else
    begin
      if(t1_valid)
      begin
        t2_valid     <= 1'b1;                       
        t2_pilot_sel <= t1_pilot_sel;
        t2_last      <= t1_last;
        if(t1_pilot_sel==2'b00)                            
        begin                                        
          /* data sub-carrier */                     
          t2_pilot_h1_i <= 27'b0;                    
          t2_pilot_h1_q <= 27'b0;                    
          t2_pilot_h2_i <= 27'b0;                    
          t2_pilot_h2_q <= 27'b0;                    
          t2_r_h1_i     <= n_t2_r_h1_i_prernd[26:8]; 
          t2_r_h2_i     <= n_t2_r_h2_i_prernd[26:8]; 
          t2_r_h1_q     <= n_t2_r_h1_q_prernd[26:8]; 
          t2_r_h2_q     <= n_t2_r_h2_q_prernd[26:8]; 
          if(!stbc)
          begin
            t2_wen <= 1'b0;
            t2_ren <= 1'b0;
          end
          else
          begin
            if(!odd)
            begin
              t2_wen <= 1'b1;
              t2_ren <= 1'b0;
            end
            else
            begin
              t2_wen <= 1'b0;
              t2_ren <= 1'b1;
            end
          end
        end                                          
        else                                         
        begin                                        
          /* pilot sub-carrier */                    
          t2_pilot_h1_i <= n_t2_r_h1_i;           
          t2_pilot_h1_q <= n_t2_r_h1_q;           
          t2_pilot_h2_i <= n_t2_r_h2_i;           
          t2_pilot_h2_q <= n_t2_r_h2_q;           
          t2_r_h1_i     <= 19'b0;                 
          t2_r_h2_i     <= 19'b0;                 
          t2_r_h1_q     <= 19'b0;                 
          t2_r_h2_q     <= 19'b0;                 
          t2_wen        <= 1'b0;
          t2_ren        <= 1'b0;
        end
      end
      else
      begin
        t2_valid <= 1'b0;
        t2_last  <= 1'b0;
        t2_wen   <= 1'b0;
        t2_ren   <= 1'b0;
      end
      
      /* stbc address */
      if(t2_valid && t2_last)
        t2_addr <= 9'd0;
      else if(t2_wen || t2_ren)
        t2_addr <= n_t2_addr;
    end
  end

  assign stbc_wen   = t2_wen;
  assign stbc_ren   = t2_ren;
  assign stbc_addr  = t2_addr;
  assign stbc_wdata = {t2_r_h1_q,t2_r_h1_i,t2_r_h2_q,t2_r_h2_i};

  /*****************************************************************************
  * S3: STBC memory address capture / read access
  *****************************************************************************/
  reg           t3_valid;
  reg  [18:0]   t3_r_h1_i;
  reg  [18:0]   t3_r_h2_i;
  reg  [18:0]   t3_r_h1_q;
  reg  [18:0]   t3_r_h2_q; 
  wire [27:0]   n_t3_pilot_i_prernd;
  wire [27:0]   n_t3_pilot_q_prernd;

  assign n_t3_pilot_i_prernd = {t2_pilot_h1_i[26],t2_pilot_h1_i} + ({28{t2_pilot_sel==2'b11}}^{t2_pilot_h2_i[26],t2_pilot_h2_i}) + 
                               {20'b0,1'b1,6'b0,t2_pilot_sel==2'b11}; // rnd8+neg fix
    
  assign n_t3_pilot_q_prernd = {t2_pilot_h1_q[26],t2_pilot_h1_q} + ({28{t2_pilot_sel==2'b11}}^{t2_pilot_h2_q[26],t2_pilot_h2_q}) + 
                               {20'b0,1'b1,6'b0,t2_pilot_sel==2'b11}; // rnd8+neg fix
 
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      t3_valid       <= 1'b0;
      t3_r_h1_i      <= 19'b0;
      t3_r_h2_i      <= 19'b0;
      t3_r_h1_q      <= 19'b0;
      t3_r_h2_q      <= 19'b0;
      t3_pilot_valid <= 1'b0;
      t3_pilot_i     <= 20'b0;
      t3_pilot_q     <= 20'b0;
    end
    else if(!enable)
    begin
      t3_valid       <= 1'b0;
      t3_r_h1_i      <= 19'b0;
      t3_r_h2_i      <= 19'b0;
      t3_r_h1_q      <= 19'b0;
      t3_r_h2_q      <= 19'b0;
      t3_pilot_valid <= 1'b0;
      t3_pilot_i     <= 20'b0;
      t3_pilot_q     <= 20'b0;
    end
    else
    begin
      if(t2_valid)
      begin
        if(t2_pilot_sel==2'b00)
        begin
          t3_valid       <= 1'b1;
          t3_r_h1_i      <= t2_r_h1_i;
          t3_r_h2_i      <= t2_r_h2_i;
          t3_r_h1_q      <= t2_r_h1_q;
          t3_r_h2_q      <= t2_r_h2_q;
          t3_pilot_valid <= 1'b0;
          t3_pilot_i     <= 20'b0;
          t3_pilot_q     <= 20'b0;
        end
        else
        begin
          t3_valid       <= 1'b0;
          t3_r_h1_i      <= 19'b0;
          t3_r_h2_i      <= 19'b0;
          t3_r_h1_q      <= 19'b0;
          t3_r_h2_q      <= 19'b0;
          t3_pilot_valid <= 1'b1;
          t3_pilot_i     <= n_t3_pilot_i_prernd[27:8];
          t3_pilot_q     <= n_t3_pilot_q_prernd[27:8];
        end
      end
      else
      begin
        t3_valid       <= 1'b0;
        t3_r_h1_i      <= 19'b0;
        t3_r_h2_i      <= 19'b0;
        t3_r_h1_q      <= 19'b0;
        t3_r_h2_q      <= 19'b0;
        t3_pilot_valid <= 1'b0;
        t3_pilot_i     <= 20'b0;
        t3_pilot_q     <= 20'b0;
      end
    end
  end

  /*****************************************************************************
  * S4: FFTMEMMUX data capture
  *****************************************************************************/
  reg           t4_valid;
  reg  [18:0]   t4_r_h1_i;
  reg  [18:0]   t4_r_h2_i;
  reg  [18:0]   t4_r_h1_q;
  reg  [18:0]   t4_r_h2_q; 
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin  
      t4_valid  <= 1'b0;
      t4_r_h1_i <= 19'b0;  
      t4_r_h2_i <= 19'b0;  
      t4_r_h1_q <= 19'b0;  
      t4_r_h2_q <= 19'b0;  
    end
    else if(!enable)
    begin
      t4_valid  <= 1'b0;
      t4_r_h1_i <= 19'b0;  
      t4_r_h2_i <= 19'b0;  
      t4_r_h1_q <= 19'b0;  
      t4_r_h2_q <= 19'b0;  
    end
    else
    begin
      if(t3_valid)
      begin
        t4_valid  <= 1'b1;
        t4_r_h1_i <= t3_r_h1_i;  
        t4_r_h1_q <= t3_r_h1_q;  
        t4_r_h2_i <= t3_r_h2_i;  
        t4_r_h2_q <= t3_r_h2_q;  
      end
      else
      begin
        t4_valid  <= 1'b0;
        t4_r_h1_i <= 19'b0;  
        t4_r_h2_i <= 19'b0;  
        t4_r_h1_q <= 19'b0;  
        t4_r_h2_q <= 19'b0;  
      end
    end
  end
  
  /*****************************************************************************
  * S5: STBC read data capture
  *****************************************************************************/
  reg           t5_valid;
  reg  [18:0]   t5_r0_h1_i;
  reg  [18:0]   t5_r0_h2_i;
  reg  [18:0]   t5_r0_h1_q;
  reg  [18:0]   t5_r0_h2_q; 
  reg  [18:0]   t5_r1_h1_i;
  reg  [18:0]   t5_r1_h2_i;
  reg  [18:0]   t5_r1_h1_q;
  reg  [18:0]   t5_r1_h2_q; 
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin  
      t5_valid   <= 1'b0;
      t5_r0_h1_i <= 19'b0;  
      t5_r0_h2_i <= 19'b0;  
      t5_r0_h1_q <= 19'b0;  
      t5_r0_h2_q <= 19'b0;  
      t5_r1_h1_i <= 19'b0;  
      t5_r1_h2_i <= 19'b0;  
      t5_r1_h1_q <= 19'b0;  
      t5_r1_h2_q <= 19'b0;
    end
    else if(!enable)
    begin
      t5_valid   <= 1'b0;
      t5_r0_h1_i <= 19'b0;  
      t5_r0_h2_i <= 19'b0;  
      t5_r0_h1_q <= 19'b0;  
      t5_r0_h2_q <= 19'b0;  
      t5_r1_h1_i <= 19'b0;  
      t5_r1_h2_i <= 19'b0;  
      t5_r1_h1_q <= 19'b0;  
      t5_r1_h2_q <= 19'b0;
    end
    else
    begin
      if(t4_valid)
      begin
        t5_valid   <= 1'b1;
        if(!stbc)
        begin
          t5_r0_h1_q <= 19'b0; 
          t5_r0_h1_i <= 19'b0; 
          t5_r0_h2_q <= 19'b0;         
          t5_r0_h2_i <= 19'b0;         
          t5_r1_h1_i <= t4_r_h1_i;                  
          t5_r1_h1_q <= t4_r_h1_q;                  
          t5_r1_h2_i <= t4_r_h2_i;
          t5_r1_h2_q <= t4_r_h2_q;
        end
        else
        begin
          if(odd)
          begin
            t5_r0_h1_q <= stbc_rdata[75:57];
            t5_r0_h1_i <= stbc_rdata[56:38];
            t5_r0_h2_q <= stbc_rdata[37:19];
            t5_r0_h2_i <= stbc_rdata[18: 0];                                      
            t5_r1_h1_i <= t4_r_h1_i;                            
            t5_r1_h1_q <= t4_r_h1_q;                            
            t5_r1_h2_i <= t4_r_h2_i;                            
            t5_r1_h2_q <= t4_r_h2_q;        
          end                                        
          else                                       
          begin                                      
            t5_r0_h1_i <= 19'b0;                     
            t5_r0_h2_i <= 19'b0;   
            t5_r0_h1_q <= 19'b0;   
            t5_r0_h2_q <= 19'b0;   
            t5_r1_h1_i <= 19'b0;   
            t5_r1_h2_i <= 19'b0;   
            t5_r1_h1_q <= 19'b0;   
            t5_r1_h2_q <= 19'b0;   
          end
        end
      end
      else
      begin
        t5_valid   <= 1'b0;
        t5_r0_h1_i <= 19'b0;  
        t5_r0_h2_i <= 19'b0;  
        t5_r0_h1_q <= 19'b0;  
        t5_r0_h2_q <= 19'b0;  
        t5_r1_h1_i <= 19'b0;  
        t5_r1_h2_i <= 19'b0;  
        t5_r1_h1_q <= 19'b0;  
        t5_r1_h2_q <= 19'b0;
      end
    end
  end

  /*****************************************************************************
  * S6: odd/even combination
  *****************************************************************************/
  wire  [19:0]  n_t6_x0_i;
  wire  [19:0]  n_t6_x0_q;
  wire  [19:0]  n_t6_x1_i;
  wire  [19:0]  n_t6_x1_q;
 
  assign n_t6_x0_i    = {t5_r0_h1_i[18],t5_r0_h1_i} + {t5_r1_h2_i[18],t5_r1_h2_i};
  assign n_t6_x0_q    = {t5_r0_h1_q[18],t5_r0_h1_q} + {t5_r1_h2_q[18],t5_r1_h2_q};
  assign n_t6_x1_i    = {t5_r0_h2_i[18],t5_r0_h2_i} + {t5_r1_h1_i[18],t5_r1_h1_i};
  assign n_t6_x1_q    = {t5_r0_h2_q[18],t5_r0_h2_q} + {t5_r1_h1_q[18],t5_r1_h1_q};

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      t6_valid <=  1'b0;
      t6_x0_i  <= 20'b0;
      t6_x0_q  <= 20'b0;
      t6_x1_i  <= 20'b0;
      t6_x1_q  <= 20'b0;
    end
    else if(!enable)
    begin
      t6_valid <=  1'b0;
      t6_x0_i  <= 20'b0;
      t6_x0_q  <= 20'b0;
      t6_x1_i  <= 20'b0;
      t6_x1_q  <= 20'b0;
    end
    else
    begin
      if(t5_valid)
      begin
        t6_valid <= 1'b1;                           
        if(!stbc)
        begin
          /* no stbc */
          t6_x0_i  <= {t5_r1_h1_i[18],t5_r1_h1_i};        
          t6_x0_q  <= {t5_r1_h1_q[18],t5_r1_h1_q}; 
          t6_x1_i  <= 20'b0; 
          t6_x1_q  <= 20'b0; 
        end
        else
        begin
          /* stbc */
          t6_x0_i  <= n_t6_x0_i;                        
          t6_x0_q  <= n_t6_x0_q;                        
          t6_x1_i  <= n_t6_x1_i;                        
          t6_x1_q  <= n_t6_x1_q;                        
        end
      end
      else
      begin
        t6_valid  <=  1'b0;         
        t6_x0_i   <= 20'b0;      
        t6_x0_q   <= 20'b0;      
        t6_x1_i   <= 20'b0;      
        t6_x1_q   <= 20'b0;      
      end      
    end
  end

  /*****************************************************************************
  * qbpsk accumulator 
  *****************************************************************************/
  wire          t6_sgn_x_i;
  wire  [19:0]  t6_abs_x_i;
  wire          t6_sgn_x_q;
  wire  [19:0]  t6_abs_x_q;
  
  wire  [25:0]  n_qbpsk_accu_i; // note: sized for nsd=48
  wire  [25:0]  n_qbpsk_accu_q;

  assign t6_sgn_x_i    = t6_x0_i[19];
  assign t6_abs_x_i    = t6_x0_i^{20{t6_sgn_x_i}};
  assign n_qbpsk_accu_i = qbpsk_accu_i + {6'b0,t6_abs_x_i} + {25'b0,t6_sgn_x_i};

  assign t6_sgn_x_q    = t6_x0_q[19];
  assign t6_abs_x_q    = t6_x0_q^{20{t6_sgn_x_q}};
  assign n_qbpsk_accu_q = qbpsk_accu_q + {6'b0,t6_abs_x_q} + {25'b0,t6_sgn_x_q};
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      qbpsk_accu_i   <= 26'b0;
      qbpsk_accu_q   <= 26'b0;
    end
    else if(!enable)
    begin
      qbpsk_accu_i   <= 26'b0;
      qbpsk_accu_q   <= 26'b0;
    end
    else
    begin
      /* qbpsk detection */
      if(qbpsk_detect_clr)
      begin
        qbpsk_accu_i <= 26'b0;
        qbpsk_accu_q <= 26'b0;
      end
      else if(qbpsk_detect_en)
      begin
        if(t6_valid)
        begin
          qbpsk_accu_i <= n_qbpsk_accu_i;
          qbpsk_accu_q <= n_qbpsk_accu_q;
        end
      end
    end
  end
  
endmodule
`default_nettype wire
