/*******************************************************************************
*  Copyright (C) by RivieraWaves.
*  This module is a confidential and proprietary property of RivieraWaves
*  and a possession or use of this module requires written permission
*  from RivieraWaves.
********************************************************************************
* $Author: $
* Company          : RivieraWaves
********************************************************************************
* $Revision: $
* $Date: $
********************************************************************************
* Dependencies     :                                       
* Description      : 11ax deinterleaver
* Simulation Notes :                                       
* Synthesis Notes  :                                        
* Application Note :                                        
* Simulator        :                                       
* Parameters       :             
* Terms & concepts : 
* Bugs             :
* Open issues and future enhancements :         
* References       :
* Revision History : 
********************************************************************************
*                                     
* $HeadURL: $
*
*******************************************************************************/
`default_nettype none
module rx_fd_deinterleaver
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input  wire        clk,
  input  wire        rst_n,

  /*****************************************************************************
  * control
  *
  * fec:       0=BCC  /1=LDPC
  * nsd:       NSD_48 / NSD_52 / NSD_108 / NSD_234
  * ruen:      0=off  / 1=on (nsd ignored, subcarrier set defined by rulen) 
  * dcm:       0=off  / 1=on
  * iss:       0=SS1  / 1=SS2  / 2=SS3 / 3=SS4
  * sigben:    0=disable / 1=special layout to store up to 4 20M symbols per buffer
  *                          the layout is detemined by nsd=48 or 52
  * sigsel:    0,1,2,3   offset of the symbol in the buffer
  *****************************************************************************/
  input  wire        enable,
  input  wire        start,
  output reg         busy,
  
  input  wire        fec,
  input  wire [ 1:0] nsd,
  input  wire [ 2:0] nbpsc,
  input  wire        ruen,
  input  wire [ 2:0] rulen,
  input  wire        dcm,
  input  wire [ 1:0] iss,
  input  wire [ 1:0] sigsel,
  
  /*****************************************************************************
  *  equalizer interface
  *****************************************************************************/
  input  wire [ 4:0] sb0,
  input  wire [ 4:0] sb1,
  input  wire [ 4:0] sb2,
  input  wire [ 4:0] sb3,
  input  wire [ 4:0] sb4,
  input  wire [ 4:0] sb5,
  input  wire [ 4:0] sb6,
  input  wire [ 4:0] sb7,
  input  wire [ 4:0] sb8,
  input  wire [ 4:0] sb9,
  input  wire        valid,
  
  /*****************************************************************************
  * sram interface
  *****************************************************************************/
  output wire        wen,
  output wire [ 8:0] waddr,
  output wire [49:0] wdata
);
  
  /*****************************************************************************
  * declaration
  *****************************************************************************/
  /* constants */
  localparam  NSD_48=2'd0, 
              NSD_52=2'd1, 
              NSD_108=2'd2, 
              NSD_234=2'd3;

  localparam  RU_26=3'd0,
              RU_52=3'd1,
              RU_106=3'd2,
              RU_242=3'd3,
              RU_484=3'd4,
              RU_996=3'd5;
  
  localparam  NBPSC_1=3'd0,
              NBPSC_2=3'd1,
              NBPSC_4=3'd2,
              NBPSC_6=3'd3,
              NBPSC_8=3'd4,
              NBPSC_10=3'd5;

  /*****************************************************************************
  * parameters decoding
  *****************************************************************************/
  reg  [ 2:0] s;    /* max(1,0.5*nbpsc) */
  reg  [ 7:0] mv;
  reg  [ 4:0] ncol,init_col,icol2,icol3,icol4;
  reg  [ 3:0] nrow,init_row,irow2,irow3,irow4;
  reg  [ 8:0] sig_offset;
  
  always @(*)
  begin
    case(nbpsc)
      NBPSC_1: {s,mv} = {3'd1, 8'b0001}; 
      NBPSC_2: {s,mv} = {3'd1, 8'b0001}; 
      NBPSC_4: {s,mv} = {3'd2, 8'b0011}; 
      NBPSC_6: {s,mv} = {3'd3, 8'b0111}; 
      default: {s,mv} = {3'd4, 8'b1111}; 
    endcase
    
    /* nonsigb symbol */                                                                                                           
    if(!ruen)                                                                                                                      
      case(nsd)                                                                                                                    
        NSD_48:    {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd16, 4'd3,  5'd0, 4'd0,  5'd0, 4'd0,  5'd0, 4'd0};  
        NSD_52:                                                                                                                    
          if(!dcm)                                                                                                                 
                   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd13, 4'd4,  5'd5, 4'd2,  5'd2, 4'd3,  5'd8, 4'd1};  
          else                                                                                                                     
                   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd13, 4'd2,  5'd0, 4'd0,  5'd0, 4'd0,  5'd0, 4'd0};  
                                                                                                                                   
        NSD_108:   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd18, 4'd6,  5'd9, 4'd4,  5'd4, 4'd5, 5'd14, 4'd3};  
        default:   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd26, 4'd9, 5'd12, 4'd8,  5'd6, 4'd4, 5'd19, 4'd3};  
      endcase                                                                                                                      
    else                                                                                                                           
    begin                                                                                                                          
      if(!dcm)                                                                                                                     
        case(rulen)                                                                                                                
          RU_26:   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = { 5'd8, 4'd3,  5'd1, 4'd1,  5'd0, 4'd2,  5'd2, 4'd0};  
          RU_52:   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd16, 4'd3,  5'd7, 4'd1,  5'd3, 4'd2, 5'd11, 4'd0};  
          RU_106:  {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd17, 4'd6,  5'd9, 4'd4,  5'd4, 4'd5, 5'd14, 4'd3};  
          default: {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd26, 4'd9, 5'd12, 4'd8,  5'd6, 4'd4, 5'd19, 4'd3};  
        endcase                                                                                                                    
      else                                                                                                                         
        case(rulen)                                                                                                                
          RU_26:   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = { 5'd4, 4'd3,  5'd1, 4'd1,  5'd0, 4'd2,  5'd2, 4'd0};  
          RU_52:   {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = { 5'd8, 4'd3,  5'd1, 4'd1,  5'd0, 4'd2,  5'd2, 4'd0};  
          RU_106:  {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd17, 4'd3,  5'd7, 4'd1,  5'd3, 4'd2, 5'd11, 4'd0};  
          default: {ncol, nrow, icol2, irow2, icol3, irow3, icol4, irow4} = {5'd13, 4'd9,  5'd6, 4'd4,  5'd3, 4'd2,  5'd9, 4'd6};  
        endcase                                                                                                                    
    end                                                                                                                            
    
    case(iss)                                                                                                                      
      2'd0:    {init_col, init_row} = {  5'd0,  4'd0};                                                                             
      2'd1:    {init_col, init_row} = { icol2, irow2};                                                                             
      2'd2:    {init_col, init_row} = { icol3, irow3};                                                                             
      default: {init_col, init_row} = { icol4, irow4};                                                                             
    endcase                                                                                                                        
      
    if(nsd==NSD_52)
      /* sig52 offset */
      case(sigsel)
        2'd0:    sig_offset = 9'd0;
        2'd1:    sig_offset = 9'd52;
        2'd2:    sig_offset = 9'd104;
        default: sig_offset = 9'd156;
      endcase
    else
      /* sig48 offset */
      case(sigsel)
        2'd0:    sig_offset = 9'd0;
        2'd1:    sig_offset = 9'd48;
        2'd2:    sig_offset = 9'd96;
        default: sig_offset = 9'd144;
      endcase
  end

  /*****************************************************************************
  * S0 : row/col sub-carrier tracking (same timing than input interface)
  *
  *  col,row   : coordinate of the sub-carrier 
  *  ccol,crow : position of the sub-carrier in the incoming stream 
  *
  *****************************************************************************/
  reg         s0_state;
 
  reg  [ 4:0] s0_ccol,      s0_col;                      
  wire [ 4:0] n_s0_ccol,    n_s0_col;  
  wire        s0_ccol_last, s0_col_last;                 
  
  reg  [ 3:0] s0_crow,      s0_row;                      
  wire [ 3:0] n_s0_crow,    n_s0_row;  
  wire        s0_crow_last, s0_row_last;                 
  
  reg  [39:0] s0_sb;
  wire        s0_valid;

  reg         s0_valid_1t;
  reg [39:0]  s0_sb_1t;
  reg [ 4:0]  s0_col_1t;
  reg [ 3:0]  s0_row_1t;
  reg         s0_crow_last_1t;
  reg         s0_ccol_last_1t;

  reg        s0_mux_sel;
  reg        s0_mux_valid;
  reg [39:0] s0_mux_sb;
  reg [ 4:0] s0_mux_col;
  reg [ 3:0] s0_mux_row;
  reg        s0_mux_crow_last;
  reg        s0_mux_ccol_last;

  assign s0_valid = valid;
  
  /* remove sub-carrier rotation */
  always @(*)
  begin
    case(nbpsc)
      NBPSC_1,NBPSC_2: /* NBPSC1: sb4 is used to de-interleave in parallel QBPSK SIG */
                   s0_sb = { 5'b0, 5'b0, 5'b0,  sb4, 5'b0, 5'b0, 5'b0,  sb0};
      NBPSC_4:
        if(!s0_col[0])
                   s0_sb = { 5'b0, 5'b0,  sb5,  sb4, 5'b0, 5'b0,  sb1,  sb0};
        else
                   s0_sb = { 5'b0, 5'b0,  sb4,  sb5, 5'b0, 5'b0,  sb0,  sb1};
      NBPSC_6:
        case(s0_col)
          5'd0,5'd3,5'd6,5'd9,5'd12,5'd15,5'd18,5'd21,5'd24:          
                   s0_sb = { 5'b0,  sb6,  sb5,  sb4, 5'b0,  sb2,  sb1,  sb0};
          5'd1,5'd4,5'd7,5'd10,5'd13,5'd16,5'd19,5'd22,5'd25:
                   s0_sb = { 5'b0,  sb5,  sb4,  sb6, 5'b0,  sb1,  sb0,  sb2};
          default: 
                   s0_sb = { 5'b0,  sb4,  sb6,  sb5, 5'b0,  sb0,  sb2,  sb1};
        endcase
      default:
        case(s0_col[1:0])
          2'd0:    s0_sb = {  sb7,  sb6,  sb5,  sb4,  sb3,  sb2,  sb1,  sb0};
          2'd1:    s0_sb = {  sb6,  sb5,  sb4,  sb7,  sb2,  sb1,  sb0,  sb3};
          2'd2:    s0_sb = {  sb5,  sb4,  sb7,  sb6,  sb1,  sb0,  sb3,  sb2};
          default: s0_sb = {  sb4,  sb7,  sb6,  sb5,  sb0,  sb3,  sb2,  sb1};
        endcase 
    endcase
  end
 
  /* count row and col */
  assign n_s0_ccol    = s0_ccol + 5'd1;
  assign s0_ccol_last = n_s0_ccol==ncol;
  assign n_s0_crow    = s0_crow + 4'd1;
  assign s0_crow_last = n_s0_crow==nrow;
  
  /* track row and col of incoming sub-carrier */
  assign n_s0_row     = s0_row + 4'd1; 
  assign s0_row_last  = n_s0_row==nrow;
  assign n_s0_col     = s0_col + 5'd1;
  assign s0_col_last  = n_s0_col==ncol;
  
  reg  [ 4:0] s0_initcol;
  wire [ 4:0] n_s0_initcol;
  assign n_s0_initcol = s0_initcol + 5'd1;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s0_state        <= 1'b0; 
      s0_mux_sel      <= 1'b0;
      s0_initcol      <= 5'd0;
      s0_ccol         <= 5'd0; 
      s0_crow         <= 4'd0; 
      s0_col          <= 5'd0; 
      s0_row          <= 4'd0; 
      s0_sb_1t        <= 40'd0;
      s0_row_1t       <= 4'd0;
      s0_col_1t       <= 5'd0;
      s0_crow_last_1t <= 4'd0;
      s0_ccol_last_1t <= 5'd0;
      s0_valid_1t     <= 1'b0;
    end
    else if(!enable)
    begin
      s0_state        <= 1'b0; 
      s0_mux_sel      <= 1'b0;
      s0_initcol      <= 5'd0;
      s0_ccol         <= 5'd0; 
      s0_crow         <= 4'd0; 
      s0_col          <= 5'd0; 
      s0_row          <= 4'd0; 
      s0_sb_1t        <= 40'd0;
      s0_row_1t       <= 4'd0;
      s0_col_1t       <= 5'd0;
      s0_crow_last_1t <= 4'd0;
      s0_ccol_last_1t <= 5'd0;
      s0_valid_1t     <= 1'b0;
    end
    else
    begin
      if(s0_mux_sel)
      begin
        s0_sb_1t        <= s0_sb;
        s0_row_1t       <= s0_row;
        s0_col_1t       <= s0_col;
        s0_crow_last_1t <= s0_crow_last;
        s0_ccol_last_1t <= s0_ccol_last;
        s0_valid_1t     <= s0_valid;
      end
      else
      begin
        s0_sb_1t        <= 40'd0;
        s0_row_1t       <= 4'd0;
        s0_col_1t       <= 5'd0;
        s0_crow_last_1t <= 4'd0;
        s0_ccol_last_1t <= 5'd0;
        s0_valid_1t     <= 1'b0;
      end
      
      if(!s0_state)
      begin
        s0_col <= 5'd0;
        s0_row <= 4'd0;
        s0_col <= 5'd0;
        if(start)
        begin
          s0_state   <= 1'b1;
          s0_mux_sel <= 1'b0;
          s0_initcol <= init_col;
          s0_col     <= init_col;
          s0_row     <= init_row;
        end
      end
      else
      begin
        if(s0_valid)
        begin
          /* count col and row */
          if(s0_ccol_last)
          begin
            if(s0_row_last)
            begin
              s0_mux_sel <= 1'b1;
              s0_initcol <= n_s0_initcol;
              s0_col     <= n_s0_initcol;
              s0_row     <= 4'd0;
            end
            else
            begin
              s0_col <= s0_initcol;
              s0_row <= n_s0_row;
            end
           
            s0_ccol <= 6'd0;
            if(s0_crow_last)
            begin
              s0_state <= 1'b0;
              s0_crow  <= 5'd0;
              s0_row   <= 4'd0;
              s0_col   <= 5'd0;
            end
            else
            begin
              s0_crow <= n_s0_crow;
            end
          end
          else
          begin
            s0_ccol <= n_s0_ccol;
            if(s0_col_last)
              s0_col <= 5'd0;
            else
              s0_col <= n_s0_col;
          end
        end
      end
    end
  end

  always @(*)
  begin
    if(!s0_mux_sel)
    begin
      s0_mux_valid     = s0_valid;
      s0_mux_sb        = s0_sb;
      s0_mux_col       = s0_col;
      s0_mux_row       = s0_row;
      s0_mux_ccol_last = s0_ccol_last;
      s0_mux_crow_last = s0_crow_last;
    end
    else
    begin
      s0_mux_valid     = s0_valid_1t;
      s0_mux_sb        = s0_sb_1t;
      s0_mux_col       = s0_col_1t;
      s0_mux_row       = s0_row_1t;
      s0_mux_ccol_last = s0_ccol_last_1t;
      s0_mux_crow_last = s0_crow_last_1t;
    end
  end

  /*****************************************************************************
  * S1: column wrap and ordering
  *****************************************************************************/
  /* wrap 7 entries fifo */
  reg          s1_locked;
  reg  [44:0]  s1_q0,s1_q1,s1_q2,s1_q3,s1_q4,s1_q5,s1_q6;
  reg  [ 2:0]  s1_wr_ptr;
  reg  [ 2:0]  s1_wr_count;
  reg          s1_wr_crow_last;
  reg  [ 3:0]  s1_wr_row;
  reg  [ 2:0]  s1_rd_ptr;
  reg  [ 2:0]  s1_rd_count;
  reg  [ 3:0]  s1_rd_row;
  reg          s1_rd_crow_last;
  
  wire [ 2:0]  n_s1_wr_ptr;
  wire [ 2:0]  n_s1_wr_count;
  wire [ 2:0]  n_s1_rd_ptr;
  wire [ 2:0]  n_s1_rd_count;
  
  assign n_s1_wr_ptr   = (s1_wr_ptr==3'd6)?3'd0:(s1_wr_ptr+3'd1);
  assign n_s1_wr_count = s1_wr_count + 3'd1;
  
  assign n_s1_rd_ptr = (s1_rd_ptr==3'd6)?3'd0:(s1_rd_ptr+3'd1);
  assign n_s1_rd_count = s1_rd_count - 3'd1;
  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1_locked       <= 1'b0;
      s1_wr_ptr       <= 3'd0;
      s1_wr_count     <= 3'd0;
      s1_wr_crow_last <= 1'b0;
      s1_wr_row       <= 4'd0;
      s1_rd_ptr       <= 3'd0;
      s1_rd_count     <= 3'd0;
      s1_rd_row       <= 4'd0;
      s1_rd_crow_last <= 1'b0;
      s1_q0           <= 46'd0;
      s1_q1           <= 46'd0;
      s1_q2           <= 46'd0;
      s1_q3           <= 46'd0;
      s1_q4           <= 46'd0;
      s1_q5           <= 46'd0;
      s1_q6           <= 46'd0; 
    end
    else if(!enable)
    begin
      s1_locked       <= 1'b0;
      s1_wr_ptr       <= 3'd0;
      s1_wr_count     <= 3'd0;
      s1_wr_crow_last <= 1'b0;
      s1_wr_row       <= 4'd0;
      s1_rd_ptr       <= 3'd0;
      s1_rd_count     <= 3'd0;
      s1_rd_row       <= 4'd0;
      s1_rd_crow_last <= 1'b0;
      s1_q0           <= 45'd0;
      s1_q1           <= 45'd0;
      s1_q2           <= 45'd0;
      s1_q3           <= 45'd0;
      s1_q4           <= 45'd0;
      s1_q5           <= 45'd0;
      s1_q6           <= 45'd0; 
    end
    else
    begin
      /* push */
      if(!s1_locked)
      begin
        if(s0_mux_valid)
        begin
          if(s0_mux_col[2:0]==3'd0)
          begin
            s1_locked <= 1'b1;
          end
          else
          begin
            s1_wr_row       <= s0_mux_row;
            s1_wr_crow_last <= s0_mux_crow_last;
            s1_wr_count     <= n_s1_wr_count;
            s1_wr_ptr       <= n_s1_wr_ptr;
            case(s1_wr_ptr)
              3'd0:    s1_q0 <= {s0_mux_col,s0_mux_sb}; 
              3'd1:    s1_q1 <= {s0_mux_col,s0_mux_sb}; 
              3'd2:    s1_q2 <= {s0_mux_col,s0_mux_sb}; 
              3'd3:    s1_q3 <= {s0_mux_col,s0_mux_sb}; 
              3'd4:    s1_q4 <= {s0_mux_col,s0_mux_sb};  
              3'd5:    s1_q5 <= {s0_mux_col,s0_mux_sb};  
              default: s1_q6 <= {s0_mux_col,s0_mux_sb};  
            endcase
          end
        end
      end
      
      /* pull */
      if(s0_mux_valid && s0_mux_ccol_last)
      begin
        s1_locked       <= 1'b0;
        s1_rd_row       <= s1_wr_row;
        s1_rd_crow_last <= s1_wr_crow_last;
        s1_rd_count     <= s1_wr_count;
        s1_wr_crow_last <= 1'b0;
        s1_wr_count     <= 3'd0;
      end
      else if(s1_rd_count!=3'd0)
      begin
        s1_rd_count <= n_s1_rd_count;
        s1_rd_ptr   <= n_s1_rd_ptr;
      end
    end
  end
  
  reg  [ 4:0] s1_q_col;
  reg  [39:0] s1_q_sb;
  
  always @(*)
  begin
    case(s1_rd_ptr)
      3'd0:    {s1_q_col,s1_q_sb} = s1_q0;
      3'd1:    {s1_q_col,s1_q_sb} = s1_q1;
      3'd2:    {s1_q_col,s1_q_sb} = s1_q2;
      3'd3:    {s1_q_col,s1_q_sb} = s1_q3;
      3'd4:    {s1_q_col,s1_q_sb} = s1_q4;
      3'd5:    {s1_q_col,s1_q_sb} = s1_q5;
      default: {s1_q_col,s1_q_sb} = s1_q6;
    endcase
  end 

  reg [ 3:0]  s1_mux_row;
  reg [ 4:0]  s1_mux_col;      
  reg         s1_mux_last;
  reg [39:0]  s1_mux_sb;       
  reg         s1_mux_valid;    

  always @(*)
  begin
    if(!s1_locked && s1_rd_count!=3'd0)                                       
    begin                                                                     
      s1_mux_row   = s1_rd_row;                                                  
      s1_mux_col   = s1_q_col;                                                   
      s1_mux_sb    = s1_q_sb;                                                    
      s1_mux_last  = s1_rd_crow_last && s1_rd_count==3'd1;                       
      s1_mux_valid = 1'b1;                                                       
    end                                                                       
    else if(s1_locked || s0_mux_col[2:0]==3'd0)                               
    begin                                                                     
      s1_mux_row   = s0_mux_row;                                                 
      s1_mux_col   = s0_mux_col;                                                 
      s1_mux_sb    = s0_mux_sb;                                                  
      s1_mux_last  = s1_wr_count==3'd0 && s0_mux_crow_last && s0_mux_ccol_last;  
      s1_mux_valid = s0_mux_valid;                                               
    end                                                                       
    else                                                                      
    begin                                                                     
      s1_mux_row   = 4'd0;                                                       
      s1_mux_col   = 5'd0;                                                       
      s1_mux_sb    = 40'b0;                                                      
      s1_mux_last  = 1'b0;                                                       
      s1_mux_valid = 1'b0;                                                       
    end                                                                       
  end
  
  wire [ 4:0] s1_mux_remcol;
  wire        s1_mux_remcol_lt_8;
  assign s1_mux_remcol      = ncol - s1_mux_col;
  assign s1_mux_remcol_lt_8 = s1_mux_remcol < 5'd8;

  /*****************************************************************************
  * S2: shift/transpose
  *****************************************************************************/
  reg [ 3:0]  s2_trans;
  reg [ 3:0]  s2_row;
  reg [ 4:0]  s2_col;      
  reg         s2_flush;
  reg [39:0]  s2;       
  reg         s2_last;    
  reg         s2_valid;    

  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2_flush <= 1'b0;
      s2_row   <= 4'd0;
      s2_col   <= 5'd0;
      s2       <= 40'b0;
      s2_trans <= 4'd0;
      s2_last  <= 1'b0;
      s2_valid <= 1'b0;
    end
    else if(!enable)
    begin
      s2_flush <= 1'b0;
      s2_row   <= 4'd0;
      s2_col   <= 5'd0;
      s2       <= 40'b0;
      s2_trans <= 4'd0;
      s2_last  <= 1'b0;
      s2_valid <= 1'b0;
    end
    else
    begin
      if(start)
        s2_flush <= 1'b0;
      else if(s2_valid && s2_last)
        s2_flush <= 1'b1;
        
      if(s1_mux_valid)
      begin
        if(s1_mux_col[2:0]==3'd0)
        begin
          if(s1_mux_remcol_lt_8)
            s2_trans <= s1_mux_remcol[3:0];
          else
            s2_trans <= 4'd8;
        end
        else
        begin
          s2_trans <= 4'd0;
        end
        
        s2_row   <= s1_mux_row;
        s2_col   <= s1_mux_col;
        s2       <= s1_mux_sb;
        s2_last  <= s1_mux_last;
        s2_valid <= s1_mux_valid;
      end
      else
      begin
        s2_valid <= 1'b0;
      end
    end
  end

  reg         s3_wen,   s4_wen,   s5_wen,   s6_wen,   s7_wen,   s8_wen,   s9_wen,   s10_wen;
  reg         s3_last,  s4_last,  s5_last,  s6_last,  s7_last,  s8_last,  s9_last,  s10_last;
  reg [ 3:0]  s3_trans, s4_trans, s5_trans, s6_trans, s7_trans, s8_trans, s9_trans;
  reg [ 3:0]  s3_row,   s4_row,   s5_row,   s6_row,   s7_row,   s8_row,   s9_row;
  reg [ 4:0]  s3_col,   s4_col,   s5_col,   s6_col,   s7_col,   s8_col,   s9_col;
  reg [39:0]  s3,       s4,       s5,       s6,       s7,       s8,       s9;
  reg [49:0]  s10; // last stage is shared with LDPC
  reg  [8:0]  s10_addr_last;
  
  
  wire [ 4:0] s2_7,s2_6,s2_5,s2_4,s2_3,s2_2,s2_1,s2_0;
  wire [ 4:0] s3_7,s3_6,s3_5,s3_4,s3_3,s3_2,s3_1,s3_0;
  wire [ 4:0] s4_7,s4_6,s4_5,s4_4,s4_3,s4_2,s4_1,s4_0;
  wire [ 4:0] s5_7,s5_6,s5_5,s5_4,s5_3,s5_2,s5_1,s5_0;
  wire [ 4:0] s6_7,s6_6,s6_5,s6_4,s6_3,s6_2,s6_1,s6_0;
  wire [ 4:0] s7_7,s7_6,s7_5,s7_4,s7_3,s7_2,s7_1,s7_0;
  wire [ 4:0] s8_7,s8_6,s8_5,s8_4,s8_3,s8_2,s8_1,s8_0;
  wire [ 4:0] s9_7,s9_6,s9_5,s9_4,s9_3,s9_2,s9_1,s9_0;
  wire [ 4:0] s10_7,s10_6,s10_5,s10_4,s10_3,s10_2,s10_1,s10_0;
  
  assign {s2_7,s2_6,s2_5,s2_4,s2_3,s2_2,s2_1,s2_0} = s2;
  assign {s3_7,s3_6,s3_5,s3_4,s3_3,s3_2,s3_1,s3_0} = s3;
  assign {s4_7,s4_6,s4_5,s4_4,s4_3,s4_2,s4_1,s4_0} = s4;
  assign {s5_7,s5_6,s5_5,s5_4,s5_3,s5_2,s5_1,s5_0} = s5;
  assign {s6_7,s6_6,s6_5,s6_4,s6_3,s6_2,s6_1,s6_0} = s6;
  assign {s7_7,s7_6,s7_5,s7_4,s7_3,s7_2,s7_1,s7_0} = s7;
  assign {s8_7,s8_6,s8_5,s8_4,s8_3,s8_2,s8_1,s8_0} = s8;
  assign {s9_7,s9_6,s9_5,s9_4,s9_3,s9_2,s9_1,s9_0} = s9;
  assign {s10_7,s10_6,s10_5,s10_4,s10_3,s10_2,s10_1,s10_0} = s10[39:0];

 
  /*****************************************************************************
  * hesigb symbol:   base = 13*row + 52*hesigbsel + 8*floor(col/8) 
  * normal symbol:   base = 26*row + 104*lsigsel  + 8*floor(col/8) 
  *****************************************************************************/
  wire [8:0]  n_s10_base, n_s10_nonsig_base, n_s10_sig48_base, n_s10_sig52_base, n_s10_addr;
  reg  [8:0]  s10_addr;
  
  /* all symbols except hesigb and lsig with subband mode */
  assign n_s10_nonsig_base =  { 4'b0,      s9_row, 1'b0} + 
                              { 2'b0,      s9_row, 3'b0} +
                              { 1'b0,      s9_row, 4'b0} +
                              { 4'b0, s9_col[4:3],3'b0};
  
  /* sig48 memory layout */
  assign n_s10_sig48_base =                   sig_offset +
                              { 1'b0,      s9_row, 4'b0} +
                              { 4'b0,  s9_col[4:3],3'b0};
                     
  /* sig52 memory layout */
  assign n_s10_sig52_base =                   sig_offset +
                              { 5'b0,            s9_row} + 
                              { 3'b0,      s9_row, 2'b0} +
                              { 2'b0,      s9_row, 3'b0} +
                              { 4'b0,  s9_col[4:3],3'b0};

  assign n_s10_base = (!ruen && nsd==NSD_48)?n_s10_sig48_base:
                      (!ruen && nsd==NSD_52)?n_s10_sig52_base:
                      n_s10_nonsig_base;
  
  assign n_s10_addr = s10_addr + {8'd0,s9_wen};
  
  // for LDPC
  always @(*)
  begin
    if (ruen) begin
      case(rulen)
        RU_26:   s10_addr_last = 9'd22;  // RU26 : 26-2pilots-2
        RU_52:   s10_addr_last = 9'd46;  // RU52 : 52-4pilots-2
        RU_106:  s10_addr_last = 9'd100; // RU106: 106-4pilots-2
        RU_242:  s10_addr_last = 9'd232; // RU242: 242-8pilots-2
        default: s10_addr_last = 9'd466; // RU484: 484-16pilots-2
      endcase
    end else begin
      case (nsd)
        NSD_48  : s10_addr_last = 9'd46;
        NSD_52  : s10_addr_last = 9'd50;  
        NSD_108 : s10_addr_last = 9'd106;
        default : s10_addr_last = 9'd232;
      endcase
    end
  end
  
  always @(posedge clk , negedge rst_n)
  begin
    if(!rst_n)
    begin
      { s3_wen, s3_last, s3_trans, s3_row, s3_col, s3} <= 55'b0;
      { s4_wen, s4_last, s4_trans, s4_row, s4_col, s4} <= 55'b0;
      { s5_wen, s5_last, s5_trans, s5_row, s5_col, s5} <= 55'b0;
      { s6_wen, s6_last, s6_trans, s6_row, s6_col, s6} <= 55'b0;
      { s7_wen, s7_last, s7_trans, s7_row, s7_col, s7} <= 55'b0;
      { s8_wen, s8_last, s8_trans, s8_row, s8_col, s8} <= 55'b0;
      { s9_wen, s9_last, s9_trans, s9_row, s9_col, s9} <= 55'b0;
      {s10_wen,s10_last,                 s10_addr,s10} <= 60'b0;
    end
    else if(!enable)
    begin
      { s3_wen, s3_last, s3_trans, s3_row, s3_col, s3} <= 55'b0;
      { s4_wen, s4_last, s4_trans, s4_row, s4_col, s4} <= 55'b0;
      { s5_wen, s5_last, s5_trans, s5_row, s5_col, s5} <= 55'b0;
      { s6_wen, s6_last, s6_trans, s6_row, s6_col, s6} <= 55'b0;
      { s7_wen, s7_last, s7_trans, s7_row, s7_col, s7} <= 55'b0;
      { s8_wen, s8_last, s8_trans, s8_row, s8_col, s8} <= 55'b0;
      { s9_wen, s9_last, s9_trans, s9_row, s9_col, s9} <= 55'b0;
      {s10_wen,s10_last,                 s10_addr,s10} <= 60'b0;
    end
    else if(fec)
    begin
      s10_last <= 1'b0; // Pulse
      if (start) begin
        s10_addr    <= 9'b111111111;
      end else
      if (valid) begin
        s10_wen  <= 1'b1;
        s10_addr <= s10_addr + 8'd1;
        case(nbpsc)
          NBPSC_1: s10 <= {  45'd0,                                              sb0};
          NBPSC_2: s10 <= {  40'd0,                                        sb4,  sb0};
          NBPSC_4: s10 <= {  30'd0,                            sb5,  sb4,  sb1,  sb0};
          NBPSC_6: s10 <= {  20'd0,                sb6,  sb5,  sb4,  sb2,  sb1,  sb0};
          // for Nbpsc=8 or 10, there is no specific SB shift in Equaliser, so just copy as they come
          default: s10 <= {  sb9, sb8, sb7,  sb6,  sb5,  sb4,  sb3,  sb2,  sb1,  sb0};
        endcase
        s10_last <= (s10_addr == s10_addr_last);
      end else begin
        s10_wen  <= 1'b0;
      end
    end
    else
    begin
      
      { s10_wen, s10_last, s10} <= 52'b0;
      /* push sub-carrier */
      if(s2_valid || s2_flush)
      begin
        if(!s2_valid)
          { s3_wen, s3_last, s3_trans, s3_row, s3_col, s3} <= 55'b0;
        else
          { s3_wen, s3_last, s3_trans, s3_row, s3_col, s3} <= {1'b0, s2_last, s2_trans, s2_row, s2_col, s2};
      
       { s4_wen, s4_last, s4_trans, s4_row, s4_col, s4} <= { s3_wen, s3_last, s3_trans, s3_row, s3_col, s3};   
       { s5_wen, s5_last, s5_trans, s5_row, s5_col, s5} <= { s4_wen, s4_last, s4_trans, s4_row, s4_col, s4};
       { s6_wen, s6_last, s6_trans, s6_row, s6_col, s6} <= { s5_wen, s5_last, s5_trans, s5_row, s5_col, s5};
       { s7_wen, s7_last, s7_trans, s7_row, s7_col, s7} <= { s6_wen, s6_last, s6_trans, s6_row, s6_col, s6};
       { s8_wen, s8_last, s8_trans, s8_row, s8_col, s8} <= { s7_wen, s7_last, s7_trans, s7_row, s7_col, s7};
       { s9_wen, s9_last, s9_trans, s9_row, s9_col, s9} <= { s8_wen, s8_last, s8_trans, s8_row, s8_col, s8};
       {s10_wen,s10_last,           s10_addr,s10[39:0]} <= { s9_wen, s9_last,               n_s10_addr, s9};
      end
      else
      begin
        /* flush memory lines */
        if(s9_wen)
        begin
          { s9_wen, s9_last, s9_trans, s9_row, s9_col, s9} <= 55'b0;
          {s10_wen,s10_last,                 s10_addr,s10[39:0]} <= { s9_wen, s9_last,               n_s10_addr, s9};
        end
        if(s8_wen)
        begin 
          { s8_wen, s8_last, s8_trans, s8_row, s8_col, s8} <= 55'b0;
          { s9_wen, s9_last, s9_trans, s9_row, s9_col, s9} <= { s8_wen, s8_last, s8_trans, s8_row, s8_col, s8};
        end
        if(s7_wen) 
        begin
          { s7_wen, s7_last, s7_trans, s7_row, s7_col, s7} <= 55'b0;
          { s8_wen, s8_last, s8_trans, s8_row, s8_col, s8} <= { s7_wen, s7_last, s7_trans, s7_row, s7_col, s7}; 
        end
        if(s6_wen)
        begin
          { s6_wen, s6_last, s6_trans, s6_row, s6_col, s6} <= 55'b0;
          { s7_wen, s7_last, s7_trans, s7_row, s7_col, s7} <= { s6_wen, s6_last, s6_trans, s6_row, s6_col, s6};
        end
        if(s5_wen)
        begin
          { s5_wen, s5_last, s5_trans, s5_row, s5_col, s5} <= 55'b0;
          { s6_wen, s6_last, s6_trans, s6_row, s6_col, s6} <= { s5_wen, s5_last, s5_trans, s5_row, s5_col, s5}; 
        end
        if(s4_wen)
        begin
          { s4_wen, s4_last, s4_trans, s4_row, s4_col, s4} <= 55'b0;
          { s5_wen, s5_last, s5_trans, s5_row, s5_col, s5} <= { s4_wen, s4_last, s4_trans, s4_row, s4_col, s4};
        end
        if(s3_wen)
        begin
          { s3_wen, s3_last, s3_trans, s3_row, s3_col, s3} <= 55'b0;
          { s4_wen, s4_last, s4_trans, s4_row, s4_col, s4} <= { s3_wen, s3_last, s3_trans, s3_row, s3_col, s3};       
        end
      end
      
      /*************************************************************************
      * transpose array
      **************************************************************************
      *  possible trans codes are :
      *
      *          ncol            trans
      *            26 =  3*8 +   2
      *            18 =  2*8 +   2
      *            17 =  2*8 +   1
      *            16 =  1*8 +   8
      *            13 =  1*8 +   5
      *             8 =  0*8 +   8
      *             4 =  0*8 +   4
      *************************************************************************/ 
      if(s2_valid || s2_flush)
      begin
        case(s9_trans)
          4'd1:
          begin
            { s10_wen, s10_last,                  s10_addr, s10[39:0]} <= { 1'b1, s9_last,               n_s10_base, s9_7,s9_6,s9_5,s9_4,s9_3,s9_2,s9_1,s9_0};
          end               
          4'd2:
          begin
            {  s9_wen,  s9_last,  s9_trans, s9_row, s9_col,  s9}       <= { 1'b1, s8_last,     4'd0,   4'd0,   5'd0, s8_7,s9_7,s8_6,s9_6,s8_5,s9_5,s8_4,s9_4};
            { s10_wen, s10_last,                  s10_addr, s10[39:0]} <= { 1'b1,    1'b0,               n_s10_base, s8_3,s9_3,s8_2,s9_2,s8_1,s9_1,s8_0,s9_0};
          end
          4'd4:
          begin
            {  s7_wen,  s7_last,  s7_trans, s7_row, s7_col,  s7}       <= { 1'b1, s6_last,     4'd0,   4'd0,   5'd0, s6_7,s7_7,s8_7,s9_7,s6_6,s7_6,s8_6,s9_6};
            {  s8_wen,  s8_last,  s8_trans, s8_row, s8_col,  s8}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s6_5,s7_5,s8_5,s9_5,s6_4,s7_4,s8_4,s9_4};
            {  s9_wen,  s9_last,  s9_trans, s9_row, s9_col,  s9}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s6_3,s7_3,s8_3,s9_3,s6_2,s7_2,s8_2,s9_2};
            { s10_wen, s10_last,                  s10_addr, s10[39:0]} <= { 1'b1,    1'b0,               n_s10_base, s6_1,s7_1,s8_1,s9_1,s6_0,s7_0,s8_0,s9_0};
          end
          4'd5:
          begin
            {  s6_wen,  s6_last,  s6_trans, s6_row, s6_col,  s6}       <= { 1'b1, s5_last,     4'd0,   4'd0,   5'd0, s5_7,s6_7,s7_7,s8_7,s9_7,s5_6,s6_6,s7_6};
            {  s7_wen,  s7_last,  s7_trans, s7_row, s7_col,  s7}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s8_6,s9_6,s5_5,s6_5,s7_5,s8_5,s9_5,s5_4};
            {  s8_wen,  s8_last,  s8_trans, s8_row, s8_col,  s8}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s6_4,s7_4,s8_4,s9_4,s5_3,s6_3,s7_3,s8_3};
            {  s9_wen,  s9_last,  s9_trans, s9_row, s9_col,  s9}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s9_3,s5_2,s6_2,s7_2,s8_2,s9_2,s5_1,s6_1};
            { s10_wen, s10_last,                  s10_addr, s10[39:0]} <= { 1'b1,    1'b0,               n_s10_base, s7_1,s8_1,s9_1,s5_0,s6_0,s7_0,s8_0,s9_0};
          end
          4'd8:
          begin
            {  s3_wen,  s3_last,  s3_trans, s3_row, s3_col,  s3}       <= { 1'b1, s2_last,     4'd0,   4'd0,   5'd0, s2_7,s3_7,s4_7,s5_7,s6_7,s7_7,s8_7,s9_7};
            {  s4_wen,  s4_last,  s4_trans, s4_row, s4_col,  s4}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s2_6,s3_6,s4_6,s5_6,s6_6,s7_6,s8_6,s9_6};
            {  s5_wen,  s5_last,  s5_trans, s5_row, s5_col,  s5}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s2_5,s3_5,s4_5,s5_5,s6_5,s7_5,s8_5,s9_5};
            {  s6_wen,  s6_last,  s6_trans, s6_row, s6_col,  s6}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s2_4,s3_4,s4_4,s5_4,s6_4,s7_4,s8_4,s9_4};
            {  s7_wen,  s7_last,  s7_trans, s7_row, s7_col,  s7}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s2_3,s3_3,s4_3,s5_3,s6_3,s7_3,s8_3,s9_3};
            {  s8_wen,  s8_last,  s8_trans, s8_row, s8_col,  s8}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s2_2,s3_2,s4_2,s5_2,s6_2,s7_2,s8_2,s9_2};
            {  s9_wen,  s9_last,  s9_trans, s9_row, s9_col,  s9}       <= { 1'b1,    1'b0,     4'd0,   4'd0,   5'd0, s2_1,s3_1,s4_1,s5_1,s6_1,s7_1,s8_1,s9_1};
            { s10_wen, s10_last,                  s10_addr, s10[39:0]} <= { 1'b1,    1'b0,               n_s10_base, s2_0,s3_0,s4_0,s5_0,s6_0,s7_0,s8_0,s9_0};
          end
          default:  ;
        endcase
      end
    end
  end

  assign wen   = s10_wen;
  assign waddr = s10_addr;
  assign wdata = s10;

  /*****************************************************************************
  * write memory
  *****************************************************************************/  
  always @(posedge clk, negedge rst_n)
  begin
    if(!rst_n)
    begin
      busy            <= 1'b0;
    end
    else if(!enable)
    begin
      busy            <= 1'b0;
    end
    else
    begin
      if(!busy)
      begin
        if(start)
          busy <= 1'b1;
      end
      else
      begin
        if(s10_wen && s10_last)           
        begin                 
          busy     <= 1'b0;   
        end                   
      end
    end
  end
  
  /*****************************************************************************
  * simu/debug
  *****************************************************************************/
  wire [ 4:0] s0_sb7,s0_sb6,s0_sb5,s0_sb4,s0_sb3,s0_sb2,s0_sb1,s0_sb0;
  assign {s0_sb7,s0_sb6,s0_sb5,s0_sb4,s0_sb3,s0_sb2,s0_sb1,s0_sb0} = s0_sb;
  
endmodule
`default_nettype wire
