`default_nettype wire
module rx_fd_ldpc_fifo
(
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire              clk,
  input wire              rst_n,
  
  /*****************************************************************************
  * system
  *****************************************************************************/
  input wire              enable,
  input wire    [ 8:0]    fifo_threshold,
  input wire    [ 1:0]    nsd,
  input wire              dcm,
  input wire              ruen,
  input wire    [ 2:0]    rulen,
  input wire    [ 2:0]    nbpsc,
  
  /*****************************************************************************
  * sc stream
  *****************************************************************************/
  output wire             in_ready,
  input wire    [4:0]     in_sb0,
  input wire    [4:0]     in_sb1,
  input wire    [4:0]     in_sb2,
  input wire    [4:0]     in_sb3,
  input wire    [4:0]     in_sb4,
  input wire    [4:0]     in_sb5,
  input wire    [4:0]     in_sb6,
  input wire    [4:0]     in_sb7,
  input wire    [4:0]     in_sb8,
  input wire    [4:0]     in_sb9,
  input wire              in_valid,
  
  /*****************************************************************************
  * sb stream
  *****************************************************************************/
  input  wire             out_ready,
  output wire  [ 4:0]     out_sb0,
  output wire  [ 4:0]     out_sb1,
  output wire  [ 4:0]     out_sb2,
  output wire  [ 4:0]     out_sb3,
  output wire  [ 4:0]     out_sb4,
  output wire  [ 4:0]     out_sb5,
  output wire  [ 4:0]     out_sb6,
  output wire  [ 4:0]     out_sb7,
  output wire  [ 4:0]     out_sb8,
  output wire  [ 4:0]     out_sb9,
  output wire  [ 4:0]     out_sb10,
  output wire  [ 4:0]     out_sb11,
  output wire  [ 3:0]     out_len,
  output wire             out_last, 
  output wire             out_valid,
  
  /*****************************************************************************
  * 
  *****************************************************************************/
  output wire            wen,
  output wire  [ 6:0]    waddr,
  output wire  [59:0]    wdata,
  output wire            ren,
  output wire  [ 6:0]    raddr,
  input  wire  [59:0]    rdata
);
  
  /*****************************************************************************
  *
  *****************************************************************************/
  localparam   NBPSC_1=3'd0,
               NBPSC_2=3'd1,  
               NBPSC_4=3'd2,  
               NBPSC_6=3'd3,  
               NBPSC_8=3'd4,  
               NBPSC_10=3'd5;  
  
  localparam   RU_26=3'd0,
               RU_52=3'd1,  
               RU_106=3'd2,  
               RU_242=3'd3,  
               RU_468=3'd4;  
   
  localparam   NSD_48=2'd0,
               NSD_52=2'd1,  
               NSD_108=2'd2,  
               NSD_234=2'd3; 
   
   
  reg [ 8:0] dec_nsd;
  reg [ 3:0] dec_nbpsc;
  reg [49:0] in_data;
  
  always @(*)
  begin
    /* nsd */
    if(!ruen)
    begin
      case(nsd)
        NSD_48:  dec_nsd = 9'd48;
        NSD_52:  dec_nsd = 9'd52;
        NSD_108: dec_nsd = 9'd108;
        default: dec_nsd = 9'd234;
      endcase
    end
    else
    begin
      if(!dcm)
        case(rulen)
          RU_26:   dec_nsd = 9'd24;
          RU_52:   dec_nsd = 9'd48;
          RU_106:  dec_nsd = 9'd102;
          RU_242:  dec_nsd = 9'd234;
          default: dec_nsd = 9'd468;
        endcase
      else
        case(rulen)
          RU_26:   dec_nsd = 9'd12;
          RU_52:   dec_nsd = 9'd24;
          RU_106:  dec_nsd = 9'd51;
          RU_242:  dec_nsd = 9'd117;
          default: dec_nsd = 9'd234;
        endcase
    end
    
    /* nbpsc */
    case(nbpsc)
      NBPSC_1: {dec_nbpsc,in_data} = { 4'd1,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0, in_sb0};
      NBPSC_2: {dec_nbpsc,in_data} = { 4'd2,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0, in_sb4, in_sb0};
      NBPSC_4: {dec_nbpsc,in_data} = { 4'd4,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0,   5'b0, in_sb5, in_sb4, in_sb1, in_sb0};
      NBPSC_6: {dec_nbpsc,in_data} = { 4'd6,   5'b0,   5'b0,   5'b0,   5'b0, in_sb6, in_sb5, in_sb4, in_sb2, in_sb1, in_sb0};
      NBPSC_8: {dec_nbpsc,in_data} = { 4'd8,   5'b0,   5'b0, in_sb7, in_sb6, in_sb5, in_sb4, in_sb3, in_sb2, in_sb1, in_sb0};
      default: {dec_nbpsc,in_data} = {4'd10, in_sb9, in_sb8, in_sb7, in_sb6, in_sb5, in_sb4, in_sb3, in_sb2, in_sb1, in_sb0};
    endcase
  end
  
  /* ncbps */
  wire [12:0] dec_ncbps;
  assign dec_ncbps = 
    ({4'b0,     dec_nsd} & {13{dec_nbpsc[0]}}) +
    ({3'b0,dec_nsd,1'b0} & {13{dec_nbpsc[1]}}) +
    ({2'b0,dec_nsd,2'b0} & {13{dec_nbpsc[2]}}) +
    ({1'b0,dec_nsd,3'b0} & {13{dec_nbpsc[3]}});

  /*****************************************************************************
  * s0: ififo input fifo
  *
  * o incoming subcarrier softbits are packed into a line of 12 sb, matching a
  *   data line of the memory, except the last line finishing the symbol that may
  *   may contains less than 12 (ncbps%12).
  *****************************************************************************/
  reg  [109:0] s0_ififo;         /* 22 sb   */
  wire [109:0] n_s0_ififo;       
  reg  [  4:0] s0_ififo_count;   /* 0-22 sb */
  wire [  4:0] n_s0_ififo_count; 
  
  wire [109:0] s0_ififo_in_sh1,s0_ififo_in_sh2,s0_ififo_in_sh3,s0_ififo_in_sh4,s0_ififo_in_sh5;
  wire [109:0] s0_ififo_out_sh1,s0_ififo_out_sh2,s0_ififo_out_sh3,s0_ififo_out_sh4;
 
  wire [  3:0] s0_ififo_wrlen,s0_ififo_rdlen;
  wire [  4:0] s0_ififo_shsel;
  
  wire [ 49:0] in_data_mask;
  
  assign in_data_mask     = in_data & {50{in_valid}};
  
  assign s0_ififo_wrlen   = dec_nbpsc & {4{in_valid}};
  
  assign s0_ififo_shsel   = s0_ififo_count - {1'b0,s0_ififo_rdlen}; 
  assign s0_ififo_in_sh1  = s0_ififo_shsel[0]?{     55'b0,in_data_mask,5'b0}:{60'b0,in_data_mask};
  assign s0_ififo_in_sh2  = s0_ififo_shsel[1]?{s0_ififo_in_sh1[ 99:0],10'b0}:s0_ififo_in_sh1;
  assign s0_ififo_in_sh3  = s0_ififo_shsel[2]?{s0_ififo_in_sh2[ 89:0],20'b0}:s0_ififo_in_sh2;
  assign s0_ififo_in_sh4  = s0_ififo_shsel[3]?{s0_ififo_in_sh3[ 69:0],40'b0}:s0_ififo_in_sh3;
  assign s0_ififo_in_sh5  = s0_ififo_shsel[4]?{s0_ififo_in_sh4[ 29:0],80'b0}:s0_ififo_in_sh4;
  
  assign s0_ififo_out_sh1 = s0_ififo_rdlen[0]?{          5'b0,s0_ififo[109:5]}:s0_ififo;
  assign s0_ififo_out_sh2 = s0_ififo_rdlen[1]?{10'b0,s0_ififo_out_sh1[109:10]}:s0_ififo_out_sh1;
  assign s0_ififo_out_sh3 = s0_ififo_rdlen[2]?{20'b0,s0_ififo_out_sh2[109:20]}:s0_ififo_out_sh2;
  assign s0_ififo_out_sh4 = s0_ififo_rdlen[3]?{40'b0,s0_ififo_out_sh3[109:40]}:s0_ififo_out_sh3;

  assign n_s0_ififo       = s0_ififo_in_sh5 | s0_ififo_out_sh4;
  assign n_s0_ififo_count = s0_ififo_count + {1'b0,s0_ififo_wrlen} - {1'b0,s0_ififo_rdlen}; 
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s0_ififo       <= 110'b0;
      s0_ififo_count <= 5'd0;
    end
    else if(!enable)
    begin
      s0_ififo       <= 110'b0;
      s0_ififo_count <= 5'd0; 
    end
    else
    begin
      s0_ififo       <= n_s0_ififo;
      s0_ififo_count <= n_s0_ififo_count; 
    end 
  end
  
  /* mask unwanted sb */
  wire [59:0] s0_ififo_mask1,s0_ififo_mask2,s0_ififo_mask3,s0_ififo_mask4;
  assign s0_ififo_mask1 = s0_ififo_rdlen[0]?{55'b0,{ 5{1'b1}}}:60'b0;
  assign s0_ififo_mask2 = s0_ififo_rdlen[1]?{s0_ififo_mask1[49:0],{10{1'b1}}}:s0_ififo_mask1;
  assign s0_ififo_mask3 = s0_ififo_rdlen[2]?{s0_ififo_mask2[39:0],{20{1'b1}}}:s0_ififo_mask2;
  assign s0_ififo_mask4 = s0_ififo_rdlen[3]?{s0_ififo_mask3[19:0],{40{1'b1}}}:s0_ififo_mask3;
  
  wire [ 4:0] d21,d20,d19,d18,d17,d16,d15,d14,d13,d12,d11,d10,d9,d8,d7,d6,d5,d4,d3,d2,d1,d0;
  assign {d21,d20,d19,d18,d17,d16,d15,d14,d13,d12,d11,d10,d9,d8,d7,d6,d5,d4,d3,d2,d1,d0} = s0_ififo;
  
  /*****************************************************************************
  * s1w
  *****************************************************************************/
  reg  [ 12:0]  s1w_count;
  wire [ 12:0]  n_s1w_count;
  wire [ 12:0]  s1w_count_incr;
  wire [ 12:0]  s1w_remaining;
  wire          s1w_last;
  wire [ 3:0]   s1w_nsb;
 
  reg           s1w_en;
  reg  [  6:0]  s1w_addr;
  reg  [ 59:0]  s1w_data;
  
  assign s1w_remaining   = dec_ncbps - s1w_count;
  assign s1w_last        = s1w_remaining<=13'd12;
  assign s1w_nsb         = s1w_last?s1w_remaining[3:0]:4'd12;
 
  assign s0_ififo_rdlen  = (s0_ififo_count>={1'b0,s1w_nsb})?s1w_nsb:4'd0;
  
  assign s1w_count_incr  = s1w_count + {9'b0,s0_ififo_rdlen};
  assign n_s1w_count     = s1w_last?13'd0:s1w_count_incr;
 
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1w_count <= 13'd0;
      s1w_en   <= 1'b0;
      s1w_addr <= 7'd0;
      s1w_data <= 60'd0;
    end
    else if(!enable)
    begin
      s1w_count <= 13'd0;
      s1w_en   <= 1'b0;
      s1w_addr <= 7'd0;
      s1w_data <= 60'd0;
    end
    else
    begin
      if(s0_ififo_rdlen!=4'd0)
      begin
        s1w_count <= n_s1w_count;
        s1w_en   <= 1'b1;
        s1w_data <= s0_ififo[59:0] & s0_ififo_mask4;
      end
      else
      begin
        s1w_en   <= 1'b0;
     end
      if(s1w_en)
      begin
        s1w_addr <= s1w_addr + 7'd1;
      end
    end
  end
  
  assign wen   = s1w_en;
  assign waddr = s1w_addr;
  assign wdata = s1w_data; 
  
  /*****************************************************************************
  * s2
  *****************************************************************************/
  reg  [ 6:0] s2_mfifo_count;
  wire [ 6:0] n_s2_mfifo_count;
  reg         s3r_en;
  
  /* use 24 memory lines as stall buffer that covers the equalizer + s0/s1 stages latency */
  assign in_ready = {2'b0,s2_mfifo_count}<fifo_threshold;
  
  assign n_s2_mfifo_count = s2_mfifo_count + {6'b0,s1w_en} - {6'b0,s3r_en};
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2_mfifo_count <= 7'd0;
    end
    else if(!enable)
    begin
      s2_mfifo_count <= 7'd0;
    end
    else
    begin
      s2_mfifo_count <= n_s2_mfifo_count;
    end
  end
  
  /*****************************************************************************
  * s1r: mem request 
  *****************************************************************************/
  wire [12:0]  n_s1r_count;
  reg  [12:0]  s1r_count;
  wire [12:0]  s1r_count_incr;
  wire [12:0]  s1r_remaining;
  wire         n_s1r_last;
  wire [ 3:0]  n_s1r_nsb;
  
  reg  [ 6:0]  s1r_mfifo_count;
  wire [ 6:0]  n_s1r_mfifo_count;
  
  wire         s1r_ofifo_rqen;
  reg  [ 2:0]  s1r_ofifo_rqcount;
  wire [ 2:0]  n_s1r_ofifo_rqcount;
  
  reg          s1r_en;
  reg  [ 6:0]  s1r_addr;
  reg  [ 3:0]  s1r_nsb;
  reg          s1r_last;

  wire         s4_ofifo_rden;
  
  /* sb count */
  assign s1r_remaining       = dec_ncbps - s1r_count;
  assign n_s1r_last          = s1r_remaining<=13'd12;
  assign n_s1r_nsb           = n_s1r_last?s1r_remaining[3:0]:4'd12;

  assign s1r_count_incr      = s1r_count + {9'd0,n_s1r_nsb};
  assign n_s1r_count         = n_s1r_last?13'd0:s1r_count_incr;
  
  /* ofifo */
  assign s1r_ofifo_rqen      = s1r_mfifo_count!=7'd0 && s1r_ofifo_rqcount<4'd4;
  assign n_s1r_ofifo_rqcount = s1r_ofifo_rqcount + {2'b0,s1r_ofifo_rqen} - {2'b0,s4_ofifo_rden};
  
  /* mfifo */
  assign n_s1r_mfifo_count   = s1r_mfifo_count + {6'b0,s1w_en} - {6'b0,s1r_ofifo_rqen};
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s1r_count         <= 13'd0;
      s1r_mfifo_count   <= 7'd0;
      s1r_ofifo_rqcount <= 3'd0;
      s1r_en            <= 1'b0;
      s1r_addr          <= 7'd0;
      s1r_nsb           <= 4'd0;
      s1r_last          <= 1'b0; 
    end
    else if(!enable)
    begin
      s1r_count         <= 13'd0;
      s1r_mfifo_count   <= 7'd0;
      s1r_ofifo_rqcount <= 3'd0;
      s1r_en            <= 1'b0;
      s1r_addr          <= 7'd0;
      s1r_nsb           <= 4'd0;
      s1r_last          <= 1'b0; 
    end
    else
    begin
      s1r_mfifo_count   <= n_s1r_mfifo_count;
      s1r_ofifo_rqcount <= n_s1r_ofifo_rqcount; 
      
      if(s1r_ofifo_rqen)
      begin
        s1r_count <= n_s1r_count;
        s1r_en    <= 1'b1;
        s1r_nsb   <= n_s1r_nsb;
        s1r_last  <= n_s1r_last; 
      end
      else
      begin
        s1r_en    <= 1'b0;
        s1r_nsb   <= 4'd0;
        s1r_last  <= 1'b0; 
      end
      if(s1r_en)
      begin
        s1r_addr  <= s1r_addr + 7'd1;
      end
    end
  end
  
  assign ren   = s1r_en;
  assign raddr = s1r_addr;
  
  /*****************************************************************************
  * s2r, s3r
  *****************************************************************************/
  reg        s2r_en;
  reg  [3:0] s2r_nsb,  s3r_nsb;
  reg        s2r_last, s3r_last;
  reg [59:0] s3r_data;
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s2r_en            <= 1'b0;
      s2r_nsb           <= 4'd0;
      s2r_last          <= 1'b0; 
      s3r_en            <= 1'b0;
      s3r_nsb           <= 4'd0;
      s3r_last          <= 1'b0;
      s3r_data          <= 60'd0; 
    end
    else if(!enable)
    begin
      s2r_en            <= 1'b0;
      s2r_nsb           <= 4'd0;
      s2r_last          <= 1'b0; 
      s3r_en            <= 1'b0;
      s3r_nsb           <= 4'd0;
      s3r_last          <= 1'b0;
      s3r_data          <= 60'd0; 
    end
    else
    begin
      s2r_en            <= s1r_en;
      s2r_nsb           <= s1r_nsb;
      s2r_last          <= s1r_last; 
      s3r_en            <= s2r_en;
      s3r_nsb           <= s2r_nsb;
      s3r_last          <= s2r_last;
      s3r_data          <= rdata; 
    end
  end
  
  wire [ 64:0] s3r_data_mask;
  assign s3r_data_mask = {s3r_last,s3r_nsb,s3r_data} & {65{s3r_en}};
 
  /*****************************************************************************
  * s4 ofifo
  *
  *  4 entries of 65 bits {last,len[3:0],data[59:0]}
  *
  *****************************************************************************/
  reg  [259:0] s4_ofifo;
  wire [259:0] n_s4_ofifo;
  wire [259:0] s4_ofifo_in_sh1,s4_ofifo_in_sh2;
  wire [259:0] s4_ofifo_out_sh1;
  
  reg  [  2:0] s4_ofifo_count;
  wire [  2:0] n_s4_ofifo_count;
  
  wire         s4_ofifo_wren;
  wire [  1:0] s4_ofifo_in_shsel;
  
  assign s4_ofifo_wren    = s3r_en;
  assign s4_ofifo_rden    = out_ready && s4_ofifo_count!=3'd0;
  assign s4_ofifo_in_shsel= s4_ofifo_count[1:0] - {1'b0,s4_ofifo_rden};
  
  assign s4_ofifo_in_sh1  = s4_ofifo_in_shsel[0]?{   130'd0,s3r_data_mask,65'd0}:{195'd0,s3r_data_mask};
  assign s4_ofifo_in_sh2  = s4_ofifo_in_shsel[1]?{s4_ofifo_in_sh1[129:0],130'd0}:s4_ofifo_in_sh1;
 
  assign s4_ofifo_out_sh1 = s4_ofifo_rden?{65'd0,s4_ofifo[259:65]}:s4_ofifo;
  
  assign n_s4_ofifo       = s4_ofifo_out_sh1 | s4_ofifo_in_sh2;
  assign n_s4_ofifo_count = s4_ofifo_count + {2'b0,s4_ofifo_wren} - {2'b0,s4_ofifo_rden};
  
  always @(posedge clk,negedge rst_n)
  begin
    if(!rst_n)
    begin
      s4_ofifo       <= 260'd0;
      s4_ofifo_count <= 4'd0; 
    end
    else if(!enable)
    begin
      s4_ofifo       <= 260'd0;
      s4_ofifo_count <= 4'd0; 
    end
    else
    begin
      s4_ofifo       <= n_s4_ofifo;
      s4_ofifo_count <= n_s4_ofifo_count; 
    end
  end
  
  assign {out_last,out_len} = s4_ofifo[64:60];
  assign {out_sb11,out_sb10,out_sb9,out_sb8,out_sb7,out_sb6,out_sb5,out_sb4,out_sb3,out_sb2,out_sb1,out_sb0} 
            = s4_ofifo[59:0];
    
  assign out_valid = s4_ofifo!=4'd0;
  
  
`ifdef RW_SIMU_ON
  integer mfifo_max;
  always @(posedge clk, negedge rst_n)
  begin:b_fifo_filling_checker
    reg v_enable_1t;
    if(!rst_n)
    begin
      v_enable_1t <= 1'b0;
      mfifo_max   <= 'd0;
    end
    else
    begin
      v_enable_1t <= enable;
      if( (s2_mfifo_count + {6'b0,s1w_en})>  mfifo_max)
      begin
        mfifo_max = (s2_mfifo_count + {6'b0,s1w_en});
        if(mfifo_max>='d128)
        begin
          $write("error:%m: mfifo overflow %0d\n",mfifo_max);
          $write("Simulation FAILURE\n");                                                               
          $stop(); 
        end
      end
    end
  end
`endif

endmodule
`default_nettype none
