//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39989 $
// $Date: 2019-10-03 15:33:10 +0200 (Thu, 03 Oct 2019) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of Rx Time Domain module
//                    This block includes:
//                    1) RX Time Domain Finite State Machine
//                    2) DC Offset Estimation and Compensation
//                    3) Time Domain Freq. Offset Estimation and Compensation
//                    4) Timing Boundary Estimation
// Simulation Notes :    
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/RxTDTop/verilog/rtl/RxTDTop.v $
//
//////////////////////////////////////////////////////////////////////////////

`default_nettype none

module RxTDTop (
  ///////////////////////////////////////////////
  // Reset
  ///////////////////////////////////////////////
  input  wire                                    nPhyRst,
  
  ///////////////////////////////////////////////
  // Clocks (120MHz)
  ///////////////////////////////////////////////
  input  wire                                    PhyClk,
  input  wire                                    TBEGClk,
  input  wire                                    TDFOEstGClk,
  input  wire                                    TDCompGClk,

  ///////////////////////////////////////////////
  // Registers
  ///////////////////////////////////////////////
  // To Time Domain DC offset
  input  wire        [6:0]                       CfgRegDelaySync,
  input  wire        [3:0]                       CfgRegStartDC,
  input  wire        [3:0]                       CfgRegStartHTDC,
  input  wire        [6:0]                       CfgRegWaitHTSTF,
  input  wire                                    CfgRegTDDCEn,
  
  // To Time Domain FSM
  input  wire                                    CfgRegForce20,         // 20 MHz BW Mode only
  input  wire                                    CfgRegForce40,         // 20-40 MHz BW Mode only
  input  wire        [11:0]                      CfgRegPlatFallTimeOut, // Time-out for plateau fall detection
  input  wire        [7:0]                       CfgRegTdCycRotVal20,
  input  wire        [7:0]                       CfgRegTdCycRotVal40,
  input  wire        [7:0]                       CfgRegTdCycRotVal80,
  input  wire        [4:0]                       CfgRegTdAdjust20ShortGI, // Adjust short GI timing boundary for 20MHz frames
  input  wire        [4:0]                       CfgRegTdAdjust40ShortGI, // Adjust short GI timing boundary for 40MHz frames
  input  wire        [4:0]                       CfgRegTdAdjust80ShortGI, // Adjust short GI timing boundary for 80MHz frames
  input  wire        [7:0]                       CfgRegTdSyncOffset2080,  // Adjust timing boundary for 80MHz frames
  input  wire        [7:0]                       CfgRegTdSyncOffset2040,  // Adjust timing boundary for 40MHz frames
  input  wire        [7:0]                       CfgRegTdSyncOffset20,    // Adjust timing boundary for 20MHz frames
  input  wire        [7:0]                       CfgRegTdDcHtStfMargin,   // Margin for HT-STF indication to TDDC
  input  wire                                    CfgRegSkipMidamble1x,    // Skip midamble HE-LTF 1x
  input  wire                                    CfgRegSkipMidamble2x,    // Skip midamble HE-LTF 2x
  input  wire                                    CfgRegSkipMidamble4x,    // Skip midamble HE-LTF 4x
  
  // To Time Domain Timing Boundary Estimation
  input  wire        [7:0]                       CfgRegTBECountAdjust20,
  input  wire        [6:0]                       CfgRegPeakSearchStart,
  input  wire        [6:0]                       CfgRegPeakSearchStop,
  input  wire        [4:0]                       CfgRegPeakSearchDelta,
  input  wire                                    CfgRegPeakSearchNdlEn,
  input  wire        [2:0]                       CfgRegPeakSearchNdlIndex,
  input  wire        [7:0]                       CfgRegTBEBias,
  
  // To Time Domain Frequency Offset Estimation and compensation
  input  wire        [9:0]                       CfgRegAutoCorrCompareRatioLow,
  input  wire        [9:0]                       CfgRegAutoCorrCompareRatioHigh,
  input  wire        [3:0]                       CfgRegAutoCorrFallCount,
  input  wire        [8:0]                       CfgRegAutoCorrPlatSumStart,
  input  wire        [8:0]                       CfgRegAutoCorrPlatSumEnd,
  input  wire        [8:0]                       CfgRegAutoCorrPlatFallSearchStart,
  input  wire        [3:0]                       CfgRegAutoCorrPlatFallIndexOffset,
  input  wire        [8:0]                       CfgRegAutoCorrPhSumStart,
  input  wire        [8:0]                       CfgRegAutoCorrPhSumEnd,
  input  wire                                    CfgRegTDFOCompEn,
  input  wire                                    CfgRegTDFOFineEstEn,
  input  wire                                    CfgRegTDFOCpeSlopeEn,
  
  // To status register
  output wire        [7:0]                       StatusTBECount,        // TBE counter
  output wire        [12:0]                      StatusMaxCorrValSum,   // Plateau fall counter
  output wire        [20:0]                      StatusCoarseFOAngle,   // Coarse FO Angle
  output wire        [20:0]                      StatusFineFOAngle,     // Fine FO Angle

  ///////////////////////////////////////////////
  // Data from Rx FE Chains
  ///////////////////////////////////////////////
  // Primary 20 MHz
  input  wire        [12:0]                      RxDataIn20PRe0,  // Real Comp of Primary Data Rx0
  input  wire        [12:0]                      RxDataIn20PIm0,  // Imag Comp of Primary Data Rx0
`ifdef RW_NX_DERIV_PATH1
  input  wire        [12:0]                      RxDataIn20PRe1,  // Real Comp of Primary Data Rx1
  input  wire        [12:0]                      RxDataIn20PIm1,  // Imag Comp of Primary Data Rx1
`endif
  input  wire                                    RxDataValidIn20, // Qualifies Rx Data Input
  // Secondary 20 MHz
  input  wire        [12:0]                      RxDataIn20SRe,   // Real Comp of Secondary Data Rx0
  input  wire        [12:0]                      RxDataIn20SIm,   // Imag Comp of Secondary Data Rx0
  // Primary 40 MHz
  input  wire        [12:0]                      RxDataIn40PRe0,  // Real Comp of Primary Data Rx0
  input  wire        [12:0]                      RxDataIn40PIm0,  // Imag Comp of Primary Data Rx0
`ifdef RW_NX_DERIV_PATH1
  input  wire        [12:0]                      RxDataIn40PRe1,  // Real Comp of Primary Data Rx1
  input  wire        [12:0]                      RxDataIn40PIm1,  // Imag Comp of Primary Data Rx1
`endif
  input  wire                                    RxDataValidIn40, // Qualifies Rx Data Input
  // Primary 80 MHz
  input  wire        [12:0]                      RxDataIn80PRe0,  // Real Comp of Primary Data Rx0
  input  wire        [12:0]                      RxDataIn80PIm0,  // Imag Comp of Primary Data Rx0
`ifdef RW_NX_DERIV_PATH1
  input  wire        [12:0]                      RxDataIn80PRe1,  // Real Comp of Primary Data Rx1
  input  wire        [12:0]                      RxDataIn80PIm1,  // Imag Comp of Primary Data Rx1
`endif
  input  wire                                    RxDataValidIn80, // Qualifies Rx Data Input

  ///////////////////////////////////////////////
  // AGC
  ///////////////////////////////////////////////
  input  wire                                    AGCOFDMHTSTF,     // Indicates that AGC is updating gain during HTSTF
  //
  output wire                                    AGCStartP,        // Trigger for AGC during HT-STF
  output wire                                    AGCGainUpdateP,   // Trigger for AGC for gain computation during HT-STF
  output wire                                    AGCTBECountValid, // Valid indicator for TBECount
  output wire                                    AGCPlatFallP,     // Valid indicator for plateau fall
  output wire                                    AGCLegFrame,      // Legacy frame indication 
  output wire                                    AGCVHTFrame,      // VHT frame indication 
  output wire                                    AGCNoACIMargin,   // Frame with no ACI margin
  output wire                                    AGCBWSup20,       // Bandwidth more than 20MHz indication 
  output wire                                    AGCBWEq20,        // Bandwidth 20MHz indication 
  output wire                                    AGCBWEq40,        // Bandwidth 40MHz indication 
  output wire                                    AGCBWEq80,        // Bandwidth 80MHz indication 
  output wire        [7:0]                       AGCFOAngle,       // FO Angle

  ///////////////////////////////////////////////
  // TD Shared Memory
  ///////////////////////////////////////////////
  input  wire        [12:0]                      TDMem0RdDataRe0,    // Real Read Data 0 Rx Path 0
  input  wire        [12:0]                      TDMem0RdDataIm0,    // Imag Read Data 0 Rx Path 0
  input  wire        [12:0]                      TDMem1RdDataRe0,    // Real Read Data 1 Rx Path 0
  input  wire        [12:0]                      TDMem1RdDataIm0,    // Imag Read Data 1 Rx Path 0
`ifdef RW_NX_DERIV_PATH1
  input  wire        [12:0]                      TDMem0RdDataRe1,    // Real Read Data 0 Rx Path 1
  input  wire        [12:0]                      TDMem0RdDataIm1,    // Imag Read Data 0 Rx Path 1
  input  wire        [12:0]                      TDMem1RdDataRe1,    // Real Read Data 1 Rx Path 1
  input  wire        [12:0]                      TDMem1RdDataIm1,    // Imag Read Data 1 Rx Path 1
`endif
  input  wire        [12:0]                      TDMem0RdData20SRe,  // Real Read Data 0 Secondary Rx Path
  input  wire        [12:0]                      TDMem0RdData20SIm,  // Imag Read Data 0 Secondary Rx Path
  input  wire        [12:0]                      TDMem1RdData20SRe,  // Real Read Data 1 Secondary Rx Path
  input  wire        [12:0]                      TDMem1RdData20SIm,  // Imag Read Data 1 Secondary Rx Path
  //
  output wire                                    TDMemRdEn,          // Read Enable
  output wire        [7:0]                       TDMemRdAddr,        // Read Address
  output wire                                    TDMemWrEn,          // Write Enable
  output wire        [12:0]                      TDMemWrDataRe0,     // Real Write Data Rx Path 0
  output wire        [12:0]                      TDMemWrDataIm0,     // Imag Write Data Rx Path 0
`ifdef RW_NX_DERIV_PATH1
  output wire        [12:0]                      TDMemWrDataRe1,     // Real Write Data Rx Path 1
  output wire        [12:0]                      TDMemWrDataIm1,     // Imag Write Data Rx Path 1
`endif
  output wire        [12:0]                      TDMemWrData20SRe,   // Real Write Data Secondary Rx Path
  output wire        [12:0]                      TDMemWrData20SIm,   // Imag Write Data Secondary Rx Path
  output wire        [9:0]                       TDMemWrAddr,        // Write Address

  ///////////////////////////////////////////////
  // Sample Memory
  ///////////////////////////////////////////////
  input  wire        [12:0]                      SMemRdDataRe0,     // Read Data Real Rx0
  input  wire        [12:0]                      SMemRdDataIm0,     // Read Data Imag Rx0
  input  wire        [12:0]                      SMemRdData20SRe,   // Read Data Real Secondary
  input  wire        [12:0]                      SMemRdData20SIm,   // Read Data Imag Secondary
  //
  output wire                                    SMemRdEn,          // Read Enable
  output wire        [5:0]                       SMemRdAddr,        // Read Address
  output wire                                    SMemWrEn,          // Write Enable
  output wire        [5:0]                       SMemWrAddr,        // Write Address
  output wire        [12:0]                      SMemWrDataRe0,     // Write Data Real Rx0
  output wire        [12:0]                      SMemWrDataIm0,     // Write Data Imag Rx0
  output wire        [12:0]                      SMemWrData20SRe,   // Write Data Real Secondary Rx
  output wire        [12:0]                      SMemWrData20SIm,   // Write Data Imag Secondary Rx
`ifdef RW_NX_DERIV_PATH1
  input  wire        [12:0]                      SMemRdDataRe1,     // Read Data Real Rx1
  input  wire        [12:0]                      SMemRdDataIm1,     // Read Data Imag Rx1
  //
  output wire        [12:0]                      SMemWrDataRe1,     // Write Data Real Rx1
  output wire        [12:0]                      SMemWrDataIm1,     // Write Data Imag Rx1
`endif

  ///////////////////////////////////////////////
  // FFT control
  ///////////////////////////////////////////////
  input  wire                                    FFTDone,           // FFT is done
  input  wire        [1:0]                       FDFFTBufRdPtr,     // FFT Read buffer pointer
  //
  output wire        [1:0]                       TDFFTBufWrPtr,     // FFT Write buffer pointer
  output wire        [1:0]                       FFTBufPtr,         // FFT buffer select
  output wire                                    FFTStart,          // FFT start
  output wire        [2:0]                       FFTLen,            // FFT Length
  output wire        [2:0]                       FFTScale,          // FFT scale
            
  ///////////////////////////////////////////////
  // Main Tx/Rx FSM
  ///////////////////////////////////////////////
  input  wire                                    RxTDEnable,        // Rx TD Enable
  //
  output wire                                    RxTDDone,          // Rx TD end
  output wire                                    PlatTimeout,       // Plateau fall time-out
  output wire                                    FFTBufOverflow,    // FFT buffer overflow

  ///////////////////////////////////////////////
  // Rx frequency domain
  ///////////////////////////////////////////////
  input  wire        [3:0]                       FDFormatMod,       // FD frame format
  input  wire        [5:0]                       FDTOffsetComp,     // Sample skip signal from FD offset block
  input  wire                                    FDDone,            // FD is done
  input  wire        [21:0]                      FDCpeSlope,        // CPE Slope

  ///////////////////////////////////////////////
  // From Rx bit domain : frame parameters
  ///////////////////////////////////////////////
  input  wire        [2:0]                       FrameParamNSTS,
  input  wire        [1:0]                       FrameParamNESS,
  input  wire        [2:0]                       FrameParamNHeltf,
`ifdef RW_MUMIMO_RX_EN
  input  wire                                    FrameParamMuMIMO,
  input  wire        [2:0]                       FrameParam1MuNSTS,
`endif // RW_MUMIMO_RX_EN
  input  wire        [1:0]                       FrameParamGIType,    // 0-0.4us / 1-0.8us / 2-1.6us / 3-3.2us
  input  wire        [1:0]                       FrameParamHeltfType, // 0-3.2us (1x) / 1-6.4us (2x) / 2-12.8us (4x)
  input  wire                                    FrameParamDoppler,
  input  wire                                    FrameParamMidamble,
  input  wire        [1:0]                       FrameParamCBW,
  input  wire                                    FrameParamVHTNDP,
  input  wire        [1:0]                       FrameParamNSD,
  input  wire                                    FrameParam1Update,
  //
  input  wire        [2:0]                       FrameParamTPE,       // Indicates HE Packet Extension duration
  input  wire        [8:0]                       FrameParamNMA,
  input  wire        [15:0]                      FrameParamNSym,
  input  wire                                    FrameParam8Update,
  //
  input  wire        [7:0]                       FrameParamNHeSigB,
  input  wire                                    FrameParam7Update,

  ///////////////////////////////////////////////
  // CRM for clock gating
  ///////////////////////////////////////////////
  output wire                                    TBEGClkEn,
  output wire                                    TDFOEstGClkEn,

  ///////////////////////////////////////////////
  // Debug Ports
  ///////////////////////////////////////////////
  output wire        [3:0]                       RxTDFSMCurrState
  );                                                                                                                        

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
 reg                             SMemWrDataValid_1t;
 reg                             SMemRdDataValid_1t;
 reg                             SMemRdDataValid_2t;
 reg                             SMemRdDataValid_3t;
 reg                             SMemEnableRd_1t;
 reg                             SMemEnableRd_2t;
 reg         [12:0]              RxDataInRe0SMem_1t;
 reg         [12:0]              RxDataInIm0SMem_1t;
 reg         [12:0]              RxDataIn20SReSMem_1t;
 reg         [12:0]              RxDataIn20SImSMem_1t;
 reg         [12:0]              RxDataInRe0SMem_2t;
 reg         [12:0]              RxDataInIm0SMem_2t;
 reg         [12:0]              RxDataInRe0SMem_3t;
 reg         [12:0]              RxDataInIm0SMem_3t;
`ifdef RW_NX_DERIV_PATH1
 reg         [12:0]              RxDataInRe1SMem_1t;
 reg         [12:0]              RxDataInIm1SMem_1t;
 reg         [12:0]              RxDataInRe1SMem_2t;
 reg         [12:0]              RxDataInIm1SMem_2t;
 reg         [12:0]              RxDataInRe1SMem_3t;
 reg         [12:0]              RxDataInIm1SMem_3t;
`endif

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
 // Signals for FSM
 wire                            TBEEnable;     // enable TBE
 wire                            RxDataValidInFSM;
 wire        [1:0]               TDBandwidth;   // TD frame BW
 wire        [1:0]               TDBufWrPtr;    // FFT Write buffer pointer

 // Signals for TBE
 wire        [1:0]               RxRFChainEn;
 wire        [9:0]               TBE16DRxI0;
 wire        [9:0]               TBE16DRxQ0;
 wire        [9:0]               TBE64DRxI0;
 wire        [9:0]               TBE64DRxQ0;
 wire        [9:0]               RxLTFRe0;
 wire        [9:0]               RxLTFIm0;
`ifdef RW_NX_DERIV_PATH1
 wire        [9:0]               TBE16DRxI1;
 wire        [9:0]               TBE16DRxQ1;
 wire        [9:0]               TBE64DRxI1;
 wire        [9:0]               TBE64DRxQ1;
 wire        [9:0]               RxLTFRe1;
 wire        [9:0]               RxLTFIm1;
`endif
 wire                            RxLTFValid;
 wire                            RxDataValidIn20Lock;
 wire                            RxDataValidIn40Lock;
 wire                            RxDataValidIn80Lock;

 // Signals for DC offset
 wire                            TDDCDataSynch;
 wire                            TDDCDataPhase;
 wire                            TDDCSynch;
 wire                            TDDCClear;
 wire        [1:0]               TDDCSizeGI;
 wire        [1:0]               TDDCSizeSymb;
 wire        [1:0]               TDDCBandwidth;
 wire        [12:0]              RxDataInRe0DCMux;
 wire        [12:0]              RxDataInIm0DCMux;
 wire                            RxDataValidDC;
 wire        [12:0]              RxDataDCRe0; //Real Comp of Data 0
 wire        [12:0]              RxDataDCIm0; //Imag Comp of Data 0
`ifdef RW_NX_DERIV_PATH1
 wire        [12:0]              RxDataInRe1DCMux;
 wire        [12:0]              RxDataInIm1DCMux;
 wire        [12:0]              RxDataDCRe1; //Real Comp of Data 1
 wire        [12:0]              RxDataDCIm1; //Imag Comp of Data 1
`endif

 // Signals for TDFO
 wire                            TDFOStrtP;         // TDFO Estimation Start
 wire                            DataCompStartP;    // Start Data Compensation
 wire        [12:0]              TDFODataOutRe0;    // Write Data Rx Path 0
 wire        [12:0]              TDFODataOutIm0;    // Write Data Rx Path 0
 wire        [12:0]              TDFODataOut20SRe;  // Write Data Secondary Rx Path
 wire        [12:0]              TDFODataOut20SIm;  // Write Data Secondary Rx Path
 wire                            CmpLTFDoneP;       // LTF complete signal from TDFO
 wire                            FineFODone;        // Fine FO complete signal from TDFO
 wire                            TDFORxDataValid;
 wire                            RxDataEnableIn;
 wire        [12:0]              TDFODataInRe0Mux;
 wire        [12:0]              TDFODataInIm0Mux;
 wire        [12:0]              TDFODataIn20SReMux;
 wire        [12:0]              TDFODataIn20SImMux;
`ifdef RW_NX_DERIV_PATH1
 wire        [12:0]              TDFODataOutRe1;   // Write Data Rx Path 1
 wire        [12:0]              TDFODataOutIm1;   // Write Data Rx Path 1
 wire        [12:0]              TDFODataInRe1Mux;
 wire        [12:0]              TDFODataInIm1Mux;
`endif
 wire                            FFTMemSel;
 wire                            TDFOMemWrSel;
 wire        [7:0]               TDFOMemWrAddr;       // Write Address
 wire                            TDFOMemWrEn;         // Write Enable
 wire        [12:0]              TDFOMemWrDataRe0;    // Real Write Data Rx Path 0
 wire        [12:0]              TDFOMemWrDataIm0;    // Imag Write Data Rx Path 0
 wire        [12:0]              TDFOMemWrData20SRe;  // Real Write Data Secondary Rx Path
 wire        [12:0]              TDFOMemWrData20SIm;  // Imag Write Data Secondary Rx Path
`ifdef RW_NX_DERIV_PATH1
 wire        [12:0]              TDFOMemWrDataRe1;    // Real Write Data Rx Path 1
 wire        [12:0]              TDFOMemWrDataIm1;    // Imag Write Data Rx Path 1
`endif

 // Signal for Sample memory
 wire                            RxDataValidInSMem;
 wire        [12:0]              RxDataInRe0SMem;
 wire        [12:0]              RxDataInIm0SMem;
 wire        [12:0]              RxDataIn20SReSMem;
 wire        [12:0]              RxDataIn20SImSMem;
 wire                            SMemMuxSel;
 wire                            SMemMuxPreLast;
 wire                            SMemMuxPreLast80;
 wire                            SMemMuxLast;
 wire                            SMemMuxLast80_m2;
 wire                            SMemMuxLast80_m1;
 wire                            SMemMuxLast80;
 wire                            SMemEnableRd;
 wire                            SMemRdDataValid;
 wire                            SMemEnableWr;
 wire                            SMemWrDataValid;
 wire                            SMemWrDataValidMux;
`ifdef RW_NX_DERIV_PATH1
 wire        [12:0]              RxDataInRe1SMem;
 wire        [12:0]              RxDataInIm1SMem;
`endif

 // Signal for TD shared memory
 wire        [9:0]               FFTMemWrAddr;    // Write Address
 wire                            FFTMemStop;      // Stop write FFT
 wire                            FFTMemWrEn;      // Write Enable
 wire                            FFTMemWrEnMask;  // Write Enable masked

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// Time Domain FSM
// ----------------------------------


assign RxDataValidInFSM = (!CfgRegForce20 && TDBandwidth == 2'd0) ? RxDataValidIn20Lock :
                          (!CfgRegForce20 && TDBandwidth == 2'd1) ? RxDataValidIn40Lock : RxDataEnableIn;

// Instantiation of RxTimeDomainStMc
RxTimeDomainStMc U_RxTimeDomainStMc (
  .PhyClk                  (PhyClk),
  .TDCompGClk              (TDCompGClk),
  .nPhyRst                 (nPhyRst),
  // AGC
  .AGCOFDMHTSTF            (AGCOFDMHTSTF),
  .AGCStartP               (AGCStartP),
  .AGCGainUpdateP          (AGCGainUpdateP),
  .AGCLegFrame             (AGCLegFrame),
  .AGCVHTFrame             (AGCVHTFrame),
  .AGCNoACIMargin          (AGCNoACIMargin),
  .AGCBWSup20              (AGCBWSup20),
  .AGCBWEq20               (AGCBWEq20),
  .AGCBWEq40               (AGCBWEq40),
  .AGCBWEq80               (AGCBWEq80),
  // Registers
  .CfgRegPlatFallTimeOut   (CfgRegPlatFallTimeOut),
  .CfgRegForce20           (CfgRegForce20),
  .CfgRegForce40           (CfgRegForce40),
  .CfgRegTdCycRotVal20     (CfgRegTdCycRotVal20),
  .CfgRegTdCycRotVal40     (CfgRegTdCycRotVal40),
  .CfgRegTdCycRotVal80     (CfgRegTdCycRotVal80),
  .CfgRegTdAdjust20ShortGI (CfgRegTdAdjust20ShortGI),
  .CfgRegTdAdjust40ShortGI (CfgRegTdAdjust40ShortGI),
  .CfgRegTdAdjust80ShortGI (CfgRegTdAdjust80ShortGI),
  .CfgRegTdSyncOffset2080  (CfgRegTdSyncOffset2080),
  .CfgRegTdSyncOffset2040  (CfgRegTdSyncOffset2040),
  .CfgRegTdSyncOffset20    (CfgRegTdSyncOffset20),
  .CfgRegTdDcHtStfMargin   (CfgRegTdDcHtStfMargin),
  .CfgRegSkipMidamble1x    (CfgRegSkipMidamble1x),
  .CfgRegSkipMidamble2x    (CfgRegSkipMidamble2x),
  .CfgRegSkipMidamble4x    (CfgRegSkipMidamble4x),
  // Rx FD
  .FDFormatMod             (FDFormatMod),
  .FDTOffsetComp           (FDTOffsetComp),
  .FDDone                  (FDDone),
  // Rx BD
  .FrameParamNSTS          (FrameParamNSTS),
  .FrameParamNESS          (FrameParamNESS),
  .FrameParamNHeltf        (FrameParamNHeltf),
`ifdef RW_MUMIMO_RX_EN
  .FrameParamMuMIMO        (FrameParamMuMIMO),
  .FrameParam1MuNSTS       (FrameParam1MuNSTS),
`endif // RW_MUMIMO_RX_EN
  .FrameParamGIType        (FrameParamGIType),
  .FrameParamHeltfType     (FrameParamHeltfType),
  .FrameParamDoppler       (FrameParamDoppler),
  .FrameParamMidamble      (FrameParamMidamble),
  .FrameParamCBW           (FrameParamCBW),
  .FrameParamVHTNDP        (FrameParamVHTNDP),
  .FrameParamNSD           (FrameParamNSD),
  .FrameParam1Update       (FrameParam1Update),
  .FrameParamTPE           (FrameParamTPE),   
  .FrameParamNMA           (FrameParamNMA),   
  .FrameParamNSym          (FrameParamNSym),
  .FrameParam8Update       (FrameParam8Update),
  .FrameParamNHeSigB       (FrameParamNHeSigB),
  .FrameParam7Update       (FrameParam7Update),
  // TBE
  .TBECount                (StatusTBECount),
  .TBEEnable               (TBEEnable),
  .TDBandwidth             (TDBandwidth),
  // FFT control
  .FFTDone                 (FFTDone),
  .FDFFTBufRdPtr           (FDFFTBufRdPtr),
  .TDFFTBufWrPtr           (TDBufWrPtr),
  .FFTBufPtr               (FFTBufPtr),
  .FFTStart                (FFTStart),
  .FFTLen                  (FFTLen),
  .FFTScale                (FFTScale),
  // FFT memory
  .FFTMemWrAddr            (FFTMemWrAddr),
  .FFTMemStop              (FFTMemStop),
  // Main Tx/Rx
  .RxTDEnable              (RxTDEnable),
  .RxTDDone                (RxTDDone),
  .PlatTimeout             (PlatTimeout),
  .FFTBufOverflow          (FFTBufOverflow),
  // TD FO
  .PlatFall                (AGCPlatFallP),
  .CmpLTFDoneP             (CmpLTFDoneP),
  .FineFODone              (FineFODone),
  .FFTMemWrEnIn            (FFTMemWrEn),
  .TDFOStrtP               (TDFOStrtP),
  .DataCompStartP          (DataCompStartP),
  // FE
  .RxDataValidIn           (RxDataValidInFSM),
  // Sample memory
  .RxDataValidInSMem       (RxDataValidInSMem),
  .SMemMuxSel              (SMemMuxSel),
  .SMemMuxPreLast          (SMemMuxPreLast),
  .SMemMuxPreLast80        (SMemMuxPreLast80),
  .SMemMuxLast80_m2        (SMemMuxLast80_m2),
  .SMemMuxLast80_m1        (SMemMuxLast80_m1),
  .SMemMuxLast80           (SMemMuxLast80),
  .SMemMuxLast             (SMemMuxLast),
  .SMemEnableWr            (SMemEnableWr),
  .SMemWrDataValid         (SMemWrDataValid),
  .SMemEnableRd            (SMemEnableRd),
  .SMemRdDataValid         (SMemRdDataValid),
  .SMemWrAddr              (SMemWrAddr),
  .SMemRdAddr              (SMemRdAddr),
  .SMemRdEn                (SMemRdEn),
  // TD DC
  .TDDCDataSynch           (TDDCDataSynch),
  .TDDCDataPhase           (TDDCDataPhase),
  .TDDCSizeGI              (TDDCSizeGI),
  .TDDCSizeSymb            (TDDCSizeSymb),
  .TDDCSynch               (TDDCSynch),
  .TDDCClear               (TDDCClear),
  // CRM
  .TBEGClkEn               (TBEGClkEn),
  .TDFOEstGClkEn           (TDFOEstGClkEn),
  // Diag
  .CurrState               (RxTDFSMCurrState)
  );


// Time Domain DC offset
// ----------------------------------

assign TDDCBandwidth = (CfgRegForce20) ? 2'd0 : (CfgRegForce40) ? 2'd1 : 2'd2;

// Limit data valid toggling with RxTDEnable
assign RxDataValidIn20Lock = RxDataValidIn20 & RxTDEnable;
assign RxDataValidIn40Lock = RxDataValidIn40 & RxTDEnable;
assign RxDataValidIn80Lock = RxDataValidIn80 & RxTDEnable;

// Muxing versus 20MHz or 40MHz or 80MHz
assign RxDataEnableIn      = (!CfgRegForce40 && !CfgRegForce20) ? RxDataValidIn80Lock : 
                             (!CfgRegForce20)                   ? RxDataValidIn40Lock : RxDataValidIn20Lock;
assign RxDataInRe0DCMux    = (!CfgRegForce40 && !CfgRegForce20) ? RxDataIn80PRe0      :
                             (!CfgRegForce20)                   ? RxDataIn40PRe0      : RxDataIn20PRe0;
assign RxDataInIm0DCMux    = (!CfgRegForce40 && !CfgRegForce20) ? RxDataIn80PIm0      :
                             (!CfgRegForce20)                   ? RxDataIn40PIm0      : RxDataIn20PIm0;

// Instantiation of TDDCOffset
TDDCOffset U_TDDCOffset (
    // Clock and Reset
    .PhyClk         (TDCompGClk),
    .nPhyRst        (nPhyRst),
    // Register
    .StartDC        (CfgRegStartDC),
    .StartHTDC      (CfgRegStartHTDC),
    .WaitSync       (CfgRegDelaySync),
    .WaitHTSTF      (CfgRegWaitHTSTF),
    .DCTrackingEn   (CfgRegTDDCEn),
    .Bandwidth      (TDDCBandwidth),
    // Control
    .DCOffEn        (RxTDEnable),
    .Clear          (TDDCClear),
    .PrDataPhase    (TDDCDataPhase),
    .Synch          (TDDCSynch),
    .SizeGI         (TDDCSizeGI),
    .SizeSymb       (TDDCSizeSymb),
    .DataSynch      (TDDCDataSynch),
    // Data
    .RxDataEnableIn (RxDataEnableIn),
    .RxDataInRe     (RxDataInRe0DCMux),
    .RxDataInIm     (RxDataInIm0DCMux),
`ifdef RW_NX_DERIV_PATH1
    .RxDataInRe1    (RxDataInRe1DCMux),
    .RxDataInIm1    (RxDataInIm1DCMux),
`endif
    .RxDataOutRe    (RxDataDCRe0),
    .RxDataOutIm    (RxDataDCIm0),
`ifdef RW_NX_DERIV_PATH1
    .RxDataOutRe1   (RxDataDCRe1),
    .RxDataOutIm1   (RxDataDCIm1),
`endif
    .RxDataOutValid (RxDataValidDC)
    );

`ifdef RW_NX_DERIV_PATH1
    assign RxDataInRe1DCMux = (!CfgRegForce40 && !CfgRegForce20) ? RxDataIn80PRe1 :
                              (!CfgRegForce20)                   ? RxDataIn40PRe1 : RxDataIn20PRe1;
    assign RxDataInIm1DCMux = (!CfgRegForce40 && !CfgRegForce20) ? RxDataIn80PIm1 :
                              (!CfgRegForce20)                   ? RxDataIn40PIm1 : RxDataIn20PIm1;
`endif


// Time Domain sample memory management
// ------------------------------------

// Buffering the data before sending the data to TDFO block
// This is to make sure that the L-SIG portion is not lost
// when TDFO is transferring out L1+L2/2 to FFT memory.
assign RxDataValidInSMem = (!CfgRegForce20 && !CfgRegForce40 && TDBandwidth == 2'd1) ? RxDataValidIn40Lock :
                           (!CfgRegForce20 && TDBandwidth == 2'd0)                   ? RxDataValidIn20Lock : RxDataValidDC;
assign RxDataInRe0SMem   = (!CfgRegForce20 && !CfgRegForce40 && TDBandwidth == 2'd1) ? RxDataIn40PRe0      :
                           (!CfgRegForce20 && TDBandwidth == 2'd0)                   ? RxDataIn20PRe0      : RxDataDCRe0;
assign RxDataInIm0SMem   = (!CfgRegForce20 && !CfgRegForce40 && TDBandwidth == 2'd1) ? RxDataIn40PIm0      :
                           (!CfgRegForce20 && TDBandwidth == 2'd0)                   ? RxDataIn20PIm0      : RxDataDCIm0;
assign RxDataIn20SReSMem = RxDataIn20SRe;
assign RxDataIn20SImSMem = RxDataIn20SIm;

// Data to Sample Memory
assign SMemWrDataRe0   = RxDataInRe0SMem;
assign SMemWrDataIm0   = RxDataInIm0SMem;
`ifndef RW_NX_DERIV_CHBW20ONLY
assign SMemWrData20SRe = RxDataIn20SReSMem;
assign SMemWrData20SIm = RxDataIn20SImSMem;
`else
assign SMemWrData20SRe = 13'b0;
assign SMemWrData20SIm = 13'b0;
`endif

`ifdef RW_NX_DERIV_PATH1
    assign RxDataInRe1SMem = (!CfgRegForce20 && !CfgRegForce40 && TDBandwidth == 2'd1) ? RxDataIn40PRe1 :
                             (!CfgRegForce20 && TDBandwidth == 2'd0)                   ? RxDataIn20PRe1 : RxDataDCRe1;
    assign RxDataInIm1SMem = (!CfgRegForce20 && !CfgRegForce40 && TDBandwidth == 2'd1) ? RxDataIn40PIm1 :
                             (!CfgRegForce20 && TDBandwidth == 2'd0)                   ? RxDataIn20PIm1 : RxDataDCIm1;
    assign SMemWrDataRe1 = RxDataInRe1SMem;
    assign SMemWrDataIm1 = RxDataInIm1SMem;
`endif

// Primary band
assign TDFODataInRe0Mux = (SMemMuxSel && SMemEnableRd_2t) ? SMemRdDataRe0 :
                          (SMemMuxPreLast80)              ? RxDataInRe0SMem_3t :
                          (SMemMuxPreLast)                ? ((TDBandwidth == 2'd2) ? RxDataInRe0SMem_2t : RxDataInRe0SMem_1t) :
                          (SMemMuxLast80_m2)              ? RxDataInRe0SMem_2t :
                          (SMemMuxLast80_m1)              ? RxDataInRe0SMem_1t :
                          (SMemMuxLast80)                 ? RxDataInRe0SMem_1t :
                                                            RxDataInRe0SMem;
assign TDFODataInIm0Mux = (SMemMuxSel && SMemEnableRd_2t) ? SMemRdDataIm0 :
                          (SMemMuxPreLast80)              ? RxDataInIm0SMem_3t :
                          (SMemMuxPreLast)                ? ((TDBandwidth == 2'd2) ? RxDataInIm0SMem_2t : RxDataInIm0SMem_1t) :
                          (SMemMuxLast80_m2)              ? RxDataInIm0SMem_2t :
                          (SMemMuxLast80_m1)              ? RxDataInIm0SMem_1t :
                          (SMemMuxLast80)                 ? RxDataInIm0SMem_1t :
                                                            RxDataInIm0SMem;
`ifdef RW_NX_DERIV_PATH1
assign TDFODataInRe1Mux = (SMemMuxSel && SMemEnableRd_2t) ? SMemRdDataRe1 :
                          (SMemMuxPreLast80)              ? RxDataInRe0SMem_3t :
                          (SMemMuxPreLast)                ? ((TDBandwidth == 2'd2) ? RxDataInRe0SMem_2t : RxDataInRe0SMem_1t) :
                          (SMemMuxLast80)                 ? RxDataInRe0SMem_1t :
                                                            RxDataInRe1SMem;
assign TDFODataInIm1Mux = (SMemMuxSel && SMemEnableRd_2t) ? SMemRdDataIm1 :
                          (SMemMuxPreLast80)              ? RxDataInIm1SMem_3t :
                          (SMemMuxPreLast)                ? ((TDBandwidth == 2'd2) ? RxDataInIm1SMem_2t : RxDataInIm1SMem_1t) :
                          (SMemMuxLast80)                 ? RxDataInRe1SMem_1t :
                                                            RxDataInIm1SMem;
`endif
// Secondary band
`ifndef RW_NX_DERIV_CHBW20ONLY
assign TDFODataIn20SReMux = (SMemMuxSel && SMemEnableRd_2t) ? SMemRdData20SRe :
                            (SMemMuxPreLast)                ? RxDataIn20SReSMem_1t :
                                                              RxDataIn20SReSMem;
assign TDFODataIn20SImMux = (SMemMuxSel && SMemEnableRd_2t) ? SMemRdData20SIm :
                            (SMemMuxPreLast)                ? RxDataIn20SImSMem_1t :
                                                              RxDataIn20SImSMem;
`else
assign TDFODataIn20SReMux = 13'b0;
assign TDFODataIn20SImMux = 13'b0;
`endif

// Write enable port 0 to Sample Memory
assign SMemWrDataValidMux = (TDBandwidth == 2'd2) ? RxDataValidInSMem : SMemWrDataValid_1t;
assign SMemWrEn           = SMemMuxSel & SMemEnableWr & SMemWrDataValidMux;

// Delayed signals for Sample memory control
always @(posedge PhyClk or negedge nPhyRst) begin
  if(nPhyRst==1'b0) begin
    SMemWrDataValid_1t   <= 1'b0;
    //
    SMemEnableRd_1t      <= 1'b0;
    SMemEnableRd_2t      <= 1'b0;
    //
    SMemRdDataValid_1t   <= 1'b0;
    SMemRdDataValid_2t   <= 1'b0;
    SMemRdDataValid_3t   <= 1'b0;
    //
    RxDataInRe0SMem_1t   <= 13'b0;
    RxDataInIm0SMem_1t   <= 13'b0;
    RxDataInRe0SMem_2t   <= 13'b0;
    RxDataInIm0SMem_2t   <= 13'b0;
    RxDataInRe0SMem_3t   <= 13'b0;
    RxDataInIm0SMem_3t   <= 13'b0;
`ifdef RW_NX_DERIV_PATH1
    RxDataInRe1SMem_1t   <= 13'b0;
    RxDataInIm1SMem_1t   <= 13'b0;
    RxDataInRe1SMem_2t   <= 13'b0;
    RxDataInIm1SMem_2t   <= 13'b0;
    RxDataInRe1SMem_3t   <= 13'b0;
    RxDataInIm1SMem_3t   <= 13'b0;
`endif
    RxDataIn20SReSMem_1t <= 13'b0;
    RxDataIn20SImSMem_1t <= 13'b0;
  end
  else begin
    SMemWrDataValid_1t <= SMemWrDataValid;
    //
    SMemEnableRd_1t    <= SMemEnableRd;
    SMemEnableRd_2t    <= SMemEnableRd_1t;
    //
    SMemRdDataValid_1t <= SMemRdDataValid;
    SMemRdDataValid_2t <= SMemRdDataValid_1t;
    SMemRdDataValid_3t <= SMemRdDataValid_2t;
    //
     RxDataInRe0SMem_2t   <= RxDataInRe0SMem_1t;
     RxDataInIm0SMem_2t   <= RxDataInIm0SMem_1t;
     RxDataInRe0SMem_3t   <= RxDataInRe0SMem_2t;
     RxDataInIm0SMem_3t   <= RxDataInIm0SMem_2t;
`ifdef RW_NX_DERIV_PATH1
     RxDataInRe1SMem_2t   <= RxDataInRe1SMem_1t;
     RxDataInIm1SMem_2t   <= RxDataInIm1SMem_1t;
     RxDataInRe1SMem_3t   <= RxDataInRe1SMem_2t;
     RxDataInIm1SMem_3t   <= RxDataInIm1SMem_2t;
`endif
//     if (SMemRdDataValid_2t) begin
      RxDataInRe0SMem_1t   <= RxDataInRe0SMem;
      RxDataInIm0SMem_1t   <= RxDataInIm0SMem;
`ifdef RW_NX_DERIV_PATH1
      RxDataInRe1SMem_1t   <= RxDataInRe1SMem;
      RxDataInIm1SMem_1t   <= RxDataInIm1SMem;
`endif
      RxDataIn20SReSMem_1t <= RxDataIn20SReSMem;
      RxDataIn20SImSMem_1t <= RxDataIn20SImSMem;
//     end
  end
end

// Modifying the RxDataValid to TDFO so that there are no pulses during the data buffering stage into Sample memory
assign TDFORxDataValid = (SMemMuxSel && !SMemEnableRd_2t) ? 1'b0 : 
                         (SMemMuxSel && SMemEnableRd_2t)  ? SMemRdDataValid_3t :
                                                            RxDataValidInSMem | SMemMuxPreLast80 | SMemMuxPreLast |
                                                            SMemMuxLast80_m2  | SMemMuxLast80_m1 | SMemMuxLast80  | SMemMuxLast;
                                                            


// Time Domain Frequency Offset Estimation and compensation
// --------------------------------------------------------

// Instantiation of TDFOTop
TDFOTop  U_TDFOTop (
    // Clock and Reset Manager interface
    .PhyClk                      (TDCompGClk),
    .TDFOEstGClk                 (TDFOEstGClk),
    .nPhyRst                     (nPhyRst),
    // Rx TD FSM interface
    .Bandwidth                   (TDBandwidth),
    .TDFOStrtP                   (TDFOStrtP),
    .DataCmpStrtP                (DataCompStartP),
    .TDFOEn                      (RxTDEnable),
    .PlatFallP                   (AGCPlatFallP),
    .CmpLTFDoneP                 (CmpLTFDoneP),
    .FineFODone                  (FineFODone),
    // Rx FD interface
    .FDCpeSlope                  (FDCpeSlope),
    // AGC interface
    .FOAngle                     (AGCFOAngle),
    // Registers interface
    .Force20                     (CfgRegForce20),
    .Force40                     (CfgRegForce40),
    .AutoCorrCompareRatioLow     (CfgRegAutoCorrCompareRatioLow),
    .AutoCorrCompareRatioHigh    (CfgRegAutoCorrCompareRatioHigh),
    .AutoCorrFallCount           (CfgRegAutoCorrFallCount),
    .AutoCorrPlatSumStart        (CfgRegAutoCorrPlatSumStart),
    .AutoCorrPlatSumEnd          (CfgRegAutoCorrPlatSumEnd),
    .AutoCorrPlatFallSearchStart (CfgRegAutoCorrPlatFallSearchStart),
    .AutoCorrPlatFallIndexOffset (CfgRegAutoCorrPlatFallIndexOffset),
    .AutoCorrPhSumStart          (CfgRegAutoCorrPhSumStart),
    .AutoCorrPhSumEnd            (CfgRegAutoCorrPhSumEnd),
    .TDFOCompEn                  (CfgRegTDFOCompEn),
    .TDFOFineEstEn               (CfgRegTDFOFineEstEn),
    .TDFOCpeSlopeEn              (CfgRegTDFOCpeSlopeEn),
    //
    .CoarseFOAngle               (StatusCoarseFOAngle),
    .FineFOAngle                 (StatusFineFOAngle),
    // FD interface
`ifdef RW_NX_TDFO_NSR_EN
    .RatioNoiseSignal            (),       // Unused
    .RatioShift                  (),       // Unused
`endif
    // Timing Boundary Estimation interface
    .TBECount                    (StatusTBECount),
    .TBECountValid               (AGCTBECountValid),
    .TBE16DRxI0                  (TBE16DRxI0),
    .TBE16DRxQ0                  (TBE16DRxQ0),
`ifdef RW_NX_DERIV_PATH1
    .TBE16DRxI1                  (TBE16DRxI1),
    .TBE16DRxQ1                  (TBE16DRxQ1),
`endif
    .TBE64DRxI0                  (TBE64DRxI0),
    .TBE64DRxQ0                  (TBE64DRxQ0),
`ifdef RW_NX_DERIV_PATH1
    .TBE64DRxI1                  (TBE64DRxI1),
    .TBE64DRxQ1                  (TBE64DRxQ1),
`endif
    .RxLTFRe0                    (RxLTFRe0), 
    .RxLTFIm0                    (RxLTFIm0), 
`ifdef RW_NX_DERIV_PATH1
    .RxLTFRe1                    (RxLTFRe1), 
    .RxLTFIm1                    (RxLTFIm1), 
`endif
    .RxLTFValid                  (RxLTFValid),
    // Data path
    .RxDataInRe0                 (TDFODataInRe0Mux),
    .RxDataInIm0                 (TDFODataInIm0Mux),
`ifdef RW_NX_DERIV_PATH1
    .RxDataInRe1                 (TDFODataInRe1Mux),
    .RxDataInIm1                 (TDFODataInIm1Mux),
`endif
    .RxDataValid                 (TDFORxDataValid),
    .RxDataIn20SRe               (TDFODataIn20SReMux),
    .RxDataIn20SIm               (TDFODataIn20SImMux),
    // TDFO Memory interface
    .TDFOMem0RdDataRe0           (TDMem0RdDataRe0),
    .TDFOMem0RdDataIm0           (TDMem0RdDataIm0),
    .TDFOMem1RdDataRe0           (TDMem1RdDataRe0),
    .TDFOMem1RdDataIm0           (TDMem1RdDataIm0),
`ifdef RW_NX_DERIV_PATH1
    .TDFOMem0RdDataRe1           (TDMem0RdDataRe1),
    .TDFOMem0RdDataIm1           (TDMem0RdDataIm1),
    .TDFOMem1RdDataRe1           (TDMem1RdDataRe1),
    .TDFOMem1RdDataIm1           (TDMem1RdDataIm1),
`endif
    .TDFOMem0RdData20SRe         (TDMem0RdData20SRe),
    .TDFOMem0RdData20SIm         (TDMem0RdData20SIm),
    .TDFOMem1RdData20SRe         (TDMem1RdData20SRe),
    .TDFOMem1RdData20SIm         (TDMem1RdData20SIm),
    .TDFOMemRdEn                 (TDMemRdEn),
    .TDFOMemRdAddr               (TDMemRdAddr),
    .TDFOMemWrEn                 (TDFOMemWrEn),
    .TDFOMemWrDataRe0            (TDFOMemWrDataRe0),
    .TDFOMemWrDataIm0            (TDFOMemWrDataIm0),
`ifdef RW_NX_DERIV_PATH1
    .TDFOMemWrDataRe1            (TDFOMemWrDataRe1),
    .TDFOMemWrDataIm1            (TDFOMemWrDataIm1),
`endif
    .TDFOMemWrData20SRe          (TDFOMemWrData20SRe),
    .TDFOMemWrData20SIm          (TDFOMemWrData20SIm),
    .TDFOMemWrAddr               (TDFOMemWrAddr),
    .TDFOMemWrSel                (TDFOMemWrSel),
    // FFT Memory interface
    .FFTMemWrDataRe0             (TDFODataOutRe0),
    .FFTMemWrDataIm0             (TDFODataOutIm0),
`ifdef RW_NX_DERIV_PATH1
    .FFTMemWrDataRe1             (TDFODataOutRe1),
    .FFTMemWrDataIm1             (TDFODataOutIm1),
`endif
    .FFTMemWrData20SRe           (TDFODataOut20SRe),
    .FFTMemWrData20SIm           (TDFODataOut20SIm),
    .FFTMemWrEn                  (FFTMemWrEn),
    .FFTMemSel                   (FFTMemSel)
    );

// Mask FFTMemWrEn for last symbol
assign FFTMemWrEnMask = FFTMemWrEn & ~FFTMemStop;

// Time Domain memory port sharing
// --------------------------------------------------------
assign TDMemWrAddr       = (FFTMemSel) ? FFTMemWrAddr     : {2'b0,TDFOMemWrAddr};
assign TDMemWrEn         = (FFTMemSel) ? FFTMemWrEnMask   : TDFOMemWrEn;
assign TDMemWrDataRe0    = (FFTMemSel) ? TDFODataOutRe0   : TDFOMemWrDataRe0;
assign TDMemWrDataIm0    = (FFTMemSel) ? TDFODataOutIm0   : TDFOMemWrDataIm0;
assign TDMemWrData20SRe  = (FFTMemSel) ? TDFODataOut20SRe : TDFOMemWrData20SRe;
assign TDMemWrData20SIm  = (FFTMemSel) ? TDFODataOut20SIm : TDFOMemWrData20SIm;
assign TDFFTBufWrPtr     = (FFTMemSel) ? TDBufWrPtr       : {1'b0,TDFOMemWrSel};

`ifdef RW_NX_DERIV_PATH1
assign TDMemWrDataRe1 = (FFTMemSel) ? TDFODataOutRe1 : TDFOMemWrDataRe1;
assign TDMemWrDataIm1 = (FFTMemSel) ? TDFODataOutIm1 : TDFOMemWrDataIm1;
`endif


// Time Domain Timing Boundary Estimation
// --------------------------------------------------------
assign RxRFChainEn[0] = 1'b1;
`ifdef RW_NX_DERIV_PATH1
  assign RxRFChainEn[1] = 1'b1;
`else
  assign RxRFChainEn[1] = 1'b0;
`endif

// Instantiation of TBETop
TBETop U_TBETop (
    //Control
    .nPhyRst                (nPhyRst),
    .PhyClk                 (TBEGClk),   // Gated Clock
    .TBEEnable              (TBEEnable),
    //Data
    .RxDataValidP           (RxLTFValid),
    .RxI0                   (RxLTFRe0),
    .RxQ0                   (RxLTFIm0),
`ifdef RW_NX_DERIV_PATH1
    .RxI1                   (RxLTFRe1),
    .RxQ1                   (RxLTFIm1),
`endif
    //Config registers
    .PeakSearchDelta        (CfgRegPeakSearchDelta),
    .PeakSearchNdlEn        (CfgRegPeakSearchNdlEn),
    .PeakSearchNdlIndex     (CfgRegPeakSearchNdlIndex),
    .RxRFChainEn            (RxRFChainEn),
    .TBECountAdjust20       (CfgRegTBECountAdjust20),
    .PeakSearchStartLowSNR  (CfgRegPeakSearchStart),
    .PeakSearchStopLowSNR   (CfgRegPeakSearchStop),
    .TBEBias                (CfgRegTBEBias),
    //
    .TBE16DRxI0             (TBE16DRxI0),
    .TBE16DRxQ0             (TBE16DRxQ0),
    .TBE64DRxI0             (TBE64DRxI0),
    .TBE64DRxQ0             (TBE64DRxQ0),
`ifdef RW_NX_DERIV_PATH1
    .TBE16DRxI1             (TBE16DRxI1),
    .TBE16DRxQ1             (TBE16DRxQ1),
    .TBE64DRxI1             (TBE64DRxI1),
    .TBE64DRxQ1             (TBE64DRxQ1),
`endif
    .TBECount               (StatusTBECount),
    .TBECountValid          (AGCTBECountValid),
    .MaxCorrValSum          (StatusMaxCorrValSum)
    );

endmodule

`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
