////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39999 $
// $Date: 2019-10-04 10:50:57 +0200 (Fri, 04 Oct 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : This module is the RW WLAN nX Receive Time Domain FSM
//                    It controls all the receive operations taking 
//                    place before the FFT, that is
//                    1) DC and Frequency offset estimation control
//                    2) Synchronization (Timing Boundary estimation) and 
//                       LTFs time domain summing
//                    3) Frequency offset and DC offset compensations
//                    4) FFT Write address generation
//                    This FSM has a linear part (active during the preamble
//                    portion of the packet) for handling all the 
//                    estimations performed on the preambles and a loop
//                    over HT-LTFs, SIG and data symbols
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :  
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
//
// 
/////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

module RxTimeDomainStMc (

            ///////////////////////////////////////////////
            // Clock and Reset Manager interface
            ///////////////////////////////////////////////
            input  wire          PhyClk,            // PHY clock
            input  wire          TDCompGClk,        // TD compensation clock
            input  wire          nPhyRst,           // Active low asynchronous reset
            //
            output reg           TBEGClkEn,         // Clock Gating enable for TBE block
            output reg           TDFOEstGClkEn,     // Clock Gating for TDFO Estim block
            
            ///////////////////////////////////////////////
            // Registers interface
            ///////////////////////////////////////////////
            input  wire          CfgRegForce40,           // CfgRegForce40 = 1 allows reception of 20/40MHz frames only
            input  wire          CfgRegForce20,           // CfgRegForce20 = 1 allows reception of 20MHz frames only
            //
            input  wire [11:0]   CfgRegPlatFallTimeOut,   // Max time out value for plateau fall
            //
            input  wire [7:0]    CfgRegTdCycRotVal20,
            input  wire [7:0]    CfgRegTdCycRotVal40,
            input  wire [7:0]    CfgRegTdCycRotVal80,
            //
            input  wire [4:0]    CfgRegTdAdjust20ShortGI, // Adjust short GI timing boundary for 20MHz frames
            input  wire [4:0]    CfgRegTdAdjust40ShortGI, // Adjust short GI timing boundary for 40MHz frames
            input  wire [4:0]    CfgRegTdAdjust80ShortGI, // Adjust short GI timing boundary for 80MHz frames
            //
            input  wire [7:0]    CfgRegTdSyncOffset2080,  // Adjust timing boundary for 80MHz frames from 20MHz TDBW
            input  wire [7:0]    CfgRegTdSyncOffset2040,  // Adjust timing boundary for 40MHz frames from 20MHz TDBW
            input  wire [7:0]    CfgRegTdSyncOffset20,    // Adjust timing boundary for 20MHz frames
            //
            input  wire [7:0]    CfgRegTdDcHtStfMargin,   // Margin for HT-STF indication to TDDC
            //
            input  wire          CfgRegSkipMidamble1x,    // Skip midamble HE-LTF 1x
            input  wire          CfgRegSkipMidamble2x,    // Skip midamble HE-LTF 2x
            input  wire          CfgRegSkipMidamble4x,    // Skip midamble HE-LTF 4x

            ///////////////////////////////////////////////
            // AGC interface
            ///////////////////////////////////////////////
            input  wire          AGCOFDMHTSTF,      // Indicates that AGC is updating gain during HTSTF
            //
            output reg           AGCStartP,         // Trigger for AGC during HT-STF in Mixed mode and VHT
            output reg           AGCGainUpdateP,    // Trigger for AGC for gain computation
            output reg           AGCLegFrame,       // Legacy frame indication 
            output reg           AGCVHTFrame,       // VHT frame indication 
            output reg           AGCNoACIMargin,    // Frame with no ACI margin
            output reg           AGCBWSup20,        // Bandwidth more than 20MHz indication 
            output reg           AGCBWEq20,         // Bandwidth 20MHz indication 
            output reg           AGCBWEq40,         // Bandwidth 40MHz indication 
            output reg           AGCBWEq80,         // Bandwidth 80MHz indication 

            ///////////////////////////////////////////////
            // RX Bit Domain interface
            ///////////////////////////////////////////////
            input  wire [2:0]    FrameParamNSTS,      // Number of HT-LTFs
            input  wire [1:0]    FrameParamNESS,      // Number of E-HT-LTFs
            input  wire [2:0]    FrameParamNHeltf,    // Number of HE-LTFs
            input  wire [1:0]    FrameParamGIType,    // 0-0.4us / 1-0.8us / 2-1.6us / 3-3.2us
            input  wire [1:0]    FrameParamHeltfType, // 0-3.2us (1x) / 1-6.4us (2x) / 2-12.8us (4x)
            input  wire          FrameParamDoppler,   // Frame with midamble
            input  wire          FrameParamMidamble,  // Midamble periodicity
            input  wire [1:0]    FrameParamCBW,       // Bandwidth in Frequency Domain
            input  wire          FrameParamVHTNDP,    // Frame is VHT NDP
            input  wire [1:0]    FrameParamNSD,       // Number of subcarriers data
`ifdef RW_MUMIMO_RX_EN
            input  wire          FrameParamMuMIMO,    // Frame is Mu-MIMO
            input  wire [2:0]    FrameParam1MuNSTS,   // Number of VHT-LTFs in Mu-MIMO
`endif // RW_MUMIMO_RX_EN
            input  wire          FrameParam1Update,   // Indicates that above parameters are valid
            
            input  wire [15:0]   FrameParamNSym,      // Number of data symbols
            input  wire [2:0]    FrameParamTPE,       // Indicates HE Packet Extension duration
            input  wire [8:0]    FrameParamNMA,       // Number of midamble periods
            input  wire          FrameParam8Update,   // Indicates that above parameter is valid
            
            input  wire [7:0]    FrameParamNHeSigB,   // Number of HE-SIG-B symbols
            input  wire          FrameParam7Update,   // Indicates that above parameter is valid

            ///////////////////////////////////////////////
            // RX Frequency Domain interface
            ///////////////////////////////////////////////
            input  wire [3:0]    FDFormatMod,       // 0,1: legacy, 2: HT-MM, 3: HT-GF, 4: VHT
                                                    // 5: HE-SU, 6: HE-MU, 7: HE-ER, 8: HE-TB
            input  wire [5:0]    FDTOffsetComp,     // Sample skip signal from FD offset block
            input  wire          FDDone,            // FD is done

            ///////////////////////////////////////////////
            // FFT control
            ///////////////////////////////////////////////
            input  wire          FFTDone,           // FFT is done
            input  wire [1:0]    FDFFTBufRdPtr,     // FFT Read buffer pointer
            //
            output wire [1:0]    TDFFTBufWrPtr,     // FFT Write buffer select
            output wire [1:0]    FFTBufPtr,         // FFT buffer select
            output reg           FFTStart,          // FFT start
            output reg  [ 2:0]   FFTLen,            // FFT length (0: FFT64, 1: FFT128, 2: FFT256, 3: FFT512, 6: 2x FFT64, 7: NO FFT)
            output reg  [ 2:0]   FFTScale,          // FFT scale (0: /8, 1: /8v2, 2: /16, 3: /16v2, 4: /32)
            output reg           FFTBufOverflow,    // FFT buffer overflow
            output wire          FFTMemStop,        // FFT write stop

            output reg  [9:0]    FFTMemWrAddr,      // FFT Write Address
            
            ///////////////////////////////////////////////
            // TD Frequency offset estimation interface
            ///////////////////////////////////////////////
            input  wire          PlatFall,          // STF Autocorrelation Plateau Fall
            input  wire          CmpLTFDoneP,       // LTF transfer to FFT complete
            input  wire          FineFODone,        // Fine FO estimation over
            input  wire          RxDataValidIn,     // Valid input from Front end
            input  wire          FFTMemWrEnIn,      // Valid input from TDFO
            //
            output wire          TDFOStrtP,         // Start FO Compensation
            output reg           DataCompStartP,    // Start FO Compensation of coarse+fine for data portion
            output wire [1:0]    TDBandwidth,       // Current bandwidth of the time domain part :
                                                    // 0: 20 MHz
                                                    // 1: 40 MHz
                                                    // 2: 80 MHz
                                                    // Set with signal field CBW when available

            ///////////////////////////////////////////////
            // Timing Boundary Estimation interface
            ///////////////////////////////////////////////
            input  wire [7:0]    TBECount,          // Indicates middle of LTF 1
            //
            output reg           TBEEnable,         // TBE enable for starting cross-correlation

            ///////////////////////////////////////////////
            // Main RX/TX FSM interface
            ///////////////////////////////////////////////
            input  wire          RxTDEnable,        // RX Time Domain enable
            //
            output reg           RxTDDone,          // RX TD done to main RX/TX FSM
                                                    // indicates end of Time domain
                                                    // processing for the packet
            output reg           PlatTimeout,       // Signals that a time out happened without
                                                    // Plateau fall being detected
            
            ///////////////////////////////////////////////
            // DC offset estimation interface
            ///////////////////////////////////////////////
            input  wire          TDDCDataSynch,     // Accumulation window for DC synchronization
            //
            output reg           TDDCDataPhase,     // Indication that data phase is being processed
            output reg  [1:0]    TDDCSizeGI,        // Indication GI size to TD DC Offset
            output reg  [1:0]    TDDCSizeSymb,      // Indication symbol size to TD DC Offset
            output reg           TDDCSynch,         // Synch pulse to TD DC Offset
            output wire          TDDCClear,         // Clear pulse to TD DC Offset

            ///////////////////////////////////////////////
            // Sample memory control
            ///////////////////////////////////////////////
            input  wire          RxDataValidInSMem,   // Valid input for Sample memory
            //
            output reg           SMemMuxSel,          // Enable data path from Sample memory
            output reg           SMemMuxPreLast80,    // Load pre-last Sample for 80MHz BW
            output reg           SMemMuxPreLast,      // Load pre-last Sample
            output reg           SMemMuxLast80_m2,    // Load last Sample minus 2 for 80MHz BW
            output reg           SMemMuxLast80_m1,    // Load last Sample minus 1 for 80MHz BW
            output reg           SMemMuxLast80,       // Load last Sample for 80MHz BW
            output reg           SMemMuxLast,         // Disable data path from Sample memory to real time
            output reg           SMemEnableWr,        // Enable writing to Sample memory
            output wire          SMemWrDataValid,     // Write data valid
            output reg           SMemEnableRd,        // Enable reading from Sample memory
            output wire          SMemRdDataValid,     // Read data valid
            output wire [5:0]    SMemWrAddr,          // Write Address
            output wire [5:0]    SMemRdAddr,          // Read Address
            output wire          SMemRdEn,            // Read Enable

            ///////////////////////////////////////////////
            // Diagnostic ports
            ///////////////////////////////////////////////
            output reg [3:0]     CurrState            // Current state of the TD FSM
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
// FSM related params
localparam IDLE             = 4'd0;
localparam STF              = 4'd1;
localparam LTF              = 4'd2;
localparam WAIT_GIEND1      = 4'd3;
localparam FFT_XFERLOOP     = 4'd4;
localparam SMEM_XFER        = 4'd5;
localparam FFT_STARTLOOP    = 4'd6;
localparam WAIT_GIEND2      = 4'd7;
localparam START_HTSTF      = 4'd8;
localparam AGCUNLOCK_WAIT   = 4'd9;
localparam WAIT_HTSTF_END   = 4'd10;
localparam WAIT_HTLTF_GIEND = 4'd11;
localparam RXEND            = 4'd12;
localparam AGCRELOCK_WAIT   = 4'd13;
localparam RESET_DC_UNLOCK  = 4'd14;
localparam WAIT_PE_END      = 4'd15;
                                
// Symbol type params
localparam LSIG    = 4'd0;
localparam HTSIG   = 4'd1;
localparam HTSTF   = 4'd2;
localparam HTLTF   = 4'd3;
localparam LDATA   = 4'd4;
localparam HTDATA  = 4'd5;
localparam VHTSIGA = 4'd6;
localparam VHTSTF  = 4'd7;
localparam VHTLTF  = 4'd8;
localparam VHTSIGB = 4'd9;
localparam VHTDATA = 4'd10;
localparam HESIGA  = 4'd11;
localparam HESTF   = 4'd12;
localparam HELTF   = 4'd13;
localparam HESIGB  = 4'd14;
localparam HEDATA  = 4'd15;

// Encoding of FDFormatMod
localparam MODE_LEG   = 4'd0;
localparam MODE_DUP   = 4'd1;
localparam MODE_MM    = 4'd2;
localparam MODE_GF    = 4'd3;
localparam MODE_VHT   = 4'd4;
localparam MODE_HE_SU = 4'd5;
localparam MODE_HE_MU = 4'd6;
localparam MODE_HE_ER = 4'd7;
localparam MODE_HE_TB = 4'd8;

// Encoding of HeltfType
localparam [1:0] HE_1xLTF = 2'd0;
localparam [1:0] HE_2xLTF = 2'd1;
localparam [1:0] HE_4xLTF = 2'd2;

// Timeout params
localparam AGCHTSTF_TIMEOUT  = 9'd276;  // AGC timeout of 2.3us @120MHz before AGCOFDMHTSTF set to 1
localparam INBANDPOW_TIMEOUT = 11'd120; // Inband power timeout of 1us @120MHz after AGCStartP generation

// Default BW
localparam TDBW_DEFAULT = 2'd0;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg [3:0]              NextState;         // Next state of the TD FSM
reg [3:0]              PrevState;         // Previous state of the TD FSM

reg [3:0]              SymbolType;        // The current symbol type being processed
                                          // by the Time Domain domain part
reg [10:0]             SampleCounter;     // Incoming samples counter, keeping track
                                          // of the symbol synchronization
                                          // domain part of the modem
reg [2:0]              HTValidCounter;    // Samples counter used during HT-STF, keeping track
                                          // of the data valid when AGC is reseting FE
reg [2:0]              HTValidCounterSel; // End value for HTValidCounter
                                          // domain data is sent (alternatively 0/1/2)
reg [8:0]              GILength;          // GI_length computed
reg [15:0]             TDSymbolCounter;   // Number of symbols processed by time domain
reg [1:0]              TDBWInter;         // Current bandwidth of the time domain inline with symbols
reg [1:0]              TDBWEarly;         // Early current bandwidth of the time domain
reg [13:0]             PlatFallTimer;
reg                    CoarseFOEn;        // Enable coarse FO comp for packet
reg                    CSFlag;
reg [5:0]              FDTOffsetComp_1t;
reg [5:0]              FDTOffsetComp_2t;
reg                    LargeGIMask;
reg                    HTSTFNoUnlock;     // Signals that AGC unlock has to be masked during HT-STF
reg                    NDPGFNoUnlock;     // Signals that AGC unlock has to be masked for GF frames
reg                    RxTDEnable_1t;
reg                    TBEEnable_1t;
reg                    TBEEnable_2t;
reg                    CmpLTFDoneP_1t;
reg                    CmpLTFDoneP_2t;
reg                    CmpLTFDoneFlag;
reg [8:0]              SampleDCCounter;
reg                    TDDCDataSynch_1t;
reg                    TDDCSynchDone;
reg                    TDDCDataSynchDone;
reg                    TDSymbolCounterIncr_1t;
reg                    TDSymbolCounterIncr_2t;
reg [6:0]              SMemWrAddrInter;
reg [6:0]              SMemRdAddrInter;
reg                    SMemRdAddrUpdateP;   // Update read address of Sample memory according to bandwidth
reg [7:0]              SMemOffset;          // Read address offset of Sample memory
reg [2:0]              SMemValidCounter;    // Samples counter used during HT-LTF, keeping track
                                            // of the data valid when AGC is reseting FE
reg [2:0]              SMemValidCounterEnd; // End value for SMemValidCounter
reg [2:0]              SMemValidCounterSel;
reg                    SMemRdAddrWait;      // Wait read address of Sample memory for flow control
reg                    SMemRdAddrWait_1t;
reg                    SMemRdAddrWaitFFT;   // Wait read address of Sample memory for flow control
reg                    SMemFlush;           // Flush Sample memory
reg                    SMemMuxPrePreLast80; // Load sample for smem transfer ending
reg                    SMemRdToggle;        // Toggle reading from Sample memory
reg  [3:0]             Nhtltf;
reg                    TDStopFFT;           // Stop FFT for NDP and 1 symbol packets
reg                    TickSymbolFFT;
reg                    NoMoreFFT;
reg                    TDCompOnGoing;
reg                    TDLastSymbol;        // Last symbol flag
reg  [1:0]             n_FFTBufPtr;
reg  [1:0]             TDFFTBufWrPtrInt;
reg  [1:0]             FFTBufPtrInt;
reg                    FFTState;
reg                    TDBW20;
reg                    TDBW40;
reg  [3:0]             FDFormatMod_1t;
reg                    FDDone_1t;
reg  [9:0]             HTSTFEndWaitAddr;
reg  [1:0]             SizeSymb;            // 0-3.2us / 1-6.4us / 2-12.8us
reg                    FrameParam8Update_1t;
reg  [4:0]             MidambleSymbolCount;
reg                    MidambleSymbol;
reg                    MidambleSymbol_1t;
reg  [9:0]             NMACount;
reg                    MidambleSkipFFT;

// Internal Wires Declarations
wire [8:0]             SymbolLengthLTF;
wire [12:0]            PlatFallTimeOut;
wire [13:0]            PlatFallTimeOutMod;
wire [9:0]             SymbolEndAddr;
wire [10:0]            FFTEndAddr;
wire [5:0]             TDSyncSkip;
wire [7:0]             TdCycRotVal;
wire [10:0]            AddrMaxOff;
wire [10:0]            FFTStartAddr;
wire [8:0]             FFTStartAddrLTF;
wire                   TDRxDataValid;
wire                   FFTMemWrEnMux;
wire                   PlatFallTimerEnd;
wire [2:0]             HTValidCounterEnd; // End value for HTValidCounter
wire                   FormatModeHTSTF;
wire                   LastHESTF;
wire                   MaskHESTFClear;
wire                   TDSymbolCounterIncr;
wire [10:0]            SampleCounterEnd;
wire [8:0]             PECounterEnd;
wire [6:0]             SMemRdAddrAdjust;
wire                   SMemDataValid;
wire                   SMemAlmostEmpty;
wire [6:0]             SMemWrNextAddr;
wire [6:0]             SMemRdNextAddr;
wire [8:0]             SampleDCCounterLimit;
wire [3:0]             Nhteltf;
wire [4:0]             Sumhtltf;
wire [15:0]            NheSigB;
wire signed [9:0]      TdAdjustShortGI;
wire [9:0]             TdAdjustSyncSkip;
wire [15:0]            TDNSym;         // Number of total symbols
wire                   TDBufEmpty;
wire                   n_FFTBufOverflow;                                                   
wire [15:0]            PreambleNSym;   // Number of preamble symbols
wire [15:0]            PreambleNSymOffset;
wire [15:0]            FrameParamNMAShift;
wire [15:0]            MidambleNSym;   // Number of midamble symbols
wire [2:0]             VHTFrameParamNSTS;
wire [4:0]             MidambleSymbolCountLimit;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// signal resampled for timing relax
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
  begin
    FDFormatMod_1t <= 4'b0;
    FDDone_1t      <= 1'b0;
  end
  else
  begin
    FDFormatMod_1t <= FDFormatMod;
    FDDone_1t      <= FDDone;
  end
end

// Synch pulse to TD DC Offset block at the beginning of GI of data symbols
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst == 1'b0)
    TDDCSynch <= 1'b0;
  else if (((CurrState == WAIT_GIEND1) && (PrevState != WAIT_GIEND1)) ||
           (!TDDCSynchDone && TDDCDataPhase && (SampleDCCounter == SampleDCCounterLimit)))
    TDDCSynch <= 1'b1;
  else
    TDDCSynch <= 1'b0;
end

// Set DC sample counter limit according to BW mode configuration
assign SampleDCCounterLimit = (CfgRegForce20) ? ((FrameParamGIType == 2'd2) ? 9'd64   : 
                                                 (FrameParamGIType == 2'd3) ? 9'd260  : 
                                                                              9'd452) :
                              (CfgRegForce40) ? ((FrameParamGIType == 2'd2) ? 9'd80   : 
                                                 (FrameParamGIType == 2'd3) ? 9'd273  :
                                                                              9'd465) :
                                                ((FrameParamGIType == 2'd2) ? 9'd80   : // TBD for 80MHz HE
                                                 (FrameParamGIType == 2'd3) ? 9'd273  : // TBD for 80MHz HE
                                                                              9'd471);

// Sample counter for TD DC synchro generation after HT-STF
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst == 1'b0) begin
    SampleDCCounter   <= 9'b0;
    TDDCDataSynch_1t  <= 1'b0;
    TDDCSynchDone     <= 1'b0;
    TDDCDataSynchDone <= 1'b0;
  end
  else if(PrevState == IDLE) begin
    SampleDCCounter   <= 9'b0;
    TDDCDataSynch_1t  <= 1'b0;
    TDDCSynchDone     <= 1'b0;
    TDDCDataSynchDone <= 1'b0;
  end
  else begin
    if ((TDDCDataSynch && !TDDCDataSynch_1t && !TDDCDataSynchDone) || (SampleDCCounter == 9'd479)) // 4us @120MHz
      SampleDCCounter <= 9'd0;
    else if (!CoarseFOEn || TDDCSynchDone)
      SampleDCCounter <= SampleDCCounter + 9'd1;
    
    TDDCDataSynch_1t <= TDDCDataSynch;
    
    if (TDDCSynchDone && TDDCDataSynch && !TDDCDataSynch_1t)
      TDDCDataSynchDone <= 1'b1;
    
    if (TDDCSynch)
      TDDCSynchDone <= 1'b1;
    else if (TDDCClear)
      TDDCSynchDone <= 1'b0;
  end
end

// TD DC Offset clear after HT-STF
assign TDDCClear = (CurrState == RESET_DC_UNLOCK || TDFOStrtP) ? 1'b1 : 1'b0;

// GI size information to TD DC Offset block
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst == 1'b0)
    TDDCSizeGI <= 2'd1;
  else if(PrevState == IDLE)
    TDDCSizeGI <= 2'd1;
  else if (FrameParam1Update &&  
    (((TDSymbolCounter > PreambleNSym) && TDDCDataSynch && (SymbolType == HTLTF || SymbolType == VHTSIGB)) || (SymbolType == HELTF)))
    TDDCSizeGI <= FrameParamGIType;
end

// Symbol size information to TD DC Offset block
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst == 1'b0)
    TDDCSizeSymb <= 2'd0;
  else if(PrevState == IDLE)
    TDDCSizeSymb <= 2'd0;
  else if (FrameParam1Update && TDDCDataSynch) begin
    if (SymbolType == HEDATA)
      TDDCSizeSymb <= 2'd2;
    else if (SymbolType == HELTF)
      TDDCSizeSymb <= FrameParamHeltfType;
    else   
      TDDCSizeSymb <= 2'd0;
  end
end

// Data / Preamble information to TD DC Offset block
always@(*)
begin
case (CurrState)
   IDLE, STF, AGCRELOCK_WAIT, RESET_DC_UNLOCK :
     TDDCDataPhase = 1'b0; // Indication that HT-STF is being processed
   WAIT_HTSTF_END : begin
     if(((FFTMemWrAddr < HTSTFEndWaitAddr) || !LastHESTF) && !TDDCSynchDone)
       TDDCDataPhase = 1'b0; // Indication that HT-STF is being processed
     else
       TDDCDataPhase = 1'b1;
   end
   default : begin
     TDDCDataPhase = 1'b1;
   end
 endcase
end // always

// Indicates the number of samples in a L-LTF : maximum limited to 40MHz BW
assign SymbolLengthLTF = 9'd64;

// Symbol size information from frame paramater
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst == 1'b0)
    SizeSymb <= 2'd0;
  else if(PrevState == IDLE)
    SizeSymb <= 2'd0;
  else if (FrameParam1Update) begin
    if (SymbolType == HELTF)
      SizeSymb <= FrameParamHeltfType;
    else if (SymbolType == HEDATA)
      SizeSymb <= 2'd2;
    else
      SizeSymb <= 2'd0;
  end
end

// Calculation of GI length. This length is used by the FSM for removing the GI
// The calculation is as follows
// Legacy mode 
//   GILength = 16 in 20 MHz BW, 32 in 40 MHz BW
// Mixed mode
//   GILength = 16/32  in 20/40 BW for preambles + signal fields
//   Short GI : GILength= 8/16 in 20/40 BW for data symbols
//   Long GI  : GILength= 16/32 in 20/40 BW for data symbols
// GF mode
//   GILength = 16/32  in 20/40 BW for preambles + signal fields
//   Short GI : GILength= 8/16 in 20/40 BW for data symbols
//   Long GI  : GILength= 16/32 in 20/40 BW for data symbols
// VHT mode
//   GILength = 16/32  in 20/40 BW for preambles + signal fields
//   Short GI : GILength= 8/16/32 in 20/40/80 BW for data symbols
//   Long GI  : GILength= 16/32/64 in 20/40/80 BW for data symbols
// HE mode
//   GILength = 16/32/64  in 20 BW for preambles + signal fields
//              32/64/128 in 40 BW for preambles + signal fields
//   GI size  : GILength= 16/32/64 in 20 BW for data symbols
//              GILength= 32/64/128 in 40 BW for data symbols
//              GILength= 64/128/256 in 80 BW for data symbols
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    GILength <= 9'd16;
  else if(PrevState == IDLE) begin
    if (TDBW20)
      GILength <= 9'd16;
    else
      GILength <= 9'd32;
  end
  else if (TDSymbolCounterIncr_2t) begin
    case (SymbolType)
      LSIG,LDATA,HTSIG,HTSTF,HTLTF,VHTSIGA,VHTSTF,VHTLTF,VHTSIGB,HESIGA,HESTF,HESIGB: begin
        if (TDBW20)
          GILength <= 9'd16;
        else if (TDBW40)
          GILength <= 9'd32;
        else
          GILength <= 9'd64;
      end
      HTDATA,VHTDATA : begin
        if (FrameParamGIType == 2'd0 && FrameParam1Update) begin
          if (TDBW20)
            GILength <= 9'd8;
          else if (TDBW40)
            GILength <= 9'd16;
          else
            GILength <= 9'd32;
        end
        else begin
          if (TDBW20)
            GILength <= 9'd16;
          else if (TDBW40)
            GILength <= 9'd32;
          else
            GILength <= 9'd64;
        end
      end
      HELTF,HEDATA : begin
        if (FrameParamGIType == 2'd1 && FrameParam1Update) begin
          if (TDBW20)
            GILength <= 9'd16;
          else if (TDBW40)
            GILength <= 9'd32;
          else
            GILength <= 9'd64;
        end
        else if (FrameParamGIType == 2'd2 && FrameParam1Update) begin
          if (TDBW20)
            GILength <= 9'd32;
          else if (TDBW40)
            GILength <= 9'd64;
          else
            GILength <= 9'd128;
        end
        else begin
          if (TDBW20)
            GILength <= 9'd64;
          else if (TDBW40)
            GILength <= 9'd128;
          else
            GILength <= 9'd256;
        end
      end
      // pragma coverage block = off
      default :
        GILength <= 9'd16;
      // pragma coverage block = on
    endcase
  end
end

// TDSymbolCounterIncr delayed for GILength generation
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    TDSymbolCounterIncr_1t <= 1'b0;
    TDSymbolCounterIncr_2t <= 1'b0;
  end
  else begin
    TDSymbolCounterIncr_1t <= TDSymbolCounterIncr;
    TDSymbolCounterIncr_2t <= TDSymbolCounterIncr_1t;
  end
end

// CSFlag is a Cyclic Shift flag indicating whether the last 6/12/24
// samples of a symbol are written into the FFT memory.
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if(nPhyRst == 1'b0)
    CSFlag <= 1'b0;
  else if(PrevState == IDLE)
    CSFlag <= 1'b0;
  else if(CurrState != FFT_XFERLOOP && NextState == FFT_XFERLOOP) 
    CSFlag <= 1'b1;
  else if(FFTMemWrAddr < FFTStartAddr[9:0])
    CSFlag <= 1'b0;
end

// CmpLTFDoneFlag is a flag indicating LTF transfer to FFT is complete
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst == 1'b0)
    CmpLTFDoneFlag <= 1'b0;
  else if(PrevState == IDLE)
    CmpLTFDoneFlag <= 1'b0;
  else if(CmpLTFDoneP)
    CmpLTFDoneFlag <= 1'b1;
end

// Decode number of HT-LTF and HE-LTF
always @*
begin
  // For HT-MM and HT-GF 
  if(FDFormatMod_1t == MODE_MM || FDFormatMod_1t == MODE_GF) begin
    if (FrameParamNSTS == 3'd0) begin
      if (FDFormatMod_1t == MODE_GF)
        Nhtltf = 4'd0;
      else
        Nhtltf = 4'd1;
    end
    else if (FrameParamNSTS == 3'd1) begin
      if (FDFormatMod_1t == MODE_GF)
        Nhtltf = 4'd1;
      else
        Nhtltf = 4'd2;
    end
    else
      Nhtltf = 4'd4;
  end
  // For VHT
  else if (FDFormatMod_1t == MODE_VHT) begin
    if (VHTFrameParamNSTS == 3'd0)
      Nhtltf = 4'd1;
    else if (VHTFrameParamNSTS == 3'd1)
      Nhtltf = 4'd2;
    else if (VHTFrameParamNSTS == 3'd2)
      Nhtltf = 4'd4;
    else if (VHTFrameParamNSTS == 3'd3)
      Nhtltf = 4'd4;
    else if (VHTFrameParamNSTS == 3'd4)
      Nhtltf = 4'd6;
    else if (VHTFrameParamNSTS == 3'd5)
      Nhtltf = 4'd6;
    else
      Nhtltf = 4'd8;
  end
  // For HE
  else begin
    if (FrameParamNHeltf == 3'd0)
      Nhtltf = 4'd1;
    else if (FrameParamNHeltf == 3'd1)
      Nhtltf = 4'd2;
    else if (FrameParamNHeltf == 3'd3)
      Nhtltf = 4'd4;
    else if (FrameParamNHeltf == 3'd5)
      Nhtltf = 4'd6;
    else
      Nhtltf = 4'd8;
  end
end

// Decode number of VHT-LTF
assign VHTFrameParamNSTS = 
`ifdef RW_MUMIMO_RX_EN
                           (FrameParamMuMIMO) ? FrameParam1MuNSTS : // Total NSTS
`endif // RW_MUMIMO_RX_EN
                            FrameParamNSTS;                         // SU NSTS

// Decode number of HT-ELTF
assign Nhteltf = (FrameParamNESS == 2'd3) ? 4'd4 : {2'b0,FrameParamNESS};

// Decode number of HT-LTF + HT-ELTF
assign Sumhtltf = {1'b0,Nhtltf} + {1'b0,Nhteltf};

// Decode number of HE-Sig-B symbols
assign NheSigB = {8'b0,FrameParamNHeSigB};
 
// The symbol type determines the current symbol being
// processed by the time domain part of the Modem
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    SymbolType <= LSIG;
  end
  else if(PrevState == IDLE || TDSymbolCounter == 16'd0)
    SymbolType <= LSIG;
  else begin
    case (FDFormatMod_1t)
      MODE_LEG,MODE_DUP: begin // Legacy and Legacy duplicate
        if(TDSymbolCounter < 16'd2)
          SymbolType <= LSIG;
        else
          SymbolType <= LDATA;
      end
      MODE_MM: begin // Mixed mode
        if(TDSymbolCounter < 16'd4)
          SymbolType <= HTSIG;
        else if(TDSymbolCounter == 16'd4)
          SymbolType <= HTSTF;
        else if(TDSymbolCounter <= (16'd4 + {11'b0,Sumhtltf}))
          SymbolType <= HTLTF;
        else
          SymbolType <= HTDATA;
      end
      MODE_GF: begin // Greenfield Mode
        if(TDSymbolCounter < 16'd3)
          SymbolType <= HTSIG;
        else if(TDSymbolCounter == 16'd3) // Force as HTLTF, as 1st symbol will be always in long GI
          SymbolType <= HTLTF;            // because GF Nss=1 short GI is fordidden
        else if(TDSymbolCounter <= (16'd2 + {11'b0,Sumhtltf}))
          SymbolType <= HTLTF;
        else
          SymbolType <= HTDATA;
      end
      MODE_VHT: begin // VHT mode
        if(TDSymbolCounter <= 16'd4)
          SymbolType <= VHTSTF;
        else if(TDSymbolCounter <= (16'd4 + {12'b0,Nhtltf}))
          SymbolType <= VHTLTF;
        else if(TDSymbolCounter <= (16'd5 + {12'b0,Nhtltf}))
          SymbolType <= VHTSIGB;
        else
          SymbolType <= VHTDATA;
      end
      MODE_HE_SU: begin // HE-SU mode
        if(TDSymbolCounter <= 16'd4)
          SymbolType <= HESIGA;
        else if(TDSymbolCounter == 16'd5)
          SymbolType <= HESTF;
        else if((TDSymbolCounter <= (16'd5 + {12'b0,Nhtltf})) ||
          (FrameParamDoppler && FrameParam1Update && MidambleSymbol))
          SymbolType <= HELTF;
        else
          SymbolType <= HEDATA;
      end
      MODE_HE_MU: begin // HE-MU mode
        if(TDSymbolCounter <= 16'd4)
          SymbolType <= HESIGA;
        else if((TDSymbolCounter <= (16'd4 + NheSigB)) || (TDSymbolCounter == 16'd5))
          SymbolType <= HESIGB;
        else if(TDSymbolCounter == (16'd5 + NheSigB))
          SymbolType <= HESTF;
        else if((TDSymbolCounter <= (16'd5 + NheSigB + {12'b0,Nhtltf})) ||
          (FrameParamDoppler && FrameParam1Update && MidambleSymbol))
          SymbolType <= HELTF;
        else
          SymbolType <= HEDATA;
      end
      MODE_HE_ER: begin // HE-ER mode
        if(TDSymbolCounter <= 16'd6)
          SymbolType <= HESIGA;
        else if(TDSymbolCounter == 16'd7)
          SymbolType <= HESTF;
        else if((TDSymbolCounter <= (16'd7 + {12'b0,Nhtltf})) ||
          (FrameParamDoppler && FrameParam1Update && MidambleSymbol))
          SymbolType <= HELTF;
        else
          SymbolType <= HEDATA;
      end
      MODE_HE_TB: begin // HE-TB mode
        if(TDSymbolCounter <= 16'd4)
          SymbolType <= HESIGA;
        else if(TDSymbolCounter <= 16'd6)
          SymbolType <= HESTF;
        else if((TDSymbolCounter <= (16'd6 + {12'b0,Nhtltf})) ||
          (FrameParamDoppler && FrameParam1Update && MidambleSymbol))
          SymbolType <= HELTF;
        else
          SymbolType <= HEDATA;
      end
      // pragma coverage block = off
      default: SymbolType <= LSIG;
      // pragma coverage block = on
    endcase
  end
end

// Logic for generation of Sample Skip value(+1/-1)
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    FDTOffsetComp_1t <= 6'b0;
    FDTOffsetComp_2t <= 6'b0;
  end
  else if(PrevState == IDLE) begin
    FDTOffsetComp_1t <= 6'b0;
    FDTOffsetComp_2t <= 6'b0;
  end
  else begin
    FDTOffsetComp_1t <= FDTOffsetComp;
    if((CurrState == WAIT_GIEND2) && (CurrState != NextState) && !LargeGIMask)
      FDTOffsetComp_2t <= FDTOffsetComp_1t;
  end
end

always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    LargeGIMask <= 1'b0;
  else if(PrevState == IDLE)
    LargeGIMask <= 1'b0;
  else if((CurrState == WAIT_GIEND2) && (FDTOffsetComp_1t != FDTOffsetComp) && !LargeGIMask)
    LargeGIMask <= 1'b1;
  else if(NextState == FFT_STARTLOOP)
    LargeGIMask <= 1'b0;
end

assign TDSyncSkip = (FDTOffsetComp_1t - FDTOffsetComp_2t) & ~{6{LargeGIMask}};

// Adjust FFT window from FD timing offset compensation
assign TdAdjustSyncSkip = (TDSyncSkip[1:0] == 2'b01) ? 10'd2 : // Sample Repeat
                          (TDSyncSkip[1:0] == 2'b11) ? 10'd0 : // Sample Skip
                                                       10'd1;  // No shift

// TDFOEstGClkEn signal is used as a clock gating enable
// for the TDFO Estimation block
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    TDFOEstGClkEn <= 1'b0;
  else if((PrevState == RXEND) || CmpLTFDoneP_2t)
    TDFOEstGClkEn <= 1'b0;
  else if(NextState==STF && CurrState != STF)
    TDFOEstGClkEn <= 1'b1;
end

// CmpLTFDoneP delayed for TDFO clock enable
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    CmpLTFDoneP_1t <= 1'b0;
    CmpLTFDoneP_2t <= 1'b0;
  end
  else begin
    CmpLTFDoneP_1t <= CmpLTFDoneP;
    CmpLTFDoneP_2t <= CmpLTFDoneP_1t;
  end
end

// TBEEnable signal is used to start the TBE Cross correlation
// for estimating the symbol boundaries
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    TBEEnable <= 1'b0;
  else if(PrevState == IDLE)
    TBEEnable <= 1'b0;
  else if(CurrState == STF && NextState == LTF)
    TBEEnable <= 1'b1;
  else if(CurrState!=WAIT_GIEND1 && PrevState == WAIT_GIEND1)
    TBEEnable <= 1'b0;
end

// RxTDEnable delayed for TBEGClkEn generation
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    RxTDEnable_1t <= 1'b0;
  else
    RxTDEnable_1t <= RxTDEnable;
end

// TBEGClkEn signal is used as a clock gating enable for the TBE block
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    TBEGClkEn <= 1'b0;
  else if(RxTDEnable && !RxTDEnable_1t)
    TBEGClkEn <= 1'b1;
  else if(TBEEnable_2t && !TBEEnable_1t)
    TBEGClkEn <= 1'b0;
end

// TBEEnable delayed for clock enable
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    TBEEnable_1t <= 1'b0;
    TBEEnable_2t <= 1'b0;
  end
  else begin
    TBEEnable_1t <= TBEEnable;
    TBEEnable_2t <= TBEEnable_1t;
  end
end

// TDFFTBufWrPtr signal is used to select the FFT buffer
// into which the current symbol is going to be written
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    TDFFTBufWrPtrInt <= 2'd0;
  else if(CurrState == IDLE && PrevState != RXEND && PrevState != IDLE)
    // Clear in case of rx abort, in normal case last state must be RXEND
    TDFFTBufWrPtrInt <= 2'd0;
  else if((PrevState == IDLE || PrevState == WAIT_PE_END) && TDBufEmpty && (FDDone_1t || (CurrState == STF)))
    // Clear at the end of the last symbol FFT
    TDFFTBufWrPtrInt <= 2'd0;
  else if(TickSymbolFFT && !NoMoreFFT) begin
    case (TDFFTBufWrPtrInt)
      2'd0:    TDFFTBufWrPtrInt <= 2'd1;
      2'd1:    TDFFTBufWrPtrInt <= 2'd2;
      2'd2:    TDFFTBufWrPtrInt <= 2'd3;
      default: TDFFTBufWrPtrInt <= 2'd0;
    endcase
  end
end

// Assign FFT buffer pointers to internal registers with specific synthesis attributes
// in order to prevent encoding error with Synplify
assign TDFFTBufWrPtr = TDFFTBufWrPtrInt;
assign FFTBufPtr     = FFTBufPtrInt;

// FFT control
assign TDBufEmpty       = (TDFFTBufWrPtrInt == FFTBufPtrInt);
assign n_FFTBufOverflow = (TDFFTBufWrPtrInt[0]==FDFFTBufRdPtr[0]) & (TDFFTBufWrPtrInt[1]!=FDFFTBufRdPtr[1]);

always @(*) 
begin
  case(FFTBufPtrInt)
    2'd0:    n_FFTBufPtr = 2'd1;
    2'd1:    n_FFTBufPtr = 2'd2;
    2'd2:    n_FFTBufPtr = 2'd3;
    default: n_FFTBufPtr = 2'd0;
  endcase
end

always @(posedge PhyClk,negedge nPhyRst)
begin
  if (!nPhyRst) begin
    FFTState       <= 1'b0;
    FFTStart       <= 1'b0;
    FFTLen         <= 3'd7;
    FFTScale       <= 3'd0;
    FFTBufPtrInt   <= 2'b0;
    FFTBufOverflow <= 1'b0;
  end
  else begin
    // Pulse to start FFT
    FFTStart <= 1'b0;
    
    // FFT length and scale
    if((PrevState == STF) || (PrevState == LTF))
    begin
      FFTLen   <= 3'd7; // Set FFT length as no FFT during TDFO computation
      FFTScale <= 3'd0; // Set default value for preamble symbols
    end
    else if((PrevState != IDLE) && (FDFormatMod_1t <= MODE_VHT)) begin
      if (TDBufEmpty) begin
        if (FineFODone) begin
          if (!CfgRegForce20 && (SymbolType == LSIG))
            FFTLen <= 3'd6; // Set FFT length as 2x FFT64
          else
            FFTLen <= {1'b0,TDBWEarly}; // Set FFT length as TD BW for non-HT, HT and VHT frames
        end
        if (TDBW20)
          FFTScale <= 3'd0;
        else if (TDBW40)
          FFTScale <= 3'd1;
        else
          FFTScale <= 3'd2;
      end
    end
    else if((PrevState == WAIT_GIEND2) || (PrevState == WAIT_HTLTF_GIEND)) begin
      if (TDBW20) begin
        if (SymbolType == HESIGB && !CfgRegForce20)
          FFTLen <= 3'd6;                        // Set FFT length as 2x FFT64 for HESIGB
        else
          FFTLen <= {1'b0,SizeSymb};              // Set FFT length as symbol size for HE frames
        if (SymbolType == HELTF)
          FFTScale <= {1'b0,FrameParamHeltfType}; // Set scaling according to HELTF 1x, 2x, 4x
        else if (SymbolType == HEDATA)
          FFTScale <= 3'd2;                       // Set 2 for HE format FFT 256
      end
      else if (TDBW40) begin
        FFTLen <= {1'b0,SizeSymb} + 2'd1;                // Set FFT length as symbol size for HE frames + 1
        if (SymbolType == HELTF)
          FFTScale <= {1'b0,FrameParamHeltfType} + 3'd1; // Set scaling according to HELTF (1x, 2x, 4x) + 1
        else if (SymbolType == HEDATA)
          FFTScale <= 3'd3;                              // Set 3 for HE format FFT 512
      end
      else begin
        FFTLen <= {1'b0,SizeSymb} + 2'd2;                // Set FFT length as symbol size for HE frames + 2
        if (SymbolType == HELTF)
          FFTScale <= {1'b0,FrameParamHeltfType} + 3'd2; // Set scaling according to HELTF (1x, 2x, 4x) + 2
        else if (SymbolType == HEDATA)
          FFTScale <= 3'd4;                              // Set 4 for HE format FFT 1024
      end
    end
    
    // FFT buffer overflow, mask it when RxFD is done (Mu-MIMO case)
    if((NextState == STF) || !RxTDEnable || FDDone_1t)
      FFTBufOverflow <= 1'b0;
    else
      FFTBufOverflow <= n_FFTBufOverflow;

  /*****************************************************************************
  * assert_FFT_overflow: detected if distance between FFT read/write pointer > 1
  *****************************************************************************/
  // pragma coverage block = off, expr = off, toggle = off
`ifdef RW_SIMU_ON
    if(FFTBufOverflow)
      $display("ASSERT @%t:%m:assert_FFT_overflow: FFT buffer overflow situation detected !",$time); 
`endif
    // pragma coverage block = on, expr = on, toggle = on
    
    // FFT state
    if(CurrState == IDLE && PrevState != RXEND && PrevState != IDLE) begin
      // Clear in case of rx abort, in normal case last state must be RXEND
      FFTState     <= 1'b0;
      FFTBufPtrInt <= 2'b0;            
    end
    else if((PrevState == IDLE || PrevState == WAIT_PE_END) && TDBufEmpty) begin
      FFTState <= 1'b0;
        // Clear only at the end of the last symbol FFT
        // to let buffer toggling to be seen by FD
        if (FDDone_1t || (CurrState == STF))
          FFTBufPtrInt <= 2'b0;            
    end
    else if(!FFTState) begin
      // FFT not running
      if(!TDBufEmpty && (PrevState != IDLE) && !FDDone_1t) begin
        // new buffer, launch FFT processing
        FFTStart <= 1'b1;
        FFTState <= 1'b1;
      end
    end
    else begin
      // FFT running
      if(FFTDone) begin
        // FFT processing completed
        FFTBufPtrInt <= n_FFTBufPtr;
        FFTState     <= 1'b0;
      end
    end
  end
end

// HTValidCounter is a counter used to generate internal data valid during
// HT/VHT/HE-STF where AGC will reset FE and FFTMemWrEnIn will stop toggling
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    HTValidCounter    <= 3'b0;
    HTValidCounterSel <= 3'b0;
  end
  else if(PrevState == IDLE) begin
    HTValidCounter    <= 3'b0;
    HTValidCounterSel <= 3'b0;
  end
  else begin
    if (FFTMemWrEnIn && CurrState == FFT_XFERLOOP)
      HTValidCounterSel <= HTValidCounter;
    if (HTValidCounter < HTValidCounterEnd)
      HTValidCounter <= HTValidCounter + 3'b1;
    else
      HTValidCounter <= 3'b0;
  end
end

assign HTValidCounterEnd = (TDBW40) ? 3'd2 : 3'd5;

assign TDRxDataValid = ((HTValidCounter == HTValidCounterSel) && ((TDSymbolCounter <= PreambleNSym) || NDPGFNoUnlock)) ? 1'b1 : 1'b0;

// SampleCounter counts the number of time domain samples
// and its value is used for STF, LTF and WAIT_GIEND1 states for deciding events such as
// end of FFT Memory Write, end of TDBW estimation, end of AGC Lock timeout and GI removal done.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    SampleCounter <= 11'b0;
  // Initialization of SampleCounter at the beginning of reception
  else if(PrevState == IDLE)
    SampleCounter <= 11'b0;
  // Initialization at L-STF end
  else if(CurrState == STF && NextState == LTF)
    SampleCounter <= 11'b0;
  // Initialization at L-LTF2 end
  else if(CurrState == LTF && NextState == WAIT_GIEND1)
    SampleCounter <= 11'b0;
  // Initialization at AGCUNLOCK_WAIT start
  else if(CurrState != AGCUNLOCK_WAIT && NextState == AGCUNLOCK_WAIT)
    SampleCounter <= 11'b0;
  // Initialization at WAIT_PE_END start
  else if(CurrState != WAIT_PE_END && NextState == WAIT_PE_END)
    SampleCounter <= 11'b0;
  // Increment SampleCounter if RxDataValidIn is high or in AGCUNLOCK_WAIT for AGC time-out
  else if((RxDataValidIn && CoarseFOEn) || (CurrState == AGCUNLOCK_WAIT) || (CurrState == WAIT_PE_END && FFTMemWrEnIn))
    SampleCounter <= SampleCounter + 11'd1;
end


assign TdCycRotVal = (TDBW20) ? CfgRegTdCycRotVal20 :
                     (TDBW40) ? CfgRegTdCycRotVal40 : CfgRegTdCycRotVal80;
assign AddrMaxOff  = (TDBW20) ? ((SizeSymb == 2'd0) ? 11'd64  :
                                 (SizeSymb == 2'd1) ? 11'd128 : 11'd256) :
                     (TDBW40) ? ((SizeSymb == 2'd0) ? 11'd128 :
                                 (SizeSymb == 2'd1) ? 11'd256 : 11'd512) :
                                ((SizeSymb == 2'd0) ? 11'd256 :
                                 (SizeSymb == 2'd1) ? 11'd512 : 11'd1024);

assign FFTStartAddr      = AddrMaxOff - {3'b0,TdCycRotVal};
assign FFTStartAddrLTF   = 9'd64  - {1'b0,CfgRegTdCycRotVal20};
assign FFTEndAddr        = AddrMaxOff - 11'd1;

assign FFTMemWrEnMux     = (HTSTFNoUnlock) ? TDRxDataValid : FFTMemWrEnIn;

// FFTMemWrAddr is generated for L1+L2/2 as well as for data portion of the symbol
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst == 1'b0)
    FFTMemWrAddr <= 10'b0;

  else if(CurrState == IDLE)
    FFTMemWrAddr <= 10'b0;

  // FFT start loop initialization for next coming GI
  else if((CurrState != FFT_STARTLOOP) && (NextState == FFT_STARTLOOP))
    FFTMemWrAddr <= 10'b0;

  // First LTF
  else if((CurrState == LTF) && (PrevState != LTF))
    FFTMemWrAddr <= {1'b0,FFTStartAddrLTF};

  // FFT data symbol loop operation initialization
  else if((CurrState != FFT_XFERLOOP) && (NextState == FFT_XFERLOOP))
    FFTMemWrAddr <= FFTStartAddr[9:0];
  
  // GI of first HT-LTF
  else if((CurrState != WAIT_HTLTF_GIEND) && (NextState == WAIT_HTLTF_GIEND))
    FFTMemWrAddr <= 10'b0;

  // Any symbol ending
  else if(FFTMemWrEnMux && (FFTMemWrAddr == FFTEndAddr[9:0]))
    FFTMemWrAddr <= 10'b0;

  // Else always increment
  else if(FFTMemWrEnMux && !FFTMemStop)
    FFTMemWrAddr <= FFTMemWrAddr + 10'd1;
end

// The Time domain BW register indicates the Channel Bandwidth
// Till the signal field gets decoded, this register will
// hold the TD bandwidth estimator, after the HT/VHT-SIG field gets decoded,
// the Bandwidth gets updated by the CBW bit of HT/VHT-SIG at the WAIT_HTSTF_END boundary
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if (nPhyRst == 1'b0) begin
    TDBWInter <= TDBW_DEFAULT;
    TDBWEarly <= TDBW_DEFAULT;
  end
  else if(CfgRegForce20) begin
    TDBWInter <= 2'd0;
    TDBWEarly <= 2'd0;
  end
  else if(CurrState == IDLE) begin
    TDBWInter <= TDBW_DEFAULT;
    TDBWEarly <= TDBW_DEFAULT;
  end
  else if(FrameParam1Update && FormatModeHTSTF) begin
    if (HTSTFNoUnlock)
    begin
      if (FrameParamNSD == 2'b0 && (FDFormatMod_1t == MODE_MM || FDFormatMod_1t == MODE_GF))
        TDBWEarly <= 2'b0; // HT mcs32
      else if (CfgRegForce40 && (FrameParamCBW > 2'd1))
        TDBWEarly <= 2'd1; // HE 80MHz in 40MHz BW
      else
        TDBWEarly <= FrameParamCBW;
    end
    if ((CurrState == WAIT_HTSTF_END) && (NextState != WAIT_HTSTF_END))
    begin
      if (FrameParamNSD == 2'b0 && (FDFormatMod_1t == MODE_MM || FDFormatMod_1t == MODE_GF))
        TDBWInter <= 2'b0; // HT mcs32
      else if (CfgRegForce40 && (FrameParamCBW > 2'd1))
        TDBWInter <= 2'd1; // HE 80MHz in 40MHz BW
      else
        TDBWInter <= FrameParamCBW;
    end
  end
end

// Frame mode with Hx-STF -> MODE_MM or MODE_VHT or MODE_HE_SU or MODE_HE_MU or MODE_HE_ER or MODE_HE_TB
assign FormatModeHTSTF = (FDFormatMod_1t == MODE_MM    || FDFormatMod_1t == MODE_VHT   ||
                          FDFormatMod_1t == MODE_HE_SU || FDFormatMod_1t == MODE_HE_MU ||
                          FDFormatMod_1t == MODE_HE_ER || FDFormatMod_1t == MODE_HE_TB) ? 1'b1 : 1'b0;

// 2nd HE-STF with HE-TB frame
assign LastHESTF = (FDFormatMod_1t == MODE_HE_TB) ? (TDSymbolCounter == 16'd6) : 1'b1;

// Mask 2nd HE-STF clear with HE-TB frame
assign MaskHESTFClear = (FDFormatMod_1t == MODE_HE_TB) ? ~LastHESTF : 1'b1;

// Time domain BW
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if (nPhyRst == 1'b0) begin
    TDBW20 <= ~TDBW_DEFAULT[0];
    TDBW40 <= TDBW_DEFAULT[0]; // Default 40MHz
  end
  else begin
    TDBW20 <= (TDBWInter == 2'd0); // Time domain BW=20MHz
    TDBW40 <= (TDBWInter == 2'd1); // Time domain BW=40MHz
  end
end

// Data Compensation Start required by TDFO block for compensation of data part
// DataCompStartP should be asserted 2 times :
// 1) before TDFO receives SIG samples
// 2) before TDFO receives (V)HT-LTF samples due to AGC gain update
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
     DataCompStartP <= 1'b0;
  else if (((NextState==FFT_XFERLOOP) && ((CurrState == WAIT_GIEND1) || (CurrState==SMEM_XFER))) ||
        (FormatModeHTSTF && (((CurrState != WAIT_HTLTF_GIEND) && (NextState == WAIT_HTLTF_GIEND)) || SMemRdAddrWait_1t)))
     DataCompStartP <= 1'b1;
  else
     DataCompStartP <= 1'b0;
end

// Start TDFO block as soon as RxTDEnable is active
assign TDFOStrtP = RxTDEnable & ~TDCompOnGoing & ~HTSTFNoUnlock;

// Ensure that TD compensation clock is available before sending 
// start pulse to TDFO and TDDC
always @(posedge TDCompGClk or negedge nPhyRst)
begin
  if(nPhyRst==1'b0)
    TDCompOnGoing <= 1'b0;
  else
    TDCompOnGoing <= RxTDEnable;
end

// TDSymbolCounter keeps track of the
// number of symbols written to the FFT Memory
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if (nPhyRst == 1'b0)
    TDSymbolCounter <= 16'b0;
  else if(PrevState == IDLE || PrevState == RXEND)
    TDSymbolCounter <= 16'b0;
  else if(TDSymbolCounterIncr)
    TDSymbolCounter <= TDSymbolCounter + 16'd1;
end

assign TDSymbolCounterIncr = (NextState == FFT_STARTLOOP)                                     |
                             (CurrState == WAIT_HTLTF_GIEND && PrevState != WAIT_HTLTF_GIEND) |
                             CmpLTFDoneP_2t;

// MidambleSymbolCount keeps track of the number of data symbols for MidambleSymbol periodicity
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    MidambleSymbolCount <= 5'b0;
    MidambleSymbol      <= 1'b0;
    NMACount            <= 10'b0;
    MidambleSkipFFT     <= 1'b0;
  end
  else if(PrevState == IDLE || PrevState == RXEND)
  begin
    MidambleSymbolCount <= 5'b0;
    MidambleSymbol      <= 1'b0;
    NMACount            <= 10'b0;
    MidambleSkipFFT     <= 1'b0;
  end
  else
  begin
    // Counter for MidambleSymbol periodicity
    if((CurrState == FFT_STARTLOOP) && FrameParamDoppler && FrameParam1Update && (TDSymbolCounter > PreambleNSym + 16'd2))
    begin
      if (((SymbolType == HEDATA) && (MidambleSymbolCount < MidambleSymbolCountLimit)) ||
          ((SymbolType == HELTF)  && (MidambleSymbolCount < {1'b0,Nhtltf} - 5'd1)))
        MidambleSymbolCount <= MidambleSymbolCount + 5'd1;
      else
      begin
        MidambleSymbolCount <= 5'b0;
        MidambleSymbol      <= ~MidambleSymbol & (NMACount < {1'b0,FrameParamNMA});
      end
    end
    // Counter for NMA limit on falling edge of MidambleSymbol
    if (!MidambleSymbol && MidambleSymbol_1t)
      NMACount <= NMACount + 10'd1;
    // Skip midamble FFT according to config register
    if (MidambleSymbol_1t)
    begin
      if (FrameParamHeltfType == HE_1xLTF)
        MidambleSkipFFT <= CfgRegSkipMidamble1x;
      else if (FrameParamHeltfType == HE_2xLTF)
        MidambleSkipFFT <= CfgRegSkipMidamble2x;
      else
        MidambleSkipFFT <= CfgRegSkipMidamble4x;
    end
    else
      MidambleSkipFFT <= 1'b0;
  end
end

assign MidambleSymbolCountLimit = (FrameParamMidamble) ? 5'd19 : 5'd9;

always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    MidambleSymbol_1t <= 1'b0;
  else
    MidambleSymbol_1t <= MidambleSymbol;
end

// Decode number of preamble symbols minus 2
assign PreambleNSymOffset = (FDFormatMod_1t == MODE_MM)    ? 16'd3 :
                            (FDFormatMod_1t == MODE_GF)    ? 16'd1 :
                            (FDFormatMod_1t == MODE_VHT)   ? 16'd4 :
                            (FDFormatMod_1t == MODE_HE_SU) ? 16'd4 :
                            (FDFormatMod_1t == MODE_HE_MU) ? 16'd4 + NheSigB :
                            (FDFormatMod_1t == MODE_HE_ER) ? 16'd6 :
                                                             16'd5;
assign PreambleNSym = (FDFormatMod_1t == MODE_LEG || FDFormatMod_1t == MODE_DUP) ? 16'd0 :
                                                                      {11'b0,Sumhtltf} + PreambleNSymOffset;
// Decode number of midamble symbols
assign FrameParamNMAShift = (FrameParamNHeltf == 3'd0) ? {7'b0,FrameParamNMA}      :
                            (FrameParamNHeltf == 3'd1) ? {6'b0,FrameParamNMA,1'b0} : {5'b0,FrameParamNMA,2'b0};
assign MidambleNSym       = (FrameParamDoppler && FrameParam1Update) ? FrameParamNMAShift : 16'b0;

// Total number of TD symbols
assign TDNSym = PreambleNSym + MidambleNSym + FrameParamNSym;

// FrameParam8Update delayed for TDLastSymbol generation
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    FrameParam8Update_1t <= 1'b0;
  else
    FrameParam8Update_1t <= FrameParam8Update;
end

// TDLastSymbol flag for RxTDDone generation
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if (nPhyRst == 1'b0)
    TDLastSymbol <= 1'b0;
  else if(PrevState == IDLE)
    TDLastSymbol <= 1'b0;
  else if (FrameParam8Update_1t && (TDSymbolCounter > TDNSym))
    TDLastSymbol <= 1'b1;
end

// TDStopFFT flag for RxTDDone generation in case of NDP case for GF Nss=1 / VHT
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if (nPhyRst == 1'b0)
    TDStopFFT <= 1'b0;
  else if(PrevState == IDLE)
    TDStopFFT <= 1'b0;
  else if(TDLastSymbol && ((TDSymbolCounter - TDNSym) > 16'd1))
    TDStopFFT <= 1'b1;
end

// NoMoreFFT flag for RxTDDone generation in case of HE PE
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if (nPhyRst == 1'b0)
    NoMoreFFT <= 1'b0;
  else if(PrevState == IDLE)
    NoMoreFFT <= 1'b0;
  else if((TickSymbolFFT && (CurrState == WAIT_PE_END)) || FDDone_1t)
    NoMoreFFT <= 1'b1;
end

// FFTMemStop flag for stopping FFT write enable
assign FFTMemStop = (PrevState == WAIT_PE_END) | RxTDDone;

// Plateau fall timer for time out generation
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    PlatFallTimer <= 14'b0;
  else if(CurrState == STF)
    PlatFallTimer <= PlatFallTimer + 14'b1;
  else
    PlatFallTimer <= 14'b0;
end

// Delay value for plateau fall time out
assign PlatFallTimeOut    = {2'b0,CfgRegPlatFallTimeOut[11:1]} + {1'b0,CfgRegPlatFallTimeOut};
assign PlatFallTimeOutMod = {PlatFallTimeOut,1'b0};

assign PlatFallTimerEnd = (PlatFallTimer == PlatFallTimeOutMod) ? 1'b1 : 1'b0;

// Generation of PlatTimeout signal to Main TxRx state machine
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    PlatTimeout <= 1'b0;
  else if(PrevState == IDLE)
    PlatTimeout <= 1'b0;
  else if((CurrState == STF) && PlatFallTimerEnd)
    PlatTimeout <= 1'b1;
end

// Generation of RxTDDone signal to Main TxRx state machine
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    RxTDDone <= 1'b0;
  else if(CurrState == RXEND)
    RxTDDone <= 1'b1;
  else
    RxTDDone <= 1'b0;
end

// Generation of contol signals for the Sample memory at the input of TDFO
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    SMemMuxSel <= 1'b0;
  else if(PrevState == IDLE)
    SMemMuxSel <= 1'b0;
  else if(SMemFlush && SMemRdDataValid && (TDBWInter != 2'd2))
    SMemMuxSel <= 1'b0;
  else if(SMemMuxPrePreLast80 && (TDBWInter == 2'd2))
    SMemMuxSel <= 1'b0;
  else if((NextState == SMEM_XFER) || ((TDBWEarly != TDBWInter) && FrameParam1Update))
    SMemMuxSel <= 1'b1;
end

// Free Sample memory and load real time data flow
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    SMemFlush <= 1'b0;
  else if(PrevState == IDLE)
    SMemFlush <= 1'b0;
  else if (SMemFlush && SMemRdDataValid)
    SMemFlush <= 1'b0;
  else if(SMemMuxSel && SMemRdDataValid && SMemAlmostEmpty && ((SymbolType == HELTF)  || 
                                                               (SymbolType == VHTLTF) ||
                                                               (SymbolType == HTLTF)))
    SMemFlush <= 1'b1;
end

always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    SMemMuxPreLast      <= 1'b0;
    SMemMuxLast         <= 1'b0;
    //
    SMemMuxPrePreLast80 <= 1'b0;
    SMemMuxPreLast80    <= 1'b0;
    SMemMuxLast80_m2    <= 1'b0;
    SMemMuxLast80_m1    <= 1'b0;
    SMemMuxLast80       <= 1'b0;
  end
  else if(PrevState == IDLE) begin
    SMemMuxPreLast      <= 1'b0;
    SMemMuxLast         <= 1'b0;
    SMemMuxPrePreLast80 <= 1'b0;
    SMemMuxPreLast80    <= 1'b0;
    SMemMuxLast80_m2    <= 1'b0;
    SMemMuxLast80_m1    <= 1'b0;
    SMemMuxLast80       <= 1'b0;
  end
  else if (TDBWInter == 2'd2) begin
    SMemMuxPrePreLast80 <= SMemFlush & SMemRdDataValid;
    SMemMuxPreLast80    <= SMemMuxPrePreLast80;
    SMemMuxPreLast      <= SMemMuxPreLast80;
    SMemMuxLast80_m2    <= SMemMuxPreLast;
    SMemMuxLast80_m1    <= SMemMuxLast80_m2;
    SMemMuxLast80       <= SMemMuxLast80_m1;
    SMemMuxLast         <= SMemMuxLast80;
  end
  else begin
    SMemMuxPreLast      <= SMemFlush & SMemRdDataValid;
    SMemMuxLast         <= SMemMuxPreLast;
    SMemMuxPrePreLast80 <= 1'b0;
    SMemMuxPreLast80    <= 1'b0;
    SMemMuxLast80_m2    <= 1'b0;
    SMemMuxLast80_m1    <= 1'b0;
    SMemMuxLast80       <= 1'b0;
  end
end

// Starts writing into the Sample memory at the input of TDFO
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    SMemEnableWr <= 1'b0;
  else if(PrevState == IDLE || !SMemMuxSel)
    SMemEnableWr <= 1'b0;
  else if((NextState == SMEM_XFER) || ((TDBWEarly != TDBWInter) && FrameParam1Update))
    SMemEnableWr <= 1'b1;
end

// Starts reading from the Sample memory at the input of TDFO
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    SMemEnableRd <= 1'b0;
  else if(PrevState == IDLE || !SMemMuxSel)
    SMemEnableRd <= 1'b0;
  else if(SMemEnableWr && ((NextState == FFT_XFERLOOP) || SMemRdAddrUpdateP))
    SMemEnableRd <= 1'b1;
end

// Update Sample memory read address
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    SMemRdAddrUpdateP <= 1'b0;
    SMemOffset        <= 8'b0;
    SMemRdAddrWait    <= 1'b0;
    SMemRdAddrWait_1t <= 1'b0;
    SMemRdAddrWaitFFT <= 1'b0;
    SMemRdToggle      <= 1'b0;
  end
  else if(PrevState == IDLE) begin
    SMemRdAddrUpdateP <= 1'b0;
    SMemOffset        <= 8'b0;
    SMemRdAddrWait    <= 1'b0;
    SMemRdAddrWait_1t <= 1'b0;
    SMemRdAddrWaitFFT <= 1'b0;
    SMemRdToggle      <= 1'b0;
  end
  else begin
    if((CurrState != WAIT_HTLTF_GIEND) && (NextState == WAIT_HTLTF_GIEND) && ((TDBWInter != TDBWEarly)))
      SMemRdAddrUpdateP <= 1'b1;
    else
      SMemRdAddrUpdateP <= 1'b0;
    
    if (SMemRdEn)
      SMemRdToggle <= ~SMemRdToggle | CfgRegForce20 | (FFTLen != 3'd6);
    else
      SMemRdToggle <= 1'b0;

    if ((FDFormatMod_1t == MODE_GF) && SMemEnableRd && (SymbolType == HTDATA)) begin // in GF 20MHz, wait FrameParam1Update
      SMemRdAddrWait    <= ~FrameParam1Update;
      SMemRdAddrWaitFFT <= ~FrameParam1Update;
    end
    else if (SMemRdAddrInter[6:0] == SMemWrAddrInter[6:0] + 7'd1)
      SMemRdAddrWaitFFT <= 1'b0;
    else if (SMemRdAddrInter[6:0] == SMemWrAddrInter[6:0] - 7'd1)
      SMemRdAddrWait <= 1'b0;

    if (SMemRdDataValid)
      SMemRdAddrWait_1t <= SMemRdAddrWait;

    if (TDBWEarly == 2'd2)       // 20->80
      SMemOffset <= CfgRegTdSyncOffset2080;
    else                         // 20->40
      SMemOffset <= CfgRegTdSyncOffset2040;
  end
end

// SMemValidCounter is a counter used to generate internal data valid during
// HT/VHT-STF where AGC will reset FE and RxDataValidInSMem will stop toggling
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    SMemValidCounter     <= 3'b0;
    SMemValidCounterSel  <= 3'b0;
    SMemValidCounterEnd  <= 3'd2;
  end
  else if(PrevState == IDLE) begin
    SMemValidCounter     <= 3'b0;
    SMemValidCounterSel  <= 3'b0;
    SMemValidCounterEnd  <= 3'd2;
  end
  else begin
    if (CurrState == FFT_XFERLOOP) begin
      if (RxDataValidInSMem && (TDBWEarly != 2'd2))
        SMemValidCounterSel <= SMemValidCounter;
      else if (!RxDataValidInSMem && (TDBWEarly == 2'd2))
        SMemValidCounterSel <= SMemValidCounter;
    end
    if (SMemValidCounter < SMemValidCounterEnd)
      SMemValidCounter <= SMemValidCounter + 3'b1;
    else
      SMemValidCounter <= 3'b0;
    if (SMemDataValid) begin
      if (TDBWEarly == 2'd0)
        SMemValidCounterEnd <= 3'd5;
      else
        SMemValidCounterEnd <= 3'd2;
    end
  end
end

assign SMemDataValid   = (SMemValidCounter == SMemValidCounterSel) ? 1'b1 : 1'b0;

assign SMemWrDataValid = (CurrState == AGCUNLOCK_WAIT || CurrState == AGCRELOCK_WAIT || CurrState == RESET_DC_UNLOCK) ? 
                         ((TDBWEarly == 2'd2) ? ~SMemDataValid : SMemDataValid) : RxDataValidInSMem;

assign SMemAlmostEmpty = (SMemWrAddrInter[5:0] == SMemRdNextAddr[5:0]) & SMemRdEn;

assign SMemRdDataValid = (!SMemAlmostEmpty && !SMemRdAddrWait) ? SMemRdToggle : (SMemWrDataValid & ~SMemRdAddrWaitFFT);

assign SMemWrAddr = SMemWrAddrInter[5:0];
assign SMemRdAddr = SMemRdAddrInter[5:0];

// Next sample memory write address
assign SMemWrNextAddr  = SMemWrAddrInter + 7'd1;

// Sample memory write address
always @(posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst==1'b0)
     SMemWrAddrInter <= 7'd0;
  else if(!SMemMuxSel)
     SMemWrAddrInter <= 7'd0;
  else if(SMemWrDataValid)
     SMemWrAddrInter <= SMemWrNextAddr;
end

// Next sample memory read address
assign SMemRdNextAddr = SMemRdAddrInter + 7'd1;

// Sample memory read address
always @(posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst==1'b0)
     SMemRdAddrInter <= 7'd0;
  else if(!SMemMuxSel)
     SMemRdAddrInter <= 7'd0;
  else if(SMemRdAddrUpdateP)
     SMemRdAddrInter <= SMemRdAddrAdjust;
  else if(SMemRdDataValid && SMemEnableRd && !SMemRdAddrWait)
     SMemRdAddrInter <= SMemRdNextAddr;
end

assign SMemRdEn = SMemMuxSel & SMemEnableRd;

// Update SMemRdAddr on SMemRdAddrUpdateP
assign SMemRdAddrAdjust = SMemWrAddrInter - $signed(SMemOffset[6:0]);

// HTSTFNoUnlock is in HT-STF during AGC gain re-estimate
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    HTSTFNoUnlock <= 1'b0;
  else if((PrevState == IDLE) || (NextState == WAIT_HTLTF_GIEND))
    HTSTFNoUnlock <= 1'b0;
  else if(CurrState == START_HTSTF)
    HTSTFNoUnlock <= 1'b1;
end

// NDPGFNoUnlock is in HT-LTF for GF frames in case of NDP 
// in order to avoid AGC abort from Main FSM
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    NDPGFNoUnlock <= 1'b0;
  else if(PrevState == IDLE)
    NDPGFNoUnlock <= 1'b0;
  else if((FDFormatMod_1t == MODE_GF) && (TDSymbolCounter == 16'd3 || TDSymbolCounter == 16'd4))
    NDPGFNoUnlock <= 1'b1;
  else
    NDPGFNoUnlock <= 1'b0;
end

// AGC trigger during Hx-STF for inband power estimation
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst==1'b0)
    AGCStartP <= 1'b0;
  else if ((CurrState == FFT_STARTLOOP) && !TDLastSymbol && 
           (((TDSymbolCounter == 16'd3) && ((FDFormatMod_1t == MODE_MM)  ||   // For HT-MM
                                            (FDFormatMod_1t == MODE_LEG) ||   // For VHT, in advance we are in legacy/duplicate
                                            (FDFormatMod_1t == MODE_DUP)))    // before Q-BPSK differenciation
           ||
            ((TDSymbolCounter == 16'd4) && ((FDFormatMod_1t == MODE_HE_SU) || // For HE_SU and HE_TB
                                            (FDFormatMod_1t == MODE_HE_TB))) 
           ||
            ((TDSymbolCounter == 16'd6) && (FDFormatMod_1t == MODE_HE_ER))    // For HE_ER
           ||
            ((TDSymbolCounter == (16'd4 + NheSigB)) && (FDFormatMod_1t == MODE_HE_MU) && FrameParam7Update))) // For HE_MU
    AGCStartP <= 1'b1;                                         
  else
    AGCStartP <= 1'b0;
end

// AGC gain update trigger during Hx-STF for HT-MM , VHT and HE
// Send also trigger in case of GF only for AGC FSM (without AGCStartP)
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if(nPhyRst==1'b0)
    AGCGainUpdateP <= 1'b0;
  else if(PrevState == IDLE)
    AGCGainUpdateP <= 1'b0;
  else if ((FormatModeHTSTF && (FDFormatMod_1t != MODE_HE_MU)) || (FDFormatMod_1t == MODE_GF))
    AGCGainUpdateP <= FrameParam1Update;
  else if ((PrevState == AGCUNLOCK_WAIT) && (SampleCounter == INBANDPOW_TIMEOUT)) // For HE_MU only
    AGCGainUpdateP <= 1'b1;
  else
    AGCGainUpdateP <= 1'b0;
end

// End of FFT address according to frame bandwidth 
assign SymbolEndAddr = FFTEndAddr[9:0] - {2'b0,TdCycRotVal};

// Adjust FFT window for short GI frames else set to 0 for other GI frames
assign TdAdjustShortGI = ((FrameParamGIType == 2'd0) && FrameParam1Update) ? 
                         ((TDBW20) ? {{5{CfgRegTdAdjust20ShortGI[4]}}, CfgRegTdAdjust20ShortGI}  :
                          (TDBW40) ? {{5{CfgRegTdAdjust40ShortGI[4]}}, CfgRegTdAdjust40ShortGI}  :
                                     {{5{CfgRegTdAdjust80ShortGI[4]}}, CfgRegTdAdjust80ShortGI}) : 10'd0;

// Register end of HT-STF wait address
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    HTSTFEndWaitAddr <= 10'd57;
  else if (CurrState == LTF)
    HTSTFEndWaitAddr <= SymbolEndAddr[9:0] - {2'b0,CfgRegTdDcHtStfMargin};
end

// SampleCounterEnd for AGC lock time-out and packet extension
assign SampleCounterEnd = (PrevState == WAIT_PE_END)     ? ((TDBW20) ? {2'b0,PECounterEnd}       :
                                                            (TDBW40) ? {1'b0,PECounterEnd,1'b0}  :
                                                                            {PECounterEnd,2'b0}) :
                          (FDFormatMod_1t == MODE_HE_TB) ? {AGCHTSTF_TIMEOUT,1'b0} : {1'b0,AGCHTSTF_TIMEOUT};

// PECounterEnd for packet extension
assign PECounterEnd = (FrameParamTPE == 3'd0) ? 9'd0   : // 0 us
                      (FrameParamTPE == 3'd1) ? 9'd80  : // 4 us
                      (FrameParamTPE == 3'd2) ? 9'd160 : // 8 us
                      (FrameParamTPE == 3'd3) ? 9'd240 : // 12 us
                                                9'd320;  // 16 us

// Assignment of Previous State to detect rx abort
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    PrevState <= IDLE;
  else
    PrevState <= CurrState;
end

// Assignment of Next State to the Current State register
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    CurrState <= IDLE;
  else if (!RxTDEnable)
    CurrState <= IDLE;
  else
    CurrState <= NextState;
end

// The following combinational block decides the next state 
// based on the Current State and the present inputs. 
// It also generates some of the control outputs.
always@(*)
begin
case (CurrState)
   // Idle state : wait for start from Main TX/RX FSM
   // On assertion of RX TD On, jump to STF state 
   IDLE       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     if(RxTDEnable)
       NextState = STF;
     else
       NextState = IDLE;
   end  // IDLE

   // STF state : start freq offset estimation
   // Start Coarse offset estimation
   // Store Bandwidth, Frequency shift from AGC
   // Init SampleCounter
   STF       : begin
     CoarseFOEn        = 1'b1;
     TickSymbolFFT     = 1'b0;

     if(PlatFallTimerEnd)
       NextState = RXEND;
     else if(PlatFall)
       NextState = LTF;
     else
       NextState = STF;
   end //STF

   // LTF state
   // Store coarse frequency offset estimate
   // Start TBE
   // Increment SampleCounter
   // Check TBECount
   LTF       : begin
     CoarseFOEn        = 1'b1;
     TickSymbolFFT     = 1'b0;

//      if(SampleCounter == ({1'b0,SymbolLengthLTF} + {2'b0,SymbolLengthLTF[8:1]} + {2'b0,TBECount} + {9'b0,CfgRegForce20}))
     if(SampleCounter == ({2'b0,SymbolLengthLTF} + {3'b0,TBECount} + {3'b0,CfgRegTdSyncOffset20}))
       NextState = WAIT_GIEND1;
     else
       NextState = LTF;
   end //LTF

   // In this state, we wait for the GI interval and then proceed with FFT write
   WAIT_GIEND1 : begin
     CoarseFOEn        = 1'b1;
     if(CmpLTFDoneP)
        TickSymbolFFT = 1'b1;
     else 
        TickSymbolFFT = 1'b0;

     if((SampleCounter==({2'b0,GILength}-11'd1)) && RxDataValidIn)
        NextState = SMEM_XFER;
     else
        NextState = WAIT_GIEND1;
   end //WAIT_GIEND1

   // This state is used for transferring the buffered data 
   // stored in the Sample Memory into the TDFO.
   SMEM_XFER       : begin
     CoarseFOEn        = 1'b1;
     if(CmpLTFDoneP)
        TickSymbolFFT = 1'b1;
     else 
        TickSymbolFFT = 1'b0;
     
     if ((CmpLTFDoneP || CmpLTFDoneFlag) && RxDataValidIn)
        NextState = FFT_XFERLOOP;
     else 
        NextState = SMEM_XFER;
   end

   // FFT Transfer for symbols after LTF
   // enable DC offset comp
   // Enable FO compensation in coarse+fine mode.
   // Stop Time domain at the end of the frame
   FFT_XFERLOOP       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     if(TDStopFFT)
       NextState = RXEND;        // NDP case for GF Nss=1 / VHT
     else if(FFTMemWrEnMux && (FFTMemWrAddr == SymbolEndAddr[9:0])) begin
       if (TDLastSymbol) begin
         if (FDFormatMod_1t <= MODE_VHT)
           NextState = RXEND;        // End of frame
         else
           NextState = WAIT_PE_END;  // Wait HE packet extension
       end
       else begin
         NextState = FFT_STARTLOOP;
         if ((SymbolType != HESTF) && !MidambleSkipFFT) // Generate TickSymbolFFT one cc in advance for 80MHz BW
           TickSymbolFFT   = 1'b1;                      // still compatible with 20/40 MHz BW
       end
     end
     else if (!CSFlag && (((TDSymbolCounter == 16'd4) && (FDFormatMod_1t == MODE_MM    || FDFormatMod_1t == MODE_VHT))   ||
                          ((TDSymbolCounter == 16'd5) && (FDFormatMod_1t == MODE_HE_SU || FDFormatMod_1t == MODE_HE_TB)) ||
                          ((TDSymbolCounter == 16'd6) && (FDFormatMod_1t == MODE_HE_TB))                                 ||
                          ((TDSymbolCounter == 16'd7) && (FDFormatMod_1t == MODE_HE_ER))                                 ||
                          ((TDSymbolCounter == (16'd5 + NheSigB)) && (FDFormatMod_1t == MODE_HE_MU) && FrameParam7Update)))
       NextState = START_HTSTF;  // STF symbol in HT-MM , VHT and HE frames
     else
       NextState = FFT_XFERLOOP;
   end //FFT_XFERLOOP

   // Start FFT processing
   // Increment symbol count
   FFT_STARTLOOP       : begin
     CoarseFOEn        = 1'b0;
//      if ((SymbolType != HESTF) && !MidambleSkipFFT)
//        TickSymbolFFT   = 1'b1;
//      else
       TickSymbolFFT   = 1'b0;
     NextState         = WAIT_GIEND2;
   end //FFT_STARTLOOP

   // Wait for symbol G.I end
   WAIT_GIEND2       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;
     
     if((FFTMemWrAddr==({1'b0,GILength}-TdAdjustSyncSkip)) && FFTMemWrEnMux)
       NextState = FFT_XFERLOOP;
     else
       NextState = WAIT_GIEND2;
   end //WAIT_GIEND2

   // Recompute sample counter at the beginning of HT-STF, rerun DC estimation
   START_HTSTF       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     if (SMemMuxSel || RxDataValidIn)
       NextState = AGCUNLOCK_WAIT;
     else
       NextState = START_HTSTF;
   end //START_HTSTF

   // Wait for AGC Unlock time-out
   AGCUNLOCK_WAIT       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     // If gain was not updated, re-do a DC offset estimation,
     // else wait for end of HT-STF
     if(FFTMemWrEnMux && (FFTMemWrAddr >= (HTSTFEndWaitAddr-10'd2)))
       NextState = WAIT_HTSTF_END; // Margin for TDDCDataPhase generation accuracy
     else if ((SampleCounter == SampleCounterEnd) && !AGCOFDMHTSTF)
       NextState = WAIT_HTSTF_END; // AGC lock time-out
     else if (AGCOFDMHTSTF)
       NextState = AGCRELOCK_WAIT;
     else
       NextState = AGCUNLOCK_WAIT;
   end //AGCUNLOCK_WAIT

   // Wait for AGC Lock
   AGCRELOCK_WAIT       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     if(FFTMemWrEnMux && (FFTMemWrAddr == (HTSTFEndWaitAddr-10'd3)))
       NextState = RESET_DC_UNLOCK;
     // Gain was updated then re-do a DC offset calibration.
     else if (!AGCOFDMHTSTF)
       NextState = RESET_DC_UNLOCK;
     else
       NextState = AGCRELOCK_WAIT;
   end //AGCRELOCK_WAIT

   // Restart DC estimation after AGC re-lock
   RESET_DC_UNLOCK       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     NextState = WAIT_HTSTF_END;
   end //RESET_DC_UNLOCK

   // Wait for end of HT-STF
   WAIT_HTSTF_END       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     if(FFTMemWrEnMux && (FFTMemWrAddr == SymbolEndAddr[9:0])) begin
       if (LastHESTF)
         NextState = WAIT_HTLTF_GIEND;
       else
         NextState = FFT_STARTLOOP;
     end
     else
       NextState = WAIT_HTSTF_END;
   end //WAIT_HTSTF_END

   // Wait for GI of the HT-LTF
   WAIT_HTLTF_GIEND       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = 1'b0;

     if((FFTMemWrAddr==({1'b0,GILength}-10'd1+$signed(TdAdjustShortGI))) && FFTMemWrEnIn) // For short GI : shift from register
       NextState = FFT_XFERLOOP;
     else
       NextState = WAIT_HTLTF_GIEND;
   end //WAIT_HTLTF_GIEND

   // Wait end of HE packet extension
   // Increment SampleCounter
   WAIT_PE_END       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = TDBufEmpty & ~FFTState & ~NoMoreFFT;

     if ((SampleCounter >= SampleCounterEnd) && FrameParam8Update)
       NextState = RXEND;
     else
       NextState = WAIT_PE_END;
   end //WAIT_PE_END

   // End of time domain FSM : send RxTDDone
   default       : begin
     CoarseFOEn        = 1'b0;
     TickSymbolFFT     = TDBufEmpty & ~RxTDDone & ~NoMoreFFT & ~PlatTimeout & ~TDStopFFT;

     if (!RxTDEnable)
       NextState = IDLE;
     else
       NextState = RXEND;
   end //RXEND

 endcase
end // always


// Outputs assignment
assign TDBandwidth = TDBWEarly;


// Time domain to AGC, add output FFs for FF to FF resynchronization
always @ (posedge PhyClk or negedge nPhyRst) 
begin
  if (nPhyRst == 1'b0) begin
    AGCBWSup20        <= TDBW_DEFAULT[0]; // Default 40MHz
    AGCBWEq20         <= ~TDBW_DEFAULT[0];
    AGCBWEq40         <= TDBW_DEFAULT[0]; // Default 40MHz
    AGCBWEq80         <= 1'b0;
    AGCLegFrame       <= 1'b0;
    AGCVHTFrame       <= 1'b0;
    AGCNoACIMargin    <= 1'b0;
  end
  else begin
    AGCBWSup20        <= (TDBWEarly != 2'd0);
    AGCBWEq20         <= (TDBWEarly == 2'd0);
    AGCBWEq40         <= (TDBWEarly == 2'd1);
    AGCBWEq80         <= (TDBWEarly == 2'd2);
    AGCLegFrame       <= (FDFormatMod_1t == MODE_LEG) | (FDFormatMod_1t == MODE_DUP);
    AGCVHTFrame       <= (FDFormatMod_1t >= MODE_VHT); // Set also HE formats for MCS definition (MCS 8 to 11)
    AGCNoACIMargin    <= (FDFormatMod_1t == MODE_HE_MU) |
                         (FrameParam1Update & (FrameParamVHTNDP // Frame is VHT NDP
`ifdef RW_MUMIMO_RX_EN
                         | FrameParamMuMIMO                     // Frame is VHT Mu-MIMO
`endif // RW_MUMIMO_RX_EN
                         ));
  end
end


// Assigning Strings to States so that it would be easy to debug
`ifdef RW_SIMU_ON

  // pragma coverage block = off, expr = off, toggle = off

reg [25*8:0]           TDCurrStateStr;
reg [25*8:0]           TDNextStateStr;
reg [25*8:0]           TDPrevStateStr;
reg [25*8:0]           SymbolTypeStr;
reg                    assert_RXTD_SMEM_overwrite;

always@(*)
begin
  case (CurrState)
    IDLE                : TDCurrStateStr = {"IDLE"};
    STF                 : TDCurrStateStr = {"STF"};
    LTF                 : TDCurrStateStr = {"LTF"};
    WAIT_GIEND1         : TDCurrStateStr = {"WAIT_GIEND1"};
    FFT_XFERLOOP        : TDCurrStateStr = {"FFT_XFERLOOP"};
    SMEM_XFER           : TDCurrStateStr = {"SMEM_XFER"};
    FFT_STARTLOOP       : TDCurrStateStr = {"FFT_STARTLOOP"};
    WAIT_GIEND2         : TDCurrStateStr = {"WAIT_GIEND2"};
    START_HTSTF         : TDCurrStateStr = {"START_HTSTF"};
    AGCUNLOCK_WAIT      : TDCurrStateStr = {"AGCUNLOCK_WAIT"};
    AGCRELOCK_WAIT      : TDCurrStateStr = {"AGCRELOCK_WAIT"};
    RESET_DC_UNLOCK     : TDCurrStateStr = {"RESET_DC_UNLOCK"};
    WAIT_HTSTF_END      : TDCurrStateStr = {"WAIT_HTSTF_END"};
    WAIT_HTLTF_GIEND    : TDCurrStateStr = {"WAIT_HTLTF_GIEND"};
    WAIT_PE_END         : TDCurrStateStr = {"WAIT_PE_END"};
    RXEND               : TDCurrStateStr = {"RXEND"};
    default             : TDCurrStateStr = {"UNKNOWN"};
  endcase
end
always@(*)
begin
  case (NextState)
    IDLE                : TDNextStateStr = {"IDLE"};
    STF                 : TDNextStateStr = {"STF"};
    LTF                 : TDNextStateStr = {"LTF"};
    WAIT_GIEND1         : TDNextStateStr = {"WAIT_GIEND1"};
    FFT_XFERLOOP        : TDNextStateStr = {"FFT_XFERLOOP"};
    SMEM_XFER           : TDNextStateStr = {"SMEM_XFER"};
    FFT_STARTLOOP       : TDNextStateStr = {"FFT_STARTLOOP"};
    WAIT_GIEND2         : TDNextStateStr = {"WAIT_GIEND2"};
    START_HTSTF         : TDNextStateStr = {"START_HTSTF"};
    AGCUNLOCK_WAIT      : TDNextStateStr = {"AGCUNLOCK_WAIT"};
    AGCRELOCK_WAIT      : TDNextStateStr = {"AGCRELOCK_WAIT"};
    RESET_DC_UNLOCK     : TDNextStateStr = {"RESET_DC_UNLOCK"};
    WAIT_HTSTF_END      : TDNextStateStr = {"WAIT_HTSTF_END"};
    WAIT_HTLTF_GIEND    : TDNextStateStr = {"WAIT_HTLTF_GIEND"};
    WAIT_PE_END         : TDNextStateStr = {"WAIT_PE_END"};
    RXEND               : TDNextStateStr = {"RXEND"};
    default             : TDNextStateStr = {"UNKNOWN"};
  endcase
end
always@(*)
begin
  case (PrevState)
    IDLE                : TDPrevStateStr = {"IDLE"};
    STF                 : TDPrevStateStr = {"STF"};
    LTF                 : TDPrevStateStr = {"LTF"};
    WAIT_GIEND1         : TDPrevStateStr = {"WAIT_GIEND1"};
    FFT_XFERLOOP        : TDPrevStateStr = {"FFT_XFERLOOP"};
    SMEM_XFER           : TDPrevStateStr = {"SMEM_XFER"};
    FFT_STARTLOOP       : TDPrevStateStr = {"FFT_STARTLOOP"};
    WAIT_GIEND2         : TDPrevStateStr = {"WAIT_GIEND2"};
    START_HTSTF         : TDPrevStateStr = {"START_HTSTF"};
    AGCUNLOCK_WAIT      : TDPrevStateStr = {"AGCUNLOCK_WAIT"};
    AGCRELOCK_WAIT      : TDPrevStateStr = {"AGCRELOCK_WAIT"};
    RESET_DC_UNLOCK     : TDPrevStateStr = {"RESET_DC_UNLOCK"};
    WAIT_HTSTF_END      : TDPrevStateStr = {"WAIT_HTSTF_END"};
    WAIT_HTLTF_GIEND    : TDPrevStateStr = {"WAIT_HTLTF_GIEND"};
    WAIT_PE_END         : TDPrevStateStr = {"WAIT_PE_END"};
    RXEND               : TDPrevStateStr = {"RXEND"};
    default             : TDPrevStateStr = {"UNKNOWN"};
  endcase
end
always@(*)
begin
  case (SymbolType)
    LSIG    : SymbolTypeStr = {"LSIG"};
    HTSIG   : SymbolTypeStr = {"HTSIG"};
    HTSTF   : SymbolTypeStr = {"HTSTF"};
    HTLTF   : SymbolTypeStr = {"HTLTF"};
    LDATA   : SymbolTypeStr = {"LDATA"};
    HTDATA  : SymbolTypeStr = {"HTDATA"};
    VHTSIGA : SymbolTypeStr = {"VHTSIGA"};
    VHTSTF  : SymbolTypeStr = {"VHTSTF"};
    VHTLTF  : SymbolTypeStr = {"VHTLTF"};
    VHTSIGB : SymbolTypeStr = {"VHTSIGB"};
    VHTDATA : SymbolTypeStr = {"VHTDATA"};
    HESIGA  : SymbolTypeStr = {"HESIGA"};
    HESTF   : SymbolTypeStr = {"HESTF"};
    HELTF   : SymbolTypeStr = {"HELTF"};
    HESIGB  : SymbolTypeStr = {"HESIGB"};
    HEDATA  : SymbolTypeStr = {"HEDATA"};
    default : SymbolTypeStr = {"UNKNOWN"};
  endcase
end

  wire   SMemFull;
  assign SMemFull = (SMemWrAddrInter == {~SMemRdAddrInter[6],SMemRdAddrInter[5:0]}) & SMemWrDataValid;

  always @(posedge PhyClk)
  begin
    if((SMemWrAddrInter[6:0] == SMemRdAddrInter[6:0]) &&
       SMemRdEn && SMemWrDataValid && (CurrState != WAIT_HTLTF_GIEND)) begin
      $display("ASSERT:%m:assert_RXTD_SMEM_overwrite: SMEM overwrite situation detected !");
      assert_RXTD_SMEM_overwrite <= 1'b1;
    end
  end

  // pragma coverage block = on, expr = on, toggle = on

`endif

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
