////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandebu $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 28383 $
// $Date: 2016-10-03 19:05:05 +0200 (Mon, 03 Oct 2016) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Complex Multiplier used in TB Estimate Block             
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                        
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TBE/verilog/rtl/CorrMult.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

`default_nettype none

module CorrMult #(parameter       RX_IN_WIDTH = 5,
                  parameter [1:0] RECN        = 2'b0, //Real part of LTF
                  parameter [1:0] IMCN        = 2'b0  //Imag part of LTF

                 )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                                   PhyClk, //PHY Clock
            input   wire                                   nPhyRst,//PHY Reset
            //Control
            input   wire                                   ComputeOn, //Indication from Controller to start Computation 
                                                                      //Used as Data Enable for FFs
            //Data
            input   wire    signed    [RX_IN_WIDTH-1:0]    RxI, //Real Component of Rx Samples
            input   wire    signed    [RX_IN_WIDTH-1:0]    RxQ, //Imag Component of Rx Samples

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   reg   signed    [RX_IN_WIDTH+1:0]     ICorr,
            output   reg   signed    [RX_IN_WIDTH+1:0]     QCorr
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires, Registers Declarations
//////////////////////////////////////////////////////////////////////////////
wire    signed    [RX_IN_WIDTH:0]    PT1;//Product Term 1
wire    signed    [RX_IN_WIDTH:0]    PT2;//Product Term 2
wire    signed    [RX_IN_WIDTH:0]    PT3;//Product Term 3
wire    signed    [RX_IN_WIDTH:0]    PT4;//Product Term 4

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This block multiplies the 5 bit complex number (RxI + jRxQ) with a
//3 bit complex number Cn (ReCn + jImCn). Both the complex numbers are in
//2's complement format. The output of this block is given out as ICorr
//(real part) and QCorr (imaginary part).
//The product of RxI + jRxQ and ReCn + jImCn can be represented as
//(RxI*ReCn - RxQ*ImCn) + j(RxI*ImCn + RxQ*ReCn). The terms in the first
//bracket correspond to ICorr and the terms in the second bracket give
//QCorr. Each CorrMux is a specialized real multiplier generating one of
//four product terms.

//Generating the four product terms using four CorrMux modules

//CorrMux1 gives RxI*ReCn -> indicated as PT1
CorrMux     #(.RX_IN_WIDTH(RX_IN_WIDTH),
              .MUXCN(RECN))
  U_CorrMux1 (.A(RxI), .Y(PT1));

//CorrMux2 gives RxQ*ImCn -> indicated as PT2
CorrMux     #(.RX_IN_WIDTH(RX_IN_WIDTH),
              .MUXCN(IMCN))
  U_CorrMux2 (.A(RxQ), .Y(PT2));

//CorrMux3 gives RxI*ImCn -> indicated as PT3
CorrMux     #(.RX_IN_WIDTH(RX_IN_WIDTH),
              .MUXCN(IMCN))
  U_CorrMux3 (.A(RxI), .Y(PT3));

//CorrMux4 gives RxQ*ReCn -> indicated as PT4
CorrMux     #(.RX_IN_WIDTH(RX_IN_WIDTH),
              .MUXCN(RECN))
  U_CorrMux4 (.A(RxQ), .Y(PT4));


//Computing the Sum Terms and Registering the Outputs
always @ (posedge PhyClk or negedge nPhyRst)
   begin: Output_Block
      if (nPhyRst == 1'b0) begin
         ICorr <= $signed({{RX_IN_WIDTH+2}{1'b0}});
         QCorr <= $signed({{RX_IN_WIDTH+2}{1'b0}});
      end
      else if (ComputeOn == 1'b1) begin
         ICorr <= PT1 - PT2; //RxI*ReCn - RxQ*ImCn
         QCorr <= PT3 + PT4; //RxI*ImCn + RxQ*ReCn
      end
   end //Output_Block

endmodule //CorrMult

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
