////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 36904 $
// $Date: 2019-01-03 17:15:54 +0100 (Thu, 03 Jan 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Generates the Cross Correlation Value for 1 Rx Chain       
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TBE/verilog/rtl/CrossCorr.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

`default_nettype none

module CrossCorr #(parameter SUM_WIDTH      = 11,
                   parameter RX_IN_WIDTH    = 10,
                   parameter ROUND_WIDTH_CC = 13,
                   parameter CORRVAL_WIDTH  = 11
                 )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input  wire                            PhyClk, //PHY Clock
            input  wire                            nPhyRst, //PHY Reset
            //Control
            input  wire                            ComputeOn, //Indication from Controller to start Corrval Computation
            input  wire                            RxDataValidP,//Rx Samples Valid indication from Front End Blocks
            input  wire                            ISumInValidP,
            input  wire                            ISumOutValidP,
            input  wire                            CCEnable, //Signal to start storing incoming RxI/RxQ
            //Data
            input  wire signed [RX_IN_WIDTH-1:0]   RxI, //Inphase Component of Rx signal
            input  wire signed [RX_IN_WIDTH-1:0]   RxQ, //Quadrature Comp. of Rx signal

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            //Correlation
            output wire        [CORRVAL_WIDTH-1:0] CorrVal,
            //Delayed data to TDFO
            output wire signed [RX_IN_WIDTH-1:0]   RxI16D,
            output wire signed [RX_IN_WIDTH-1:0]   RxQ16D,
            output wire signed [RX_IN_WIDTH-1:0]   RxI64D,
            output wire signed [RX_IN_WIDTH-1:0]   RxQ64D
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam signed   [1:0]    CONST_ZERO    = 2'sd0;  //  0
localparam signed   [1:0]    CONST_N1      = -2'sd1; // -1
localparam signed   [1:0]    CONST_1       = 2'sd1;  //  1

//Prestore Cn Values
//These values are taken from the algorithm section of the TBE DD.
//The values are first reversed in order, then conjugated, and stored here.

//Real Cn Values
localparam signed   [127:0]  RECN          =  {CONST_ZERO,CONST_1,CONST_N1,CONST_ZERO,
                                              CONST_1,CONST_N1,CONST_N1,CONST_ZERO,
                                              CONST_N1,CONST_N1,CONST_1,CONST_1,
                                              CONST_N1,CONST_N1,CONST_ZERO,CONST_1,
                                              CONST_1,CONST_ZERO,CONST_1,CONST_ZERO,
                                              CONST_N1,CONST_ZERO,CONST_1,CONST_1,
                                              CONST_ZERO,CONST_N1,CONST_1,CONST_ZERO,
                                              CONST_1,CONST_ZERO,CONST_ZERO,CONST_1,
                                              CONST_ZERO,CONST_ZERO,CONST_1,CONST_ZERO,
                                              CONST_1,CONST_N1,CONST_ZERO,CONST_1,
                                              CONST_1,CONST_ZERO,CONST_N1,CONST_ZERO,
                                              CONST_1,CONST_ZERO,CONST_1,CONST_1,
                                              CONST_ZERO,CONST_N1,CONST_N1,CONST_1,
                                              CONST_1,CONST_N1,CONST_N1,CONST_ZERO,
                                              CONST_N1,CONST_N1,CONST_1,CONST_ZERO,
                                              CONST_N1,CONST_1,CONST_ZERO,CONST_N1};

//Imaginary Cn Values
localparam signed   [127:0]  IMCN          =  {CONST_1,CONST_1,CONST_1,CONST_1,
                                              CONST_N1,CONST_ZERO,CONST_ZERO,CONST_N1,
                                              CONST_ZERO,CONST_1,CONST_ZERO,CONST_1,
                                              CONST_1,CONST_ZERO,CONST_1,CONST_N1,
                                              CONST_ZERO,CONST_1,CONST_ZERO,CONST_N1,
                                              CONST_N1,CONST_N1,CONST_ZERO,CONST_ZERO,
                                              CONST_N1,CONST_N1,CONST_N1,CONST_ZERO,
                                              CONST_1,CONST_N1,CONST_N1,CONST_ZERO,
                                              CONST_1,CONST_1,CONST_N1,CONST_ZERO,
                                              CONST_1,CONST_1, CONST_1,CONST_ZERO,
                                              CONST_ZERO,CONST_1,CONST_1,CONST_1,
                                              CONST_ZERO,CONST_N1,CONST_ZERO,CONST_1,
                                              CONST_N1,CONST_ZERO,CONST_N1,CONST_N1,
                                              CONST_ZERO,CONST_N1,CONST_ZERO,CONST_1,
                                              CONST_ZERO,CONST_ZERO,CONST_1,CONST_N1,
                                              CONST_N1,CONST_N1,CONST_N1,CONST_ZERO};

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires, Registers & Var declarations
//////////////////////////////////////////////////////////////////////////////
reg     signed    [RX_IN_WIDTH-1:0]    RxIStore[63:0];
reg     signed    [RX_IN_WIDTH-1:0]    RxQStore[63:0];
reg                                    CCEnableD;
reg     signed    [SUM_WIDTH-1:0]      ISum; //Real Sum

wire    signed    [RX_IN_WIDTH+1:0]    ICorr[63:0];
wire    signed    [RX_IN_WIDTH+1:0]    QCorr[63:0];
wire    signed    [SUM_WIDTH-1:0]      Sum; //Output from CorrSum
wire    signed    [SUM_WIDTH-1:0]      QSum; //Im Sum
wire    signed    [RX_IN_WIDTH+1:0]    ln_tab[63:0]; //Input to CorrSum

genvar i,k,o,p,q,r;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//The main components of the CrossCorr Block are:
//1. Two 5 bit wide register arrays of depth 64 to store the
//   latest 64 RxI / RxQ samples
//2. 64 instances of CorrMult block. CorrMult is a complex multiplier used
//   for multiplication of Cn andthe RxI / RxQ samples.
//3. 2 instances of CorrSum block. CorrSum is a pipeline adder. One instance
//   adds the 64 Real outputs from the 64 CorrMult blocks, and the other
//   one adds the imaginary outputs from the 64 CorrMult blocks.
//4. ModApprox block, which calculates the approximate value of the
//   modulus of a complex number.
//5. Other misc blocks like Round, Sat_Signed, and Sat_Signed_Sym

//Pipeline Buffer for storing RxI and RxQ
//The first register stores the incoming sample
always @ (posedge PhyClk or negedge nPhyRst)
   begin
      if (nPhyRst == 1'b0) begin
         RxIStore[0] <= $signed({{RX_IN_WIDTH}{1'b0}});
         RxQStore[0] <= $signed({{RX_IN_WIDTH}{1'b0}});
      end
      else if (CCEnable != CCEnableD) begin
         //Reset at the start of computation for every packets
         RxIStore[0] <= $signed({{RX_IN_WIDTH}{1'b0}});
         RxQStore[0] <= $signed({{RX_IN_WIDTH}{1'b0}});
      end
      else if (RxDataValidP == 1'b1) begin
         //Store the incoming value
         RxIStore[0] <= RxI;
         RxQStore[0] <= RxQ;
      end
    end

//After that the samples are shifted through the buffer
//Stages 2 through 64
generate
   for (i = 1; i<64; i=i+1) begin: Rx_Buffer
      always @ (posedge PhyClk or negedge nPhyRst)
         begin
            if (nPhyRst == 1'b0) begin
               RxIStore[i] <= $signed({{RX_IN_WIDTH}{1'b0}});
               RxQStore[i] <= $signed({{RX_IN_WIDTH}{1'b0}});
            end
            else if (CCEnable != CCEnableD) begin
               //Reset at the start of computation for every packets
               RxIStore[i] <= $signed({{RX_IN_WIDTH}{1'b0}});
               RxQStore[i] <= $signed({{RX_IN_WIDTH}{1'b0}});
            end
            else if (RxDataValidP == 1'b1) begin
               //Take in the data from the previous stage
               RxIStore[i] <= RxIStore[i-1];
               RxQStore[i] <= RxQStore[i-1];
            end
          end
   end
endgenerate

//Delay line outputs to TDFO
assign RxI16D = RxIStore[15];
assign RxQ16D = RxQStore[15];
assign RxI64D = RxIStore[63];
assign RxQ64D = RxQStore[63];

//The Buffer outputs are tapped and connected to the CorrMult Blocks
//Taps 0 to 63
generate
   for (k = 0; k<64; k=k+1) begin: CorrMult_Instances
       CorrMult #(
                  .RX_IN_WIDTH(RX_IN_WIDTH),
                  .RECN(RECN[2*k+1:2*k]),
                  .IMCN(IMCN[2*k+1:2*k])
                 )
                  U_CorrMult(
                             //Inputs
                             .nPhyRst(nPhyRst),
                             .PhyClk(PhyClk),
                             .ComputeOn(ComputeOn),
                             .RxI(RxIStore[k]),
                             .RxQ(RxQStore[k]),
                             //Outputs
                             .ICorr(ICorr[k]),
                             .QCorr(QCorr[k])
                            );
   end
endgenerate


//Summing the 64 ICorr and 64 QCorr outputs using one CorrSum Block
//The ISumInValidP signal is used to distinguish between ICorr &
//QCorr values.

for (r = 0; r<64; r=r+1) begin: CorrSumInputs20
   assign ln_tab[r] = ISumInValidP ? ICorr[r] : QCorr[r];
end

CorrSum #(
          .RX_IN_WIDTH(RX_IN_WIDTH),
          .SUM_WIDTH(SUM_WIDTH),
          .ROUND_WIDTH(ROUND_WIDTH_CC)
         )
          U_CorrSum(
                     //Inputs
                     .nPhyRst(nPhyRst),
                     .PhyClk(PhyClk),
                     .ComputeOn(ComputeOn),
                     .ln0(ln_tab[0]),
                     .ln1(ln_tab[1]),
                     .ln2(ln_tab[2]),
                     .ln3(ln_tab[3]),
                     .ln4(ln_tab[4]),
                     .ln5(ln_tab[5]),
                     .ln6(ln_tab[6]),
                     .ln7(ln_tab[7]),
                     .ln8(ln_tab[8]),
                     .ln9(ln_tab[9]),
                     .ln10(ln_tab[10]),
                     .ln11(ln_tab[11]),
                     .ln12(ln_tab[12]),
                     .ln13(ln_tab[13]),
                     .ln14(ln_tab[14]),
                     .ln15(ln_tab[15]),
                     .ln16(ln_tab[16]),
                     .ln17(ln_tab[17]),
                     .ln18(ln_tab[18]),
                     .ln19(ln_tab[19]),
                     .ln20(ln_tab[20]),
                     .ln21(ln_tab[21]),
                     .ln22(ln_tab[22]),
                     .ln23(ln_tab[23]),
                     .ln24(ln_tab[24]),
                     .ln25(ln_tab[25]),
                     .ln26(ln_tab[26]),
                     .ln27(ln_tab[27]),
                     .ln28(ln_tab[28]),
                     .ln29(ln_tab[29]),
                     .ln30(ln_tab[30]),
                     .ln31(ln_tab[31]),
                     .ln32(ln_tab[32]),
                     .ln33(ln_tab[33]),
                     .ln34(ln_tab[34]),
                     .ln35(ln_tab[35]),
                     .ln36(ln_tab[36]),
                     .ln37(ln_tab[37]),
                     .ln38(ln_tab[38]),
                     .ln39(ln_tab[39]),
                     .ln40(ln_tab[40]),
                     .ln41(ln_tab[41]),
                     .ln42(ln_tab[42]),
                     .ln43(ln_tab[43]),
                     .ln44(ln_tab[44]),
                     .ln45(ln_tab[45]),
                     .ln46(ln_tab[46]),
                     .ln47(ln_tab[47]),
                     .ln48(ln_tab[48]),
                     .ln49(ln_tab[49]),
                     .ln50(ln_tab[50]),
                     .ln51(ln_tab[51]),
                     .ln52(ln_tab[52]),
                     .ln53(ln_tab[53]),
                     .ln54(ln_tab[54]),
                     .ln55(ln_tab[55]),
                     .ln56(ln_tab[56]),
                     .ln57(ln_tab[57]),
                     .ln58(ln_tab[58]),
                     .ln59(ln_tab[59]),
                     .ln60(ln_tab[60]),
                     .ln61(ln_tab[61]),
                     .ln62(ln_tab[62]),
                     .ln63(ln_tab[63]),
                     //Output
                     .Sum(Sum)
                     );

//Registering ISum
//This is required as ISum value will be available only for one clock.
//After that the output will represent QSum
always @ (posedge PhyClk or negedge nPhyRst)
   begin: ISum_Blk
      if (nPhyRst == 1'b0)
          ISum <= $signed({{SUM_WIDTH}{1'b0}});
      else if (CCEnable == 1'b1 && ISumOutValidP == 1'b1)
         ISum <= Sum;
end //ISum_Blk

//Assigning QSum
assign QSum = Sum;

//Finding the absolute value of (ISum +jQSum) -> CorrVal
ModApprox #(
            .SUM_WIDTH(SUM_WIDTH),
            .CORRVAL_WIDTH(CORRVAL_WIDTH)
           )
            U_ModApprox(
                        //Inputs
                        .PhyClk(PhyClk),
                        .nPhyRst(nPhyRst),
                        .ComputeOn(ComputeOn),
                        .ISum(ISum),
                        .QSum(QSum),
                        //Output
                        .CorrVal(CorrVal)
                       );


//Generate delayed version of CCEnable for resetting buffer
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CCEnable_Delay_Blk
      if (nPhyRst == 1'b0)
         CCEnableD <= 1'b0;
      else
         CCEnableD <= CCEnable;
    end //CCEnable_Delay_Blk


endmodule //CrossCorr

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
