////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39440 $
// $Date: 2019-07-15 18:26:53 +0200 (Mon, 15 Jul 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Peak Search Block of Cross Correlator      
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TBE/verilog/rtl/PeakSearch.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

`default_nettype none

module PeakSearch #(parameter CORRVAL_WIDTH = 11,
                    parameter ROUND_WIDTH_PS = 13
                   )(

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                                  PhyClk,
            input   wire                                  nPhyRst,

            //Config Registers
            input   wire      [4:0]                       PeakSearchDelta,            
            input   wire                                  PeakSearchNdlEn,    // Delay line enable
            input   wire      [2:0]                       PeakSearchNdlIndex, // Delay line selection

            //Control
            input   wire                                  ComputeOn,
            input   wire                                  PeakSearchOn,
            input   wire                                  CorrValValidP,

            //Data
            input   wire      [CORRVAL_WIDTH-1:0]         CorrVal0,
            input   wire      [CORRVAL_WIDTH-1:0]         CorrVal1,
            input   wire      [CORRVAL_WIDTH-1:0]         CorrVal2,
            input   wire      [CORRVAL_WIDTH-1:0]         CorrVal3,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output   reg                                  MaxDetP,
            output  wire      [ROUND_WIDTH_PS-1:0]        MaxCorrValSum
            );


//////////////////////////////////////////////////////////////////////////////
//  Internal Wires, Registers & Var Declarations
//////////////////////////////////////////////////////////////////////////////
wire              [CORRVAL_WIDTH+1:0]  CorrValSum;
wire              [ROUND_WIDTH_PS-1:0] CorrValSumAvgRnd;

wire                                   Compare;
wire              [CORRVAL_WIDTH+1:0]  CorrValSumSel;
wire              [CORRVAL_WIDTH+5:0]  CorrValSumAvgInt;

reg               [CORRVAL_WIDTH+1:0]  CorrValSumReg[7:0];
reg               [CORRVAL_WIDTH+5:0]  CorrValSumAvg;
reg                                    ComputeOn_1t;
reg               [ROUND_WIDTH_PS-1:0] CorrValSumAvgPS;
reg               [ROUND_WIDTH_PS-1:0] iMaxCorrValSum;

reg                                    CorrValValidP_1t;
reg                                    CorrValValidP_2t;

genvar i;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This block searches for the peak in the cross correlation value.
//The values from all the 4 Rx chains are summed up and then averaged
//over a period of 0.4 us tipically (set by PeakSearchNdlIndex).
//This average value is used by the peak detect logic to find the peak
//in the window where PeakSearchOn is asserted

//Step 1 - Sum of the CorrVal from all the RxChains (1 to 4)
assign CorrValSum = CorrVal0 + CorrVal1 + CorrVal2 + CorrVal3;

//Step 2 - Find the average on a sliding window of 0.4us duration
//This amounts to 8 values in 20 MHz case

//Shift Register
//The first register stores the latest Sum
always @ (posedge PhyClk or negedge nPhyRst)
   begin:CorrValSumReg0_Block
      if(nPhyRst == 1'b0)
         CorrValSumReg[0] <= {{(CORRVAL_WIDTH+2)}{1'b0}};
      else if (ComputeOn == 1'b0 && ComputeOn_1t == 1'b1)
         //Reset at the end of computation for every packet
         CorrValSumReg[0] <= {{(CORRVAL_WIDTH+2)}{1'b0}};
      else if (ComputeOn == 1'b1 && CorrValValidP == 1'b1)
         //Take in the new sum
         CorrValSumReg[0] <= CorrValSum;
   end //CorrValSumReg0_Block

//After that the samples are shifted through the shift register
generate
   for (i = 1; i<8; i=i+1) begin:CorrValSum_ShitReg
      always @ (posedge PhyClk or negedge nPhyRst)
         begin
            if (nPhyRst == 1'b0)
               CorrValSumReg[i] <= {{(CORRVAL_WIDTH+2)}{1'b0}};
            else if (ComputeOn == 1'b0 && ComputeOn_1t == 1'b1)
               //Reset at the end of computation for every packet
               CorrValSumReg[i] <= {{(CORRVAL_WIDTH+2)}{1'b0}};
            else if (ComputeOn == 1'b1 && CorrValValidP == 1'b1)
               //Take in the value of the previous stage
               CorrValSumReg[i] <= CorrValSumReg[i-1];
         end
   end //CorrValSum_ShitReg
endgenerate

//The last PeakSearchNdlIndex samples are accumulated
assign CorrValSumSel = (PeakSearchNdlIndex == 3'd0) ? CorrValSumReg[0] :
                       (PeakSearchNdlIndex == 3'd1) ? CorrValSumReg[1] :
                       (PeakSearchNdlIndex == 3'd2) ? CorrValSumReg[2] :
                       (PeakSearchNdlIndex == 3'd3) ? CorrValSumReg[3] :
                       (PeakSearchNdlIndex == 3'd4) ? CorrValSumReg[4] :
                       (PeakSearchNdlIndex == 3'd5) ? CorrValSumReg[5] :
                       (PeakSearchNdlIndex == 3'd6) ? CorrValSumReg[6] :
                                                      CorrValSumReg[7];

//The last 8 samples are accumulated
assign CorrValSumAvgInt = CorrValSumAvg + {4'b0,CorrValSum};

always @ (posedge PhyClk or negedge nPhyRst)
   begin: Moving_AvgBlock
      if (nPhyRst == 1'b0)
         CorrValSumAvg <= {{(CORRVAL_WIDTH+6)}{1'b0}};
      else if (ComputeOn == 1'b0 && ComputeOn_1t == 1'b1)
         //Reset at the end of computation for every packet
         CorrValSumAvg <= {{(CORRVAL_WIDTH+6)}{1'b0}};
      else if (ComputeOn == 1'b1 && CorrValValidP == 1'b1)
         CorrValSumAvg <= CorrValSumAvgInt - {4'b0,CorrValSumSel};
   end //Moving_AvgBlock

//Generate delayed version of ComputeOn for resetting Accumulator
always @ (posedge PhyClk or negedge nPhyRst)
   begin: ComputeOn_Delay_Blk
      if (nPhyRst == 1'b0)
         ComputeOn_1t <= 1'b0;
      else
         ComputeOn_1t <= ComputeOn;
    end //ComputeOn_Delay_Blk

//Generate delayed version of CorrValValidP
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrValValid_Delay_Blk
      if (nPhyRst == 1'b0)
      begin
         CorrValValidP_1t <= 1'b0;
         CorrValValidP_2t <= 1'b0;
      end
      else
      begin
         CorrValValidP_1t <= CorrValValidP;
         CorrValValidP_2t <= CorrValValidP_1t;
      end
    end //CorrValValid_Delay_Blk

//Round CorrValSumAvg to ROUND_WIDTH_PS
//Round by 3 LSBs
Round #(
        .INPUT_WIDTH(CORRVAL_WIDTH+5),
        .OUTPUT_WIDTH(ROUND_WIDTH_PS)
       ) U_ROUND_PS2(
                     .InputData($signed(CorrValSumAvg[CORRVAL_WIDTH+4:0])),
                     .RoundData(CorrValSumAvgRnd[ROUND_WIDTH_PS-1:0])
                    );

//Register before Peak Detect Logic
always @ (posedge PhyClk or negedge nPhyRst)
   begin: CorrValSumAvgPS_Block
      if (nPhyRst == 1'b0)
         CorrValSumAvgPS <= {{ROUND_WIDTH_PS}{1'b0}};
      else if (ComputeOn == 1'b0 && ComputeOn_1t == 1'b1)
         //Reset at the end of computation for every packet
         CorrValSumAvgPS <= {{ROUND_WIDTH_PS}{1'b0}};
      else if (ComputeOn && CorrValValidP_1t)
      begin
         if (PeakSearchNdlEn)
           CorrValSumAvgPS <= CorrValSumAvgRnd;
         else
           CorrValSumAvgPS <= CorrValSum;
      end
   end //CorrValSumAvgPS_Block


//Step 3 - Peak Detection
assign Compare = (CorrValSumAvgPS > (iMaxCorrValSum + {{{(ROUND_WIDTH_PS-5)}{1'b0}}, PeakSearchDelta})) ? CorrValValidP_2t : 1'b0;

//Store the incoming value in the max register if it is
//more than the current value
always @ (posedge PhyClk or negedge nPhyRst)
   begin: iMaxCorrValSum_Block
      if (nPhyRst == 1'b0)
         iMaxCorrValSum <= {{ROUND_WIDTH_PS}{1'b0}};
      else if (ComputeOn == 1'b1 && ComputeOn_1t == 1'b0)
         //Reset when ComputeOn goes high
         iMaxCorrValSum <= {{ROUND_WIDTH_PS}{1'b0}};
      else if (PeakSearchOn == 1'b1 && Compare == 1'b1)
         //Incoming value is greater than stored max
         iMaxCorrValSum <= CorrValSumAvgPS;
   end //iMaxCorrValSum_Block

assign MaxCorrValSum = iMaxCorrValSum;

//Detect Peaks by comparing incoming value with stored value
always @ (posedge PhyClk or negedge nPhyRst)
   begin: MaxDetP_Block
      if (nPhyRst == 1'b0)
         MaxDetP <= 1'b0;
      else if (ComputeOn == 1'b0 && ComputeOn_1t == 1'b1)
         //Reset at the end of computation for every packet
         MaxDetP <= 1'b0;
      else if (PeakSearchOn == 1'b1)
      begin
         //Peak Search Window Enabled
         if (Compare == 1'b1)
            //Peak Found!
            MaxDetP <= 1'b1;
         else
            //Not a Peak
            MaxDetP <= 1'b0;
      end
   end //MaxDetP_Block

endmodule //PeakSearch

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
