//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 39440 $
// $Date: 2019-07-15 18:26:53 +0200 (Mon, 15 Jul 2019) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : TBE Controller Block      
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TBE/verilog/rtl/TBECntrl.v $
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


`default_nettype none

module TBECntrl (

            ///////////////////////////////////////////////
            // Inputs
            ///////////////////////////////////////////////
            //Clock and Reset
            input   wire                            PhyClk,   // PHY Clock
            input   wire                            nPhyRst,  // PHY Reset

            //Control
            input   wire                            TBEEnable,    // Enable for TBE Block @T1
            input   wire                            RxDataValidP, // Rx Samples Valid indication from Front End Blocks

            input   wire                            MaxDetP,      // Pulse from peak search
            
            //Config Registers
            input   wire        [1:0]               RxRFChainEn,
            input   wire        [7:0]               TBECountAdjust20,
            input   wire        [6:0]               PeakSearchStartLowSNR,
            input   wire        [6:0]               PeakSearchStopLowSNR,
            input   wire        [7:0]               TBEBias,

            ///////////////////////////////////////////////
            // Outputs
            ///////////////////////////////////////////////
            output  wire        [1:0]               CCEnable,
            output  wire                            ComputeOn,
            output  reg                             PeakSearchOn,
            output  wire                            CorrValValidP,
            output  wire                            ISumInValidP,
            output  wire                            ISumOutValidP,
            output  reg                             TBECountValid,
            output  reg         [7:0]               TBECount
            );


//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
//FSM Parameters
  //Size
localparam FSM_SIZE     = 6;
  //States
localparam IDLE         = 6'b000001;
localparam BUFFR        = 6'b000010;
localparam COMPUTE      = 6'b000100;
localparam PEAKSEARCH   = 6'b001000;
localparam RESULT       = 6'b010000;
localparam WAIT_END_LTF = 6'b100000;

//Other Internal Parameters
//Hardware Latencies (in terms of PhyClk)
localparam [7:0] HW_ADJ_120MHZ = 8'd6;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
reg                                    iComputeOn;
reg               [FSM_SIZE-1:0]       CurrState;
reg               [FSM_SIZE-1:0]       NextState;
reg               [7:0]                DataValidReg;
reg               [7:0]                SampleCount;
reg                                    iComputeOnD;
reg               [6:0]                TimCnt;
reg                                    RxDataValidD1P;
reg                                    RxDataValidD2P;
reg                                    RxDataValidD3P;
reg                                    RxDataValidD4P;
reg                                    MaxDetP_1t;

//////////////////////////////////////////////////////////////////////////////
// Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
wire              [6:0]                GotoCOMPUTE;
wire              [6:0]                GotoPEAKSEARCH;
wire              [6:0]                GotoRESULT;
wire              [7:0]                SampleCountBiased;
wire                                   TBECount20Unit;

//String definition to display FSM states in simulation environment
`ifdef RW_SIMU_ON
reg        [20*8:0]                TBECSStr;
reg        [20*8:0]                TBENSStr;
`endif //RW_SIMU_ON


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//This block implements a Controller FSM for the TBE Block, and related
//Timing Controls.

//FSM with One Hot Encoding
//Sequential Block
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FSM_Seq
      if (nPhyRst == 1'b0)
         CurrState <= IDLE;
      else if (TBEEnable == 1'b0)
         //From RxCntrl FSM
         CurrState <= IDLE;
      else 
         CurrState <= NextState;
   end //FSM_Seq

//Combinatorial Block
always @ (*)
   begin: FSM_Combo
      case(CurrState)

         IDLE: begin
            //@T1 from RxCntrl FSM
            NextState = BUFFR;
         end

         BUFFR: begin
            if (TimCnt == GotoCOMPUTE)
               //@T2 - 0.5 us;
               NextState = COMPUTE;
            else
               NextState = BUFFR;
         end

         COMPUTE: begin
            if (TimCnt == GotoPEAKSEARCH)
               //@T2; also noting hardware latencies
               NextState = PEAKSEARCH;
            else
               NextState = COMPUTE;
         end

         PEAKSEARCH: begin
            if (TimCnt == GotoRESULT) 
               //@T3; also noting hardware latencies
               NextState = RESULT;
            else
               NextState = PEAKSEARCH;
         end

         RESULT: begin
               NextState = WAIT_END_LTF;
         end

         WAIT_END_LTF: begin
               NextState = WAIT_END_LTF;
         end

         //pragma coverage off
         default: NextState = IDLE;
         //pragma coverage on

      endcase
   end //FSM_Combo

//Outputs Block
always @ (posedge PhyClk or negedge nPhyRst)
   begin: FSM_Out
      if (nPhyRst == 1'b0) begin
         iComputeOn    <= 1'b0;
         PeakSearchOn  <= 1'b0;
         TBECountValid <= 1'b0;
      end
      else begin
         case (CurrState)

            IDLE: begin
               iComputeOn    <= 1'b0;
               PeakSearchOn  <= 1'b0;
               TBECountValid <= 1'b0;
            end

            BUFFR: begin
               iComputeOn    <= 1'b0;
               PeakSearchOn  <= 1'b0;
               TBECountValid <= 1'b0;
            end

            COMPUTE: begin
               iComputeOn    <= 1'b1;
               PeakSearchOn  <= 1'b0;
               TBECountValid <= 1'b0;
            end

            PEAKSEARCH: begin
               iComputeOn    <= 1'b1;
               PeakSearchOn  <= 1'b1;
               TBECountValid <= 1'b0;
            end

            RESULT: begin
               iComputeOn    <= 1'b0;
               PeakSearchOn  <= 1'b0;
               TBECountValid <= 1'b1;
            end

            WAIT_END_LTF: begin
               iComputeOn    <= 1'b0;
               PeakSearchOn  <= 1'b0;
               TBECountValid <= 1'b0;
            end

            //pragma coverage off
            default: begin
               iComputeOn    <= 1'b0;
               PeakSearchOn  <= 1'b0;
               TBECountValid <= 1'b0;
            end
            //pragma coverage on

         endcase
      end
   end //FSM_Out

//Generating CCEnable directly from TBEEnable
assign CCEnable = ({2{TBEEnable}} & RxRFChainEn);

assign ComputeOn = iComputeOn;

//Timing Counter - Counts PhyClks when TBEEnable is asserted
//Maximum Window from T1 to T3 can be 3.2 us as per the algorithm
//So a 7 bit counter will suffice
//Also, the first increment of TimCnt is aligned to the first increment
//of SampleCnt. This ensures uniformity, irrespective of the point where
//the first RxDataValidP arrives, after TBEEnable has been asserted
always @ (posedge PhyClk or negedge nPhyRst)
   begin: TimCnt_Blk
      if (nPhyRst == 1'b0)
        TimCnt <= 7'd0;
      else if(TBEEnable == 1'b0) 
        //Init Counter
        TimCnt <= 7'd0;
      else if (RxDataValidP == 1'b1)
        TimCnt <= TimCnt + 7'b1;
   end //TimCnt_Blk

//Timing Signals for FSM
//Takes into account the HW Latencies
//Also accounts for the fact that the TBE Algorithm Implementation closes
//the peak search window 1 sample earlier than what the register specifies.
//Hence deduct 1 TimCnts
assign GotoCOMPUTE    = PeakSearchStartLowSNR - 7'd10;
assign GotoPEAKSEARCH = PeakSearchStartLowSNR + 7'd1;
assign GotoRESULT     = PeakSearchStopLowSNR  + 7'd2;

//Sample Counter - Count RxDataValidP pulses when TBEEnable is high
//Maximum Window from T1 to T3 can be 3.2 us as per the algorithm
//Allowing twice the value i.e. 6.4 us, we can have 256 samples at 40 MHz
//So an 8 bit counter will suffice
always @ (posedge PhyClk or negedge nPhyRst)
   begin: SampleCount_Blk
      if (nPhyRst == 1'b0)
        SampleCount <= 8'b0;
      else if(TBEEnable == 1'b0)
        //Counter Init to 0
        SampleCount <= 8'b0;
      else if (RxDataValidP)
        //Count RxI/Q samples
        SampleCount <= SampleCount + 8'b1;
   end //SampleCount_Blk

//Generate final value of TBECount by latching value of SampleCount
//at places where MaxDetP is asserted. The last MaxDetP corresponds
//to the correct value of TBECount.
//Also take HW latencies and bias in to account.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: TBECount_Blk
      if (nPhyRst == 1'b0)
        TBECount <= 8'b0;
      else if ((TBEEnable == 1'b1) && (MaxDetP_1t == 1'b1))
        //Max Detected
        TBECount <= SampleCountBiased;
   end //TBECount_Blk

//Counter adjustement from registers and HW latency
assign SampleCountBiased = SampleCount - TBEBias - TBECountAdjust20 + HW_ADJ_120MHZ - {7'b0,TBECount20Unit};

assign TBECount20Unit = RxDataValidD1P | RxDataValidD2P |
                        RxDataValidD3P | RxDataValidD4P;

//Generate delayed version of RxDataValidP for TBECount20Unit
always @ (posedge PhyClk or negedge nPhyRst)
   begin: RxDataValid_Delay_Blk
      if (nPhyRst == 1'b0) begin
         RxDataValidD1P <= 1'b0;
         RxDataValidD2P <= 1'b0;
         RxDataValidD3P <= 1'b0;
         RxDataValidD4P <= 1'b0;
         MaxDetP_1t     <= 1'b0;
      end
      else begin
         RxDataValidD1P <= RxDataValidP;
         RxDataValidD2P <= RxDataValidD1P;
         RxDataValidD3P <= RxDataValidD2P;
         RxDataValidD4P <= RxDataValidD3P;
         MaxDetP_1t     <= MaxDetP;
      end
    end //RxDataValid_Delay_Blk

//Generate delayed version of iComputeOn for resetting DataValidReg
always @ (posedge PhyClk or negedge nPhyRst)
   begin: iComputeOn_Delay_Blk
      if (nPhyRst == 1'b0)
         iComputeOnD <= 1'b0;
      else
         iComputeOnD <= iComputeOn;
    end //iComputeOn_Delay_Blk

//Generate DataValid for the different blocks.
always @ (posedge PhyClk or negedge nPhyRst)
   begin: DataValidReg_Block
      if(nPhyRst == 1'b0)
         DataValidReg <= 8'b0;
      else
         if(iComputeOn == 1'b0 && iComputeOnD == 1'b1)
            DataValidReg <= 8'b0;
         else if(iComputeOn == 1'b0)
            DataValidReg <= DataValidReg;
         else
            DataValidReg[7:0] <= {DataValidReg[6:0],RxDataValidP};
   end //DataValidReg_Block

//2 Delays for the inputs to the CorrSum Block (for generating ISum)
assign ISumInValidP = DataValidReg[1];

//5 Delays for the outputs of the CorrSum Block (for ISum)
assign ISumOutValidP = DataValidReg[4];

//8 Delays for the DataValid going to PeakSearch Block
assign CorrValValidP = DataValidReg[7];

//Additional Code to ease verification
//Display FSM State as a string in RTL simulation waveform
//pragma coverage off
`ifdef RW_SIMU_ON
always @ (*)
   begin: String_Blk
      case (CurrState)
         IDLE         : TBECSStr  = {"IDLE"};
         BUFFR        : TBECSStr  = {"BUFFR"};
         COMPUTE      : TBECSStr  = {"COMPUTE"};
         PEAKSEARCH   : TBECSStr  = {"PEAKSEARCH"};
         RESULT       : TBECSStr  = {"RESULT"};
         WAIT_END_LTF : TBECSStr  = {"WAIT_END_LTF"};
         default      : TBECSStr  = {"UNKNOWN"};
      endcase

      case (NextState)
         IDLE         : TBENSStr  = {"IDLE"};
         BUFFR        : TBENSStr  = {"BUFFR"};
         COMPUTE      : TBENSStr  = {"COMPUTE"};
         PEAKSEARCH   : TBENSStr  = {"PEAKSEARCH"};
         RESULT       : TBENSStr  = {"RESULT"};
         WAIT_END_LTF : TBENSStr  = {"WAIT_END_LTF"};
         default      : TBENSStr  = {"UNKNOWN"};
      endcase

   end //String_Blk
`endif //RW_SIMU_ON
//pragma coverage on

endmodule //TBECntrl

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
