//////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 34361 $
// $Date: 2018-06-22 11:57:21 +0200 (Fri, 22 Jun 2018) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Time Domain DC Offset Compensation.
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TDDCOffset/verilog/rtl/TDDCOffsetComp.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
`default_nettype none

module  TDDCOffsetComp
            (
            ///////////////////////////////////////////////
            // Clock & Reset
            ///////////////////////////////////////////////
            input  wire                     PhyClk,
            input  wire                     nPhyRst,

            ///////////////////////////////////////////////
            // Control interface
            ///////////////////////////////////////////////
            input   wire                    DCCompEn,       // Enable TD DC Offset Compensation

            ///////////////////////////////////////////////
            // DC estimation
            ///////////////////////////////////////////////
            input  wire signed    [14:0]    DCEstRe,
            input  wire signed    [14:0]    DCEstIm,
`ifdef RW_NX_DERIV_PATH1
            input  wire signed    [14:0]    DCEstRe1,
            input  wire signed    [14:0]    DCEstIm1,
`endif
            input  wire                     DCEstValid,

            ///////////////////////////////////////////////
            // Data interface
            ///////////////////////////////////////////////
            // Rx input with valid signal
            input  wire signed    [12:0]    RxDataInRe,
            input  wire signed    [12:0]    RxDataInIm,
`ifdef RW_NX_DERIV_PATH1
            input  wire signed    [12:0]    RxDataInRe1,
            input  wire signed    [12:0]    RxDataInIm1,
`endif
            input  wire                     RxDataInValid,
            //
            // Rx Output Port with valid signal
            output reg  signed    [12:0]    RxDataOutRe,
            output reg  signed    [12:0]    RxDataOutIm,
`ifdef RW_NX_DERIV_PATH1
            output reg  signed    [12:0]    RxDataOutRe1,
            output reg  signed    [12:0]    RxDataOutIm1,
`endif
            output reg                      RxDataOutValid
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declaration
//////////////////////////////////////////////////////////////////////////////
localparam signed [12:0] SIG_WIDTH_13_PARAM_0 = 13'b0;
localparam signed [14:0] SIG_WIDTH_15_PARAM_0 = 15'b0;
localparam signed [15:0] SIG_WIDTH_16_PARAM_0 = 16'b0;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
// Output of Round block.
wire        signed    [13:0]    RndCompensatedOutputRe;
wire        signed    [13:0]    RndCompensatedOutputIm;
`ifdef RW_NX_DERIV_PATH1
wire        signed    [13:0]    RndCompensatedOutputRe1;
wire        signed    [13:0]    RndCompensatedOutputIm1;
`endif
// Output of Saturation block.
wire        signed    [12:0]    SatCompensatedOutputRe;
wire        signed    [12:0]    SatCompensatedOutputIm;
`ifdef RW_NX_DERIV_PATH1
wire        signed    [12:0]    SatCompensatedOutputRe1;
wire        signed    [12:0]    SatCompensatedOutputIm1;
`endif

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declaration
//////////////////////////////////////////////////////////////////////////////

// Raw compensated output before saturation and rounding.
reg         signed    [15:0]    CompensatedOutputRe;
reg         signed    [15:0]    CompensatedOutputIm;
`ifdef RW_NX_DERIV_PATH1
reg         signed    [15:0]    CompensatedOutputRe1;
reg         signed    [15:0]    CompensatedOutputIm1;
`endif

// Delay of valid signals.
reg                             RxDataInValidD1;
reg                             RxDataInValidD2;
reg                             RxDataInValidD3;
reg         signed    [12:0]    RxDataInReD;
reg         signed    [12:0]    RxDataInImD;
`ifdef RW_NX_DERIV_PATH1
reg         signed    [12:0]    RxDataInRe1D;
reg         signed    [12:0]    RxDataInIm1D;
`endif

// Shifted Output.
reg         signed    [14:0]    ShftRxDataInRe;
reg         signed    [14:0]    ShftRxDataInIm;
`ifdef RW_NX_DERIV_PATH1
reg         signed    [14:0]    ShftRxDataInRe1;
reg         signed    [14:0]    ShftRxDataInIm1;
`endif

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// DC Compensation
// Input data shift by 2 and subtract DC Estimation value.
// Till 1st est is not calculated do shift and round operations.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    RxDataInValidD1     <= 1'b0;
    RxDataInValidD2     <= 1'b0;
    RxDataInValidD3     <= 1'b0;
    RxDataInReD         <= SIG_WIDTH_13_PARAM_0;
    RxDataInImD         <= SIG_WIDTH_13_PARAM_0;
    CompensatedOutputRe <= SIG_WIDTH_16_PARAM_0;
    CompensatedOutputIm <= SIG_WIDTH_16_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    RxDataInRe1D         <= SIG_WIDTH_13_PARAM_0;
    RxDataInIm1D         <= SIG_WIDTH_13_PARAM_0;
    CompensatedOutputRe1 <= SIG_WIDTH_16_PARAM_0;
    CompensatedOutputIm1 <= SIG_WIDTH_16_PARAM_0;
`endif
  end
  else if (!DCCompEn)begin
    RxDataInValidD1     <= 1'b0;
    RxDataInValidD2     <= 1'b0;
    RxDataInValidD3     <= 1'b0;
    RxDataInReD         <= SIG_WIDTH_13_PARAM_0;
    RxDataInImD         <= SIG_WIDTH_13_PARAM_0;
    CompensatedOutputRe <= SIG_WIDTH_16_PARAM_0;
    CompensatedOutputIm <= SIG_WIDTH_16_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    RxDataInRe1D         <= SIG_WIDTH_13_PARAM_0;
    RxDataInIm1D         <= SIG_WIDTH_13_PARAM_0;
    CompensatedOutputRe1 <= SIG_WIDTH_16_PARAM_0;
    CompensatedOutputIm1 <= SIG_WIDTH_16_PARAM_0;
`endif
  end
  else begin 
    RxDataInValidD1    <= RxDataInValid;
    RxDataInValidD2    <= RxDataInValidD1;
    RxDataInValidD3    <= RxDataInValidD2;
    RxDataInReD        <= RxDataInRe;
    RxDataInImD        <= RxDataInIm;
`ifdef RW_NX_DERIV_PATH1
    RxDataInRe1D        <= RxDataInRe1;
    RxDataInIm1D        <= RxDataInIm1;
`endif
    if (DCEstValid) begin
      CompensatedOutputRe <= {ShftRxDataInRe[14],ShftRxDataInRe} - {DCEstRe[14],DCEstRe};
      CompensatedOutputIm <= {ShftRxDataInIm[14],ShftRxDataInIm} - {DCEstIm[14],DCEstIm};
`ifdef RW_NX_DERIV_PATH1
      CompensatedOutputRe1 <= {ShftRxDataInRe1[14],ShftRxDataInRe1} - {DCEstRe1[14],DCEstRe1};
      CompensatedOutputIm1 <= {ShftRxDataInIm1[14],ShftRxDataInIm1} - {DCEstIm1[14],DCEstIm1};
`endif
    end
    else begin
      CompensatedOutputRe <= {ShftRxDataInRe[14],ShftRxDataInRe};
      CompensatedOutputIm <= {ShftRxDataInIm[14],ShftRxDataInIm};
`ifdef RW_NX_DERIV_PATH1
      CompensatedOutputRe1 <= {ShftRxDataInRe1[14],ShftRxDataInRe1};
      CompensatedOutputIm1 <= {ShftRxDataInIm1[14],ShftRxDataInIm1};
`endif
    end
  end
end

// This part needs changes if downsampling frequency changes.
// Shifting is done in clock edge which can be done in 
// combinationatorial process to save one clock.This wont affect
// timing requirements.
// Shift Input value by 2.
// Register DC Estmation value.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    ShftRxDataInRe <= SIG_WIDTH_15_PARAM_0;
    ShftRxDataInIm <= SIG_WIDTH_15_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    ShftRxDataInRe1 <= SIG_WIDTH_15_PARAM_0;
    ShftRxDataInIm1 <= SIG_WIDTH_15_PARAM_0;
`endif
  end
  else if (!DCCompEn)begin
    ShftRxDataInRe <= SIG_WIDTH_15_PARAM_0;
    ShftRxDataInIm <= SIG_WIDTH_15_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    ShftRxDataInRe1 <= SIG_WIDTH_15_PARAM_0;
    ShftRxDataInIm1 <= SIG_WIDTH_15_PARAM_0;
`endif
  end
  else if (RxDataInValidD1)  begin
    ShftRxDataInRe <= {RxDataInReD,2'b0};
    ShftRxDataInIm <= {RxDataInImD,2'b0};
`ifdef RW_NX_DERIV_PATH1
    ShftRxDataInRe1 <= {RxDataInRe1D,2'b0};
    ShftRxDataInIm1 <= {RxDataInIm1D,2'b0};
`endif
  end  
end 

// Register output of saturation block before passing to 
// output port.
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    RxDataOutRe    <= SIG_WIDTH_13_PARAM_0;
    RxDataOutIm    <= SIG_WIDTH_13_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    RxDataOutRe1    <= SIG_WIDTH_13_PARAM_0;
    RxDataOutIm1    <= SIG_WIDTH_13_PARAM_0;
`endif
    RxDataOutValid <= 1'b0;
  end
  else if (!DCCompEn) begin
    RxDataOutRe    <= SIG_WIDTH_13_PARAM_0;
    RxDataOutIm    <= SIG_WIDTH_13_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    RxDataOutRe1    <= SIG_WIDTH_13_PARAM_0;
    RxDataOutIm1    <= SIG_WIDTH_13_PARAM_0;
`endif
    RxDataOutValid <= 1'b0;
  end
  else begin
    if (RxDataInValidD3) begin
      RxDataOutRe  <= SatCompensatedOutputRe;
      RxDataOutIm  <= SatCompensatedOutputIm;
`ifdef RW_NX_DERIV_PATH1
      RxDataOutRe1  <= SatCompensatedOutputRe1;
      RxDataOutIm1  <= SatCompensatedOutputIm1;
`endif
    end
    RxDataOutValid <= RxDataInValidD3;
  end
end


// Round by 2 bits.
Round #(
            .INPUT_WIDTH(16),
            .OUTPUT_WIDTH(14)
           )
            U_ROUND_RE(
                        .InputData(CompensatedOutputRe),
                        .RoundData(RndCompensatedOutputRe)
                       );

Round #(
            .INPUT_WIDTH(16),
            .OUTPUT_WIDTH(14)
           )
            U_ROUND_IM(
                        .InputData(CompensatedOutputIm),
                        .RoundData(RndCompensatedOutputIm)
                       );

// Instantiation of Saturation component
SatSymSigned #(
               .INPUT_WIDTH(14),
               .OUTPUT_WIDTH(13)
              )
               U_SAT_RE(
                        .InputData(RndCompensatedOutputRe),
                        .SatSymData(SatCompensatedOutputRe)
                        );

SatSymSigned #(
               .INPUT_WIDTH(14),
               .OUTPUT_WIDTH(13)
              )
               U_SAT_IM(
                        .InputData(RndCompensatedOutputIm),
                        .SatSymData(SatCompensatedOutputIm)
                        );

`ifdef RW_NX_DERIV_PATH1
// Round by 2 bits.
Round #(
            .INPUT_WIDTH(16),
            .OUTPUT_WIDTH(14)
           )
            U_ROUND_RE1(
                        .InputData(CompensatedOutputRe1),
                        .RoundData(RndCompensatedOutputRe1)
                       );

Round #(
            .INPUT_WIDTH(16),
            .OUTPUT_WIDTH(14)
           )
            U_ROUND_IM1(
                        .InputData(CompensatedOutputIm1),
                        .RoundData(RndCompensatedOutputIm1)
                       );

// Instantiation of Saturation component
SatSymSigned #(
               .INPUT_WIDTH(14),
               .OUTPUT_WIDTH(13)
              )
               U_SAT_RE1(
                        .InputData(RndCompensatedOutputRe1),
                        .SatSymData(SatCompensatedOutputRe1)
                        );

SatSymSigned #(
               .INPUT_WIDTH(14),
               .OUTPUT_WIDTH(13)
              )
               U_SAT_IM1(
                        .InputData(RndCompensatedOutputIm1),
                        .SatSymData(SatCompensatedOutputIm1)
                        );
`endif

endmodule

`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
