/////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//--------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//--------------------------------------------------------------------------
// $Revision: 39956 $
// $Date: 2019-09-30 18:25:20 +0200 (Mon, 30 Sep 2019) $
// -------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Time Domain DC Offset Estimation.                                                  
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// -------------------------------------------------------------------------
//                                                                          
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/PHYSUBSYS/HDMCORE/OFDMACORE/OFDMRXCORE/OFDMRXTD/TDDCOffset/verilog/rtl/TDDCOffsetEst.v $
// 
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
`default_nettype none

module  TDDCOffsetEst (

            ///////////////////////////////////////////////
            // Clock & Reset
            ///////////////////////////////////////////////
            input   wire                    PhyClk,
            input   wire                    nPhyRst,
            
            ///////////////////////////////////////////////
            // Register interface
            ///////////////////////////////////////////////
            input   wire                    DCTrackingEn,
            // Channel bandwidth
            input   wire          [1:0]     Bandwidth,
            // Initial Delay before L-STF Estimation
            input   wire          [3:0]     StartDC,
            // Initial Delay before HT-STF Estimation
            input   wire          [3:0]     StartHTDC,
            // Wait number of samples before accumulation window
            input   wire          [6:0]     WaitSync,
            // Wait number of samples for HT-STF accumulation window
            input   wire          [6:0]     WaitHTSTF,

            ///////////////////////////////////////////////
            // Control interface
            ///////////////////////////////////////////////
            // Block enable
            input   wire                    DCEstEn,
            // PrDataPhase =0 Preamble Phase ,PrDataPhase=1 Data Phase
            input   wire                    PrDataPhase,
            // Clear DC estimation
            input   wire                    Clear,
            // Pulse from RxTD FSM for starting estimation
            input   wire                    Synch,
            // GI information : 0-0.4us / 1-0.8us / 2-1.6us / 3-3.2us
            input   wire          [1:0]     SizeGI,
            // Symbol information : 0-3.2us / 1-6.4us / 2-12.8us
            input   wire          [1:0]     SizeSymb,
            //
            // Accumulation window to RxTD FSM for short GI synchronization
            output  reg                     DataSynch,
            
            ///////////////////////////////////////////////
            // Data interface
            ///////////////////////////////////////////////
            input   wire                    RxDataInValid,
            input   wire signed   [12:0]    RxDataInRe,
            input   wire signed   [12:0]    RxDataInIm,
`ifdef RW_NX_DERIV_PATH1
            input   wire signed   [12:0]    RxDataInRe1,
            input   wire signed   [12:0]    RxDataInIm1,
`endif

            ///////////////////////////////////////////////
            // DC estimation interface
            ///////////////////////////////////////////////
            output  reg signed    [14:0]    DCEstRe,
            output  reg signed    [14:0]    DCEstIm,
`ifdef RW_NX_DERIV_PATH1
            output  reg signed    [14:0]    DCEstRe1,
            output  reg signed    [14:0]    DCEstIm1,
`endif
            output  reg                     DCEstValid
            );

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////
localparam signed [14:0] SIG_WIDTH_15_PARAM_0 = 15'b0;
localparam signed [20:0] SIG_WIDTH_21_PARAM_0 = 21'b0;
localparam signed [24:0] SIG_WIDTH_25_PARAM_0 = 25'b0;
localparam signed [25:0] SIG_WIDTH_26_PARAM_0 = 26'b0;
localparam signed [29:0] SIG_WIDTH_30_PARAM_0 = 30'b0;
localparam signed [31:0] SIG_WIDTH_32_PARAM_0 = 32'b0;
localparam signed [34:0] SIG_WIDTH_35_PARAM_0 = 35'd0;
localparam signed [35:0] SIG_WIDTH_36_PARAM_0 = 36'd0;

//////////////////////////////////////////////////////////////////////////////
//  Internal Wires Declarations
//////////////////////////////////////////////////////////////////////////////
// Output of Rounding operation.
wire        signed    [31:0]    RndKTableMulAccMRe_Rnd3;
wire        signed    [31:0]    RndKTableMulAccMIm_Rnd3;
wire        signed    [30:0]    RndKTableMulAccMRe_Rnd4;
wire        signed    [30:0]    RndKTableMulAccMIm_Rnd4;
wire        signed    [29:0]    RndKTableMulAccMRe_Rnd5;
wire        signed    [29:0]    RndKTableMulAccMIm_Rnd5;
wire        signed    [28:0]    RndKTableMulAccMRe_Rnd6;
wire        signed    [28:0]    RndKTableMulAccMIm_Rnd6;
wire        signed    [27:0]    RndKTableMulAccMRe_Rnd7;
wire        signed    [27:0]    RndKTableMulAccMIm_Rnd7;
wire        signed    [26:0]    RndKTableMulAccMRe_Rnd8;
wire        signed    [26:0]    RndKTableMulAccMIm_Rnd8;
wire        signed    [25:0]    RndGTableMulLamdaRe;
wire        signed    [25:0]    RndGTableMulLamdaIm;
`ifdef RW_NX_DERIV_PATH1
wire        signed    [31:0]    RndKTableMulAccMRe1_Rnd3;
wire        signed    [31:0]    RndKTableMulAccMIm1_Rnd3;
wire        signed    [30:0]    RndKTableMulAccMRe1_Rnd4;
wire        signed    [30:0]    RndKTableMulAccMIm1_Rnd4;
wire        signed    [29:0]    RndKTableMulAccMRe1_Rnd5;
wire        signed    [29:0]    RndKTableMulAccMIm1_Rnd5;
wire        signed    [28:0]    RndKTableMulAccMRe1_Rnd6;
wire        signed    [28:0]    RndKTableMulAccMIm1_Rnd6;
wire        signed    [27:0]    RndKTableMulAccMRe1_Rnd7;
wire        signed    [27:0]    RndKTableMulAccMIm1_Rnd7;
wire        signed    [26:0]    RndKTableMulAccMRe1_Rnd8;
wire        signed    [26:0]    RndKTableMulAccMIm1_Rnd8;
wire        signed    [25:0]    RndGTableMulLamdaRe1;
wire        signed    [25:0]    RndGTableMulLamdaIm1;
`endif

// K & G Table.
wire                  [12:0]    KTable[58:0];
wire                  [10:0]    GTable[58:0];

// Output from saturation block.
wire        signed    [24:0]    SatKMulAccRe;
wire        signed    [24:0]    SatKMulAccIm;
wire        signed    [24:0]    SatLambdaRe;
wire        signed    [24:0]    SatLambdaIm;
wire        signed    [14:0]    SatDCEstRe;
wire        signed    [14:0]    SatDCEstIm;
`ifdef RW_NX_DERIV_PATH1
wire        signed    [24:0]    SatKMulAccRe1;
wire        signed    [24:0]    SatKMulAccIm1;
wire        signed    [24:0]    SatLambdaRe1;
wire        signed    [24:0]    SatLambdaIm1;
wire        signed    [14:0]    SatDCEstRe1;
wire        signed    [14:0]    SatDCEstIm1;
`endif

// No of samples depending on Bandwidth.
wire                  [6:0]     NoOfSamplesPrPh;    // Preamble phase.
wire                  [6:0]     NoOfSamplesPrPhMux; // Preamble phase mux.
wire                  [9:0]     NoOfSamplesDataPh;  // Data Phase.
wire                  [7:0]     NoOfSamplesGIPh;    // Guard Interval Phase.
wire                  [2:0]     RndSel;             // Select rounding

wire                            LoadAccu;
wire                            LoadAccuSTF;
wire                            EarlyStartDC;

// Accumulator control.
wire                            CountSamplesEnd;

// HTSTF Accumulator size (0.4us or 0.8us).
wire                            HTSTFWindow;

//////////////////////////////////////////////////////////////////////////////
// Internal Registers Declarations
//////////////////////////////////////////////////////////////////////////////
// Accumulator.
reg         signed    [20:0]    RxDataRe;
reg         signed    [20:0]    RxDataIm;
`ifdef RW_NX_DERIV_PATH1
reg         signed    [20:0]    RxDataRe1;
reg         signed    [20:0]    RxDataIm1;
`endif

// Counter for no.of samples.
reg                   [9:0]     Count;
reg                             CountDisable;

// Registering K & G Table.
reg                   [12:0]    KTableVal;
reg                   [10:0]    GTableVal;

// K Multiplied by Accumulator.
reg         signed    [34:0]    KTableMulAccMRe;
reg         signed    [34:0]    KTableMulAccMIm;
`ifdef RW_NX_DERIV_PATH1
reg         signed    [34:0]    KTableMulAccMRe1;
reg         signed    [34:0]    KTableMulAccMIm1;
`endif

// G Multiplied by Previous DC Estimation value.
reg         signed    [35:0]    GTableMulLamdaRe;
reg         signed    [35:0]    GTableMulLamdaIm;
`ifdef RW_NX_DERIV_PATH1
reg         signed    [35:0]    GTableMulLamdaRe1;
reg         signed    [35:0]    GTableMulLamdaIm1;
`endif

// Previous short preamble dc est.value.
reg         signed    [29:0]    LambdaRe;
reg         signed    [29:0]    LambdaIm;
`ifdef RW_NX_DERIV_PATH1
reg         signed    [29:0]    LambdaRe1;
reg         signed    [29:0]    LambdaIm1;
`endif

// Delayed signals.
reg                             RxDataInValid_1t;

// Counter for no.of Symbols.
reg                   [6:0]     Ind;

// AccuValid for assigning outputs.
reg                             AccuValid;
reg                             AccuValid_1t;
reg                             AccuValid_2t;

// Flag for skipping starting samples. 
reg                             FlagClear;
reg                             FlagStartDC;

// Counter for skipping starting samples.
reg                  [3:0]      CountStartDC;

// Counter for skipping synchro samples.
reg                   [6:0]     CountSamples;
reg                             CountSamplesEn;

// Counter for HT-STF DC computation
reg                   [6:0]     CountSamplesSTF;

// Control signal generated from Synch Pulse.
reg                             Control;
reg                             Control_1t;
reg                             Control_2t;
reg                             Control_3t;

// Flag signifies HT-STF.
reg                             FlagHTSTF;
reg                             FlagHTSTFValid;

// Registering PrDataPhase input.
reg                             PrDataPhase_1t;

reg        signed    [25:0]     RndGTableMulLamdaRe_1t;
reg        signed    [25:0]     RndGTableMulLamdaIm_1t;
`ifdef RW_NX_DERIV_PATH1
reg        signed    [25:0]     RndGTableMulLamdaRe1_1t;
reg        signed    [25:0]     RndGTableMulLamdaIm1_1t;
`endif

// Registering saturation output.
reg        signed    [24:0]     SatLambdaRe_1t;
reg        signed    [24:0]     SatLambdaIm_1t;
`ifdef RW_NX_DERIV_PATH1
reg        signed    [24:0]     SatLambdaRe1_1t;
reg        signed    [24:0]     SatLambdaIm1_1t;
`endif

// Input to Saturation block.
reg        signed    [31:0]     RndKMulAcc2SatRe;
reg        signed    [31:0]     RndKMulAcc2SatIm;
`ifdef RW_NX_DERIV_PATH1
reg        signed    [31:0]     RndKMulAcc2SatRe1;
reg        signed    [31:0]     RndKMulAcc2SatIm1;
`endif

// Synchro is done
reg                             SynchDone;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// KTable
assign KTable[0]  = 13'd4096;
assign KTable[1]  = 13'd2048;
assign KTable[2]  = 13'd1365;
assign KTable[3]  = 13'd1024;
assign KTable[4]  = 13'd68;
assign KTable[5]  = 13'd64;
assign KTable[6]  = 13'd60;
assign KTable[7]  = 13'd57;
assign KTable[8]  = 13'd53;
assign KTable[9]  = 13'd50;
assign KTable[10] = 13'd47;
assign KTable[11] = 13'd44;
assign KTable[12] = 13'd42;
assign KTable[13] = 13'd39;
assign KTable[14] = 13'd37;
assign KTable[15] = 13'd35;
assign KTable[16] = 13'd33;
assign KTable[17] = 13'd31;
assign KTable[18] = 13'd29;
assign KTable[19] = 13'd27;
assign KTable[20] = 13'd25;
assign KTable[21] = 13'd24;
assign KTable[22] = 13'd22;
assign KTable[23] = 13'd21;
assign KTable[24] = 13'd20;
assign KTable[25] = 13'd19;
assign KTable[26] = 13'd18;
assign KTable[27] = 13'd17;
assign KTable[28] = 13'd16;
assign KTable[29] = 13'd15;
assign KTable[30] = 13'd14;
assign KTable[31] = 13'd13;
assign KTable[32] = 13'd12;
assign KTable[33] = 13'd11;
assign KTable[34] = 13'd11;
assign KTable[35] = 13'd10;
assign KTable[36] = 13'd10;
assign KTable[37] = 13'd9;
assign KTable[38] = 13'd8;
assign KTable[39] = 13'd8;
assign KTable[40] = 13'd7;
assign KTable[41] = 13'd7;
assign KTable[42] = 13'd7;
assign KTable[43] = 13'd6;
assign KTable[44] = 13'd6;
assign KTable[45] = 13'd5;
assign KTable[46] = 13'd5;
assign KTable[47] = 13'd5;
assign KTable[48] = 13'd5;
assign KTable[49] = 13'd4;
assign KTable[50] = 13'd4;
assign KTable[51] = 13'd4;
assign KTable[52] = 13'd4;
assign KTable[53] = 13'd3;
assign KTable[54] = 13'd3;
assign KTable[55] = 13'd3;
assign KTable[56] = 13'd3;
assign KTable[57] = 13'd3;
assign KTable[58] = 13'd2;

// GTable
assign GTable[0]  = 11'd0;
assign GTable[1]  = 11'd512;
assign GTable[2]  = 11'd683;
assign GTable[3]  = 11'd768;
assign GTable[4]  = 11'd956;
assign GTable[5]  = 11'd960;
assign GTable[6]  = 11'd964;
assign GTable[7]  = 11'd967;
assign GTable[8]  = 11'd971;
assign GTable[9]  = 11'd974;
assign GTable[10] = 11'd977;
assign GTable[11] = 11'd980;
assign GTable[12] = 11'd982;
assign GTable[13] = 11'd985;
assign GTable[14] = 11'd987;
assign GTable[15] = 11'd989;
assign GTable[16] = 11'd991;
assign GTable[17] = 11'd993;
assign GTable[18] = 11'd995;
assign GTable[19] = 11'd997;
assign GTable[20] = 11'd999;
assign GTable[21] = 11'd1000;
assign GTable[22] = 11'd1002;
assign GTable[23] = 11'd1003;
assign GTable[24] = 11'd1004;
assign GTable[25] = 11'd1005;
assign GTable[26] = 11'd1006;
assign GTable[27] = 11'd1007;
assign GTable[28] = 11'd1008;
assign GTable[29] = 11'd1009;
assign GTable[30] = 11'd1010;
assign GTable[31] = 11'd1011;
assign GTable[32] = 11'd1012;
assign GTable[33] = 11'd1013;
assign GTable[34] = 11'd1013;
assign GTable[35] = 11'd1014;
assign GTable[36] = 11'd1014;
assign GTable[37] = 11'd1015;
assign GTable[38] = 11'd1016;
assign GTable[39] = 11'd1016;
assign GTable[40] = 11'd1017;
assign GTable[41] = 11'd1017;
assign GTable[42] = 11'd1017;
assign GTable[43] = 11'd1018;
assign GTable[44] = 11'd1018;
assign GTable[45] = 11'd1019;
assign GTable[46] = 11'd1019;
assign GTable[47] = 11'd1019;
assign GTable[48] = 11'd1019;
assign GTable[49] = 11'd1020;
assign GTable[50] = 11'd1020;
assign GTable[51] = 11'd1020;
assign GTable[52] = 11'd1020;
assign GTable[53] = 11'd1021;
assign GTable[54] = 11'd1021;
assign GTable[55] = 11'd1021;
assign GTable[56] = 11'd1021;
assign GTable[57] = 11'd1021;
assign GTable[58] = 11'd1022;

// Number of Samples in L-STF
assign NoOfSamplesPrPh =  (Bandwidth==2'd0) ? 7'd15 : 
                          (Bandwidth==2'd1) ? 7'd31 : 7'd63;

// Number of Samples in HT-STF
assign NoOfSamplesPrPhMux = (FlagHTSTF) ? WaitHTSTF : NoOfSamplesPrPh;

// HT-STF estimation in 0.8us->1 or 0.4us->0
assign HTSTFWindow = (Bandwidth == 2'd0 && WaitHTSTF == 7'd15) ? 1'b1 :
                     (Bandwidth == 2'd1 && WaitHTSTF == 7'd31) ? 1'b1 :
                     (Bandwidth == 2'd2 && WaitHTSTF == 7'd63) ? 1'b1 :
                                                                 1'b0;

// Number of Samples in Data Phase
assign NoOfSamplesDataPh = (Bandwidth == 2'd0) ? ((SizeSymb==2'd2) ? 10'd255  :
                                                  (SizeSymb==2'd1) ? 10'd127  : 10'd63)  :
                           (Bandwidth == 2'd1) ? ((SizeSymb==2'd2) ? 10'd511  :
                                                  (SizeSymb==2'd1) ? 10'd255  : 10'd127) : 
                                                 ((SizeSymb==2'd2) ? 10'd1023 :
                                                  (SizeSymb==2'd1) ? 10'd511  : 10'd255);

// Select rounding for RndKMulAcc2Sat
assign RndSel = (Bandwidth == 2'd0) ? ((SizeSymb==2'd2) ? 3'd2  :
                                       (SizeSymb==2'd1) ? 3'd1  : 3'd0) :
                (Bandwidth == 2'd1) ? ((SizeSymb==2'd2) ? 3'd3  :
                                       (SizeSymb==2'd1) ? 3'd2  : 3'd1) : 
                                      ((SizeSymb==2'd2) ? 3'd4  :
                                       (SizeSymb==2'd1) ? 3'd3  : 3'd2);

// Number of Samples in Guard Interval Phase
assign NoOfSamplesGIPh = (Bandwidth == 2'd0) ? ((SizeGI == 2'd0) ? 8'd7  :
                                                (SizeGI == 2'd1) ? 8'd15 :
                                                (SizeGI == 2'd2) ? 8'd31 : 8'd63)  :
                         (Bandwidth == 2'd1) ? ((SizeGI == 2'd0) ? 8'd15 :
                                                (SizeGI == 2'd1) ? 8'd31 :
                                                (SizeGI == 2'd2) ? 8'd63 : 8'd127) :
                                               ((SizeGI == 2'd0) ? 8'd31 :
                                                (SizeGI == 2'd1) ? 8'd63 :
                                                (SizeGI == 2'd2) ? 8'd127 : 8'd255);

// Input to saturation table.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    RndKMulAcc2SatRe        <= SIG_WIDTH_32_PARAM_0;
    RndKMulAcc2SatIm        <= SIG_WIDTH_32_PARAM_0;
    RndGTableMulLamdaRe_1t  <= SIG_WIDTH_26_PARAM_0;
    RndGTableMulLamdaIm_1t  <= SIG_WIDTH_26_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    RndKMulAcc2SatRe1       <= SIG_WIDTH_32_PARAM_0;
    RndKMulAcc2SatIm1       <= SIG_WIDTH_32_PARAM_0;
    RndGTableMulLamdaRe1_1t <= SIG_WIDTH_26_PARAM_0;
    RndGTableMulLamdaIm1_1t <= SIG_WIDTH_26_PARAM_0;
`endif
    
  end
  else begin  
    RndGTableMulLamdaRe_1t <= RndGTableMulLamdaRe;
    RndGTableMulLamdaIm_1t <= RndGTableMulLamdaIm;
`ifdef RW_NX_DERIV_PATH1
    RndGTableMulLamdaRe1_1t <= RndGTableMulLamdaRe1;
    RndGTableMulLamdaIm1_1t <= RndGTableMulLamdaIm1;
`endif

    if (AccuValid) begin
    
      if ((Bandwidth==2'd0) && !HTSTFWindow && FlagHTSTF) begin
        RndKMulAcc2SatRe <= RndKTableMulAccMRe_Rnd3;
        RndKMulAcc2SatIm <= RndKTableMulAccMIm_Rnd3;
  `ifdef RW_NX_DERIV_PATH1
        RndKMulAcc2SatRe1 <= RndKTableMulAccMRe1_Rnd3;
        RndKMulAcc2SatIm1 <= RndKTableMulAccMIm1_Rnd3;
  `endif
      end
      else if ((RndSel==3'd0) || (Bandwidth == 2'd1 && !HTSTFWindow && FlagHTSTF)) begin
        RndKMulAcc2SatRe <= {RndKTableMulAccMRe_Rnd4[30], RndKTableMulAccMRe_Rnd4};
        RndKMulAcc2SatIm <= {RndKTableMulAccMIm_Rnd4[30], RndKTableMulAccMIm_Rnd4};
  `ifdef RW_NX_DERIV_PATH1
        RndKMulAcc2SatRe1 <= {RndKTableMulAccMRe1_Rnd4[30], RndKTableMulAccMRe1_Rnd4};
        RndKMulAcc2SatIm1 <= {RndKTableMulAccMIm1_Rnd4[30], RndKTableMulAccMIm1_Rnd4};
  `endif
      end
      else if ((RndSel==3'd1) || (Bandwidth == 2'd2 && !HTSTFWindow && FlagHTSTF)) begin
        RndKMulAcc2SatRe <= {{2{RndKTableMulAccMRe_Rnd5[29]}}, RndKTableMulAccMRe_Rnd5};
        RndKMulAcc2SatIm <= {{2{RndKTableMulAccMIm_Rnd5[29]}}, RndKTableMulAccMIm_Rnd5};
  `ifdef RW_NX_DERIV_PATH1
        RndKMulAcc2SatRe1 <= {{2{RndKTableMulAccMRe1_Rnd5[29]}}, RndKTableMulAccMRe1_Rnd5};
        RndKMulAcc2SatIm1 <= {{2{RndKTableMulAccMIm1_Rnd5[29]}}, RndKTableMulAccMIm1_Rnd5};
  `endif
      end
      else if (RndSel==3'd2) begin
        RndKMulAcc2SatRe <= {{3{RndKTableMulAccMRe_Rnd6[28]}}, RndKTableMulAccMRe_Rnd6};
        RndKMulAcc2SatIm <= {{3{RndKTableMulAccMIm_Rnd6[28]}}, RndKTableMulAccMIm_Rnd6};
  `ifdef RW_NX_DERIV_PATH1
        RndKMulAcc2SatRe1 <= {{3{RndKTableMulAccMRe1_Rnd6[28]}}, RndKTableMulAccMRe1_Rnd6};
        RndKMulAcc2SatIm1 <= {{3{RndKTableMulAccMIm1_Rnd6[28]}}, RndKTableMulAccMIm1_Rnd6};
  `endif
      end
      else if (RndSel==3'd3) begin
        RndKMulAcc2SatRe <= {{4{RndKTableMulAccMRe_Rnd7[27]}}, RndKTableMulAccMRe_Rnd7};
        RndKMulAcc2SatIm <= {{4{RndKTableMulAccMIm_Rnd7[27]}}, RndKTableMulAccMIm_Rnd7};
  `ifdef RW_NX_DERIV_PATH1
        RndKMulAcc2SatRe1 <= {{4{RndKTableMulAccMRe1_Rnd7[27]}}, RndKTableMulAccMRe1_Rnd7};
        RndKMulAcc2SatIm1 <= {{4{RndKTableMulAccMIm1_Rnd7[27]}}, RndKTableMulAccMIm1_Rnd7};
  `endif
      end
      else begin
        RndKMulAcc2SatRe <= {{5{RndKTableMulAccMRe_Rnd8[26]}}, RndKTableMulAccMRe_Rnd8};
        RndKMulAcc2SatIm <= {{5{RndKTableMulAccMIm_Rnd8[26]}}, RndKTableMulAccMIm_Rnd8};
  `ifdef RW_NX_DERIV_PATH1
        RndKMulAcc2SatRe1 <= {{5{RndKTableMulAccMRe1_Rnd8[26]}}, RndKTableMulAccMRe1_Rnd8};
        RndKMulAcc2SatIm1 <= {{5{RndKTableMulAccMIm1_Rnd8[26]}}, RndKTableMulAccMIm1_Rnd8};
  `endif
      end
    end
  end
end

// Makes flag high for HT/VHT/HE-STF.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    FlagHTSTF <= 1'b0;
  else if ((FlagHTSTF && SynchDone) || !DCEstEn)
    FlagHTSTF <= 1'b0;
  else if (!PrDataPhase && PrDataPhase_1t)
    FlagHTSTF <= 1'b1;
end

// Makes flag high for HT/VHT/HE-STF first accumulator valid.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    FlagHTSTFValid <= 1'b0;
  else if (!DCEstEn || Clear)
    FlagHTSTFValid <= 1'b0;
  else if (FlagHTSTF && (CountSamplesSTF==7'b0))
    FlagHTSTFValid <= 1'b1;
end

// Sample counter enable is for generating control signal depending on synch pulse.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    CountSamplesEn <= 1'b0;
  else if (Synch)
    CountSamplesEn <= 1'b1;
  else if (Control && !Control_1t)
    CountSamplesEn <= 1'b0;
end

// Generates control level signal for accumulator.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    CountSamples <= 7'd0;
    Control      <= 1'b0;
  end
  else if (Clear || !DCTrackingEn || !DCEstEn) begin
    CountSamples <= 7'd0;
    Control      <= 1'b0;
  end
  else if (RxDataInValid_1t) begin
    // Pulse generation
    if (Control_1t)
      Control      <= 1'b0;
    if (!Control && CountSamplesEnd) begin
      CountSamples <= 7'd0;
      Control      <= 1'b1;
    end 
    else if (CountSamplesEn) begin
      CountSamples <= CountSamples + 7'd1;
    end
  end
end

// Accumulator control for timing synchronisation adjustment
assign CountSamplesEnd = (FlagHTSTF) ? (CountSamples > 7'd0) & CountSamplesEn :
                                       (CountSamples > WaitSync);

// Sample counter enable is for generating control signal depending on synch pulse.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    CountSamplesSTF  <= 7'b0;
  else if (!DCEstEn)
    CountSamplesSTF  <= 7'b0;
  else if (Control_1t)
    CountSamplesSTF  <= 7'b0;
  else begin
    if (RxDataInValid) begin
      if (CountSamplesSTF == NoOfSamplesPrPh)
        CountSamplesSTF  <= 7'b0;
      else
        CountSamplesSTF <= CountSamplesSTF + 7'd1;
    end
  end
end

assign LoadAccuSTF = (CountSamplesSTF == 7'b0) & (Ind < 7'd4);

// Accumulates input value
always@(posedge PhyClk or negedge nPhyRst)
begin:Accumulator
  if (nPhyRst == 1'b0) begin
    RxDataRe <= SIG_WIDTH_21_PARAM_0;
    RxDataIm <= SIG_WIDTH_21_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    RxDataRe1 <= SIG_WIDTH_21_PARAM_0;
    RxDataIm1 <= SIG_WIDTH_21_PARAM_0;
`endif
  end
  else if (RxDataInValid && (FlagStartDC || EarlyStartDC)) begin
    if (LoadAccu) begin
      RxDataRe <= {{8{RxDataInRe[12]}},RxDataInRe};
      RxDataIm <= {{8{RxDataInIm[12]}},RxDataInIm};
`ifdef RW_NX_DERIV_PATH1
      RxDataRe1 <= {{8{RxDataInRe1[12]}},RxDataInRe1};
      RxDataIm1 <= {{8{RxDataInIm1[12]}},RxDataInIm1};
`endif
    end
    else begin
      RxDataRe <= RxDataRe + {{8{RxDataInRe[12]}},RxDataInRe};
      RxDataIm <= RxDataIm + {{8{RxDataInIm[12]}},RxDataInIm};
`ifdef RW_NX_DERIV_PATH1
      RxDataRe1 <= RxDataRe1 + {{8{RxDataInRe1[12]}},RxDataInRe1};
      RxDataIm1 <= RxDataIm1 + {{8{RxDataInIm1[12]}},RxDataInIm1};
`endif
    end
  end
end

// Generating delays of control signals.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    RxDataInValid_1t <= 1'b0;
    PrDataPhase_1t   <= 1'b0;
    Control_1t       <= 1'b0;
    Control_2t       <= 1'b0;
    Control_3t       <= 1'b0;
    AccuValid_1t     <= 1'b0;
    AccuValid_2t     <= 1'b0;
  end
  else if (!DCEstEn) begin
    RxDataInValid_1t <= 1'b0;
    PrDataPhase_1t   <= 1'b0;
    Control_1t       <= 1'b0;
    Control_2t       <= 1'b0;
    Control_3t       <= 1'b0;
    AccuValid_1t     <= 1'b0;
    AccuValid_2t     <= 1'b0;
  end
  else begin
    RxDataInValid_1t <= RxDataInValid;
    PrDataPhase_1t   <= PrDataPhase;
    if (RxDataInValid) begin
      Control_1t       <= Control;
      Control_2t       <= Control_1t & ~CountDisable;
      Control_3t       <= Control_2t;
    end
    AccuValid_1t     <= AccuValid;
    AccuValid_2t     <= AccuValid_1t;
  end
end

// Increment index to keep a count of number of STF.
always@(posedge PhyClk or negedge nPhyRst)
begin:IncrementInd
  if (nPhyRst == 1'b0)
    Ind <= 7'd0;
  else if (Clear || FlagClear || !DCTrackingEn || !DCEstEn)
    Ind <= 7'd0;
  else if (PrDataPhase && !PrDataPhase_1t)
    Ind <= 7'd4;
  else if (AccuValid && Ind < 7'd58)
    Ind <= Ind + 7'd1;
end

// Control logic to skip starting # samples.
always@(posedge PhyClk or negedge nPhyRst)
begin:CounterStartDC
  if (nPhyRst == 1'b0)begin
    CountStartDC <= 4'd0;
    FlagStartDC  <= 1'b0;
    FlagClear    <= 1'b0;
  end
  else begin
    if (Clear)
      FlagClear <= 1'b1;
    else if (((Count == 10'd1) && FlagClear) || !DCEstEn)
      FlagClear <= 1'b0;

    if (Clear || !DCTrackingEn || !DCEstEn) begin 
      CountStartDC <= 4'd0;
      FlagStartDC  <= 1'b0;
    end 
    else if (FlagClear) begin
      if (EarlyStartDC)
        FlagStartDC <= 1'b1;
      else if (RxDataInValid) begin
        CountStartDC <= CountStartDC + 4'd1;
        FlagStartDC  <= 1'b0;
      end
    end
  end
end

assign EarlyStartDC = (FlagHTSTF) ? ((CountStartDC == StartHTDC) & ~Clear) : 
                                    ((CountStartDC == StartDC)   & ~Clear);

// This toggle decides accumulation window between GI and data samples
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    DataSynch <= 1'b0;
  else if (FlagClear || !DCTrackingEn || !DCEstEn)
     DataSynch <= 1'b0;
  else if (PrDataPhase && LoadAccu && !CountDisable && RxDataInValid && FlagStartDC)
     DataSynch <= ~DataSynch;
end

// Disable count between end of STF and beginning of synchro
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    CountDisable <= 1'b0;
  else if (Control_1t || FlagClear || !DCTrackingEn || !DCEstEn)
    CountDisable <= 1'b0; 
  else if ((FlagHTSTF && !PrDataPhase) || (PrDataPhase && !PrDataPhase_1t))
    CountDisable <= 1'b1;
end

// Synchro done
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    SynchDone <= 1'b0;
  else if (Clear || !DCTrackingEn || (!PrDataPhase && PrDataPhase_1t) || !DCEstEn)
    SynchDone <= 1'b0;
  else if (Control_3t)
    SynchDone <= 1'b1;
end

// Reset counter when it counts till NoOfSamplesPrPh/NoOfSamplesDataPh
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)
    Count <= 10'd0;
  else if (Clear || !DCTrackingEn || !DCEstEn)
    Count <= 10'd0;
  else if (((!FlagStartDC && Bandwidth != 2'd2) || !EarlyStartDC) && FlagClear)
    Count <= 10'd0;
  else if (CountDisable)
    Count <= 10'd1;
  else if (RxDataInValid) begin
    if (((Count == {3'b0,NoOfSamplesPrPhMux}) && !PrDataPhase && (Ind < 7'd4))             ||
       (((Count == NoOfSamplesDataPh) || (!DataSynch && (Count == {2'b0,NoOfSamplesGIPh})) ||
        (!SynchDone && (Control_1t && !Control_2t))) && PrDataPhase))
      Count <= 10'd0;
    else
      Count <= Count + 10'd1;
  end
end 

// Accumulator load
assign LoadAccu = (FlagHTSTF & ~PrDataPhase) ? LoadAccuSTF : (Count == 10'd0) & DCEstEn;

// Register outputs.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    DCEstRe    <= SIG_WIDTH_15_PARAM_0;
    DCEstIm    <= SIG_WIDTH_15_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    DCEstRe1   <= SIG_WIDTH_15_PARAM_0;
    DCEstIm1   <= SIG_WIDTH_15_PARAM_0;
`endif
    DCEstValid <= 1'b0;
  end
  else if (FlagClear || !DCTrackingEn || !DCEstEn) begin
    DCEstRe    <= SIG_WIDTH_15_PARAM_0;
    DCEstIm    <= SIG_WIDTH_15_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    DCEstRe1   <= SIG_WIDTH_15_PARAM_0;
    DCEstIm1   <= SIG_WIDTH_15_PARAM_0;
`endif
    DCEstValid <= 1'b0;
  end
  else if (AccuValid_2t) begin
    DCEstRe    <= SatDCEstRe;
    DCEstIm    <= SatDCEstIm;
`ifdef RW_NX_DERIV_PATH1
    DCEstRe1   <= SatDCEstRe1;
    DCEstIm1   <= SatDCEstIm1;
`endif
    DCEstValid <= 1'b1;
  end
end 

// Register Tables.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    KTableVal <= 13'd4096;
    GTableVal <= 11'b0;
  end
  else begin
    KTableVal <= KTable[Ind];
    GTableVal <= GTable[Ind];
  end
end 

// Register lamda saturation.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0)begin
    SatLambdaRe_1t <= SIG_WIDTH_25_PARAM_0;
    SatLambdaIm_1t <= SIG_WIDTH_25_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    SatLambdaRe1_1t <= SIG_WIDTH_25_PARAM_0;
    SatLambdaIm1_1t <= SIG_WIDTH_25_PARAM_0;
`endif
  end
  else begin
    SatLambdaRe_1t <= SatLambdaRe;
    SatLambdaIm_1t <= SatLambdaIm;
`ifdef RW_NX_DERIV_PATH1
    SatLambdaRe1_1t <= SatLambdaRe1;
    SatLambdaIm1_1t <= SatLambdaIm1;
`endif
  end
end 

// Multiply accumulated output with KTable.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    KTableMulAccMRe <= SIG_WIDTH_35_PARAM_0;
    KTableMulAccMIm <= SIG_WIDTH_35_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    KTableMulAccMRe1 <= SIG_WIDTH_35_PARAM_0;
    KTableMulAccMIm1 <= SIG_WIDTH_35_PARAM_0;
`endif
    AccuValid       <= 1'b0;
  end
  else if (FlagClear || !DCTrackingEn || !DCEstEn) begin
    KTableMulAccMRe <= SIG_WIDTH_35_PARAM_0;
    KTableMulAccMIm <= SIG_WIDTH_35_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    KTableMulAccMRe1 <= SIG_WIDTH_35_PARAM_0;
    KTableMulAccMIm1 <= SIG_WIDTH_35_PARAM_0;
`endif
    AccuValid       <= 1'b0;
  end
  else begin
    if (LoadAccu && RxDataInValid_1t && (DataSynch || (!PrDataPhase && (FlagHTSTFValid || !FlagHTSTF)))) begin
      KTableMulAccMRe <= RxDataRe * $signed({2'b0,KTableVal});
      KTableMulAccMIm <= RxDataIm * $signed({2'b0,KTableVal});
`ifdef RW_NX_DERIV_PATH1
      KTableMulAccMRe1 <= RxDataRe1 * $signed({2'b0,KTableVal});
      KTableMulAccMIm1 <= RxDataIm1 * $signed({2'b0,KTableVal});
`endif
      AccuValid       <= 1'b1;
    end
    else
      AccuValid       <= 1'b0;
  end
end

// Multiply Previous STF lambda with GTable.
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    GTableMulLamdaRe  <= SIG_WIDTH_36_PARAM_0;
    GTableMulLamdaIm  <= SIG_WIDTH_36_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    GTableMulLamdaRe1 <= SIG_WIDTH_36_PARAM_0;
    GTableMulLamdaIm1 <= SIG_WIDTH_36_PARAM_0;
`endif
  end
  else if (FlagClear || !DCTrackingEn || !DCEstEn) begin
    GTableMulLamdaRe  <= SIG_WIDTH_36_PARAM_0;
    GTableMulLamdaIm  <= SIG_WIDTH_36_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    GTableMulLamdaRe1 <= SIG_WIDTH_36_PARAM_0;
    GTableMulLamdaIm1 <= SIG_WIDTH_36_PARAM_0;
`endif
  end
  else if (LoadAccu && RxDataInValid_1t && (DataSynch || !PrDataPhase)) begin
    GTableMulLamdaRe  <= SatLambdaRe_1t * $signed({1'b0,GTableVal});
    GTableMulLamdaIm  <= SatLambdaIm_1t * $signed({1'b0,GTableVal});
`ifdef RW_NX_DERIV_PATH1
    GTableMulLamdaRe1 <= SatLambdaRe1_1t * $signed({1'b0,GTableVal});
    GTableMulLamdaIm1 <= SatLambdaIm1_1t * $signed({1'b0,GTableVal});
`endif
  end
end

// Addition of KMulAccumulator & GMulLambda
always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    LambdaRe  <= SIG_WIDTH_30_PARAM_0; 
    LambdaIm  <= SIG_WIDTH_30_PARAM_0; 
`ifdef RW_NX_DERIV_PATH1
    LambdaRe1 <= SIG_WIDTH_30_PARAM_0; 
    LambdaIm1 <= SIG_WIDTH_30_PARAM_0; 
`endif
  end
  else if (FlagClear || !DCTrackingEn || !DCEstEn) begin
    LambdaRe  <= SIG_WIDTH_30_PARAM_0;
    LambdaIm  <= SIG_WIDTH_30_PARAM_0;
`ifdef RW_NX_DERIV_PATH1
    LambdaRe1 <= SIG_WIDTH_30_PARAM_0;
    LambdaIm1 <= SIG_WIDTH_30_PARAM_0;
`endif
  end
  else begin
    LambdaRe <= {{5{SatKMulAccRe[24]}},SatKMulAccRe} + {{4{RndGTableMulLamdaRe_1t[25]}},RndGTableMulLamdaRe_1t};
    LambdaIm <= {{5{SatKMulAccIm[24]}},SatKMulAccIm} + {{4{RndGTableMulLamdaIm_1t[25]}},RndGTableMulLamdaIm_1t};
`ifdef RW_NX_DERIV_PATH1
    LambdaRe1 <= {{5{SatKMulAccRe1[24]}},SatKMulAccRe1} + {{4{RndGTableMulLamdaRe1_1t[25]}},RndGTableMulLamdaRe1_1t};
    LambdaIm1 <= {{5{SatKMulAccIm1[24]}},SatKMulAccIm1} + {{4{RndGTableMulLamdaIm1_1t[25]}},RndGTableMulLamdaIm1_1t};
`endif
  end
end

// Round by 3 bits if Bandwidth is 20MHz and HTSTF Window is 0.4us
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(32))
  U_ROUND_RE_Rnd3(
        .InputData(KTableMulAccMRe),
        .RoundData(RndKTableMulAccMRe_Rnd3)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(32))
  U_ROUND_IM_Rnd3(
        .InputData(KTableMulAccMIm),
        .RoundData(RndKTableMulAccMIm_Rnd3)
        );

// Round by 4 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(31))
  U_ROUND_RE_Rnd4(
        .InputData(KTableMulAccMRe),
        .RoundData(RndKTableMulAccMRe_Rnd4)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(31))
  U_ROUND_IM_Rnd4(
        .InputData(KTableMulAccMIm),
        .RoundData(RndKTableMulAccMIm_Rnd4)
        );

// Round by 5 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(30))
  U_ROUND_RE_Rnd5(
        .InputData(KTableMulAccMRe),
        .RoundData(RndKTableMulAccMRe_Rnd5)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(30))
  U_ROUND_IM_Rnd5(
        .InputData(KTableMulAccMIm),
        .RoundData(RndKTableMulAccMIm_Rnd5)
        );

// Round by 6 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(29))
  U_ROUND_RE_Rnd6(
        .InputData(KTableMulAccMRe),
        .RoundData(RndKTableMulAccMRe_Rnd6)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(29))
  U_ROUND_IM_Rnd6(
        .InputData(KTableMulAccMIm),
        .RoundData(RndKTableMulAccMIm_Rnd6)
        );

// Round by 7 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(28))
  U_ROUND_RE_Rnd7(
        .InputData(KTableMulAccMRe),
        .RoundData(RndKTableMulAccMRe_Rnd7)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(28))
  U_ROUND_IM_Rnd7(
        .InputData(KTableMulAccMIm),
        .RoundData(RndKTableMulAccMIm_Rnd7)
        );

// Round by 8 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(27))
  U_ROUND_RE_Rnd8(
        .InputData(KTableMulAccMRe),
        .RoundData(RndKTableMulAccMRe_Rnd8)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(27))
  U_ROUND_IM_Rnd8(
        .InputData(KTableMulAccMIm),
        .RoundData(RndKTableMulAccMIm_Rnd8)
        );

// Instantiation of Saturation component
SatSigned #(.INPUT_WIDTH(32),
            .OUTPUT_WIDTH(25))
  U1_SAT_RE(
            .InputData(RndKMulAcc2SatRe),
            .SatData(SatKMulAccRe)
            );

SatSigned #(.INPUT_WIDTH(32),
            .OUTPUT_WIDTH(25))
  U1_SAT_IM(
            .InputData(RndKMulAcc2SatIm),
            .SatData(SatKMulAccIm)
            );

// Round by 10 bits.
Round #(.INPUT_WIDTH(36),
        .OUTPUT_WIDTH(26))
  U4_ROUND_RE(
        .InputData(GTableMulLamdaRe),
        .RoundData(RndGTableMulLamdaRe)
        );

Round #(.INPUT_WIDTH(36),
        .OUTPUT_WIDTH(26))
  U4_ROUND_IM(
        .InputData(GTableMulLamdaIm),
        .RoundData(RndGTableMulLamdaIm)
        );

// Instantiation of Saturation component
SatSigned #(.INPUT_WIDTH(30),
            .OUTPUT_WIDTH(25))
  U2_SAT_RE(
            .InputData(LambdaRe),
            .SatData(SatLambdaRe)
            );

SatSigned #(.INPUT_WIDTH(30),
            .OUTPUT_WIDTH(25))
  U2_SAT_IM(
            .InputData(LambdaIm),
            .SatData(SatLambdaIm)
            );

// Round by 10 bits.
Round #(.INPUT_WIDTH(25),
        .OUTPUT_WIDTH(15))
  U5_ROUND_RE(
        .InputData(SatLambdaRe),
        .RoundData(SatDCEstRe)
        );

Round #(.INPUT_WIDTH(25),
        .OUTPUT_WIDTH(15))
  U5_ROUND_IM(
        .InputData(SatLambdaIm),
        .RoundData(SatDCEstIm)
        );

`ifdef RW_NX_DERIV_PATH1
// Round by 3 bits if HTSTF Window is 0.4us in 20MHz BW
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(32))
  U_ROUND_RE1_Rnd3(
        .InputData(KTableMulAccMRe1),
        .RoundData(RndKTableMulAccMRe1_Rnd3)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(32))
  U_ROUND_IM1_Rnd3(
        .InputData(KTableMulAccMIm1),
        .RoundData(RndKTableMulAccMIm1_Rnd3)
        );

// Round by 4 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(31))
  U_ROUND_RE1_Rnd4(
        .InputData(KTableMulAccMRe1),
        .RoundData(RndKTableMulAccMRe1_Rnd4)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(31))
  U_ROUND_IM1_Rnd4(
        .InputData(KTableMulAccMIm1),
        .RoundData(RndKTableMulAccMIm1_Rnd4)
        );

// Round by 5 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(30))
  U_ROUND_RE1_Rnd5(
        .InputData(KTableMulAccMRe1),
        .RoundData(RndKTableMulAccMRe1_Rnd5)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(30))
  U_ROUND_IM1_Rnd5(
        .InputData(KTableMulAccMIm1),
        .RoundData(RndKTableMulAccMIm1_Rnd5)
        );

// Round by 6 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(29))
  U_ROUND_RE1_Rnd6(
        .InputData(KTableMulAccMRe1),
        .RoundData(RndKTableMulAccMRe1_Rnd6)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(29))
  U_ROUND_IM1_Rnd6(
        .InputData(KTableMulAccMIm1),
        .RoundData(RndKTableMulAccMIm1_Rnd6)
        );

// Round by 7 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(28))
  U_ROUND_RE1_Rnd7(
        .InputData(KTableMulAccMRe1),
        .RoundData(RndKTableMulAccMRe1_Rnd7)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(28))
  U_ROUND_IM1_Rnd7(
        .InputData(KTableMulAccMIm1),
        .RoundData(RndKTableMulAccMIm1_Rnd7)
        );

// Round by 8 bits
Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(27))
  U_ROUND_RE1_Rnd8(
        .InputData(KTableMulAccMRe1),
        .RoundData(RndKTableMulAccMRe1_Rnd8)
        );

Round #(.INPUT_WIDTH(35),
        .OUTPUT_WIDTH(27))
  U_ROUND_IM1_Rnd8(
        .InputData(KTableMulAccMIm1),
        .RoundData(RndKTableMulAccMIm1_Rnd8)
        );

// Instantiation of Saturation component
SatSigned #(.INPUT_WIDTH(32),
            .OUTPUT_WIDTH(25))
  U1_SAT_RE1(
            .InputData(RndKMulAcc2SatRe1),
            .SatData(SatKMulAccRe1)
            );

SatSigned #(.INPUT_WIDTH(32),
            .OUTPUT_WIDTH(25))
  U1_SAT_IM1(
            .InputData(RndKMulAcc2SatIm1),
            .SatData(SatKMulAccIm1)
            );

// Round by 10 bits.
Round #(.INPUT_WIDTH(36),
        .OUTPUT_WIDTH(26))
  U4_ROUND_RE1(
        .InputData(GTableMulLamdaRe1),
        .RoundData(RndGTableMulLamdaRe1)
        );

Round #(.INPUT_WIDTH(36),
        .OUTPUT_WIDTH(26))
  U4_ROUND_IM1(
        .InputData(GTableMulLamdaIm1),
        .RoundData(RndGTableMulLamdaIm1)
        );

// Instantiation of Saturation component
SatSigned #(.INPUT_WIDTH(30),
            .OUTPUT_WIDTH(25))
  U2_SAT_RE1(
            .InputData(LambdaRe1),
            .SatData(SatLambdaRe1)
            );

SatSigned #(.INPUT_WIDTH(30),
            .OUTPUT_WIDTH(25))
  U2_SAT_IM1(
            .InputData(LambdaIm1),
            .SatData(SatLambdaIm1)
            );

// Round by 10 bits.
Round #(.INPUT_WIDTH(25),
        .OUTPUT_WIDTH(15))
  U5_ROUND_RE1(
        .InputData(SatLambdaRe1),
        .RoundData(SatDCEstRe1)
        );

Round #(.INPUT_WIDTH(25),
        .OUTPUT_WIDTH(15))
  U5_ROUND_IM1(
        .InputData(SatLambdaIm1),
        .RoundData(SatDCEstIm1)
        );

`endif


// Assigning accumulator so that it would be easy to debug
`ifdef RW_SIMU_ON
reg signed [20:0] sumValReDebug;
reg signed [20:0] sumValImDebug;

always@(posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    sumValReDebug <= 21'b0;
    sumValImDebug <= 21'b0;
  end
  else if (LoadAccu && RxDataInValid_1t && FlagStartDC && (DataSynch || !PrDataPhase)) begin
     sumValReDebug <= RxDataRe;
     sumValImDebug <= RxDataIm;
   end
end

`endif

endmodule

`default_nettype wire

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
