//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: cvandeburie $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 34361 $
// $Date: 2018-06-22 11:57:21 +0200 (Fri, 22 Jun 2018) $
// ---------------------------------------------------------------------------
// Dependencies     :                                                       
// Description      : Time Domain Frequency Offset - Modulus Calc
// Simulation Notes :                                                       
// Synthesis Notes  :                                                       
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// $HeadURL:  $
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
module SqrtOf2Sqrs # (
  parameter INWIDTH  = 12 // Min 12
)(

  //Clock and Reset
  input   wire                  nPhyRst, // Active LOW Reset
  input   wire                  PhyClk,  // PHY Clock

  //Control Signals
  input   wire                  Enable,

  //Data
  input   wire [INWIDTH-1:0]    DataInRe, //Real Comp of Data
  input   wire [INWIDTH-1:0]    DataInIm, //Imag Comp of Data

  output  reg                   ModulusValid,
  output  wire [2*INWIDTH-11:0] Modulus
);

//////////////////////////////////////////////////////////////////////////////
// Local Parameters Declarations
//////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////////
//  Internal Signals Declarations
//////////////////////////////////////////////////////////////////////////////
reg            [3:0] StateCount;
wire           [3:0] StateCountMax;
wire   [INWIDTH-1:0] AbsDataInRe, MinusDataInRe;
wire   [INWIDTH-1:0] AbsDataInIm, MinusDataInIm;
wire   [INWIDTH-1:0] NextxMaxAbs, NextyMinAbs;
reg    [INWIDTH-1:0] xMaxAbs, yMinAbs;
wire                 AbsReBiggerAbsIm;
reg    [INWIDTH-1:0] MultIn0, MultIn1, Beta;
reg  [2*INWIDTH-2:0] MultRes;
wire                 ySmaller, ySmallerValid;
reg  [2*INWIDTH-1:0] Accu;


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////


assign StateCountMax = 4'd13;

// Counter for multiplier muxing
always @ (posedge PhyClk or negedge nPhyRst)
begin: Mult_Mux_Blk
  if (nPhyRst == 1'b0) begin
    StateCount   <= 4'd0;
  end else if (Enable == 1'b1) begin
    if (StateCount<StateCountMax) begin
      if ((ySmaller==1'b1) && (ySmallerValid==1'b1))
        StateCount <= 4'd10;
      else
        StateCount <= StateCount+4'd1;
    end
  end else begin
    StateCount   <= 4'd0;
  end
end //Mult_Mux_Blk


// Compute min/max of Im and Re parts
assign MinusDataInRe = ~DataInRe + {{(INWIDTH-1){1'b0}},1'b1};
assign AbsDataInRe   = DataInRe[INWIDTH-1] ? MinusDataInRe : DataInRe;

assign MinusDataInIm = ~DataInIm + {{(INWIDTH-1){1'b0}},1'b1};
assign AbsDataInIm   = DataInIm[INWIDTH-1] ? MinusDataInIm : DataInIm;

assign AbsReBiggerAbsIm = (AbsDataInRe>AbsDataInIm);
assign NextxMaxAbs = AbsReBiggerAbsIm ? AbsDataInRe : AbsDataInIm;
assign NextyMinAbs = AbsReBiggerAbsIm ? AbsDataInIm : AbsDataInRe;

always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    xMaxAbs <= {INWIDTH{1'b0}};
    yMinAbs <= {INWIDTH{1'b0}};
  end else if (Enable == 1'b1) begin
    if (StateCount==4'd0) begin
      xMaxAbs <= NextxMaxAbs; // x in Matlab code
      yMinAbs <= NextyMinAbs; // y in Matlab code
    end
  end
end

assign ySmaller      = ( {{(INWIDTH-11){1'b0}},yMinAbs,10'd0} < MultRes );
assign ySmallerValid = (StateCount < 4'd10) && (StateCount > 4'd2);

// Multiplier with multiplexed inputs
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    MultIn0 <= {INWIDTH{1'b0}};
    MultIn1 <= {INWIDTH{1'b0}};
    Beta    <= {INWIDTH{1'b0}};
    MultRes <= {(2*INWIDTH-1){1'b0}};
  end else if (Enable == 1'b1) begin
    // Multiplier output
    MultRes <= {{(INWIDTH-1){1'b0}},MultIn0} * {{(INWIDTH-1){1'b0}},MultIn1};
    
    // Multiplier inputs, Beta
    case (StateCount)
      4'd1   : begin
        MultIn0 <= xMaxAbs;
        MultIn1 <= 'd101; // round(tan(1*pi/32)*2^10)
      end
      4'd2   : begin
        MultIn0 <= xMaxAbs;
        MultIn1 <= 'd204; // round(tan(2*pi/32)*2^10)
      end
      4'd3   : begin
        MultIn0 <= xMaxAbs;
        if (ySmaller == 1'b1)
          MultIn1 <= 'd1023; // alpha
        else
          MultIn1 <= 'd311; // round(tan(3*pi/32)*2^10)
        Beta    <= 'd50;
      end
      4'd4   : begin
        MultIn0 <= xMaxAbs;
        if (ySmaller == 1'b1)
          MultIn1 <= 'd1014; // alpha
        else
          MultIn1 <= 'd424; // round(tan(4*pi/32)*2^10)
        Beta    <= 'd150;
      end
      4'd5   : begin
        MultIn0 <= xMaxAbs;
        if (ySmaller == 1'b1)
          MultIn1 <= 'd994; // alpha
        else
          MultIn1 <= 'd547; // round(tan(5*pi/32)*2^10)
        Beta    <= 'd249;
      end
      4'd6   : begin
        MultIn0 <= xMaxAbs;
        if (ySmaller == 1'b1)
          MultIn1 <= 'd965; // alpha
        else
          MultIn1 <= 'd684; // round(tan(6*pi/32)*2^10)
        Beta    <= 'd345;
      end
      4'd7   : begin
        MultIn0 <= xMaxAbs;
        if (ySmaller == 1'b1)
          MultIn1 <= 'd926; // alpha
        else
          MultIn1 <= 'd840; // round(tan(7*pi/32)*2^10)
        Beta    <= 'd438;
      end
      4'd8   : begin
        MultIn0 <= xMaxAbs;
        if (ySmaller == 1'b1)
          MultIn1 <= 'd879; // alpha
        Beta    <= 'd527;
      end
      4'd9   : begin
        MultIn0 <= xMaxAbs;
        if (ySmaller == 1'b1) begin
          MultIn1 <= 'd823; // alpha
          Beta    <= 'd610;
        end else begin // last default values for alpha and beta
          MultIn1 <= 'd759; // alpha 
          Beta    <= 'd688;
        end
      end
      4'd10   : begin
        MultIn0 <= yMinAbs;
        MultIn1 <= Beta;
      end
      default: begin
      end
    endcase
  end
end


// Accumulator for the last multiplier results
always @ (posedge PhyClk or negedge nPhyRst)
begin
  if (nPhyRst == 1'b0) begin
    Accu         <= {(2*INWIDTH){1'b0}};
    ModulusValid <= 1'b0;
  end else if (Enable == 1'b1) begin
    case (StateCount)
      4'd11  : Accu <= {1'b0,MultRes};
      4'd12  : begin
        Accu <= Accu + {1'b0,MultRes} + 'b1000000000;
        ModulusValid <= 1'b1;
      end
      default: begin
      end
    endcase
  end else begin
    ModulusValid <= 1'b0;
  end
end

assign Modulus = Accu[2*INWIDTH-1:10];

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
